/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package ch.abacus.designcockpit.ide;

import ch.abacus.lib.ui.renderer.common.*;
import ch.abacus.lib.ui.JAPanel;
import ch.abacus.lib.ui.JAScrollPane;
import ch.abacus.lib.ui.customizer.CustomizerDataElementInterface;
import ch.abacus.lib.ui.customizer.CustomizerDataTableInterface;
import ch.abacus.lib.ui.customizer.CustomizerDataConnectionInterface;

import javax.swing.*;
import javax.swing.event.TreeModelEvent;
import javax.swing.event.TreeModelListener;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;
import java.awt.*;
import java.awt.event.*;

/**
 * Title:        uifactory
 * Description:  This is a program design tool that creates Abacus java applications using the UI Factory.  The output of this tool is an xml document that is processed by the UI factory to create the java programs.
 * Copyright:    Copyright (c) 2001
 * Company:      Abacus Research
 * @author Michael Gouker (Cagey Logic)
 * @version 1.0
 */


public class IDEComponentPalettePane extends JAPanel {
    public SuperDesignCockpit theDesignCockpit;
    DefaultTreeModel theTreeModel;
    IDEComponentPalette theTree;
    IDEComponentPaletteNode baseClass = null;

    JPopupMenu popupMenu = null;

    public IDEComponentPalettePane(SuperDesignCockpit objDesignCockpit) {
        theDesignCockpit = objDesignCockpit;

        String sComponentBranchTitle = theDesignCockpit.theLanguageManager.getMessage("ComponentsPane.ComponentsBranch.label", "Components");
        baseClass = new IDEComponentPaletteNode(sComponentBranchTitle);

        theTreeModel = new DefaultTreeModel(baseClass);
        theTreeModel.addTreeModelListener(new MyTreeModelListener());
        theTree = new IDEComponentPalette(theTreeModel, theDesignCockpit);
        //if (theDesignCockpit.theProgramMode == SuperDesignCockpit.JAVABUILDER)
        //    theTree.setEditable(true);
        theTree.getSelectionModel().setSelectionMode
                (TreeSelectionModel.SINGLE_TREE_SELECTION);
        theTree.setShowsRootHandles(true);
        theTree.addKeyListener(new ComponentPaletteKeyListener(this));
        theTree.addMouseListener(new ComponentPaletteMouseListener(theDesignCockpit));


        JAScrollPane theScrollPane = new JAScrollPane(theTree);

        //#ALEX, Small UI mod
        //setLayout(new GridLayout(1, 0));
        setLayout( new BorderLayout() );

        setBorder ( BorderFactory.createMatteBorder ( 0,1,1,1, Color.GRAY  ));
        theScrollPane.setBorder ( null);

        String sPaneTitle = theDesignCockpit.theLanguageManager.getMessage("ComponentsPane.label", "Components");
        add( new IDETitleBar ( sPaneTitle ), BorderLayout.NORTH );
        add(theScrollPane, BorderLayout.CENTER );


        // Set up popup menu.
        createContextMenu();
    }

    private void createContextMenu () {
        if (theDesignCockpit.isCustomizing())
            return; // don't display this menu if Customizing

        AddClassAction addClassAction;
        EditClassAction editClassAction;
        DeleteClassAction deleteClassAction;
        SaveMetadataAction saveMetadataAction;
        HelpAction helpAction;

        JMenuItem addClassItem, deleteClassItem, editClassItem, saveMetadataItem, helpItem;

        popupMenu = new JPopupMenu();
        addClassAction = new AddClassAction("Add Class", theDesignCockpit);
        addClassItem = new JMenuItem(addClassAction);
        editClassAction = new EditClassAction("Edit Class", theDesignCockpit);
        editClassItem = new JMenuItem(editClassAction);
        deleteClassAction = new DeleteClassAction("Delete Class", theDesignCockpit);
        deleteClassItem = new JMenuItem(deleteClassAction);
        saveMetadataAction = new SaveMetadataAction("Save Classes", theDesignCockpit);
        saveMetadataItem = new JMenuItem(saveMetadataAction);
        helpAction = new HelpAction("Help", theDesignCockpit);
        helpItem = new JMenuItem(helpAction);
        popupMenu.add(saveMetadataItem);
        popupMenu.addSeparator();
        popupMenu.add(addClassItem);
        popupMenu.add(editClassItem);
        popupMenu.add(deleteClassItem);
        popupMenu.addSeparator();
        popupMenu.add(helpItem);
    }

    /** Remove all nodes except the root node. */
    public void clear() {
        baseClass.removeAllChildren();
        theTreeModel.reload();
    }

    /** Remove the currently selected node. */
    public void removeCurrentNode() {
        TreePath currentSelection = theTree.getSelectionPath();
        if (currentSelection != null) {
            IDEComponentPaletteNode currentNode = (IDEComponentPaletteNode)
                    (currentSelection.getLastPathComponent());
            MetaClass theMetaClass = currentNode.getMetaClass();
            theMetaClass.DeleteClass();
            IDEComponentPaletteNode parent = (IDEComponentPaletteNode) (currentNode.getParent());
            if (parent != null) {
                theTreeModel.removeNodeFromParent(currentNode);
                return;
            }
        }

    }

    public void removeNode(IDEComponentPaletteNode nodeToRemove) {
        MetaClass theMetaClass = nodeToRemove.getMetaClass();
        theMetaClass.DeleteClass();
        IDEComponentPaletteNode parent = (IDEComponentPaletteNode) (nodeToRemove.getParent());
        if (parent != null)
            theTreeModel.removeNodeFromParent(nodeToRemove);
    }

    /** Add child to the currently selected node. */
    public IDEComponentPaletteNode addClass(String theNodeText) {
        IDEComponentPaletteNode parentNode = null;
        TreePath parentPath = theTree.getSelectionPath();

        if (parentPath == null) {
            parentNode = baseClass;
        } else {
            parentNode = (IDEComponentPaletteNode)
                    (parentPath.getLastPathComponent());
        }

        return addClass(parentNode, theNodeText, true);
    }

    public IDEComponentPaletteNode addClass(IDEComponentPaletteNode parent, String theNodeText) {
        return addClass(parent, theNodeText, false);
    }

    public IDEComponentPaletteNode addClass(IDEComponentPaletteNode parent,
                                            String theNodeText,
                                            boolean shouldBeVisible) {
        IDEComponentPaletteNode childNode =
                new IDEComponentPaletteNode(theNodeText);

        if (parent == null) {
            parent = baseClass;
        }

        theTreeModel.insertNodeInto(childNode, parent,
                parent.getChildCount());

        // Make sure the user can see the lovely new node.
        if (shouldBeVisible) {
            theTree.scrollPathToVisible(new TreePath(childNode.getPath()));
        }
        return childNode;
    }

    public IDEComponentPaletteNode addCustomizerDataConnection(IDEComponentPaletteNode parent,
                                            CustomizerDataConnectionInterface theDataConnection) {
        IDEComponentPaletteNode childNode = new IDEComponentPaletteNode(theDataConnection);

        if (parent == null) {
            parent = baseClass;
        }

        theTreeModel.insertNodeInto(childNode, parent, parent.getChildCount());

        return childNode;
    }

    /**
     * Adds a Customizer node to the tree.
     * @param parent - the parent node to attach to
     * @param metaClass - the metaclass that the node represents
     * @param theDataElement - the CustomizerDataElementInterface this node represents
     * @return a IDEComponentPaletteNode instance
     */
    public IDEComponentPaletteNode addClass(IDEComponentPaletteNode parent,
                                            CustomizerDataElementInterface theDataElement,
                                            MetaClass metaClass) {
        IDEComponentPaletteNode childNode = new IDEComponentPaletteNode(metaClass, theDataElement);

        if (parent == null)
            parent = baseClass;

        theTreeModel.insertNodeInto(childNode, parent, parent.getChildCount());

        return childNode;
    }

    public IDEComponentPaletteNode addCustomizerDataTable(IDEComponentPaletteNode parent,
                                            CustomizerDataTableInterface theDataTable,
                                            boolean bMakeVisible,
                                            boolean bDeferElements) {
        IDEComponentPaletteNode childNode = new IDEComponentPaletteNode(theDataTable, bDeferElements);

        if (parent == null)
            parent = baseClass;

        theTreeModel.insertNodeInto(childNode, parent, parent.getChildCount());

        if (bMakeVisible) {
            theTree.scrollPathToVisible(new TreePath(childNode.getPath()));
        }

        return childNode;
    }

    boolean loadClass(MetaClass theFirstClass, IDEComponentPaletteNode thisNode) {
        // A node in the treeview for every class in the heirarchy.
        MetaClass currClass = theFirstClass;
        while (currClass != null) {
            if (currClass.getMetadata().bHidden == false) {  // hides all subclasses too.
                IDEComponentPaletteNode theNewNode = new IDEComponentPaletteNode(currClass);
                if (thisNode == null)
                    theTreeModel.insertNodeInto(theNewNode, baseClass,
                            baseClass.getChildCount());
                else
                    theTreeModel.insertNodeInto(theNewNode, thisNode,
                            thisNode.getChildCount());
                if (currClass.getFirstClass() != null)
                    loadClass(currClass.getFirstClass(), theNewNode);
            }
            currClass = currClass.getNextSibling();
        }

        return true;
    }

    boolean load(MetaClass theFirstClass, IDEComponentPaletteNode thisNode) {
        resetContents();
        boolean bClassLoaded = loadClass(theFirstClass, thisNode);
        if (theDesignCockpit.isCustomizing())
            theDesignCockpit.theCustomizer.loadFormElements();

        return bClassLoaded;
    }

    public void addClass(MetaClass currClass) {
        if (currClass != null) {
            IDEComponentPaletteNode theSelectedNode = (IDEComponentPaletteNode) theTree.getLastSelectedPathComponent();
            IDEComponentPaletteNode theNewNode = new IDEComponentPaletteNode(currClass);
            theTreeModel.insertNodeInto(theNewNode, theSelectedNode,
                    theSelectedNode.getChildCount());
            currClass.theTreeNode = theNewNode;
        }
    }

    public void newClassNode() {
        IDEComponentPaletteNode theSelectedNode = (IDEComponentPaletteNode) theTree.getLastSelectedPathComponent();
        if (theSelectedNode != null) {
            MetaClass theSuperClass = theSelectedNode.getMetaClass();
            MetaClass theSubClass = null;
            MetaProjectHeader theProgram = null;
            try {
                theProgram = theDesignCockpit.getProgramMetadata();
            } catch (HammerException e1) {
                return;  // no point in going on.
            }
            if (theSuperClass == null) {
//                String sNewClassName = getNewClassName();
                String sNewClassName = "newClass";
                MetaClassDetail theClassMetadata = new MetaClassDetail(sNewClassName, sNewClassName);
                theSubClass = new MetaClass(theClassMetadata,
                        null, theDesignCockpit.getDesignProject().theMetaProject.getMetadataDispenser(), true);
                theDesignCockpit.getDesignProject().theMetaProject.addClass(theSubClass);
            } else
                theSubClass = theSuperClass.subclassClass(theProgram.sPackageName, theDesignCockpit.getMode());
            if (theSubClass != null) {
                addClass(theSubClass);
                if (theSuperClass == null)
                    theDesignCockpit.getDesignProject().theMetaProject.addInstanceTracker(theSubClass);
                else {
                    MetaClass theSuperClassTracker = theDesignCockpit.getDesignProject().theMetaProject.findInstanceTracker(theSuperClass.getMetadata().sClassName);
                    theSuperClassTracker.addInstanceTracker(theSubClass);
                }
            }

        }
    }

    public void resetContents() {
        baseClass.removeAllChildren();
        theTreeModel = new DefaultTreeModel(baseClass);
        theTree.setModel(theTreeModel);
    }

    public void clearSelection(IDEComponentPaletteNode theNode) {
        if (theNode.getChildCount() != 0) {
            theNode = (IDEComponentPaletteNode) theNode.getFirstChild();
            while (theNode != null) {
                theNode.setSelected(false);
                clearSelection(theNode);
                theNode = (IDEComponentPaletteNode) theNode.getNextSibling();
            }
        }
    }


    public void clearSelection() {
        IDEComponentPaletteNode theNode = baseClass;
        if (theNode != null) {
            if (theNode.getChildCount() != 0) {
                theNode = (IDEComponentPaletteNode) theNode.getFirstChild();
                while (theNode != null) {
                    theNode.setSelected(false);
                    clearSelection(theNode);
                    theNode = (IDEComponentPaletteNode) theNode.getNextSibling();
                }
            }
        }
        repaint();
    }

    int countSelections(IDEComponentPaletteNode theNode, int iInitValue) {
        int iSelectionCount = iInitValue;
        while (theNode != null) {
            if (theNode.isSelected() == true)
                iSelectionCount++;
            if (theNode.getChildCount() != 0)
                iSelectionCount = countSelections((IDEComponentPaletteNode) theNode.getFirstChild(), iSelectionCount);
            theNode = (IDEComponentPaletteNode) theNode.getNextSibling();
        }
        return iSelectionCount;
    }

    int collectSelections(IDEComponentPaletteNode theNode, int iIndex, MetaClass[] theSelections) {
        while (theNode != null) {
            if (theNode.isSelected() == true)
                theSelections[iIndex++] = theNode.getMetaClass();
            if (theNode.getChildCount() != 0)
                iIndex = collectSelections((IDEComponentPaletteNode) theNode.getFirstChild(), iIndex, theSelections);
            theNode = (IDEComponentPaletteNode) theNode.getNextSibling();
        }
        return iIndex;
    }

    MetaClass[] getSelections() {
        IDEComponentPaletteNode theCurrentNode = (IDEComponentPaletteNode) baseClass.getFirstChild();
        int iSelectionCount = countSelections(theCurrentNode, 0);
        MetaClass[] theSelections = new MetaClass[iSelectionCount];
        collectSelections(theCurrentNode, 0, theSelections);
        return theSelections;
    }

    class AddClassAction extends AbstractAction {
        SuperDesignCockpit theDesignCockpit = null;

        public AddClassAction(String label, Icon icon, SuperDesignCockpit objDesignCockpit) {
            super(label, icon);
            theDesignCockpit = objDesignCockpit;
        }

        public AddClassAction(String label, SuperDesignCockpit objDesignCockpit) {
            super(label);
            theDesignCockpit = objDesignCockpit;
        }

        public void actionPerformed(ActionEvent evt) {
            IDEComponentPaletteNode theSelectedNode = (IDEComponentPaletteNode)
                    theTree.getLastSelectedPathComponent();
            if (theSelectedNode != null)
                newClassNode();
        }
    }

    class DeleteClassAction extends AbstractAction {
        SuperDesignCockpit theDesignCockpit = null;

        public DeleteClassAction(String label, Icon icon, SuperDesignCockpit objDesignCockpit) {
            super(label, icon);
            theDesignCockpit = objDesignCockpit;
        }

        public DeleteClassAction(String label, SuperDesignCockpit objDesignCockpit) {
            super(label);
            theDesignCockpit = objDesignCockpit;
        }

        public void actionPerformed(ActionEvent evt) {
            IDEComponentPaletteNode theSelectedNode = (IDEComponentPaletteNode)
                    theTree.getLastSelectedPathComponent();
            if (theSelectedNode != null)
                removeNode(theSelectedNode);
        }
    }

    class EditClassAction extends AbstractAction {
        SuperDesignCockpit theDesignCockpit = null;

        public EditClassAction(String label, Icon icon, SuperDesignCockpit objDesignCockpit) {
            super(label, icon);
            theDesignCockpit = objDesignCockpit;
        }

        public EditClassAction(String label, SuperDesignCockpit objDesignCockpit) {
            super(label);
            theDesignCockpit = objDesignCockpit;
        }

        public void actionPerformed(ActionEvent evt) {
            theTree.PopupClassTool();
        }
    }

    class SaveMetadataAction extends AbstractAction {
        SuperDesignCockpit theDesignCockpit = null;

        public SaveMetadataAction(String label, Icon icon, SuperDesignCockpit objDesignCockpit) {
            super(label, icon);
            theDesignCockpit = objDesignCockpit;
        }

        public SaveMetadataAction(String label, SuperDesignCockpit objDesignCockpit) {
            super(label);
            theDesignCockpit = objDesignCockpit;
        }

        public void actionPerformed(ActionEvent evt) {
            theDesignCockpit.theCurrentDesignProject.theMetaProject.getMetadataDispenser().theMetadataDocument.writeMetaDocument();
        }
    }

    class HelpAction extends AbstractAction {
        SuperDesignCockpit theDesignCockpit = null;

        public HelpAction(String label, Icon icon, SuperDesignCockpit objDesignCockpit) {
            super(label, icon);
            theDesignCockpit = objDesignCockpit;
        }

        public HelpAction(String label, SuperDesignCockpit objDesignCockpit) {
            super(label);
            theDesignCockpit = objDesignCockpit;
        }

        public void actionPerformed(ActionEvent evt) {
        }
    }

    class ComponentPaletteMouseListener implements MouseListener {
        SuperDesignCockpit theDesignCockpit = null;

        public ComponentPaletteMouseListener(SuperDesignCockpit theDesignCockpit) {
            this.theDesignCockpit = theDesignCockpit;
        }

        public void mouseClicked(MouseEvent evt) {
            if (theDesignCockpit.isCustomizing())
                return; // don't show the tool if Cutomizing

            if (theDesignCockpit.theProgramMode == SuperDesignCockpit.JAVABUILDER) {
                if (evt.getClickCount() == 2) {
                    IDEComponentPalette theComponentPalette = (IDEComponentPalette) evt.getSource();
                    theComponentPalette.PopupClassTool();
                }
            }
        }

        public void mouseEntered(MouseEvent evt) {
        }

        public void mouseExited(MouseEvent evt) {
        }

        public void mousePressed(MouseEvent evt) {
            showPopup(evt);
        }

        public void mouseReleased(MouseEvent evt) {
            showPopup(evt);
        }

        private void showPopup(MouseEvent evt) {
            if (theDesignCockpit.theProgramMode == SuperDesignCockpit.JAVABUILDER) {
                if (evt.isPopupTrigger() && popupMenu != null)
                    popupMenu.show(evt.getComponent(), evt.getX(), evt.getY());
            }
        }
    }

    class ComponentPaletteKeyListener extends KeyAdapter {
        IDEComponentPalettePane theComponentPalettePane = null;

        public ComponentPaletteKeyListener(IDEComponentPalettePane objComponentPalettePane) {
            super();
            theComponentPalettePane = objComponentPalettePane;
        }

        public void keyPressed(KeyEvent e) {
            if (theComponentPalettePane.theDesignCockpit.theProgramMode == SuperDesignCockpit.JAVABUILDER) {
                if (e.getKeyCode() == e.VK_DELETE) {
                    IDEComponentPaletteNode theSelectedNode = (IDEComponentPaletteNode)
                            theComponentPalettePane.theTree.getLastSelectedPathComponent();
                    theComponentPalettePane.removeNode(theSelectedNode);
                }
            }
        }
    }

    class MyTreeModelListener implements TreeModelListener {
        public void treeNodesChanged(TreeModelEvent e) {
            IDEComponentPaletteNode node;
            node = (IDEComponentPaletteNode)
                    (e.getTreePath().getLastPathComponent());

            /*
             * If the event lists children, then the changed
             * node is the child of the node we've already
             * gotten.  Otherwise, the changed node and the
             * specified node are the same.
             */
            try {
                int index = e.getChildIndices()[0];
                node = (IDEComponentPaletteNode)
                        (node.getChildAt(index));
                node.getMetaClass().getMetadata().sClassName = (String) node.getUserObject();
            } catch (NullPointerException exc) {
            }

        }

        public void treeNodesInserted(TreeModelEvent e) {
        }

        public void treeNodesRemoved(TreeModelEvent e) {
        }

        public void treeStructureChanged(TreeModelEvent e) {
        }
    }


}



