/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package ch.abacus.designcockpit.ide;

import ch.abacus.designcockpit.ide.propertyinspector.IDEPropertyInspectorInterface;
import ch.abacus.designcockpit.ide.propertyinspector.IDEAbacusPropertyInspector;
import ch.abacus.lib.ui.renderer.common.*;

import javax.swing.*;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableColumn;
import javax.swing.event.ChangeEvent;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.ArrayList;

/**
 * Created by IntelliJ IDEA.
 * User: Stuart Booth (Live Clue, inc)
 * Date: Jul 24, 2005
 * Time: 3:21:45 PM
 */
public class IDEPropertyInspectorController {
    private final SuperDesignCockpit m_superDesignCockpit;
    IDEPropertyInspectorInterface m_propertyInspector;
    PropertyTable m_defaultPropertyInspector;
    private LanguageManager m_languageManager;
    private MetaObject[] m_selectedComponents;

    public IDEPropertyInspectorController(SuperDesignCockpit superDesignCockpit) {
        this.m_superDesignCockpit = superDesignCockpit;
        m_languageManager = superDesignCockpit.theLanguageManager;

        m_propertyInspector = createPropertyInspector();
        if (m_propertyInspector == null) {
            PropertyTableModel thePropertyTableModel = new PropertyTableModel(m_superDesignCockpit);

            m_defaultPropertyInspector = new PropertyTable(thePropertyTableModel, m_superDesignCockpit);
        }
    }

    public void newObjectsSelected() {
        if (m_propertyInspector == null)
            return;


        final DesignProject designProject = m_superDesignCockpit.getDesignProject();
        int iSelObjCount = designProject.getSelectedObjectCount();
        //System.out.println("Selected = " +iSelObjCount);

        MetaObject[] selectedObjects = new MetaObject[iSelObjCount];
        for (int iSelObj = 0; iSelObj < iSelObjCount; iSelObj++) {
            MetaObject metaObject = designProject.getSelectedObject(iSelObj);
            //if (metaObject != null)
            //System.out.println("   " +metaObject.getName());
            selectedObjects[iSelObj] = metaObject;
        }

        if (! Arrays.equals(selectedObjects, m_selectedComponents)) {
            m_selectedComponents = selectedObjects;

            m_propertyInspector.clearSelectedObjects();
            m_propertyInspector.newObjectsSelected(selectedObjects);
        } else {
            m_propertyInspector.updateSelectedProperties();
        }
    }

    private IDEPropertyInspectorInterface createPropertyInspector() {
        // load the appropriate Property Inspector...
        //String sPropertyInspector = "ch.abacus.designcockpit.ide.propertyinspector.IDEAbacusPropertyInspector2";
        String sPropertyInspector = m_superDesignCockpit.getConfigurationManager().getConfigurationVariable("PROPERTY_INSPECTOR", "");
        if (sPropertyInspector.equals(""))
            return null;

        Class propertyInspector;
        IDEPropertyInspectorInterface inspectorInterface = null;

        String sErrorMessage = null;

        try {
            propertyInspector = Class.forName(sPropertyInspector);
            try {
                final Constructor constructor = propertyInspector.getConstructor((Class[]) null);
                try {
                    inspectorInterface = (IDEPropertyInspectorInterface) constructor.newInstance((Object[]) null);
                } catch (InstantiationException e) {
                    sErrorMessage = e.getMessage();
                    inspectorInterface = null;
                } catch (IllegalAccessException e) {
                    sErrorMessage = e.getMessage();
                    inspectorInterface = null;
                } catch (InvocationTargetException e) {
                    sErrorMessage = e.getMessage();
                    inspectorInterface = null;
                } catch (ClassCastException e) {
                    sErrorMessage = sPropertyInspector + " does not implement the IDEPropertyInspectorInterface, and cannot be used.";
                    inspectorInterface = null;
                }

            } catch (NoSuchMethodException e) {
                sErrorMessage = sPropertyInspector + " does not have a default Constructor, and cannot be used.";
                inspectorInterface = null;
            }
        } catch (ClassNotFoundException e) {
            sErrorMessage = "the class \"" + sPropertyInspector + "\" cannot be found.";
            inspectorInterface = null;
        }

        if (inspectorInterface != null)
            inspectorInterface.init(this);
        else {
            System.out.println("Property Inspector configuration is invalid: " + sErrorMessage);
        }
        return inspectorInterface;
    }

    public void refreshPropertyInspector() {
        if (m_propertyInspector != null)
            newObjectsSelected();
        else
            m_defaultPropertyInspector.Load();

    }

    public SuperDesignCockpit getSuperDesignCockpit() {
        return m_superDesignCockpit;
    }

    public Object getPropertyValue(MetaObject theMetaObject, String sPropertyName) {

        final MetaClass metaClass = theMetaObject.theClass;
        MetaPropertyDescriptor theProperty = metaClass.getProperty(sPropertyName, true);
        if (theProperty == null)
            return null;

        //sPropertyName = theProperty.getName();
        if (theMetaObject.isFrame() || theMetaObject.isAbalet()) {
            if (sPropertyName.startsWith("Location"))
                return new String("0");
        }
        int iDotPos = sPropertyName.indexOf(".");
        String sKey = sPropertyName;
        if (iDotPos != -1)
            sKey = sPropertyName.substring(0, iDotPos);
        String sType = theProperty.getClassName();
        MetaProperty theProp = theMetaObject.getPropertyMetadata(sKey);

        // We added check for open source to be on safe side must be removed if
        // the Abacus DC needs to support object reference
        if(theProperty.bObjectReference == true && this.getSuperDesignCockpit().getOpenSourceState()==true) // MHC
        {
            DesignProject theDesignProject = getSuperDesignCockpit().getDesignProject();
            ArrayList arrObj = theDesignProject.findObjectsByClass(sType);
            theProperty.setObjectReferenceList(arrObj);
        }
        else if(theProperty.getDataAwareness()==true && this.getSuperDesignCockpit().getOpenSourceState()==true) // MHC
        {
            DesignProject theDesignProject = getSuperDesignCockpit().getDesignProject();

            MetaObject mt = theDesignProject.getLastObjectSelected();
            Object objFieldAccess = mt.getSwingObject();

            Class fldAccsClass = objFieldAccess.getClass();
            Method[] availMthds = fldAccsClass.getMethods();

            for(int i=0;i < availMthds.length;i++)
            {
              Method chMth = availMthds[i];
              String methodName = chMth.getName();
              if( methodName.equalsIgnoreCase("getFieldNames") ||
                  methodName.equalsIgnoreCase("getTableNames"))
              {
                  try
                  {
                    ArrayList fields = (ArrayList)chMth.invoke(objFieldAccess, (Object[])null);
                    if(fields!=null)
                    {
                        for(int n=0;n < fields.size();n++)
                        {
                            try
                            {
                                theProperty.setValueChoice(n,fields.get(n).toString());
                            }
                            catch(Exception e)
                            {
                                theProperty.addValueChoice(fields.get(n).toString());
                            }
                        }
                    }
                    else
                     return null;
                  }
                  catch(Exception e)
                  {
                      e.printStackTrace();
                  }
//                  break;
              }
            }
        }
        else if (theProp == null) {
            if ((sType.endsWith("oolean") == false) && (theProperty.getDefaultValue() != null)) {
                // translate constants
                String sDefault = theProperty.getDefaultValue();
                Object oMatch = theProperty.getExpandedValue(theMetaObject.theDesignProject, sDefault);
                return oMatch;

            }
            else
                return null;
        }

        int iIndex = 0;
        MetaPropertyValueEx pvValue = theMetaObject.getPropertyValue(sPropertyName, iIndex);


        //  Check if there is a constant group for the property class.
        MetaConstantGroup theConstantTable = theMetaObject.theDesignProject.findConstantGroup(theProperty.getIncludeTypeConstants());
        if (theConstantTable != null) {
            if (pvValue != null) {
                String sConstantValue = pvValue.getLocalString();
                String sMatch = theConstantTable.getKey(sConstantValue);
                if (sMatch != null) {
                    if (sType.endsWith("oolean"))
                        return new Boolean(sMatch);
                    return sMatch;
                }
                return sConstantValue;
            }
        } else {
            MetaConstantGroupCollection theConstantTableCollection = theMetaObject.theDesignProject.findConstantGroupCollection(theProperty.getIncludeTypeConstants());
            if (theConstantTableCollection != null) {
                if (pvValue != null) {
                    String sConstantValue = pvValue.getLocalString();
                    String sMatch = theConstantTableCollection.getKey(sConstantValue);
                    if (sMatch != null) {
                        if (sType.endsWith("oolean"))
                            return new Boolean(sMatch);
                        return sMatch;
                    }
                    return sConstantValue;
                }
            }

        }
        //  If there is maybe we need to translate.
        if (pvValue == null) {
            Object oValue = theProperty.getDefaultValue();
            if (oValue == null)
                oValue = theProperty.getValueChoice(0);
            pvValue = new MetaPropertyValueEx(sType, oValue, true, theMetaObject.theDesignProject, theProperty.bAllowMultipleLanguages);
        } else {
            Object oValue = pvValue.getNativeValue();
            return oValue;
        }
        String sRetVal = pvValue.getStringValue();
        if (sType.endsWith("oolean"))
            return new Boolean(sRetVal);
        else
            return sRetVal;
    }

    public void setPropertyValue(MetaObject theMetaObject, String sPropertyName, Object theNewValue) {
        //if (propertyValueChangeDisallowed(theNewValue, row))
        //     return;

        //MetaObject theMetaObject = theTable.theDesignCockpit.getDesignProject().getLastObjectSelected();
        final MetaClass metaClass = theMetaObject.theClass;
        MetaPropertyDescriptor theProperty = metaClass.getProperty(sPropertyName, true);

        if (theProperty.getName().equalsIgnoreCase("Name")) {
            String sNewName = (String) theNewValue;
            // if set to "". then we must disallow that change...
            if (sNewName.trim().length() ==0) {
                return;
            }

            // We need to verify that the name is not already used. If not, we have to disqualify the change...
            if (isNameChangePermitted(sNewName) == false) {
                MetaObject metaObject = m_superDesignCockpit.getDesignProject().findObject(sNewName);
                if (!metaObject.getName().equalsIgnoreCase(theMetaObject.getName())) {
                    MetaObject metaObjectParent = metaObject.theParentObject;
                    String sMessage;
                    if (metaObjectParent != null) {
                        sMessage = m_languageManager.expandMessage("DuplcateObjectNames.hasParent.label", "That name is already used by another object, which is a child of \"{0}\"", metaObject.theParentObject.getName());
                    } else {
                        sMessage = m_languageManager.getMessage("DuplcateObjectNames.noParent.label", "That name is already used by another object, the project's first object");
                    }
                    String sTitle = m_languageManager.getMessage("DuplcateObjectNames.label", "Duplicate Object Name");
                    //JOptionPane.showMessageDialog(m_superDesignCockpit, sMessage, sTitle, JOptionPane.ERROR_MESSAGE);
                    m_superDesignCockpit.theSourceViewer.setMessagesText(sTitle + " - " + sMessage);
                    m_superDesignCockpit.theSourceViewer.showMessageEditor();
                    return;
                }
            }
        }

        // if the property is an "Anchor", then we must reset the offsets...
//        if ("|AnchorLeft|AnchorTop|AnchorRight|AnchorBottom|".indexOf("|" +sPropertyName +"|") != -1) {
//            System.out.println("ANCHORS CHANGED");
//            theMetaObject.bAnchoringOff = true;
//        }

        MetaConstantGroup theConstantTable = theMetaObject.theDesignProject.findConstantGroup(theProperty.getIncludeTypeConstants());
        MetaConstantGroupCollection theConstantTableGroup = null;
        if (theConstantTable != null) {
            if (theNewValue instanceof String) {
                String sMatch = theConstantTable.get((String) theNewValue);
                if (sMatch != null)
                    theNewValue = sMatch;
            }
        } else {
            theConstantTableGroup = theMetaObject.theDesignProject.findConstantGroupCollection(theProperty.getIncludeTypeConstants());
            if (theConstantTableGroup != null) {
                if (theNewValue instanceof String) {
                    String sMatch = theConstantTableGroup.get((String) theNewValue);
                    if (sMatch != null)
                        theNewValue = sMatch;
                }
            }
        }

        //String sPropertyName = theProperty.getName();
        theMetaObject.theDesignProject.setProjectChangedState(true);
        if (theMetaObject.isFrame() || theMetaObject.isAbalet()) {
            if (sPropertyName.startsWith("Location"))
                theNewValue = new String("0");
        }
        // indexed values are handled in the helper.
        MetaPropertyValueEx pvOldValue = theMetaObject.getPropertyValue(sPropertyName, 0);
        String sOldValue = null;
        if (pvOldValue != null)
            sOldValue = pvOldValue.getStringValue();
        String sNewReverseLookup = null;
        String sOldReverseLookup = null;
        if (theConstantTable != null) {
            sNewReverseLookup = theConstantTable.getKey((String) theNewValue);
            sOldReverseLookup = theConstantTable.getKey(sOldValue);
        } else if (theConstantTableGroup != null) {
            sNewReverseLookup = theConstantTableGroup.getKey((String) theNewValue);
            sOldReverseLookup = theConstantTableGroup.getKey(sOldValue);
        } else if (theNewValue.equals(sOldValue) == false) {
            propertyEdited(theMetaObject, sPropertyName, theProperty.getOrder(), theNewValue);
            return;
        }
        if ((theNewValue.equals(sOldValue) == false) &&
                (sNewReverseLookup != null) &&
                (sNewReverseLookup.equals(sOldReverseLookup) == false)) {
            propertyEdited(theMetaObject, sPropertyName, theProperty.getOrder(), theNewValue);
        }
    }

    /**
     * isNameChangePermitted is used to determine whether a component can be renamed.
     * Component names need to be unique within a project, so a rename is not permitted if
     * another component is already using the name you request.
     *
     * @param sNewName String - the name to change to
     * @return boolean - true indicates the name change is permitted; false indicates it is not.
     */
    private boolean isNameChangePermitted(String sNewName) {
        MetaObject mo; // reference to the metaobject with sNewName name. If name is not used, then it will be null.
        mo = m_superDesignCockpit.getDesignProject().findObject(sNewName);

        return (mo == null);
    }

    public void propertyEdited(MetaObject theMetaObject, String sPropertyName, int iOrder, Object theNewValue) {
        m_superDesignCockpit.getDesignProject().theUndoRedoController.createChangePropertyEvent(theMetaObject, sPropertyName, 0, 0);
        theMetaObject.setPropertyValue(sPropertyName, 0, iOrder, theNewValue, false);
        theMetaObject.refresh();
        if (theMetaObject.theClass.getMetadata().containerType.getContainerType() != MetaContainerType.CONTAINER_FRAME) {
            if (theMetaObject.theVisualObject != null)
                theMetaObject.theVisualObject.repaint();
        } else {
            if (theMetaObject.iMode == AbaMetaDataUser.DESIGN_MODE) {
                m_superDesignCockpit.getDesignProject().getMetaDataUser().propertyEditedRepaint(theMetaObject);
            }
        }
    }


    public void editingCanceled() {
        if (m_defaultPropertyInspector != null)
            m_defaultPropertyInspector.editingCanceled(new ChangeEvent(this));

        if (m_propertyInspector != null)
            m_propertyInspector.editingCanceled(new ChangeEvent(this));

    }

    public void resetContents() {
        if (m_defaultPropertyInspector != null)
            m_defaultPropertyInspector.resetContents();

        if (m_propertyInspector != null)
            newObjectsSelected();

    }

    public void repaint() {
        if (m_defaultPropertyInspector != null)
            m_defaultPropertyInspector.repaint();

        if (m_defaultPropertyInspector != null)
            m_defaultPropertyInspector.repaint();
    }

    public JComponent getEmbeddedPropertyInspector() {
        if (m_propertyInspector != null)
            return m_propertyInspector.getPropertyInspectorComponent();
        else
            return m_defaultPropertyInspector;
    }

    public void showHelper(MetaObject metaObject, String sPropertyName) {
        editingCanceled();
        m_superDesignCockpit.showHelper(metaObject, sPropertyName);
    }

}
