/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package ch.abacus.designcockpit.ide.languages;

import ch.abacus.lib.ui.JAFrame;

import javax.swing.*;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.util.*;

/**
 * Created by IntelliJ IDEA.
 * User: Stuart
 * Date: Oct 1, 2004
 * Time: 11:15:33 AM
 */
public class VerifyLanguageFiles extends JAFrame {
    //File firstFile = null;
    //File secondFile = null;
    ResourceBundle theFirstBundle = null;
    ResourceBundle theSecondBundle = null;
    boolean bError = false; // set true if any individual test fails
    String sDirLanguageFile = null;


    public VerifyLanguageFiles () {
        addWindowListener(new CloseWindow());
    }

    public void verify() {
        //selectFiles();
        processFiles();

        if (bError)
            System.out.println("\n***ERRORS WERE FOUND ***");
        else
            System.out.println("\nEveything ok");
    }

    /**
     * checks ResourceBundles to ensure they have the same keys defined
     * @return
     */
    private boolean hasSameKeys () {
        // 1. create a hashmap of the first resource bundle
        // 2. ensure each key in the second resource-bundle exists in the first bundle
        // 3. if it does, remove it from first bundle
        // 4. if not, show error
        // 5. repeat step 2 until all keys in the second bundle are tested
        // 5. any keys left in the first bundle are surplus, so report error

        boolean bErrors = false; // errors found in the test?

        // create a hashset of the first resource-bundle...
        HashSet firstSet = new HashSet();
        Enumeration firstBundleKeys = theFirstBundle.getKeys();
        while (firstBundleKeys.hasMoreElements()) {
            String sKey = (String) firstBundleKeys.nextElement();
            firstSet.add(sKey);
        }

        // for each key in the second resource-bundle, ensure it is in the first hashmap, and if it is remove it
        Enumeration secondBundleKeys = theSecondBundle.getKeys();
        while (secondBundleKeys.hasMoreElements()) {
            String sKey = (String) secondBundleKeys.nextElement();

            if (firstSet.contains(sKey)) {
                firstSet.remove(sKey);
            } else {
                System.out.println("Error: The \"" + sKey + "\" key in the German file is missing in the English file");
                bErrors = true;
            }
        }

        // Any items left in the first map didn't exist in the second bundle...
        Iterator iterator = firstSet.iterator();
        while (iterator.hasNext()) {
            String sKey = (String) iterator.next();
            System.out.println("Error: The \"" + sKey + "\" key in the English file is missing in the German file");
            bErrors = true;
        }

        return bErrors;
    }

    /**
     * checks ResourceBundles to ensure they Keys have different values (meaning it was copied, but not translated)
     *
     * @return
     */
    private boolean hasSameStrings () {
        boolean bErrors = false; // errors found in the test?

        Enumeration firstBundleKeys = theFirstBundle.getKeys();
        while (firstBundleKeys.hasMoreElements()) {
            String sKey = (String) firstBundleKeys.nextElement();
            try {
                String sValue = theFirstBundle.getString(sKey);
                String sSecondValue = theSecondBundle.getString(sKey);

                if (sValue.equals(sSecondValue)) {
                    if (!(sKey.endsWith(".image") || sKey.endsWith(".mnemonic") || sKey.endsWith(".accelerator"))) {
                        System.out.println("Warning: The \"" + sKey + "\" key has the same value of \"" + sValue + "\" in both language files");
                    }
                } else {
                    if (sKey.endsWith(".image")) { // warn if the images have changed. This is not an error!
                        System.out.println("Warning: The \"" + sKey + "\" key has a value of \"" + sValue + "\" in the English file, but \"" + sSecondValue + "\" in the German file");
                    }

                }
            } catch (MissingResourceException e) {
            } // ignore null keys - will have been reported in the hasSameKeys() method
        }

        return bErrors;
    }

    /**
     * checks ResourceBundles to ensure they all mnemonics are contained in the label
     *
     * @return
     */
    private boolean hasValidMnemonics () {
        boolean bErrors = false; // errors found in the test?

        Enumeration firstBundleKeys = theFirstBundle.getKeys();
        while (firstBundleKeys.hasMoreElements()) {
            String sKey = (String) firstBundleKeys.nextElement();
            try {

                if (sKey.endsWith(".mnemonic")) {
                    String sFirstMnemonic = theFirstBundle.getString(sKey);
                    String sLabelKey = sKey.substring(0, sKey.lastIndexOf(".mnemonic")) + ".label"; //
                    String sFirstLabel = null;
                    try {
                        sFirstLabel = theFirstBundle.getString(sLabelKey);

                        // now check that the mnemonic is in the label
                        if (sFirstLabel.toUpperCase().indexOf(sFirstMnemonic) == -1) {
                            System.out.println("Error (English File): The mnemonic of the key \"" + sKey + "\" (Value =\"" + sFirstMnemonic + "\") is not contained in the associated label, \"" + sLabelKey + "\" (Value = \"" + sFirstLabel + "\")");
                            bErrors = true;
                        }
                    } catch (MissingResourceException e) {
                        System.out.println("Error (English File): The key \"" + sKey + "\" does not have an associated \"" + sLabelKey + "\" key");
                        bErrors = true;
                    }

                    // Now check the German file...
                    String sSecondMnemonic = theSecondBundle.getString(sKey);
                    String sSecondLabel = null;
                    try {
                        sSecondLabel = theSecondBundle.getString(sLabelKey);

                        // now check that the mnemonic is in the label
                        if (sSecondLabel.toUpperCase().indexOf(sSecondMnemonic) == -1) {
                            System.out.println("Error (German File): The mnemonic of the key \"" + sKey + "\" (Value =\"" + sSecondMnemonic + "\") is not contained in the associated label, \"" + sLabelKey + "\" (Value = \"" + sSecondLabel + "\")");
                            bErrors = true;
                        }
                    } catch (MissingResourceException e) {
                        System.out.println("Error (German File): The key \"" + sKey + "\" does not have an associated \"" + sLabelKey + "\" key");
                        bErrors = true;
                    }
                }

            } catch (MissingResourceException e) {
            } // ignore null keys - will have been reported in the hasSameKeys() method
        }

        return bErrors;
    }

    private void processFiles () {
        boolean bTestError; // used to determine if each indiovidual test failed

        // Process the "Actions" file...
        System.out.println("Examining the Action files...");
        String sBundleName = sDirLanguageFile;
        sBundleName += "actions";

        try {
            theFirstBundle = ResourceBundle.getBundle(sBundleName, new Locale("en", "", ""));
        } catch (MissingResourceException e) {
            System.out.println("Could not find actions_en (Using path: " +sDirLanguageFile +")");
            System.exit(2);
        }
        try {
            theSecondBundle = ResourceBundle.getBundle(sBundleName, new Locale("de", "", ""));
        } catch (MissingResourceException e) {
            System.out.println("Could not find actions_de (Using path: " +sDirLanguageFile +")");
            System.exit(2);
        }

        bTestError = hasSameKeys();
        if (bTestError)
            bError = true;

        bTestError = hasSameStrings();
        if (bTestError)
            bError = true;

        bTestError = hasValidMnemonics();
        if (bTestError)
            bError = true;

        // Now examine the "Frame Components" file...
        System.out.println("\nExamining the Frame Components files...");
        sBundleName = sDirLanguageFile;
        sBundleName += "frameComponents";

        try {

            theFirstBundle = ResourceBundle.getBundle(sBundleName, new Locale("en", "", ""));
        } catch (MissingResourceException e) {
            System.out.println("Could not find frameComponents_en (Using path: " +sDirLanguageFile +")");
            System.exit(2);
        }
        try {
            theSecondBundle = ResourceBundle.getBundle(sBundleName, new Locale("de", "", ""));
        } catch (MissingResourceException e) {
            System.out.println("Could not find frameComponents_de (Using path: " +sDirLanguageFile +")");
            System.exit(2);
        }

        bTestError = hasSameKeys();
        if (bTestError)
            bError = true;

        bTestError = hasSameStrings();
        if (bTestError)
            bError = true;

        bTestError = hasValidMnemonics();
        if (bTestError)
            bError = true;

        // Now examine the "Dialogs" file...
        System.out.println("\nExamining the Dialog files...");
        sBundleName = sDirLanguageFile;
        sBundleName += "dialogs";

        try {

            theFirstBundle = ResourceBundle.getBundle(sBundleName, new Locale("en", "", ""));
        } catch (MissingResourceException e) {
            System.out.println("Could not find dialogs_en (Using path: " +sDirLanguageFile +")");
            System.exit(2);
        }
        try {
            theSecondBundle = ResourceBundle.getBundle(sBundleName, new Locale("de", "", ""));
        } catch (MissingResourceException e) {
            System.out.println("Could not find dialogs_de (Using path: " +sDirLanguageFile +")");
            System.exit(2);
        }

        bTestError = hasSameKeys();
        if (bTestError)
            bError = true;

        bTestError = hasSameStrings();
        if (bTestError)
            bError = true;

        bTestError = hasValidMnemonics();
        if (bTestError)
            bError = true;

        // Now examine the "Errors" file...
        System.out.println("\nExamining the Error-Message files...");
        sBundleName = sDirLanguageFile;
        sBundleName += "errors";

        try {

            theFirstBundle = ResourceBundle.getBundle(sBundleName, new Locale("en", "", ""));
        } catch (MissingResourceException e) {
            System.out.println("Could not find errors_en (Using path: " +sDirLanguageFile +")");
            System.exit(2);
        }
        try {
            theSecondBundle = ResourceBundle.getBundle(sBundleName, new Locale("de", "", ""));
        } catch (MissingResourceException e) {
            System.out.println("Could not find errors_de (Using path: " +sDirLanguageFile +")");
            System.exit(2);
        }

        bTestError = hasSameKeys();
        if (bTestError)
            bError = true;

        bTestError = hasSameStrings();
        if (bTestError)
            bError = true;

        bTestError = hasValidMnemonics();
        if (bTestError)
            bError = true;
    }

//    private void selectFiles () {
//        JFileChooser chooser = new JFileChooser();
//        LanguageFileFilter filter = new LanguageFileFilter();
//        filter.addExtension("properties");
//        filter.setDescription("Language Files");
//        chooser.setFileFilter(filter);
//
//        firstFile = pickLanguageFile(chooser, "Pick First File");
//        if (firstFile == null)
//            System.exit(0);
//
//        secondFile = pickLanguageFile(chooser, "Pick Second File");
//        if (secondFile == null)
//            System.exit(0);
//
//    }


    File pickLanguageFile (JFileChooser chooser, String sLabel) {
        chooser.setDialogTitle(sLabel);
        int returnVal = chooser.showOpenDialog(/*parent*/null);
        if (returnVal == JFileChooser.APPROVE_OPTION)
            return chooser.getSelectedFile();
        else
            return null;
    }

    public static void main (String[] args) {
        VerifyLanguageFiles verifyLanguageFiles = new VerifyLanguageFiles();
        if (args.length > 0) {
            String sPath = args[0];
            if (!sPath.endsWith(File.separator))
                sPath += File.separator;
            verifyLanguageFiles.sDirLanguageFile = sPath;
        } else {
            System.out.println("Cannot continue. You must pass the path containing the files (this path must be in the Class Path)");
        }

        verifyLanguageFiles.verify();
        //verifyLanguageFiles.setVisible(true);
        //verifyLanguageFiles.show();

    }

    private class CloseWindow extends WindowAdapter {
        public void windowClosing (WindowEvent e) {
            System.exit(0);
        }
    }
}
