/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package ch.abacus.designcockpit.ide;

import ch.abacus.lib.ui.renderer.common.*;
import ch.abacus.lib.ui.JAPanel;
import ch.abacus.lib.ui.JAScrollPane;
import ch.abacus.lib.ui.JATabbedPane;
import ch.abacus.lib.ui.JALabel;

import javax.swing.*;
import javax.swing.event.TreeModelEvent;
import javax.swing.event.TreeModelListener;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;
import javax.swing.tree.DefaultTreeCellRenderer;
import java.awt.*;
import java.awt.event.*;

/**
 * Title:        uifactory
 * Description:  This is a program design tool that creates Abacus java applications using the UI Factory.  The output of this tool is an xml document that is processed by the UI factory to create the java programs.
 * Copyright:    Copyright (c) 2001
 * Company:      Abacus Research
 * @author Michael Gouker (Cagey Logic)
 * @version 1.0
 */

class ObjectTreeViewMouseListener implements MouseListener {
    ObjectTreeViewPane theObjectTreeViewPane;

    public ObjectTreeViewMouseListener(ObjectTreeViewPane objObjectTreeViewPane) {
        theObjectTreeViewPane = objObjectTreeViewPane;
    }

    public void mouseClicked(MouseEvent evt) {
    }

    public void mouseEntered(MouseEvent evt) {
    }

    public void mouseExited(MouseEvent evt) {
    }

    public void mousePressed(MouseEvent evt) {
        showPopup(evt);
    }

    public void mouseReleased(MouseEvent evt) {
        showPopup(evt);
    }

    private void showPopup(MouseEvent evt) {
        if (evt.isPopupTrigger()) {
            theObjectTreeViewPane.initPopupMenu(theObjectTreeViewPane.theDesignCockpit);
            theObjectTreeViewPane.theDesignCockpit.theLastPopupMenu = theObjectTreeViewPane.thePopupMenu;
            int iRow = theObjectTreeViewPane.theTree.getRowForLocation(evt.getX(), evt.getY());
            if (iRow > 0) {
                ObjectTreeViewNode theSelectedNode = (ObjectTreeViewNode) theObjectTreeViewPane.theTree.getLastSelectedPathComponent();
                if (theSelectedNode != null) {
                    MetaObject theMetaObject = theSelectedNode.getMetaObject();
                    theObjectTreeViewPane.thePopupMenu.show(theObjectTreeViewPane,
                            evt.getX(), evt.getY(), theMetaObject);
                    JViewport theViewport = theObjectTreeViewPane.theScrollPane.getViewport();
                    Point ptTopLeft = theViewport.getViewPosition();
                    Point ptThisObjectOrigin = theObjectTreeViewPane.getLocationOnScreen();
                    theObjectTreeViewPane.thePopupMenu.setLocation(evt.getX(),
                            evt.getY() - (int) ptTopLeft.getY() + (int) ptThisObjectOrigin.getY());
                }
            }
        }
    }
}

public class ObjectTreeViewPane extends JAPanel {
    protected ObjectTreeViewNode theRootNode;
    protected DefaultTreeModel theTreeModel;
    protected ObjectTreeView theTree;
    public SuperDesignCockpit theDesignCockpit = null;
    public IDEObjectPopupMenu thePopupMenu = null;
    public JAScrollPane theScrollPane;

    public ObjectTreeViewPane(SuperDesignCockpit objDesignCockpit) {
        theRootNode = new ObjectTreeViewNode("Program");
        theTreeModel = new DefaultTreeModel(theRootNode);
        theTreeModel.addTreeModelListener(new ObjectTreeViewModelListener(objDesignCockpit));
        theDesignCockpit = objDesignCockpit;
        theTree = new ObjectTreeView(theTreeModel, theDesignCockpit);
        theTree.setEditable(false);
        theTree.getSelectionModel().setSelectionMode
                (TreeSelectionModel.SINGLE_TREE_SELECTION);
        theTree.setShowsRootHandles(true);
        theTree.addTreeSelectionListener(new ObjectTreeViewSelectionListener());
        theTree.addKeyListener(new ObjectTreeViewKeyListener(this));
        theTree.addMouseListener(new ObjectTreeViewMouseListener(this));
        theScrollPane = new JAScrollPane(theTree);

        //#ALEX, Small UI mod
        //setLayout(new GridLayout(1, 0));
        setLayout( new BorderLayout() );

        String sPaneTitle = theDesignCockpit.theLanguageManager.getMessage("ObjectPane.label", "Object Hierarchy");
        add( new IDETitleBar ( sPaneTitle ), BorderLayout.NORTH );
        add(theScrollPane, BorderLayout.CENTER );
        theScrollPane.setBorder ( null);
        setBorder ( BorderFactory.createMatteBorder ( 0,1,1,1, Color.GRAY  ));

        add(theScrollPane);
    }

    public void initPopupMenu(SuperDesignCockpit theDesignCockpit) {
        thePopupMenu = new IDEObjectPopupMenu(theDesignCockpit);
    }

    /** Remove all nodes except the root node. */
    public void clear() {
        theRootNode.removeAllChildren();
        theTreeModel.reload();
    }

    /** Remove the currently selected node. */
    public void removeCurrentNode() {
        TreePath currentSelection = theTree.getSelectionPath();
        if (currentSelection != null) {
            ObjectTreeViewNode currentNode = (ObjectTreeViewNode)
                    (currentSelection.getLastPathComponent());
            MetaObject theMetaObject = currentNode.getMetaObject();
            theMetaObject.DeleteObject(theDesignCockpit.theCurrentDesignProject.theMetaProject, theDesignCockpit);
            ObjectTreeViewNode parent = (ObjectTreeViewNode) (currentNode.getParent());
            if (parent != null) {
                theTreeModel.removeNodeFromParent(currentNode);
                return;
            }
        }
    }

    public void removeNode(ObjectTreeViewNode nodeToRemove) {
        MetaObject theMetaObject = nodeToRemove.getMetaObject();
        theMetaObject.DeleteObject(theDesignCockpit.theCurrentDesignProject.theMetaProject, theDesignCockpit);
        ObjectTreeViewNode parent = (ObjectTreeViewNode) (nodeToRemove.getParent());
        if (parent != null)
            theTreeModel.removeNodeFromParent(nodeToRemove);
    }

    public boolean removeObject(MetaObject theMetaObject) {
        ObjectTreeViewNode nodeToRemove = theMetaObject.getTreeNode();
        if (nodeToRemove == null)
            return false;
        ObjectTreeViewNode parent = (ObjectTreeViewNode) (nodeToRemove.getParent());
        if (parent != null)
            theTreeModel.removeNodeFromParent(nodeToRemove);
        else
            return false;
        return true;
    }

    /** Add child to the currently selected node. */
    public ObjectTreeViewNode addObject(String theNodeText) {
        ObjectTreeViewNode parentNode = null;
        TreePath parentPath = theTree.getSelectionPath();

        if (parentPath == null) {
            parentNode = theRootNode;
        } else {
            parentNode = (ObjectTreeViewNode)
                    (parentPath.getLastPathComponent());
        }

        return addObject(parentNode, theNodeText, true);
    }

    public ObjectTreeViewNode addObject(ObjectTreeViewNode parent, String theNodeText) {
        return addObject(parent, theNodeText, false);
    }

    public ObjectTreeViewNode addObject(ObjectTreeViewNode parent,
                                        String theNodeText,
                                        boolean shouldBeVisible) {
        ObjectTreeViewNode childNode =
                new ObjectTreeViewNode(theNodeText);

        if (parent == null) {
            parent = theRootNode;
        }

        theTreeModel.insertNodeInto(childNode, parent,
                parent.getChildCount());

        // Make sure the user can see the lovely new node.
        if (shouldBeVisible) {
            theTree.scrollPathToVisible(new TreePath(childNode.getPath()));
        }
        return childNode;
    }

    protected boolean load(MetaObject theFirstObject, ObjectTreeViewNode thisNode) {
        // Add an entry in the tree view for every design object in the project.
        MetaObject currObj = theFirstObject;
        while (currObj != null) {
            ObjectTreeViewNode theNewNode = new ObjectTreeViewNode(currObj);
            if (thisNode == null)
                theTreeModel.insertNodeInto(theNewNode, theRootNode,
                        theRootNode.getChildCount());
            else
                theTreeModel.insertNodeInto(theNewNode, thisNode,
                        thisNode.getChildCount());

            currObj.setTreeNode(theNewNode);
            if (currObj.theFirstChild != null)
                load(currObj.theFirstChild, theNewNode);
            currObj = currObj.theNextObject;
        }

        if (this.theDesignCockpit.isCustomizing())
            theTree.expandAll(); // this would be a nice config item to add as time becomes available

        return true;
    }

    public void moveDown() {
        ObjectTreeViewNode theSelectedNode = (ObjectTreeViewNode) theTree.getLastSelectedPathComponent();

        if (theSelectedNode.getMetaObject().isTabPage()) { // we must prevent tabpages moving!
            return;
        }

        ObjectTreeViewNode theNextNode = (ObjectTreeViewNode) theSelectedNode.getNextSibling();
        // Change these around.
        if (theNextNode != null) {
            ObjectTreeViewNode theParent = (ObjectTreeViewNode) theSelectedNode.getParent();
            theTree.theModel.removeNodeFromParent(theNextNode);
            theTree.theModel.insertNodeInto(theNextNode,
                    theParent,
                    theParent.getIndex(theSelectedNode));
            //theTree.updateUI();  SWB: removed as it wasn't needed, and even caused an Exception!
            // Fix the metaobjects.
            MetaObject objParent = theParent.getMetaObject();
            MetaObject objChild1 = theNextNode.getMetaObject();
            MetaObject objChild2 = theSelectedNode.getMetaObject();
            objParent.exchangeChildren(objChild1, objChild2);
        }
    }

    public void moveUp() {
        ObjectTreeViewNode theSelectedNode = (ObjectTreeViewNode) theTree.getLastSelectedPathComponent();

        if (theSelectedNode.getMetaObject().isTabPage()) { // we must prevent tabpages moving!
            return;
        }

        ObjectTreeViewNode thePreviousNode = (ObjectTreeViewNode) theSelectedNode.getPreviousSibling();
        // Change these around.
        if (thePreviousNode != null) {
            ObjectTreeViewNode theParent = (ObjectTreeViewNode) theSelectedNode.getParent();
            theTree.theModel.removeNodeFromParent(theSelectedNode);
            theTree.theModel.insertNodeInto(theSelectedNode,
                    theParent,
                    theParent.getIndex(thePreviousNode));
            TreePath path = new TreePath(theSelectedNode.getPath());
            if (path != null)
                theTree.setSelectionPath(path);
            //theTree.updateUI(); SWB: removed as it wasn't needed, and even caused an Exception!
            // Fix the metaobjects.
            MetaObject objParent = theParent.getMetaObject();
            MetaObject objChild1 = thePreviousNode.getMetaObject();
            MetaObject objChild2 = theSelectedNode.getMetaObject();
            objParent.exchangeChildren(objChild1, objChild2);
        }
    }

    public void addObjects(MetaObject currObj) {
        addObject(currObj, currObj.theParentObject);
        currObj = currObj.theFirstChild;
        while (currObj != null) {
            addObjects(currObj);
            currObj = currObj.theNextObject;
        }
    }

    public void addObject(MetaObject currObj, MetaObject theParent) {
        if (currObj != null) {
            if (theParent == null)
                theParent = theDesignCockpit.getSelectedObject();
            // Can only add nodes to container types.
            if ((theParent == null) || (theParent.theClass.getMetadata().containerType.getContainerType() == MetaContainerType.CONTAINER_NONE)) {
                theParent = theDesignCockpit.getSelectedFrame();
            }
            // Make sure that the object we are inserting into isn't the same as the object we
            // are inserting.  This can happen if the selected frame is the top object.
            if ((theParent != null) && (theParent.equals(currObj)))
                theParent = null;
            ObjectTreeViewNode theNewNode = new ObjectTreeViewNode(currObj);
            currObj.setTreeNode(theNewNode);
            if (theParent != null) {
                ObjectTreeViewNode theParentNode = theParent.getTreeNode();
                theTreeModel.insertNodeInto(theNewNode, theParentNode,
                        theParentNode.getChildCount());
            } else {
                theTreeModel.insertNodeInto(theNewNode, theRootNode,
                        theRootNode.getChildCount());
                // Select this new frame automatically.
                if (currObj != null) {
                    if ((currObj.theClass.getMetadata().containerType.getContainerType() == MetaContainerType.CONTAINER_FRAME) ||
                            (currObj.theClass.getMetadata().containerType.getContainerType() == MetaContainerType.CONTAINER_ABALET) ||
                            (currObj.theClass.getMetadata().containerType.getContainerType() == MetaContainerType.CONTAINER_TABBEDPANE))
                        theDesignCockpit.SelectContainer(currObj);
                    theDesignCockpit.SelectObject(currObj, false);
                }
            }
            theTree.setFolderIcons();
        }
    }

    public void redisplayProject() {
        clear();
        final MetaProject metaProject = theDesignCockpit.theCurrentDesignProject.theMetaProject;
        load(metaProject.getFirstObject(), null);
    }
}


class ObjectTreeViewKeyListener extends KeyAdapter {
    ObjectTreeViewPane theTreePane = null;

    public ObjectTreeViewKeyListener(ObjectTreeViewPane objTreePane) {
        super();
        theTreePane = objTreePane;
    }

    public void keyPressed(KeyEvent e) {
        if (e.getKeyCode() == KeyEvent.VK_DELETE) {
            //ObjectTreeViewNode theSelectedNode = (ObjectTreeViewNode) theTreePane.theTree.getLastSelectedPathComponent();
            //theTreePane.removeNode(theSelectedNode);
            //theTreePane.theDesignCockpit.theViewDisplayer.updateUI();

            theTreePane.theTree.theCockpit.theLastPopupMenu = theTreePane.theTree.theCockpit.theObjectTreeViewPane.thePopupMenu;
            theTreePane.theTree.theCockpit.theLastPopupMenu.cutSelectedObjectAction.actionPerformed(new ActionEvent(this, 0, null));

        }
        if (e.getKeyCode() == KeyEvent.VK_UP) {
            if (e.isShiftDown()) {
                theTreePane.moveUp();
                e.consume();
            }
        }
        if (e.getKeyCode() == KeyEvent.VK_DOWN) {
            if (e.isShiftDown()) {
                theTreePane.moveDown();
                e.consume();
            }
        }
    }

}


