/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
 * Created by IntelliJ IDEA.
 * User: Stuart Booth (Live Clue, inc)
 * Date: Jul 16, 2005
 * Time: 1:05:37 PM
 */
package ch.abacus.designcockpit.ide.propertyinspector;

import ch.abacus.lib.ui.renderer.common.*;
import ch.abacus.lib.ui.propertyinspector.core.Property;
import ch.abacus.lib.ui.propertyinspector.core.Group;
import ch.abacus.lib.ui.propertyinspector.core.Groups;
import ch.abacus.lib.ui.propertyinspector.display.PropertyEditorInterface;
import ch.abacus.lib.ui.propertyinspector.display.PropertyEditorEvent;
import ch.abacus.lib.ui.propertyinspector.display.PropertyValueEditorInterface;
import ch.abacus.designcockpit.ide.SuperDesignCockpit;
import ch.abacus.designcockpit.ide.DesignProject;
import ch.abacus.designcockpit.ide.IDEPropertyInspectorController;

import javax.swing.*;
import javax.swing.table.TableCellRenderer;
import java.util.HashMap;
import java.awt.*;

public class PropertyFactory {
    private HashMap<SimpleProperty, Property> m_properties;
    private HashMap<String, Group> m_groups;
    private PropertyRegistry m_propertyRegistry;
    private final IDEPropertyInspectorController m_idePropertyInspectorController;
    private final SuperDesignCockpit m_cockpit;
    private MyPropertyEditorInterface m_myPropertyEditorInterface;

    public PropertyFactory(PropertyRegistry propertyRegistry, IDEPropertyInspectorController idePropertyInspectorController) {
        m_propertyRegistry = propertyRegistry;
        m_idePropertyInspectorController = idePropertyInspectorController;
        m_cockpit = idePropertyInspectorController.getSuperDesignCockpit();
        m_properties = new HashMap<SimpleProperty, Property>();
        m_groups = new HashMap<String, Group>();
        m_myPropertyEditorInterface = new MyPropertyEditorInterface();
    }

    private Property createObjectReferenceProperty(String sFullName, String sName, String sHint, String sGroup,
     String sType, String sConstants, boolean bSingleSelection, boolean bReadOnly, boolean bHelper, boolean bObjectRef)
    {

       SimpleProperty simpleProperty = new SimpleProperty(sName, sType, null);
       Property property = m_properties.get(simpleProperty);
       Group group = getGroup(sGroup);
       final PropertyValueEditorInterface editor = m_propertyRegistry.getEditor(sName, null);

       if(property==null)
       {
           property = new Property(PropertyIds.Dummy,
               sName,
               sHint,
               group,
               m_propertyRegistry.getRenderer(sType, sConstants),
               editor,
               m_myPropertyEditorInterface);

            property.setData(sFullName);
            m_properties.put(simpleProperty, property);
       }

       return property;
    }

    private Property createObjectDataAwareProperty(String sFullName, String sName, String sHint, String sGroup,
     String sType, String sConstants, boolean bSingleSelection, boolean bReadOnly, boolean bHelper, boolean bObjectDataAware)
    {

       SimpleProperty simpleProperty = new SimpleProperty(sName, sType, null);
       Property property = m_properties.get(simpleProperty);
       Group group = getGroup(sGroup);
       final PropertyValueEditorInterface editor = m_propertyRegistry.getEditor(sName, null);

       if(property==null)
       {
           property = new Property(PropertyIds.Dummy,
               sName,
               sHint,
               group,
               m_propertyRegistry.getRenderer(sType, sConstants),
               editor,
               m_myPropertyEditorInterface);

           if (bSingleSelection)
               property.setSingleSelection(true);

            property.setData(sFullName);
            m_properties.put(simpleProperty, property);
       }

       return property;
    }

    private Property getProperty(String sFullName, String sName, String sHint, String sGroup, String sType, String sConstants, boolean bSingleSelection, boolean bReadOnly, boolean bHelper) {
        SimpleProperty simpleProperty = new SimpleProperty(sFullName, sType, sConstants );
        Property property = m_properties.get(simpleProperty);
        if (property == null) {
            //System.out.println("Add: " +simpleProperty + " " + simpleProperty.hashCode() + " " + m_properties.size());
            Group group = getGroup(sGroup);

            final PropertyValueEditorInterface editor = m_propertyRegistry.getEditor(sType, sConstants);
            property = new Property(PropertyIds.Dummy,
                sName,
                sHint,
                group,
                m_propertyRegistry.getRenderer(sType, sConstants),
                editor,
                m_myPropertyEditorInterface);

            if (bSingleSelection)
                property.setSingleSelection(true);

            if (bReadOnly) {
                // For the Customizer, we want to make read-only properties only disable the edit,
                //  if a helper is associated - the button and label are still enabled.
                if (m_cockpit.isCustomizing() && bHelper)
                    property.setValueEditorReadOnly(true);
                else
                    property.setReadOnly(true);
            }

            property.setData(sFullName);

            m_properties.put(simpleProperty, property);
        }

        return property;
    }

    public Property getProperty(MetaPropertyDescriptor metaPropertyDescriptor) {
        String sFullName = getFullPropertyName(metaPropertyDescriptor);
        String sName = getPropertyName(metaPropertyDescriptor);
        String sGroup = getGroupFromName(sFullName);
        String sHint = null;
        String sType = metaPropertyDescriptor.getClassName();
        String sConstants = metaPropertyDescriptor.getIncludeTypeConstants();
        boolean bSingleSelect = sName.equalsIgnoreCase("Name");
        boolean bReadOnly = metaPropertyDescriptor.isReadOnly();
        boolean bObjectRef = metaPropertyDescriptor.bObjectReference; // Change to func
        boolean bObjectPromptList = metaPropertyDescriptor.bObjectPromptList; // Change to func
        boolean bDataAwarePromptList = metaPropertyDescriptor.getDataAwareness();//

        final String sHelperClass = metaPropertyDescriptor.getHelperClass();
        boolean bHelper = (sHelperClass != null  && sHelperClass.length() >0);

        Property passProp= null;

        if(metaPropertyDescriptor.bObjectReference==true)
            passProp= createObjectReferenceProperty(sFullName, sName, sHint, sGroup,
                    sType, sConstants, bSingleSelect, bReadOnly, bHelper, bObjectRef);
        else if(metaPropertyDescriptor.getDataAwareness()==true)
            passProp= createObjectDataAwareProperty(sFullName, sName, sHint, sGroup,
                    sType, sConstants, bSingleSelect, bReadOnly, bHelper, bDataAwarePromptList);
        else
            passProp= getProperty(sFullName, sName, sHint, sGroup, sType, sConstants, bSingleSelect, bReadOnly, bHelper);

        return passProp;
    }

    private String getGroupFromName(String sName) {
        String sGroup = "General";
        final int iIndex = sName.lastIndexOf('.');

        if (iIndex > 0) {
            sGroup = sName.substring(0, iIndex);
        }

        return sGroup;
    }

    /**
     * Returns the fully-qualified name of the Property. For example, Location.Top for a sub-property
     * @param metaPropertyDescriptor
     * @return
     */
    private String getFullPropertyName (MetaPropertyDescriptor metaPropertyDescriptor) {
        String sName = metaPropertyDescriptor.getName();
        while ((metaPropertyDescriptor = metaPropertyDescriptor.theSuperProperty) != null) {
            sName = metaPropertyDescriptor.getName() + "." + sName;
        }

        return sName;
    }

    private String getPropertyName (MetaPropertyDescriptor metaPropertyDescriptor) {
        String sName = metaPropertyDescriptor.getName();

        return sName;
    }


    private Group getGroup(String sGroup) {
        Group group = m_groups.get(sGroup);
        if (group == null) {
            // may be a group, or a sub-group...
            String[] sGroups = sGroup.split("\\.");

            if (sGroups.length == 1) {
                // simple group...
                group = new Group(null, sGroup, null, null);
                m_groups.put(sGroup, group);

            } else {
                // contains sub-group(s)...
                Group parentGroup = getGroup(sGroups[0]);
                Groups groups = parentGroup.getGroups();
                for (int iGroup = 0; iGroup < groups.size(); iGroup ++) {
                    final Group childGroup = groups.get(iGroup);
                    if (childGroup.getName().equals(sGroups[1]) ) {
                        return childGroup;
                    }
                }
                // subgroup doesn't exist...
                group = new Group(parentGroup, sGroups[1], null, null);

            }

        }
        return group;
    }

    static enum PropertyIds {
        Dummy
    };

    private class MyPropertyEditorInterface implements PropertyEditorInterface {

        public void edit(PropertyEditorEvent event) {
            final DesignProject designProject = m_cockpit.getDesignProject();
            final Property property = event.getDisplayProperty().getProperty();
            //System.out.println("Edit: " + property);
            MetaObject mo = designProject.getSelectedObject();
            m_idePropertyInspectorController.showHelper(mo, (String) property.getData());
        }

        public boolean isVisible(Property property) {
            //System.out.println("Is Visible: " +property.getName());
            final DesignProject designProject = m_cockpit.getDesignProject();
            if (designProject.getSelectedObjectCount() > 1)
                return false;

            final MetaObject metaObject = designProject.getSelectedObject();
            if (metaObject != null) {
                MetaPropertyDescriptor theProp = metaObject.theClass.getProperty((String) property.getData(), true);
                return (theProp.getHelperClass() != null && theProp.getHelperClass().length() >0);                
            }


            return false;
        }

    }

    /**
     * Used to compare logical property values to make a physical Property
     */
    private class SimpleProperty {
        private String m_name;
        private String m_type;
        private String m_constants;

        public SimpleProperty(String sName, String sType, String sConstants) {
            m_name = sName;
            m_type = sType;
            m_constants = sConstants;
        }

        public boolean equals(Object obj) {
            if (this == obj)
                return true;

            if (obj == null)
                return false;

            if (getClass() != obj.getClass())
                return false;

            SimpleProperty other = (SimpleProperty) obj;

            boolean bSame = m_name.equals(other.m_name) && m_type.equals(other.m_type);
            if (bSame) {
                if (m_constants == null)
                    bSame = (other.m_constants == null);
                else
                    bSame = m_constants.equals(other.m_constants);
            }
            return bSame;
        }

        public int hashCode() {
            int iHashCode = 7 * m_name.hashCode() + 11 * m_type.hashCode();
            if (m_constants != null)
                iHashCode += 13 * m_constants.hashCode();

            return iHashCode;
        }

        public String toString() {
            return m_name + " " + m_type + " " + m_constants;
        }


    }
}


