/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package ch.abacus.designcockpit.ide.propertyinspector;

import ch.abacus.designcockpit.ide.SuperDesignCockpit;
import ch.abacus.lib.ui.propertyinspector.core.StringEntry;
import ch.abacus.lib.ui.renderer.common.*;
import ch.abacus.lib.ui.propertyinspector.display.*;
import ch.abacus.lib.ui.propertyinspector.display.editor.BooleanEditor;
import ch.abacus.lib.ui.propertyinspector.display.editor.EditableStringListEditor;
import ch.abacus.lib.ui.propertyinspector.display.renderer.StringRenderer;
import ch.abacus.lib.ui.propertyinspector.display.renderer.ColorIcon;

import java.util.ArrayList;
import java.awt.*;

/**
 * Created by IntelliJ IDEA.
 * User: Stuart Booth (Live Clue, inc)
 * Date: Jul 16, 2005
 * Time: 9:58:00 PM
 */
public class PropertyRegistry {
    private final PropertyInspector m_propertyInspector;
    private final SuperDesignCockpit m_superDesignCockpit;
    private PropertyRendererRegistry m_rendererRegistry;
    private PropertyEditorRegistry m_editorRegistry;
    private ConstantListFactory m_constantListFactory;

    public PropertyRegistry(PropertyInspector propertyInspector, SuperDesignCockpit superDesignCockpit) {
        m_propertyInspector = propertyInspector;
        m_superDesignCockpit = superDesignCockpit;

        m_rendererRegistry = m_propertyInspector.getRendererRegistry();
        m_editorRegistry = m_propertyInspector.getEditorRegistry();

        m_constantListFactory = new ConstantListFactory(superDesignCockpit.getMetadataDispenser());

        registerRenderers();
        registerEditors();

        m_propertyInspector.initialize();
    }

    private void registerRenderers() {

//        /**
//         * Define a few values and displaynames for a combobox editor..
//         */

        m_rendererRegistry.register("boolean", new DCBooleanRenderer());
        m_rendererRegistry.register("-default-", new StringRenderer());
    }

    public void registerColorRenderer(String sConstants) {
        final String sRendererName = "java.awt.Color-" + sConstants;

        final PropertyRendererInterface propertyRendererInterface = m_rendererRegistry.get(sRendererName);
        if (propertyRendererInterface == null)
            m_rendererRegistry.register(sRendererName, new ColorPropertyValueRenderer(m_superDesignCockpit, sConstants));
    }


    private void registerEditors() {
        m_editorRegistry.register("int", new MyIntegerEditor()); //IntegerEditor(m_globalInterface));
        m_editorRegistry.register("boolean", new BooleanEditor());
        m_editorRegistry.register("-default-", new MyStringEditor());
    }

    private void registerConstantListEditor(String sName, ArrayList list) {
        final String name = "ConstantList-" + sName;
        if (m_editorRegistry.get(name) == null) {
            DynamicListEditor baseStringListEditor = new DynamicListEditor(list);
            //final EditableStringListEditor baseStringListEditor = new EditableStringListEditor(list);
            baseStringListEditor.initialized();
            m_editorRegistry.register(name, baseStringListEditor);
        }

    }

    private void registerDataSourceList(String sName, ArrayList list) {
        final String name = sName;
        if (m_editorRegistry.get(name) == null) {
            final ObjecReferenceListEditor baseDatasourceList = new ObjecReferenceListEditor(false,list);
            baseDatasourceList.initialized();
            m_editorRegistry.register(name, baseDatasourceList);
        }

    }


    private void registerConstantListColorEditor(String sName, ArrayList list) {
        final String name = "ConstantList-" + sName;
        if (m_editorRegistry.get(name) == null) {

            ArrayList icons = new ArrayList();
            for (int iItem =0; iItem < list.size(); iItem++) {
                StringEntry se = (StringEntry) list.get(iItem);
                icons.add(new ColorIcon(getItemColor(sName, se.getTranslatedName())));
            }

            DynamicListEditor baseStringListEditor = new DynamicListEditor(false, list, icons);
            baseStringListEditor.initialized();
            m_editorRegistry.register(name, baseStringListEditor);
        }

    }

    public PropertyRendererInterface getRenderer(String sType, String sConstants) {
        if (sType.equals("java.awt.Color")) {
            if (sConstants != null) {
                registerColorRenderer(sConstants);
                sType += "-" + sConstants;
            }

        }
        PropertyRendererInterface renderer = m_rendererRegistry.get(sType);
        if (renderer == null)
            renderer = m_rendererRegistry.get("-default-");
        return renderer;
    }

    public PropertyValueEditorInterface getEditor(String sType, String sConstants) {

        boolean bIsDataSource = sType.equalsIgnoreCase("DataSource");
        boolean bIsdbField = (sType.equalsIgnoreCase("FieldName")
                    || sType.equalsIgnoreCase("OrderByField")) ? true : false;
        boolean bIsdbTables = sType.equalsIgnoreCase("dbTables") ? true : false;


        if (sConstants != null) {
            boolean bIsColor = sType.equals("java.awt.Color");
            sType = "ConstantList-" + sConstants;
            final ArrayList constantList = m_constantListFactory.getConstantList(sConstants);

            if (bIsColor)
                registerConstantListColorEditor(sConstants, constantList);
            else
                registerConstantListEditor(sConstants, constantList);
        }

        if(bIsDataSource)
        {
            sType = "-ObjReferenceList-" + sType;
            sConstants = sType;
            final ArrayList constantList = m_constantListFactory.getConstantList(sConstants);
            registerDataSourceList(sConstants, constantList);
        }
        else if(bIsdbTables)
        {
            sType = "-dbFieldNameList-" + sType;
            sConstants = sType;
            final ArrayList constantTableList = m_constantListFactory.getConstantList(sConstants);
            registerDataSourceList(sConstants, constantTableList);
        }
        else if(bIsdbField)
        {
            sType = "-dbFieldNameList-" + sType;
            sConstants = sType;
            final ArrayList constantList = m_constantListFactory.getConstantList(sConstants);
            registerDataSourceList(sConstants, constantList);
        }

        PropertyValueEditorInterface editor = m_editorRegistry.get(sType);
        if (editor == null)
            editor = m_editorRegistry.get("-default-");
        return editor;
    }

    Color getItemColor(String sConstantGroup, String avalue)
        {
            Object theResolvedValue=null;
            final MetadataDispenser metadataDispenser = m_superDesignCockpit.theMetadataDispenser;
            MetaConstantGroupCollection theConstantTableGroup =
                    metadataDispenser.findConstantGroupCollection(sConstantGroup);
//todo SWB Needed?        if ( avalue==null )
//            avalue = theProperty.getDefaultValue();
            Object nv = theConstantTableGroup.get(avalue);
            try
            {
                final HammerClassCustomLoader loader = m_superDesignCockpit.getClassLoader().getLoader();
                theResolvedValue = MetaConstantGroup.resolve(nv, metadataDispenser, loader);
            }
            catch (HammerException e) { e.printStackTrace(); }

            return (Color)theResolvedValue;
        }

     public PropertyEditorRegistry getEditorRegistry()
     {
         return m_editorRegistry;
     }
}
