/**
 * Title:        Comedia Beans
 * Module:       Tetris Game Bean
 * Copyright:    Copyright (c) 2001
 * Company:      Capella Development Group
 * @authors Gregory Grigorenko, Sergey Seroukhov
 * @version 1.0
 */

package org.comedia.game;

import javax.swing.*;
import java.awt.*;
import java.util.*;
import java.awt.event.*;

/**
 * Implements the Tetris game java bean.
 * This bean can be resized to fits the draw rectangle.
 * <p>
 * <center><img src="CTetris.gif"></center>
 * <p>
 * Tetris supports keyboard interaction.
 * <ul>
 * <li>LEFT - move left
 * <li>RIGHT - move right
 * <li>UP - rotate
 * <li>DOWN - move down
 * <li>SPACE - drop
 * <li>PAUSE - pause
 * </ul>
 */
public class CTetris extends JComponent implements Runnable, KeyListener,
  MouseListener {
  /**
   * The default inset size.
   */
  private final static int DEFAULT_INSET_SIZE = 10;

  /**
   * The default cell x number.
   */
  private final static int DEFAULT_X_CELL_NUMBER = 11;

  /**
   * The default cell y number.
   */
  private final static int DEFAULT_Y_CELL_NUMBER = 20;

  /**
   * The default background color for the game map.
   */
  private final static Color DEFAULT_BACK_COLOR = Color.lightGray;

  /**
   * The speedup constant
   */
  private static final int SPEEDUP = 50;

  /**
   * The minimim delay constant.
   */
  private static final int MIN_DELAY = 200;

  /**
   * The maximum delay constant.
   */
  private static final int MAX_DELAY = 1000;

  /**
   * The current cell x number.
   */
  private int xCellNum = DEFAULT_X_CELL_NUMBER;

  /**
   * The current cell y number.
   */
  private int yCellNum = DEFAULT_Y_CELL_NUMBER;

  /**
   * The layout metrics.
   */
  private int xInset;     // x inset width
  private int yInset;     // y inset height
  private int mapWidth;   // width of the map
  private int mapHeight;  // height of the map
  private int cellSide;   // cell side size
  private int scoreWidth; // width of score

  /**
   * The tetris components.
   */
  private CTetrisMap map = null;

  /**
   * The game variables.
   */
  private int score = 0;                // current score value
  private int figureCount = 0;          // current figires count
  private int lineCount = 0;            // current line count
  private int delay = MAX_DELAY;        // fall delay, 2 sec at start
  private Random random = new Random(); // random numbers generator
  private CFigure nextFigure = null;    // next figure
  private CFigure currentFigure = null; // current figure
  private Thread thread = null;         // current animation thread
  private boolean flick = true;
  private String message = "GAME OVER";
  private boolean active = false;
  private boolean over = true;

  /**
   * The table of high scores.
   */
  protected CHighScores highScores = new CHighScores();

  /**
   * Constructs this class with default properties.
   */
  public CTetris() {
    this.setRequestFocusEnabled(true);
    this.setOpaque(false);
    this.addKeyListener(this);
    this.addMouseListener(this);

    map = new CTetrisMap();
    thread = new Thread(this);
    thread.start();

//    startNewGame();
//    active = false;
  }

  /**
   * Runs the animation thread of this class.
   */
  public void run() {
    while (true) {
      try {
	thread.sleep(delay);
      } catch (Exception e) {};

      flick = !flick;

      if (active && !over)
        stepDown();
      else if (message.length() > 0 && flick)
        this.repaint();
    }
  }

  /**
   * Moves the current figure one row down.
   */
  synchronized void stepDown() {
    if (currentFigure == null)
      return;
    CFigure newFigure = new CFigure(currentFigure);
    newFigure.moveDown();
    if (!map.checkFit(newFigure))
      dropDown();
    else
      currentFigure = newFigure;
    this.repaint();
  }

  /**
   * Switches the new and current figures.
   */
  void switchFigure() {
    if (nextFigure == null) {
      nextFigure = new CFigure(xCellNum / 2 - 1, 0);
      currentFigure = new CFigure(xCellNum / 2 - 1, 0);
    } else {
      currentFigure = nextFigure;
      nextFigure = new CFigure(xCellNum / 2 - 1, 0);
    }
    if (!map.checkFit(currentFigure)) {
      currentFigure = null;
      gameOver();
    }
  }

  /**
   * Starts the new game.
   */
  public void startNewGame() {
    map = new CTetrisMap();

    figureCount = 0;
    lineCount = 0;
    score = 0;
    delay = MAX_DELAY;
    active = true;
    over = false;
    message = "";

    nextFigure = null;
    switchFigure();

    this.repaint();
  }

  /**
   * Drops the current figure down.
   */
  synchronized void dropDown() {
    do {
      currentFigure.moveDown();
    } while (map.checkFit(currentFigure));

    currentFigure.moveUp();
    map.addFigure(currentFigure);

    figureCount++;
    int deadCount = map.checkMap();
    lineCount += deadCount;

    if (delay > MIN_DELAY)
      delay -= SPEEDUP * deadCount;
    if (delay < MIN_DELAY)
      delay = MIN_DELAY;

    score = (lineCount * 50) + figureCount;

    switchFigure();
    this.repaint();

    if (!map.checkFit(currentFigure))
      gameOver();
  }

  /**
   * Finishes the game.
   */
  public void gameOver() {
    active = false;
    over = true;
    message = "GAME OVER";
    this.repaint();

    if (highScores.getMinimumScore() >= score && highScores.isFull()) {
      JOptionPane.showMessageDialog(null, "Your score is " + score,
        "Game Over", JOptionPane.INFORMATION_MESSAGE);
    } else {
      String player = JOptionPane.showInputDialog(null, "Congratulations!\n"
        + "You have reached " + score + " score", "Enter you name",
        JOptionPane.INFORMATION_MESSAGE);
      if (player != null)
        highScores.addScore(player.trim(), score);
    }

    startNewGame();
  }

  /**
   * Draws the next figure.
   * @param g the current graphics context.
   * @param x the x offset for the figure.
   * @param y the y offset for the figure.
   */
  private void drawNextFigure(Graphics g, int x, int y) {
    if (nextFigure != null)
      nextFigure.draw(g, x, y);
  }

  /**
   * Paints this class.
   * @param g the current graphics context.
   */
  public void paint(Graphics g) {
    Font font = findFont(g, "Score: 00000", scoreWidth, cellSide);
    g.setFont(font);
    int xScoreOffs = xInset + mapWidth + DEFAULT_INSET_SIZE;
    FontMetrics m = g.getFontMetrics(g.getFont());
    int fontHeight = m.getHeight() - m.getDescent() * 2;

    map.drawMap(g);
    g.setColor(Color.black);
    g.drawRect(xInset, yInset - 1, mapWidth + 1, mapHeight + 1);
    if (currentFigure != null)
      currentFigure.draw(g, currentFigure.column * cellSide + xInset,
        currentFigure.row * cellSide + yInset);

    if (message.length() > 0/* && flick*/) {
      int xOffs = (mapWidth - m.stringWidth(message)) / 2;
      int yOffs = (mapHeight - fontHeight) / 2;
      font = new Font(g.getFont().getName(), Font.BOLD, g.getFont().getSize());
      g.setColor(Color.white);
      g.drawString(message, xInset + xOffs + 1, yInset + yOffs + 1);
      g.setColor(Color.black);
      g.drawString(message, xInset + xOffs, yInset + yOffs);
    }

    drawNextFigure(g, xScoreOffs, yInset + fontHeight * 4);

    g.setColor(Color.black);
    g.drawString("Score: " + score, xScoreOffs, yInset + fontHeight);
    g.drawString("Next:", xScoreOffs, yInset + fontHeight * 3);

    int yScoreOffs = yInset + fontHeight * 10;
    g.drawString("High scores:", xScoreOffs, yScoreOffs);
    yScoreOffs += fontHeight / 2;
    for (int i = 0; i < highScores.getLength(); i++) {
      yScoreOffs += fontHeight;
      g.drawString(highScores.formatScore(i, 5), xScoreOffs, yScoreOffs);
    }
  }

  /**
   * Finds a font size which better fits the specified message to given sizes.
   * @param g current graphical context.
   * @param s string message.
   * @param w width of the rectangle.
   * @param h height of the rectangle.
   */
  private Font findFont(Graphics g, String s, int w, int h) {
    int size = 6;
    Font font = new Font(this.getFont().getName(), this.getFont().getStyle(), size);
    while (true) {
      FontMetrics m = g.getFontMetrics(font);
      if (m.getHeight() > h || m.stringWidth(s) > w)
        break;
      size++;
      font = new Font(this.getFont().getName(), this.getFont().getStyle(), size);
    }
    return font;
  }

  /**
   * Recount layout of all internal components.
   */
  public void doLayout() {
    Dimension d = this.getSize();
    Dimension dMap = new Dimension((int)(d.width * 0.7), d.height);
    Dimension dScore = new Dimension((int)(d.width * 0.3), d.height);

    xInset = DEFAULT_INSET_SIZE;
    mapWidth = dMap.width - xInset * 2;
    yInset = DEFAULT_INSET_SIZE;
    mapHeight = dMap.height - yInset * 2;
    scoreWidth = dScore.width - xInset;

    if (mapHeight > mapWidth / xCellNum * yCellNum) {
      cellSide = (int) (mapWidth / xCellNum);
      mapWidth = cellSide * xCellNum;
      mapHeight = cellSide * yCellNum;
      yInset = (int) ((dMap.height - mapHeight) / 2);
    } else {
      cellSide = (int) (mapHeight / yCellNum);
      mapWidth = cellSide * xCellNum;
      mapHeight = cellSide * yCellNum;
      xInset = (int) ((dMap.width - mapWidth) / 2);
    }

    this.repaint();
  }

  /**
   * Starts the games.
   */
  public void start() {
    active = true;
  }

  /**
   * Suspends the game.
   */
  public void stop() {
    active = false;
  }

  /**
   * Handles Key Typed event from user interface.
   * @param e an object which describes occured event.
   */
  public void keyTyped(KeyEvent e) {
  }

  /**
   * Handles Key Pressed event from user interface.
   * @param e an object which describes occured event.
   */
  public void keyPressed(KeyEvent e) {
    CFigure newFigure = new CFigure(currentFigure);

    if (e.getKeyCode() == KeyEvent.VK_PAUSE) {
      active = !active;
      if (active) message = "";
      else message = "PAUSED";
      this.repaint();
      return;
    }

    if (active && !over) {
      switch(e.getKeyCode()) {
        case KeyEvent.VK_LEFT :
          newFigure.moveLeft();
          break;
        case KeyEvent.VK_RIGHT :
          newFigure.moveRight();
          break;
        case KeyEvent.VK_DOWN :
          stepDown();
          return;
        case KeyEvent.VK_UP :
          newFigure.rotate();
          break;
        case KeyEvent.VK_SPACE :
          dropDown();
          return;
        default:
          return;
      }

      if (map.checkFit(newFigure))
        currentFigure = newFigure;
      this.repaint();
    }
  }

  /**
   * Handles Mouse Clicked event from user interface.
   * @param e an object which describes occured event.
   */
  public void mouseClicked(MouseEvent e) {
  }

  /**
   * Handles Mouse Pressed event from user interface.
   * @param e an object which describes occured event.
   */
  public void mousePressed(MouseEvent e) {
    this.requestFocus();
  }

  /**
   * Handles Mouse Released event from user interface.
   * @param e an object which describes occured event.
   */
  public void mouseReleased(MouseEvent e) {
  }

  /**
   * Handles Mouse Clicked Entered event from user interface.
   * @param e an object which describes occured event.
   */
  public void mouseEntered(MouseEvent e) {
  }

  /**
   * Handles Mouse Exited event from user interface.
   * @param e an object which describes occured event.
   */
  public void mouseExited(MouseEvent e) {
  }

  /**
   * Handles Key Released event from user interface.
   * @param e an object which describes occured event.
   */
  public void keyReleased(KeyEvent e) {
  }

  /**
   * Draws a color box on the screen.
   * @param g a current graphic context.
   * @param c a box color.
   * @param x x offset.
   * @param y y offset.
   */
  void drawBox(Graphics g, Color c, int x, int y) {
    int rx = x + cellSide - 1;
    int ry = y + cellSide - 1;

    if (c == DEFAULT_BACK_COLOR) {
      g.setColor(DEFAULT_BACK_COLOR);
      g.fillRect(x, y, cellSide, cellSide);
      g.setColor(Color.gray);
      g.drawLine(x, y, x, ry);
    } else {
      g.setColor(c);
      g.fillRect(x, y, cellSide, cellSide);
      g.setColor(Color.black);
      g.drawLine(x, ry, rx, ry);
      g.drawLine(rx, y, rx, ry);
      g.setColor(Color.white);
      g.drawLine(x, y, rx, y);
      g.drawLine(x, y, x, ry);
    }
  }

  /**
   * Presents a tetris figure.
   */
  private class CFigure {
    String[] figureShapes = {
      "@@@|@.", "@@| @| @.", "  @|@@@.", "@ |@ |@@.",
      "@@@|  @.", " @| @|@@.", "@  |@@@.", "@@|@ |@.",
      "@@ | @@.", " @|@@|@.", " @@|@@.", "@ |@@| @.",
      "@@@@.", "@|@|@|@.", "@@@| @.", " @|@@| @.",
      " @ |@@@.", "@ |@@|@.", "@@|@@.", "@@@|@  |@@@.",
      "@@@|@ @|@ @.", "@@@|  @|@@@.", "@ @|@ @|@@@.",
      "@ @|@ @|@@@.", "@@@|@  |@@@.", "@@@|@ @|@ @.",
      "@@@|  @|@@@.", "@@|@@|@ @.", "@@@| @@|@.",
      "@ @| @@| @@.", "  @|@@|@@@.", "@@@|@@|@@@.",
      "@@@|@@@|@ @.","@@@| @@|@@@.","@ @|@@@|@@@."
    };

    int[] baseShapes = {0,4,8,10,12,14,18,19,23,27,31};

    int[] maxShapes =  {3,7,9,11,13,17,18,22,26,30,34 };

    Color[] figureColors = {Color.green, Color.blue, Color.cyan,
      Color.orange, Color.pink, Color.yellow, Color.magenta, Color.red,
      Color.red,Color.red,Color.red
    };

    int baseShape;      // base shape index in FigShape
    int maxShape;       // max shape index in FigShape
    int currentShape;   // current figure shape
    int figureIndex;    // Index of the figure (in Base-, MaxShapes)
    int row;            // line (0 = base)
    int column;         // column (0 = base)
    Color figureColor;  // Color

    /**
     * Draws this figure on the screen.
     * @param g the current graphic context.
     * @param x x coordinate
     * @param y y coordinate
     */
    public void draw(Graphics g, int x, int y)  {
      String Shape = figureShapes[currentShape];
      int index = 0;
      int column = x;
      char ch;

      while ((ch = Shape.charAt(index++)) != '.') {
        if (ch == '@')
          drawBox(g, figureColor, x, y);
        x += cellSide;
        if (ch == '|') {
          y += cellSide;
          x = column;
        }
      }
    }

    /**
     * Constructs a figure and assignes a main properties.
     * @param column an initial column for this figure.
     * @param row an initial row for this figure.
     * @param figureIndex an index of figure scheme.
     */
    public CFigure(int column, int row, int figureIndex){
      this.column = column;
      this.row = row;
      this.figureIndex = figureIndex;
      this.baseShape = baseShapes[figureIndex];
      this.maxShape = maxShapes[figureIndex];
      this.currentShape = baseShape;
      this.figureColor = figureColors[figureIndex];
    }

    /**
     * Constructs a random figure.
     * @param column an initial column for this figure.
     * @param row an initial row for this figure.
     */
    public CFigure(int column, int row) {
      this(column, row, Math.abs(random.nextInt()) % (11 - 4));
    }

    /**
     * Construct an empty figure and copies a specified figure instance.
     * @param figure a copy instance of figure.
     */
    public CFigure(CFigure figure) {
      this(figure.column, figure.row, figure.figureIndex);
      currentShape = figure.currentShape;
    }

    /**
     * Moves this figure to the left.
     */
    void moveLeft() {
      column--;
    }

    /**
     * Moves this figure to the right.
     */
    void moveRight() {
      column++;
    }

    /**
     * Moves this figure to the down.
     */
    void moveDown() {
      row++;
    }

    /**
     * Moves this figure to the up.
     */
    void moveUp() {
      row--;
    }

    /**
     * Rotates this figure.
     */
    void rotate() {
      currentShape++;
      if (currentShape > maxShape)
        currentShape = baseShape;
    }
  }

  /**
   * Presents the game map for Tetris.
   */
  private class CTetrisMap {
    /**
     * The map content cells;
     */
    Color[][] cells = null;

    /**
     * Constructs this map with default parameters.
     */
    public CTetrisMap() {
      cells = new Color[yCellNum][xCellNum];
      clear();
    }

    /**
     * Clears this map class.
     */
    public void clear() {
      for (int r = 0; r < yCellNum; r++)
        for (int c = 0; c < xCellNum; c++)
          cells[r][c] = DEFAULT_BACK_COLOR;
    }

    /**
     * Draws the game map.
     * @param g the current graphics context.
     */
    public void drawMap(Graphics g) {
      int x = xInset;
      int y = yInset;

      for (int r = 0; r < yCellNum; r++ ) {
        for (int c = 0; c < xCellNum; c++ ) {
          drawBox(g, cells[r][c], x, y);
          x += cellSide;
        }
        y += cellSide;
        x = xInset;
      }
    }

    /**
     * Checks is specified fugure fits to this map.
     * @param figure a figure to check.
     * @result <code>TRUE</code> if figure fits to this map and
     *   <code>FALSE</code> otherwise.
     */
    public boolean checkFit(CFigure figure) {
      String Shape = figure.figureShapes[figure.currentShape];
      int i = 0;
      int column = figure.column;
      int row = figure.row;
      char ch;
      while ((ch = Shape.charAt(i++)) != '.') {
        if (ch == '@') {
          if ((column < 0) || (row < 0) || (column >= xCellNum) || (row >= yCellNum) ||
            (cells[row][column] != DEFAULT_BACK_COLOR))
            return false;
        }
        column++;
        if (ch == '|') {
          row++;
          column = figure.column;
        }
      }
      return true;
    }

    /**
     * Removes the specified row from this map.
     * @param row a row number to remove from this map.
     */
    private void removeRow(int row) {
      do {
        java.lang.System.arraycopy(cells[row-1], 0, cells[row], 0,
          cells[row-1].length);
      } while (--row > 0);
    }

    /**
     * Checks the content of this map and removes filled rows.
     * @resut a number of removed rows.
     */
    public int checkMap() {
      int row = 1;
      int result = 0;

      do {
        int column = 0;
        while (column < xCellNum) {
          if (cells[row][column] == DEFAULT_BACK_COLOR)
            break;
          column++;
        }
        if (column == xCellNum) {
          removeRow(row);
          result++;
        }
      } while (++row < yCellNum);
      return result;
    }

    /**
     * Adds a figure to this map.
     * @param figure a figure to add to this map.
     */
    public void addFigure(CFigure figure) {
      String shape = figure.figureShapes[figure.currentShape];
      int index = 0;
      int column = figure.column;
      int row = figure.row;
      char ch;

      while ((ch = shape.charAt(index++)) != '.') {
        if (ch == '@')
          cells[row][column] = figure.figureColor;
        column++;
        if (ch == '|') {
          row++;
          column = figure.column;
        }
      }
    }

  }

  /**
   * Implements a hi score table item.
   */
  protected class CHiScore {
    public String player = "";
    public int score = 0;
  }

  /**
   * Runs this Tetris game as a standalone application.
   * @param args command line arguments.
   */
  public static void main(String[] args) {
    JFrame frame = new JFrame("Tetris");
    CTetris tetris = new CTetris();
    frame.getContentPane().add(tetris, BorderLayout.CENTER);
    frame.setSize(300, 500);
    frame.setLocation(300, 100);
    frame.setDefaultCloseOperation(frame.EXIT_ON_CLOSE);
    frame.show();
    tetris.requestFocus();
    tetris.startNewGame();
  }
}
