/**
 * Title:        Comedia Beans
 * Copyright:    Copyright (c) 2001
 * Company:      Capella Development Group
 * @author Sergey Seroukhov
 * @version 1.0
 */

package org.comedia.ui;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import javax.swing.*;
import javax.swing.tree.*;
import javax.swing.text.*;
import javax.swing.event.*;
import org.comedia.text.*;

/**
 * Presents an editor bean with syntax highlighting, smart indents and other
 * features.
 * <p><img src="CSyntaxEditor.gif">
 * <p>Usage example:
 * <pre>
 * JFrame frame = new JFrame("Comedia Syntax Editor Test");
 * CSyntaxEditor editor = new CSyntaxEditor();
 * JScrollPane scroll = new JScrollPane();
 * scroll.setViewportView(editor);
 * editor.setSyntaxHighlighter(new CJavaHighlighter());
 * editor.read(new FileReader("d:\\test.java"), null);
 * </pre>
 */
public class CSyntaxEditor extends CEditor implements DocumentListener {
  /**
   * The indent flag.
   */
  private boolean indent = true;

  /**
   * The tab space length.
   */
  private int tabSpace = 2;

  /**
   * The highlighting thread.
   */
  private CHighlightThread thread = null;

//  public boolean highlight = false;

  /**
   * The syntax highlighter.
   */
  private CAbstractHighlighter syntaxHighlighter = null;

  /**
   * Constructs this class with default parameters.
   */
  public CSyntaxEditor() {
//    JEditorPane.super();
    setEditorKit(new CSyntaxEditorKit());
    this.setFont(new Font("Monospaced", 0, 12));
  }

  /**
   * Constructs a new class, with a specified document model.
   * @param doc the document model
   */
  public CSyntaxEditor(CSyntaxDocument doc) {
    this();
    setDocument(doc);
  }

  /**
   * Sets a new document model for this editor.
   * @param doc a new document model.
   */
  public void setDocument(Document doc) {
    if (!(doc instanceof CSyntaxDocument))
//      throw new IllegalArgumentException("Model must be CSyntaxDocument");
      return;

    if (getDocument() != null) {
      getDocument().removeDocumentListener(this);
    }
    super.setDocument(doc);
    if (getDocument() != null) {
      getDocument().addDocumentListener(this);
      performHighlight();
    }
  }

  /**
   * Performs an event when edited text was changed.
   * @param e an object which describes an event.
   **/
  public void changedUpdate(DocumentEvent e) {
    if (syntaxHighlighter != null && (thread == null || !thread.ready)) {
      try {
        int pos = getCaretPosition();
        String text = getDocument().getText(0, getDocument().getLength());
        int start = text.lastIndexOf('\n', pos-2);
        start = (start < 0) ? 0 : start;
        int end = text.indexOf('\n', pos-1);
        end = (end < 0) ? text.length()-1 : end;
        int unclosedComment = syntaxHighlighter.locateUnclosedComment(text, start);
        start = (unclosedComment >= 0) ? unclosedComment : start;
        if (thread == null) {
          thread = new CHighlightThread();
          thread.start();
        }
        thread.start = start;
        thread.end = end;
        thread.ready = true;
      }
      catch (Exception ex) {
        ex.printStackTrace();
      }
    }
  }

  /**
   * Performs an event when edited text was inserted.
   * @param e an object which describes an event.
   **/
  public void insertUpdate(DocumentEvent e) {
    changedUpdate(e);
  }

  /**
   * Performs an event when edited text was removed.
   * @param e an object which describes an event.
   **/
  public void removeUpdate(DocumentEvent e) {
    changedUpdate(e);
  }

  /**
   * Performs the syntax highlighting for the whole text.
   */
  public void performHighlight() {
    performHighlight(0, getDocument().getLength()-1);
  }

  /**
   * Performs the syntax highlighting.
   * @param start a start position.
   * @param end an end position.
   */
  public synchronized void performHighlight(int start, int end) {
    if (getDocument() instanceof CSyntaxDocument && syntaxHighlighter != null) {
      ((CSyntaxDocument) getDocument()).performHighlight(syntaxHighlighter,
        start, end);
    }
  }

  /**
   * Checks the current indent status.
   * @result <code>TRUE</code> if indent is turned on, <code>FALSE</code>
   *   otherwise.
   */
  public boolean isIndent() {
    return indent;
  }

  /**
   * Sets a new indent status.
   * @param indent <code>TRUE</code> to turn indent on, <code>FALSE</code>
   *   otherwise.
   */
  public void setIndent(boolean indent) {
    this.indent = indent;
  }

  /**
   * Gets the current tab space length.
   * @result the current tab space length.
   */
  public int getTabSpace() {
    return tabSpace;
  }

  /**
   * Sets a new tab space length.
   * @param tabSpace a new tab space length.
   */
  public void setTabSpace(int tabSpace) {
    this.tabSpace = tabSpace;
  }

  /**
   * Gets the current syntax highlighter.
   * @result the current syntax highlighter.
   */
  public CAbstractHighlighter getSyntaxHighlighter() {
    return syntaxHighlighter;
  }

  /**
   * Sets a new syntax highlighter.
   * @param highlighter a new syntax highlighter.
   */
  public void setSyntaxHighlighter(CAbstractHighlighter highlighter) {
    this.syntaxHighlighter = highlighter;
  }

  /**
   * Creates the EditorKit to use by default.  This
   * is implemented to return javax.swing.text.StyledEditorKit.
   * @return the editor kit
   */
  protected EditorKit createDefaultEditorKit() {
    return new CSyntaxEditorKit();
  }

  /**
   * Extracts a line from the specified position.
   * @param text a piece of text.
   * @result an extracted text line.
   */
  private static String extractLineFromEnd(String text) {
    int pos = text.lastIndexOf('\n');
    if (pos < 0) return text;
    if (pos >= text.length()-1) return "";
    return text.substring(pos + 1);
  }

  /**
   * Defines the indent of this string.
   * @param line a text line.
   * @result an indent of this line.
   */
  private static int defineIndent(String line) {
    int indent = 0;
    while (indent < line.length() && line.charAt(indent) == ' ')
      indent++;
    return indent;
  }

  /**
   * Removes line last line demiliters from last text line.
   * @param text a piece of text.
   * @result an extracted text line.
   */
  private static String removeLastLine(String text) {
    int pos = text.lastIndexOf('\n');
    return (pos > 0) ? text.substring(0, pos) : "";
  }

  /**
   * Creates a string repeating other one.
   * @param piece a piece of the string.
   * @param repeat a number of repeat times.
   * @result a created string.
   */
  private String createString(String piece, int repeat) {
    String result = "";
    for (int i = 0; i < repeat; i++)
      result += piece;
    return result;
  }

  /**
   * Invoked when key event is accepted from user.
   * @param e the event received.
   */
  protected void processKeyEvent(KeyEvent e) {
    final int PIECE_LENGTH = 1024;

    if (e.isAltDown() || e.isMetaDown() || e.isAltGraphDown() ||
      e.getID() != e.KEY_PRESSED) {
      super.processKeyEvent(e);
      return;
    }

    if ((e.getKeyCode() == e.VK_ENTER || e.getKeyCode() == e.VK_BACK_SPACE
      || e.getKeyCode() == e.VK_TAB) && !e.isShiftDown() && !e.isControlDown()
      && indent) {

      // Prepare text parameters
      int pos = getCaretPosition();
      String piece = "";
      try {
        piece = getDocument().getText(0, pos);
      }
      catch (Exception ex) {
        ex.printStackTrace();
      }
      String line = extractLineFromEnd(piece);
      int currIndent = defineIndent(line);
      piece = removeLastLine(piece.substring(0, piece.length() - line.length()));

      // Process ENTER key
      if (e.getKeyCode() == e.VK_ENTER) {
        try {
          while (piece.length() > 0 && currIndent > 0 && currIndent == line.length()) {
            line = extractLineFromEnd(piece);
            currIndent = defineIndent(line);
            piece = removeLastLine(piece.substring(0, piece.length() - line.length()));
          }
          getDocument().insertString(getCaretPosition(),
            "\n" + createString(" ", currIndent), null);
        }
        catch (Exception ex) {
          ex.printStackTrace();
        }
      // Process BACKSPACE key
      } else if (e.getKeyCode() == e.VK_BACK_SPACE) {
        if (currIndent == line.length()) {
          String lastLine = extractLineFromEnd(piece);
          int lastIndent = defineIndent(lastLine);
          piece = removeLastLine(piece.substring(0, piece.length() - lastLine.length()));
          while (piece.length() > 0 && lastIndent > 0 && lastIndent >= currIndent) {
            lastLine = extractLineFromEnd(piece);
            lastIndent = defineIndent(lastLine);
            piece = removeLastLine(piece.substring(0, piece.length() - lastLine.length()));
          }
          if (currIndent > lastIndent) {
            try {
              getDocument().remove(getCaretPosition() - currIndent + lastIndent + 1,
                currIndent - lastIndent - 1);
            }
            catch (Exception ex) {
              ex.printStackTrace();
            }
          } else if (currIndent <= lastIndent) {
            try {
              getDocument().remove(getCaretPosition() - currIndent + 1, currIndent - 1);
            }
            catch (Exception ex) {
              ex.printStackTrace();
            }
          }
        }
      // Process TAB key
      } else if (e.getKeyCode() == e.VK_TAB) {
        try {
          getDocument().insertString(getCaretPosition(),
            createString(" ", tabSpace), null);
        }
        catch (Exception ex) {
          ex.printStackTrace();
        }
      }

      e.consume();
    } else
      super.processKeyEvent(e);
  }

  /**
   * Implements a thread which makes a syntax highlighting.
   */
  private class CHighlightThread extends Thread {
    public int start = -1;
    public int end = -1;
    public boolean ready = false;

    public void run() {
      try {
        while (true) {
          if (ready) {
            performHighlight(start, end);
            ready = false;
          } else sleep(150);
          sleep(50);
        }
      }
      catch (Exception e) {
        e.printStackTrace();
      }
    }
  }

  /**
   * The main routine to run this module as standalone application.
   */
  public static void main(String[] args) {
    JFrame frame = new JFrame("Comedia Syntax Editor Test");
    CSyntaxEditor editor = new CSyntaxEditor();
    JScrollPane scroll = new JScrollPane();
    scroll.setViewportView(editor);
    try {
      editor.setSyntaxHighlighter(new CJavaHighlighter());
      editor.read(new FileReader("d:\\test.java"), null);
    }
    catch (Exception e) {
//      System.out.println("Load file problem: " + e.getMessage());
      e.printStackTrace();
    }
    frame.getContentPane().add(scroll, BorderLayout.CENTER);
    frame.setLocation(100, 100);
    frame.setSize(305, 320);
    frame.setDefaultCloseOperation(frame.EXIT_ON_CLOSE);
    frame.show();
  }
}