/**
 * Title:        Comedia Beans
 * Copyright:    Copyright (c) 2001
 * Company:      Capella Development Group
 * @author Sergey Seroukhov
 * @version 1.0
 */

package org.comedia.ui;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;

/**
 * Implements a MS Access like scroller for tables with row header.
 * <p><img src="CTableScroller.gif">
 * <p>Usage example:
 * <pre>
 * JFrame frame = new JFrame("Comedia Data Grid Test");
 * JTable table = new JTable(
 *   new String[][] {{"1x1", "1x2","1x3"}, {"2x1", "2x2", "2x3"},
 *   {"3x1", "3x2", "3x3"}, {"4x1", "4x2", "4x3"}},
 *   new String[] {"Column 1", "Column 2", "Column 3"});
 *
 * CTableScroller scroll = new CTableScroller(table);
 * frame.getContentPane().add(scroll, BorderLayout.CENTER);
 * </pre>
 */
public class CTableScroller extends JScrollPane implements ListSelectionListener,
  TableModelListener, ItemListener {
  /**
   * The default indicator column width.
   */
  final private static int HEADER_WIDTH = 18;

  /**
   * The related table control.
   */
  private JTable table = null;

  /**
   * The header container.
   */
  private JPanel headerPanel = new JPanel();

  /**
   * The internal panel with row indicators.
   */
  private JPanel internalPanel = new JPanel();

  /**
   * The scroller corner component.
   */
  private JComponent corner = null;

  /**
   * The last selection row.
   */
  private int lastSelection = 0;

  /**
   * Constructs this class with default properties.
   */
  public CTableScroller() {
    this(null);
  }

  /**
   * Constructs this class and assigns main properties.
   * @param table a related table control.
   */
  public CTableScroller(JTable table) {
    setTable(table);

    JViewport viewport = new JViewport();
    headerPanel.add(internalPanel);
    ((FlowLayout) headerPanel.getLayout()).setHgap(0);
    ((FlowLayout) headerPanel.getLayout()).setVgap(0);
    viewport.setView(headerPanel);
    setRowHeader(viewport);
    corner = createBlock();
    setCorner(JScrollPane.UPPER_LEFT_CORNER, corner);
    setViewportView(table);

    updateContent();
  }

  /**
   * Gets the related table control.
   * @result the related table control.
   */
  public JTable getTable() {
    return table;
  }

  /**
   * Sets a new related table control.
   * @param table a related table control.
   */
  public void setTable(JTable table) {
    if (this.table != null) {
      this.table.getSelectionModel().removeListSelectionListener(this);
      this.table.getModel().removeTableModelListener(this);
      this.setViewportView(null);
      if (this.table instanceof CDataTable)
        ((CDataTable) this.table).removeItemListener(this);
    }
    this.table = table;
    this.lastSelection = 0;
    if (this.table != null) {
      this.table.getSelectionModel().addListSelectionListener(this);
      this.table.getModel().addTableModelListener(this);
      this.setViewportView(this.table);
      if (this.table instanceof CDataTable)
        ((CDataTable) this.table).addItemListener(this);
      updateContent();
    }
  }

  /**
   * Performs an event when selection of the related table is changed.
   * @param e an object which describes occured event.
   */
  public void valueChanged(ListSelectionEvent e) {
    updateContent();
  }

  /**
   * Performs an event when model of the related table is changed.
   * @param e an object which describes occured event.
   */
  public void tableChanged(TableModelEvent e) {
    updateContent();
  }

  /**
   * Performs an event when status of table element is changed.
   * @param e an object which describes occured event.
   */
  public void itemStateChanged(ItemEvent e) {
    updateContent();
  }

  /**
   * Create a special block control.
   * @result a created block control.
   */
  private JComponent createBlock() {
    JLabel block = new JLabel("");
    block.setOpaque(true);
//    block.setBackground(SystemColor.control);
    block.setBorder(UIManager.getBorder("TableHeader.cellBorder"));
    block.setPreferredSize(
      new Dimension(HEADER_WIDTH, (table != null)? table.getRowHeight(): 0));
    block.setHorizontalAlignment(JLabel.CENTER);
    block.setVerticalAlignment(JLabel.CENTER);
    return block;
  }

  /**
   * Updates the contents of this scroller.
   */
  private void updateContent() {
    if (table == null) {
      internalPanel.removeAll();
      validate();
      return;
    }

    // Adjust blocks count
    internalPanel.setLayout(new GridLayout(table.getRowCount(), 1));
    while (internalPanel.getComponentCount() > table.getRowCount())
      internalPanel.remove(internalPanel.getComponent(internalPanel.getComponentCount()-1));
    while (internalPanel.getComponentCount() < table.getRowCount())
      internalPanel.add(createBlock());

    // Set marks on blocks
    int currentRow = (table.getSelectedRow() >= 0) ? table.getSelectedRow() : 0;
    if (table.getSelectedRowCount() > 1)
      currentRow = lastSelection;
    else lastSelection = currentRow;
    int insertRow = ((table instanceof CDataTable) && ((CDataTable) table).isInsert()) ?
      table.getRowCount() - 1: -1;
    for (int i = 0; i < internalPanel.getComponentCount(); i++) {
      JLabel current = (JLabel) internalPanel.getComponent(i);
      current.setPreferredSize(new Dimension(HEADER_WIDTH, table.getRowHeight()));
      if (i == table.getEditingRow() &&
        i == currentRow && table instanceof CDataTable) {
        CEncodedIcon icon = CEncodedIcon.PEN_SIGN;
        icon.setColor(SystemColor.controlText);
        current.setIcon(icon);
      } else if (i == currentRow) {
        CEncodedIcon icon = CEncodedIcon.BIG_RIGHT_ARROW;
        icon.setColor(SystemColor.controlText);
        current.setIcon(icon);
      } else if (i == insertRow) {
        CEncodedIcon icon = CEncodedIcon.STAR_SIGN;
        icon.setColor(SystemColor.controlText);
        current.setIcon(icon);
      } else
        current.setIcon(null);
    }

    validate();
  }

  /**
   * Updates Look&Feel of user interface.
   */
  public void updateUI() {
    super.updateUI();
    if (internalPanel != null) {
      corner.setBorder(UIManager.getBorder("TableHeader.cellBorder"));
     for (int i = 0; i < internalPanel.getComponentCount(); i++) {
        JLabel current = (JLabel) internalPanel.getComponent(i);
        current.setBorder(UIManager.getBorder("TableHeader.cellBorder"));
      }
    }
  }

  /**
   * The main routine to run this module as standalone application.
   */
  public static void main(String[] args) {
    JFrame frame = new JFrame("Comedia Data Grid Test");
    JTable table = new JTable(
      new String[][] {{"1x1", "1x2","1x3"}, {"2x1", "2x2", "2x3"},
        {"3x1", "3x2", "3x3"}, {"4x1", "4x2", "4x3"}},
      new String[] {"Column 1", "Column 2", "Column 3"});
/*
    table.setShowHorizontalLines(false);
    table.setShowVerticalLines(false);
    table.setColumnSelectionAllowed(false);
    table.setCellSelectionEnabled(false);
*/
    CTableScroller scroll = new CTableScroller(table);
    frame.getContentPane().add(scroll, BorderLayout.CENTER);

    frame.setLocation(100, 100);
    frame.setSize(500, 300);
    frame.setDefaultCloseOperation(frame.EXIT_ON_CLOSE);
    frame.show();

    try {
      UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
      SwingUtilities.updateComponentTreeUI(scroll);
    }
    catch(Exception e) {}
  }

}