/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

/** Creator:
 * 15.04.2005 16:35:28 Misteli
 *
 * Maintainer:
 * 15.04.2005 16:35:28 Misteli
 *
 * Last Modification:
 * $Id: PropertyModel.java,v 1.5 2005/11/29 13:55:19 misteli Exp $
 *
 * Copyright (c) 2003 ABACUS Research AG, All Rights Reserved
 */

package ch.abacus.lib.ui.propertyinspector.display;

import ch.abacus.lib.ui.JATable;
import ch.abacus.lib.ui.propertyinspector.core.AccessorInterface;
import ch.abacus.lib.ui.propertyinspector.core.Clazz;
import ch.abacus.lib.ui.propertyinspector.core.Clazzes;
import ch.abacus.lib.ui.propertyinspector.core.Group;
import ch.abacus.lib.ui.propertyinspector.core.GroupEditorInterface;
import ch.abacus.lib.ui.propertyinspector.core.Property;
import ch.abacus.lib.ui.propertyinspector.core.SelectionList;
import ch.abacus.lib.ui.propertyinspector.display.editor.PropertyInspectorCellEditor;
import ch.abacus.lib.ui.propertyinspector.display.renderer.PropertyInspectorCellRenderer;

import javax.swing.SwingUtilities;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import java.awt.Component;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.HashMap;

public class PropertyModel extends AbstractTableModel {

  private ArrayList displayProperties = new ArrayList();
  private boolean updating = true;
  private PropertyInspector propertyInspector;
  private Clazzes clazzes = new Clazzes();
  private SelectionList selection = new SelectionList();
  private AccessorInterface accessor;
  private DependenciesInterface dependencyChecker;
  private PropertyDefinitionInterface propertyDefinition;
  private DisplayProperties visibleProperties;
  private JATable table;
  private boolean sortAlphabetically;
  private boolean grouped;
  private Property defaultProperty;
  private PropertyInspectorCellRenderer properyInspectorRenderer;
  private PropertyInspectorCellEditor propertyInspectorCellEditor;
  private TableEventInterface tableEvent;
  private KeyAdapter tableKeyAdapter = new KeyAdapter() {
    public void keyPressed(KeyEvent e) {
      if (!table.isEditing()) {
        if (e.getKeyCode() == KeyEvent.VK_ENTER) {
          e.consume();
          editCurrentRow();
        }
      }
      else
        table.endEditing();
    }
  };
  private MouseAdapter mouseAdapter = new MouseAdapter() {
    public void mouseClicked(MouseEvent e) {
      Point p = e.getPoint();
      int row = table.rowAtPoint(p);
      if (row != -1) {
        Object value = getValueAt(row, 0);
        if (value instanceof DisplayGroup) {
          TableColumn column = table.getColumnModel().getColumn(0);
          int width = column.getWidth();
          column = table.getColumnModel().getColumn(1);
          width += column.getWidth();
          DisplayGroup group = ((DisplayGroup) value);
          if (p.x > width - 16 && p.x < width && group.getGroup().getEditor() != null) {
            GroupEditorInterface editor = group.getGroup().getEditor();
            editor.edit(e, group.getGroup());
          }
          else {
            group.getGroup().setFolded(!group.getGroup().isFolded());
            changeSelection();
          }
        }
      }
      if (tableEvent != null)
        tableEvent.propertySelectionChanged(table.getSelectedRow());
    }
  };

  PropertyModel(PropertyInspector propertyInspector, JATable table) {
    this.table = table;
    this.grouped = true;
    this.sortAlphabetically = false;
    this.propertyInspector = propertyInspector;
  }

  public TableEventInterface getTableEvent() {
    return tableEvent;
  }

  public void setTableEvent(TableEventInterface tableEvent) {
    this.tableEvent = tableEvent;
  }

  public HashMap<String, String> getLanguageStrings() {
    return propertyInspector.getLanguageStrings();
  }

  public boolean isSortAlphabetically() {
    return sortAlphabetically;
  }

  public void setSortAlphabetically(boolean sortAlphabetically) {
    if (sortAlphabetically == this.sortAlphabetically)
      return;
    this.sortAlphabetically = sortAlphabetically;
    changeSelection();
  }

  public boolean isGrouped() {
    return grouped;
  }

  public void setGrouped(boolean grouped) {
    if (grouped == this.grouped)
      return;
    this.grouped = grouped;
    changeSelection();
  }

  public Property getDefaultProperty() {
    return defaultProperty;
  }

  public void setDefaultProperty(Property defaultProperty) {
    this.defaultProperty = defaultProperty;
  }

  public void refreshProperty(Property property) {
    Object display;
    for (int i=0; i<displayProperties.size(); i++) {
      display = displayProperties.get(i);
      if (display instanceof DisplayProperty && ((DisplayProperty) display).getProperty() == property) {
        ((DisplayProperty) display).getValues().clear();
        for (int a=0; a<selection.size(); a++)
          ((DisplayProperty) display).getValues().add(accessor.getValue(property, selection.get(a)));
        Rectangle rect = table.getCellRect(i, 1, true);
        table.repaint(rect);
        break;
      }
    }
  }

  public void initialize() {
    properyInspectorRenderer = new PropertyInspectorCellRenderer(table);
    propertyInspectorCellEditor = new PropertyInspectorCellEditor(table);
    TableColumnModel columnModel = table.getColumnModel();
    columnModel.getColumn(0).setCellRenderer(properyInspectorRenderer);
    columnModel.getColumn(1).setCellRenderer(properyInspectorRenderer);
    columnModel.getColumn(1).setCellEditor(propertyInspectorCellEditor);
    table.addMouseListener(mouseAdapter);
    table.addKeyListener(tableKeyAdapter);
    table.setFocusCycleRoot(true);
    table.setFocusTraversalKeysEnabled(false);
  }

  public int getColumnCount() {
    return 2;
  }

  public int getRowCount() {
    return displayProperties.size();
  }

  public AccessorInterface getAccessor() {
    return accessor;
  }

  public void setAccessor(AccessorInterface accessor) {
    this.accessor = accessor;
  }

  public PropertyDefinitionInterface getPropertyDefinition() {
    return propertyDefinition;
  }

  public void setPropertyDefinition(PropertyDefinitionInterface propertyDefinition) {
    this.propertyDefinition = propertyDefinition;
  }

  public DependenciesInterface getDependencyChecker() {
    return dependencyChecker;
  }

  public void setDependencyChecker(DependenciesInterface dependencyChecker) {
    this.dependencyChecker = dependencyChecker;
  }

  public Object getValueAt(int rowIndex, int columnIndex) {
    return displayProperties.get(rowIndex);
  }

  public boolean isCellEditable(int rowIndex, int columnIndex) {
    Object object = displayProperties.get(rowIndex);
    if (object instanceof DisplayGroup)
      return false;
    boolean ok = columnIndex > 0;
    if (object instanceof DisplayProperty) {
      DisplayProperty p = (DisplayProperty) object;
      ok = ok && !p.getProperty().isReadOnly();
      ok = ok && !(p.getProperty().isSingleSelection() && selection.size() > 1);
      ok = ok && p.getProperty().getEditor() != null;
      ok = ok && p.isEnabled();
    }
    return ok;
  }

  public void editCurrentRow() {
    int row = table.getSelectedRow();
    if (row != -1)
      editRow(row);
  }

  private int getNearestRow() {
    int row = table.getSelectedRow();
    if (row == -1)
      for (int i=0; i<displayProperties.size(); i++)
        if (displayProperties.get(i) instanceof DisplayProperty && ((DisplayProperty) displayProperties.get(i)).getProperty() == defaultProperty)
          return i;
    if (displayProperties.size() > 0)
      return 0;
    return -1;
  }

  public void editAnyRow() {
    int row = getNearestRow();
    if (row != -1)
      editRow(row);
  }

  public void editRow(int row) {
    table.editCellAt(row, 1);
    SwingUtilities.invokeLater(new Runnable() {
      public void run() {
        Component c = table.getEditorComponent();
        if (c != null)
          c.requestFocus();
      }
    });
  }

  public void setSelection(Object[] objects) {
    if (table.isEditing())
      table.endEditing();
    selection.clear();
    beginUpdate();
    try {
      for (int i=0; i<objects.length; i++)
        addSelection(objects[i]);
      changeSelection();
    }
    finally {
      endUpdate();
    }
  }

  private void addPropertiesToSelection(DisplayProperties selection, ObjectInstance instance, Clazz clazz) {
    Property property;
    DisplayProperty p;
    for (int i=0; i<clazz.getProperties().size(); i++) {
      property = clazz.getProperties().get(i);
      p = selection.findByProperty(property, true);
      p.getValues().add(accessor.getValue(property, instance));
    }
  }

  protected Clazz findClazzFromInstance(Clazzes clazzes, ObjectInstance instance) {
    if (instance.getClassName() != null)
      return clazzes.findByInstance(instance.getClassName());
    else
      return clazzes.findByInstance(instance.getObject());
  }

  public void changeSelection() {
    if (!updating || (displayProperties.size() == 0 && selection.size() == 0))
      return;

    // Alle Properties aller zutreffenden Klassen der Instanzen in eine Liste einfgen..
    visibleProperties = new DisplayProperties();
    ObjectInstance instance;
    for (int i=0; i<selection.size(); i++) {
      instance = selection.get(i);
      Clazz clazz = findClazzFromInstance(clazzes, instance);
      while (clazz != null) {
        addPropertiesToSelection(visibleProperties, selection.get(i), clazz);
        clazz = clazz.getParent();
      }
    }
    if (propertyDefinition != null)
      propertyDefinition.addPropertyDefinition(selection, visibleProperties);

    // Und nun alle nicht sichtbaren, da nicht in allen Klassen vorkommen, Properties rausknallen..
    DisplayProperty p;
    for (int i=visibleProperties.size()-1; i>=0; i--) {
      p = visibleProperties.get(i);
      if (p.getValues().size() != selection.size())
        visibleProperties.remove(p);
    }

    // Und nun die DisplayGruppen zusammenstellen..
    DisplayGroup root = new DisplayGroup(null);
    DisplayGroup displayGroup;
    Group group;
    for (int i=0; i<visibleProperties.size(); i++) {
      p = visibleProperties.get(i);
      if (grouped) {
        group = p.getProperty().getGroup();
        ArrayList<Group> list = new ArrayList<Group>();
        if (group != null)
          do {
            list.add(0, group);
            group = group.getParent();
          } while (group != null);
        displayGroup = root;
        DisplayGroup parent = root;
        for (int gg=0; gg<list.size(); gg++) {
          group = list.get(gg);
          displayGroup = displayGroup.findByGroup(group);
          if (displayGroup == null) {
            displayGroup = new DisplayGroup(group);
            parent.getChildren().add(displayGroup);
          }
          if (gg == list.size() - 1) // das unterste..
            displayGroup.getProperties().add(p);
          parent = displayGroup;
        }
      }
      else
        root.getProperties().add(p);
    }

    // Und zuguterletzt alle visuellen Gruppen sortieren..
    if (grouped) {
      root.getChildren().sort(sortAlphabetically);
      sortGroups(root);
    }
    else
      root.getProperties().sort(sortAlphabetically);

    // Und nun die endgltigen Gruppen zusammenstaucheln..
    displayProperties.clear();
    flattenDisplayGroups(root);

    // Dependencies updaten..
    if (dependencyChecker != null)
      for (int i=0; i<visibleProperties.size(); i++) {
        p = visibleProperties.get(i);
        DisplayProperties dependencies = createDependencyListFor(p.getProperty());
        dependencyChecker.checkDependencies(p, p.getValues().getCurrentValue(), dependencies);
      }
    // Und Tabelle refreshen..
    fireTableDataChanged();
  }

  public DisplayProperties getVisibleProperties() {
    return visibleProperties;
  }

  private boolean isGroupVisible(Group group) {
    boolean ok = true;
    while (group != null) {
      ok &= !group.isFolded();
      group = group.getParent();
    }
    return ok;
  }

  private void sortGroups(DisplayGroup root) {
    root.getProperties().sort(sortAlphabetically);
    for (int i=0; i<root.getChildren().size(); i++)
      sortGroups(root.getChildren().get(i));
  }

  private void flattenDisplayGroups(DisplayGroup root) {
    if (root.getGroup() != null)
      if (isGroupVisible(root.getGroup().getParent()))
        displayProperties.add(root);
    DisplayProperty property;
    for (int i=0; i<root.getProperties().size(); i++) {
      property = root.getProperties().get(i);
      if (grouped && !isGroupVisible(property.getProperty().getGroup()))
        continue;
      displayProperties.add(root.getProperties().get(i));
    }
    for (int i=0; i<root.getChildren().size(); i++)
      flattenDisplayGroups(root.getChildren().get(i));
  }

  public void beginUpdate() {
    updating = false;
  }

  public void endUpdate() {
    updating = true;
    changeSelection();
  }

  public void addSelection(String className, Object object) {
    selection.add(new ObjectInstance(className, object));
    changeSelection();
  }

  public void addSelection(Object object) {
    addSelection(null, object);
  }

  public Clazzes getClazzes() {
    return clazzes;
  }

  public ArrayList<ObjectInstance> getSelection() {
    return selection;
  }

  private DisplayProperties createDependencyListFor(Property property) {
    DisplayProperties dependencies = new DisplayProperties();
    for (int a=0; a<property.getDependencies().size(); a++) {
      Property dependency = property.getDependencies().get(a);
      DisplayProperty dp = visibleProperties.findByProperty(dependency, false);
      if (dp == null)
        dp = new DisplayProperty(dependency);
      dp.setEnabled(true);
      dependencies.add(dp);
    }
    return dependencies;
  }

  public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
    if (!((PropertyInspectorCellEditor) table.getCellEditor()).getCurrentEditor().isModified())
      return;
    DisplayProperty displayProperty = (DisplayProperty) displayProperties.get(rowIndex);
    displayProperty.getValues().clear();
    Property property = displayProperty.getProperty();
    accessor.beginUpdate();
    try {
      for (int i=0; i<selection.size(); i++) {
        accessor.setValue(property, selection.get(i), aValue);
        displayProperty.getValues().add(aValue);
        if (dependencyChecker != null && property.getDependencies().size() > 0) {
          DisplayProperties dependencies = createDependencyListFor(property);
          dependencyChecker.checkDependencies(displayProperty, aValue, dependencies);
        }
        fireTableDataChanged();
      }
    }
    finally {
      accessor.endUpdate();
    }
  }

}
