/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package ch.abacus.lib.ui.renderer.common;

/**
 * <p>Title: uifactory</p>
 * <p>Description: </p>
 * <p>Copyright: Copyright (c) 2001</p>
 * <p>Company: Abacus Research</p>
 * @author Michael Gouker (Cagey Logic)
 * @version 1.0
 *
 *      <Parameter>
 <Order> numeric value that denotes the number of this
 parameter </Order>  <-- optional (not included if only one)
 {Type declaration}
 <Value> text value that denotes the data for this parameter
 </Value>  <--- optional (if simple type - otherwise type declaration
 contains simple types - at some point it resolves to simple types)
 </Parameter>
 *
 */

public class MetaParameter {
    public int theOrder = 0;
    public MetaType theComplexType = null;
    public String theSimpleType = null;  //basic types are without params
    public MetaPropertyValueEx theValue = null;  // exists when there is a simple type.
    public String theName = null;  // Used for hints.
    public MetaParameter thePreviousParameter = null;
    public MetaParameter theNextParameter = null;
    boolean bValid = false;
    public MetaProject theDesignProject = null;
    boolean bAllowMultipleLanguages = false;
    boolean bObjectReference = false;

    public MetaParameter() {
    }

    public MetaParameter(int theOrder, MetaType theComplexType, String theName, MetaProject theDesignProject,
                         boolean bAllowMultipleLanguages, boolean bObjectReference) {
        this.theOrder = theOrder;
        this.theComplexType = theComplexType;
        this.theName = theName;
        bValid = true;
        this.theDesignProject = theDesignProject;
        this.bAllowMultipleLanguages = bAllowMultipleLanguages;
        this.bObjectReference = bObjectReference;
    }

    public MetaParameter(int theOrder, MetaType theComplexType, MetaProject theDesignProject,
                         boolean bAllowMultipleLanguages, boolean bObjectReference) {
        this.theOrder = theOrder;
        this.theComplexType = theComplexType;
        bValid = true;
        this.theDesignProject = theDesignProject;
        this.bAllowMultipleLanguages = bAllowMultipleLanguages;
        this.bObjectReference = bObjectReference;
    }

    public MetaParameter(int theOrder, MetaPropertyValueEx objValue,
                         boolean bAllowMultipleLanguages, boolean bObjectReference) {
        this.theOrder = theOrder;
        this.theValue = objValue;
        if (objValue != null)
            this.theSimpleType = MetaPropertyValueEx.translateType(objValue.getType());
        else
            this.theSimpleType = "java.lang.String";
        theDesignProject = objValue.theDesignProject;
        bValid = true;
        this.bAllowMultipleLanguages = bAllowMultipleLanguages;
        this.bObjectReference = bObjectReference;
    }

    public MetaParameter(int theOrder, String theSimpleType, String sValue, MetaProject theDesignProject,
                         boolean bAllowMultipleLanguages, boolean bObjectReference) {
        this.theOrder = theOrder;
        this.theSimpleType = theSimpleType;
        this.bAllowMultipleLanguages = bAllowMultipleLanguages;
        this.bObjectReference = bObjectReference;
        this.theDesignProject = theDesignProject;
        this.theValue = new MetaPropertyValueEx(theSimpleType, sValue, theDesignProject, bAllowMultipleLanguages);
    }

    public MetaParameter(int theOrder, MetaPropertyValueEx objValue, String theName, MetaProject theDesignProject,
                         boolean bAllowMultipleLanguages, boolean bObjectReference) {
        this.theOrder = theOrder;
        if (objValue != null) {
            this.theSimpleType = MetaPropertyValueEx.translateType(objValue.getType());
            objValue.theDesignProject = theDesignProject;
        } else
            this.theSimpleType = new String("java.lang.String");
        this.theName = theName;
        this.theValue = objValue;
        this.theDesignProject = theDesignProject;
        bValid = true;
        this.bAllowMultipleLanguages = bAllowMultipleLanguages;
        this.bObjectReference = bObjectReference;
    }

    // order 0 constructors. We should chain constructors here instead.  Less code.
    public MetaParameter(MetaType theComplexType, String theName, MetaProject theDesignProject,
                         boolean bAllowMultipleLanguages, boolean bObjectReference) {
        this.theComplexType = theComplexType;
        this.theName = theName;
        bValid = true;
        this.theDesignProject = theDesignProject;
        this.bAllowMultipleLanguages = bAllowMultipleLanguages;
        this.bObjectReference = bObjectReference;
    }

    public MetaParameter(MetaType theComplexType, MetaProject theDesignProject,
                         boolean bAllowMultipleLanguages, boolean bObjectReference) {
        this.theComplexType = theComplexType;
        bValid = true;
        this.theDesignProject = theDesignProject;
        this.bAllowMultipleLanguages = bAllowMultipleLanguages;
        this.bObjectReference = bObjectReference;
    }

    public MetaParameter(MetaPropertyValueEx objValue, MetaProject theDesignProject,
                         boolean bAllowMultipleLanguages, boolean bObjectReference) {
        this.theValue = objValue;
        if (objValue != null) {
            this.theSimpleType = MetaPropertyValueEx.translateType(objValue.getType());
            objValue.theDesignProject = theDesignProject;
        } else
            this.theSimpleType = new String("java.lang.String");
        this.theDesignProject = theDesignProject;
        bValid = true;
        this.bAllowMultipleLanguages = bAllowMultipleLanguages;
        this.bObjectReference = bObjectReference;
    }

    public MetaParameter(String theSimpleType, String sValue, MetaProject theDesignProject,
                         boolean bAllowMultipleLanguages, boolean bObjectReference) {
        this.theSimpleType = theSimpleType;
        this.theDesignProject = theDesignProject;
        this.bAllowMultipleLanguages = bAllowMultipleLanguages;
        this.theValue = new MetaPropertyValueEx(theSimpleType, sValue, theDesignProject, bAllowMultipleLanguages);
        this.bObjectReference = bObjectReference;
    }

    /**
     * searchParameterTree - Searches a tree of parameters for a named value.
     *
     * @param sShortName - the Name of the component at this level.
     * @return value is null or a MetaParameter child component with the
     * name that is supplied in sShortName.
     */
    public MetaParameter searchParameterTree(String sShortName) {
        MetaParameter theTestParameter = this;
        while (theTestParameter != null) {
            if (sShortName.equals(theName))
                return theTestParameter;
            if (theComplexType != null) {
                MetaParameter testSubParameter = theComplexType.theFirstParameter;
                while (testSubParameter != null) {
                    if (sShortName.equals(testSubParameter.theName))
                        return testSubParameter;
                    testSubParameter = testSubParameter.theNextParameter;
                }
            }
            theTestParameter = theTestParameter.theNextParameter;
        }
        return null;
    }

    public void set(String sType, String sValue, MetaProject theDesignProject) {
        if (theValue == null)
            theValue = new MetaPropertyValueEx(sType, sValue, theDesignProject, bAllowMultipleLanguages);
        else {
            theValue.setLocalString(sValue);
            theValue.theDesignProject = theDesignProject;
        }
    }

    public void set(String sType, String sValue, boolean bCast, MetaProject theDesignProject) {
        if (theValue == null)
            theValue = new MetaPropertyValueEx(sType, sValue, bCast, theDesignProject, bAllowMultipleLanguages);
        else {
            theValue.setLocalString(sValue);
            theValue.theDesignProject = theDesignProject;
        }
    }

    public void attachToProject(MetaProject theProj) {
        this.theDesignProject = theProj;
        if (theValue != null)
            theValue.theDesignProject = theProj;
        if (theComplexType != null) {
            theComplexType.attachToProject(theProj);
        }
    }

    public boolean save(electric.xml.Element theParent) {
        // create the parameter
        electric.xml.Element xmlParameter = new electric.xml.Element("Parameter");
        theParent.addElement(xmlParameter);
        electric.xml.Element xmlOrder = new electric.xml.Element("Order");
        xmlOrder.setText(new Integer(theOrder).toString());
        if ((theName != null) && (theName.length() > 0)) {
            electric.xml.Element xmlName = new electric.xml.Element("Name");
            xmlName.setText(theName);
            xmlParameter.addElement(xmlName);
        }
        xmlParameter.addElement(xmlOrder);
        // obj ref
        if (bObjectReference) {
            electric.xml.Element xmlObjRef = new electric.xml.Element("ObjectReference");
            xmlParameter.addElement(xmlObjRef);
        }
        // add Value.
        if (theComplexType != null)
            theComplexType.save(xmlParameter);
        else {
            electric.xml.Element xmlSimpleType = new electric.xml.Element("SimpleType");
            xmlSimpleType.setText(theSimpleType);
            xmlParameter.addElement(xmlSimpleType);
        }
        if (theValue != null) {
            if ((theValue.getLanguageCount() == 1) || (bAllowMultipleLanguages == false)) {
                electric.xml.Element xmlValue = new electric.xml.Element("Value");
                xmlValue.setText(theValue.getStringValue());
                xmlParameter.addElement(xmlValue);
            } else {
                String sDefaultValue = theValue.getLanguageDependentValue(HammerLanguagePresentation.DEFAULT);
                for (int iLang = 0; iLang <= HammerLanguagePresentation.LAST_LANGUAGE; iLang++) {
                    HammerLanguagePresentation thisLanguage = theValue.theDesignProject.getLanguage(iLang);
                    if (thisLanguage != null) {
                        String sLanguageValue = theValue.getLanguageDependentValue(iLang);
                        if ((iLang == 0) || (sLanguageValue.equals(sDefaultValue) == false))
                            thisLanguage.saveValue(xmlParameter, sLanguageValue);
                    }
                }
            }
        }
        return true;
    }

    public boolean load(electric.xml.Element theElement, int iOldVersionFixIndex) {
        electric.xml.Element xmlOrder = theElement.getElement("Order");
        String sOrder = xmlOrder.getTextString();
        theOrder = new Integer(sOrder).intValue();
        if (this.theDesignProject.needsFixed())
            theOrder = iOldVersionFixIndex;
        electric.xml.Element xmlObjectReference = theElement.getElement("ObjectReference");
        if (xmlObjectReference != null)
            bObjectReference = true;
        else
            bObjectReference = false;
        electric.xml.Element xmlType = theElement.getElement("Type");
        if (xmlType != null) {
            theComplexType = new MetaType(theDesignProject);
            theComplexType.load(xmlType);
        } else {
            electric.xml.Element xmlSimpleType = theElement.getElement("SimpleType");
            if (xmlSimpleType != null)
                theSimpleType = xmlSimpleType.getTextString();
            else
                theSimpleType = "java.lang.String";
            electric.xml.Elements xmlLanguages = theElement.getElements("Language");
            if ((xmlLanguages == null) || (xmlLanguages.size() == 0)) {
                electric.xml.Element xmlValue = theElement.getElement("Value");
                String sValue = null;
                if (xmlValue != null)
                    sValue = xmlValue.getTextString();
                if (sValue == null)
                    sValue = new String("");
                theValue = new MetaPropertyValueEx(theSimpleType, sValue, theDesignProject, false);
            } else {
                String[] theValues = new String[HammerLanguagePresentation.LAST_LANGUAGE + 1];
                electric.xml.Element xmlLanguage = xmlLanguages.next();
                while (xmlLanguage != null) {
                    int theLanguage = HammerLanguagePresentation.DEFAULT;
                    String theAttribute = xmlLanguage.getAttribute("Name");
                    try {
                        Integer theValue = new Integer(theAttribute);
                        theLanguage = theValue.intValue();
                    } catch (java.lang.NumberFormatException e1) {
                    }
                    HammerLanguagePresentation thisLanguage = theDesignProject.getLanguage(theLanguage);
                    theValues = thisLanguage.loadValue(xmlLanguage, theValues);
                    xmlLanguage = xmlLanguages.next();
                }
                theValue = new MetaPropertyValueEx(theSimpleType, theValues, theDesignProject, true);
                this.bAllowMultipleLanguages = true; // SWB: Mar 16,2005 - fix to allow multiple languages on loaded projects. 
            }
        }
        electric.xml.Element xmlName = theElement.getElement("Name");
        if (xmlName != null)
            theName = xmlName.getTextString();

        return true;
    }

    public MetaParameter Clone() {
        if (theComplexType != null)
            return new MetaParameter(theOrder, theComplexType.Clone(), theName, theDesignProject,
                    bAllowMultipleLanguages, bObjectReference);
        else {
            MetaParameter newParameter = new MetaParameter(theOrder, (theValue != null) ? theValue.Clone() : null, theName,
                    theDesignProject, bAllowMultipleLanguages, bObjectReference);
            newParameter.theSimpleType = theSimpleType;
            return newParameter;
        }
    }

    public Object Create() {
        if (theSimpleType != null) {
            return theValue;
        } else
            return this.theComplexType.Create(null, null);
    }

    public String getLiteralValue() {
        if (theSimpleType != null)
            return theValue.getStringValue();
        else
            return theComplexType.getLiteralValue();
    }

    public void setMnemonic() {
        if (theSimpleType != null)
            theValue.setMnemonic();
        else
            theComplexType.setMnemonic();
    }

    public boolean getMnemonic() {
        if (theSimpleType != null)
            return theValue.bMnemonic;
        else
            return theComplexType.getMnemonic();
    }


}
