/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
/*
 * Created by IntelliJ IDEA.
 * User: Administrator
 * Date: Aug 2, 2004
 * Time: 12:27:49 AM
 */
package ch.abacus.lib.ui.renderer.common.generator;

import ch.abacus.lib.ui.renderer.common.*;
import ch.abacus.lib.ui.JAScrollPane;

import javax.swing.*;
import java.util.HashMap;
import java.util.Vector;
import java.util.ArrayList;
import java.io.StringReader;
import java.io.StringWriter;

public class AbacusCodeGenerator implements ProjectCodeGenerator {

    ProjectDocument theDocument;
    public Vector vectorConstants = new Vector(0);
    private Vector vRequiredImports;
    private boolean bNLSTranslation    = true;
    private boolean bgenerateAccessor  = true;

    /**
     * Variables - The key is the identifier, the value is the value of the identifier
     */
    HashMap Variables = new HashMap();
    /**
     * VariableClasses - The key is the classname, the value stored is the number of vars.
     */
    HashMap VariableClasses = new HashMap();

    BufferCollection theBufferCollection = new BufferCollection();
    HashMap theLookupTable = new HashMap();
    ListenerCollection theListenerCollection = new ListenerCollection();

    public AbacusCodeGenerator(ProjectDocument theDocument) {
        this.theDocument = theDocument;
        vRequiredImports = new Vector();
    }

    public ProjectDocument getProjectDocument() {
        return theDocument;
    }

    public void setProjectDocument(ProjectDocument theDocument) {
        this.theDocument = theDocument;
    }

    // lookup table.
    private String getValue(MetaObject theObject, String sKey) {
        sKey = theObject.getName() + "|" + sKey;
        return (String) this.theLookupTable.get(sKey);
    }

    private void setValue(MetaObject theObject, String sKey, String sValue) {
        sKey = theObject.getName() + "|" + sKey;
        theLookupTable.put(sKey, sValue);
    }

    private String getIdentifier(MetaObject theObject) {
        return getValue(theObject, "Identifier");
    }

    private void setIdentifier(MetaObject theObject, String value) {
        setValue(theObject, "Identifier", value);
    }

    private String getContainerName(MetaObject theObject) {
        return getValue(theObject, "ContainerName");
    }

    private void setContainerName(MetaObject theObject, String value) {
        setValue(theObject, "ContainerName", value);
    }

    private String getLayoutName(MetaObject theObject) {
        return getValue(theObject, "LayoutName");
    }

    private void setLayoutName(MetaObject theObject, String value) {
        setValue(theObject, "LayoutName", value);
    }

    private String getGroupName(MetaObject theObject) {
        return getValue(theObject, "GroupName");
    }

    private void setGroupName(MetaObject theObject, String value) {
        setValue(theObject, "GroupName", value);
    }

    // code generation.
    //
    // define class
    // define members
    // define object accessors
    // define constructor  (creates objects, calls property setting, calls attachment code)
    // define property setting
    // define attachment code
    //

    public void generateObjectCode(MetaObject theObject) throws HammerException {
        String sProjectName = theObject.theDesignProject.getName();
        int iIndex = sProjectName.indexOf(".");
        sProjectName = sProjectName.substring(0, iIndex);
        // Generate object class and code. // MHC Open Source
        if(this.getProjectDocument().isOpenSource() && theObject.getObjectCode()!=null)
            generateObjecClass(theObject);
        //sIdentifier = "AbaRenderer_"+ sProjectName + "_" + getName();
        setIdentifier(theObject, theObject.getName());
        generateMember(theObject);
        // Retreives the flag everytime an object is looked up in case in needs to change
        // later to an object property
        bgenerateAccessor=theObject.theDesignProject.isGenerateAccessor();
        generateAccessor(theObject);
        generateBuilder(theObject);
        generatePropertySettings(theObject);
        generateAttachmentCode(theObject);
        MetaObject objTest = theObject.theFirstChild;
        while (objTest != null) {
            generateObjectCode(objTest);
            objTest = objTest.theNextObject;
        }
    }

    public void generateMember(MetaObject theObject) {
        String sClassName = theObject.theClass.theMetadata.sFullClassName;

        if(theObject.isPatched() && this.getProjectDocument().isOpenSource()==true)
        {
            sClassName = theObject.getPatchedClassName();
        }
        else
        {
            int iLastDot = sClassName.lastIndexOf(".");
            sClassName = sClassName.substring(iLastDot + 1);
        }

        String sDeclaration = "    private " + sClassName + " " + getIdentifier(theObject) + ";";
        theBufferCollection.sDeclBuffer += sDeclaration;
        theBufferCollection.sDeclBuffer += "\r\n";
        int iContainerType = theObject.getContainerType();
        if ((iContainerType == MetaContainerType.CONTAINER_FRAME) || (iContainerType == MetaContainerType.CONTAINER_ABALET))
            theBufferCollection.sDeclBuffer += "Container contentPane;\r\n";
    }

    public void generateObjecClass(MetaObject theObject) {
        String sObjectName = theObject.getName();
        String sObjectBasedClassName = "$$$" + sObjectName + "$$$";
        String sSuperClassName = theObject.theClass.theMetadata.sFullClassName;
        int iLastDot = sSuperClassName.lastIndexOf(".");
        sSuperClassName= sSuperClassName.substring(iLastDot + 1);
        String sObjectBasedClass = "class " + sObjectBasedClassName + " extends " + sSuperClassName + "{" + "\r\n";
        sObjectBasedClass += theObject.getObjectCode() + "\r\n" + "}" + "\r\n";
        theObject.setPatchedClassName(sObjectBasedClassName);
        theObject.setPatchedStatus(true);
        theBufferCollection.sAccessorBuffer += sObjectBasedClass;
    }

    public void generateAccessor(MetaObject theObject) {
        String sClassName = theObject.theClass.theMetadata.sFullClassName;
        int iLastDot = sClassName.lastIndexOf(".");
        sClassName = sClassName.substring(iLastDot + 1);
        if (theObject.equals(theObject.theDesignProject.getFirstObject())) {
            String sAccessorCode = "public Object getRenderingDestination() {\r\n";
            sAccessorCode += "return " + getIdentifier(theObject) + ";\r\n}\r\n\r\n";
            theBufferCollection.sAccessorBuffer += sAccessorCode;
        }
        String sAccessorCode = new String();
        if(bgenerateAccessor)
        {
            if(theObject.isPatched() && this.getProjectDocument().isOpenSource()==true)
            {
                sAccessorCode = "public " + theObject.getPatchedClassName() + " get" + theObject.getName() + "() {\r\n";
                sAccessorCode += "return " + getIdentifier(theObject) + ";\r\n}\r\n\r\n";
            }
            else
            {
                sAccessorCode = "public " + sClassName + " get" + theObject.getName() + "() {\r\n";
                sAccessorCode += "return " + getIdentifier(theObject) + ";\r\n}\r\n\r\n";
            }
        }

        theBufferCollection.sAccessorBuffer += sAccessorCode;
    }

    public void generateBuilder(MetaObject theObject) {
        // MHC Code changed in order to add new object source code functionality
        // march 31st - 2005 -- MHC for open source version

        String sfullClassName = theObject.theClass.theMetadata.sFullClassName;
        String sClassName = theObject.theClass.theMetadata.sClassName;
        String sBuildCode = null;

        if(theObject.isPatched()==true && this.getProjectDocument().isOpenSource()==true)
        {
            sfullClassName = theObject.getPatchedClassName();
            sClassName = theObject.getPatchedClassName();
            sBuildCode = getIdentifier(theObject) +
                    " = new " + sfullClassName + "();" + "\n\r"  ;
            sBuildCode += "theRenderer.createInterfaceObjectEx(" +
                    "\""+  theObject.getName() + "\"" + "," + theObject.getName() + ");";
        }
        else
        {
            int iLastDot = sfullClassName.lastIndexOf(".");
            sfullClassName= sfullClassName.substring(iLastDot + 1);
            sBuildCode = getIdentifier(theObject) +
                    " = (" + sfullClassName +
                    ") theRenderer.createInterfaceObject(\"" +
                    theObject.getName() + "\", \"" +
                    sClassName + "\");";
        }
        if (theObject.isFrame() || theObject.isAbalet()) {
            theBufferCollection.sAssignmentBuffer += "if (bMakeFrame) {\r\n";
        }
        theBufferCollection.sAssignmentBuffer += sBuildCode;
        theBufferCollection.sAssignmentBuffer += "\r\n";
    }

    public void generatePropertySettings(MetaObject theObject) throws HammerException {
        String sSetCode = "";
        MetaProperty theProperty = theObject.theFirstProperty;

        String metaClassName = theProperty.theObject.theType.theName;
        bNLSTranslation=theObject.theDesignProject.isNLSCodeTranslation();
        while (theProperty != null) {
            String sPropertySet = generatePropertyCode(theProperty);
            if (theProperty.thePropertyInfo != null) {
                String sSetMethod = theProperty.thePropertyInfo.getSetMethod();
                boolean bIsObjReference = theProperty.thePropertyInfo.bObjectReference;// MHC

                if ((sSetMethod != null) && (sSetMethod.length() > 0) && (bIsObjReference == false)) {
                    sSetCode += getIdentifier(theObject) + "." + theProperty.thePropertyInfo.getSetMethod() + "(" + sPropertySet + ")";
                    sSetCode += ";\r\n";
                } else if ((sSetMethod != null) && (sSetMethod.length() > 0) && (bIsObjReference == true)) // MHC
                {
                    String sProjectName = theObject.theDesignProject.getName();
                    int iIndex = sProjectName.indexOf(".");
                    sProjectName = sProjectName.substring(0, iIndex);
                    sSetCode += getIdentifier(theObject) + "." + theProperty.thePropertyInfo.getSetMethod() + "(" + sPropertySet + ")";
                    sSetCode += ";\r\n";
                }
                if (theObject.bAutoEmbed) {
                    if (sSetMethod.equals("setSize")) {
                        sSetCode += "scr_" + getIdentifier(theObject) + "." + theProperty.thePropertyInfo.getSetMethod() + "(" + sPropertySet + ")";
                        sSetCode += ";\r\n";
                    }
                    if (sSetMethod.equals("setLocation")) {
                        sSetCode += "scr_" + getIdentifier(theObject) + "." + theProperty.thePropertyInfo.getSetMethod() + "(" + sPropertySet + ")";
                        sSetCode += ";\r\n";
                    }
                }
            }
            theProperty = theProperty.theNextProperty;
        }
        // add events.
        sSetCode = generateObjectEventClasses(sSetCode, theObject);

        theBufferCollection.sPropertyMethodBuffer += "\r\nvoid set_" + getIdentifier(theObject) + "_properties(AbaRendererInterface theRenderer) {\r\n" +
                sSetCode + "\r\n}\r\n";
        if (theObject.isFrame() || theObject.isAbalet()) {
            theBufferCollection.sPropertyBuffer += "if (bMakeFrame)\r\n";
        }

        if (metaClassName.equalsIgnoreCase("JSSDataSource"))
            theBufferCollection.sPropertyBuffer = "set_" + getIdentifier(theObject) + "_properties(theRenderer);\r\n" + theBufferCollection.sPropertyBuffer;
        else
            theBufferCollection.sPropertyBuffer += "set_" + getIdentifier(theObject) + "_properties(theRenderer);\r\n";
    }

    public void generateAttachmentCode(MetaObject theObject) {
        int iContainerType = theObject.getContainerType();
        // Do layout construction for the current object.
        String sLayoutConstruction = "";
        switch (iContainerType) {
            case MetaContainerType.CONTAINER_TABBEDPANE:
                setContainerName(theObject, getIdentifier(theObject));
                break;
            case MetaContainerType.CONTAINER_BUTTONGROUP:
                setGroupName(theObject, "grp_" + getIdentifier(theObject));
                theBufferCollection.sDeclBuffer += "ButtonGroup " + getGroupName(theObject) + " = new ButtonGroup();";
                setContainerName(theObject, getIdentifier(theObject));
                setLayoutName(theObject, "lm_" + getIdentifier(theObject));
                theBufferCollection.sDeclBuffer += "AnchoringLayoutManager " + getLayoutName(theObject) + " = new AnchoringLayoutManager();" + "\r\n";
                sLayoutConstruction += getContainerName(theObject) + ".setLayout(" + getLayoutName(theObject) + ");\r\n";
                break;
            case MetaContainerType.CONTAINER_TABPAGE:
            case MetaContainerType.CONTAINER_PANEL:
                setLayoutName(theObject, "lm_" + getIdentifier(theObject));
                setContainerName(theObject, getIdentifier(theObject));
                theBufferCollection.sDeclBuffer += "AnchoringLayoutManager " + getLayoutName(theObject) + " = new AnchoringLayoutManager();" + "\r\n";
                sLayoutConstruction += getContainerName(theObject) + ".setLayout(" + getLayoutName(theObject) + ");\r\n";
                break;
            case MetaContainerType.CONTAINER_ABALET:
            case MetaContainerType.CONTAINER_FRAME:
                setLayoutName(theObject, "lmRenderingObject");
                setContainerName(theObject, "contentPane");
                theBufferCollection.sDeclBuffer += "AnchoringLayoutManager " + getLayoutName(theObject) + " = new AnchoringLayoutManager();" + "\r\n";
                sLayoutConstruction += getContainerName(theObject) + "= " + getIdentifier(theObject) + ".getContentPane();" + "\r\n";
                sLayoutConstruction += "theRenderer.addFocus(null," + getIdentifier(theObject) + ");\r\n";
                sLayoutConstruction += "}\r\n" + getContainerName(theObject) + ".setLayout(" + getLayoutName(theObject) + ");\r\n";
                break;
            case MetaContainerType.CONTAINER_NONE:
            case MetaContainerType.CONTAINER_SCROLLPANE:
                break;
            case MetaContainerType.CONTAINER_HSPLIT:
            case MetaContainerType.CONTAINER_VSPLIT:
                setContainerName(theObject, getIdentifier(theObject));
                break;
            case MetaContainerType.CONTAINER_SCROLLING_PANEL:
                setContainerName(theObject, getIdentifier(theObject));
                sLayoutConstruction += getContainerName(theObject) + ".setLayout(null);\r\n";
                break;
            case MetaContainerType.CONTAINER_JMENUBAR:
            case MetaContainerType.CONTAINER_JMENU:
            case MetaContainerType.CONTAINER_JMENUITEM:
            case MetaContainerType.CONTAINER_JTOOLBAR:
                setContainerName(theObject, getIdentifier(theObject));
                break;
        }
        String sVariableToAttach = getIdentifier(theObject);
        if (theObject.bAutoEmbed) {
            theBufferCollection.sDeclBuffer += "JAScrollPane scr_" + getIdentifier(theObject) + ";\r\n";
            sVariableToAttach = "scr_" + getIdentifier(theObject);
            theBufferCollection.sAssignmentBuffer += "scr_" + getIdentifier(theObject) + " = new JAScrollPane(" + getIdentifier(theObject) + ");\r\n";
            if (iContainerType == MetaContainerType.CONTAINER_SCROLLING_PANEL)
                theBufferCollection.sAssignmentBuffer += getIdentifier(theObject) +
                        ".setScrollPane(scr_" + getIdentifier(theObject) + ");\r\n";
        }
        // Do attachment.
        String sAttachmentCode = "";
        MetaObject theParentContainer = theObject.getVisualContainerObject(true);

        if (theParentContainer != null) {
            int iParentContainerType = theParentContainer.getContainerType();
            switch (iParentContainerType) {
                case MetaContainerType.CONTAINER_NONE:
                    // this should throw an exception.
                    break;
                case MetaContainerType.CONTAINER_SCROLLPANE:
                    {
                        JAScrollPane thePane = new JAScrollPane();
                        JViewport theViewport = new JViewport();
                        theViewport.setView(theObject.theVisualObject);
                        thePane.setViewport(theViewport);
                        sAttachmentCode = "JViewport vp_" + getIdentifier(theParentContainer) + "= new JViewport();\r\n";
                        sAttachmentCode += "vp_" + getIdentifier(theParentContainer) + ".setView(" + getIdentifier(theObject) + ");\r\n";
                        sAttachmentCode += getContainerName(theParentContainer) + ".setViewport(vp_" + getIdentifier(theParentContainer) + ");\r\n";
                        theParentContainer = theParentContainer.getVisualContainerObject(true);
                        sAttachmentCode += "theRenderer.addFocus(" + getIdentifier(theParentContainer) + "," + getIdentifier(theObject) + ");\r\n";
                    }
                    break;
                case MetaContainerType.CONTAINER_SCROLLING_PANEL:
                    sAttachmentCode += getContainerName(theParentContainer) + ".add(" + sVariableToAttach + ");\r\n";
                    sAttachmentCode += "theRenderer.addFocus(" + getIdentifier(theParentContainer) + "," + sVariableToAttach + ");\r\n";
                    break;
                case MetaContainerType.CONTAINER_BUTTONGROUP:
                    sAttachmentCode = "if (" + getIdentifier(theObject) + " instanceof JRadioButton)\r\n";
                    sAttachmentCode = getGroupName(theParentContainer) + ".add(" + getIdentifier(theObject) + ");\r\n";
//                    theParentContainer = theParentContainer.getVisualContainerObject(true);
                    // no break!
                case MetaContainerType.CONTAINER_PANEL:
                case MetaContainerType.CONTAINER_TABPAGE:
                case MetaContainerType.CONTAINER_ABALET:
                case MetaContainerType.CONTAINER_FRAME:
                    {
                        sAttachmentCode += getContainerName(theParentContainer) + ".add(\"" + sVariableToAttach + "\"," + sVariableToAttach + ");\r\n";
                        sAttachmentCode += getLayoutName(theParentContainer) + ".setAnchoring(" + sVariableToAttach + "," +
                                ((theObject.bLeftAnchoring) ? "true" : "false") + "," +
                                ((theObject.bRightAnchoring) ? "true" : "false") + "," +
                                ((theObject.bTopAnchoring) ? "true" : "false") + "," +
                                ((theObject.bBottomAnchoring) ? "true" : "false") + ");\r\n";
                        sAttachmentCode += "theRenderer.addFocus(" + getIdentifier(theParentContainer) + "," + sVariableToAttach + ");\r\n";
                        if(theObject.getSwingObject() instanceof JMenuBar)
                          sAttachmentCode += theObject.theParentObject.getName() + ".setJMenuBar("  + sVariableToAttach + ");\r\n";
                    }
                    break;
                case MetaContainerType.CONTAINER_TABBEDPANE:
                    {
                        int iIndex = theParentContainer.getChildIndex(theObject);
                        MetaPropertyValueEx theTabTitle = theParentContainer.getPropertyValue("TabTitle", iIndex);
                        String sTabTitle = "Tab " + iIndex;
                        if (theTabTitle != null) {
                            sTabTitle = theTabTitle.getStringValue();

                            if (isNLSTranslationRequired(theTabTitle, sTabTitle)) {
                                sAttachmentCode += getContainerName(theParentContainer) + ".addTab(theRenderer.doNLSTranslation(\"" + sTabTitle + "\"), " + getIdentifier(theObject) + ");\r\n";
                            }
                            else if (isLocalLanguageRequired(theTabTitle)) {
                                String sTestValue = "theRenderer.getCurrentLanguageValue(" +getLanguageValuesString(theTabTitle) +")";

                                sAttachmentCode += getContainerName(theParentContainer) + ".addTab(" + sTestValue + ", " + getIdentifier(theObject) + ");\r\n";
                            }
                            else { // straight text
                                sAttachmentCode += getContainerName(theParentContainer) + ".addTab(\"" + sTabTitle + "\", " + getIdentifier(theObject) + ");\r\n";
                                //System.out.println(sAttachmentCode);
                            }

                        }
                        sAttachmentCode += "theRenderer.addFocus(" + getIdentifier(theParentContainer) + "," + getIdentifier(theObject) + ");\r\n";
                    }
                    break;
                case MetaContainerType.CONTAINER_HSPLIT:
                    {
                        if (theObject.equals(theParentContainer.theFirstChild))
                            sAttachmentCode += getContainerName(theParentContainer) + ".setLeftComponent(" + sVariableToAttach + ");\r\n";
                        else {
                            sAttachmentCode += getContainerName(theParentContainer) + ".setRightComponent(" + sVariableToAttach + ");\r\n";
                            String sDividerLocation = theParentContainer.getPropertyValue("DividerLocation", 0).getLocalString();
                            sAttachmentCode += getContainerName(theParentContainer) + ".resetToPreferredSizes();\r\n";
                            sAttachmentCode += getContainerName(theParentContainer) + ".setDividerLocation(" + sDividerLocation + ");\r\n";
                        }
                    }
                    break;
                case MetaContainerType.CONTAINER_VSPLIT:
                    {
                        if (theObject.equals(theParentContainer.theFirstChild))
                            sAttachmentCode += getContainerName(theParentContainer) + ".setTopComponent(" + sVariableToAttach + ");\r\n";
                        else {
                            sAttachmentCode += getContainerName(theParentContainer) + ".setBottomComponent(" + sVariableToAttach + ");\r\n";
                            String sDividerLocation = theParentContainer.getPropertyValue("DividerLocation", 0).getLocalString();
                            sAttachmentCode += getContainerName(theParentContainer) + ".resetToPreferredSizes();\r\n";
                            sAttachmentCode += getContainerName(theParentContainer) + ".setDividerLocation(" + sDividerLocation + ");\r\n";
                        }
                    }
                    break;

                case MetaContainerType.CONTAINER_JMENUBAR:
                case MetaContainerType.CONTAINER_JMENU:
                case MetaContainerType.CONTAINER_JMENUITEM:
                case MetaContainerType.CONTAINER_JTOOLBAR:
                    {
                        sAttachmentCode += getContainerName(theParentContainer) + ".add("  + sVariableToAttach + ");\r\n";
                    }
                    break;
            }
        }
        theBufferCollection.sAssignmentBuffer += sLayoutConstruction;
        theBufferCollection.sAttachmentBuffer += sAttachmentCode;
    }

    /**
     * generateCode - This is the code generator for the optimized output option that builds java
     * programs for compilation.
     *
     */

    public void generateProjectCode(String sClassName, MetaProject theProject) throws HammerException {
        Variables = new HashMap(0);
        VariableClasses = new HashMap(0);
        vectorConstants = new Vector(0);
        generatePrologCode(theProject, sClassName);
        MetaObject objTest = theProject.getFirstObject();
        while (objTest != null) {
            generateObjectCode(objTest);
            objTest = objTest.theNextObject;
        }
        generateEpilogCode(theProject);
    }

    /**
     * doImports - This method writes imports for a given class, if necessary.
     * It adds the imports to a vector that is later processed to generate code.
     *
     * @param theProject MetaProject - this is the project to generate import statements for.
     * @param theClass  MetaClass - This is the class that is being tested for imports.
     * @param vImports  Vector - These are the imports so far.
     * @return  Vector - The imports after processing.
     */
    public Vector doImports(MetaProject theProject, MetaClass theClass, Vector vImports) {
        String sSuperClass = theClass.theMetadata.sFullClassName;
        if ((theClass.theFirstObject != null)) { // mg - the test should work!!! && theClass.hasRealObjects()) {
            if ((sSuperClass != null) && (sSuperClass.length() > 0)) {
                try {
                    theProject.getMetaDataUser().getClassLoader().getLoader().loadClass(sSuperClass);
                    // If we can load the class add the import.
                if (vImports.contains(sSuperClass) == false)
                    vImports.add(sSuperClass);
                } catch (ClassNotFoundException e) {
                    // class not found - write the class declaration so it will compile.

                    String sClassDefinition = "class " + sSuperClass.substring(sSuperClass.lastIndexOf(".")+1);
                    // MG 1/15/2005 - fix for when metadata.meta is wrong or class really has no superclass!
                    String sConstructor = "";
                    if ((theClass.theMetadata.getSuperClassName()!=null) && (theClass.theMetadata.getSuperClassName().length()>0)) {
                        sClassDefinition  += (" extends " + theClass.theMetadata.getSuperClassName());
                    }
                    else {
                        sConstructor = "public "+sSuperClass.substring(sSuperClass.lastIndexOf(".")+1)+"() {};\r\n";
                    }
                    sClassDefinition += " {\r\n"+sConstructor+"}\r\n\r\n";
                    theBufferCollection.sAppendedClassBuffer += sClassDefinition;
                }
            }
        }
        theClass = theClass.getFirstClass();
        while (theClass != null) {
            vImports = doImports(theProject, theClass, vImports);
            theClass = theClass.getNextSibling();
        }
        return vImports;
    }

    /**
     * doImports - This method collects import statements from all the classes of all the objects of
     * a project.
     *
     * @param theProject MetaProject = the project to generate imports for
     * @return String - a buffer of import statements.
     */
    public String doImports(MetaProject theProject) {
        String sImports = "";
        Vector vImports = new Vector();
        MetaClass theClass = theProject.getFirstClass();
        // make sure we don't add stupid things.
        vImports.add("java.lang.Object");
        while (theClass != null) {
            vImports = doImports(theProject, theClass, vImports);
            theClass = theClass.getNextSibling();
        }

        String externalImports  = generateExternalImports(theProject);

        sImports += "import java.awt.*;\r\n";
        sImports += "import java.util.*;\r\n";
        sImports += "import javax.swing.*;\r\n";
        sImports += "import javax.swing.event.*;\r\n";
        sImports += "import java.awt.event.*;\r\n";
        sImports += "import ch.abacus.lib.ui.renderer.abaRenderer.AbaRendererInterface;\r\n";
        sImports += "import ch.abacus.lib.ui.renderer.common.RenderingProjectInterface;\r\n";
        sImports += "import ch.abacus.lib.ui.layout.AnchoringLayoutManager;\r\n";
        if(this.getProjectDocument().isOpenSource()==true)
            sImports += "import ch.abacus.lib.ui.renderer.common.jdbc.*;\r\n"; // MHC Only add the line when open source
        sImports += externalImports; // MHC
        sImports += "import ch.abacus.lib.ui.JAScrollPane;\r\n"; // SWB. I don't really like this. It would be better to go through the classes used to see if any of them use a scrollpane and output this conditionally.
        for (int i = 0; i < vImports.size(); i++) {
            String sImport = (String) vImports.get(i);
            sImports += "import " + sImport + ";\r\n";
        }
        return sImports;
    }

    // New function for imports based on the object tree.
    //
    public String doImportsObjectBased(MetaProject theProject) {
            String sImports = "";
            String externalImports  = generateExternalImports(theProject);
            String additionalImports = "";

            externalImports = generateExternalImports(theProject);
            if(this.getProjectDocument().isOpenSource())
            {
                additionalImports = generateAdditionalImports(theProject);
            }

            sImports += "import java.awt.*;\r\n";
            sImports += "import java.util.*;\r\n";
            sImports += "import javax.swing.*;\r\n";
            sImports += "import javax.swing.event.*;\r\n";
            sImports += "import java.awt.event.*;\r\n";
            sImports += "import ch.abacus.lib.ui.renderer.abaRenderer.AbaRendererInterface;\r\n";
            sImports += "import ch.abacus.lib.ui.renderer.common.RenderingProjectInterface;\r\n";
            sImports += "import ch.abacus.lib.ui.layout.AnchoringLayoutManager;\r\n";
            if(this.getProjectDocument().isOpenSource()==true)
                sImports += "import ch.abacus.lib.ui.renderer.common.jdbc.*;\r\n"; // MHC
            sImports += externalImports; // MHC
            sImports += additionalImports; // MHC Additional imports section --- open source addition
            sImports += "import ch.abacus.lib.ui.JAScrollPane;\r\n";
            for (int i = 0; i < vRequiredImports.size(); i++) {
                    String sImport = (String) vRequiredImports.get(i);
                sImports += "import " + sImport + ";\r\n";
            }

            return sImports;
        }

    /**
     * addLocalNLSDocuments - write code to load local nls documents in rendered class.
     * @param theProject - MetaProject - the project to add nls docs for.
     */
    public void addLocalNLSDocuments(MetaProject theProject) {
        int iDocCount = theProject.theLocalNLSDocuments.size();
        for (int i = 0; i < iDocCount; i++) {
            HammerNLSAccess theNLSDoc = (HammerNLSAccess) theProject.theLocalNLSDocuments.get(i);
            theBufferCollection.sAssignmentBuffer += "theRenderer.addNLSDocument(\"" +
                    MetaConstantGroup.preserveEscapeSequences(theNLSDoc.sDocument) +
                    "\",\"" + theNLSDoc.sKey + "\");\r\n";
        }
    }

    // MHC
    public void generateRequiredImports(MetaObject topObject)
    {
        if(topObject!=null)
        {
            String importClass = topObject.theType.theFullName;
            // if class in metada is incorrect I do not handle that case. But if metadata is wrong
            // why should the project compile? Mario
            if (vRequiredImports.contains(importClass) == false)
              vRequiredImports.add(importClass);

            MetaObject objTest = topObject.theFirstChild;

            while (objTest != null) {
                generateRequiredImports(objTest);
                objTest = objTest.theNextObject;
            }
        }
     }
    // MHC //
    /**
     * generatePrologCode - This method generates the code that occurs before the class declaration.
     *
     */

    public void generatePrologCode(MetaProject theProject, String sClassName) {
         // Added function so the imports are based on  the obj tree
         //String sImports = doImports(theProject);

        generateRequiredImports(theProject.getFirstObject());
        String sImports = doImportsObjectBased(theProject);
        // make sure there is no extension trailing the classname.
        int iMatch = sClassName.indexOf('.');
        if (iMatch != -1) {
            sClassName = sClassName.substring(0, iMatch);
        }
        String sExtInits = generateExternalInit(theProject);
        theBufferCollection.sImportBuffer = sImports + "\r\n";
        theBufferCollection.sDeclBuffer += "\r\n" + "public class " + sClassName + " implements RenderingProjectInterface {\r\n";
        theBufferCollection.sAssignmentBuffer1 = "public " + sClassName + "(AbaRendererInterface theRenderer, JFrame theFrame) {\r\n";
        theBufferCollection.sAssignmentBuffer1 += "contentPane = theFrame.getContentPane();\r\n";
        theBufferCollection.sAssignmentBuffer1 += "init(theRenderer, false); \r\n}\r\n";
        theBufferCollection.sAssignmentBuffer2 = "public " + sClassName + "(AbaRendererInterface theRenderer, JComponent theComponent) {\r\n";
        theBufferCollection.sAssignmentBuffer2 += "contentPane = (Container)theComponent;\r\n";
        theBufferCollection.sAssignmentBuffer2 += "init(theRenderer, false); \r\n}\r\n";
        theBufferCollection.sAssignmentBuffer3 = "public " + sClassName + "(AbaRendererInterface theRenderer) {\r\n";
        theBufferCollection.sAssignmentBuffer3 += "init(theRenderer, true);\r\n";
        //String sMainObjectIdentifier = sClassName + "_" + theFirstObject.getName();
        String sMainObjectIdentifier = theProject.getFirstObject().getName();   // MHC
        theBufferCollection.sAssignmentBuffer3 += "// do frame insets\r\n";
        theBufferCollection.sAssignmentBuffer3 += "Dimension dimSize = " + sMainObjectIdentifier + ".getSize();\r\n";
        theBufferCollection.sAssignmentBuffer3 += sMainObjectIdentifier + ".setVisible(false);\r\n";
        theBufferCollection.sAssignmentBuffer3 += sMainObjectIdentifier + ".show();\r\n";
        theBufferCollection.sAssignmentBuffer3 += "Insets theInsets = " + sMainObjectIdentifier + ".getInsets();\r\n";
        theBufferCollection.sAssignmentBuffer3 += sMainObjectIdentifier +
                ".setSize((int) dimSize.getWidth() + theInsets.left + theInsets.right," +
                "(int) dimSize.getHeight() + theInsets.top + theInsets.bottom);\r\n" +
                sMainObjectIdentifier + ".setVisible(true);\r\n"
                + "}\r\n";
        theBufferCollection.sAssignmentBuffer = "void init(AbaRendererInterface theRenderer, boolean bMakeFrame) {\r\n";
        theBufferCollection.sAssignmentBuffer += sExtInits;

        addLocalNLSDocuments(theProject);
    }

    /**
     if (theFabricatedObject instanceof JFrame) {
     JFrame theFrame = (JFrame) theFabricatedObject;
     Dimension dimSize = theFrame.getSize();
     // This is how you have to fix the non-client area sizing.
     // It has to be _exactly_ like this.
     theFrame.setVisible(false);
     theFrame.show();
     Insets theInsets = theFrame.getInsets();
     theFrame.setSize((int) dimSize.getWidth() + theInsets.left + theInsets.right,
     (int) dimSize.getHeight() + theInsets.top + theInsets.bottom);
     }

     */

    ListenerDiscriminator findDiscriminator(ArrayList theDiscriminators, MetaClassDetail theClass) {
        int iDiscriminators = theDiscriminators.size();
        for (int i = 0; i < iDiscriminators; i++) {
            ListenerDiscriminator theDiscriminator = (ListenerDiscriminator) theDiscriminators.get(i);
            if (theDiscriminator.theClass.equals(theClass))
                return theDiscriminator;
        }
        return null;
    }

    public void listenerPrint(String sOutputString) {
        theBufferCollection.sEventClassBuffer += sOutputString;
    }

    public void listenerPrintln(String sOutputString) {
        sOutputString += "\r\n";
        listenerPrint(sOutputString);
    }

    public String generateObjectEventClasses(String sSetCode, MetaObject theObject) {
        // Make a class for each listener for the object.
        // The class name is identifier of object plus name of listener concatenated
        // Implements is the name of the listener
        // One method for each event
        // Class is written to buffer that will be added before epilog code.

        MetaMethodLinkage theMethod = theObject.theFirstMethod;
        ArrayList theListeners = new ArrayList(0);
        ArrayList theDiscriminators = new ArrayList(0);
        ArrayList theSpecialDiscriminators = new ArrayList(0);
        while (theMethod != null) {
            MetaClassDetail theClass = theMethod.theListener;
            if (theListeners.contains(theClass) == false) {
                theListeners.add(theMethod.theListener);
                ListenerDiscriminator theDiscriminator = new ListenerDiscriminator(theClass);
                theDiscriminators.add(theDiscriminator);
                theDiscriminator.addMethod(theMethod);
            } else {
                ListenerDiscriminator theDiscriminator = findDiscriminator(theDiscriminators, theClass);
                theDiscriminator.addMethod(theMethod);
            }
            theMethod = theMethod.theNextMethod;
        }
        // remove listeners that have no special implementation.
        for (int i = 0; i < theDiscriminators.size(); i++) {
            ListenerDiscriminator theDiscriminator = (ListenerDiscriminator) theDiscriminators.get(i);
            if (theDiscriminator.isSpecial() == true)
                theSpecialDiscriminators.add(theDiscriminator);
        }

        // generate classes for the others.
        for (int i = 0; i < theSpecialDiscriminators.size(); i++) {
            ListenerDiscriminator theDiscriminator = (ListenerDiscriminator) theSpecialDiscriminators.get(i);
            MetaClassDetail theListener = theDiscriminator.theClass;
            String sListenerName = theListener.sClassName;
            String sOutputIdentifier = getIdentifier(theObject);
            int iMatchingListener = theListenerCollection.getDefaultListenerClass(sListenerName);

            sSetCode += sOutputIdentifier + ".";
            ListenerEntry theEntry = theListenerCollection.objDefaultListeners[iMatchingListener];
            sSetCode += theEntry.getAddListenerMethod() + "(";
            sSetCode += "new " + sOutputIdentifier + "$$$" + sListenerName + "());\r\n";
            listenerPrintln("/***__@Listener: [" + sOutputIdentifier + "$$$" + sListenerName + "] ***/");
            listenerPrint("class " + sOutputIdentifier + "$$$" + sListenerName);
            // Check if it's a listener or an adapter.  Adapter extends.  Listener implements interface.
            boolean bIsAdapter = (sListenerName.lastIndexOf("Adapter") == -1) ? false : true;
            if (bIsAdapter)
                listenerPrintln(" extends " + sListenerName + "  {");
            else
                listenerPrintln(" implements " + sListenerName + "  {");
            // Now write the methods.
            for (int j = 0; j < theDiscriminator.theMethods.size(); j++) {
                MetaMethodLinkage theListenerMethodLink = (MetaMethodLinkage) theDiscriminator.theMethods.get(j);
                MetaMethod theListenerMethod = theListenerMethodLink.theMethod;
                String sCode = theListenerMethod.getCode();
                if (sCode.trim().length() != 0) {
                    listenerPrintln("/***__@Method: [" + theListenerMethod.getMethodName() + "] ***/");
                    listenerPrintln(sCode);
                    listenerPrintln("/***__@@Method: [" + theListenerMethod.getMethodName() + "] ***/");
                }
                theListenerMethod = theListenerMethod.getNextMethod();
            }
            // Close the class.
            listenerPrintln("  }");
            listenerPrintln("/***__@@Listener: [" + sListenerName + "] ***/");
        }
        return sSetCode;
    }


    /**
     * generateEpilogCode - This method writes any code that can only be written after processing.
     * This includes some types of declarations (constants etc) and the code that finishes the
     * class declaration.
     *
     */

    public void generateEpilogCode(MetaProject theProject) {
        String sProjectName = theProject.getName();
        int iIndex = sProjectName.indexOf(".");
        sProjectName = sProjectName.substring(0, iIndex);
        // close up the init function.
        theBufferCollection.sDeclBuffer += "Object oTemp;\r\n";
        for (int i = 0; i < vectorConstants.size(); i++) {
            String sCombinedKey = (String) vectorConstants.get(i);
            int iMatch1 = sCombinedKey.indexOf("|?|");
            int iMatch2 = sCombinedKey.indexOf("|?|", iMatch1 + 1);
            String sClassName = sCombinedKey.substring(0, iMatch1);
            String sConstantName = sCombinedKey.substring(iMatch1 + 3, iMatch2);
            String sValue = sCombinedKey.substring(iMatch2 + 3);
            theBufferCollection.sDeclBuffer += sClassName + " " + sValue + ";\r\n";
            if (sClassName.equals("int")) {
                theBufferCollection.sAssignmentBuffer += "oTemp = theRenderer.createConstantObject(\"" + sValue + "\",\"" + sConstantName + "\");\r\n";
                theBufferCollection.sAssignmentBuffer += "if (oTemp instanceof String)\r\n";
                theBufferCollection.sAssignmentBuffer += "oTemp = new Integer((String)oTemp);\r\n";
                theBufferCollection.sAssignmentBuffer += sValue + " = ((Integer)oTemp).intValue();\r\n";
            } else {
                theBufferCollection.sAssignmentBuffer += sValue + " = (" + sClassName + ") theRenderer.createConstantObject(\"" + sValue + "\",\"" + sConstantName + "\");\r\n";
            }
        }
        Object[] identifiers = Variables.keySet().toArray();
        for (int i = 0; i < identifiers.length; i++) {
            String sKey = (String) identifiers[i];
            int iMatch = sKey.indexOf("|*|");
            String sClassName = sKey.substring(0, iMatch);
            String identifier = sKey.substring(iMatch + 3);
            String sValue = (String) Variables.get(sKey);
            theBufferCollection.sDeclBuffer += sClassName + " " + identifier + ";";
            theBufferCollection.sAssignmentBuffer += identifier + " = (" + sClassName + ") theRenderer.resolveConstantValue(\"" + sValue + "\");\r\n";
        }
        theBufferCollection.sAccessorBuffer += "\r\n" + theBufferCollection.sPropertyMethodBuffer;
        // Add the event classes.
    }


    public String generatePropertyCode(MetaProperty theProperty) throws HammerException {
        String sReturnValue = "";
        int iSize = theProperty.theIndexedValue.size();
        for (int i = 0; i < iSize; i++) {
            if (sReturnValue.length() > 0)
                sReturnValue += ",";
            MetaParameter theParameter = (MetaParameter) theProperty.theIndexedValue.get(i);
            sReturnValue += generateParameterCode(theParameter, theProperty.thePropertyInfo);
        }
        return sReturnValue;
    }

    public String generateParameterCode(MetaParameter theParameter, MetaPropertyDescriptor thePropDesc) throws HammerException {
        String sTestValue = theParameter.getLiteralValue();
        if (thePropDesc == null)
            return "";
        final String theSimpleType = theParameter.theSimpleType;
        final boolean bIsString = (theSimpleType != null) && (theSimpleType.equalsIgnoreCase("java.lang.String") || (theSimpleType.equalsIgnoreCase("String")));
        if (bIsString) {
            if (sTestValue.indexOf('\\') != -1) {
                sTestValue = MetaConstantGroup.preserveEscapeSequences("\"" + sTestValue + "\"");
            } else if (sTestValue.indexOf('\"') != -1) {
                sTestValue = "\"" + MetaConstantGroup.protectQuotes(sTestValue) + "\"";
            } else if (isNLSTranslationRequired(theParameter.theValue, sTestValue)) {
                sTestValue = "theRenderer.doNLSTranslation(\"" + sTestValue + "\")";  // surround by quotes.
            } else if (isLocalLanguageRequired(theParameter.theValue)) {
                sTestValue = "theRenderer.getCurrentLanguageValue(" +getLanguageValuesString(theParameter.theValue) +")";  // surround by quotes.
            } else {
                sTestValue = "\"" + sTestValue + "\"";  // surround by quotes.
            }
        } else if (theParameter.getMnemonic() & theDocument.isOpenSource() == false) {
            if (thePropDesc.getName() != null) {
                if (thePropDesc.getName().equalsIgnoreCase("mnemonic"))
                    sTestValue = "theRenderer.doNLSMnemonicTranslation(\"" + sTestValue + "\")";  // surround by quotes.
            }
        }
        return doCodeGenerationForConstants(theParameter, theParameter.theDesignProject, sTestValue, thePropDesc);
    }

    /**
     * Returns true if the property value requires converting with NLS
     */
    private boolean isNLSTranslationRequired(MetaPropertyValueEx theValue, String sTestValue) {
        if (this.bNLSTranslation == false) // global NLS is switched off
            return false;

        // Removed for bug reported by Hervo (old projects don't have the Value's multiple languages flag always set)
        //if (theValue.isMultipleLanguages() == false) // this property does not support languages
        //    return false;

        if (sTestValue.startsWith("@AbaNLS.")) // test to ensure this string isn't a literal
            return true;

        return false;
    }

    /**
     * Returns true if the property value requires converting to a local language at runtime
     */
    private boolean isLocalLanguageRequired(MetaPropertyValueEx theValue) {
        if (this.bNLSTranslation == false && theDocument.isOpenSource()) // global NLS is switched off
            return false;

        if (theValue.isMultipleLanguages() == false) // this property does not support languages
            return false;

        String[] values = getLanguageValues(theValue);

        for (int iLanguage = 0; iLanguage < values.length -1; iLanguage++) {
            String s1 = values[iLanguage];
            String s2 = values[iLanguage +1];

            if (! (s1 != null && s1.equals(s2)))
                return true;
        }

        return false;
    }

    /**
     * Returns a String[] of the language strings
     */
    private String[] getLanguageValues(MetaPropertyValueEx theValue) {
        int iNumLanguages = HammerLanguagePresentation.LAST_LANGUAGE +1;
        String[] values = new String[iNumLanguages];
        for (int iLanguage = 0; iLanguage < iNumLanguages; iLanguage++) {
            values[iLanguage] = theValue.getLanguageDependentValue(iLanguage);
        }

        return values;
    }


    /**
     * Returns a string to output for local languages strings (that is, not NLS langauge strings)
     */
    private String getLanguageValuesString(MetaPropertyValueEx theValue) {
        String[] values = getLanguageValues(theValue);
        String s ="";
        for (int iValue = 0; iValue < values.length; iValue++) {
            if (iValue >0)
                s+= ", ";
            s += "\"" +values[iValue] +"\"";
        }

        return "new String[] {" + s +"}";

    }


    public String doCodeGenerationForConstants(MetaParameter theParameter, MetaProject theProject, String sTestValue, MetaPropertyDescriptor thePropDesc) throws HammerException {
        String sMatch = null;
        if ((thePropDesc != null) && (thePropDesc.getIncludeTypeConstants() != null)) {
            MetaConstantGroup theConstantTable = theProject.findConstantGroup(thePropDesc.getIncludeTypeConstants());
            if (theConstantTable != null) {
                sMatch = theConstantTable.getKey(sTestValue);
            } else {
                MetaConstantGroupCollection theConstantTableCollection =
                        theProject.findConstantGroupCollection(thePropDesc.getIncludeTypeConstants());
                if (theConstantTableCollection != null) {
                    sMatch = theConstantTableCollection.getKey(sTestValue);
                }
            }
            if (sMatch != null) {
                String sCombinedKey = thePropDesc.getClassName() + "|?|" + thePropDesc.getIncludeTypeConstants() + "|?|" + sMatch;
                if (isConstantMatch(sMatch) == false) {
                    vectorConstants.add(sCombinedKey);
                }
                sTestValue = sMatch;
            } else {
                // look for an object definition.  Rule out standard classes.
                Object theThing = MetaConstantGroup.resolve(sTestValue,
                        theParameter.theDesignProject.getMetadataDispenser(),
                        theParameter.theDesignProject.getMetaDataUser().getClassLoader().getLoader());
                String sClass = thePropDesc.getClassName();
                if (sClass.equals("String") == false) {
                    if (theThing instanceof String == false) {
                        // See if it is made already.
                        // If it is not present add it.
                        sMatch = lookupIdentifier(thePropDesc.getClassName(), sTestValue);
                        if (sMatch == null)
                            sMatch = makeIdentifier(thePropDesc.getClassName(), sTestValue);
                        sTestValue = sMatch;
                    }
                }
            }
        }
        else {
            String sClassName = thePropDesc.getClassName();
            if ((sClassName.endsWith("String")==false) && (sTestValue.indexOf("construct")!=-1)) {
//                Object theThing = MetaConstantGroup.resolve(sTestValue,
//                        theParameter.theDesignProject.getMetadataDispenser(),
//                        theParameter.theDesignProject.getMetaDataUser().getClassLoader().getLoader());
                sMatch = lookupIdentifier(thePropDesc.getClassName(), sTestValue);
                if (sMatch == null)
                    sMatch = makeIdentifier(thePropDesc.getClassName(), sTestValue);
                sTestValue = sMatch;
            }
        }
        return sTestValue;
    }
    /**
     * This method is used to filter duplicate symbol names so that they are not generated twice<br>
     * Only the final part of the symbol name is checked for duplication.
     * @param sMatch
     * @return true, if the symbol has alraedy been defined; false, otherwise.
     */
    private boolean isConstantMatch (String sMatch) {
        sMatch = "|" + sMatch;
        for (int i = 0 ; i < vectorConstants.size() ; i++) {
            String sElement = (String)vectorConstants.get(i);
            if (sElement.endsWith(sMatch))
                return true;
        }
        return false;

    }

    public String getDeclarationBuffer() {
        return theBufferCollection.sDeclBuffer;
    }

    public String getAssignmentBuffer() {
        return theBufferCollection.sAssignmentBuffer;
    }

    public BufferCollection getBufferCollection() {
        return theBufferCollection;
    }

    public void setAssignmentBuffer(String sAssignmentBuffer) {
        theBufferCollection.sAssignmentBuffer = sAssignmentBuffer;
    }

    public void setDeclarationBuffer(String sDeclarationBuffer) {
        theBufferCollection.sDeclBuffer = sDeclarationBuffer;
    }

    /**
     * lookupIdentifier - Simple method to search for an identifier from a hash map.  It pairs
     * the class name with the test value to return the unique key.  This is used to lookup vars for
     * constant declarations.
     *
     * @param className   String - Class name of the variable.
     * @param sTestValue  String - Value to assign.
     * @return String - the identifier associated.
     */
    public String lookupIdentifier(String className, String sTestValue) {
        return (String) Variables.get(className + "|*|" + sTestValue);
    }

    /**
     * makeIdentifier  - makes a unique identifier to use for a given value of a given class.
     * This is used to associate named values and constants in code generation.
     *
     * @param className String - Class name of the variable.
     * @param sTestValue String - Value to assign.
     * @return String - the Identifier to use to reference this value of this class.
     */
    public String makeIdentifier(String className, String sTestValue) {
        Integer classVariableCount = (Integer) VariableClasses.get(className);
        int iClassVarCount = 0;
        if (classVariableCount != null) {
            try {
                iClassVarCount = classVariableCount.intValue() + 1;
            } catch (NumberFormatException e1) {
                // stays 0
            }
        }
        String sIdentifier = className;
        int iDotPos = sIdentifier.lastIndexOf(".");
        sIdentifier = className.substring(iDotPos+1) + "_" + iClassVarCount;
        Variables.put(className + "|*|" + sIdentifier, sTestValue);
        VariableClasses.put(className, new Integer(iClassVarCount));
        return sIdentifier;
    }

    public ListenerCollection getListenerCollection() {
        return theListenerCollection;
    }

    public void setListenerCollection(ListenerCollection theListenerCollection) {
        this.theListenerCollection = theListenerCollection;
    }

    public String getFormattedSource () {                       // use default delims
        String sSourceCode = theBufferCollection.sImportBuffer +
                "\r\n" + theBufferCollection.sDeclBuffer +
                "\r\n" + theBufferCollection.sAccessorBuffer +
                "\r\n" + theBufferCollection.sAssignmentBuffer1 + "\r\n" +
                "\r\n" + theBufferCollection.sAssignmentBuffer2 + "\r\n" +
                "\r\n" + theBufferCollection.sAssignmentBuffer3 + "\r\n" +
                theBufferCollection.sEventClassBuffer + "\r\n" +
                theBufferCollection.sAssignmentBuffer + "\r\n" +
                theBufferCollection.sPropertyBuffer + "\r\n" +
                theBufferCollection.sAttachmentBuffer + "\r\n" + "}\r\n}\r\n"; //+theBufferCollection.sAppendedClassBuffer;
        com.ibm.cf.CodeFormatter theFormatter = new com.ibm.cf.CodeFormatter();
        StringReader theReader = new StringReader(sSourceCode);
        StringWriter theWriter = new StringWriter(sSourceCode.length());
        theFormatter.formatCode(theReader, theWriter);
        return theWriter.getBuffer().toString();
    }

    private String generateExternalImports(MetaProject theProject)
    {
        String retVal = "\n";

        if(this.getProjectDocument().isOpenSource()==true)
        {
            int    extImportCode    = theProject.DEFAULT_LOOK;
            String externLAF= theProject.getGenCodeLAF();
            extImportCode=theProject.getGenCodeLookFeel(externLAF);

            if(extImportCode==theProject.PLASTICXP_LOOK)
            {
                retVal="import com.jgoodies.plaf.plastic.PlasticXPLookAndFeel;\r\n";
            }
            else if(extImportCode==theProject.PLASTIC3D_LOOK)
            {
               retVal="import com.jgoodies.plaf.plastic.Plastic3DLookAndFeel;\r\n";
            }
            else if(extImportCode==theProject.PLASTIC_LOOK)
            {
               retVal="import com.jgoodies.plaf.plastic.PlasticLookAndFeel;\r\n";
            }
        }

        return retVal;
    }

    private String generateExternalInit(MetaProject theProject)
    {
        String retVal = "\n";

        if(this.getProjectDocument().isOpenSource()==true)
        {
            int    extImportCode    = theProject.DEFAULT_LOOK;
            String externLAF= theProject.getGenCodeLAF();
            extImportCode=theProject.getGenCodeLookFeel(externLAF);

            if(extImportCode==theProject.PLASTICXP_LOOK)
            {
                retVal= "try {UIManager.setLookAndFeel(new PlasticXPLookAndFeel());}catch (Exception e) {} \n\r"; // MHC
            }
            else if(extImportCode==theProject.PLASTIC3D_LOOK)
            {
               retVal="try {UIManager.setLookAndFeel(new Plastic3DLookAndFeel());}catch (Exception e) {} \n\r";
            }
            else if(extImportCode==theProject.PLASTIC_LOOK)
            {
               retVal="try {UIManager.setLookAndFeel(new PlasticLookAndFeel());}catch (Exception e) {} \n\r";
            }
        }

        return retVal;
    }

    private String generateAdditionalImports(MetaProject theProject)
    {
        String retVal = "\n";

        retVal=theProject.getAdditionalImports() + "\n";

        return retVal;
    }

    public void reset() {
        theBufferCollection = new BufferCollection();
    }
}