/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

/* $Header: /cvs/v2006_cdnew/aba/java/src/ch/abacus/lib/ui/renderer/programmableCompiler/Test.java,v 1.6 2004/08/31 16:25:50 gouker Exp $
 * $Log: Test.java,v $
 * Revision 1.6  2004/08/31 16:25:50  gouker
 * Changes for optimization.  Droplet code generator changes.
 * This code still has some debugging that must be removed.
 *
 * Revision 1.5  2003/12/12 16:26:56  gouker
 * Moved source code from design cockpit to renderer
 * Merged in all 11/30 fixes for resource losses
 * Added metadata dispenser
 *
 * Revision 1.1  2003/07/02 18:53:18  gouker
 * Design Cockpit - New check in after refactoring.  In progress still so don't even try to build yet.
 *
 * Thanks!
 *
 * Revision 1.1  2003/03/24 14:59:57  gouker
 * First time checking in design cockpit.  This time it allows renderer.
 *
 * Revision 1.2  1998/11/06 10:37:03  graham
 * Added support for multiple packages and inner classes.
 *
 * Revision 1.1  1998/07/17 09:00:00  graham
 * Initial revision
 *
 */

/* Copyright (C) 1998 Graham Kirby
 *
 * This library is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Library General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE. See the GNU Library General Public License for more details.
 *
 * To receive a copy of the GNU Library General Public License, write to the Free
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */

package ch.abacus.lib.ui.renderer.programmableCompiler;

import java.io.File;
import java.lang.reflect.Method;

/**
 * Tests the <A HREF="ProgrammableCompiler.html">ProgrammableCompiler</A> class.<P>
 *
 * The <A HREF="../../compiler/Test.java">source code</A> is available.
 *
 * @author	Graham Kirby (<A HREF="mailto:graham@dcs.st-and.ac.uk">graham@dcs.st-and.ac.uk</A>)
 * @version 1.2 2-Nov-98
 */
public class Test {

    /**
     * Illustrates various uses of the <A HREF="ProgrammableCompiler.html">ProgrammableCompiler</A> class.
     * Prints results of testing some of the following options:<P>
     *
     * <UL>
     *   <LI>single class / multiple classes
     *   <LI>no package / single package / multiple packages
     *   <LI>single / multiple compilation by same compiler
     *   <LI>multiple compilation by different compilers
     *   <LI>static / non-static method invoked
     *   <LI>void method / result returned
     *   <LI>compiled classes stated explicitly / inferred
     *   <LI>returned classes stated explicitly / inferred
     *   <LI>single class per definition / multiple classes per definition
     *   <LI>inner classes / no inner classes
     *   <LI>direct class / shell invocation
     *   <LI>compilation errors / no errors
     * </UL>
     *
     * @param args	the command line arguments, which are ignored
     */
    public static void main(String args[]) {

        String packageHeader = "package p.q.r;\n";

        // Single class.
        String defn1 = "public class Test1 {\n" +
                "  public static void m1() {\n" +
                "    System.out.println (\"Test OK.\");\n" +
                "  }\n" +
                "\n" +
                "  public void m2() {\n" +
                "    System.out.println (\"Test OK.\");\n" +
                "  }\n" +
                "\n" +
                "  public String m3 (String s) {\n" +
                "    return \"Test OK.\";\n" +
                "  }\n" +
                "}\n";

        // Call to a different class.
        String defn2 = "public class Test2 {\n" +
                "  public static void m1() {\n" +
                "    Test1 t = new Test1();\n" +
                "    t.m2 ();\n" +
                "  }\n" +
                "}\n";

        // Call to a different package.
        String defn3 = "public class Test3 {\n" +
                "  public static void m1() {\n" +
                "    p.q.r.Test2 t = new p.q.r.Test2();\n" +
                "    t.m1 ();\n" +
                "  }\n" +
                "}\n";

        // Non-public class.
        String defn4 = "public class Test4 {\n" +
                "  public static void m1() {\n" +
                "    Test5 t = new Test5();\n" +
                "    t.m1 ();\n" +
                "  }\n" +
                "}\n" +
                "\n" +
                "class Test5 {\n" +
                "  void m1() {\n" +
                "    System.out.println (\"Test OK.\");\n" +
                "  }\n" +
                "}\n";

        // Inner class.
        String defn5 = "public class Test6 {\n" +
                "  public void m1() {\n" +
                "    Test7 t = new Test7();\n" +
                "    t.m1 ();\n" +
                "  }\n" +
                "\n" +
                "  class Test7 {\n" +
                "    void m1() {\n" +
                "      System.out.println (\"Test OK.\");\n" +
                "    }\n" +
                "  }\n" +
                "}\n";

        // Mutually recursive packages.
        String defn6 = "package a.b.c;\n" +
                "import d.e.f.*;\n" +
                "public class Test8 {\n" +
                "  public void m1() {\n" +
                "    Test9 t = new Test9 (this);\n" +
                "    t.m1 ();\n" +
                "  }\n" +
                "\n" +
                "  public void m2() {\n" +
                "    System.out.println (\"Test OK.\");\n" +
                "  }\n" +
                "}\n";

        String defn7 = "package d.e.f;\n" +
                "import a.b.c.*;\n" +
                "public class Test9 {\n" +
                "  Test8 t8;\n" +
                "  public Test9 (Test8 t8) {\n" +
                "    this.t8 = t8;\n" +
                "  }\n" +
                "\n" +
                "  public void m1() {\n" +
                "    t8.m2 ();\n" +
                "  }\n" +
                "}\n";

        /************************************************************************************************************/

        // Run finalizers on exit to clean up transient files.
        //System.runFinalizersOnExit (true);

        /************************************************************************************************************/

        // Simplest test - single class, no package, calling static void method.

        try {

            System.out.println("Single class.");

            Class c = (new ProgrammableCompiler()).compileClass(defn1);
            Method m = c.getMethod("m1", new Class[0]);
            m.invoke(null, new Object[0]);

            System.out.println();
        } catch (Exception e) {
            System.out.println("Exception: " + e.getMessage());
        }

        /************************************************************************************************************/

        // Same within a package.

        try {

            System.out.println("Single class in package.");

            Class c = (new ProgrammableCompiler()).compileClass(packageHeader + defn1);
            Method m = c.getMethod("m1", new Class[0]);
            m.invoke(null, new Object[0]);

            System.out.println();
        } catch (Exception e) {
            System.out.println("Exception: " + e.getMessage());
        }

        /************************************************************************************************************/

        // Non static method.

        try {

            System.out.println("Single class / non-static method.");

            Class c = (new ProgrammableCompiler()).compileClass(defn1);
            Method m = c.getMethod("m2", new Class[0]);
            Object o = c.newInstance();
            m.invoke(o, new Object[0]);

            System.out.println();
        } catch (Exception e) {
            System.out.println("Exception: " + e.getMessage());
        }

        /************************************************************************************************************/

        // Parameter and non-void result.

        try {

            System.out.println("Single class / parameter and result.");

            Class c = (new ProgrammableCompiler()).compileClass(defn1);
            Object[] params = {"test parameter"};
            Class[] paramTypes = {params[0].getClass()};
            Method m = c.getMethod("m3", paramTypes);
            Object o = c.newInstance();
            String s = (String) m.invoke(o, params);

            System.out.println(s);
            System.out.println();
        } catch (Exception e) {
            System.out.println("Exception: " + e.getMessage());
        }

        /************************************************************************************************************/

        // Two class definitions.

        try {

            System.out.println("Two classes.");

            String[] defns = {defn1, defn2};
            Class c = (new ProgrammableCompiler()).compileClass(defns, "Test2");
            Method m = c.getMethod("m1", new Class[0]);
            Object o = c.newInstance();
            String s = (String) m.invoke(o, new Object[0]);

            System.out.println();
        } catch (Exception e) {
            System.out.println("Exception: " + e.getMessage());
        }

        /************************************************************************************************************/

        // Separate compilation.

        try {

            System.out.println("Separate compilation.");

            ProgrammableCompiler dc = new ProgrammableCompiler();

            String[] defns = {packageHeader + defn1, packageHeader + defn2};
            Class c1 = dc.compileClass(defns, "p.q.r.Test2");

            Class c2 = dc.compileClass(defn3);
            Method m = c2.getMethod("m1", new Class[0]);
            Object o = c2.newInstance();
            String s = (String) m.invoke(o, new Object[0]);
            System.out.println();
        } catch (Exception e) {
            System.out.println("Exception: " + e.getMessage());
        }

        /************************************************************************************************************/

        // Multiple classes in same definition.

        try {

            System.out.println("Multiple classes in same definition.");

            ProgrammableCompiler dc = new ProgrammableCompiler();

            // Main class inferred.
            Class c1 = dc.compileClass(defn4);

            // Main class explicit.
            Class c2 = dc.compileClass(defn4, "Test4");
            Method m = c2.getMethod("m1", new Class[0]);
            Object o = c2.newInstance();
            m.invoke(o, new Object[0]);

            System.out.println();
        } catch (Exception e) {
            System.out.println("Exception: " + e.getMessage());
        }

        /************************************************************************************************************/

        // Multiple classes, multiple definitions.

        try {

            System.out.println("Multiple classes, multiple definitions.");

            String[] defns = {defn1, defn4};
            String[] mainClasses = {"Test1", "Test4"};

            ProgrammableCompiler dc = new ProgrammableCompiler();

            Class c = dc.compileClass(defns, mainClasses, "Test4");
            Method m = c.getMethod("m1", new Class[0]);
            Object o = c.newInstance();
            m.invoke(o, new Object[0]);

            System.out.println();
        } catch (Exception e) {
            System.out.println("Exception: " + e.getMessage());
        }

        /************************************************************************************************************/

        // Multiple classes returned.

        try {

            System.out.println("Multiple classes returned.");

            String[] defns = {defn1, defn4};
            String[] mainClasses = {"Test1", "Test4"};
            String[] expectedClasses = {"Test1", "Test4", "Test5"};

            ProgrammableCompiler dc = new ProgrammableCompiler();

            Class[] classes1 = dc.compileClasses(defns);

            checkClassNames(classes1, expectedClasses);

            Class[] classes2 = dc.compileClasses(defns, mainClasses);

            checkClassNames(classes2, expectedClasses);

            System.out.println();
        } catch (Exception e) {
            System.out.println("Exception: " + e.getMessage());
        }

        /************************************************************************************************************/

        // Multiple specified classes returned.

        try {

            System.out.println("Multiple specified classes returned.");

            String[] defns = {defn1, defn4};
            String[] mainClasses = {"Test1", "Test4"};

            ProgrammableCompiler dc = new ProgrammableCompiler();

            Class[] classes = dc.compileSpecificClasses(defns, mainClasses);

            checkClassNames(classes, mainClasses);

            System.out.println();
        } catch (Exception e) {
            System.out.println("Exception: " + e.getMessage());
        }

        /************************************************************************************************************/

        // Multiple classes specified and returned.

        try {

            System.out.println("Multiple classes specified and returned.");

            String[] defns = {defn1, defn4};
            String[] mainClasses = {"Test1", "Test4"};

            ProgrammableCompiler dc = new ProgrammableCompiler();

            Class[] classes = dc.compileClasses(defns, mainClasses, mainClasses);

            checkClassNames(classes, mainClasses);

            System.out.println();
        } catch (Exception e) {
            System.out.println("Exception: " + e.getMessage());
        }

        /************************************************************************************************************/

        // Multiple recursive packages.

        try {

            System.out.println("Multiple recursive packages.");

            String[] defns = {defn6, defn7};

            ProgrammableCompiler dc = new ProgrammableCompiler();

            Class c = dc.compileClass(defns, "a.b.c.Test8");
            Method m = c.getMethod("m1", new Class[0]);
            Object o = c.newInstance();
            m.invoke(o, new Object[0]);

            System.out.println();
        } catch (Exception e) {
            System.out.println("Exception: " + e.getMessage());
        }

        /************************************************************************************************************/

        // Multiple independent compilations.

        try {

            System.out.println("Multiple independent compilations.");

            String[] defns1 = {defn6, defn7};
            String[] defns2 = {defn1, defn4};
            String[] expectedClasses = {"Test1", "Test4", "Test5"};

            ProgrammableCompiler dc = new ProgrammableCompiler();

            Class[] classes1 = dc.compileClasses(defns1);
            Class[] classes2 = dc.compileClasses(defns2);

            checkClassNames(classes2, expectedClasses);

            System.out.println();
        } catch (Exception e) {
            System.out.println("Exception: " + e.getMessage());
        }

        /************************************************************************************************************/

        // Inner class.

        try {

            System.out.println("Inner class.");

            String[] defns = {defn5};
            String[] expectedClasses = {"Test6$Test7", "Test6"};

            ProgrammableCompiler dc = new ProgrammableCompiler();

            Class[] classes = dc.compileClasses(defns);

            checkClassNames(classes, expectedClasses);

            System.out.println();
        } catch (Exception e) {
            System.out.println("Exception: " + e.getMessage());
        }

        /************************************************************************************************************/

        // Compile to byte arrays.

        try {

            System.out.println("Compile to byte arrays.");

            String[] defns = {defn1, defn4};
            int[] expectedByteArrayLengths = {574, 320, 442};

            ProgrammableCompiler dc = new ProgrammableCompiler();

            byte[][] byteArrays = dc.compileClassesToBytes(defns);

            checkByteArrayLengths(byteArrays, expectedByteArrayLengths);

            System.out.println();
        } catch (Exception e) {
            System.out.println("Exception: " + e.getMessage());
        }

        /************************************************************************************************************/

        // Compile to files.


        try {

            System.out.println("Compile to files.");

            String[] defns = {defn1, defn4};
            int[] expectedFileLengths = {574, 320, 442};

            ProgrammableCompiler dc = new ProgrammableCompiler();

            File[] files = dc.compileClassesToFiles(defns);

            checkFileLengths(files, expectedFileLengths);

            System.out.println();
        } catch (Exception e) {
            System.out.println("Exception: " + e.getMessage());
        }

        /************************************************************************************************************/

        // Manually setting class path.

        try {

            System.out.println("Manually setting class path.");

            ProgrammableCompiler dc1 = new ProgrammableCompiler();

            String[] defns = {packageHeader + defn1, packageHeader + defn2};
            Class c1 = dc1.compileClass(defns, "p.q.r.Test2");

            ProgrammableCompiler dc2 = new ProgrammableCompiler();

            dc2.classPathVector.insertElementAt(dc1.classPathVector.elementAt(0), 0);

            Class c2 = dc2.compileClass(defn3);
            Method m = c2.getMethod("m1", new Class[0]);
            Object o = c2.newInstance();
            String s = (String) m.invoke(o, new Object[0]);

            System.out.println();
        } catch (Exception e) {
            System.out.println("Exception: " + e.getMessage());
        }

        /************************************************************************************************************/

        // Compiler class invocation only.

        try {

            System.out.println("Compiler class invocation only.");

            ProgrammableCompiler dc = new ProgrammableCompiler();

            if (!dc.isCompilerClassAvailable())
                System.out.println("Class invocation not available.");
            else {
                dc.useCompilerClass = true;
                dc.useShell = false;

                Class c = dc.compileClass(defn1);
                Method m = c.getMethod("m1", new Class[0]);
                m.invoke(null, new Object[0]);
            }

            System.out.println();
        } catch (Exception e) {
            System.out.println("Exception: " + e.getMessage());
        }

        /************************************************************************************************************/

        // Shell invocation only.

        try {

            System.out.println("Shell invocation only.");

            ProgrammableCompiler dc = new ProgrammableCompiler();

            if (!dc.isShellAvailable())
                System.out.println("Shell invocation not available.");
            else {
                dc.useCompilerClass = false;
                dc.useShell = true;

                Class c = dc.compileClass(defn1);
                Method m = c.getMethod("m1", new Class[0]);
                m.invoke(null, new Object[0]);
            }

            System.out.println();
        } catch (Exception e) {
            System.out.println("Exception: " + e.getMessage());
        }

        /************************************************************************************************************/

        // Invalid class definitions (compilation should fail).

        try {

            System.out.println("Invalid class definitions (compilation should fail).");

            String[] defns = {"xxx " + defn1, "yyy " + defn4};

            ProgrammableCompiler dc = new ProgrammableCompiler();

            Class[] classes = dc.compileClasses(defns);

            System.out.println();
        } catch (Exception e) {
            System.out.println("Exception: " + e.getMessage());
        }

        /************************************************************************************************************/

        System.out.println("Finished testing.");

    }

    protected static void printClassNames(Class[] classes) {

        System.out.println("Classes returned:\n");

        for (int i = 0; i < classes.length; i++) {

            System.out.println("   " + classes[i].getName());
        }
    }

    protected static void printBytes(byte[][] byteArrays) {

        System.out.println("Byte arrays:\n");

        for (int i = 0; i < byteArrays.length; i++) {

            System.out.println("Array length: " + byteArrays[i].length);

            for (int j = 0; j < byteArrays[i].length; j++) {

                System.out.print(byteArrays[i][j]);
                System.out.print(" ");
            }

            System.out.println();
        }
    }

    protected static void checkClassNames(Class[] classes, String[] expectedClassNames) throws Exception {

        if (classes.length != expectedClassNames.length) throw new Exception("Unexpected number of classes");

        String[] generatedClassNames = new String[classes.length];
        for (int i = 0; i < classes.length; i++) generatedClassNames[i] = classes[i].getName();

        sort(generatedClassNames);
        sort(expectedClassNames);

        for (int i = 0; i < classes.length; i++)
            if (!generatedClassNames[i].equals(expectedClassNames[i])) throw new Exception("Unexpected class");

        System.out.println("Test OK.");
    }

    protected static void sort(String[] strings) {

        int noStrings = strings.length;

        for (int i = 0; i < noStrings - 1; i++)
            for (int j = 0; j < noStrings - i - 1; j++)

                if (strings[j].compareTo(strings[j + 1]) > 0) {

                    String temp = strings[j + 1];
                    strings[j + 1] = strings[j];
                    strings[j] = temp;
                }
    }

    protected static void checkByteArrayLengths(byte[][] byteArrays, int[] expectedByteArrayLengths) throws Exception {

        if (byteArrays.length != expectedByteArrayLengths.length) throw new Exception("Unexpected number of byte arrays");

        for (int i = 0; i < byteArrays.length; i++) {

            if (byteArrays[i].length != expectedByteArrayLengths[i]) throw new Exception("Unexpected byte array length");
        }

        System.out.println("Test OK.");
    }

    protected static void checkFileLengths(File[] files, int[] expectedFileLengths) throws Exception {

        if (files.length != expectedFileLengths.length) throw new Exception("Unexpected number of files");

        for (int i = 0; i < files.length; i++) {

            if (files[i].length() != expectedFileLengths[i]) throw new Exception("Unexpected file length");
        }

        System.out.println("Test OK.");
    }
}
