/*
 * Copyright (C) 2003-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.model;

/**
 * A form event.
 *
 * @author Karl Tauber
 * @since 2.0
 */
public class FormEvent
{
	public static final String FIELD_HANDLER = "handler";
	public static final String FIELD_PASS_PARAMS = "passParams";
	/** @since 3.0 */
	public static final String FIELD_PROPERTY_NAME = "propertyName";

	static final FormEvent[] EMPTY_EVENTS = {};

	private final String listener;
	private final String listenerMethod;
	private String handler;
	private boolean passParams;
	private String propertyName;

	private FormComponent component;

	/**
	 * Constructs a form event.
	 *
	 * @param listener The name of the listener interface
	 *                 (e.g. "java.awt.event.ActionListener").
	 * @param listenerMethod The name of the method in the listener interface
	 *                 that should trigger the handler (e.g. "actionPerformed").
	 * @param handler The name of the event handler method.
	 * @param passParams If true, pass listener method parameters to the handler method.
	 */
	public FormEvent( String listener, String listenerMethod, String handler, boolean passParams ) {
		this( listener, listenerMethod, handler, passParams, null );
	}

	/**
	 * Constructs a form event.
	 *
	 * @param listener The name of the listener interface
	 *                 (e.g. "java.awt.event.ActionListener").
	 * @param listenerMethod The name of the method in the listener interface
	 *                 that should trigger the handler (e.g. "actionPerformed").
	 * @param handler The name of the event handler method.
	 * @param passParams If true, pass listener method parameters to the handler method.
	 * @param propertyName The property name used to add PropertyChangeListeners.
	 * @since 3.0
	 */
	public FormEvent( String listener, String listenerMethod, String handler,
		boolean passParams, String propertyName )
	{
		this.listener = listener;
		this.listenerMethod = listenerMethod;
		this.handler = handler;
		this.passParams = passParams;
		this.propertyName = propertyName;
	}

	/**
	 * Copy constructor.
	 */
	public FormEvent( FormEvent orig ) {
		listener = orig.listener;
		listenerMethod = orig.listenerMethod;
		handler = orig.handler;
		passParams = orig.passParams;
		propertyName = orig.propertyName;
	}

	/**
	 * Returns the name of the listener interface
	 * (e.g. "java.awt.event.ActionListener").
	 */
	public String getListener() {
		return listener;
	}

	/**
	 * Returns the name of the method in the listener interface that should
	 * trigger the handler (e.g. "actionPerformed").
	 */
	public String getListenerMethod() {
		return listenerMethod;
	}

	/**
	 * Returns the name of the method that the event handler will invoke on the target.
	 */
	public String getHandler() {
		return handler;
	}

	/**
	 * Sets the name of the event handler method.
	 */
	public void setHandler( String handler ) {
		if( FormObject.safeEquals( this.handler, handler ) )
			return;

		String oldHandler = this.handler;
		this.handler = handler;

		fireFieldChanged( FIELD_HANDLER, oldHandler, handler );
	}

	/**
	 * Returns whether the listener method parameters should passed to the handler method.
	 */
	public boolean getPassParams() {
		return passParams;
	}

	/**
	 * Sets whether the listener method parameters should passed to the handler method.
	 */
	public void setPassParams( boolean passParams ) {
		if( this.passParams == passParams )
			return;

		boolean oldPassParams = this.passParams;
		this.passParams = passParams;

		fireFieldChanged( FIELD_PASS_PARAMS, Boolean.valueOf( oldPassParams ),
											 Boolean.valueOf( passParams ) );
	}

	/**
	 * Returns the property name used to add PropertyChangeListeners.
	 * @since 3.0
	 */
	public String getPropertyName() {
		return propertyName;
	}

	/**
	 * Sets the property name used to add PropertyChangeListeners.
	 * @since 3.0
	 */
	public void setPropertyName( String propertyName ) {
		if( FormObject.safeEquals( this.propertyName, propertyName ) )
			return;

		String oldPropertyName = this.propertyName;
		this.propertyName = propertyName;

		fireFieldChanged( FIELD_PROPERTY_NAME, oldPropertyName, propertyName );
	}

	/**
	 * Returns the form component for this object.
	 */
	public FormComponent getComponent() {
		return component;
	}

	void setComponent( FormComponent component ) {
		if( this.component != null && component != null )
			throw new IllegalStateException( "Already attached (current=\""
						+ this.component + "\", new=\"" + component + "\")." );

		this.component = component;
	}

	private void fireFieldChanged( String fieldName, Object oldValue, Object newValue ) {
		FormModel model = (component != null) ? component.getModel() : null;
		if( model != null && model.eventProvider != null )
			model.eventProvider.fireEventFieldChanged( this, fieldName, oldValue, newValue );
	}

	/**
	 * Returns a string representation of the object.
	 */
	@Override
	public String toString() {
		return FormObject.unqualifiedClassName( getClass() )
			+ "[listener=" + listener + ",listenerMethod=" + listenerMethod
			+ ",handler=" + handler + ",passParams=" + passParams + "] "
			+ super.toString();
	}
}
