/*
 * Copyright (C) 2003-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.model;

/**
 * A form model represents a JFormDesigner form in memory.
 * <p>
 * {@link com.jformdesigner.runtime.FormLoader} can be used to load a .jfd file and
 * {@link com.jformdesigner.runtime.FormSaver} to save a form model to a .jfd file.
 * {@link com.jformdesigner.runtime.FormCreator} can create Swing components from
 * a form model.
 * <p>
 * Example:
 * <pre>
 *   FormContainer panel = ... // see {@link FormContainer}
 *
 *   FormRoot root = new FormRoot();
 *   root.add(panel);
 *
 *   FormModel model = new FormModel();
 *   model.setContentType("form/swing");
 *   model.setRoot(root);
 * </pre>
 *
 * @author Karl Tauber
 */
public class FormModel
	extends FormObject
{
	private ClassLoader classLoader;
	private String contentType;
	private FormRoot root;

	FormModelEventProvider eventProvider;
	private boolean undoRecording = true;

	FormNameRegistry nameRegistry;

	/**
	 * The header comment of the file or null.
	 *
	 * @since 3.0
	 */
	public String fileHeader;

	/**
	 * Constructs a form model.
	 */
	public FormModel() {
		// Initialize to context class loader of current thread because
		// XMLDecoder uses this class loader to load classes.
		this.classLoader = Thread.currentThread().getContextClassLoader();
	}

	/**
	 * Returns the class loader used to load the form model.
	 */
	public ClassLoader getClassLoader() {
		return classLoader;
	}

	/**
	 * For internal use only.
	 */
	// not named setClassLoader() to avoid that XMLEncoder tries to encode it
	public void set_ClassLoader( ClassLoader classLoader ) {
		this.classLoader = classLoader;
	}

	/**
	 * Returns the content type of the form model.
	 */
	public String getContentType() {
		return contentType;
	}

	/**
	 * Sets the content type of the form model.
	 * <p>
	 * Use <code>form/swing</code> for Swing forms.
	 */
	public void setContentType( String contentType ) {
		this.contentType = contentType;
	}

	/**
	 * Returns the root of this form model.
	 */
	public FormRoot getRoot() {
		return root;
	}

	/**
	 * Sets the root of this form model.
	 * The root is a container for all top-level components in a form model.
	 */
	public void setRoot( FormRoot root ) {
		if( this.root == root )
			return;

		if( this.root != null )
			this.root.setModel( null );

		this.root = root;
		nameRegistry = null;

		if( root != null )
			root.setModel( this );

		if( eventProvider != null )
			eventProvider.fireFormChanged();
	}

	/**
	 * Returns the form component for the specified component name.
	 */
	public FormComponent getFormComponent( String name ) {
		if( nameRegistry == null )
			nameRegistry = new FormNameRegistry( this );
		return nameRegistry.getFormComponent( name );
	}

	/**
	 * Returns all form component names used in this form model.
	 */
	public String[] getNames() {
		if( nameRegistry == null )
			nameRegistry = new FormNameRegistry( this );
		return nameRegistry.getNames();
	}

	/**
	 * Accepts the given visitor.
	 * The visitor's {@link FormComponentVisitor#visit} is called
	 * with all form components of this form model.
	 *
	 * @param visitor The visitor.
	 * @return The result of {@link FormComponentVisitor#visit}.
	 * @since 3.0
	 */
	public boolean accept( FormComponentVisitor visitor ) {
		if( root == null )
			return true;
		return root.accept( visitor );
	}

	/**
	 * Accepts the given visitor.
	 * The visitor's {@link FormBindingVisitor#visit} is called
	 * with all form bindings of this form model.
	 *
	 * @param visitor The visitor.
	 * @return The result of {@link FormBindingVisitor#visit}.
	 * @since 5.0
	 */
	public boolean accept( FormBindingVisitor visitor ) {
		if( root == null )
			return true;
		return root.accept( visitor );
	}

	// necessary for FormObject
	@Override
	void firePropertyChanged( String name, int index, Object oldValue, Object newValue ) {
		if( eventProvider != null )
			eventProvider.fireModelPropertyChanged( name, index, oldValue, newValue );
	}

	/**
	 * For internal use only.
	 */
	public boolean isUndoRecording() {
		return undoRecording;
	}

	/**
	 * For internal use only.
	 */
	public boolean setUndoRecording( boolean undoRecording ) {
		boolean oldUndoRecording = this.undoRecording;
		this.undoRecording = undoRecording;
		return oldUndoRecording;
	}

	@Override
	void updateReferences( String oldName, String newName ) {
		if( root != null && undoRecording )
			root.updateReferences( oldName, newName );
	}

	/**
	 * For internal use only.
	 *
	 * @since 3.1
	 */
	public FormModelEventProvider getEventProvider() {
		return eventProvider;
	}

	/**
	 * For internal use only.
	 *
	 * @since 3.1
	 */
	// not named setEventProvider() to avoid that XMLEncoder tries to encode it
	public void set_EventProvider( FormModelEventProvider eventProvider ) {
		if( this.eventProvider != null )
			throw new IllegalStateException();

		this.eventProvider = eventProvider;
	}
}
