/*
 * Copyright (C) 2003-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.model;

import java.util.HashMap;

/**
 * @author Karl Tauber
 */
final class FormNameRegistry
{
	/** Maps names to form components (String, FormComponent). */
	private final HashMap<String, FormComponent> namesMap = new HashMap<String, FormComponent>();

	FormNameRegistry( FormModel model ) {
		FormContainer root = model.getRoot();
		if( root != null )
			addNames( root );
	}

	private void addNames( FormComponent component ) {
		String name = component.getName();
		if( name != null )
			namesMap.put( name, component );

		if( component instanceof FormContainer ) {
			FormContainer container = (FormContainer) component;

			if( container.getMenuBar() != null )
				addNames( container.getMenuBar() );

			int count = container.getComponentCount();
			for( int i = 0; i < count; i++ )
				addNames( container.getComponent( i ) );
		}
	}

	private void removeNames( FormComponent component ) {
		String name = component.getName();
		if( name != null )
			namesMap.remove( name );

		if( component instanceof FormContainer ) {
			FormContainer container = (FormContainer) component;

			if( container.getMenuBar() != null )
				removeNames( container.getMenuBar() );

			int count = container.getComponentCount();
			for( int i = 0; i < count; i++ )
				removeNames( container.getComponent( i ) );
		}
	}

	FormComponent getFormComponent( String name ) {
		return namesMap.get( name );
	}

	String[] getNames() {
		return namesMap.keySet().toArray( new String[namesMap.size()] );
	}

	void componentAdded( FormComponent component ) {
		addNames( component );
	}

	void componentRemoved( FormComponent component ) {
		removeNames( component );
	}

	void componentFieldChanged( FormComponent component, String fieldName,
		Object oldValue, Object newValue )
	{
		if( fieldName == FormComponent.FIELD_NAME ) {
			// renamed
			namesMap.remove( oldValue );
			namesMap.put( (String) newValue, component );
		} else if( fieldName == FormContainer.FIELD_MENU_BAR ) {
			if( oldValue != null )
				removeNames( (FormComponent) oldValue );
			if( newValue != null )
				addNames( (FormComponent) newValue );
		}
	}
}
