/*
 * Copyright (C) 2003-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.model;

import java.awt.Font;

/**
 * Used to specify derived fonts.
 * <p>
 * This class extends java.awt.Font so that it can be used everywhere,
 * but especially as title font in TitledBorder.
 *
 * @author Karl Tauber
 * @since 3.1
 */
public class SwingDerivedFont
	extends Font
{
	/** The new font name; or null. */
	private final String nameChange;

	/**
	 * The font style change; or zero.
	 * The lower 16 bits are added; the upper 16 bits are removed.
	 */
	private final int styleChange;

	/** The absolute or relative font size; or zero. */
	private final int sizeChange;

	/** Specifies whether the size changes absolute or relative. */
	private final boolean absoluteSize;

	/**
	 * Constructs a derived font.
	 *
	 * @param nameChange The new font name; or null.
	 * @param styleChange The font style change; or zero.
	 *        The lower 16 bits are added; the upper 16 bits are removed.
	 * @param sizeChange The absolute or relative font size; or zero.
	 * @param absoluteSize Specifies whether the size changes absolute or relative.
	 */
	public SwingDerivedFont( String nameChange, int styleChange, int sizeChange,
		boolean absoluteSize )
	{
		super( "Dialog", Font.PLAIN, 12 );

		this.nameChange = nameChange;
		this.styleChange = styleChange;
		this.sizeChange = sizeChange;
		this.absoluteSize = absoluteSize;
	}

	/**
	 * Constructs a derived font based on the given font.
	 *
	 * @param font The base font.
	 * @param nameChange The new font name; or null.
	 * @param styleChange The font style change; or zero.
	 *        The lower 16 bits are added; the upper 16 bits are removed.
	 * @param sizeChange The absolute or relative font size; or zero.
	 * @param absoluteSize Specifies whether the size changes absolute or relative.
	 */
	public SwingDerivedFont( Font font, String nameChange, int styleChange,
		int sizeChange, boolean absoluteSize )
	{
		super( (nameChange != null) ? nameChange : font.getName(),
			font.getStyle() & ~((styleChange >> 16) & 0xffff) | (styleChange & 0xffff),
			absoluteSize ? sizeChange : (font.getSize() + sizeChange) );

		this.nameChange = nameChange;
		this.styleChange = styleChange;
		this.sizeChange = sizeChange;
		this.absoluteSize = absoluteSize;
	}

	/**
	 * Returns the new font name; or null.
	 */
	public String getNameChange() {
		return nameChange;
	}

	/**
	 * Returns the font style change; or zero.
	 * The lower 16 bits are added; the upper 16 bits are removed.
	 */
	public int getStyleChange() {
		return styleChange;
	}

	/**
	 * Returns the absolute or relative font size; or zero.
	 */
	public int getSizeChange() {
		return sizeChange;
	}

	/**
	 * Returns whether the size changes absolute or relative.
	 */
	public boolean isAbsoluteSize() {
		return absoluteSize;
	}

	/**
	 * Creates a derived font.
	 */
	public Font derive( Font font ) {
		if( font == null )
			return this;

		int style = font.getStyle() & ~((styleChange >> 16) & 0xffff) | (styleChange & 0xffff);
		int size = absoluteSize ? sizeChange : (font.getSize() + sizeChange);

		if( nameChange != null )
			return new Font( nameChange, style, size );
		else if( styleChange != 0 && sizeChange != 0 )
			return font.deriveFont( style, size );
		else if( styleChange != 0 )
			return font.deriveFont( style );
		else if( sizeChange != 0 )
			return font.deriveFont( (float) size );
		else
			return font; // nothing changed
	}

	/**
	 * Determines whether another object is equal to this object.
	 */
	@Override
	public boolean equals( Object obj ) {
		if( !(obj instanceof SwingDerivedFont) )
			return false;
		SwingDerivedFont f = (SwingDerivedFont) obj;
		return FormObject.safeEquals( nameChange, f.nameChange ) &&
			   styleChange == f.styleChange &&
			   sizeChange == f.sizeChange &&
			   absoluteSize == f.absoluteSize;
	}

	/**
	 * Returns the hash code for this object.
	 */
	@Override
	public int hashCode() {
		int hash = styleChange + sizeChange * 31;
		if( absoluteSize )
			hash += 61;
		if( nameChange != null )
			hash += nameChange.hashCode();
		return hash;
	}

	/**
	 * Returns a string representation of the object.
	 */
	@Override
	public String toString() {
		StringBuffer buf = new StringBuffer();
		if( (styleChange & Font.BOLD) != 0 )
			buf.append( "+BOLD " );
		if( (styleChange & Font.ITALIC) != 0 )
			buf.append( "+ITALIC " );
		if( ((styleChange >> 16) & Font.BOLD) != 0 )
			buf.append( "-BOLD " );
		if( ((styleChange >> 16) & Font.ITALIC) != 0 )
			buf.append( "-ITALIC " );
		buf.append( '(' ).append( Integer.toHexString( styleChange ) ).append( ')' );

		return FormObject.unqualifiedClassName( getClass() )
			+ "[nameChange=" + nameChange + ",styleChange=" + buf
			+ ",sizeChange=" + sizeChange + ",absoluteSize=" + absoluteSize + "]";
	}
}
