/*
 * Copyright (C) 2003-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.runtime;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.InputEvent;
import java.beans.*;
import java.lang.reflect.*;
import java.util.*;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import com.jgoodies.forms.factories.DefaultComponentFactory;
import com.jformdesigner.model.*;

/**
 * Creates instances of Swing components from a form model and
 * provides methods to access components.
 * Use {@link FormLoader} to load a form model from a JFormDesigner
 * .jfd file into memory.
 * <p>
 * First invoke one of the <code>create</code> methods.
 * Then you can use the getter methods to access the created Swing components.
 * Use the standard Swing methods to add listeners, set values or get values.
 * <p>
 * Example:
 * <pre>
 * // load the .jfd file into memory
 * FormModel formModel = FormLoader.load( "com/jformdesigner/examples/LoaderExample.jfd" );
 *
 * // create a dialog
 * FormCreator formCreator = new FormCreator( formModel );
 * JDialog dialog = formCreator.createDialog( null );
 *
 * // get references to components
 * JTextField nameField = formCreator.getTextField( "nameField" );
 * JCheckBox checkBox = formCreator.getCheckBox( "checkBox" );
 * JButton okButton = formCreator.getButton( "okButton" );
 *
 * // set values and add listeners
 * nameField.setText( "enter name here" );
 * okButton.addActionListener( new ActionListener() {
 * 	public void actionPerformed( ActionEvent e ) {
 * 		// add your code...
 * 	}
 * } );
 *
 * // show dialog
 * dialog.setModal( true );
 * dialog.pack();
 * dialog.show();
 *
 * System.out.println( nameField.getText() );
 * System.out.println( checkBox.isSelected() );
 * </pre>
 * <p>
 * If you've assigned event handlers in the form, use {@link #setTarget(Object)}
 * to set a target for the events.
 *
 * @author Karl Tauber
 */
public class FormCreator
	implements ComponentProvider, BeanProvider
{
	private static final String PROP_I18N_BUNDLE_PACKAGE = "i18n.bundlePackage";
	private static final String PROP_I18N_BUNDLE_NAME = "i18n.bundleName";
	private static final String PROP_BUTTON_GROUP = "$buttonGroup";
	private static final String PROP_SET_COMPONENT_NAMES = "$setComponentNames";
	private static final String PROP_LOCATION_POLICY = "$locationPolicy";
	private static final String PROP_SIZE_POLICY = "$sizePolicy";
	private static final String CONTAINER_DELEGATE = "containerDelegate";
	private static final String DESIGNER_CONTAINER_DELEGATE = "designerContainerDelegate";
	private static final String CLIENT_PREFIX = "$client.";
	private static final String ACTION_PREFIX = "$action.";
	private static final String I18N_KEY_PREFIX = "i18nKey=";

	private final FormModel model;
	private final Locale locale;
	private ClassLoader loader;
	private boolean setComponentNames;
	private Object target;

	/** Maps form components to beans <FormComponent,Object>. */
	private final IdentityHashMap<FormComponent, Object> componentMap = new IdentityHashMap<FormComponent, Object>();

	/** Maps button group names to button groups <String,ButtonGroup>. */
	private HashMap<String, ButtonGroup> buttonGroupMap;

	/** Maps resource bundle names to resource bundles <String,ResourceBundle>. */
	private HashMap<String, ResourceBundle> bundleMap;

	private ResourceBundle defaultBundle;
	private String defaultBaseName;

	/**
	 * Creates a new <code>FormCreator</code> for the specified form model.
	 *
	 * @param model The form model, from which the components should be created.
	 */
	public FormCreator( FormModel model ) {
		this( model, null );
	}

	/**
	 * Creates a new <code>FormCreator</code> for the specified form model
	 * using the specified class loader.
	 *
	 * @param model The form model, from which the components should be created.
	 * @param classLoader The class loader used to load components.
	 */
	public FormCreator( FormModel model, ClassLoader classLoader ) {
		this( model, null, classLoader );
	}

	/**
	 * Creates a new <code>FormCreator</code> for the specified form model
	 * using the specified locale and class loader.
	 *
	 * @param model The form model, from which the components should be created.
	 * @param locale The locale used to load resource bundles.
	 * @param classLoader The class loader used to load components.
	 * @since 2.0
	 */
	public FormCreator( FormModel model, Locale locale, ClassLoader classLoader ) {
		if( model == null )
			throw new IllegalArgumentException( "model is null" );

		this.model = model;
		this.locale = (locale != null) ? locale : Locale.getDefault();
		this.loader = (classLoader != null) ? classLoader : getClass().getClassLoader();
		if( loader == null )
			loader = ClassLoader.getSystemClassLoader();

		if( loader == null )
			throw new IllegalArgumentException( "Unable to get class loader. "
				+ "getClass().getClassLoader() and ClassLoader.getSystemClassLoader() returned null" );

		setComponentNames = model.getRoot().getPropertyBoolean( PROP_SET_COMPONENT_NAMES );
	}

	/**
	 * Returns the form model.
	 *
	 * @since 3.0
	 */
	public FormModel getModel() {
		return model;
	}

	/**
	 * Returns the locale used to load resource bundles.
	 *
	 * @since 3.0
	 */
	public Locale getLocale() {
		return locale;
	}

	/**
	 * Returns the class loader used to load components.
	 *
	 * @since 3.0
	 */
	public ClassLoader getLoader() {
		return loader;
	}

	/**
	 * Returns the object that handles events.
	 *
	 * @since 2.0
	 */
	public Object getTarget() {
		return target;
	}

	/**
	 * Set the object that will handle events.
	 * <p>
	 * This object must have handler methods as specified in the form.
	 * E.g. if you've specified an actionPerformed() event for a JButton and
	 * set the handler method to "myButtonPressed", the target object must
	 * implement following method:
	 * <pre>
	 * private void myButtonPressed() {
	 *     // handler code
	 * }
	 * </pre>
	 * If you've set the "Pass parameters" flag to true for the event,
	 * you can get the listener parameters:
	 * <pre>
	 * private void myButtonPressed(ActionEvent e) {
	 *     // handler code
	 * }
	 * </pre>
	 *
	 * @since 2.0
	 */
	public void setTarget( Object target ) {
		if( componentMap.size() > 0 )
			throw new IllegalStateException( "Invoke setTarget() before creating components." );

		this.target = target;
	}

	/**
	 * Returns whether <code>java.awt.Component.setName()</code> will be invoked
	 * on all components of the form. Default is <code>false</code>.
	 *
	 * @since 2.0.1
	 */
	public boolean isSetComponentNames() {
		return setComponentNames;
	}

	/**
	 * Sets whether <code>java.awt.Component.setName()</code> will be invoked
	 * on all components of the form.
	 *
	 * @since 2.0.1
	 */
	public void setSetComponentNames( boolean setComponentNames ) {
		this.setComponentNames = setComponentNames;
	}

	/**
	 * Creates all components.
	 * Use one of the getter methods to access beans.
	 *
	 * @since 2.0
	 */
	public void createAll()
		throws Exception
	{
		FormRoot root = model.getRoot();
		int count = root.getComponentCount();
		for( int i = 0; i < count; i++ )
			create( root.getComponent( i ) );
	}

	/**
	 * Creates the component hierarchy for the first top level form component.
	 *
	 * @return The created component hierarchy.
	 */
	public Component create()
		throws Exception
	{
		FormComponent comp = getFirstVisualTopLevel();
		if( comp == null )
			throw new IllegalStateException( "Form does not have a top-level component." );

		return (Component) create( comp );
	}

	/**
	 * Creates the component hierarchy for the first top level form component,
	 * which must be an instance of <code>javax.swing.JPanel</code>.
	 * Convenience method that invokes {@link #create()} and
	 * casts the result to <code>JPanel</code>.
	 *
	 * @return The created panel.
	 */
	public JPanel createPanel()
		throws Exception
	{
		return (JPanel) create();
	}

	/**
	 * Creates the component hierarchy for the first top level form component,
	 * which must be an instance of <code>javax.swing.JDialog</code>.
	 * Convenience method that invokes {@link #createWindow(Window)} and
	 * casts the result to <code>JDialog</code>.
	 *
	 * @param owner The window to act as owner; or <code>null</code>.
	 * @return The created dialog.
	 */
	public JDialog createDialog( Window owner )
		throws Exception
	{
		return (JDialog) createWindow( owner );
	}

	/**
	 * Creates the component hierarchy for the first top level form component,
	 * which must be an instance of <code>java.awt.Window</code>.
	 *
	 * @param owner The window to act as owner; or <code>null</code>.
	 * @return The created window.
	 */
	public Window createWindow( Window owner )
		throws Exception
	{
		FormComponent comp = getFirstVisualTopLevel();
		if( comp == null )
			throw new IllegalStateException( "Form does not have a top-level component." );

		return (Window) create( comp, owner );
	}

	/**
	 * Creates the component hierarchy for the specified name.
	 *
	 * @param name The name of the form component.
	 * @return The created component hierarchy.
	 */
	public Component create( String name )
		throws Exception
	{
		return create( name, null );
	}

	/**
	 * Creates the component hierarchy for the specified name and passes the
	 * specified owner window to the constructor, if the component is an
	 * instance of <code>java.awt.Window</code>.
	 *
	 * @param name The name of the form component.
	 * @param owner The window to act as owner; or <code>null</code>.
	 * @return The created component hierarchy.
	 */
	public Component create( String name, Window owner )
		throws Exception
	{
		FormComponent comp = model.getFormComponent( name );
		if( comp == null )
			throw new NoSuchComponentException( "Unknown component \"" + name + "\"." );

		return (Component) create( comp, owner );
	}

	/**
	 * Creates the component hierarchy for the specified form component.
	 */
	Object create( FormComponent formComp )
		throws Exception
	{
		return create( formComp, null );
	}

	Object create( FormComponent formComp, Window owner )
		throws Exception
	{
		Object c = getComponent( formComp );
		if( c == null ) {
			c = createComponents( formComp, owner );
			setReferences( formComp );
		}
		return c;
	}

	private FormComponent getFirstVisualTopLevel() {
		// find first visual under root
		FormRoot root = model.getRoot();
		int count = root.getComponentCount();
		if( count == 0 )
			return null;
		for( int i = 0; i < count; i++ ) {
			FormComponent comp = root.getComponent( i );
			if( comp instanceof FormNonVisual )
				continue;
			return comp;
		}
		return null;
	}

	/**
	 * Creates the bindings.
	 *
	 * @since 5.0
	 */
	@SuppressWarnings({ "unchecked", "rawtypes" })
	public Object[] createBindings()
		throws Exception
	{
		FormBindingGroup[] bindingGroups = model.getRoot().getBindingGroups();
		if( bindingGroups.length == 0 )
			return null;

		Object[] groups = new Object[bindingGroups.length];
		for( int i = 0; i < bindingGroups.length; i++ ) {
			BindingCreator bindingCreator = BindingCreatorRegistry.createBindingCreator(
					bindingGroups[i].getBindingGroupClass() );
			bindingCreator.setFormCreator( this );

			Object group = bindingCreator.createBindingGroup( bindingGroups[i] );
			if( bindingGroups[i].getPropertyBoolean( FormBindingGroup.PROP_BOUND, true ) )
				bindingCreator.bindGroup( group );

			groups[i] = group;
		}

		return groups;
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the name specifies a non-visual bean.
	 */
	public Component getComponent( String name ) throws NoSuchComponentException {
		return (Component) getBean( name );
	}

	/**
	 * Returns the bean for the specified name.
	 * This can be a visual or a non-visual bean.
	 *
	 * @param name The name of the bean.
	 * @return The bean.
	 * @throws NoSuchComponentException If a bean with the specified name
	 * 							does not exist or the bean is not created.
	 * @since 2.0
	 */
	public Object getBean( String name ) throws NoSuchComponentException {
		FormComponent formComp = model.getFormComponent( name );
		if( formComp == null )
			throw new NoSuchComponentException( "Unknown component \"" + name + "\"." );

		Object c = getComponent( formComp );
		if( c == null )
			throw new NoSuchComponentException( "Component \"" + name + "\" is not created." );
		return c;
	}

	/**
	 * Returns the bean for the specified name.
	 * This can be a visual or a non-visual bean.
	 *
	 * @param name The name of the bean.
	 * @param create If true, create the bean if necessary.
	 * @return The bean.
	 * @since 5.0
	 */
	public Object getBean( String name, boolean create ) throws Exception {
		FormComponent formComp = model.getFormComponent( name );
		if( formComp == null )
			throw new NoSuchComponentException( "Unknown component \"" + name + "\"." );

		return create( formComp );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JButton</code>.
	 */
	public JButton getButton( String name ) throws NoSuchComponentException {
		return (JButton) getComponent( name );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JCheckBox</code>.
	 */
	public JCheckBox getCheckBox( String name ) throws NoSuchComponentException {
		return (JCheckBox) getComponent( name );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JCheckBoxMenuItem</code>.
	 * @since 2.0.5
	 */
	public JCheckBoxMenuItem getCheckBoxMenuItem( String name ) throws NoSuchComponentException {
		return (JCheckBoxMenuItem) getComponent( name );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JComboBox</code>.
	 */
	public JComboBox getComboBox( String name ) throws NoSuchComponentException {
		return (JComboBox) getComponent( name );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JDialog</code>.
	 */
	public JDialog getDialog( String name ) throws NoSuchComponentException {
		return (JDialog) getComponent( name );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JEditorPane</code>.
	 */
	public JEditorPane getEditorPane( String name ) throws NoSuchComponentException {
		return (JEditorPane) getComponent( name );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JFormattedTextField</code>.
	 */
	public JFormattedTextField getFormattedTextField( String name ) throws NoSuchComponentException {
		return (JFormattedTextField) getComponent( name );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JFrame</code>.
	 */
	public JFrame getFrame( String name ) throws NoSuchComponentException {
		return (JFrame) getComponent( name );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JLabel</code>.
	 */
	public JLabel getLabel( String name ) throws NoSuchComponentException {
		return (JLabel) getComponent( name );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JList</code>.
	 */
	public JList getList( String name ) throws NoSuchComponentException {
		return (JList) getComponent( name );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JMenu</code>.
	 * @since 2.0.5
	 */
	public JMenu getMenu( String name ) throws NoSuchComponentException {
		return (JMenu) getComponent( name );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JMenuBar</code>.
	 * @since 2.0.5
	 */
	public JMenuBar getMenuBar( String name ) throws NoSuchComponentException {
		return (JMenuBar) getComponent( name );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JMenuItem</code>.
	 * @since 2.0.5
	 */
	public JMenuItem getMenuItem( String name ) throws NoSuchComponentException {
		return (JMenuItem) getComponent( name );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JPanel</code>.
	 */
	public JPanel getPanel( String name ) throws NoSuchComponentException {
		return (JPanel) getComponent( name );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JPasswordField</code>.
	 */
	public JPasswordField getPasswordField( String name ) throws NoSuchComponentException {
		return (JPasswordField) getComponent( name );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JPopupMenu</code>.
	 * @since 2.0.5
	 */
	public JPopupMenu getPopupMenu( String name ) throws NoSuchComponentException {
		return (JPopupMenu) getComponent( name );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JProgressBar</code>.
	 */
	public JProgressBar getProgressBar( String name ) throws NoSuchComponentException {
		return (JProgressBar) getComponent( name );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JRadioButton</code>.
	 */
	public JRadioButton getRadioButton( String name ) throws NoSuchComponentException {
		return (JRadioButton) getComponent( name );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JRadioButtonMenuItem</code>.
	 * @since 2.0.5
	 */
	public JRadioButtonMenuItem getRadioButtonMenuItem( String name ) throws NoSuchComponentException {
		return (JRadioButtonMenuItem) getComponent( name );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JScrollBar</code>.
	 */
	public JScrollBar getScrollBar( String name ) throws NoSuchComponentException {
		return (JScrollBar) getComponent( name );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JScrollPane</code>.
	 */
	public JScrollPane getScrollPane( String name ) throws NoSuchComponentException {
		return (JScrollPane) getComponent( name );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JSeparator</code>.
	 */
	public JSeparator getSeparator( String name ) throws NoSuchComponentException {
		return (JSeparator) getComponent( name );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JSlider</code>.
	 */
	public JSlider getSlider( String name ) throws NoSuchComponentException {
		return (JSlider) getComponent( name );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JSpinner</code>.
	 */
	public JSpinner getSpinner( String name ) throws NoSuchComponentException {
		return (JSpinner) getComponent( name );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JSplitPane</code>.
	 */
	public JSplitPane getSplitPane( String name ) throws NoSuchComponentException {
		return (JSplitPane) getComponent( name );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JTabbedPane</code>.
	 */
	public JTabbedPane getTabbedPane( String name ) throws NoSuchComponentException {
		return (JTabbedPane) getComponent( name );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JTable</code>.
	 */
	public JTable getTable( String name ) throws NoSuchComponentException {
		return (JTable) getComponent( name );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JTextArea</code>.
	 */
	public JTextArea getTextArea( String name ) throws NoSuchComponentException {
		return (JTextArea) getComponent( name );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JTextField</code>.
	 */
	public JTextField getTextField( String name ) throws NoSuchComponentException {
		return (JTextField) getComponent( name );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JTextPane</code>.
	 */
	public JTextPane getTextPane( String name ) throws NoSuchComponentException {
		return (JTextPane) getComponent( name );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JToggleButton</code>.
	 */
	public JToggleButton getToggleButton( String name ) throws NoSuchComponentException {
		return (JToggleButton) getComponent( name );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JToolBar</code>.
	 */
	public JToolBar getToolBar( String name ) throws NoSuchComponentException {
		return (JToolBar) getComponent( name );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JTree</code>.
	 */
	public JTree getTree( String name ) throws NoSuchComponentException {
		return (JTree) getComponent( name );
	}

	/**
	 * Returns the component for the specified name.
	 *
	 * @param name The name of the component.
	 * @return The component.
	 * @throws NoSuchComponentException If a component with the specified name
	 * 							does not exist or the component is not created.
	 * @throws ClassCastException If the component is not a instance of <code>JWindow</code>.
	 */
	public JWindow getWindow( String name ) throws NoSuchComponentException {
		return (JWindow) getComponent( name );
	}

	/**
	 * Returns the component for the specified form component;
	 * or <code>null</code> if the component is not yet created.
	 */
	private Object getComponent( FormComponent formComp ) {
		return componentMap.get( formComp );
	}

	/**
	 * Creates the component hierarchy for the specified form component.
	 */
	private Object createComponents( FormComponent formComp, Window owner )
		throws Exception
	{
		Object c = getComponent( formComp );
		if( c != null )
			return c; // already created

		if( owner != null && formComp instanceof FormWindow )
			c = createWindowComponent( (FormWindow) formComp, owner );
		else
			c = createComponent( formComp );

		componentMap.put( formComp, c );

		Object groupRef = formComp.getProperty( PROP_BUTTON_GROUP );
		if( groupRef instanceof FormReference && c instanceof AbstractButton ) {
			String buttonGroupName = ((FormReference)groupRef).getName();
			ButtonGroup buttonGroup = (buttonGroupMap != null)
				? buttonGroupMap.get( buttonGroupName ) : null;
			if( buttonGroup == null ) {
				FormComponent buttonGroupComp = model.getFormComponent( buttonGroupName );
				if( buttonGroupComp != null ) {
					buttonGroup = (ButtonGroup) componentMap.get( buttonGroupComp );
					if( buttonGroup == null ) {
						buttonGroup = (ButtonGroup) createComponent( buttonGroupComp );
						componentMap.put( buttonGroupComp, buttonGroup );
					}
				} else
					buttonGroup = new ButtonGroup();

				if( buttonGroupMap == null )
					buttonGroupMap = new HashMap<String, ButtonGroup>(); // create lazy
				buttonGroupMap.put( buttonGroupName, buttonGroup );
			}
			buttonGroup.add( (AbstractButton) c );
		}

		if( formComp instanceof FormContainer ) {
			Container container = getContainerDelegate( (Container) c );

			FormContainer formCont = (FormContainer) formComp;
			createMenuBar( formCont, (Container) c );

			FormLayoutManager formLayout = formCont.getLayout();
			LayoutCreator layoutCreator = null;
			if( formLayout != null ) {
				layoutCreator = LayoutCreatorRegistry.createLayoutCreator( formLayout.getLayoutClass() );
				layoutCreator.setFormCreator( this );
				LayoutManager layout = layoutCreator.createLayoutManager( container, formLayout );
				if( layout != null )
					container.setLayout( layout );
			}

			int count = formCont.getComponentCount();
			for( int i = 0; i < count; i++ ) {
				FormComponent fc = formCont.getComponent( i );
				FormLayoutConstraints formConstraints = null;
				Component c2 = (Component) createComponents( fc, null );
				Object constraints = null;
				if( formLayout != null ) {
					formConstraints = formLayout.getConstraints( fc );
					if( formConstraints != null )
						constraints = createConstraints( layoutCreator, formConstraints );
				}

				if( layoutCreator != null )
					layoutCreator.addComponentToContainer( container, c2,
											constraints, -1, formConstraints );
				else
					container.add( c2, constraints );
			}

			if( layoutCreator != null )
				layoutCreator.finishLayoutInitialization( container, formLayout );

			if( formComp instanceof FormWindow && c instanceof Window ) {
				Window w = (Window) c;
				int sizePolicy = formComp.getPropertyInt( PROP_SIZE_POLICY );
				if( sizePolicy == 0 )
					w.pack();
				else if( sizePolicy == 1 ) {
					FormLayoutConstraints formCons = formComp.getConstraints();
					if( formCons != null ) {
						Dimension size = (Dimension) formCons.getProperty( "size" );
						if( size != null )
							w.setSize( size );
					}
				}

				int locationPolicy = formComp.getPropertyInt( PROP_LOCATION_POLICY );
				if( locationPolicy == 0 )
					w.setLocationRelativeTo(w.getOwner());
				else if( locationPolicy == 1 )
					w.setLocationRelativeTo(null);
			}
		}

		return c;
	}

	private void createMenuBar( FormContainer formCont, Component c )
		throws Exception
	{
		FormComponent menuBarComp = formCont.getMenuBar();
		if( menuBarComp == null )
			return;

		JMenuBar menuBar = (JMenuBar) createComponents( menuBarComp, null );

		Method setMethod = c.getClass().getMethod( "setJMenuBar",
										new Class[] { JMenuBar.class } );
		setMethod.invoke( c, new Object[] { menuBar } );
	}

	private Container getContainerDelegate( Container container )
		throws Exception
	{
		Class<?> containerClass = container.getClass();

		// avoid creation of BeanInfo for well known components
		if( containerClass == JPanel.class || // not using instanceof because of subclasses
			container instanceof JTabbedPane )
		  return container;

		// special case for JScrollPane
		if( container instanceof JScrollPane )
			return container;

		// use container delegate from BeanInfo
		String delegateMethodName = null;
		try {
			BeanInfoEx beanInfo = IntrospectorEx.getBeanInfoEx( containerClass );
			if( beanInfo != null ) {
				BeanDescriptor beanDesc = beanInfo.getBeanDescriptor();
				if( beanDesc != null ) {
					delegateMethodName = (String) beanDesc.getValue( CONTAINER_DELEGATE );
					if( delegateMethodName == null )
						delegateMethodName = (String) beanDesc.getValue( DESIGNER_CONTAINER_DELEGATE );
				}
			}
		} catch( IntrospectionException ex ) {
			return container;
		}

		// temporary fix
		if( delegateMethodName == null &&
			containerClass.getName().equals( "com.jgoodies.uif_lite.panel.SimpleInternalFrame" ) )
		  delegateMethodName = "getContentPane";

		if( delegateMethodName != null ) {
			Method m = container.getClass().getMethod( delegateMethodName, (Class[]) null );
			return (Container) m.invoke( container, (Object[]) null );
		}

		// special case for RootPaneContainer and JRootPane
		if( container instanceof RootPaneContainer )
			return ((RootPaneContainer)container).getContentPane();
		if( container instanceof JRootPane )
			return ((JRootPane)container).getContentPane();

		return container;
	}

	private void setReferences( FormComponent formComp )
		throws Exception
	{
		int refCount = formComp.getReferenceCount();
		if( refCount > 0 ) {
			Object c = getComponent( formComp );
			FormModel model = formComp.getModel();
			BeanInfoEx beanInfo = IntrospectorEx.getBeanInfoEx( c.getClass() );

			for( Map.Entry<String, Object> entry : formComp.properties() ) {
				String name = entry.getKey();
				Object value = entry.getValue();

				if( value instanceof FormReference ) {
					FormComponent refComp = model.getFormComponent(
												((FormReference)value).getName() );
					if( refComp == null )
						continue;
					Object ref = getComponent( refComp );
					if( ref == null ) {
						// create bean on demand
						ref = create( refComp );
					}
					setObjectProperty( c, name, ref, beanInfo );

					if( --refCount <= 0 )
						break;
				}
			}
		}

		if( formComp instanceof FormContainer ) {
			FormContainer formCont = (FormContainer) formComp;

			if( formCont.getMenuBar() != null )
				setReferences( formCont.getMenuBar() );

			int count = formCont.getComponentCount();
			for( int i = 0; i < count; i++ )
				setReferences( formCont.getComponent( i ) );
		}
	}

	/**
	 * Creates one component and sets its properties.
	 */
	private Object createComponent( FormComponent formComp )
		throws Exception
	{
		String className = formComp.getClassName();

		Object c;
		if( className.startsWith( "com.jformdesigner.designer.wrapper." ) )
			c = createWrapperComponent( formComp );
		else {
			Class<?> beanClass = loader.loadClass( className );
			if( beanClass == JComponent.class )
				c = new JComponent() {};
			else if( beanClass == AbstractAction.class )
				c = new AbstractAction() { public void actionPerformed( ActionEvent e ) {} };
			else
				c = newComponentInstance( beanClass, formComp.getName() );
			setObjectProperties( c, formComp, null, true );
		}

		if( setComponentNames && c instanceof Component && formComp.getProperty( "name" ) == null )
			((Component)c).setName( formComp.getName() );

		addEventListeners( formComp, c );

		return c;
	}

	/**
	 * Creates a new instance of the given bean class.
	 * The default implementation is "<code>return beanClass.newInstance();</code>".
	 * <p>
	 * Override this method to implement special component creation.
	 *
	 * @param beanClass The component class.
	 * @param compName The name of the component.
	 * @return A new instance of the component.
	 * @since 3.0
	 */
	protected Object newComponentInstance( Class<?> beanClass, String compName )
		throws Exception
	{
		return beanClass.newInstance();
	}

	/**
	 * Creates one window and sets its properties.
	 */
	private Component createWindowComponent( FormWindow formComp, Window owner )
		throws Exception
	{
		String className = formComp.getClassName();

		Class<?> beanClass = loader.loadClass( className );

		Component c;
		if( owner instanceof Frame ) {
			Constructor<?> constructor = beanClass.getConstructor( new Class[] { Frame.class } );
			c = (Component) constructor.newInstance( new Object[] { owner } );
		} else if( owner instanceof Dialog ) {
			Constructor<?> constructor = beanClass.getConstructor( new Class[] { Dialog.class } );
			c = (Component) constructor.newInstance( new Object[] { owner } );
		} else {
			c = (Component) beanClass.newInstance();
		}
		setObjectProperties( c, formComp, null, true );

		if( setComponentNames && formComp.getProperty( "name" ) == null )
			c.setName( formComp.getName() );

		addEventListeners( formComp, c );

		return c;
	}

	/**
	 * Creates wrapper components (JGoodies, spacers) and sets its properties.
	 */
	private Component createWrapperComponent( FormComponent formComp )
		throws Exception
	{
		String className = formComp.getClassName();
		DefaultComponentFactory factory = DefaultComponentFactory.getInstance();

		if( className.equals( "com.jformdesigner.designer.wrapper.JGoodiesFormsLabel" ) ) {
			String text = getPropertyStringI18n( formComp, "textWithMnemonic", "" );
			Component c = factory.createLabel( text );
			setObjectProperties( c, formComp, "textWithMnemonic", true );
			return c;
		}

		if( className.equals( "com.jformdesigner.designer.wrapper.JGoodiesFormsSeparator" ) ) {
			String text = getPropertyStringI18n( formComp, "text", "" );
			int alignment = formComp.getPropertyInt( "alignment", SwingConstants.LEFT );
			return factory.createSeparator( text, alignment );
		}

		if( className.equals( "com.jformdesigner.designer.wrapper.JGoodiesFormsTitle" ) ) {
			String text = getPropertyStringI18n( formComp, "textWithMnemonic", "" );
			Component c = factory.createTitle( text );
			setObjectProperties( c, formComp, "textWithMnemonic", true );
			return c;
		}

		if( className.equals( "com.jformdesigner.designer.wrapper.HSpacer" ) ||
			className.equals( "com.jformdesigner.designer.wrapper.VSpacer" ) )
		  return new JPanel( null );

		throw new ClassNotFoundException( className );
	}

	/**
	 * Creates a constraints object and sets its properties.
	 */
	private Object createConstraints( LayoutCreator layoutCreator, FormLayoutConstraints formConstraints )
		throws Exception
	{
		Class<?> constraintsClass = formConstraints.getConstraintsClass();
		if( constraintsClass == null )
			return formConstraints;
		else if( constraintsClass == String.class ) {
			return formConstraints.getProperty( FormLayoutConstraints.PROP_VALUE );
		} else {
			Object constraints = layoutCreator.createLayoutConstraints( formConstraints );
			if( constraints == null ) {
				constraints = constraintsClass.newInstance();
				setObjectProperties( constraints, formConstraints, null, false );
			}
			return constraints;
		}
	}

	/**
	 * Sets all properties of the specified component.
	 */
	private void setObjectProperties( Object target, FormObject formObject,
									  String exclude, boolean useBeanInfo )
		throws Exception
	{
		if( formObject.getPropertyCount() == 0 )
			return;

		BeanInfoEx beanInfo = useBeanInfo ? IntrospectorEx.getBeanInfoEx( target.getClass() ) : null;
		for( Map.Entry<String, Object> entry : formObject.properties() ) {
			String name = entry.getKey();
			if( name.startsWith( "$" ) &&
				!name.startsWith( CLIENT_PREFIX ) &&
				!name.startsWith( ACTION_PREFIX ) )
			  continue;
			if( exclude != null && name.equals( exclude ) )
				continue;

			Object value = entry.getValue();
			if( value instanceof FormReference )
				continue;

			if( value != null )
				setObjectProperty( target, name, value, beanInfo );
		}
	}

	/**
	 * Sets one property of the specified component.
	 */
	private void setObjectProperty( Object target, String name, Object value, BeanInfoEx beanInfo )
		throws Exception
	{
		if( name.startsWith( "$" ) ) {
			if( name.startsWith( CLIENT_PREFIX ) && target instanceof JComponent ) {
				String key = name.substring( CLIENT_PREFIX.length() );
				if( value == FormObject.NULL_VALUE )
					value = null;
				value = ObjectCloner.cloneObject( value );
				((JComponent)target).putClientProperty( key, value );
			} else if( name.startsWith( ACTION_PREFIX ) && target instanceof Action ) {
				String key = name.substring( ACTION_PREFIX.length() );
				if( value == FormObject.NULL_VALUE )
					value = null;
				else if( value instanceof SwingIcon )
					value = ((SwingIcon)value).createIcon( loader );
				else if( value instanceof KeyStroke )
					value = updateKeyStroke( (KeyStroke) value );
				else if( value instanceof FormMessage ) {
					value = getString( (FormMessage) value );

					boolean isMnemonicIndex = key.equals( "SwingDisplayedMnemonicIndexKey" ); // Java 6
					if( key.equals( Action.MNEMONIC_KEY ) || isMnemonicIndex )
						value = convertI18nMnemonic( value, isMnemonicIndex );
				}
				value = ObjectCloner.cloneObject( value );
				((Action)target).putValue( key, value );
			}
			return;
		}

		boolean isI18nValue = false;
		if( value == FormObject.NULL_VALUE )
			value = null;
		else if( value instanceof SwingBorder )
			value = UIManager.getBorder( ((SwingBorder)value).getKey() );
		else if( value instanceof SwingColor )
			value = UIManager.getColor( ((SwingColor)value).getKey() );
		else if( value instanceof SwingDerivedFont )
			value = ((SwingDerivedFont)value).derive( (Font) getObjectProperty( target, name, beanInfo ) );
		else if( value instanceof SwingFont )
			value = UIManager.getFont( ((SwingFont)value).getKey() );
		else if( value instanceof SwingIcon )
			value = ((SwingIcon)value).createIcon( loader );
		else if( value instanceof Border )
			value = updateBorder( (Border) value );
		else if( value instanceof KeyStroke )
			value = updateKeyStroke( (KeyStroke) value );
		else if( value instanceof FormMessage ) {
			value = getString( (FormMessage) value );
			isI18nValue = true;
		} else if( value instanceof FormMessageArray ) {
			FormMessageArray messages = (FormMessageArray) value;
			int length = messages.getLength();
			String[] strings = new String[length];
			for( int i = 0; i < length; i++ )
				strings[i] = getString( messages.getMessage( i ) );
			value = strings;
		} else if( value instanceof Integer &&
				   ("mnemonic".equals( name ) ||
					"displayedMnemonic".equals( name ) ||
					"approveButtonMnemonic".equals( name )) )
		{
			int ch = ((Integer)value).intValue();
			if( ch >= 'a' && ch <= 'z' )
				value = new Integer( ch - ('a' - 'A') );
		}

		value = ObjectCloner.cloneObject( value );

		if( beanInfo != null ) {
			PropertyDescriptor desc = beanInfo.getPropertyDescriptor( name );
			if( desc != null ) {
				if( value instanceof ImageIcon && Image.class.isAssignableFrom( desc.getPropertyType() ) )
					value = ((ImageIcon)value).getImage();

				if( isI18nValue && value instanceof String ) {
					// special code for mnemonics and mnemonic indices
					if( desc.getPropertyType() == Integer.TYPE )
						value = convertI18nMnemonic( value, name.endsWith( "Index" ) );
				}

				Method writeMethod = desc.getWriteMethod();
				if( writeMethod != null ) {
					writeMethod.invoke( target, new Object[] { value } );
					postSetObjectProperty( target, name, value );
					return;
				}
			}
		}

		StringBuffer setMethodName = new StringBuffer( 3 + name.length() );
		setMethodName.append( "set" ).append( name );
		setMethodName.setCharAt( 3, Character.toUpperCase( setMethodName.charAt( 3 ) ) );

		Statement st = new Statement( target, setMethodName.toString(),
									  new Object[] { value } );
		try {
			// invoke "set" method
			st.execute();
		} catch( NoSuchMethodException ex ) {
			try {
				// "set" method not found --> try to set field directly
				Field field = target.getClass().getField( name );
				field.set( target, value );
			} catch( NoSuchFieldException ex2 ) {
				throw ex; // throw original NoSuchMethodException
			}
		}
		postSetObjectProperty( target, name, value );
	}

	private Object getObjectProperty( Object target, String name, BeanInfoEx beanInfo )
		throws Exception
	{
		if( beanInfo != null ) {
			PropertyDescriptor desc = beanInfo.getPropertyDescriptor( name );
			if( desc != null ) {
				Method readMethod = desc.getReadMethod();
				if( readMethod != null )
					return readMethod.invoke( target, (Object[]) null );
			}
		}

		return null;
	}

	private Border updateBorder( Border border ) {
		if( border instanceof TitledBorder ) {
			// replace i18n key with real string
			TitledBorder titledBorder = (TitledBorder) border;
			String title = titledBorder.getTitle();
			if( title.startsWith( I18N_KEY_PREFIX ) ) {
				String newTitle = getString( new FormMessage( null,
					title.substring( I18N_KEY_PREFIX.length() ) ) );
				return RuntimeUtils.cloneTitledBorder( titledBorder, newTitle );
			}
		} else if( border instanceof MatteBorder ) {
			// create icon for matte border
			MatteBorder matteBorder = (MatteBorder) border;
			Icon tileIcon = matteBorder.getTileIcon();
			if( tileIcon instanceof SwingIcon )
				return new MatteBorder( RuntimeUtils.getRawMatteBorderInsets( matteBorder ),
					((SwingIcon)tileIcon).createIcon( loader ) );
		} else if( border instanceof CompoundBorder ) {
			CompoundBorder compBorder = (CompoundBorder) border;
			Border outside = updateBorder( compBorder.getOutsideBorder() );
			Border inside = updateBorder( compBorder.getInsideBorder() );
			if( outside != compBorder.getOutsideBorder() ||
				inside != compBorder.getInsideBorder() )
			  return new CompoundBorder( outside, inside );
		}
		return border;
	}

	private KeyStroke updateKeyStroke( KeyStroke keyStroke ) {
		int modifiers = keyStroke.getModifiers();
		if( (modifiers & InputEvent.BUTTON3_DOWN_MASK) == 0 )
			return keyStroke;

		modifiers &= ~(InputEvent.CTRL_MASK | InputEvent.CTRL_DOWN_MASK |
			InputEvent.META_MASK | InputEvent.META_DOWN_MASK | InputEvent.BUTTON3_DOWN_MASK);
		modifiers |= Toolkit.getDefaultToolkit().getMenuShortcutKeyMask();
		return KeyStroke.getKeyStroke( keyStroke.getKeyCode(), modifiers, keyStroke.isOnKeyRelease() );
	}

	private Integer convertI18nMnemonic( Object value, boolean isMnemonicIndex ) {
		String str = (String) value;
		if( isMnemonicIndex )
			return Integer.valueOf( str );
		else {
			char ch = str.charAt( 0 );
			if( ch >= 'a' && ch <= 'z' )
				ch -= 'a' - 'A';
			return new Integer( ch );
		}
	}

	private void postSetObjectProperty( Object target, String name, Object value ) {
		if( name.equals( "model" ) && target instanceof JTable && value instanceof SwingTableModel ) {
			// apply column infos
			applyColumnInfos( (JTable) target, (SwingTableModel) value );
		}
	}

	private void applyColumnInfos( JTable table, SwingTableModel model ) {
		TableColumnModel cm = table.getColumnModel();

		int columnCount = model.getColumnCount();
		for( int i = 0; i < columnCount; i++ ) {
			SwingTableColumn colInfo = model.getColumnInfo( i );
			if( colInfo == null )
				continue;

			TableColumn col = cm.getColumn( i );

			if( !colInfo.getResizable() )
				col.setResizable( false );

			if( colInfo.getMinWidth() != 0 )
				col.setMinWidth( colInfo.getMinWidth() );

			if( colInfo.getMaxWidth() != 0 )
				col.setMaxWidth( colInfo.getMaxWidth() );

			if( colInfo.getPreferredWidth() != 0 )
				col.setPreferredWidth( colInfo.getPreferredWidth() );

			Object[] values = colInfo.getValues();
			Class<?> type = model.getColumnClass( i );
			if( values != null && values.length > 0 &&
				(type == Object.class || type == String.class) )
			{
				col.setCellEditor( new DefaultCellEditor(
					new JComboBox( new DefaultComboBoxModel( values ) ) ) );
			}
		}
	}

	String getPropertyStringI18n( FormObject formObject, String name, String def ) {
		Object value = formObject.getProperty( name );
		if( value instanceof String )
			return (String) value;
		else if( value instanceof FormMessage )
			return getString( (FormMessage) value );
		else
			return def;
	}

	String getString( FormMessage message ) {
		String baseName = message.getBaseName();
		String key = message.getKey();

		if( baseName == null ) {
			if( defaultBundle == null ) {
				String bundlePackage = model.getPropertyString( PROP_I18N_BUNDLE_PACKAGE );
				String bundleName = model.getPropertyString( PROP_I18N_BUNDLE_NAME );
				defaultBaseName = (bundlePackage != null)
							? (bundlePackage + '.' + bundleName) : bundleName;
				defaultBundle = getI18nBundle( defaultBaseName, locale, loader );
			}
			return getI18nString( defaultBundle, defaultBaseName, key );
		} else {
			ResourceBundle bundle = (bundleMap != null) ? bundleMap.get( baseName ) : null;
			if( bundle == null ) {
				bundle = getI18nBundle( baseName, locale, loader );
				if( bundleMap == null )
					bundleMap = new HashMap<String, ResourceBundle>(); // create lazy
				bundleMap.put( baseName, bundle );
			}
			return getI18nString( bundle, baseName, key );
		}
	}

	/**
	 * Gets the resource bundle for the given base name.
	 * The default implementation is "<code>return ResourceBundle.getBundle(baseName, locale, loader);</code>".
	 * <p>
	 * Override this method to implement special bundle loading.
	 *
	 * @param baseName The base name of the resource bundle, a fully qualified class name.
	 * @param locale The locale for which a resource bundle is desired.
	 * @param loader The class loader from which to load the resource bundle.
	 * @return The resource bundle for the given base name and locale.
	 * @since 3.0
	 */
	protected ResourceBundle getI18nBundle( String baseName, Locale locale, ClassLoader loader ) {
		return ResourceBundle.getBundle( baseName, locale, loader );
	}

	/**
	 * Gets a string for the given key from the given resource bundle or one of its parents.
	 * The default implementation is "<code>return bundle.getString(key);</code>".
	 * <p>
	 * Override this method to implement special string loading.
	 *
	 * @param bundle The resource bundle to get the string from.
	 * @param baseName The base name of the resource bundle, a fully qualified class name.
	 * @param key The key for the desired string.
	 * @return The string for the given key.
	 * @since 3.0
	 */
	protected String getI18nString( ResourceBundle bundle, String baseName, String key ) {
		return bundle.getString( key );
	}

	void updateUIValues() {
		for( Map.Entry<FormComponent, Object> e : componentMap.entrySet() ) {
			FormComponent formComp = e.getKey();
			Object bean = e.getValue();
			BeanInfoEx beanInfo = null;

			for( Map.Entry<String, Object> entry : formComp.properties() ) {
				String name = entry.getKey();
				Object value = entry.getValue();

				if( value instanceof SwingResource ) {
					try {
						if( beanInfo == null ) {
							// get BeanInfo lazy
							beanInfo = IntrospectorEx.getBeanInfoEx( bean.getClass() );
						}
						setObjectProperty( bean, name, value, beanInfo );
					} catch( Exception ex ) {
						// ignore
					}
				}
			}
		}
	}

	private void addEventListeners( FormComponent formComp, Object c )
		throws Exception
	{
		if( target == null )
			return;

		FormEvent[] events = formComp.getEvents();
		if( events.length == 0 )
			return;

		BeanInfoEx beanInfo = IntrospectorEx.getBeanInfoEx( c.getClass() );

		for( int i = 0; i < events.length; i++ ) {
			FormEvent event = events[i];
			Class<?> listenerClass = loader.loadClass( event.getListener() );

			EventSetDescriptor eventSetDesc = beanInfo.getEventSetDescriptor( event.getListener() );
			if( eventSetDesc == null )
				throw new IllegalStateException( "Component " + formComp.getName()
					+ " (class " + formComp.getClassName() + ") does not support "
					+ event.getListener() + "." );

			Method listenerMethod = findListenerMethod( eventSetDesc, event.getListenerMethod() );
			if( listenerMethod == null )
				throw new IllegalStateException( "Component " + formComp.getName()
					+ ": Event listener " + event.getListener() + " does not have a "
					+ event.getListenerMethod() + " method." );

			HandlerInvoker handlerInvoker = new HandlerInvoker( target,
				listenerMethod, event.getHandler(), event.getPassParams() );
			Object listener = Proxy.newProxyInstance( loader,
				new Class[] { listenerClass }, handlerInvoker );

			if( listenerClass == PropertyChangeListener.class &&
				event.getPropertyName() != null )
			{
				Method addMethod = c.getClass().getMethod( "addPropertyChangeListener",
					new Class[] { String.class, PropertyChangeListener.class } );
				addMethod.invoke( c, new Object[] { event.getPropertyName(), listener } );
			} else {
				Method addMethod = eventSetDesc.getAddListenerMethod();
				addMethod.invoke( c, new Object[] { listener } );
			}
		}
	}

	private Method findListenerMethod( EventSetDescriptor eventSetDesc, String listenerMethod ) {
		Method[] methods = eventSetDesc.getListenerMethods();
		for( int i = 0; i < methods.length; i++ ) {
			Method method = methods[i];
			if( listenerMethod.equals( method.getName() ) )
				return method;
		}
		return null;
	}

	//---- class HandlerInvoker -----------------------------------------------

	private static class HandlerInvoker
		implements InvocationHandler
	{
		private final Object target;
		private final String listenerMethod;
		private final Method targetMethod;
		private final boolean passParams;

		HandlerInvoker( Object target, Method listenerMethod,
						String handler, boolean passParams )
			throws Exception
		{
			this.target = target;
			this.listenerMethod = listenerMethod.getName();

			Class<?> targetClass = target.getClass();
			Method targetMethod = null;

			if( passParams ) {
				Class<?>[] paramTypes = listenerMethod.getParameterTypes();
				try {
					targetMethod = getDeclaredMethod( targetClass, handler, paramTypes );
				} catch( NoSuchMethodException ex ) {
					// don't pass parameters
					passParams = false;
				}
			}

			if( targetMethod == null )
				targetMethod = getDeclaredMethod( targetClass, handler, null );

			// make sure that we can invoke private methods
			targetMethod.setAccessible( true );

			this.targetMethod = targetMethod;
			this.passParams = passParams;
		}

		private static Method getDeclaredMethod( Class<?> targetClass, String name, Class<?>[] paramTypes )
			throws NoSuchMethodException, SecurityException
		{
			try {
				return targetClass.getDeclaredMethod( name, paramTypes );
			} catch( NoSuchMethodException ex ) {
				try {
					Class<?> superclass = targetClass.getSuperclass();
					if( superclass != Object.class && superclass != null )
						return getDeclaredMethod( superclass, name, paramTypes );
				} catch( NoSuchMethodException ex2 ) {
					// throw outer exception below
				}
				throw ex;
			}
		}

		public Object invoke( Object proxy, Method method, Object[] args )
			throws Throwable
		{
			if( method.getDeclaringClass() != Object.class ) {
				if( listenerMethod.equals( method.getName() ) )
					return targetMethod.invoke( target, passParams ? args : null );
			} else {
				// handle java.lang.Object methods
				if( "equals".equals( method.getName() ) )
					return Boolean.valueOf( proxy == args[0] );
				if( "hashCode".equals( method.getName() ) )
					return new Integer( hashCode() );
			}
			return null;
		}
	}
}
