/*
 * Copyright (C) 2003-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.runtime;

import java.beans.XMLDecoder;
import java.io.*;
import java.lang.reflect.InvocationTargetException;
import java.net.URL;
import java.util.Arrays;
import javax.xml.parsers.FactoryConfigurationError;
import com.jformdesigner.model.FormModel;

/**
 * Loads the form model from a JFormDesigner .jfd file into memory.
 * Use {@link FormCreator} to create Swing component instances.
 * <p>
 * The separation of the file loading and the component creation into two classes
 * ({@link FormLoader} and {@link FormCreator}) enables you to cache the form model
 * in memory. Using {@link FormCreator} it's possible to create multiple instances
 * of a form from one form model.
 *
 * @author Karl Tauber
 */
public class FormLoader
{
	private FormLoader() {
	}

	/**
	 * Loads a form model from the specified resource using the default
	 * class loader. Uses {@link ClassLoader#getResourceAsStream(java.lang.String)}
	 * to locate and load the form file.
	 *
	 * @param resourceName The name of the resource containing a form
	 * 		(e.g. "com/jformdesigner/examples/LoaderExample.jfd").
	 * @return The form model.
	 * @throws Exception See {@link #load(InputStream)} for details.
	 */
	public static FormModel load( String resourceName )
		throws Exception
	{
		ClassLoader classLoader = FormLoader.class.getClassLoader();
		try {
			InputStream in = classLoader.getResourceAsStream( resourceName );
			if( in == null ) {
				// resource not found --> try classloader of invoker
				// (required when a custom bean uses FormLoader in JFormDesigner)
				Class<?>[] stack = new ContextGetter().getClassContext();
				classLoader = stack[2].getClassLoader();
				if( classLoader != FormLoader.class.getClassLoader() )
					in = classLoader.getResourceAsStream( resourceName );
			}
			return load( in, classLoader );
		} catch( MultiException ex ) {
			throw ex.getExceptions()[0];
		}
	}

	/**
	 * Loads a form model from the specified resource using the specified
	 * class loader. Uses {@link ClassLoader#getResourceAsStream(java.lang.String)}
	 * to locate and load the form file.
	 *
	 * @param resourceName The name of the resource containing a form
	 * 		(e.g. "com/jformdesigner/examples/LoaderExample.jfd").
	 * @param classLoader The class loader.
	 * @return The form model.
	 * @throws Exception See {@link #load(InputStream)} for details.
	 */
	public static FormModel load( String resourceName, ClassLoader classLoader )
		throws Exception
	{
		try {
			return load( classLoader.getResourceAsStream( resourceName ), classLoader );
		} catch( MultiException ex ) {
			throw ex.getExceptions()[0];
		}
	}

	/**
	 * Loads a form model from the specified file.
	 *
	 * @param file The file containing a form.
	 * @return The form model.
	 * @throws Exception See {@link #load(InputStream)} for details.
	 */
	public static FormModel load( File file )
		throws Exception
	{
		return load( new FileInputStream( file ) );
	}

	/**
	 * Loads a form model from the given input stream.
	 * Use this method if you want read a form e.g. from a database.
	 * <p>
	 * A <code>BufferedInputStream</code> is used to improve performance.
	 *
	 * @param in The input stream. Closed when this method returns.
	 * @return The form model.
	 * @throws IllegalArgumentException If the input stream is <code>null</code>.
	 * @throws IOException If an error occurred when reading from the input stream.
	 * @throws SAXParseException If an error occurred when parsing the XML content.
	 * @throws ClassNotFoundException If a class used in the XML content is not found.
	 * @throws ClassCastException If the root object in the XML content is not a <code>FormModel</code>.
	 * @throws Exception If an other error occurred when decoding the XML content.
	 */
	public static FormModel load( InputStream in )
		throws Exception
	{
		try {
			return load( in, null );
		} catch( MultiException ex ) {
			throw ex.getExceptions()[0];
		}
	}

	/**
	 * Loads a form model from the given input stream.
	 * Use this method if you want read a form e.g. from a database.
	 * <p>
	 * A <code>BufferedInputStream</code> is used to improve performance.
	 *
	 * @param in The input stream. Closed when this method returns.
	 * @param classLoader The class loader used to load classes.
	 * @return The form model.
	 * @throws IllegalArgumentException If the input stream is <code>null</code>.
	 * @throws MultiException If a problem occurred when encoding the form model to XML.
	 * @throws ClassCastException If the root object in the XML content is not a <code>FormModel</code>.
	 *
	 * @since 1.0.1
	 */
	public static FormModel load( InputStream in, ClassLoader classLoader )
		throws MultiException
	{
		if( in == null )
			throw new IllegalArgumentException( "Input stream is null." );

		if( !(in instanceof ByteArrayInputStream) &&
			!(in instanceof BufferedInputStream) )
		  in = new BufferedInputStream( in );

		in = new V1FilterInputStream( in );
		FileHeaderInputStream headerIn = new FileHeaderInputStream( in );
		in = headerIn;

		Thread currentThread = Thread.currentThread();
		ClassLoader oldContextClassLoader = null;
		if( classLoader != null ) {
			oldContextClassLoader = currentThread.getContextClassLoader();
			currentThread.setContextClassLoader( new ContextClassLoader( classLoader, oldContextClassLoader ) );
		}

		try {
			XMLExceptionListener exceptionListener = new XMLExceptionListener();
			XMLDecoder decoder = new XMLDecoder( in, null, exceptionListener );
			Object result = decoder.readObject();
			decoder.close();

			Exception[] exceptions = exceptionListener.getExceptions();
			if( exceptions != null )
				throw new MultiException( "Failed to decode.", exceptions );

			FormModel model = (FormModel) result;
			if( classLoader != null )
				model.set_ClassLoader( classLoader );
			model.fileHeader = headerIn.getFileHeader();
			return model;
		} catch( FactoryConfigurationError ex ) {
			// may be thrown by XMLDecoder.readObject()
			throw new MultiException( "Failed to decode.",
				new Exception[] { new InvocationTargetException( ex ) } );
		} catch( JFDMLException ex ) {
			throw new MultiException( "Form is in JFDML format and requires JFormDesigner 5.1 or later.", new Exception[0] );
		} finally {
			if( classLoader != null )
				currentThread.setContextClassLoader( oldContextClassLoader );

			try {
				in.close();
			} catch( IOException ex ) {
				// ignore
			}
		}
	}

	//---- class ContextGetter ------------------------------------------------

	/**
	 * Used to make a protected method public.
	 */
	private static final class ContextGetter
		extends SecurityManager
	{
		@Override
		public Class<?>[] getClassContext() {
			return super.getClassContext();
		}
	}

	//---- class ContextClassLoader -------------------------------------------

	/**
	 * This class loader is necessary in rare cases where a XML library
	 * (e.g. xercesImpl.jar) is in the lib/ext folder of the JRE, but the
	 * given form class loader (e.g. a Eclipse plug-in class loader) does
	 * not find classes from the lib/ext folder.
	 *
	 * Without this class loader, a "javax.xml.parsers.FactoryConfigurationError:
	 * Provider org.apache.xerces.jaxp.SAXParserFactoryImpl not found" is
	 * thrown in the above scenario (xercesImpl.jar and Eclipse) because
	 * javax.xml.parsers.FactoryFinder.findJarServiceProvider() finds the file
	 * "META-INF/services/javax.xml.parsers.SAXParserFactory" from xercesImpl.jar
	 * (using the ClassLoader.getSystemResourceAsStream()), but
	 * FactoryFinder.newInstance() can not find the class (because classLoader is null).
	 *
	 * This class loader therefore delegates to the old context class loader.
	 *
	 * @since 3.1
	 */
	private static class ContextClassLoader
		extends ClassLoader
	{
		private final ClassLoader oldContextClassLoader;

		ContextClassLoader( ClassLoader classLoader, ClassLoader oldContextClassLoader ) {
			super( classLoader );
			this.oldContextClassLoader = oldContextClassLoader;
		}

		@Override
		protected Class<?> findClass( String name )
			throws ClassNotFoundException
		{
			try {
				return getParent().loadClass( name );
			} catch( ClassNotFoundException ex ) {
				try {
					return oldContextClassLoader.loadClass( name );
				} catch( ClassNotFoundException ex2 ) {
					// throw original exception
					throw ex;
				}
			}
		}

		@Override
		public URL getResource( String name ) {
			URL url = super.getResource( name );
			if( url == null )
				url = oldContextClassLoader.getResource( name );
			return url;
		}
	}

	//---- class FileHeaderInputStream ----------------------------------------

	private static class FileHeaderInputStream
		extends FilterInputStream
	{
		private static final byte[] HEADER_END_MARKER = "\n<java".getBytes();

		private byte[] buffer;
		private int count;
		private int endMarkerIndex;
		private boolean finished;

		private boolean jfdmlChecked;
		private byte[] firstBytes = new byte["JFDML".length()];
		private int firstBytesIndex;

		FileHeaderInputStream( InputStream in ) {
			super( in );
		}

		@Override
		public int read() throws IOException {
			int b = super.read();
			if( !finished && b != -1 )
				record( b );
			if( !jfdmlChecked ) {
				if( firstBytesIndex < firstBytes.length )
					firstBytes[firstBytesIndex++] = (byte) b;
				else
					checkJFDML();
			}
			return b;
		}

		@Override
		public int read( byte[] b, int off, int len ) throws IOException {
			int length = super.read( b, off, len );
			if( !finished ) {
				for( int i = 0; i < length; i++ ) {
					if( !finished )
						record( b[i] );
				}
			}
			if( !jfdmlChecked && length > 0 ) {
				int l = Math.min( length, firstBytes.length - firstBytesIndex );
				System.arraycopy( b, off, firstBytes, firstBytesIndex, l );
				firstBytesIndex += l;
				if( firstBytesIndex >= firstBytes.length )
					checkJFDML();
			}
			return length;
		}

		private void record( int b ) {
			if( buffer != null ) {
				if( count >= buffer.length ) {
					// increase buffer size
					int newLength = (buffer.length < 100) ? 512 : (buffer.length * 2);
					byte[] newBuffer = new byte[newLength];
					System.arraycopy( buffer, 0, newBuffer, 0, buffer.length );
					buffer = newBuffer;
				}

				// record character
				buffer[count++] = (byte) b;

				// check for header end
				if( b == HEADER_END_MARKER[endMarkerIndex] ) {
					endMarkerIndex++;
					if( endMarkerIndex == HEADER_END_MARKER.length ) {
						finished = true;
						count -= HEADER_END_MARKER.length;
					}
				} else
					endMarkerIndex = 0;
			} else if( b == '\n' ) {
				// start recording after first newline character
				buffer = new byte[HEADER_END_MARKER.length];
				if( b == HEADER_END_MARKER[0] ) {
					endMarkerIndex++;
					buffer[count++] = (byte) b;
				}
			}
		}

		public String getFileHeader() {
			if( count == 0 )
				return null;

			try {
				String s = new String( buffer, 0, count, "UTF-8" );
				s = s.trim();
				if( s.startsWith( "<!--" ) && s.endsWith( "-->" ) ) {
					int beginIndex = "<!--".length();
					int endIndex = s.length() - "-->".length();
					if( s.charAt( beginIndex ) == '\n' )
						beginIndex++;
					if( s.charAt( endIndex - 1 ) == '\n' )
						endIndex--;
					return s.substring( beginIndex, endIndex );
				}
			} catch( UnsupportedEncodingException ex ) {
				// ignore (should not occur)
			}
			return null;
		}

		private void checkJFDML() throws IOException {
			jfdmlChecked = true;

			if( Arrays.equals( firstBytes, "JFDML".getBytes() ) )
				throw new JFDMLException();
		}
	}

	//---- class JFDMLException -----------------------------------------------

	private static class JFDMLException
		extends RuntimeException
	{
	}
}
