/*-
 * Javoids -- Javoids is an asteroids based game (that look nothing like the original).
 * 
 * Copyright (C) 1999-2006 Patrick Mallette
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * I can be reached at parickmallette@rogers.com
 */
package javoids;

import java.awt.Component;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Label;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.Properties;
import java.util.StringTokenizer;

import javax.swing.JOptionPane;
import javax.swing.JPanel;

/**
 * A dialog to show a player's high score list.
 * @author mallette
 */
public class HighScorePanel extends JPanel
{
  /** This is the version used for serializing/deserializing (storing/retrieving) this object */
  private static final long     serialVersionUID   = 1L;
  /** an anonnymous name entry */
  private final static String   ANONYMOUS_NAME     = Messages.getString("HighScorePanel.Anonymous"); //$NON-NLS-1$
  /** an anonnymous score entry */
  private final static String   ANONYMOUS_SCORE    = "0";                                           //$NON-NLS-1$
  /** an anonnymous duration entry */
  private final static String   ANONYMOUS_DURATION = "00:00:00:00";                                 //$NON-NLS-1$
  /** an anonnymous level entry */
  private final static String   ANONYMOUS_LEVEL    = "0";                                           //$NON-NLS-1$
  /** maximum number of entries */
  private final static int      MAX_ENTRIES        = 10;
  /** a list of high score information */
  private ArrayList<ScoreEntry> highScores         = new ArrayList<ScoreEntry>(HighScorePanel.MAX_ENTRIES);

  /**
   * Contructor (default) In this constructor an empty high score list is displayed (usually because the high score table could not be loaded)
   */
  public HighScorePanel()
  {
    this.setLayout(new GridBagLayout());
    for (int row = 0;row < HighScorePanel.MAX_ENTRIES;row++)
      this.highScores.add(new ScoreEntry(HighScorePanel.ANONYMOUS_NAME,HighScorePanel.ANONYMOUS_SCORE,HighScorePanel.ANONYMOUS_DURATION,HighScorePanel.ANONYMOUS_LEVEL));
    this.populatePanel();
  }

  /**
   * Contructor In this constructor a high score list is passed for display.
   * @param scores a list of user names and high scores
   */
  public HighScorePanel(ArrayList<ScoreEntry> scores)
  {
    this.highScores = scores;
    this.setLayout(new GridBagLayout());
    this.populatePanel();
  }

  /**
   * Fill the high score panel with the high score table.
   */
  private void populatePanel()
  {
    int row = 0;
    this.removeAll();
    Collections.sort(this.highScores);
    for (ScoreEntry scoreEntry : this.highScores)
    {
      Label label1 = new Label(scoreEntry.name);
      Label label2 = new Label(scoreEntry.score);
      Label label3 = new Label(scoreEntry.duration);
      Label label4 = new Label(scoreEntry.level);
      this.add(label1,new GridBagConstraints(0,row,1,1,1.0,0.5,GridBagConstraints.WEST,GridBagConstraints.EAST,new Insets(0,0,0,0),0,0));
      this.add(label2,new GridBagConstraints(1,row,1,1,0.0,0.5,GridBagConstraints.WEST,GridBagConstraints.EAST,new Insets(0,0,0,0),0,0));
      this.add(label3,new GridBagConstraints(2,row,1,1,0.0,0.5,GridBagConstraints.WEST,GridBagConstraints.EAST,new Insets(0,0,0,0),0,0));
      this.add(label4,new GridBagConstraints(3,row,1,1,0.0,0.5,GridBagConstraints.WEST,GridBagConstraints.EAST,new Insets(0,0,0,0),0,0));
      row++;
    }
  }

  /**
   * @param component the frame owning this dialog
   * @param score the player's score
   * @param duration the duration of the game
   * @param level the level that was reached
   * @return true if the score was inseted into the list
   */
  public boolean insertScore(Component component,int score,String duration,int level)
  {
    Collections.sort(this.highScores);
    int i = HighScorePanel.MAX_ENTRIES - 1;
    boolean found = false;

    while (!found && i >= 0)
      if (score > Integer.valueOf(this.highScores.get(i).score).intValue())
        found = true;
      else
        i--;
    if (found)
    {
      String name;
      if (i >= 0)
        name = this.highScores.get(i).name;
      else
        name = HighScorePanel.ANONYMOUS_NAME;
      name = JOptionPane.showInputDialog(component,Messages.getString("HighScorePanel.NamePrompt"),name); //$NON-NLS-1$
      if (name == null)
        name = HighScorePanel.ANONYMOUS_NAME;
      this.highScores.add(new ScoreEntry(name,String.valueOf(score),duration,String.valueOf(level)));
      Collections.sort(this.highScores);
      this.highScores = new ArrayList<ScoreEntry>(this.highScores.subList(0,HighScorePanel.MAX_ENTRIES));
      this.populatePanel();
    }
    return found;
  }

  /**
   * @param url the location of the high score table
   * @return true if the high score table was loaded into the dialog
   */
  public boolean load(URL url) // get the high scores from a URL
  {
    ArrayList<ScoreEntry> scores = new ArrayList<ScoreEntry>(HighScorePanel.MAX_ENTRIES);
    String key = null;
    String value = null;
    boolean success = true;
    
    try
    {
      int i = 0;
      
      if (Debug.debugHighScores)
        System.out.printf(Messages.getString("HighScorePanel.LoadingHighScores")); //$NON-NLS-1$
      if (url != null)
      {
        FileInputStream inputStream = new FileInputStream(url.getFile());
        Properties properties = new Properties();
        Enumeration<?> enumeration;
        properties.load(inputStream);
        enumeration = properties.propertyNames();
        while (enumeration.hasMoreElements() && i < HighScorePanel.MAX_ENTRIES)
        {
          if (Debug.debugHighScoresInternal)
            System.out.print(Messages.getString("KeyMap.Period")); //$NON-NLS-1$
          key = (String)enumeration.nextElement();
          value = properties.getProperty(key);
          if (Debug.debugHighScoresInternal)
            System.out.printf(Messages.getString("HighScorePanel.DebugKeyValue"),key,value); //$NON-NLS-1$
          StringTokenizer tokenizer = new StringTokenizer(value,"|"); //$NON-NLS-1$
          String name = tokenizer.nextToken();
          String score = tokenizer.nextToken();
          String duration = tokenizer.nextToken();
          String level = tokenizer.nextToken();
          try
          {
            Integer.valueOf(score);
          }
          catch (NumberFormatException numberFormatException)
          {
            System.out.printf(Messages.getString("HighScorePanel.ErrorHighScoreTable"),value); //$NON-NLS-1$
            name = HighScorePanel.ANONYMOUS_NAME;
            score = "0"; //$NON-NLS-1$
          }
          scores.add(new ScoreEntry(name,score,duration,level));
          i++;
        }
        if (Debug.debugHighScores)
          System.out.printf(Messages.getString("HighScorePanel.Done")); //$NON-NLS-1$
      }
      else if (Debug.debugHighScores)
        System.out.printf(Messages.getString("HighScorePanel.ErrorHighScoreTable")); //$NON-NLS-1$
    }
    catch (FileNotFoundException fileNotFoundException)
    {
      System.out.printf(Messages.getString("HighScorePanel.UnableFindHighScore"),url.toString()); //$NON-NLS-1$
      success = false;
    }
    catch (IOException iOException)
    {
      System.out.printf(Messages.getString("HighScorePanel.UnableFindHighScore"),url.toString(),iOException.getMessage()); //$NON-NLS-1$
      success = false;
    }
    if (success)
    {
      Collections.sort(scores);
      if (scores.size() > 0)
        this.highScores = scores;
      this.populatePanel();
    }
    return success;
  }

  /**
   * @param url the location where the high score table will be saved
   */
  public void save(URL url) // save the high scores to a URL
  {
    try
    {
      if (Debug.debugHighScores)
        System.out.printf(Messages.getString("HighScorePanel.SavingHighScores")); //$NON-NLS-1$
      FileOutputStream outputStream = new FileOutputStream(url.getFile());
      Properties properties = new Properties();
      String key;
      String data;
      int row = 0;
      Collections.sort(this.highScores);
      for (ScoreEntry scoreEntry : this.highScores)
      {
        if (Debug.debugHighScoresInternal)
          System.out.print(Messages.getString("KeyMap.Period")); //$NON-NLS-1$
        key = Integer.toString(row);
        data = String.format(Messages.getString("HighScorePanel.ScoreEntry"),scoreEntry.name,scoreEntry.score,scoreEntry.duration,scoreEntry.level); //$NON-NLS-1$
        if (Debug.debugHighScoresInternal)
          System.out.printf(Messages.getString("HighScorePanel.DebugKeyValue"),key,data); //$NON-NLS-1$
        if (key != null && data != null)
          properties.put(key,data);
        row++;
      }
      properties.store(outputStream,null);
      if (Debug.debugHighScores)
        System.out.printf(Messages.getString("HighScorePanel.Done")); //$NON-NLS-1$
    }
    catch (IOException iOException)
    {
      System.out.printf(Messages.getString("HighScorePanel.ErrorHighScoreProperties"),iOException.getMessage()); //$NON-NLS-1$
    }
    this.populatePanel();
  }
}

/**
 * A class for holding the high score information of one entry of the high score table.
 * @author mallette
 */
class ScoreEntry implements Comparable<ScoreEntry>
{
  /** player's name */
  String name;
  /** player's score */
  String score;
  /** plaey's duration of play */
  String duration;
  /** player's level reched */
  String level;

  /**
   * @param _name the player's name
   * @param _score the player's score
   * @param _duration the duration of play
   * @param _level the level reached
   */
  ScoreEntry(String _name,String _score,String _duration,String _level)
  {
    this.name = _name;
    this.score = _score;
    this.duration = _duration;
    this.level = _level;
  }

  /**
   * Compare one score entry to another.
   * @param scoreEntry the other score entry to compare this score entry to.
   * @return the results of the comparison
   */
  /*
   * (non-Javadoc)
   * @see java.lang.Comparable#compareTo(T)
   */
  public int compareTo(ScoreEntry scoreEntry)
  {
    return Integer.valueOf(scoreEntry.score).intValue() - Integer.valueOf(this.score).intValue();
  }
}
