/*-
 * Javoids -- Javoids is an asteroids based game (that look nothing like the original).
 * 
 * Copyright (C) 1999-2006 Patrick Mallette
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * I can be reached at parickmallette@rogers.com
 */
package javoids;

import static javoids.BasicSprite.Gravity.SINK;
import static javoids.ImageMap.Name.MINE1;
import static javoids.ImageMap.Name.PACMAN_FRUIT;
import static javoids.Shapes.Shape.POINT;
import static javoids.SoundMap.Sound.MINEDIE;
import static javoids.SoundMap.Sound.MINEHIT;
import static javoids.SoundMap.Sound.PACMAN_EATDOT;
import static javoids.SoundMap.Sound.PACMAN_EATFRUIT;

/* Mine----------------------- */
/**
 * A mine class that damages ship sprites when they collide.
 * @author mallette
 */
public final class Mine extends ShieldedSprite
{
  /** This is the version used for serializing/deserializing (storing/retrieving) this object */
  private static final long   serialVersionUID = 1L;
  /** The maximum number of these sprites allowed in the game */
  protected final static int  MAX_NUMBER       = 4;
  /** the sprite's maximum size */
  protected final static int  MAX_SIZE         = 100;
  /** the sprite's minimum size */
  protected final static int  MIN_SIZE         = 20;
  /** the sprite's default size */
  protected final static int  DEFAULT_SIZE     = 30;  // This should match the size of the image!
  /** the chance the sprite will move */
  private final static double CHANCE_MOVE      = 0.60;
  /** the chance the sprite will rotate */
  private final static double CHANCE_ROTATE    = 0.35;
  /** the amount to rotate by */
  private final static double ROTATION         = 30;

  /**
   * Constructor
   * @param parent the parent of this sprite
   * @param health the health information
   * @param move the movment information
   * @param _size the size this should be
   */
  public Mine(BasicSprite parent,Health health,Move move,int _size)
  {
    super(parent,health,move,null,_size,MINE1,Shapes.getAreas(POINT,_size,Mine.MAX_SIZE),Shapes.getColors(POINT));
    this.setHoming(true); // all mines are homing, but 1 % are gravity homing mines
    if (Math.random() < 0.01)
      this.setGravity(SINK);
    if (Debug.debugMine)
    {
      this.setHealth(health);
      this.setMove(new Move(move.getScreen(),0,0,0,0));
    }
    if (Math.random() < Mine.CHANCE_MOVE)
      move.setMaxVelocity(0);
    if (Math.random() < Mine.CHANCE_ROTATE)
      this.setRotation(Math.random() * Mine.ROTATION * (Math.random() < 0.5 ? -1 : 1));
    this.accelerate(move.getMaxVelocity());
    this.setShape(POINT);
    this.setDisplayAreas(false);
  }

  /**
   * @return the maximum size
   */
  public int getMaximumSize()
  {
    return Mine.MAX_SIZE;
  }

  /**
   * @return the minimum size
   */
  public int getMinimumSize()
  {
    return Mine.MIN_SIZE;
  }

  /**
   * @return the default size
   */
  public int getDefaultSize()
  {
    return Mine.DEFAULT_SIZE;
  }

  /**
   * @param _size the desired sprite size
   * @return the desired size, or the minimum, or maximum size allowable if under or over the limits respectively.
   */
  public static int getValidSize(int _size)
  {
    return _size >= Mine.MIN_SIZE ? _size <= Mine.MAX_SIZE ? _size : Mine.MAX_SIZE : Mine.MIN_SIZE;
  }

  /**
   * @param _damage the damage this sprite sustained
   * @return the new damage level
   */
  /*
   * (non-Javadoc)
   * @see javoids.BasicSprite#modifyDamage(int)
   */
  @Override
  public int modifyDamage(int _damage)
  {
    this.setPoints(this.getDamage());
    int damage = super.modifyDamage(_damage);
    if (this.isAlive())
      Media.play(BasicSprite.getPacmanGame() ? PACMAN_EATDOT : MINEHIT);
    else
      Media.play(BasicSprite.getPacmanGame() ? PACMAN_EATFRUIT : MINEDIE);
    return damage;
  }

  /**
   * @return the image name the sprite is currently using
   */
  /*
   * (non-Javadoc)
   * @see javoids.Sprite#getImageNumber()
   */
  @Override
  public ImageMap.Name getImageNumber()
  {
    return BasicSprite.getPacmanGame() ? PACMAN_FRUIT : this.getOriginalImage();
  }
}
/* Mine----------------------- */
