/*-
 * Javoids -- Javoids is an asteroids based game (that look nothing like the original).
 * 
 * Copyright (C) 1999-2006 Patrick Mallette
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * I can be reached at parickmallette@rogers.com
 */
package javoids;

import static javoids.Shapes.Shape.CIRCLE;
import static javoids.Shapes.Shape.HEXAGON;
import static javoids.Shapes.Shape.JAVOID;
import static javoids.Shapes.Shape.LANDER;
import static javoids.Shapes.Shape.LINE_2;
import static javoids.Shapes.Shape.LINE_4;
import static javoids.Shapes.Shape.NO_SHAPE;
import static javoids.Shapes.Shape.PENTAGON;
import static javoids.Shapes.Shape.POINT;
import static javoids.Shapes.Shape.SPIKES;
import static javoids.Shapes.Shape.SQUARE;
import static javoids.Shapes.Shape.STAR;
import static javoids.Shapes.Shape.TRIANGLE30;
import static javoids.Shapes.Shape.TRIANGLE45;

import java.awt.Color;
import java.awt.geom.AffineTransform;
import java.awt.geom.Area;
import java.awt.geom.Ellipse2D;
import java.awt.geom.GeneralPath;
import java.awt.geom.Rectangle2D;
import java.io.Serializable;
import java.util.HashMap;
import java.util.Vector;

/* Shapes-------------- */
/**
 * A class to represent the shapes in this game.
 * @author mallette
 */
public final class Shapes implements Serializable
{
  /**
   * An enum of available shapes.
   * @author mallette
   */
  // types
  public enum Shape
  {
    /** a place holder to represent that the item has no shape */
    NO_SHAPE,
    /** a single point */
    POINT,
    /** a circle */
    CIRCLE,
    /** an isosceles triangle with a 30 degree angle at the top */
    TRIANGLE30,
    /** a square (4 sided shape with equal side lengths) */
    SQUARE,
    /** a pentagon (5 sided shape with equal side lengths) */
    PENTAGON,
    /** a hexagon (6 sided shape with equal side lengths) */
    HEXAGON,
    /** a 6 pointed star */
    STAR,
    /** a lunar lander like shape (kind of looks like a bug with the current colors) */
    LANDER,
    /** a Javoid (kind of like an asteroid, but more extensible lol) */
    JAVOID,
    /** an isosceles triangle with a 45 degree angle at the top */
    TRIANGLE45,
    /** two lines slanted at 45 degree angles in front of a shielded sprite */
    LINE_2,
    /** four lines slanted at 45 degree angles in front and behind a shielded sprite */
    LINE_4,
    /** siz lines like an asterisk */
    SPIKES;
  }

  /** This is the version used for serializing/deserializing (storing/retrieving) this object */
  private static final long                   serialVersionUID = 1L;
  /** constant for sin(30 degrees) */
  private final static double                 sin30            = Math.sin(30 * Common.toRadians); // 30 makes an equilateral triangle
  /** constant for cos(30 degrees) */
  private final static double                 cos30            = Math.cos(30 * Common.toRadians); // 30 makes an equilateral triangle
  /** constant for sin(30 degrees) / 2 */
  private final static double                 sin30_2          = Shapes.sin30 / 2;
  /** constant for cos(30 degrees) / 2 */
  private final static double                 cos30_2          = Shapes.cos30 / 2;
  /** constant for sin(45 degrees) */
  private final static double                 sin45            = Math.sin(45 * Common.toRadians);
  /** constant for cos(45 degrees) */
  private final static double                 cos45            = Math.cos(45 * Common.toRadians);
  /** the maximum size for a shape */
  protected final static int                  MAX_SIZE         = 100;
  /** the minimum size for a shape */
  protected final static int                  MIN_SIZE         = 1;
  /** the default size of a shape */
  protected final static int                  DEFAULT_SIZE     = 20;                      // This should match the size of the image!
  /** a mapping of shape names to areas that make up the shape */
  private static HashMap<Shape,Vector<Area>>  shapeMap = new HashMap<Shape,Vector<Area>>();
  /** a mapping of shape names to colors that go with the areas of the shape */
  private static HashMap<Shape,Vector<Color>> colorMap = new HashMap<Shape,Vector<Color>>();

  static
  {
    Rectangle2D.Double rectangleLeft = new Rectangle2D.Double(-Shapes.MAX_SIZE / 2,-Shapes.MAX_SIZE / 4,4,Shapes.MAX_SIZE / 2);
    Rectangle2D.Double rectangleRight = new Rectangle2D.Double(Shapes.MAX_SIZE / 2,-Shapes.MAX_SIZE / 4,4,Shapes.MAX_SIZE / 2);
    Rectangle2D.Double rectangle;
    double tempSize = Shapes.MAX_SIZE - 4;
    Area area = null;
    Area area1 = null;
    Area area2 = null;
    GeneralPath path;
    AffineTransform transform;
    Vector<Area> areas;
    Vector<Color> colors;

    // make "no shape"
    areas = new Vector<Area>();
    colors = new Vector<Color>();
    area = new Area(new Ellipse2D.Double(-Shapes.MAX_SIZE / 2,-Shapes.MAX_SIZE / 2,Shapes.MAX_SIZE,Shapes.MAX_SIZE));
    areas.add(area);
    colors.add(Color.black);
    Shapes.shapeMap.put(NO_SHAPE,areas);
    Shapes.colorMap.put(NO_SHAPE,colors);
    // make point shape
    areas = new Vector<Area>();
    colors = new Vector<Color>();
    area = new Area(new Ellipse2D.Double(-Shapes.MAX_SIZE / 2,-Shapes.MAX_SIZE / 2,Shapes.MAX_SIZE,Shapes.MAX_SIZE));
    areas.add(area);
    colors.add(new Color(128,64,0,255)); // brown
    Shapes.shapeMap.put(POINT,areas);
    Shapes.colorMap.put(POINT,colors);
    // make circle shape
    areas = new Vector<Area>();
    colors = new Vector<Color>();
    area = new Area(new Ellipse2D.Double(-Shapes.MAX_SIZE / 2,-Shapes.MAX_SIZE / 2,Shapes.MAX_SIZE,Shapes.MAX_SIZE));
    area.subtract(new Area(new Ellipse2D.Double(-tempSize / 2,-tempSize / 2,tempSize,tempSize)));
    areas.add(area);
    colors.add(Color.black);
    Shapes.shapeMap.put(CIRCLE,areas);
    Shapes.colorMap.put(CIRCLE,colors);
    // make triangle30 shape
    areas = new Vector<Area>();
    colors = new Vector<Color>();
    path = new GeneralPath();
    path.moveTo((int)(-Shapes.MAX_SIZE * Shapes.cos30_2),(int)(Shapes.MAX_SIZE * Shapes.sin30));
    path.lineTo(0,-Shapes.MAX_SIZE / 2);
    path.lineTo((int)(Shapes.MAX_SIZE * Shapes.cos30_2),(int)(Shapes.MAX_SIZE * Shapes.sin30));
    area = new Area(path);
    areas.add(area);
    colors.add(Color.red);
    Shapes.shapeMap.put(TRIANGLE30,areas);
    Shapes.colorMap.put(TRIANGLE30,colors);
    // make triangle45 shape
    areas = new Vector<Area>();
    colors = new Vector<Color>();
    path = new GeneralPath();
    path.moveTo((int)(-Shapes.MAX_SIZE * Shapes.cos45 / 2),(int)(Shapes.MAX_SIZE * Shapes.sin45 / 2));
    path.lineTo(0,-Shapes.MAX_SIZE / 2);
    path.lineTo((int)(Shapes.MAX_SIZE * Shapes.cos45 / 2),(int)(Shapes.MAX_SIZE * Shapes.sin45 / 2));
    area = new Area(path);
    areas.add(area);
    colors.add(Color.red);
    Shapes.shapeMap.put(TRIANGLE45,areas);
    Shapes.colorMap.put(TRIANGLE45,colors);
    // make star shape
    areas = new Vector<Area>();
    colors = new Vector<Color>();
    path = new GeneralPath();
    path.moveTo((int)(-Shapes.MAX_SIZE * Shapes.cos30_2),(int)(Shapes.MAX_SIZE * Shapes.sin30_2));
    path.lineTo(0,-Shapes.MAX_SIZE / 2);
    path.lineTo((int)(Shapes.MAX_SIZE * Shapes.cos30_2),(int)(Shapes.MAX_SIZE * Shapes.sin30_2));
    area = new Area(path);
    transform = new AffineTransform();
    transform.rotate(90.0 * Common.toRadians);
    path.transform(transform);
    area.add(new Area(path));
    transform.rotate(90.0 * Common.toRadians);
    path.transform(transform);
    area.add(new Area(path));
    transform.rotate(90.0 * Common.toRadians);
    path.transform(transform);
    area.add(new Area(path));
    areas.add(area);
    colors.add(Color.red);
    Shapes.shapeMap.put(STAR,areas);
    Shapes.colorMap.put(STAR,colors);
    // make square shape
    areas = new Vector<Area>();
    colors = new Vector<Color>();
    area = new Area(new Rectangle2D.Double(-Shapes.MAX_SIZE / 2,-Shapes.MAX_SIZE / 2,Shapes.MAX_SIZE,Shapes.MAX_SIZE));
    areas.add(area);
    colors.add(Color.red);
    Shapes.shapeMap.put(SQUARE,areas);
    Shapes.colorMap.put(SQUARE,colors);
    // make pentagon shape
    areas = new Vector<Area>();
    colors = new Vector<Color>();
    path = new GeneralPath();
    path.moveTo((int)(-Shapes.MAX_SIZE * Shapes.cos45 / 2),Shapes.MAX_SIZE / 2);
    path.lineTo(-Shapes.MAX_SIZE / 2,0);
    path.lineTo(0,-Shapes.MAX_SIZE / 2);
    path.lineTo(Shapes.MAX_SIZE / 2,0);
    path.lineTo((int)(Shapes.MAX_SIZE * Shapes.cos45 / 2),Shapes.MAX_SIZE / 2);
    area = new Area(path);
    areas.add(area);
    colors.add(Color.red);
    Shapes.shapeMap.put(PENTAGON,areas);
    Shapes.colorMap.put(PENTAGON,colors);
    // make hexagon shape
    areas = new Vector<Area>();
    colors = new Vector<Color>();
    path = new GeneralPath();
    path.moveTo(-Shapes.MAX_SIZE / 2,(int)(Shapes.MAX_SIZE * Shapes.sin45 / 2));
    path.lineTo(-Shapes.MAX_SIZE / 2,(int)(-Shapes.MAX_SIZE * Shapes.sin45 / 2));
    path.lineTo(0,-Shapes.MAX_SIZE * 3 / 4);
    path.lineTo(Shapes.MAX_SIZE / 2,(int)(-Shapes.MAX_SIZE * Shapes.sin45 / 2));
    path.lineTo(Shapes.MAX_SIZE / 2,(int)(Shapes.MAX_SIZE * Shapes.sin45 / 2));
    path.lineTo(0,Shapes.MAX_SIZE * 3 / 4);
    area = new Area(path);
    areas.add(area);
    colors.add(Color.red);
    Shapes.shapeMap.put(HEXAGON,areas);
    Shapes.colorMap.put(HEXAGON,colors);
    // make lander shape
    areas = new Vector<Area>();
    colors = new Vector<Color>();
    area = new Area(new Ellipse2D.Double(-Shapes.MAX_SIZE / 4,-Shapes.MAX_SIZE / 2,Shapes.MAX_SIZE / 2.0,Shapes.MAX_SIZE / 2.0));
    areas.add(area);
    colors.add(Color.cyan);
    area = new Area(new Ellipse2D.Double(-Shapes.MAX_SIZE / 2,-Shapes.MAX_SIZE / 4,Shapes.MAX_SIZE,Shapes.MAX_SIZE / 2.0));
    areas.add(area);
    colors.add(Color.green);
    path = new GeneralPath();
    path.moveTo(0,0);
    path.lineTo(-Shapes.MAX_SIZE / 2,Shapes.MAX_SIZE / 2);
    path.lineTo(0,Shapes.MAX_SIZE / 10);
    path.lineTo(0,0);
    path.lineTo(Shapes.MAX_SIZE / 2,Shapes.MAX_SIZE / 2);
    path.lineTo(0,Shapes.MAX_SIZE / 10);
    path.lineTo(0,0);
    area = new Area(path);
    areas.add(area);
    colors.add(Color.red);
    Shapes.shapeMap.put(LANDER,areas);
    Shapes.colorMap.put(LANDER,colors);
    // make javoid shape
    areas = new Vector<Area>();
    colors = new Vector<Color>();
    path = new GeneralPath();
    path.moveTo(-50,-20);
    path.lineTo(-10,-50);
    path.lineTo(30,-45);
    path.lineTo(50,15);
    path.lineTo(10,10);
    path.lineTo(0,50);
    path.lineTo(-40,30);
    path.lineTo(-50,-20);
    area = new Area(path);
    areas.add(area);
    colors.add(Color.black);
    transform = new AffineTransform();
    transform.scale(0.95,0.95);
    path.transform(transform);
    area = new Area(path);
    areas.add(area);
    colors.add(new Color(128,64,0,255));
    Shapes.shapeMap.put(JAVOID,areas);
    Shapes.colorMap.put(JAVOID,colors);
    // make 2 line shape
    areas = new Vector<Area>();
    colors = new Vector<Color>();
    area1 = new Area(rectangleLeft);
    transform = new AffineTransform();
    transform.rotate(45 * Common.toRadians);
    area1.transform(transform);
    area2 = new Area(rectangleRight);
    transform = new AffineTransform();
    transform.rotate(-45 * Common.toRadians);
    area2.transform(transform);
    area = new Area();
    area.add(new Area(area1));
    area.add(new Area(area2));
    areas.add(area);
    colors.add(Color.white);
    Shapes.shapeMap.put(LINE_2,areas);
    Shapes.colorMap.put(LINE_2,colors);
    // make 4 line shape
    areas = new Vector<Area>();
    colors = new Vector<Color>();
    area1 = new Area(rectangleLeft);
    transform = new AffineTransform();
    transform.rotate(45 * Common.toRadians);
    area1.transform(transform);
    area2 = new Area(rectangleRight);
    transform = new AffineTransform();
    transform.rotate(-45 * Common.toRadians);
    area2.transform(transform);
    area = new Area();
    area.add(new Area(area1));
    area.add(new Area(area2));
    area1 = new Area(rectangleLeft);
    transform = new AffineTransform();
    transform.rotate(-135 * Common.toRadians);
    area1.transform(transform);
    area2 = new Area(rectangleRight);
    transform = new AffineTransform();
    transform.rotate(135 * Common.toRadians);
    area2.transform(transform);
    area.add(new Area(area1));
    area.add(new Area(area2));
    areas.add(area);
    colors.add(Color.white);
    Shapes.shapeMap.put(LINE_4,areas);
    Shapes.colorMap.put(LINE_4,colors);
    // make spikes
    areas = new Vector<Area>();
    colors = new Vector<Color>();
    transform = new AffineTransform();
    rectangle = new Rectangle2D.Double(-2,-Shapes.MAX_SIZE / 2,4,Shapes.MAX_SIZE / 2);
    transform.rotate(45 * Common.toRadians);
    area = new Area(rectangle);
    for (int i = 0;i < 8;i++)
    {
      area.add(new Area(rectangle));
      area.transform(transform);
    }
    areas.add(area);
    colors.add(Color.white);
    Shapes.shapeMap.put(SPIKES,areas);
    Shapes.colorMap.put(SPIKES,colors);
  }

  /**
   * @param number the integer representation of a shape (in order of the shape name appearing in the enum)
   * @return the shape corresponding to the number, or no shape if the number is invalid
   */
  public static Shapes.Shape getShape(int number)
  {
    Shapes.Shape shape = NO_SHAPE;
    if (number >= 0 && number < Shape.values().length)
      shape = Shape.values()[number];
    else
    {
      shape = NO_SHAPE;
      System.out.printf(Messages.getString("Shapes.ShapeMissing"),Integer.valueOf(number)); //$NON-NLS-1$
    }
    return shape;
  }

  /**
   * @param shape the name of a shape
   * @param _size the size of the areas used to represent this shape
   * @param maximumSize the maximum size this shape can be
   * @return the areas representing the shape
   */
  public static Vector<Area> getAreas(Shapes.Shape shape,int _size,int maximumSize)
  {
    Vector<Area> vector = new Vector<Area>();
    double multiplier = (double)_size / maximumSize;
    if (Shapes.shapeMap.containsKey(shape))
    {
      AffineTransform transform = new AffineTransform();
      Area newArea = null;

      if (Shapes.shapeMap.get(shape).size() > 0)
      {
        transform.scale(multiplier,multiplier);
      }
      for (Area area : Shapes.shapeMap.get(shape))
      {
        newArea = new Area(area);
        newArea.transform(transform);
        vector.add(newArea);
      }
    }
    return vector;
  }

  /**
   * @param shape the shape name
   * @return the colors associated with each shape
   */
  public static Vector<Color> getColors(Shapes.Shape shape)
  {
    Vector<Color> vector = new Vector<Color>();
    if (Shapes.colorMap.containsKey(shape))
      for (Color color : Shapes.colorMap.get(shape))
        vector.add(color);
    return vector;
  }

  /**
   * Provide a String representation of this object.
   * @return String A representation of the object for debugging.
   */
  @Override
  public String toString()
  {
    return Messages.getString("Shapes.Name") + Shapes.MAX_SIZE; //$NON-NLS-1$
  }
}
/* Shapes-------------- */
