/*-
 * Javoids -- Javoids is an asteroids based game (that look nothing like the original).
 * 
 * Copyright (C) 1999-2006 Patrick Mallette
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * I can be reached at parickmallette@rogers.com
 */
package javoids;

import static javoids.Shapes.Shape.CIRCLE;
import static javoids.Shapes.Shape.NO_SHAPE;
import static javoids.Shapes.Shape.POINT;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.geom.AffineTransform;
import java.awt.geom.Area;
import java.util.Iterator;
import java.util.Vector;

/* Sprite--------------------- */
/**
 * A sprite class to have a moving game object.
 * @author mallette
 */
public abstract class Sprite extends BasicSprite
{
  /** This is the version used for serializing/deserializing (storing/retrieving) this object */
  private static final long serialVersionUID = 1L;
  /** the shield protecting the sprite */
  private Shield            shield;
  /** the image name representing the sprite */
  private ImageMap.Name     imageNumber;
  /** the original image name */
  private ImageMap.Name     originalImageNumber;
  /** the sprite's shape */
  private Shapes.Shape      shape            = NO_SHAPE; // picture for drawn sprites
  /** the initial x coordinate */
  private double            initialX;                   // for rotating around now that a new object isn't created every time
  /** the initial y coordinate */
  private double            initialY;
  /** the areas representing this sprite's shape */
  private Vector<Area>      areas;
  /** the colors that correspond to a sprite's areas */
  private Vector<Color>     colors;

  /**
   * Constructor
   * @param parent the parent of this sprite
   * @param health the health information
   * @param move the movment information
   * @param _size the size this should be
   * @param _imageNumber the image numbeer to use for this sprite
   * @param _areas the areas used to represent or draw the sprite (needed even if using an image for area collision detection)
   * @param _colors the colors of each area (in the same order as the areas)
   */
  public Sprite(BasicSprite parent,Health health,Move move,int _size,ImageMap.Name _imageNumber,Vector<Area> _areas,Vector<Color> _colors)
  {
    super(parent,health,move,_size);
    this.setImage(_imageNumber);
    this.setAreas(_areas);
    this.setColors(_colors);
    this.setHoming(false);
    this.setPlayer(false);
    this.setAutomaticMove(false);
    this.originalImageNumber = _imageNumber;
  }

  /**
   * @return the colors associate with this sprite
   */
  /*
   * (non-Javadoc)
   * @see javoids.BasicSprite#getColors()
   */
  @Override
  public Vector<Color> getColors()
  {
    return this.colors;
  }

  /**
   * @return get the shape name for this sprite
   */
  public Shapes.Shape getShape()
  {
    return this.shape;
  }

  /**
   * @param _shape set the shape name for this sprite
   */
  public void setShape(Shapes.Shape _shape)
  {
    this.shape = _shape;
    this.setAreas();
    this.setColors();
  }

  /**
   * @return the areas associated with this sprite
   */
  /*
   * (non-Javadoc)
   * @see javoids.BasicSprite#getAreas()
   */
  @Override
  public Vector<Area> getAreas()
  {
    return this.areas;
  }

  /**
   * @param _areas the areas that represent the shape of this sprite
   */
  public void setAreas(Vector<Area> _areas)
  {
    Area area = null;
    AffineTransform transform = new AffineTransform();
    if (this.areas == null)
      this.areas = new Vector<Area>();
    this.areas.clear();
    transform.scale((double)this.getSize() / this.getMaximumSize(),(double)this.getSize() / this.getMaximumSize());
    if (_areas != null)
      for (Area _area : _areas)
      {
        area = new Area(_area);
        area.transform(transform);
        this.areas.add(area);
      }
  }

  /**
   * Set the areas that represent the sprite's shape.
   */
  private void setAreas()
  {
    this.getAreas().clear();
    switch (this.getShape())
    {
      case NO_SHAPE :
        System.out.printf(Messages.getString("Sprite.ShapeMissing")); //$NON-NLS-1$
        break;
      case POINT :
        this.getAreas().addAll(Shapes.getAreas(POINT,this.getSize(),this.getMaximumSize()));
        this.getAreas().addAll(Shapes.getAreas(CIRCLE,this.getSize(),this.getMaximumSize()));
        break;
      default :
        this.getAreas().addAll(Shapes.getAreas(this.getShape(),this.getSize(),this.getMaximumSize()));
        break;
    }
  }

  /**
   * @param _colors the colors that correspond to the areas of this sprite
   */
  public void setColors(Vector<Color> _colors)
  {
    if (this.colors == null)
      this.colors = new Vector<Color>();
    this.colors.clear();
    if (_colors != null)
      this.colors.addAll(_colors);
  }

  /**
   * Set the colors that go with the sprit'es areas.
   */
  protected void setColors()
  {
    this.getColors().clear();
    switch (this.getShape())
    {
      case NO_SHAPE :
        break;
      case POINT :
        this.getColors().addAll(Shapes.getColors(POINT));
        this.getColors().addAll(Shapes.getColors(CIRCLE));
        break;
      default :
        this.getColors().addAll(Shapes.getColors(this.getShape()));
        break;
    }
  }

  /**
   * @param _size the size of the sprite
   */
  /*
   * (non-Javadoc)
   * @see javoids.BasicSprite#setSize(int)
   */
  @Override
  public void setSize(int _size)
  {
    this.size = _size;
    this.initialX = this.getX(); // reset initial position to adjust for size changes
    this.initialY = this.getY();
  }

  /**
   * Draw the shapes or images onto the foreground image (for double buffering). Draw parts of the sprite in different locations for when parts wrap around a screen border.
   * @param g2d the graphics context used to draw this sprite
   * @param foregroundImage the image use to draw this sprite on
   */
  /*
   * (non-Javadoc)
   * @see javoids.BasicSprite#draw(java.awt.Graphics2D, java.awt.Graphics2D)
   */
  @Override
  public void draw(Graphics2D g2d,Graphics2D foregroundImage)
  {
    if (this.isAlive())
    {
      boolean xlt = this.getX() - this.getSize() / 2 < this.getScreen().x;
      boolean ylt = this.getY() - this.getSize() / 2 < this.getScreen().y;
      boolean xgt = this.getX() + this.getSize() / 2 > this.getScreen().x + this.getScreen().width;
      boolean ygt = this.getY() + this.getSize() / 2 > this.getScreen().y + this.getScreen().height;
      int screenXTotal = this.getScreen().x + this.getScreen().width;
      int screenYTotal = this.getScreen().y + this.getScreen().height;
      int modifierX = xlt ? screenXTotal : xgt ? -screenXTotal : 0;
      int modifierY = ylt ? screenYTotal : ygt ? -screenYTotal : 0;
      int totalX = (int)this.getX() + modifierX;
      int totalY = (int)this.getY() + modifierY;
      double multiplier = this.getMultiplier();
      foregroundImage.setStroke(new BasicStroke(3,BasicStroke.CAP_ROUND,BasicStroke.JOIN_ROUND)); // width,cap,join combined
      g2d.setTransform(g2d.getDeviceConfiguration().getDefaultTransform());
      g2d.translate(this.getX(),this.getY());
      g2d.rotate(this.getDirection() * Common.toRadians);
      g2d.scale(multiplier,multiplier);
      this.drawShapes(g2d,foregroundImage);
      if (xlt && ylt || xlt && ygt || xgt && ylt || xgt && ygt)
      {
        g2d.setTransform(g2d.getDeviceConfiguration().getDefaultTransform());
        g2d.translate(totalX,totalY);
        g2d.rotate(this.getDirection() * Common.toRadians);
        g2d.scale(multiplier,multiplier);
        this.drawShapes(g2d,foregroundImage);
      }
      if (xlt || xgt)
      {
        g2d.setTransform(g2d.getDeviceConfiguration().getDefaultTransform());
        g2d.translate(totalX,this.getY());
        g2d.rotate(this.getDirection() * Common.toRadians);
        g2d.scale(multiplier,multiplier);
        this.drawShapes(g2d,foregroundImage);
      }
      if (ylt || ygt)
      {
        g2d.setTransform(g2d.getDeviceConfiguration().getDefaultTransform());
        g2d.translate(this.getX(),totalY);
        g2d.rotate(this.getDirection() * Common.toRadians);
        g2d.scale(multiplier,multiplier);
        this.drawShapes(g2d,foregroundImage);
      }
      foregroundImage.setTransform(g2d.getDeviceConfiguration().getDefaultTransform()); // restore the defaults
    }
  }

  /**
   * Draw an image on the screen.
   * @param g2d the graphics context used to draw this sprite
   * @param foregroundImage the image use to draw this sprite on
   */
  private void drawImage(Graphics2D g2d,Graphics2D foregroundImage)
  {
    Image image = Media.getImage(this.getImageNumber());
    if (image != null)
    {
      g2d.translate(-this.getSize() / this.getMultiplier() / 2.0,-this.getSize() / this.getMultiplier() / 2.0);
      foregroundImage.setTransform(g2d.getTransform());
      foregroundImage.drawImage(image,null,null);
    }
  }

  /**
   * Draw a single area on the screen.
   * @param g2d the graphics context used to draw this sprite
   * @param foregroundImage the image use to draw this sprite on
   * @param area the are to draw
   */
  private void drawShape(Graphics2D g2d,Graphics2D foregroundImage,Area area)
  {
    if (area != null)
    {
      g2d.translate(0,0);
      foregroundImage.setTransform(g2d.getTransform());
      foregroundImage.fill(area);
    }
  }

  /**
   * Draw all of a sprite's shapes on the screen
   * @param g2d the graphics context used to draw this sprite
   * @param foregroundImage the image use to draw this sprite on
   */
  private void drawShapes(Graphics2D g2d,Graphics2D foregroundImage)
  {
    // there's always a pacman image (or there should be!)
    if (this.isDisplayAreas() && !BasicSprite.getPacmanGame())
    {
      Iterator<Color> iterator = this.getColors().iterator();
      for (Area area : this.getAreas())
      {
        if (iterator.hasNext())
        {
          Color color = iterator.next();
          foregroundImage.setColor(color);
        }
        else
          foregroundImage.setColor(Color.white);
        this.drawShape(g2d,foregroundImage,area);
      }
    }
    else
      this.drawImage(g2d,foregroundImage);
  }

  /**
   * @return the image name for this sprite's image
   */
  public abstract ImageMap.Name getImageNumber();

  /**
   * @return the original image name for this sprite's image
   */
  public ImageMap.Name getOriginalImage()
  {
    return this.originalImageNumber;
  }

  /**
   * @return the scaling factor for this sprite's image/shapes
   */
  /*
   * (non-Javadoc)
   * @see javoids.BasicSprite#getMultiplier()
   */
  @Override
  public double getMultiplier()
  {
    double value;

    if (!this.isDisplayAreas() && Media.getImage(this.getImageNumber()) != null)
    {
      value = this.getSize();
      value /= Media.getImage(this.getImageNumber()).getWidth(null);
    }
    else
      value = 1.0;
    return value;
  }

  /**
   * @param _imageNumber set the image name for this sprite
   */
  public void setImage(ImageMap.Name _imageNumber)
  {
    this.imageNumber = _imageNumber;
  }

  /**
   * Provide a String representation of this object.
   * @return String A representation of the object for debugging.
   */
  @Override
  public String toString()
  {
    return String.format(Messages.getString("Sprite.ToString"),super.toString(),this.shield,this.imageNumber,this.originalImageNumber,Double.valueOf(this.initialX),Double.valueOf(this.initialY),this.areas); //$NON-NLS-1$
  }
}
/* Sprite--------------------- */
