package biege;
import java.awt.geom.Point2D;
import java.text.DecimalFormat;

public class Position {
    public static int DIRECTION_NORTH = 0;
    public static int DIRECTION_SOUTH = 4;
    public static int DIRECTION_EAST = 8;
    public static int DIRECTION_WEST = 12;

    private Point2D       point; // Internal point representation

    // Format pattern for outputting the x or y value of a Position
    private static final String pattern = "######.##";

    // Used for double equality test. Two doubles are "equal" if they are
    // within EPSILON of each other.
    private static final double EPSILON = 0.0001;
 
    private static final double DELTA = 0.025;

    /**
     * Line is described by ay + bx + c = 0.
     * http://www.uni-koblenz.de/~jboedeck/uva/classLine.html
     */
    private static double ma; // Line: a coefficient
    private static double mb; // Line: b coefficient
    private static double mc; // Line: c coefficient

    // Formats position to 2 decimal places
    private static final DecimalFormat myFormatter = new DecimalFormat(pattern);

    /**
     * Ensure that a position doesn't go outside the game's boundaries in the
     * x-direction.
     * @param x incoming x-coordinate
     * @return double Returns Game.LEFT <= x-coordinate <= Game.RIGHT
     */
    public static double boundX(double x)
    {   if ( x-EPSILON < AdventureApp.LEFT) return AdventureApp.LEFT;
        if ( x+EPSILON > AdventureApp.RIGHT) return AdventureApp.RIGHT;
        return x;
    } // boundX

    /**
     * Ensure that a position doesn't go outside the game's boundaries in the
     * y-direction.
     * @param y incoming x-coordinate
     * @return double Returns Game.BOTTOM <= y-coordinate <= Game.TOP
     */
    public static double boundY(double y)
    {   if ( y-EPSILON < AdventureApp.TOP) return AdventureApp.TOP;
        if ( y+EPSILON > AdventureApp.BOTTOM) return AdventureApp.BOTTOM;
        return y;
    } // boundY

    /**
     * Creates a Position with given coordinates (newX, newY).
     * @param newX x-axis coordinate
     * @param newY y-axis coordinate
     */
    public Position(double newX, double newY)
    {   point = new Point2D.Double(newX, newY);
    } // Position

    /**
     * Creates a Position with default values of (0.0, 0.0).
     */
    public Position()
    {   this(0.0, 0.0);
    } // Position

    /**
     * Creates a Position with same coordinates as newPosition.
     * @param newPosition coordinates to assign
     */
    public Position(Position newPosition)
    {   this(newPosition.getX(), newPosition.getY());
    } // Position

    /**
     * @return double Returns the x-coordinate of the point.
     */
    public double getX()
    {   return point.getX();
    } // getX

    /**
     * @return double Returns the y-coordinate of the point.
     */
    public double getY()
    {   return point.getY();
    } // getY

    /**
     * @return Returns the location of the Position.
     */
    public Position getLocation()
    {   return new Position(this.getX(), this.getY());
    } // getLocation

    public static Position closestRectangleIntersection(Position p, double west, double north, double east, double south)
    {
      double x = p.getX();
      double y = p.getY();

      Position closest;
      Position temp;

      //Calculate the east intersection
      temp = new Position(east, y);
      //by default it is closest now
      closest = temp;
      //Calculate the west intersection

      temp = new Position(west, y);
      //See if it should be closer
      if (distance(temp, p) < distance(closest, p))
        closest = temp;

      //Calculate the north intersection
      temp = new Position(x, north);
      //See if it should be closer
      if (distance(temp, p) < distance(closest, p))
        closest = temp;

      //Calculate the south intersection
      temp = new Position(x, south);
      //See if it should be closer
      if (distance(temp, p) < distance(closest, p))
        closest = temp;

      return closest;

    }

    /**
     * Changes Position to given coordinates (newX, newY).
     * @param newX x-axis coordinate
     * @param newY y-axis coordinate
     */
    public void setLocation(double newX, double newY)
    {   point.setLocation(newX, newY);
    } // setLocation

    /**
     * Changes Position to given coordinates (newX, newY).
     * @param newX x-axis coordinate
     * @param newY y-axis coordinate
     */
    public void setLocation(Position newPosition)
    {   point.setLocation(newPosition.getX(), newPosition.getY());
    } // setLocation

    /**
     * Calculates the distance between this point and the specified point.
     * @param otherLocation other end-point
     * @return double Returns the distance between this point and the other.
     */
    public double distance(Position otherLocation)
    {   return point.distance(otherLocation.point);
    } // distance

    /**
     * Calculates the distance between this point and the specified point.
     * @param x x-axis coordinate of the other point
     * @param y y-axis coordinate of the other point
     * @return double Returns the distance between this point and the other.
     */
    public double distance(double x, double y)
    {   return point.distance(x, y);
    } // distance

    /**
     * Calculates the distance between two points.
     * @param x1 x-axis coordinate of the first point
     * @param y1 y-axis coordinate of the first point
     * @param x2 x-axis coordinate of the second point
     * @param y2 y-axis coordinate of the second point
     * @return double Returns the distance between the two points.
     */
    public static double distance(double x1, double y1, double x2, double y2)
    {   return Point2D.distance(x1, y1, x2, y2);
    } // distance

    /**
     * Calculates the distance between two points.
     * @param p1 first point
     * @param p2 second point
     * @return double Returns the distance between the two points.
     */
    public static double distance(Position p1, Position p2)
    {   return Point2D.distance(p1.getX(), p1.getY(), p2.getX(), p2.getY());
    } // distance

    /**
     * Given the current location and the goal position, determine the
     * appropriate unit direction vector.
     * @param currentLocation current location of the player
     * @param goalLocation current position of the flag
     * @return Position direction unit vector
     */
    public static Position calculateDirectionVector(Position currentLocation,
        Position goalLocation)
    {   double distance = goalLocation.distance(currentLocation, goalLocation);
        double xCoord = (goalLocation.getX() - currentLocation.getX())/distance;
        double yCoord = (goalLocation.getY() - currentLocation.getY())/distance;
        return new Position(xCoord, yCoord);
    } // calculateDirectionVector

    public static boolean isInRectangle(Position testPos, Position topLeft, Position bottomRight)
    {
      if (testPos.getX()> topLeft.getX() && testPos.getX() < bottomRight.getX())
      {
        if (testPos.getY()>topLeft.getY()&&testPos.getY()< bottomRight.getY())
          return true;
      }
      return false;
    }

    /**
     * Calculate the velocity vector given the current speed and a unit
     * direction vector.
     * @param direction unit direction vector
     * @param speed speed at the current moment in time.
     * @return Position Returns the velocity vector
     */
    public static Position calculateVelocityVector(Position direction,
        double speed)
    {   return new Position(direction.getX()*speed, direction.getY()*speed);
    } // calculateVelocityVector

    /**
     * Given the current position and a velocity vector, calculate the next
     * point.
     * @param currentLocation current location of some object
     * @param velocity velocity vector
     * @return Position Returns the new location of the object.
     */
    public static Position calculateNewPosition(Position currentLocation,
        Position velocity)
    {   double x = currentLocation.getX() + velocity.getX();
        double y = currentLocation.getY() + velocity.getY();
        return new Position(x, y);
    } // calculateNewPosition

    /**
     * Given the current and goal locations, plus the current speed, calculates
     * the next position.
     * @param currentLocation current location of some object
     * @param goalLocation location towards which the object is moving
     * @param speed current speed of the object
     * @return Position Returns the next location of the object
     */
    public static Position calculateNewPosition(Position currentLocation,
        Position goalLocation, double speed)
    {   Position directionVector;
        Position velocityVector;
        directionVector = Position.calculateDirectionVector(currentLocation,
            goalLocation);
        velocityVector = Position.calculateVelocityVector(directionVector, speed);
        Position newPosition =
            Position.calculateNewPosition(currentLocation, velocityVector);

        // To prevent overshooting the target, check if distance we want to move
        // is less than or equal to the distance we can move at the given speed.
        // If true, move to the goalLocation instead.
        if (currentLocation.distance(goalLocation) <= currentLocation.distance(newPosition))
        {   newPosition = goalLocation;
        } // if
        return newPosition;
    } // calculateLocation

    /**
     * Performs the Pythagorean Formula on the given parameters and stores the
     * results. It returns the number of found coordinates.
     * http://www.uni-koblenz.de/~jboedeck/uva/classLine.html
     * @param a a parameter in the formula
     * @param b b parameter in the formula
     * @param c c parameter in the formula
     * @param results x-value contains the first result of the formula while the
     * y-value contains the second result of the formula
     * @return int Returns the number of found x-coordinates.
     */
    private static int pythagorean(double a, double b, double c,
        Position results)
    {   int numResults = 0;                 // number of results found
        double dDiscr = b*b - 4*a*c;        // discriminant is b^2 - 4*a*c
        double s1 = -b/(2*a);               // first result possible
        double s2 = 0.0;                    // second result possible
        if (Math.abs(dDiscr) < EPSILON)     // if discriminant == 0,
        {   s1 = -b/(2*a);                  //     there is only one solution
            numResults = 1;
        } else if (dDiscr < 0)              // if discriminant < 0,
        {   numResults = 0;                 //     there are no results
        } else                              // if discriminant > 0
        {   dDiscr = Math.sqrt(dDiscr);     //     there are two solutions
            s1 = (-b + dDiscr)/(2*a);
            s2 = (-b - dDiscr)/(2*a);
            numResults = 2;
        } // if
        results.setLocation(s1, s2);
        return numResults;
    } // pythagorean

    /**
     * @param point starting point
     * @param vector unit direction vector
     * @return Position returns a new point on the line defined by point+vector
     */
    private static Position addPointToVector(Position point, Position vector)
    {   double x = point.getX() + vector.getX();
        double y = point.getY() + vector.getY();
        return new Position(x, y);
    } // addPointToVector

    /**
     * Calculate the coefficients of a line given two points on the line.
     * Sets the instance variables {ma, mb, mc}
     * @param p1 first point on the line
     * @param p2 second point on the line
     */
    private static void makeLineFromTwoPoints(Position p1, Position p2)
    {   // y + bx +c = 0 implies y = -bx - c, where -b is the slope
        // and c = -y - bx
        double dA = 1.0;
        double dB = 0.0;
        double dC = 0.0;
        double dTemp = p2.getX() - p1.getX(); // determine the slope
        if (Math.abs(dTemp) < EPSILON)
        {   // ay + bx + c = 0 with vertical slope => a = 0, b = 1
            dA = 0.0;
            dB = 1.0;
        } else
        {   // y = (-b)x - c with -b the slope of the line
            dA = 1.0;
            dB = -(p2.getY() - p1.getY())/dTemp;
        } // if
        // ay + bx + c = 0 => c = -a*y - b*x
        dC = -dA*p2.getY() - dB*p2.getX();
        ma = dA;
        mb = dB;
        mc = dC;
    } // makeLineFromTwoPoints

    /**
     * Calculation of the intersection of a line and a circle come from
     * http://www.uni-koblenz.de/~jboedeck/uva/classLine.html
     * The C++ code was then translated to Java.
     * @param center center of the circle
     * @param radius radius of the circle
     * @param vector unit vector for direction of line that intersects the circle
     * @param origin starting point of the unit vector
     * @param result1 coordinates of the first intersection point
     * @param result2 coordinates of the second intersection point
     */
    public static void calcCircleVectorIntersection(Position center,
        double radius, Position vector, Position origin,
        Position result1, Position result2)
    {   // From the direction vector, calculate the coefficients {a,b,c} of the
        // line defined by ay + bx + c = 0.
        Position.makeLineFromTwoPoints(origin,
            Position.addPointToVector(origin, vector));

        Position result = new Position();// Storage for returned calculations.
        int iSol;                        // Number of solutions calculated
        double dSol1, dSol2;             // Calculated y intersection values
        double h = center.getX();        // Location of circle's center.
        double k = center.getY();

        // line: x = -c/b (if a == 0)
        // circle: (x-h)^2 + (y-k)^2 = r^2, with h = center.x and k = center.y
        // fill in: (-c/b-h)^2 + y^2 -2ky + k^2 - r^2 = 0
        //          y^2 - 2ky + (-c/b-h)^2 + k^2 - r^2 = 0
        // and determine solutions for y using Pythagorean formula
        if (Math.abs(ma) < EPSILON)
        {   iSol = Position.pythagorean(1, -2*k, ((-mc/mb)-h)*((-mc/mb)-h)
                +k*k - radius*radius, result);
            dSol1 = result.getX();
            dSol2 = result.getY();
            result1.setLocation(-mc/mb, dSol1);
            result2.setLocation(-mc/mb, dSol2);
            return;
        } // if

        // ay + bx + c = 0 => y = -b/a x - c/a, with da = -b/a and db = -c/a
        // circle: (x-h)^2 + (y-k)^2 = r^2, with h = center.x and k = center.y
        // fill in: x^2 - 2hx + h^2 + (da*x-db)^2 - 2k(da*x-db) + k^2 - r^2 = 0
        //          x^2 - 2hx + h^2 + da^2*x^2 + 2da*db*x + db^2 - 2k - 2k*da*x
        //              - 2k*db + k^2 - r^2 = 0
        //         (1+da^2)*x^2 + 2(da*db-h-k*da)*x + h^2 + db^2 - 2k*db
        //              + k^2 - r^2
        // and determine solutions for x using pythagorean formula
        // fill in x in original line equation to get y coordinate
        double da = -mb/ma;                  // Holds partial-results
        double db = -mc/ma;
        double dA = 1 + da*da;
        double dB = 2*(da*db - h - k*da);
        double dC = h*h + db*db - 2*k*db + k*k - radius*radius;
        iSol = Position.pythagorean(dA, dB, dC, result);
        dSol1 = result.getX();
        dSol2 = result.getY();
        result1.setLocation(dSol1, da*dSol1 + db);
        result2.setLocation(dSol2, da*dSol2 + db);
    } // calcCircleVectorIntersection

    /**
     * @return String Returns the coordinates of the point as "(x,y)".
     */
    public String toString()
    {   return "(" + myFormatter.format(this.getX()) + ", "
            + myFormatter.format(this.getY()) + ")";
    } // toString

    public static void main(String [] args) {
        Position p1 = new Position(0,90);
        Position p2 = new Position(0,100);
        double speed = 5.0;

        // Test distance and new position calculations
        double distance = Position.distance(p1,p2); //p1.distance(p2);
        System.out.println(p1+" to "+p2+" is "+distance);
        Position direction = Position.calculateDirectionVector(p1, p2);
        System.out.println("direction vector is "+direction);
        Position velocity = Position.calculateVelocityVector(direction, speed);
        System.out.println("velocity vector is "+velocity);
        Position newPosition = Position.calculateNewPosition(p1, velocity);
        System.out.println("new position via method 1 is "+newPosition);
        newPosition = Position.calculateNewPosition(p1, p2, speed);
        System.out.println("new position via method 2 is "+newPosition);

        // Test calculation of intersections points of lines with a circle
        Position circleCenter = new Position(0,0);
        double radius = 0.5;
        Position p3 = new Position(0, -2);
        Position p4 = new Position(1, 1);
        Position p5 = new Position(1, 0);
        Position v3 = Position.calculateDirectionVector(p3,circleCenter);
        Position v4 = Position.calculateDirectionVector(p4,circleCenter);
        Position v5 = Position.calculateDirectionVector(p5,circleCenter);
        Position result1 = new Position();
        Position result2 = new Position();
        Position.calcCircleVectorIntersection(circleCenter, radius, v3, p3,
            result1, result2);
        System.out.println("Intersections of "+p3+" with circle are: "+result1
            +", "+result2);
        Position.calcCircleVectorIntersection(circleCenter, radius, v4, p4,
            result1, result2);
        System.out.println("Intersections of "+p4+" with circle are: "+result1
            +", "+result2);
        Position.calcCircleVectorIntersection(circleCenter, radius, v5, p5,
            result1, result2);
        System.out.println("Intersections of "+p5+" with circle are: "+result1
            +", "+result2);
    } // main

} // Position