// ---------------------------------------------------------------------------
//  Virtual Bus Implementation
//  Copyright (c) cisc 1999.
// ---------------------------------------------------------------------------
//  $Id: device.h,v 1.14 1999/07/15 12:33:53 cisc Exp $

#ifndef common_device_h
#define common_device_h

#include "types.h"

// ---------------------------------------------------------------------------
//  Device
//
class Device
{
public:
    typedef uint32 ID;
    typedef uint (Device::*InFuncPtr)(uint port);
    typedef void (Device::*OutFuncPtr)(uint port, uint data);
    typedef void (Device::*TimeFunc)(uint arg);
    struct Descriptor
    {
        const InFuncPtr* indef;
        const OutFuncPtr* outdef;
    };

public:
    Device(const ID& _id) : id(_id) {}
    virtual ~Device() {}

    const ID& GetID() const { return id; }
    virtual const Descriptor* GetDesc() const { return 0; }

public:
    virtual uint GetStatusSize() { return 0; }
    virtual bool LoadStatus(void* status) { return false; }
    virtual bool SaveStatus(void* status) { return false; }

private:
    ID id;
};


// ---------------------------------------------------------------------------
//  MemoryBus
//  ԂƃANZXi񋟂NX
//  oNƂɎANZX֐蓖Ă邱ƂłB
//
//  ANZX֐̏ꍇ́A֐̈ɓnʎqoNƂɐݒł
//  ܂AoNƂɂꂼEFCgݒ肷邱Ƃł
//  ֐Ǝ̎ʂɂ̓|C^ LSB (idbit) 𗘗p邽߁A
//  |C^ even-aligned łȂ΂ȂȂ
//
class MemoryBus
{
public:
    typedef uint (MEMCALL * ReadFuncPtr)  (void* inst, uint addr);
    typedef void (MEMCALL * WriteFuncPtr) (void* inst, uint addr, uint data);

    struct Page
    {
        void* read;
        void* write;
        void* inst;
        int wait;
    };

    enum
    {
        pagebits = 10,
        pagemask = (1 << pagebits) - 1,
        idbit = PTR_IDBIT,
    };

public:
    MemoryBus();
    ~MemoryBus();

    bool Init(uint npages, Page* pages=0);

    void SetWriteMemory(uint addr, void* ptr);
    void SetReadMemory(uint addr, void* ptr);
    void SetMemory(uint addr, void* ptr);
    void SetFunc(uint addr, void* inst, ReadFuncPtr rd, WriteFuncPtr wr);

    void SetWriteMemorys(uint addr, uint length, uint8* ptr);
    void SetReadMemorys(uint addr, uint length, uint8* ptr);
    void SetMemorys(uint addr, uint length, uint8* ptr);
    void SetFuncs(uint addr, uint length, void* inst, ReadFuncPtr rd, WriteFuncPtr wr);

    void SetWait(uint addr, uint wait);
    void SetWaits(uint addr, uint length, uint wait);

    uint Read8(uint addr);
    void Write8(uint addr, uint data);

    const Page* GetPageTable();

private:
    static void MEMCALL wrdummy(void*, uint, uint);
    static uint MEMCALL rddummy(void*, uint);

    Page* pages;
    bool ownpages;
};

// ---------------------------------------------------------------------------

template<class ID, class T>
class SimpleList
{
private:
    struct Node
    {
        ID id;
        T entry;
        Node* next;
    };

public:
    SimpleList() { node = 0; }

    bool Add(ID& id, T& t)
    {
        Node* n = new Node;
        if (n)
        {
            n->id, n->entry = t, n->next = node;
            node = n;
            return true;
        }
        else
            return false;
    }
    bool Del(ID& id)
    {
#ifndef __OS2__
        for (Node** r = &node; *r; *r=&((*r)->next))
#else
        for (Node** r = &node; *r; *r=((*r)->next))
#endif
        {
            if ((*r)->id == id)
            {
                Node* d = *r;
                *r = d->next;
                delete d;
                return true;
            }
        }
        return false;
    }
    T* Find(ID& id)
    {
        Node* n = FindNode(id);
        return n ? &n->entry : 0;
    }

private:
    Node* FindNode(ID& id)
    {
        for (Node* n = node; node; node=node->next)
            if (node->id == id)
                return node;
        return 0;
    }
    Node* node;
};


// ---------------------------------------------------------------------------
//  IO Ԃ񋟂NX
//  MemoryBus Ƃ̍ő̈Ⴂ͂ЂƂ̃oNɕ̃ANZX֐
//  ݒł邱
//
class IOBus
{
public:
    enum
    {
        iobankbits = 0,         // 1 oÑTCY(rbg)
    };

    typedef Device::InFuncPtr InFuncPtr;
    typedef Device::OutFuncPtr OutFuncPtr;

    enum ConnectRule
    {
        end = 0,
        portin  = 1,
        portout = 2,
        sync = 4,
    };
    struct Connector
    {
        ushort bank;
        uint8 rule;
        uint8 id;
    };
    struct InBank
    {
        Device* device;
        InFuncPtr func;
        InBank* next;
    };
    struct OutBank
    {
        Device* device;
        OutFuncPtr func;
        OutBank* next;
    };

public:
    IOBus();
    ~IOBus();

    bool Init(uint nports);
    bool Connect(Device* device, const Connector* connector);
    bool Disconnect(Device* device);

    InBank* GetIns() { return ins; }
    OutBank* GetOuts() { return outs; }
    uint8* GetFlags() { return flags; }

    bool ConnectIn(uint bank, Device* device, InFuncPtr func);
    bool ConnectOut(uint bank, Device* device, OutFuncPtr func);

    bool IsSyncPort(uint port);
    uint In(uint port);
    void Out(uint port, uint data);

    // inactive line is high
    static uint Active(uint data, uint bits) { return data | ~bits; }

    //
    uint GetStatusSize();
    bool SaveStatus(void* status);
    bool LoadStatus(void* status);

private:
    class DummyIO : public Device
    {
    public:
        DummyIO() : Device(0) {}
        ~DummyIO() {}

        uint dummyin(uint);
        void dummyout(uint, uint);
    };
    struct StatusTag
    {
        Device::ID id;
        uint32 size;
    };

private:
    SimpleList<Device::ID, int> devlist;

    InBank* ins;
    OutBank* outs;
    uint8* flags;

    uint banksize;
    static DummyIO dummyio;
};

// ---------------------------------------------------------------------------
//  Bus
//
class Bus : public MemoryBus, public IOBus
{
public:
    Bus() {}
    ~Bus() {}

    bool Init(uint nports, uint npages, Page* pages = 0);
};

// ---------------------------------------------------------------------------

inline bool IOBus::IsSyncPort(uint port)
{
    return (flags[port >> iobankbits] & 1) != 0;
}

#endif // common_device_h
