// ---------------------------------------------------------------------------
//  M88 - PC-8801 Emulator
//  Copyright (C) cisc 1997, 1999.
// ---------------------------------------------------------------------------
//  $Id: sound.cpp,v 1.16 1999/07/04 14:05:52 cisc Exp $

#include "headers.h"
#include "types.h"
#include "misc.h"
#include "schedule.h"
#include "pc88/sound.h"
#include "pc88/pc88.h"
#include "pc88/config.h"
#include "opnif.h"

#define LOGNAME "sound"
#include "diag.h"

using namespace PC8801;

// ---------------------------------------------------------------------------
//  Ej
//
Sound::Sound(const ID& id)
: Device(id)
{
    enabled = false;
    mixingbuf = 0;
//  beepmask = 0xa0;

    cfgflg = 0;
}

Sound::~Sound()
{
    Cleanup();
}

// ---------------------------------------------------------------------------
//  Ƃ
//  rate = SOUND_55K, 44100, 22050, 11025, 0 ̂ǂꂩ
//
bool Sound::Init(PC88* pc88, uint rate, uint32 opnclock, int bufsize)
{
    pc = pc88;
    opn1 = pc->GetOPN1();
    opn2 = pc->GetOPN2();
    prevtime = pc->GetCPUTick();
    enabled = false;
    mixthreshold = 16;

    opn1->SetSound(this);
    opn2->SetSound(this);

    if (!SetRate(rate, opnclock, bufsize))
        return false;

    return true;
}

// ---------------------------------------------------------------------------
//  [gݒ
//  rate:       SOUND_55K, 44100, 22050, 11025, 0 ̂ǂꂩ
//  clock:      OPN ɗ^NbN
//  bufsize:    obt@ (TvP?)
//
bool Sound::SetRate(uint rate, uint32 clock, int bufsize)
{
    opn1->SetRate(clock, rate);
    opn2->SetRate(clock, rate);

    if (rate == SOUND_55K)
        rate = 44100;

    enabled = false;
    samplingrate = rate;

    SoundBuffer::Cleanup();
    delete[] mixingbuf; mixingbuf = 0;

    buffersize = bufsize;
    if (bufsize > 0)
    {
        if (!SoundBuffer::Init(2, bufsize))
            return false;

        mixingbuf = new int32[2 * bufsize];
        if (!mixingbuf)
            return false;

        rate50 = rate / 50;
        tdiff = 0;
        enabled = true;
    }
    pslice = 0;
    bslice = 0;
    port40 = 0;
    bcount = 0;
    bperiod = int(2400.0 / rate * (1 << 15));
    return true;
}

// ---------------------------------------------------------------------------
//  Еt
//
void Sound::Cleanup()
{
    SoundBuffer::Cleanup();
    delete[] mixingbuf; mixingbuf = 0;
}

// ---------------------------------------------------------------------------
//  ԏ
//
void Sound::Update()
{
    uint32 currenttime = pc->GetCPUTick();

    uint32 time = currenttime - prevtime;
    if (enabled && time > mixthreshold)
    {
        prevtime = currenttime;
        // nsamples = oߎ(s) * TvO[g
        // sample = ticks * rate / clock / 100000
        // sample = ticks * (rate/50) / clock / 2000

        int a = (time * rate50 / pc->GetCPUSpeed()) + tdiff;
        int samples = a / 2000;
        tdiff = a % 2000;

        LOG1("Store = %5d samples", samples);
        Put(samples);
    }
//  LOG0("\n");
}

// ---------------------------------------------------------------------------
//  
//
void Sound::Mix(Sample* dest, int nsamples)
{
    nsamples = Min(nsamples, buffersize);
    memset(mixingbuf, 0, nsamples * 2 * sizeof(int32));

    opn1->Mix(mixingbuf, nsamples);
    opn2->Mix(mixingbuf, nsamples);

    MixBeep(mixingbuf, nsamples);

    int32* src = mixingbuf;
    for (; nsamples>0; nsamples--)
    {
        *dest++ = Limit(*src++, 32767, -32768);
        *dest++ = Limit(*src++, 32767, -32768);
    }
}

// ---------------------------------------------------------------------------
//  r[v
//
//  0-2000
//   0 -  4     1111
//   5 -  9     0111
//  10 - 14     0011
//  15 - 19     0001
//
void Sound::MixBeep(int32* dest, int nsamples)
{
    int i;
    int p = port40 & 0x80 ? 0 : -1;
    int b = port40 & 0x20 ? 0 : -1;

    uint ps = pslice, bs = bslice;
    pslice = bslice = 0;

    int sample = 0;
    for (i=0; i<4; i++)
    {
        if (ps < 500) p ^= 0x10000;
        if (bs < 500) b ^= 0x10000;

        sample += ((b & bcount) | p) & 0x10000 ? 0x800 : 0;
        bcount += bperiod;
        ps -= 500;
        bs -= 500;
    }
    *dest++ += sample; *dest++ += sample;

    if (p | b)
    {
        for (i=nsamples-1; i>0; i--)
        {
            sample = 0;
            for (int j=0; j<4; j++)
            {
                sample += ((b & bcount) | p) & 0x10000 ? 0x800 : 0;
//              sample += bcount & 0x10000 ? 0x800 : 0;
                bcount += bperiod;
            }
            *dest++ += sample; *dest++ += sample;
        }
    }
}

// ---------------------------------------------------------------------------
//  ʐݒ
//
void Sound::SetVolume(const Config* config)
{
    opn1->SetVolume(config);
    opn2->SetVolume(config);
}

// ---------------------------------------------------------------------------
//  BEEP Port ւ Out
//
void Sound::Out40(uint, uint data)
{
    data &= p40mask;
    int i = data ^ port40;
    if (i & 0xa0)
    {
        Update();
        port40 = data;
        if (i & 0x80)
            pslice = tdiff;
        if (i & 0x20)
            bslice = tdiff;
    }
}

// ---------------------------------------------------------------------------
//  ݒXV
//
void Sound::ApplyConfig(const Config* config)
{
    SetVolume(config);
    mixthreshold = (config->flags & Config::precisemixing) ? 100 : 2000;
    p40mask = (config->flags & Config::disablesing) ? 0x20 : 0xa0;
    port40 &= p40mask;
}

// ---------------------------------------------------------------------------
//  JE^̒l킹Ȃ悤IɍXV
//
void Sound::UpdateCounter(uint, uint)
{
    if ((pc->GetCPUTick() - prevtime) > 400000)
    {
        opn1->TimeEvent(1);
        opn2->TimeEvent(1);
    }
}

// ---------------------------------------------------------------------------
//  device description
//
const Device::Descriptor Sound::descriptor = { 0, outdef };

const Device::OutFuncPtr Sound::outdef[] =
{
#ifndef __OS2__
    static_cast<OutFuncPtr> (Out40),
    static_cast<OutFuncPtr> (UpdateCounter),
#else
    (Device::OutFuncPtr) (Out40),
    (Device::OutFuncPtr) (UpdateCounter),
#endif
};
