/////////////////////////////////////////////////////////////////////////////
// Name:        filefn.cpp
// Purpose:     File- and directory-related functions
// Author:      Julian Smart
// Modified by:
// Created:     29/01/98
// RCS-ID:      $Id: filefn.cpp,v 1.249 2005/09/20 09:52:14 VZ Exp $
// Copyright:   (c) 1998 Julian Smart
// Licence:     wxWindows licence
/////////////////////////////////////////////////////////////////////////////

// ============================================================================
// declarations
// ============================================================================

// ----------------------------------------------------------------------------
// headers
// ----------------------------------------------------------------------------

#if defined(__GNUG__) && !defined(NO_GCC_PRAGMA)
    #pragma implementation "filefn.h"
#endif

// For compilers that support precompilation, includes "wx.h".
#include "wx/wxprec.h"
#include "wx/defs.h"

#ifdef __BORLANDC__
    #pragma hdrstop
#endif

#include "wx/utils.h"
#include "wx/intl.h"
#include "wx/file.h" // This does include filefn.h
#include "wx/filename.h"
#include "wx/dir.h"

// there are just too many of those...
#ifdef __VISUALC__
    #pragma warning(disable:4706)   // assignment within conditional expression
#endif // VC++

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#if !defined(__WATCOMC__)
    #if !(defined(_MSC_VER) && (_MSC_VER > 800))
        #include <errno.h>
    #endif
#endif

#if defined(__WXMAC__)
    #include  "wx/mac/private.h"  // includes mac headers
#endif

#include "wx/log.h"

#ifdef __WINDOWS__
    #include "wx/msw/private.h"
    #include "wx/msw/mslu.h"

    // sys/cygwin.h is needed for cygwin_conv_to_full_win32_path()
    //
    // note that it must be included after <windows.h>
    #ifdef __GNUWIN32__
        #ifdef __CYGWIN__
            #include <sys/cygwin.h>
        #endif
    #endif // __GNUWIN32__

    // io.h is needed for _get_osfhandle()
    // Already included by filefn.h for many Windows compilers
    #if defined __MWERKS__ || defined __CYGWIN__
        #include <io.h>
    #endif
#endif // __WINDOWS__

#if defined(__VMS__)
    #include <fab.h>
#endif

// TODO: Borland probably has _wgetcwd as well?
#ifdef _MSC_VER
    #define HAVE_WGETCWD
#endif

// ----------------------------------------------------------------------------
// constants
// ----------------------------------------------------------------------------

#ifndef _MAXPATHLEN
    #define _MAXPATHLEN 1024
#endif

#ifdef __WXMAC__
#    include "MoreFilesX.h"
#endif

// ----------------------------------------------------------------------------
// private globals
// ----------------------------------------------------------------------------

// MT-FIXME: get rid of this horror and all code using it
static wxChar wxFileFunctionsBuffer[4*_MAXPATHLEN];

#if defined(__VISAGECPP__) && __IBMCPP__ >= 400
//
// VisualAge C++ V4.0 cannot have any external linkage const decs
// in headers included by more than one primary source
//
const int wxInvalidOffset = -1;
#endif

// ----------------------------------------------------------------------------
// macros
// ----------------------------------------------------------------------------

// we need to translate Mac filenames before passing them to OS functions
#define OS_FILENAME(s) (s.fn_str())

// ============================================================================
// implementation
// ============================================================================

#ifdef wxNEED_WX_UNISTD_H

WXDLLEXPORT int wxStat( const wxChar *file_name, wxStructStat *buf )
{
    return stat( wxConvFile.cWX2MB( file_name ), buf );
}

WXDLLEXPORT int wxAccess( const wxChar *pathname, int mode )
{
    return access( wxConvFile.cWX2MB( pathname ), mode );
}

WXDLLEXPORT int wxOpen( const wxChar *pathname, int flags, mode_t mode )
{
    return open( wxConvFile.cWX2MB( pathname ), flags, mode );
}

#endif
   // wxNEED_WX_UNISTD_H

// ----------------------------------------------------------------------------
// wxPathList
// ----------------------------------------------------------------------------

// IMPLEMENT_DYNAMIC_CLASS(wxPathList, wxStringList)

static inline wxChar* MYcopystring(const wxString& s)
{
    wxChar* copy = new wxChar[s.length() + 1];
    return wxStrcpy(copy, s.c_str());
}

static inline wxChar* MYcopystring(const wxChar* s)
{
    wxChar* copy = new wxChar[wxStrlen(s) + 1];
    return wxStrcpy(copy, s);
}

void wxPathList::Add (const wxString& path)
{
    wxStringList::Add (WXSTRINGCAST path);
}

// Add paths e.g. from the PATH environment variable
void wxPathList::AddEnvList (const wxString& WXUNUSED_IN_WINCE(envVariable))
{
    // No environment variables on WinCE
#ifndef __WXWINCE__
    static const wxChar PATH_TOKS[] =
#if defined(__WINDOWS__) || defined(__OS2__)
        /*
        The space has been removed from the tokenizers, otherwise a
        path such as "C:\Program Files" would be split into 2 paths:
        "C:\Program" and "Files"
        */
//        wxT(" ;"); // Don't separate with colon in DOS (used for drive)
        wxT(";"); // Don't separate with colon in DOS (used for drive)
#else
        wxT(" :;");
#endif

    wxString val ;
    if (wxGetEnv (WXSTRINGCAST envVariable, &val))
    {
        wxChar *s = MYcopystring (val);
        wxChar *save_ptr, *token = wxStrtok (s, PATH_TOKS, &save_ptr);

        if (token)
        {
            Add(token);
            while (token)
            {
                if ( (token = wxStrtok ((wxChar *) NULL, PATH_TOKS, &save_ptr))
                    != NULL )
                {
                    Add(token);
                }
            }
        }

        // suppress warning about unused variable save_ptr when wxStrtok() is a
        // macro which throws away its third argument
        save_ptr = token;

        delete [] s;
    }
#endif // !__WXWINCE__
}

// Given a full filename (with path), ensure that that file can
// be accessed again USING FILENAME ONLY by adding the path
// to the list if not already there.
void wxPathList::EnsureFileAccessible (const wxString& path)
{
    wxString path_only(wxPathOnly(path));
    if ( !path_only.empty() )
    {
        if ( !Member(path_only) )
            Add(path_only);
    }
}

bool wxPathList::Member (const wxString& path)
{
  for (wxStringList::compatibility_iterator node = GetFirst(); node; node = node->GetNext())
  {
      wxString path2( node->GetData() );
      if (
#if defined(__WINDOWS__) || defined(__OS2__) || defined(__VMS__) || defined(__WXMAC__)
      // Case INDEPENDENT
          path.CompareTo (path2, wxString::ignoreCase) == 0
#else
      // Case sensitive File System
          path.CompareTo (path2) == 0
#endif
        )
        return true;
  }
  return false;
}

wxString wxPathList::FindValidPath (const wxString& file)
{
  if (wxFileExists (wxExpandPath(wxFileFunctionsBuffer, file)))
    return wxString(wxFileFunctionsBuffer);

  wxChar buf[_MAXPATHLEN];
  wxStrcpy(buf, wxFileFunctionsBuffer);

  wxChar *filename = wxIsAbsolutePath (buf) ? wxFileNameFromPath (buf) : (wxChar *)buf;

  for (wxStringList::compatibility_iterator node = GetFirst(); node; node = node->GetNext())
    {
      const wxChar *path = node->GetData();
      wxStrcpy (wxFileFunctionsBuffer, path);
      wxChar ch = wxFileFunctionsBuffer[wxStrlen(wxFileFunctionsBuffer)-1];
      if (ch != wxT('\\') && ch != wxT('/'))
        wxStrcat (wxFileFunctionsBuffer, wxT("/"));
      wxStrcat (wxFileFunctionsBuffer, filename);
#ifdef __WINDOWS__
      wxUnix2DosFilename (wxFileFunctionsBuffer);
#endif
      if (wxFileExists (wxFileFunctionsBuffer))
      {
        return wxString(wxFileFunctionsBuffer);        // Found!
      }
    }                                // for()

  return wxEmptyString;                    // Not found
}

wxString wxPathList::FindAbsoluteValidPath (const wxString& file)
{
    wxString f = FindValidPath(file);
    if ( f.empty() || wxIsAbsolutePath(f) )
        return f;

    wxString buf;
    wxGetWorkingDirectory(wxStringBuffer(buf, _MAXPATHLEN), _MAXPATHLEN);

    if ( !wxEndsWithPathSeparator(buf) )
    {
        buf += wxFILE_SEP_PATH;
    }
    buf += f;

    return buf;
}

bool
wxFileExists (const wxString& filename)
{
   int result;
#if defined(__WXPALMOS__)
    return false;
#elif defined(__WIN32__) && !defined(__WXMICROWIN__)
    // we must use GetFileAttributes() instead of the ANSI C functions because
    // it can cope with network (UNC) paths unlike them
    DWORD ret = ::GetFileAttributes(filename);

    return (ret != (DWORD)-1) && !(ret & FILE_ATTRIBUTE_DIRECTORY);
#elif defined(__OS2__)
    FILESTATUS3 Info = {{0}};
    APIRET rc = ::DosQueryPathInfo((PSZ)(WXSTRINGCAST filename), FIL_STANDARD,
                                   (void*) &Info, sizeof(FILESTATUS3));
    return ((rc == NO_ERROR)) ||
      (rc == ERROR_SHARING_VIOLATION);
    // If we got a sharing violation, there must be something with this name.
#else // !__WIN32__
    wxStructStat st;
#ifndef wxNEED_WX_UNISTD_H
    result=wxStat( filename.fn_str() , &st) == 0 && (st.st_mode & S_IFREG);
#else
    result=wxStat( filename , &st) == 0 && (st.st_mode & S_IFREG);
#endif
    return result;
#endif // __WIN32__/!__WIN32__
}

bool
wxIsAbsolutePath (const wxString& filename)
{
    if (!filename.empty())
    {
#if defined(__WXMAC__) && !defined(__DARWIN__)
        // Classic or Carbon CodeWarrior like
        // Carbon with Apple DevTools is Unix like

        // This seems wrong to me, but there is no fix. since
        // "MacOS:MyText.txt" is absolute whereas "MyDir:MyText.txt"
        // is not. Or maybe ":MyDir:MyText.txt" has to be used? RR.
        if (filename.Find(':') != wxNOT_FOUND && filename[0] != ':')
            return true ;
#else
        // Unix like or Windows
        if (filename[0] == wxT('/'))
            return true;
#endif
#ifdef __VMS__
        if ((filename[0] == wxT('[') && filename[1] != wxT('.')))
            return true;
#endif
#if defined(__WINDOWS__) || defined(__OS2__)
        // MSDOS like
        if (filename[0] == wxT('\\') || (wxIsalpha (filename[0]) && filename[1] == wxT(':')))
            return true;
#endif
    }
    return false ;
}

/*
 * Strip off any extension (dot something) from end of file,
 * IF one exists. Inserts zero into buffer.
 *
 */

void wxStripExtension(wxChar *buffer)
{
    int len = wxStrlen(buffer);
    int i = len-1;
    while (i > 0)
    {
        if (buffer[i] == wxT('.'))
        {
            buffer[i] = 0;
            break;
        }
        i --;
    }
}

void wxStripExtension(wxString& buffer)
{
    //RN:  Be careful about the handling the case where
    //buffer.Length() == 0
    for(size_t i = buffer.Length() - 1; i != wxString::npos; --i)
    {
        if (buffer.GetChar(i) == wxT('.'))
        {
          buffer = buffer.Left(i);
          break;
        }
    }
}

// Destructive removal of /./ and /../ stuff
wxChar *wxRealPath (wxChar *path)
{
#ifdef __WXMSW__
  static const wxChar SEP = wxT('\\');
  wxUnix2DosFilename(path);
#else
  static const wxChar SEP = wxT('/');
#endif
  if (path[0] && path[1]) {
    /* MATTHEW: special case "/./x" */
    wxChar *p;
    if (path[2] == SEP && path[1] == wxT('.'))
      p = &path[0];
    else
      p = &path[2];
    for (; *p; p++)
      {
        if (*p == SEP)
          {
            if (p[1] == wxT('.') && p[2] == wxT('.') && (p[3] == SEP || p[3] == wxT('\0')))
              {
                wxChar *q;
                for (q = p - 1; q >= path && *q != SEP; q--)
                {
                    // Empty
                }

                if (q[0] == SEP && (q[1] != wxT('.') || q[2] != wxT('.') || q[3] != SEP)
                    && (q - 1 <= path || q[-1] != SEP))
                  {
                    wxStrcpy (q, p + 3);
                    if (path[0] == wxT('\0'))
                      {
                        path[0] = SEP;
                        path[1] = wxT('\0');
                      }
#if defined(__WXMSW__) || defined(__OS2__)
                    /* Check that path[2] is NULL! */
                    else if (path[1] == wxT(':') && !path[2])
                      {
                        path[2] = SEP;
                        path[3] = wxT('\0');
                      }
#endif
                    p = q - 1;
                  }
              }
            else if (p[1] == wxT('.') && (p[2] == SEP || p[2] == wxT('\0')))
              wxStrcpy (p, p + 2);
          }
      }
  }
  return path;
}

// Must be destroyed
wxChar *wxCopyAbsolutePath(const wxString& filename)
{
  if (filename.empty())
    return (wxChar *) NULL;

  if (! wxIsAbsolutePath(wxExpandPath(wxFileFunctionsBuffer, filename))) {
    wxChar  buf[_MAXPATHLEN];
    buf[0] = wxT('\0');
    wxGetWorkingDirectory(buf, WXSIZEOF(buf));
    wxChar ch = buf[wxStrlen(buf) - 1];
#ifdef __WXMSW__
    if (ch != wxT('\\') && ch != wxT('/'))
        wxStrcat(buf, wxT("\\"));
#else
    if (ch != wxT('/'))
        wxStrcat(buf, wxT("/"));
#endif
    wxStrcat(buf, wxFileFunctionsBuffer);
    return MYcopystring( wxRealPath(buf) );
  }
  return MYcopystring( wxFileFunctionsBuffer );
}

/*-
 Handles:
   ~/ => home dir
   ~user/ => user's home dir
   If the environment variable a = "foo" and b = "bar" then:
   Unix:
        $a        =>        foo
        $a$b        =>        foobar
        $a.c        =>        foo.c
        xxx$a        =>        xxxfoo
        ${a}!        =>        foo!
        $(b)!        =>        bar!
        \$a        =>        \$a
   MSDOS:
        $a        ==>        $a
        $(a)        ==>        foo
        $(a)$b        ==>        foo$b
        $(a)$(b)==>        foobar
        test.$$        ==>        test.$$
 */

/* input name in name, pathname output to buf. */

wxChar *wxExpandPath(wxChar *buf, const wxChar *name)
{
    register wxChar *d, *s, *nm;
    wxChar          lnm[_MAXPATHLEN];
    int             q;

    // Some compilers don't like this line.
//    const wxChar    trimchars[] = wxT("\n \t");

    wxChar      trimchars[4];
    trimchars[0] = wxT('\n');
    trimchars[1] = wxT(' ');
    trimchars[2] = wxT('\t');
    trimchars[3] = 0;

#ifdef __WXMSW__
    const wxChar     SEP = wxT('\\');
#else
    const wxChar     SEP = wxT('/');
#endif
    buf[0] = wxT('\0');
    if (name == NULL || *name == wxT('\0'))
        return buf;
    nm = MYcopystring(name); // Make a scratch copy
    wxChar *nm_tmp = nm;

    /* Skip leading whitespace and cr */
    while (wxStrchr((wxChar *)trimchars, *nm) != NULL)
        nm++;
    /* And strip off trailing whitespace and cr */
    s = nm + (q = wxStrlen(nm)) - 1;
    while (q-- && wxStrchr((wxChar *)trimchars, *s) != NULL)
        *s = wxT('\0');

    s = nm;
    d = lnm;
#ifdef __WXMSW__
    q = FALSE;
#else
    q = nm[0] == wxT('\\') && nm[1] == wxT('~');
#endif

    /* Expand inline environment variables */
#ifdef __VISAGECPP__
    while (*d)
    {
      *d++ = *s;
      if(*s == wxT('\\'))
      {
        *(d - 1) = *++s;
        if (*d)
        {
          s++;
          continue;
        }
        else
           break;
      }
      else
#else
    while ((*d++ = *s) != 0) {
#  ifndef __WXMSW__
        if (*s == wxT('\\')) {
            if ((*(d - 1) = *++s)!=0) {
                s++;
                continue;
            } else
                break;
        } else
#  endif
#endif
            // No env variables on WinCE
#ifndef __WXWINCE__
#ifdef __WXMSW__
        if (*s++ == wxT('$') && (*s == wxT('{') || *s == wxT(')')))
#else
        if (*s++ == wxT('$'))
#endif
        {
            register wxChar  *start = d;
            register int     braces = (*s == wxT('{') || *s == wxT('('));
            register wxChar  *value;
            while ((*d++ = *s) != 0)
                if (braces ? (*s == wxT('}') || *s == wxT(')')) : !(wxIsalnum(*s) || *s == wxT('_')) )
                    break;
                else
                    s++;
            *--d = 0;
            value = wxGetenv(braces ? start + 1 : start);
            if (value) {
                for ((d = start - 1); (*d++ = *value++) != 0;)
                {
                    // Empty
                }

                d--;
                if (braces && *s)
                    s++;
            }
        }
#endif
        // __WXWINCE__
    }

    /* Expand ~ and ~user */
    nm = lnm;
    if (nm[0] == wxT('~') && !q)
    {
        /* prefix ~ */
        if (nm[1] == SEP || nm[1] == 0)
        {        /* ~/filename */
        // FIXME: wxGetUserHome could return temporary storage in Unicode mode
            if ((s = WXSTRINGCAST wxGetUserHome(wxEmptyString)) != NULL) {
                if (*++nm)
                    nm++;
            }
        } else
        {                /* ~user/filename */
            register wxChar  *nnm;
            register wxChar  *home;
            for (s = nm; *s && *s != SEP; s++)
            {
                // Empty
            }
            int was_sep; /* MATTHEW: Was there a separator, or NULL? */
            was_sep = (*s == SEP);
            nnm = *s ? s + 1 : s;
            *s = 0;
        // FIXME: wxGetUserHome could return temporary storage in Unicode mode
            if ((home = WXSTRINGCAST wxGetUserHome(wxString(nm + 1))) == NULL)
            {
                if (was_sep) /* replace only if it was there: */
                    *s = SEP;
                s = NULL;
            }
            else
            {
                nm = nnm;
                s = home;
            }
        }
    }

    d = buf;
    if (s && *s) { /* MATTHEW: s could be NULL if user '~' didn't exist */
        /* Copy home dir */
        while (wxT('\0') != (*d++ = *s++))
          /* loop */;
        // Handle root home
        if (d - 1 > buf && *(d - 2) != SEP)
          *(d - 1) = SEP;
    }
    s = nm;
    while ((*d++ = *s++) != 0)
    {
        // Empty
    }
    delete[] nm_tmp; // clean up alloc
    /* Now clean up the buffer */
    return wxRealPath(buf);
}

/* Contract Paths to be build upon an environment variable
   component:

   example: "/usr/openwin/lib", OPENWINHOME --> ${OPENWINHOME}/lib

   The call wxExpandPath can convert these back!
 */
wxChar *
wxContractPath (const wxString& filename,
                const wxString& WXUNUSED_IN_WINCE(envname),
                const wxString& user)
{
  static wxChar dest[_MAXPATHLEN];

  if (filename.empty())
    return (wxChar *) NULL;

  wxStrcpy (dest, WXSTRINGCAST filename);
#ifdef __WXMSW__
  wxUnix2DosFilename(dest);
#endif

  // Handle environment
  const wxChar *val;
#ifndef __WXWINCE__
  wxChar *tcp;
  if (!envname.empty() && (val = wxGetenv (WXSTRINGCAST envname)) != NULL &&
     (tcp = wxStrstr (dest, val)) != NULL)
    {
        wxStrcpy (wxFileFunctionsBuffer, tcp + wxStrlen (val));
        *tcp++ = wxT('$');
        *tcp++ = wxT('{');
        wxStrcpy (tcp, WXSTRINGCAST envname);
        wxStrcat (tcp, wxT("}"));
        wxStrcat (tcp, wxFileFunctionsBuffer);
    }
#endif

  // Handle User's home (ignore root homes!)
  val = wxGetUserHome (user);
  if (!val)
    return dest;

  const size_t len = wxStrlen(val);
  if (len <= 2)
    return dest;

  if (wxStrncmp(dest, val, len) == 0)
  {
    wxStrcpy(wxFileFunctionsBuffer, wxT("~"));
    if (!user.empty())
           wxStrcat(wxFileFunctionsBuffer, (const wxChar*) user);
    wxStrcat(wxFileFunctionsBuffer, dest + len);
    wxStrcpy (dest, wxFileFunctionsBuffer);
  }

  return dest;
}

// Return just the filename, not the path (basename)
wxChar *wxFileNameFromPath (wxChar *path)
{
    wxString p = path;
    wxString n = wxFileNameFromPath(p);

    return path + p.length() - n.length();
}

wxString wxFileNameFromPath (const wxString& path)
{
    wxString name, ext;
    wxFileName::SplitPath(path, NULL, &name, &ext);

    wxString fullname = name;
    if ( !ext.empty() )
    {
        fullname << wxFILE_SEP_EXT << ext;
    }

    return fullname;
}

// Return just the directory, or NULL if no directory
wxChar *
wxPathOnly (wxChar *path)
{
    if (path && *path)
    {
        static wxChar buf[_MAXPATHLEN];

        // Local copy
        wxStrcpy (buf, path);

        int l = wxStrlen(path);
        int i = l - 1;

        // Search backward for a backward or forward slash
        while (i > -1)
        {
#if defined(__WXMAC__) && !defined(__DARWIN__)
            // Classic or Carbon CodeWarrior like
            // Carbon with Apple DevTools is Unix like
            if (path[i] == wxT(':') )
            {
                buf[i] = 0;
                return buf;
            }
#else
            // Unix like or Windows
            if (path[i] == wxT('/') || path[i] == wxT('\\'))
            {
                buf[i] = 0;
                return buf;
            }
#endif
#ifdef __VMS__
            if (path[i] == wxT(']'))
            {
                buf[i+1] = 0;
                return buf;
            }
#endif
            i --;
        }

#if defined(__WXMSW__) || defined(__OS2__)
        // Try Drive specifier
        if (wxIsalpha (buf[0]) && buf[1] == wxT(':'))
        {
            // A:junk --> A:. (since A:.\junk Not A:\junk)
            buf[2] = wxT('.');
            buf[3] = wxT('\0');
            return buf;
        }
#endif
    }
    return (wxChar *) NULL;
}

// Return just the directory, or NULL if no directory
wxString wxPathOnly (const wxString& path)
{
    if (!path.empty())
    {
        wxChar buf[_MAXPATHLEN];

        // Local copy
        wxStrcpy (buf, WXSTRINGCAST path);

        int l = path.Length();
        int i = l - 1;

        // Search backward for a backward or forward slash
        while (i > -1)
        {
#if defined(__WXMAC__) && !defined(__DARWIN__)
            // Classic or Carbon CodeWarrior like
            // Carbon with Apple DevTools is Unix like
            if (path[i] == wxT(':') )
            {
                buf[i] = 0;
                return wxString(buf);
            }
#else
            // Unix like or Windows
            if (path[i] == wxT('/') || path[i] == wxT('\\'))
            {
                // Don't return an empty string
                if (i == 0)
                    i ++;
                buf[i] = 0;
                return wxString(buf);
            }
#endif
#ifdef __VMS__
            if (path[i] == wxT(']'))
            {
                buf[i+1] = 0;
                return wxString(buf);
            }
#endif
            i --;
        }

#if defined(__WXMSW__) || defined(__OS2__)
        // Try Drive specifier
        if (wxIsalpha (buf[0]) && buf[1] == wxT(':'))
        {
            // A:junk --> A:. (since A:.\junk Not A:\junk)
            buf[2] = wxT('.');
            buf[3] = wxT('\0');
            return wxString(buf);
        }
#endif
    }
    return wxEmptyString;
}

// Utility for converting delimiters in DOS filenames to UNIX style
// and back again - or we get nasty problems with delimiters.
// Also, convert to lower case, since case is significant in UNIX.

#if defined(__WXMAC__)

#if TARGET_API_MAC_OSX
#define kDefaultPathStyle kCFURLPOSIXPathStyle
#else
#define kDefaultPathStyle kCFURLHFSPathStyle
#endif

wxString wxMacFSRefToPath( const FSRef *fsRef , CFStringRef additionalPathComponent )
{
    CFURLRef fullURLRef;
    fullURLRef = CFURLCreateFromFSRef(NULL, fsRef);
    if ( additionalPathComponent )
    {
        CFURLRef parentURLRef = fullURLRef ;
        fullURLRef = CFURLCreateCopyAppendingPathComponent(NULL, parentURLRef,
            additionalPathComponent,false);
        CFRelease( parentURLRef ) ;
    }
    CFStringRef cfString = CFURLCopyFileSystemPath(fullURLRef, kDefaultPathStyle);
    CFRelease( fullURLRef ) ;
    return wxMacCFStringHolder(cfString).AsString(wxLocale::GetSystemEncoding());
}

OSStatus wxMacPathToFSRef( const wxString&path , FSRef *fsRef )
{
    OSStatus err = noErr ;
    CFURLRef url = CFURLCreateWithFileSystemPath(kCFAllocatorDefault, wxMacCFStringHolder(path ,wxLocale::GetSystemEncoding() ) , kDefaultPathStyle, false);
    if ( NULL != url )
    {
        if ( CFURLGetFSRef(url, fsRef) == false )
            err = fnfErr ;
        CFRelease( url ) ;
    }
    else
    {
        err = fnfErr ;
    }
    return err ;
}

wxString wxMacHFSUniStrToString( ConstHFSUniStr255Param uniname )
{
    CFStringRef cfname = CFStringCreateWithCharacters( kCFAllocatorDefault,
                                                      uniname->unicode,
                                                      uniname->length );
    return wxMacCFStringHolder(cfname).AsString() ;
}

wxString wxMacFSSpec2MacFilename( const FSSpec *spec )
{
    FSRef fsRef ;
    if ( FSpMakeFSRef( spec , &fsRef) == noErr )
    {
        return wxMacFSRefToPath( &fsRef ) ;
    }
    return wxEmptyString ;
}

void wxMacFilename2FSSpec( const wxString& path , FSSpec *spec )
{
    OSStatus err = noErr ;
    FSRef fsRef ;
    wxMacPathToFSRef( path , &fsRef ) ;
    err = FSRefMakeFSSpec( &fsRef , spec ) ;
}

#endif // __WXMAC__

void
wxDos2UnixFilename (wxChar *s)
{
  if (s)
    while (*s)
      {
        if (*s == _T('\\'))
          *s = _T('/');
#ifdef __WXMSW__
        else
          *s = (wxChar)wxTolower (*s);        // Case INDEPENDENT
#endif
        s++;
      }
}

void
#if defined(__WXMSW__) || defined(__OS2__)
wxUnix2DosFilename (wxChar *s)
#else
wxUnix2DosFilename (wxChar *WXUNUSED(s) )
#endif
{
// Yes, I really mean this to happen under DOS only! JACS
#if defined(__WXMSW__) || defined(__OS2__)
  if (s)
    while (*s)
      {
        if (*s == wxT('/'))
          *s = wxT('\\');
        s++;
      }
#endif
}

// Concatenate two files to form third
bool
wxConcatFiles (const wxString& file1, const wxString& file2, const wxString& file3)
{
#if wxUSE_FILE

    wxFile in1(file1), in2(file2);
    wxTempFile out(file3);

    if ( !in1.IsOpened() || !in2.IsOpened() || !out.IsOpened() )
        return false;

    ssize_t ofs;
    unsigned char buf[1024];

    for( int i=0; i<2; i++)
    {
        wxFile *in = i==0 ? &in1 : &in2;
        do{
            if ( (ofs = in->Read(buf,WXSIZEOF(buf))) == wxInvalidOffset ) return false;
            if ( ofs > 0 )
                if ( !out.Write(buf,ofs) )
                    return false;
        } while ( ofs == (ssize_t)WXSIZEOF(buf) );
    }

    return out.Commit();

#else

    wxUnusedVar(file1);
    wxUnusedVar(file2);
    wxUnusedVar(file3);
    return false;

#endif
}

// Copy files
bool
wxCopyFile (const wxString& file1, const wxString& file2, bool overwrite)
{
#if defined(__WIN32__) && !defined(__WXMICROWIN__)
    // CopyFile() copies file attributes and modification time too, so use it
    // instead of our code if available
    //
    // NB: 3rd parameter is bFailIfExists i.e. the inverse of overwrite
    if ( !::CopyFile(file1, file2, !overwrite) )
    {
        wxLogSysError(_("Failed to copy the file '%s' to '%s'"),
                      file1.c_str(), file2.c_str());

        return false;
    }
#elif defined(__OS2__)
    if ( ::DosCopy((PSZ)file1.c_str(), (PSZ)file2.c_str(), overwrite ? DCPY_EXISTING : 0) != 0 )
        return false;
#elif defined(__PALMOS__)
    // TODO with http://www.palmos.com/dev/support/docs/protein_books/Memory_Databases_Files/
    return false;
#elif wxUSE_FILE // !Win32

    wxStructStat fbuf;
    // get permissions of file1
    if ( wxStat( file1.c_str(), &fbuf) != 0 )
    {
        // the file probably doesn't exist or we haven't the rights to read
        // from it anyhow
        wxLogSysError(_("Impossible to get permissions for file '%s'"),
                      file1.c_str());
        return false;
    }

    // open file1 for reading
    wxFile fileIn(file1, wxFile::read);
    if ( !fileIn.IsOpened() )
        return false;

    // remove file2, if it exists. This is needed for creating
    // file2 with the correct permissions in the next step
    if ( wxFileExists(file2)  && (!overwrite || !wxRemoveFile(file2)))
    {
        wxLogSysError(_("Impossible to overwrite the file '%s'"),
                      file2.c_str());
        return false;
    }

    // reset the umask as we want to create the file with exactly the same
    // permissions as the original one
    wxCHANGE_UMASK(0);

    // create file2 with the same permissions than file1 and open it for
    // writing

    wxFile fileOut;
    if ( !fileOut.Create(file2, overwrite, fbuf.st_mode & 0777) )
        return false;

    // copy contents of file1 to file2
    char buf[4096];
    size_t count;
    for ( ;; )
    {
        count = fileIn.Read(buf, WXSIZEOF(buf));
        if ( fileIn.Error() )
            return false;

        // end of file?
        if ( !count )
            break;

        if ( fileOut.Write(buf, count) < count )
            return false;
    }

    // we can expect fileIn to be closed successfully, but we should ensure
    // that fileOut was closed as some write errors (disk full) might not be
    // detected before doing this
    if ( !fileIn.Close() || !fileOut.Close() )
        return false;

#if !defined(__VISAGECPP__) && !defined(__WXMAC__) || defined(__UNIX__)
    // no chmod in VA.  Should be some permission API for HPFS386 partitions
    // however
    if ( chmod(OS_FILENAME(file2), fbuf.st_mode) != 0 )
    {
        wxLogSysError(_("Impossible to set permissions for the file '%s'"),
                      file2.c_str());
        return false;
    }
#endif // OS/2 || Mac

#else // !Win32 && ! wxUSE_FILE

    // impossible to simulate with wxWidgets API
    wxUnusedVar(file1);
    wxUnusedVar(file2);
    wxUnusedVar(overwrite);
    return false;

#endif // __WXMSW__ && __WIN32__

    return true;
}

bool
wxRenameFile (const wxString& file1, const wxString& file2)
{
#if !defined(__WXWINCE__) && !defined(__WXPALMOS__)
    // Normal system call
  if ( wxRename (file1, file2) == 0 )
    return true;
#endif

  // Try to copy
  if (wxCopyFile(file1, file2)) {
    wxRemoveFile(file1);
    return true;
  }
  // Give up
  return false;
}

bool wxRemoveFile(const wxString& file)
{
#if defined(__VISUALC__) \
 || defined(__BORLANDC__) \
 || defined(__WATCOMC__) \
 || defined(__DMC__) \
 || defined(__GNUWIN32__) \
 || (defined(__MWERKS__) && defined(__MSL__))
    int res = wxRemove(file);
#elif defined(__WXMAC__)
    int res = unlink(wxFNCONV(file));
#elif defined(__WXPALMOS__)
    int res = 1;
    // TODO with VFSFileDelete()
#else
    int res = unlink(OS_FILENAME(file));
#endif

    return res == 0;
}

bool wxMkdir(const wxString& dir, int perm)
{
#if defined(__WXPALMOS__)
    return false;
#elif defined(__WXMAC__) && !defined(__UNIX__)
    return (mkdir( wxFNCONV(dir) , 0 ) == 0);
#else // !Mac
    const wxChar *dirname = dir.c_str();

    // assume mkdir() has 2 args on non Windows-OS/2 platforms and on Windows too
    // for the GNU compiler
#if (!(defined(__WXMSW__) || defined(__OS2__) || defined(__DOS__))) || (defined(__GNUWIN32__) && !defined(__MINGW32__)) || defined(__WINE__) || defined(__WXMICROWIN__)
  #if defined(MSVCRT)
    wxUnusedVar(perm);
    if ( mkdir(wxFNCONV(dirname)) != 0 )
  #else
    if ( mkdir(wxFNCONV(dirname), perm) != 0 )
  #endif
#elif defined(__OS2__)
    wxUnusedVar(perm);
    if (::DosCreateDir((PSZ)dirname, NULL) != 0) // enhance for EAB's??
#elif defined(__DOS__)
  #if defined(__WATCOMC__)
    (void)perm;
    if ( wxMkDir(wxFNSTRINGCAST wxFNCONV(dirname)) != 0 )
  #elif defined(__DJGPP__)
    if ( mkdir(wxFNCONV(dirname), perm) != 0 )
  #else
    #error "Unsupported DOS compiler!"
  #endif
#else  // !MSW, !DOS and !OS/2 VAC++
    wxUnusedVar(perm);
#ifdef __WXWINCE__
    if ( !CreateDirectory(dirname, NULL) )
#else
    if ( wxMkDir(dir.fn_str()) != 0 )
#endif
#endif // !MSW/MSW
    {
        wxLogSysError(_("Directory '%s' couldn't be created"), dirname);

        return false;
    }

    return true;
#endif // Mac/!Mac
}

bool wxRmdir(const wxString& dir, int WXUNUSED(flags))
{
#if defined(__VMS__)
    return false; //to be changed since rmdir exists in VMS7.x
#elif defined(__OS2__)
    return (::DosDeleteDir((PSZ)dir.c_str()) == 0);
#elif defined(__WXWINCE__)
    return (CreateDirectory(dir, NULL) != 0);
#elif defined(__WXPALMOS__)
    // TODO with VFSFileRename()
    return false;
#else
    return (wxRmDir(OS_FILENAME(dir)) == 0);
#endif
}

// does the path exists? (may have or not '/' or '\\' at the end)
bool wxDirExists(const wxChar *pszPathName)
{
    wxString strPath(pszPathName);

#if defined(__WINDOWS__) || defined(__OS2__)
    // Windows fails to find directory named "c:\dir\" even if "c:\dir" exists,
    // so remove all trailing backslashes from the path - but don't do this for
    // the pathes "d:\" (which are different from "d:") nor for just "\"
    while ( wxEndsWithPathSeparator(strPath) )
    {
        size_t len = strPath.length();
        if ( len == 1 || (len == 3 && strPath[len - 2] == _T(':')) )
            break;

        strPath.Truncate(len - 1);
    }
#endif // __WINDOWS__

#ifdef __OS2__
    // OS/2 can't handle "d:", it wants either "d:\" or "d:."
    if (strPath.length() == 2 && strPath[1u] == _T(':'))
        strPath << _T('.');
#endif

#if defined(__WXPALMOS__)
    return false;
#elif defined(__WIN32__) && !defined(__WXMICROWIN__)
    // stat() can't cope with network paths
    DWORD ret = ::GetFileAttributes(strPath);

    return (ret != (DWORD)-1) && (ret & FILE_ATTRIBUTE_DIRECTORY);
#elif defined(__OS2__)
    return !((bool)(::DosSetCurrentDir((PSZ)(WXSTRINGCAST strPath))));
#else // !__WIN32__

    wxStructStat st;
#ifndef __VISAGECPP__
    return wxStat(strPath.c_str(), &st) == 0 && ((st.st_mode & S_IFMT) == S_IFDIR);
#else
    // S_IFMT not supported in VA compilers.. st_mode is a 2byte value only
    return wxStat(pszPathName, &st) == 0 && (st.st_mode == S_IFDIR);
#endif

#endif // __WIN32__/!__WIN32__
}

// Get a temporary filename, opening and closing the file.
wxChar *wxGetTempFileName(const wxString& prefix, wxChar *buf)
{
#if wxUSE_FILE
    wxString filename = wxFileName::CreateTempFileName(prefix);
    if ( filename.empty() )
        return NULL;

    if ( buf )
        wxStrcpy(buf, filename);
    else
        buf = MYcopystring(filename);

    return buf;
#else
    wxUnusedVar(prefix);
    wxUnusedVar(buf);
    // wxFileName::CreateTempFileName needs wxFile class enabled
    return NULL;
#endif
}

bool wxGetTempFileName(const wxString& prefix, wxString& buf)
{
    buf = wxGetTempFileName(prefix);

    return !buf.empty();
}

// Get first file name matching given wild card.

static wxDir *gs_dir = NULL;
static wxString gs_dirPath;

wxString wxFindFirstFile(const wxChar *spec, int flags)
{
    wxSplitPath(spec, &gs_dirPath, NULL, NULL);
    if ( gs_dirPath.empty() )
        gs_dirPath = wxT(".");
    if ( !wxEndsWithPathSeparator(gs_dirPath ) )
        gs_dirPath << wxFILE_SEP_PATH;

    if (gs_dir)
        delete gs_dir;
    gs_dir = new wxDir(gs_dirPath);

    if ( !gs_dir->IsOpened() )
    {
        wxLogSysError(_("Can not enumerate files '%s'"), spec);
        return wxEmptyString;
    }

    int dirFlags;
    switch (flags)
    {
        case wxDIR:  dirFlags = wxDIR_DIRS; break;
        case wxFILE: dirFlags = wxDIR_FILES; break;
        default:     dirFlags = wxDIR_DIRS | wxDIR_FILES; break;
    }

    wxString result;
    gs_dir->GetFirst(&result, wxFileNameFromPath(wxString(spec)), dirFlags);
    if ( result.empty() )
    {
        wxDELETE(gs_dir);
        return result;
    }

    return gs_dirPath + result;
}

wxString wxFindNextFile()
{
    wxASSERT_MSG( gs_dir, wxT("You must call wxFindFirstFile before!") );

    wxString result;
    gs_dir->GetNext(&result);

    if ( result.empty() )
    {
        wxDELETE(gs_dir);
        return result;
    }

    return gs_dirPath + result;
}


// Get current working directory.
// If buf is NULL, allocates space using new, else
// copies into buf.
wxChar *wxGetWorkingDirectory(wxChar *buf, int sz)
{
#if defined(__WXPALMOS__)
    // TODO ?
    return NULL;
#elif defined(__WXWINCE__)
    // TODO
    wxUnusedVar(buf);
    wxUnusedVar(sz);
    return NULL;
#else
    if ( !buf )
    {
        buf = new wxChar[sz + 1];
    }

    bool ok wxDUMMY_INITIALIZE(false);

    // for the compilers which have Unicode version of _getcwd(), call it
    // directly, for the others call the ANSI version and do the translation
#if !wxUSE_UNICODE
    #define cbuf buf
#else // wxUSE_UNICODE
    bool needsANSI = true;

    #if !defined(HAVE_WGETCWD) || wxUSE_UNICODE_MSLU
        char cbuf[_MAXPATHLEN];
    #endif

    #ifdef HAVE_WGETCWD
        #if wxUSE_UNICODE_MSLU
            if ( wxGetOsVersion() != wxWIN95 )
        #else
            char *cbuf = NULL; // never really used because needsANSI will always be false
        #endif
            {
                ok = _wgetcwd(buf, sz) != NULL;
                needsANSI = false;
            }
    #endif

    if ( needsANSI )
#endif // wxUSE_UNICODE
    {
    #if defined(_MSC_VER) || defined(__MINGW32__)
        ok = _getcwd(cbuf, sz) != NULL;
    #elif defined(__WXMAC__) && !defined(__DARWIN__)
        char lbuf[1024] ;
        if ( getcwd( lbuf , sizeof( lbuf ) ) )
        {
            wxString res( lbuf , *wxConvCurrent ) ;
            wxStrcpy( buf , res ) ;
            ok = true;
        }
        else
            ok = false ;
    #elif defined(__OS2__)
        APIRET rc;
        ULONG ulDriveNum = 0;
        ULONG ulDriveMap = 0;
        rc = ::DosQueryCurrentDisk(&ulDriveNum, &ulDriveMap);
        ok = rc == 0;
        if (ok)
        {
            sz -= 3;
            rc = ::DosQueryCurrentDir( 0 // current drive
                                      ,cbuf + 3
                                      ,(PULONG)&sz
                                     );
            cbuf[0] = char('A' + (ulDriveNum - 1));
            cbuf[1] = ':';
            cbuf[2] = '\\';
            ok = rc == 0;
        }
    #else // !Win32/VC++ !Mac !OS2
        ok = getcwd(cbuf, sz) != NULL;
    #endif // platform

    #if wxUSE_UNICODE && !(defined(__WXMAC__) && !defined(__DARWIN__))
        // finally convert the result to Unicode if needed
        wxConvFile.MB2WC(buf, cbuf, sz);
    #endif // wxUSE_UNICODE
    }

    if ( !ok )
    {
        wxLogSysError(_("Failed to get the working directory"));

        // VZ: the old code used to return "." on error which didn't make any
        //     sense at all to me - empty string is a better error indicator
        //     (NULL might be even better but I'm afraid this could lead to
        //     problems with the old code assuming the return is never NULL)
        buf[0] = _T('\0');
    }
    else // ok, but we might need to massage the path into the right format
    {
#ifdef __DJGPP__
        // VS: DJGPP is a strange mix of DOS and UNIX API and returns paths
        //     with / deliminers. We don't like that.
        for (wxChar *ch = buf; *ch; ch++)
        {
            if (*ch == wxT('/'))
                *ch = wxT('\\');
        }
#endif // __DJGPP__

// MBN: we hope that in the case the user is compiling a GTK+/Motif app,
//      he needs Unix as opposed to Win32 pathnames
#if defined( __CYGWIN__ ) && defined( __WINDOWS__ )
        // another example of DOS/Unix mix (Cygwin)
        wxString pathUnix = buf;
#if wxUSE_UNICODE
        char bufA[_MAXPATHLEN];
        cygwin_conv_to_full_win32_path(pathUnix.mb_str(wxConvFile), bufA);
        wxConvFile.MB2WC(buf, bufA, sz);
#else
        cygwin_conv_to_full_win32_path(pathUnix, buf);
#endif // wxUSE_UNICODE
#endif // __CYGWIN__
    }

    return buf;

#if !wxUSE_UNICODE
    #undef cbuf
#endif

#endif
    // __WXWINCE__
}

wxString wxGetCwd()
{
    wxChar *buffer = new wxChar[_MAXPATHLEN];
    wxGetWorkingDirectory(buffer, _MAXPATHLEN);
    wxString str( buffer );
    delete [] buffer;

    return str;
}

bool wxSetWorkingDirectory(const wxString& d)
{
#if defined(__OS2__)
    return (::DosSetCurrentDir((PSZ)d.c_str()) == 0);
#elif defined(__UNIX__) || defined(__WXMAC__) || defined(__DOS__)
    return (chdir(wxFNSTRINGCAST d.fn_str()) == 0);
#elif defined(__WINDOWS__)

#ifdef __WIN32__
#ifdef __WXWINCE__
    // No equivalent in WinCE
    wxUnusedVar(d);
    return false;
#else
    return (bool)(SetCurrentDirectory(d) != 0);
#endif
#else
    // Must change drive, too.
    bool isDriveSpec = ((strlen(d) > 1) && (d[1] == ':'));
    if (isDriveSpec)
    {
        wxChar firstChar = d[0];

        // To upper case
        if (firstChar > 90)
            firstChar = firstChar - 32;

        // To a drive number
        unsigned int driveNo = firstChar - 64;
        if (driveNo > 0)
        {
            unsigned int noDrives;
            _dos_setdrive(driveNo, &noDrives);
        }
    }
    bool success = (chdir(WXSTRINGCAST d) == 0);

    return success;
#endif

#endif
}

// Get the OS directory if appropriate (such as the Windows directory).
// On non-Windows platform, probably just return the empty string.
wxString wxGetOSDirectory()
{
#ifdef __WXWINCE__
    return wxString(wxT("\\Windows"));
#elif defined(__WINDOWS__) && !defined(__WXMICROWIN__)
    wxChar buf[256];
    GetWindowsDirectory(buf, 256);
    return wxString(buf);
#elif defined(__WXMAC__)
    return wxMacFindFolder(kOnSystemDisk, 'macs', false);
#else
    return wxEmptyString;
#endif
}

bool wxEndsWithPathSeparator(const wxChar *pszFileName)
{
    size_t len = wxStrlen(pszFileName);

    return len && wxIsPathSeparator(pszFileName[len - 1]);
}

// find a file in a list of directories, returns false if not found
bool wxFindFileInPath(wxString *pStr, const wxChar *pszPath, const wxChar *pszFile)
{
    // we assume that it's not empty
    wxCHECK_MSG( !wxIsEmpty(pszFile), false,
                 _T("empty file name in wxFindFileInPath"));

    // skip path separator in the beginning of the file name if present
    if ( wxIsPathSeparator(*pszFile) )
        pszFile++;

    // copy the path (strtok will modify it)
    wxChar *szPath = new wxChar[wxStrlen(pszPath) + 1];
    wxStrcpy(szPath, pszPath);

    wxString strFile;
    wxChar *pc, *save_ptr;
    for ( pc = wxStrtok(szPath, wxPATH_SEP, &save_ptr);
          pc != NULL;
          pc = wxStrtok((wxChar *) NULL, wxPATH_SEP, &save_ptr) )
    {
        // search for the file in this directory
        strFile = pc;
        if ( !wxEndsWithPathSeparator(pc) )
            strFile += wxFILE_SEP_PATH;
        strFile += pszFile;

        if ( wxFileExists(strFile) ) {
            *pStr = strFile;
            break;
        }
    }

    // suppress warning about unused variable save_ptr when wxStrtok() is a
    // macro which throws away its third argument
    save_ptr = pc;

    delete [] szPath;

    return pc != NULL;  // if true => we breaked from the loop
}

void WXDLLEXPORT wxSplitPath(const wxChar *pszFileName,
                             wxString *pstrPath,
                             wxString *pstrName,
                             wxString *pstrExt)
{
    // it can be empty, but it shouldn't be NULL
    wxCHECK_RET( pszFileName, wxT("NULL file name in wxSplitPath") );

    wxFileName::SplitPath(pszFileName, pstrPath, pstrName, pstrExt);
}

time_t WXDLLEXPORT wxFileModificationTime(const wxString& filename)
{
#if defined(__WXPALMOS__)
    return 0;
#elif defined(__WXWINCE__)
    FILETIME ftLastWrite;
    AutoHANDLE hFile(::CreateFile(filename, GENERIC_READ, FILE_SHARE_READ,
                                    NULL, 0, FILE_ATTRIBUTE_NORMAL, 0));

    if ( !hFile.IsOk() )
        return 0;

    if ( !::GetFileTime(hFile, NULL, NULL, &ftLastWrite) )
        return 0;

    // sure we want to translate to local time here?
    FILETIME ftLocal;
    if ( !::FileTimeToLocalFileTime(&ftLastWrite, &ftLocal) )
    {
        wxLogLastError(_T("FileTimeToLocalFileTime"));
    }

    // FILETIME is a counted in 100-ns since 1601-01-01, convert it to
    // number of seconds since 1970-01-01
    ULARGE_INTEGER uli;
    uli.LowPart = ftLocal.dwLowDateTime;
    uli.HighPart = ftLocal.dwHighDateTime;

    ULONGLONG ull = uli.QuadPart;
    ull /= wxULL(10000000);     // number of 100ns intervals in 1s
    ull -= wxULL(11644473600);  // 1970-01-01 - 1601-01-01 in seconds

    return wx_static_cast(time_t, ull);
#else
    wxStructStat buf;
    if ( wxStat( filename, &buf) != 0 )
        return 0;

    return buf.st_mtime;
#endif
}


// Parses the filterStr, returning the number of filters.
// Returns 0 if none or if there's a problem.
// filterStr is in the form: "All files (*.*)|*.*|JPEG Files (*.jpeg)|*.jpeg"

int WXDLLEXPORT wxParseCommonDialogsFilter(const wxString& filterStr,
                                           wxArrayString& descriptions,
                                           wxArrayString& filters)
{
    descriptions.Clear();
    filters.Clear();

    wxString str(filterStr);

    wxString description, filter;
    int pos = 0;
    while( pos != wxNOT_FOUND )
    {
        pos = str.Find(wxT('|'));
        if ( pos == wxNOT_FOUND )
        {
            // if there are no '|'s at all in the string just take the entire
            // string as filter and make description empty for later autocompletion
            if ( filters.IsEmpty() )
            {
                descriptions.Add(wxEmptyString);
                filters.Add(filterStr);
            }
            else
            {
                wxFAIL_MSG( _T("missing '|' in the wildcard string!") );
            }

            break;
        }

        description = str.Left(pos);
        str = str.Mid(pos + 1);
        pos = str.Find(wxT('|'));
        if ( pos == wxNOT_FOUND )
        {
            filter = str;
        }
        else
        {
            filter = str.Left(pos);
            str = str.Mid(pos + 1);
        }

        descriptions.Add(description);
        filters.Add(filter);
    }

#if defined(__WXMOTIF__)
    // split it so there is one wildcard per entry
    for( size_t i = 0 ; i < descriptions.GetCount() ; i++ )
    {
        pos = filters[i].Find(wxT(';'));
        if (pos != wxNOT_FOUND)
        {
            // first split only filters
            descriptions.Insert(descriptions[i],i+1);
            filters.Insert(filters[i].Mid(pos+1),i+1);
            filters[i]=filters[i].Left(pos);

            // autoreplace new filter in description with pattern:
            //     C/C++ Files(*.cpp;*.c;*.h)|*.cpp;*.c;*.h
            // cause split into:
            //     C/C++ Files(*.cpp)|*.cpp
            //     C/C++ Files(*.c;*.h)|*.c;*.h
            // and next iteration cause another split into:
            //     C/C++ Files(*.cpp)|*.cpp
            //     C/C++ Files(*.c)|*.c
            //     C/C++ Files(*.h)|*.h
            for ( size_t k=i;k<i+2;k++ )
            {
                pos = descriptions[k].Find(filters[k]);
                if (pos != wxNOT_FOUND)
                {
                    wxString before = descriptions[k].Left(pos);
                    wxString after = descriptions[k].Mid(pos+filters[k].Len());
                    pos = before.Find(_T('('),true);
                    if (pos>before.Find(_T(')'),true))
                    {
                        before = before.Left(pos+1);
                        before << filters[k];
                        pos = after.Find(_T(')'));
                        int pos1 = after.Find(_T('('));
                        if (pos != wxNOT_FOUND && (pos<pos1 || pos1==wxNOT_FOUND))
                        {
                            before << after.Mid(pos);
                            descriptions[k] = before;
                        }
                    }
                }
            }
        }
    }
#endif

    // autocompletion
    for( size_t j = 0 ; j < descriptions.GetCount() ; j++ )
    {
        if ( descriptions[j].empty() && !filters[j].empty() )
        {
            descriptions[j].Printf(_("Files (%s)"), filters[j].c_str());
        }
    }

    return filters.GetCount();
}


//------------------------------------------------------------------------
// wild character routines
//------------------------------------------------------------------------

bool wxIsWild( const wxString& pattern )
{
    wxString tmp = pattern;
    wxChar *pat = WXSTRINGCAST(tmp);
    while (*pat)
    {
        switch (*pat++)
        {
        case wxT('?'): case wxT('*'): case wxT('['): case wxT('{'):
            return true;
        case wxT('\\'):
            if (!*pat++)
                return false;
        }
    }
    return false;
}

/*
* Written By Douglas A. Lewis <dalewis@cs.Buffalo.EDU>
*
* The match procedure is public domain code (from ircII's reg.c)
*/

bool wxMatchWild( const wxString& pat, const wxString& text, bool dot_special )
{
    if (text.empty())
    {
        /* Match if both are empty. */
        return pat.empty();
    }

    const wxChar *m = pat.c_str(),
    *n = text.c_str(),
    *ma = NULL,
    *na = NULL,
    *mp = NULL,
    *np = NULL;
    int just = 0,
    pcount = 0,
    acount = 0,
    count = 0;

    if (dot_special && (*n == wxT('.')))
    {
        /* Never match so that hidden Unix files
         * are never found. */
        return false;
    }

    for (;;)
    {
        if (*m == wxT('*'))
        {
            ma = ++m;
            na = n;
            just = 1;
            mp = NULL;
            acount = count;
        }
        else if (*m == wxT('?'))
        {
            m++;
            if (!*n++)
                return false;
        }
        else
        {
            if (*m == wxT('\\'))
            {
                m++;
                /* Quoting "nothing" is a bad thing */
                if (!*m)
                    return false;
            }
            if (!*m)
            {
                /*
                * If we are out of both strings or we just
                * saw a wildcard, then we can say we have a
                * match
                */
                if (!*n)
                    return true;
                if (just)
                    return true;
                just = 0;
                goto not_matched;
            }
            /*
            * We could check for *n == NULL at this point, but
            * since it's more common to have a character there,
            * check to see if they match first (m and n) and
            * then if they don't match, THEN we can check for
            * the NULL of n
            */
            just = 0;
            if (*m == *n)
            {
                m++;
                if (*n == wxT(' '))
                    mp = NULL;
                count++;
                n++;
            }
            else
            {

                not_matched:

                /*
                * If there are no more characters in the
                * string, but we still need to find another
                * character (*m != NULL), then it will be
                * impossible to match it
                */
                if (!*n)
                    return false;
                if (mp)
                {
                    m = mp;
                    if (*np == wxT(' '))
                    {
                        mp = NULL;
                        goto check_percent;
                    }
                    n = ++np;
                    count = pcount;
                }
                else
                check_percent:

                if (ma)
                {
                    m = ma;
                    n = ++na;
                    count = acount;
                }
                else
                    return false;
            }
        }
    }
}

// Return the type of an open file
//
// Some file types on some platforms seem seekable but in fact are not.
// The main use of this function is to allow such cases to be detected
// (IsSeekable() is implemented as wxGetFileKind() == wxFILE_KIND_DISK).
//
// This is important for the archive streams, which benefit greatly from
// being able to seek on a stream, but which will produce corrupt archives
// if they unknowingly seek on a non-seekable stream.
//
// wxFILE_KIND_DISK is a good catch all return value, since other values
// disable features of the archive streams. Some other value must be returned
// for a file type that appears seekable but isn't.
//
// Known examples:
//   *  Pipes on Windows
//   *  Files on VMS with a record format other than StreamLF
//
wxFileKind wxGetFileKind(int fd)
{
#if defined __WXMSW__ && !defined __WXWINCE__ && defined wxGetOSFHandle
    switch (::GetFileType(wxGetOSFHandle(fd)) & ~FILE_TYPE_REMOTE)
    {
        case FILE_TYPE_CHAR:
            return wxFILE_KIND_TERMINAL;
        case FILE_TYPE_DISK:
            return wxFILE_KIND_DISK;
        case FILE_TYPE_PIPE:
            return wxFILE_KIND_PIPE;
    }

    return wxFILE_KIND_UNKNOWN;

#elif defined(__UNIX__)
    if (isatty(fd))
        return wxFILE_KIND_TERMINAL;

    struct stat st;
    fstat(fd, &st);

    if (S_ISFIFO(st.st_mode))
        return wxFILE_KIND_PIPE;
    if (!S_ISREG(st.st_mode))
        return wxFILE_KIND_UNKNOWN;

    #if defined(__VMS__)
        if (st.st_fab_rfm != FAB$C_STMLF)
            return wxFILE_KIND_UNKNOWN;
    #endif

    return wxFILE_KIND_DISK;

#else
    #define wxFILEKIND_STUB
    (void)fd;
    return wxFILE_KIND_DISK;
#endif
}

wxFileKind wxGetFileKind(FILE *fp)
{
    // Note: The watcom rtl dll doesn't have fileno (the static lib does).
    //       Should be fixed in version 1.4.
#if defined(wxFILEKIND_STUB) || \
        (defined(__WATCOMC__) && __WATCOMC__ <= 1230 && defined(__SW_BR))
    (void)fp;
    return wxFILE_KIND_DISK;
#else
    return fp ? wxGetFileKind(fileno(fp)) : wxFILE_KIND_UNKNOWN;
#endif
}

#ifdef __VISUALC__
    #pragma warning(default:4706)   // assignment within conditional expression
#endif // VC++
