/*************************************************************************
 *                                                                       *
 * exif.cpp                                                              *
 * EXIF specific code                                                    *
 *                                                                       *
 *************************************************************************/


#include "exif.hpp"
#include "string.hpp"

#include <stdio.h>
#include <iostream>
#include <algorithm>


////////// class IFD

const int IFD::Value::TypeLengths[12] = { 1,1,2,4,8,1,1,2,4,8,4,8 };

unsigned long IFD::Value::GetInt() const
{  switch (Type)
   {case TIFF_Byte:
    case TIFF_Undefined: // we can read undefind data as characters
      return Fetch<TtiffByte>();
    case TIFF_Short:
      return Fetch<TtiffShort>();
    case TIFF_Long:
      return Fetch<TtiffLong>();
    default:
      throw stringf("Incompatible type conversion from %u to unsigned long requested.", Type);
   }
}

long IFD::Value::GetSInt() const
{  switch (Type)
   {case TIFF_SByte:
      return Fetch<TtiffSByte>();
    case TIFF_SShort:
      return Fetch<TtiffSShort>();
    case TIFF_SLong:
      return Fetch<TtiffSLong>();
    case TIFF_Byte:
      return Fetch<TtiffByte>();
    case TIFF_Short:
      return Fetch<TtiffShort>();
    default:
      throw stringf("Incompatible type conversion from %u to signed long requested.", Type);
   }
}

IFD::TtiffRational IFD::Value::GetRational() const
{  if (Type != TIFF_Rational)
      throw stringf("Incompatible type conversion from %u to unsigned rational requested.", Type);
   TtiffRational r;
   r.numerator = Fetch<TtiffLong>(0);
   r.denominator = Fetch<TtiffLong>(1);
   return r;
}

IFD::TtiffSRational IFD::Value::GetSRational() const
{  if (Type != TIFF_SRational)
      throw stringf("Incompatible type conversion from %u to signed rational requested.", Type);
   TtiffSRational r;
   r.numerator = Fetch<TtiffSLong>(0);
   r.denominator = Fetch<TtiffSLong>(1);
   return r;
}

double IFD::Value::GetFloat() const
{  switch (Type)
   {case TIFF_Float:
      return Fetch<TtiffFloat>();
    case TIFF_Double:
      return Fetch<TtiffDouble>();
    case TIFF_Rational:
      return (double)Fetch<TtiffLong>(0) / Fetch<TtiffLong>(1);
    case TIFF_SRational:
      return (double)Fetch<TtiffSLong>(0) / Fetch<TtiffSLong>(1);
    default:
      throw stringf("Incompatible type conversion from %u to double requested.", Type);
   }
}

template <typename T>
static T ggT(T a, T b)
{  while (T c = b ^= a ^= b ^= a ^= c ^= a ^= c = a % b);
   return a;
}

void IFD::Value::CancelRational(TtiffRational& r)
{  TtiffLong ggt = ggT(r.numerator, r.denominator);
   r.numerator /= ggt;
   r.denominator /= ggt;
}

void IFD::Value::CancelRational(TtiffSRational& r)
{  TtiffSLong ggt = ggT(r.numerator, r.denominator);
   r.numerator /= ggt;
   r.denominator /= ggt;
}

void IFD::Value::toostream(std::ostream& os) const
{  switch (Type)
   {case TIFF_Byte:
    case TIFF_Short:
    case TIFF_Long:
      os << GetInt();
      break;
    case TIFF_Ascii:
      os << Fetch<TtiffAscii>();
      break;
    case TIFF_Rational:
      {  TtiffRational r = GetRational();
         CancelRational(r);
         os << r.numerator << '/' << r.denominator;
      }
      break;
    case TIFF_SByte:
    case TIFF_SShort:
    case TIFF_SLong:
      os << GetSInt();
      break;
    //case TIFF_Undefined:
    default:
      os << (int)Fetch<TtiffUndefined>();
      break;
    case TIFF_SRational:
      {  TtiffSRational r = GetSRational();
         CancelRational(r);
         os << r.numerator << '/' << r.denominator;
      }
      break;
    case TIFF_Float:
    case TIFF_Double:
      os << GetFloat();
}  }


std::string IFD::ValueList::GetASCII() const
{  if (Type != TIFF_Ascii)
      throw stringf("Incompatible type conversion from %u to string requested.", Type);
   std::string s((const char*)Data, Count);
   if (*s.rbegin() == '\0') // remove terminating zero
      s.resize(s.length()-1);
   for (std::string::iterator i = s.begin(); i != s.end(); ++i) // replace embedded zeros by tabs
      if (*i == '\0')
         *i = '\t';
   return s;
}

const IFD::Value IFD::ValueList::operator[](unsigned element) const
{  if (element >= Count)
      throw stringf("IFD element index out of bounds %u/%u.", element, Count);
   return Value(Convert, Type, Data + TypeLengths[Type-1]*element);
}

void IFD::ValueList::toostream(std::ostream& os) const
{  switch (Type)
   {case TIFF_UNDEFINED:
      os << "n/a";
      break;
    case TIFF_Ascii:
      os << GetASCII();
      break;
    case TIFF_Byte:
    case TIFF_Short:
    case TIFF_Long:
    case TIFF_Rational:
    case TIFF_SByte:
    case TIFF_SShort:
    case TIFF_SLong:
    case TIFF_SRational:
    case TIFF_Float:
    case TIFF_Double:
      os << (Value)*this; // 1st element
      for (unsigned i = 1; i < std::min(Count, 16U); ++i)
         os << ", " << (*this)[i];
      break;
    //case TIFF_Undefined:
    default:
      os << std::hex << (Value)*this; // 1st element
      for (unsigned i = 1; i < std::min(Count, 16U); ++i)
         os << ' ' << (*this)[i];
      os << std::dec;
      break;
}  }


const IFD::ValueList IFD::NoValue;

const char* IFD::TypeNames[] = {"byte", "ASCII", "short", "long", "rational", "sbyte", "undef.", "sshort", "slong", "srational", "float", "double"};

bool IFD::Parse()
{
   #ifdef DEBUG
   printf("Parsing IFD-Table at %x.\n", Data-Root);
   #endif
   int n = Fetch<TtiffShort>(); // number of elements
   if (n > 0x1000)
      throw stringf("Implausible number of elements %u in IFD table.", n);
   while (n--) // read all tags
   {  try
      {  ValueList v(Convert);
         int tag = Fetch<TtiffShort>();
         v.Type = Fetch<TtiffShort>();
         v.Count = Fetch<TtiffLong>();
         const TtiffLong d = Fetch<TtiffLong>();
         if (v.Type-1 >= 12)
            throw stringf("Invalid type code %u in IFD tag %u.", v.Type, tag);
         #ifdef DEBUG2
         printf("Entry: %x [%x] = size %x * %x at %x ...", tag, v.Type, v.Count, Value::TypeLengths[v.Type-1], d);
         #endif
         TtiffLong l = Value::TypeLengths[v.Type-1] * v.Count;
         if (l <= 4)
            v.Data = Data -4; // reference to already fetched data location
          else
         {  v.Data = Root + d;
            if (v.Data + l > DataEnd)
               throw stringf("IFD error: Data of tag %x is beyond the end of the data buffer.\n (Offset %x, Length %x, Buffer %x)", tag, d, l, DataEnd-Root);
         }
         if (IFDList.insert(IFDListType::value_type(tag, v)).second == false)
            throw stringf("Invalid duplicate tag %x in IFD.", tag);
         #ifdef DEBUG2
         printf(" added, now %u.\n", IFDList.size());
         #endif
      } catch (const std::string& s)
      {  fprintf(stderr, "%s At offset %x. Ignoring tag.\n", s.c_str(), Data-Root);
      }
   }
   // advance to next IFD if any
   TtiffLong next = Fetch<TtiffLong>();
   #ifdef DEBUG
   printf("IFD-Table finished at %x, next pointer: %x.\n", Data-Root, next);
   #endif
   Data = Root + next;
   return next != 0;
}

const IFD::ValueList& IFD::operator[](int tag) const
{  IFDListType::const_iterator i = IFDList.find(tag);
   return i == IFDList.end() ? NoValue : i->second;
}

void IFD::Reset(const Endian& convert, const char* root, const char* data, const char* dataend)
{  Convert = convert;
   Root = root;
   Data = data;
   DataEnd = dataend;
   if (Data > DataEnd)
      throw stringf("IFD error: tried to initialize IFD with data pointer field beyond the end of the data.\n (Offset %x, Datalength %x)", Data-Root, DataEnd-Root);
}

void IFD::Reset(TtiffLong offset)
{  Data = Root + offset;
   if (Data > DataEnd)
      throw stringf("IFD error: tried to access field beyond the end of the data.\n (Offset %x, Datalength %x)", offset, DataEnd-Root);
}

void IFD::itemtoostream(std::ostream& os, const IFDListType::value_type& item) const
{  os << std::hex << item.first << std::dec;
   os << " [" << TypeNames[item.second.Type-1] << ']';
   os << ": " << item.second << std::endl;
}

void IFD::toostream(std::ostream& os) const
{
   #ifdef DEBUG2
   printf("dumping %u entries.\n", IFDList.size());
   #endif
   for (IFDListType::const_iterator i = IFDList.begin(); i != IFDList.end(); ++i)
      itemtoostream(os, *i);
}


////////// class TIFF

TIFF::TIFF(const char* tiffheader, const char* dataend)
{  Reset(tiffheader, dataend);
}

void TIFF::Parse()
{  // Endianess
   TtiffShort s = Fetch<TtiffShort>(); // endian invariant so far
   switch (s)
   {default:
      throw stringf("Unknown byteOrder %04.4x.", s);
    case TIFF_II:
      Convert.Swap = false; // TODO: swap on Motorola
      #ifdef DEBUG2
      printf("found TIFF header with intel byte order.\n");
      #endif
      break;
    case TIFF_MM:
      Convert.Swap = true; // TODO: swap on Motorola
      #ifdef DEBUG2
      printf("found TIFF header with motorola byte order.\n");
      #endif
   }
   // TIFF magic
   s = Fetch<TtiffShort>();
   if (s != TIFF_42)
      throw stringf("Not a TIFF header, bad magic code %04.4x.", s);
   // IFD 0 offset
   TtiffLong offset = Fetch<TtiffLong>();
   #ifdef DEBUG
   printf("offset %p %x\n", Root, offset);
   #endif
   if (offset > 0x1000)
      throw stringf("Implausible 0th IFD offset %x.", offset);
   Reset(offset);
   // Parse IFD 0
   IFD::Parse();
}

////////// class Exif

// TODO: must be false on motorola architecture (Endianess)
static const Endian Convert4JPEG(true);

const Exif::DecodeEntry Exif::DecodeEntry::Table[] =
{  { TIFF_ImageWidth, Basic|MoreAfter, "Image width * height: ", " pixel", &DecodeEntry::DDefault }
 , { TIFF_ImageLength, Basic|MoreBefore, " * ", " pixel", &DecodeEntry::DDefault }
 , { TIFF_BitsPerSample, Detail, "Bits per component: ", " bits", &DecodeEntry::DDefault }
 , { TIFF_SamplesPerPixel, Detail, "Samples per pixel: ", "", &DecodeEntry::DDefault }
 , { TIFF_Compression, Internal, "Compression: ", "|uncompressed||||JPEG compression", &DecodeEntry::DToken }
 , { TIFF_PhotometricInterpretation, Detail, "Pixel composition: ", "||RGB||||YCbCr", &DecodeEntry::DToken }
 , { TIFF_Orientation, Detail, "Orientation: ", "|Landscape, top left to bottom right|Landscape, top right to bottom left|Landscape, bottom right to top left|Landscape, bottom left to top right|Portrait, left top to right bottom|Portrait, right top to left bottom|Portrait, right bottom to left top|Portrait, left bottom to right top", &DecodeEntry::DToken }
 , { TIFF_PlanarConfiguration, Internal, "Planar configuration: ", "|chunky format|planar format", &DecodeEntry::DToken }
 , { TIFF_YCbCrSubSampling, Detail, "Subsampling ratio of Y to C: ", "", &DecodeEntry::DDefault }
 , { TIFF_YCbCrPositioning, Detail, "Y to C positioning: ", "|centered|co-sited", &DecodeEntry::DToken }
 , { TIFF_XResolution, Basic|MoreAfter, "Resolution X * Y: ", " dpi", &DecodeEntry::DDefault }
 , { TIFF_YResolution, Basic|MoreBefore, " * ", " dpi", &DecodeEntry::DDefault }
 , { TIFF_ResolutionUnit, Basic, "Resolution Unit: ", "||inch|centimeter", &DecodeEntry::DToken }
 , { TIFF_StripOffset, Internal, "Strip offsets: ", " bytes", &DecodeEntry::DDefault }
 , { TIFF_RowsPerStrip, Internal, "Rows per strip: ", " rows", &DecodeEntry::DDefault }
 , { TIFF_StripByteCounts, Internal, "Bytes per strip: ", " bytes", &DecodeEntry::DDefault }
 , { TIFF_JpegInterchangeFormat, Detail, "JPEG thumbnail offset: ", " bytes", &DecodeEntry::DDefault }
 , { TIFF_JpegInterchangeFormatLength, Detail, "JPEG thumbnail length: ", " bytes", &DecodeEntry::DDefault }
 , { TIFF_TransferFunction, Internal, "Transfer function: ", "", &DecodeEntry::DDefault }
 , { TIFF_WhitePoint, Internal, "Chormaticity of the white point: ", "", &DecodeEntry::DDefault }
 , { TIFF_PrimaryChromaticities, Internal, "Chormaticity of the primary colors: ", "", &DecodeEntry::DDefault }
 , { TIFF_YCbCrCoefficients, Internal, "YCbCr coefficients: ", "", &DecodeEntry::DDefault }
 , { TIFF_ReferenceBlackWhite, Internal, "Black and white reference: ", "", &DecodeEntry::DDefault }
 , { TIFF_DateTime, Basic|MoreAfter, "Date/time: ", "", &DecodeEntry::DDefault }
 , { EXIF_SubSecTime, Basic|MoreBefore, ".", "", &DecodeEntry::DDefault }
 , { TIFF_ImageDescription, Basic, "Image description: ", "", &DecodeEntry::DDefault }
 , { TIFF_Make, Basic, "Equipment manufacturer: ", "", &DecodeEntry::DDefault }
 , { TIFF_Model, Basic, "Equipment model: ", "", &DecodeEntry::DDefault }
 , { TIFF_Software, Basic, "Equipment software/firmware: ", "", &DecodeEntry::DDefault }
 , { TIFF_Artist, Basic, "Artist: ", "", &DecodeEntry::DDefault }
 , { TIFF_Copyright, Basic, "Copyright: ", "", &DecodeEntry::DDefault }
 , { EXIF_ExifVersion, Basic, "EXIF version: ", "", &DecodeEntry::DasASCII }
 , { EXIF_FlashpixVersion, Basic, "Flashpix version: ", "", &DecodeEntry::DasASCII }
 , { EXIF_ColorSpace, Detail, "Color space: ", "sRGB", &DecodeEntry::DToken } // TODO: uncalibrated
 , { EXIF_PixelXDimension, Detail|MoreAfter, "Valid image width * height: ", " pixel", &DecodeEntry::DDefault }
 , { EXIF_PixelYDimension, Detail|MoreBefore, " * ", "pixel", &DecodeEntry::DDefault }
 , { EXIF_ComponentConfiguration, Internal, "Components configuration: ", "-|Y|Cb|Cr|R|G|B", &DecodeEntry::DToken }
 , { EXIF_CompressedBitsPerPixel, Detail, "Compression mode: ", " bits/pixel", &DecodeEntry::DDefault }
 , { EXIF_MakerNote, Internal, "Maker note: ", "", &DecodeEntry::DDefault } // We will usually decode this otherwise
 , { EXIF_UserComment, Basic, "User comment: ", "", &DecodeEntry::DCharSet }
 , { EXIF_DateTimeOriginal, Basic|MoreAfter, "Date/time of original data generation: ", "", &DecodeEntry::DDefault }
 , { EXIF_SubSecTimeOriginal, Basic|MoreBefore, ".", "", &DecodeEntry::DDefault }
 , { EXIF_DateTimeDigitized, Basic|MoreAfter, "Date/time of digital data generation: ", "", &DecodeEntry::DDefault }
 , { EXIF_SubSecTimeDigitized, Basic|MoreBefore, ".", "", &DecodeEntry::DDefault }
 , { EXIF_ExposureProgram, Basic, "Exposure program: ", "|manual|normal program|aperture priority|shutter priority|creative program|action program|portrait program|landscape program", &DecodeEntry::DToken }
 , { EXIF_SpectralSensitivity, Detail, "Spectral sensitivity: ", "", &DecodeEntry::DDefault }
 , { EXIF_ISOSpeedRatings, Detail, "ISO speed rating: ", " ISO", &DecodeEntry::DDefault }
 , { EXIF_OECF, Internal, "Opto electronic conversion function: ", "", &DecodeEntry::DDefault }
 , { EXIF_ExposureTime, Basic, "Exposure time: ", " seconds", &DecodeEntry::DDefault }
 , { EXIF_ShutterSpeedValue, Basic, "Shutter speed: ", " seconds", &DecodeEntry::DDefault }
 , { EXIF_ApertureValue, Basic, "Aperture: ", " APEX", &DecodeEntry::DDefault }
 , { EXIF_BrightnessValue, Basic, "Brightness: ", " APEX", &DecodeEntry::DDefault }
 , { EXIF_ExposureBiasValue, Basic, "Exposure bias: ", " APEX", &DecodeEntry::DDefault }
 , { EXIF_MaxApertureValue, Detail, "Max. lens aperture: ", " APEX", &DecodeEntry::DDefault }
 , { EXIF_SubjectDistance, Basic, "Subject distance: ", " m", &DecodeEntry::DDefault }
 , { EXIF_MeteringMode, Basic, "Metering mode: ", "unknown|average|center weighted average|spot|multi spot|pattern|partial", &DecodeEntry::DToken }
 , { EXIF_LightSource, Basic, "Light source: ", "unknown|daylight|flourescent|tungsten|flash|||||fine weather|cloudy weather|shade|daylight flourescent|day white flourescent|cool white flourescent|white flourescent||standard light A|standard light B|standard light C|D55|D65|D75|D50ISO studio tungsten", &DecodeEntry::DToken }
 , { EXIF_Flash, Basic, "Flash: ", "flash not fired|flash fired||||flash fired, return light not detected||flash fired, return light detected||flash fired, compulsory mode||||flash fired, compulsory mode, return light not detected||flash fired, compulsory mode, return light detected|"
                                   "flash not fired, compulsary mode||||||||flash not fired, auto mode|flash fired, auto mode||||flash fired, auto mode, return light not detected||flash fired, auto mode, return light detected|"
                                   "no flash||||||||||||||||"
                                   "||||||||||||||||"
                                   "|flash fired, red-eye reduction mode||||flash fired, red-eye reduction mode, return light not detected||flash fired, red-eye reduction mode, return light detected||flash fired, compulsary mode, red-eye reduction mode||||flash fired, compulsary mode, red-eye reduction mode, return light not detected||flash fired, compulsary mode, red-eye reduction mode, return light detected|"
                                   "|||||||||flash fired, auto mode, red-eye reduction mode||||flash fired, auto mode, red-eye reduction mode, return light not detected||flash fired, auto mode, red-eye reduction mode, return light detected", &DecodeEntry::DToken }
 , { EXIF_SubjectArea, Detail, "Subject area: ", " pixel", &DecodeEntry::DDefault }
 , { EXIF_FocalLength, Basic, "Focal length: ", " mm", &DecodeEntry::DDefault }
 , { EXIF_FlashEnergy, Basic, "Flash energy: ", " BCPS", &DecodeEntry::DDefault }
 , { EXIF_SpatialFrequencyResponse, Detail|MoreBefore, "Spacial frequency response: ", "", &DecodeEntry::DDefault }
 , { EXIF_FocalPlaneXResolution, Detail|MoreAfter, "Focal plane resolution X * Y: ", " dpi", &DecodeEntry::DDefault }
 , { EXIF_FocalPlaneYResolution, Detail|MoreBefore, " * ", " dpi", &DecodeEntry::DDefault }
 , { EXIF_FocalPlaneResolutionUnit, Detail, "Focal plane resolution Unit: ", "||inch|centimeter", &DecodeEntry::DToken }
 , { EXIF_SubjectLocation, Basic, "Subject location: ", " pixel", &DecodeEntry::DDefault }
 , { EXIF_ExposureIndex, Basic, "Exposure index: ", "", &DecodeEntry::DDefault }
 , { EXIF_SensingMethod, Detail, "Sensing method: ", "||one-chip color area sensor|two-chip color area sensor|three-chip color area sensor|color sequential area sensor|trilinear sensor|color sequential linear sensor", &DecodeEntry::DToken }
 , { EXIF_FileSource, Internal, "File Source: ", "|||DSC", &DecodeEntry::DToken }
 , { EXIF_SceneType, Internal, "Scene type: ", "|directly photographed", &DecodeEntry::DToken }
 , { EXIF_CFAPattern, Internal, "CFA pattern: ", "", &DecodeEntry::DDefault }
 , { EXIF_CustomRendered, Detail, "Custom rendered: ", "normal|custom", &DecodeEntry::DToken }
 , { EXIF_ExposureMode, Basic, "Exposure mode: ", "auto|manual|auto bracket", &DecodeEntry::DToken }
 , { EXIF_WhiteBalance, Basic, "White balance: ", "auto|manual", &DecodeEntry::DToken }
 , { EXIF_DigitalZoomRatio, Basic, "Digital zoom: * ", "", &DecodeEntry::DDefault }
 , { EXIF_FocalLengthIn35mmFilm, Basic, "Focal length in 35mm film: ", " mm", &DecodeEntry::DDefault }
 , { EXIF_GainControl, Detail, "Gain control: ", "|low gain up|high gain up|low gain down|high gain down", &DecodeEntry::DToken }
 , { EXIF_Contrast, Basic, "Contrast: ", "normal|soft|hard", &DecodeEntry::DToken }
 , { EXIF_Saturation, Basic, "Saturation: ", "normal|low|high", &DecodeEntry::DToken }
 , { EXIF_Sharpness, Basic, "Sharpness: ", "normal|soft|hard", &DecodeEntry::DToken }
 , { EXIF_DeviceSettingDescription, Internal, "Picture taking condition: ", "", &DecodeEntry::DDefault }
 , { EXIF_SubjectDistanceRange, Detail, "Subject distance range: ", "unknown|macro|close view|distant view", &DecodeEntry::DToken }
 , { EXIF_ImageUniqueID, Detail, "Unique image ID: ", "", &DecodeEntry::DDefault }
 , { EXIF_InteroperabilityIndex, Detail, "Interoperability index: ", "", &DecodeEntry::DDefault }
 , { EXIF_InteroperabilityVersion, Detail, "Interoperability version: ", "", &DecodeEntry::DasASCII}
};

Exif::DecodeEntry::SortedDecodeTable Exif::DecodeEntry::SortedTable = DecodeEntry::SortTable();

Exif::DecodeEntry::SortedDecodeTable Exif::DecodeEntry::SortTable()
{  std::map<int, const Exif::DecodeEntry*> sl;
   const DecodeEntry* end = DecodeEntry::Table + sizeof DecodeEntry::Table / sizeof *DecodeEntry::Table;
   const DecodeEntry* dep = DecodeEntry::Table;
   while (dep != end)
   {  sl[dep->Tag] = dep;
      ++dep;
   }
   return sl;
}

void Exif::DecodeEntry::DDefault(std::ostream& os, const IFD::ValueList& vl) const
{  os << Name << vl << Unit;
}

void Exif::DecodeEntry::DasASCII(std::ostream& os, const IFD::ValueList& vl) const
{  ValueList myvl = vl; // create a writable copy
   myvl.Count *= Value::TypeLengths[myvl.Type-1]; // ensure the same data length
   myvl.Type = TIFF_Ascii; // type override
   DDefault(os, myvl); // continue as usual
}

void Exif::DecodeEntry::DToken(std::ostream& os, const IFD::ValueList& vl) const
{  unsigned i = 0;
   os << Name;
   while (true)
   {  int v = vl[i].GetInt();
      // Search for the i-th '|' delimited substring in Unit. If beyond the end or empty => unkown.
      // We won't win the speed price here...
      const char* cp = Unit;
      for (;;)
      {  size_t n = strcspn(cp, "|");
         if (v == 0)
         {  // found
            if (n != 0)
            {  // OK
               os.write(cp, n);
               goto gotit;
            } else
               break; // error
         }
         --v;
         cp += n;
         if (*cp == 0)
            break; // end of string => error
         ++cp;
      }
      // error
      os << "unknown (" << vl << ')';
    gotit:
      if (++i >= vl.Count)
         break;
      os << ", ";
   }
}

void Exif::DecodeEntry::DCharSet(std::ostream& os, const IFD::ValueList& vl) const
{  if (vl.Count < 8 || vl.Type != TIFF_Undefined)
      DDefault(os, vl); // better than crash
   os << Name;
   if (strcmp(vl.Data, "ASCII") == 0)
   {  std::string s(vl.Data+8, vl.Count-8);
      if (*s.rbegin() == '\0') // remove terminating zero
         s.resize(s.length()-1);
      os << s;
   } else if (strcmp(vl.Data, "JIS") == 0)
   {  // TODO: Whatever charset this is
      std::string s(vl.Data+8, vl.Count-8);
      if (*s.rbegin() == '\0') // remove terminating zero
         s.resize(s.length()-1);
      os << s;
   } else if (strcmp(vl.Data, "UNICODE") == 0)
   {  // Really BAD Unicode conversion
      std::string s((vl.Count-8)/2, '.');
      const unsigned short* sp = (const unsigned short*)(vl.Data +8);
      for (unsigned i = 0; i < s.length(); ++i, ++sp)
      {  if (sp[0] == 0)
            s[i] = (char)sp[1];
         sp += 2;
      }
   } else // undefined -> standard
      os << vl;
   os << Unit;
}

const Exif::DecodeEntry* Exif::DecodeEntry::Find(int tag)
{  SortedDecodeTable::const_iterator i = SortedTable.find(tag);
   return i == SortedTable.end() ? NULL : i->second;
}

void Exif::itemtoostream(std::ostream& os, const IFDListType::value_type& item) const
{  const DecodeEntry* dep = DecodeEntry::Find(item.first);
   if (dep != NULL)
   {  if ((LastItemFlags & DecodeEntry::MoreAfter) && !(dep->Flags & DecodeEntry::MoreBefore))
         os << std::endl;
      (dep->*dep->DisplayFunction)(os, item.second); // known Tag
      if (!(dep->Flags & DecodeEntry::MoreAfter))
         os << std::endl;
      LastItemFlags = dep->Flags;
   } else if (ShowUnknown) // unknown Tag
      IFD::itemtoostream(os, item);
}

void Exif::toostream(std::ostream& os) const
{  LastItemFlags = 0;
   TIFF::toostream(os);
   if (LastItemFlags & DecodeEntry::MoreAfter)
      os << std::endl;
}

bool Exif::CompareDecodeEntry(const DecodeEntry& de, int tag)
{  return de.Tag < tag;
}

Exif::Exif(const char* fileheader, const char* dataend) throw (std::string)
 : ShowUnknown(false)
{  switch (Convert4JPEG(*(TtiffShort*)fileheader))
   {default:
      throw stringf("File seems to be neither a JPEG nor a TIFF image. Signature %04.4x.", Convert4JPEG(*(TtiffShort*)fileheader));
    case JPEG_SOI: // is JPEG
      if ( Convert4JPEG(((TtiffShort*)fileheader)[1]) != JPEG_APP1 || memcmp("Exif\0\0", fileheader+6, 6) != 0 )
         throw std::string("JPEG file has no Exif information.");
      TIFF::Reset(fileheader + 12, dataend);
      break;
    case TIFF_II:
    case TIFF_MM:
      TIFF::Reset(fileheader, dataend);
      break;
   }
}

/*Exif::~Exif()
{  delete Tiff;
}*/

void Exif::ParseSubIFD(int tag, const char* name) throw()
{  try
   {  const Value& v = (*this)[tag];
      if (v.Type != TIFF_Long || v.Data == 0)
         throw stringf("TIFF Header contains no %s SubIFD.", name);
      #ifdef DEBUG
      printf("%s info at %x\n", name, v.Data - Root);
      #endif
      Reset(v.GetInt());
      IFD::Parse(); // add values
   } catch (const std::string& msg)
   {  std::cout << msg << std::endl;
   }
}

void Exif::Parse()
{  TIFF::Parse();

   ParseSubIFD(TIFF_ExifIFDPointer, "Exif");
   ParseSubIFD(TIFF_GPSInfoIFDPointer, "GPS");
   ParseSubIFD(TIFF_InteroperabilityIFDPointer, "Interoperability");

   // try to extract maker specific data
   {  IFD::Value v = (*this)[EXIF_MakerNote];
      /*if (v.Type != 0)
      {  // try to identify MakerNote data format
         if (memcmp(v.GetRAW(), "OLYMP", 6) == 0)
            Tiff->Reset(v.Data + 7);
          else if (
         ((IFD)Tiff)->Parse(); // add values
      }
      */
   }
}

