#include "mainwindow.h"

#include <QtGui/QApplication>
#include <QtGui/QWidget>
#include <QtGui/QMenu>
#include <QtGui/QMenuBar>
#include <QtGui/QAction>
#include <QtGui/QStatusBar>
#include <QtGui/QScrollArea>
#include <QtGui/QScrollBar>
#include <QString>
#include <QtGui/QFileDialog>
#include <QtGui/QMessageBox>
#include <QPoint>
#include <QtGui/QToolButton>
#include <QtGui/QToolBar>
#include <QtGui/QSpinBox>
#include <QtGui/QTabWidget>
#include <QtGui/QGridLayout>
#include <QtGui/QSlider>
#include <QtGui/QStyle>
#include <QtGui/QStyleFactory>
#include <QDropEvent>
#include <QUrl>
#include <QClipboard>

/*!
 \brief конструктор главной формы

 \param parent указатель на родителя
*/
MainWindow::MainWindow(QWidget *parent) : QMainWindow(parent)
{
    QApplication::setStyle(QStyleFactory::create("Plastique"));
    setWindowIcon(QPixmap(":/program/icons/program/paint.png"));
    setAcceptDrops(true);

    tabBar = new QTabWidget();
    tabBar->setUsesScrollButtons(true);
    tabBar->setTabsClosable(true);
    tabBar->setMovable(true);
    setCentralWidget(tabBar);

    colorSet = new CColorWidget(0, 0, 0);
    colorSet->setStatusTip(tr("Change pen color"));
    brushSet = new CColorWidget(255, 255, 255);
    brushSet->setStatusTip(tr("Change brush color"));
    spinTricknessSize = new QSpinBox();
    spinTricknessSize->setRange(1, 30);
    spinTricknessSize->setValue(1);
    spinTricknessSize->setStatusTip(tr("Change pen trickness"));
    activeTool = NONE;
    penColor = Qt::black;
    brushColor = Qt::white;
    penSize = 1;

    loadActions();
    loadMenu();
    loadToolBars();
    loadStatBar();
    loadSignals();
    setNew(520,400);
    resize(650, 490);
}

/*!
 \brief деструктор главной формы

*/
MainWindow::~MainWindow()
{
    delete newAction;
    delete exitAction;
    delete openAction;
    delete saveAsAction;
    delete saveAction;
    delete closeAction;
    delete printAction;
    delete aboutAction;
    delete aboutQtAction;
    delete undoAction;
    delete redoAction;
    delete toBlackAction;
    delete toNegative;
    delete rotateAction;
    delete resizeImageAction;
    delete resizeAreaAction;
    delete paletteAction;
    delete setUndoSizeAction;
    delete cutAction;
    delete copyAction;
    delete pasteAction;
    delete fileMenu;
    delete editMenu;
    delete settingsMenu;
    delete aboutMenu;
    delete statBar;
    delete colorSet;
    delete brushSet;
    //delete paletteSet;
    delete gadgetsBar;
    delete paletteBar;
    delete tabBar;
}

/*!
 \brief загрузка экшенов

*/
void MainWindow::loadActions()
{
    newAction = new QAction(tr("&New"), this);
    newAction->setShortcut(QKeySequence::New);
    newAction->setIcon(QPixmap(":/file-menu/icons/file/new.png"));
    newAction->setStatusTip(tr("Create new image"));
    connect(newAction, SIGNAL(triggered()), this, SLOT(createNew()));
    exitAction = new QAction(tr("E&xit"), this);
    exitAction->setShortcut(tr("Ctrl+Q"));
    exitAction->setIcon(QPixmap(":/file-menu/icons/file/exit.png"));
    exitAction->setStatusTip(tr("Exit the application"));
    connect(exitAction, SIGNAL(triggered()), this, SLOT(close()));
    openAction = new QAction(tr("&Open..."), this);
    openAction->setShortcut(QKeySequence::Open);
    openAction->setIcon(QPixmap(":/file-menu/icons/file/open.png"));
    openAction->setStatusTip(tr("Open an existing image file"));
    connect(openAction, SIGNAL(triggered()), this, SLOT(openFile()));
    saveAsAction = new QAction(tr("Save &as..."), this);
    saveAsAction->setShortcut(QKeySequence::SaveAs);
    saveAsAction->setIcon(QPixmap(":/file-menu/icons/file/save-as.png"));
    saveAsAction->setStatusTip(tr("Save image to disk"));
    connect(saveAsAction, SIGNAL(triggered()), this, SLOT(saveAsFile()));
    saveAction = new QAction(tr("&Save"), this);
    saveAction->setShortcut(QKeySequence::Save);
    saveAction->setIcon(QPixmap(":/file-menu/icons/file/save.png"));
    saveAction->setStatusTip(tr("Save image to disk"));
    connect(saveAction, SIGNAL(triggered()), this, SLOT(saveFile()));
    closeAction = new QAction(tr("&Close"), this);
    closeAction->setShortcut(QKeySequence::Close);
    closeAction->setIcon(QPixmap(":/file-menu/icons/file/close.png"));
    closeAction->setStatusTip(tr("Close image"));
    connect(closeAction, SIGNAL(triggered()), this, SLOT(sendToCloseFile()));
    printAction = new QAction(tr("&Print"), this);
    printAction->setShortcut(QKeySequence::Print);
    printAction->setIcon(QPixmap(":/file-menu/icons/file/print.png"));
    printAction->setStatusTip(tr("Print image"));
    connect(printAction, SIGNAL(triggered()), this, SLOT(printImage()));
    aboutAction = new QAction(tr("&About"), this);
    aboutAction->setShortcut(QKeySequence::HelpContents);
    aboutAction->setIcon(QPixmap(":/about-menu/icons/about/about.png"));
    aboutAction->setStatusTip(tr("Show the application's About box"));
    connect(aboutAction, SIGNAL(triggered()), this, SLOT(about()));
    aboutQtAction = new QAction(tr("About &Qt"), this);
    aboutQtAction->setIcon(QPixmap(":/about-menu/icons/about/aboutQt.png"));
    aboutQtAction->setStatusTip(tr("Show the Qt library's About box"));
    connect(aboutQtAction, SIGNAL(triggered()), qApp, SLOT(aboutQt()));
    undoAction = new QAction(tr("&Undo"), this);
    undoAction->setShortcut(QKeySequence::Undo);
    undoAction->setIcon(QPixmap(":/edit-menu/icons/edit/undo.png"));
    undoAction->setStatusTip(tr("Undo last action"));
    undoAction->setEnabled(false);
    connect(undoAction, SIGNAL(triggered()), this, SLOT(setUndo()));
    redoAction = new QAction(tr("&Redo"), this);
    redoAction->setShortcut(QKeySequence::Redo);
    redoAction->setIcon(QPixmap(":/edit-menu/icons/edit/redo.png"));
    redoAction->setStatusTip(tr("Redo action"));
    redoAction->setEnabled(false);
    connect(redoAction, SIGNAL(triggered()), this, SLOT(setRedo()));
    toBlackAction = new QAction(tr("&Black-White"), this);
    toBlackAction->setStatusTip(tr("Convert image to black-white"));
    connect(toBlackAction, SIGNAL(triggered()), this, SLOT(setBlackWhite()));
    toNegative = new QAction(tr("&Negative"), this);
    toNegative->setStatusTip(tr("Convert image to negative"));
    connect(toNegative, SIGNAL(triggered()), this, SLOT(setNegative()));
    rotateAction = new QAction(tr("R&otate"), this);
    rotateAction->setStatusTip(tr("Rotate image"));
    connect(rotateAction, SIGNAL(triggered()), this, SLOT(setRotate()));
    resizeImageAction = new QAction(tr("&Resize image"), this);
    resizeImageAction->setStatusTip(tr("Resize image"));
    connect(resizeImageAction, SIGNAL(triggered()), this, SLOT(resizeImage()));
    resizeAreaAction = new QAction(tr("R&esize area"), this);
    resizeAreaAction->setStatusTip(tr("Resize area"));
    connect(resizeAreaAction, SIGNAL(triggered()), this, SLOT(resizeArea()));
    paletteAction = new QAction(tr("&Palette"), this);
    paletteAction->setStatusTip(tr("Show/Hide palette"));
    paletteAction->setCheckable(true);
    paletteAction->setChecked(true);
    connect(paletteAction, SIGNAL(toggled(bool)), this, SLOT(viewPalette()));
    setUndoSizeAction = new QAction(tr("&Undo size"), this);
    setUndoSizeAction->setStatusTip(tr("Set Undo/Redo size"));
    connect(setUndoSizeAction, SIGNAL(triggered()), this, SLOT(resizeStored()));
    cutAction = new QAction(tr("Cut"), this);
    cutAction->setStatusTip(tr("Cut"));
    cutAction->setShortcut(QKeySequence::Cut);
    cutAction->setIcon(QPixmap(":/edit-menu/icons/edit/cut.png"));
    connect(cutAction, SIGNAL(triggered()), this, SLOT(cutImage()));
    copyAction = new QAction(tr("Copy"), this);
    copyAction->setStatusTip(tr("Copy"));
    copyAction->setShortcut(QKeySequence::Copy);
    copyAction->setIcon(QPixmap(":/edit-menu/icons/edit/copy.png"));
    connect(copyAction, SIGNAL(triggered()), this, SLOT(copyImage()));
    pasteAction = new QAction(tr("Paste"), this);
    pasteAction->setStatusTip(tr("Paste"));
    pasteAction->setShortcut(QKeySequence::Paste);
    pasteAction->setIcon(QPixmap(":/edit-menu/icons/edit/paste.png"));
    connect(pasteAction, SIGNAL(triggered()), this, SLOT(pasteImage()));
}

/*!
 \brief метод для включения\выключения виджета палитра

*/
void MainWindow::viewPalette()
{
    if(paletteAction->isChecked())
        paletteBar->show();
    else
        paletteBar->hide();
}

/*!
 \brief загрузка меню

*/
void MainWindow::loadMenu()
{
    fileMenu = menuBar()->addMenu(tr("&File"));
    fileMenu->addAction(newAction);
    fileMenu->addAction(openAction);
    fileMenu->addAction(saveAction);
    fileMenu->addAction(saveAsAction);
    fileMenu->addAction(closeAction);
    fileMenu->addSeparator();
    fileMenu->addAction(printAction);
    fileMenu->addSeparator();
    fileMenu->addAction(exitAction);
    editMenu = menuBar()->addMenu(tr("&Edit"));
    editMenu->addAction(undoAction);
    editMenu->addAction(redoAction);
    editMenu->addSeparator();
    editMenu->addAction(copyAction);
    editMenu->addAction(pasteAction);
    editMenu->addAction(cutAction);
    filterMenu = menuBar()->addMenu(tr("&Filtres"));
    filterMenu->addAction(toBlackAction);
    filterMenu->addAction(toNegative);
    filterMenu->addSeparator();
    filterMenu->addAction(resizeImageAction);
    filterMenu->addAction(resizeAreaAction);
    filterMenu->addAction(rotateAction);
    settingsMenu = menuBar()->addMenu(tr("&Settings"));
    settingsMenu->addAction(paletteAction);
    settingsMenu->addAction(setUndoSizeAction);
    aboutMenu = menuBar()->addMenu(tr("&About"));
    aboutMenu->addAction(aboutAction);
    aboutMenu->addAction(aboutQtAction);
}

/*!
 \brief загрузка статус-бара

*/
void MainWindow::loadStatBar()
{
    statBar = new QStatusBar();
    zoomSlider = new QSlider(Qt::Horizontal);
    zoomSlider->setStatusTip("Zoom");
    zoomSlider->setFocusPolicy(Qt::ClickFocus);
    zoomSlider->setTickPosition(QSlider::NoTicks);
    zoomSlider->setFixedWidth(140);
    zoomSlider->setMinimum(0);
    zoomSlider->setMaximum(10);
    zoomSlider->setSingleStep(1);
    zoomSlider->setSliderPosition(0);
    statBar->addPermanentWidget(zoomSlider);
    setStatusBar(statBar);
}

/*!
 \brief загрузка тул-баров

*/
void MainWindow::loadToolBars()
{
    buttonCursor = new QToolButton();
    buttonCursor->setMinimumSize(QSize(25,25));
    buttonCursor->setMaximumSize(QSize(25,25));
    buttonCursor->setIcon(QPixmap(":/gadgets/icons/gadgets/cursor.png"));
    buttonCursor->setStatusTip(tr("Cursor"));
    buttonCursor->setCheckable(true);
    connect(buttonCursor, SIGNAL(toggled(bool)), this, SLOT(setCursor(bool)));
    buttonLastic = new QToolButton();
    buttonLastic->setMinimumSize(QSize(25,25));
    buttonLastic->setMaximumSize(QSize(25,25));
    buttonLastic->setIcon(QPixmap(":/gadgets/icons/gadgets/lastic.png"));
    buttonLastic->setStatusTip(tr("Lastic"));
    buttonLastic->setCheckable(true);
    connect(buttonLastic, SIGNAL(toggled(bool)), this, SLOT(setLastic(bool)));
    buttonPipet = new QToolButton();
    buttonPipet->setMinimumSize(QSize(25,25));
    buttonPipet->setMaximumSize(QSize(25,25));
    buttonPipet->setIcon(QPixmap(":/gadgets/icons/gadgets/pipette.png"));
    buttonPipet->setStatusTip(tr("Pipette"));
    buttonPipet->setCheckable(true);
    connect(buttonPipet, SIGNAL(toggled(bool)), this, SLOT(setPipet(bool)));
    buttonPen = new QToolButton();
    buttonPen->setMinimumSize(QSize(25,25));
    buttonPen->setMaximumSize(QSize(25,25));
    buttonPen->setText(tr("P"));
    buttonPen->setIcon(QPixmap(":/gadgets/icons/gadgets/pencil.png"));
    buttonPen->setStatusTip(tr("Pen"));
    buttonPen->setCheckable(true);
    connect(buttonPen, SIGNAL(toggled(bool)), this, SLOT(setPen(bool)));
    buttonLine = new QToolButton();
    buttonLine->setMinimumSize(QSize(25,25));
    buttonLine->setMaximumSize(QSize(25,25));
    buttonLine->setIcon(QPixmap(":/gadgets/icons/gadgets/line.png"));
    buttonLine->setStatusTip(tr("Line"));
    buttonLine->setCheckable(true);
    connect(buttonLine, SIGNAL(toggled(bool)), this, SLOT(setLine(bool)));
    buttonRectangle = new QToolButton();
    buttonRectangle->setMinimumSize(QSize(25,25));
    buttonRectangle->setMaximumSize(QSize(25,25));
    buttonRectangle->setIcon(QPixmap(":/gadgets/icons/gadgets/rectangle.png"));
    buttonRectangle->setStatusTip(tr("Rectangle"));
    buttonRectangle->setCheckable(true);
    connect(buttonRectangle, SIGNAL(toggled(bool)), this, SLOT(setRectangle(bool)));
    buttonEllipse = new QToolButton();
    buttonEllipse->setMinimumSize(QSize(25,25));
    buttonEllipse->setMaximumSize(QSize(25,25));
    buttonEllipse->setIcon(QPixmap(":/gadgets/icons/gadgets/ellipse.png"));
    buttonEllipse->setStatusTip(tr("Ellipse"));
    buttonEllipse->setCheckable(true);
    connect(buttonEllipse, SIGNAL(toggled(bool)), this, SLOT(setElliple(bool)));
    buttonLoupe = new QToolButton();
    buttonLoupe->setMinimumSize(QSize(25,25));
    buttonLoupe->setMaximumSize(QSize(25,25));
    buttonLoupe->setIcon(QPixmap(":/gadgets/icons/gadgets/loupe.png"));
    buttonLoupe->setStatusTip(tr("Loupe"));
    buttonLoupe->setCheckable(true);
    connect(buttonLoupe, SIGNAL(toggled(bool)), this, SLOT(setLoupe(bool)));
    buttonSpray = new QToolButton();
    buttonSpray->setMinimumSize(QSize(25,25));
    buttonSpray->setMaximumSize(QSize(25,25));
    buttonSpray->setIcon(QPixmap(":/gadgets/icons/gadgets/spray.png"));
    buttonSpray->setStatusTip(tr("Spray"));
    buttonSpray->setCheckable(true);
    connect(buttonSpray, SIGNAL(toggled(bool)), this, SLOT(setSpray(bool)));
    buttonFill = new QToolButton();
    buttonFill->setMinimumSize(QSize(25,25));
    buttonFill->setMaximumSize(QSize(25,25));
    buttonFill->setIcon(QPixmap(":/gadgets/icons/gadgets/fill.png"));
    buttonFill->setStatusTip(tr("Fill"));
    buttonFill->setCheckable(true);
    connect(buttonFill, SIGNAL(toggled(bool)), this, SLOT(setFill(bool)));
    buttonHand = new QToolButton();
    buttonHand->setMinimumSize(QSize(25,25));
    buttonHand->setMaximumSize(QSize(25,25));
    buttonHand->setIcon(QPixmap(":/gadgets/icons/gadgets/hand.png"));
    buttonHand->setStatusTip(tr("Hand"));
    buttonHand->setCheckable(true);
    connect(buttonHand, SIGNAL(toggled(bool)), this, SLOT(setHand(bool)));
    buttonSelect = new QToolButton();
    buttonSelect->setMinimumSize(QSize(25,25));
    buttonSelect->setMaximumSize(QSize(25,25));
    buttonSelect->setIcon(QPixmap(":/gadgets/icons/gadgets/select.png"));
    buttonSelect->setStatusTip(tr("Select"));
    buttonSelect->setCheckable(true);
    connect(buttonSelect, SIGNAL(toggled(bool)), this, SLOT(setSelect(bool)));

    gadgetsLayout = new QGridLayout();
    gadgetsLayout->setMargin(3);
    gadgetsLayout->addWidget(buttonCursor, 0, 0);
    gadgetsLayout->addWidget(buttonLastic, 0, 1);
    gadgetsLayout->addWidget(buttonPipet, 1, 0);
    gadgetsLayout->addWidget(buttonLoupe, 1, 1);
    gadgetsLayout->addWidget(buttonPen, 2, 0);
    gadgetsLayout->addWidget(buttonLine, 2, 1);
    gadgetsLayout->addWidget(buttonSpray, 3, 0);
    gadgetsLayout->addWidget(buttonFill, 3, 1);
    gadgetsLayout->addWidget(buttonRectangle, 4, 0);
    gadgetsLayout->addWidget(buttonEllipse, 4, 1);
    gadgetsLayout->addWidget(buttonHand, 5, 0);
    gadgetsLayout->addWidget(buttonSelect, 5, 1);

    gadgetsSetsLayout = new QGridLayout();
    gadgetsSetsLayout->setMargin(3);
    gadgetsSetsLayout->addWidget(colorSet, 0, 0);
    gadgetsSetsLayout->addWidget(brushSet, 0, 1);
    gadgetsSetsLayout->addWidget(spinTricknessSize, 1, 0, 1, 2);

    gadgetsWidget = new QWidget();
    gadgetsWidget->setLayout(gadgetsLayout);
    gadgetsSetsWidget = new QWidget();
    gadgetsSetsWidget->setLayout(gadgetsSetsLayout);

    gadgetsBar = new QToolBar("&Gadgets", this);
    addToolBar(Qt::LeftToolBarArea, gadgetsBar);
    gadgetsBar->setMovable(false);
    gadgetsBar->addWidget(gadgetsWidget);
    gadgetsBar->addSeparator();
    gadgetsBar->addWidget(gadgetsSetsWidget);

    paletteBar = new TPalleteBar();//"&Palette", this);
    addToolBar(Qt::LeftToolBarArea, paletteBar);

    //    addToolBar(Qt::NoToolBarArea, paletteBar);
//    paletteBar->
}

/*!
 \brief инициализация сигналов

*/
void MainWindow::loadSignals()
{
    connect(colorSet, SIGNAL(sendColor(QColor)), this, SLOT(setPenColor(QColor)));
    connect(paletteBar, SIGNAL(sendFirstColor(QColor)), this, SLOT(setPenColor(QColor)));
    connect(paletteBar, SIGNAL(sendFirstColor(QColor)), colorSet, SLOT(setColor(QColor)));
    connect(brushSet, SIGNAL(sendColor(QColor)), this, SLOT(setBrush(QColor)));
    connect(paletteBar, SIGNAL(sendSecondColor(QColor)), this, SLOT(setBrush(QColor)));
    connect(paletteBar, SIGNAL(sendSecondColor(QColor)), brushSet, SLOT(setColor(QColor)));
    connect(spinTricknessSize, SIGNAL(valueChanged(int)), this, SLOT(setPenTrickness(int)));
    connect(tabBar, SIGNAL(tabCloseRequested(int)), this, SLOT(closeFile(int)));
    connect(tabBar, SIGNAL(currentChanged(int)), this, SLOT(activateTab(int)));
    connect(zoomSlider, SIGNAL(valueChanged(int)), this, SLOT(setZoom(int)));
}

/*!
 \brief метод для изменения положения ползунков при использовании лупы

 \param x координата
 \param y координата
*/
void MainWindow::setScrollBarsPos(int x, int y)
{
    QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
    tempArea->horizontalScrollBar()->setSliderPosition(x);
    tempArea->verticalScrollBar()->setSliderPosition(y);
}

/*!
 \brief слот для обработки инициализации таба

 \param index номер таба
*/
void MainWindow::activateTab(int index)
{
    if(index >= 0)
    {
        tabBar->setCurrentIndex(index);
        QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
        CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
        tempView->setPenColor(penColor);
        tempView->setBrushColor(brushColor);
        tempView->setPenTrickness(penSize);
        tempView->setActiveTool(activeTool);
        tempView->setURButtons();
        zoomSlider->setValue(tempView->zoomFactor);
        if(activeTool == LOUPE)
            zoomSlider->setEnabled(false);
        else
            zoomSlider->setEnabled((true));
        if(tempView->zoomFactor > 0)
            buttonLoupe->setEnabled(false);
        else
            buttonLoupe->setEnabled(true);
    }
}

/*!
 \brief включения\выключения всех кнопок и экшенов

 \param flag индикатор ключения\отключения
*/
void MainWindow::allSetEnabled(bool flag)
{
    saveAction->setEnabled(flag);
    saveAsAction->setEnabled(flag);
    printAction->setEnabled(flag);
    closeAction->setEnabled(flag);
    buttonCursor->setEnabled(flag);
    buttonLastic->setEnabled(flag);
    buttonPen->setEnabled(flag);
    buttonLine->setEnabled(flag);
    buttonRectangle->setEnabled(flag);
    buttonEllipse->setEnabled(flag);
    buttonPipet->setEnabled(flag);
    colorSet->setEnabled(flag);
    brushSet->setEnabled(flag);
    spinTricknessSize->setEnabled(flag);
    undoAction->setEnabled(flag);
    redoAction->setEnabled(flag);
    toBlackAction->setEnabled(flag);
    toNegative->setEnabled(flag);
    resizeImageAction->setEnabled(flag);
    resizeAreaAction->setEnabled(flag);
    rotateAction->setEnabled(flag);
    paletteAction->setEnabled(flag);
    if(paletteAction->isChecked())
        paletteBar->setEnabled(flag);
    setUndoSizeAction->setEnabled(flag);
    copyAction->setEnabled(flag);
    cutAction->setEnabled(flag);
    pasteAction->setEnabled(flag);
    buttonFill->setEnabled(flag);
    buttonSpray->setEnabled(flag);
    buttonLoupe->setEnabled(flag);
    buttonSelect->setEnabled(flag);
    buttonHand->setEnabled(flag);
    zoomSlider->setEnabled(flag);
}

/*!
 \brief метод реализующий открытие файла

*/
void MainWindow::openFile()
{
    QFileDialog::Options options;
    QString filter;
    options |= QFileDialog::DontUseNativeDialog;
    QString fileName = QFileDialog::getOpenFileName(this, tr("Open File"),"",
                                                    tr("All supported (*.png *.bmp *.jpg *.gif *.jpeg *.pbm *.pgm *.ppm *.xbm *.xpm);; All (*.*)"),
                                                    &filter,
                                                    options);

    if(!fileName.isEmpty())
        open(fileName);
}

/*!
 \brief метод реализующий сохранение файла с заданием имени

*/
void MainWindow::saveAsFile()
{
    QFileDialog::Options options;
    QString filter;
    options |= QFileDialog::DontUseNativeDialog;
    QString fileName = QFileDialog::getSaveFileName(this, tr("Save File"),
                                                    "",
                                                    tr("*.png;;*.jpg;;*.jpeg;;*.bmp;;*.xbm;;*.xpm"),
                                                    &filter,
                                                    options);
    if(!fileName.isEmpty())
    {
        QApplication::setOverrideCursor(Qt::WaitCursor);
        int length(filter.size());
        for(int i(length - 1); i > 0; i--)
        {
            if(filter[i] != fileName[fileName.size() - length + i])
            {
                fileName += filter.mid(1);
                break;
            }
        }
        QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
        CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
        tempView->setSaveImage(fileName);
        fileName = tempView->setFilePath(fileName);
        tabBar->setTabText(tabBar->currentIndex(), fileName);
        statBar->showMessage(tr("Save completed"), 2000);
        QApplication::restoreOverrideCursor();
    }
}

/*!
 \brief метод реализующий сохранение файла

*/
void MainWindow::saveFile()
{
    QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
    CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
    if(tempView->getFilePath().isEmpty())
        saveAsFile();
    else
    {
        QString fileName = tempView->getFilePath();
        QString format = fileName.right(fileName.size()-1-fileName.indexOf("."));
        if(format=="gif" || format=="pbm" || format=="pgm" )
            saveAsFile();
        else
        {
            QApplication::setOverrideCursor(Qt::WaitCursor);
            tempView->setSaveImage(tempView->getFilePath());
            statBar->showMessage(tr("Save completed"), 2000);
            QApplication::restoreOverrideCursor();
        }
    }
}

/*!
 \brief метод для закрытия таба

*/
void MainWindow::sendToCloseFile()
{
    closeFile(tabBar->currentIndex());
}

/*!
 \brief сигнал для закрытия таба и виджета на нем

 \param index номер таба
*/
void MainWindow::closeFile(int index)
{
    if(toContinue(index))
    {
        tabBar->widget(index)->close();
        tabBar->removeTab(index);
        if(tabBar->count() == 0)
            allSetEnabled(false);
    }
}

/*!
 \brief инициализация формы для задания новых размеров

*/
void MainWindow::createNew()
{
    newImWindow = new FNewFile(this);
    connect(newImWindow, SIGNAL(newImage(int, int)), this, SLOT(setNew(int, int)));
    newImWindow->show();
    newImWindow->raise();
    newImWindow->activateWindow();
}

/*!
 \brief метод для создания таба с новым изображением заданных размеров

 \param width ширина
 \param height высота
*/
void MainWindow::setNew(int width, int height)
{
    QApplication::setOverrideCursor(Qt::WaitCursor);
    QScrollArea *imageArea = new QScrollArea();
    imageArea->setAttribute(Qt::WA_DeleteOnClose);
    imageArea->setWidget(new CImageCV(width, height));
    imageArea->setBackgroundRole(QPalette::Dark);
    tabBar->addTab(imageArea, "new");
    tabBar->setCurrentIndex(tabBar->count()-1);
    CImageCV *tempView = qobject_cast<CImageCV*>(imageArea->widget());
    tempView->setPenColor(penColor);
    tempView->setBrushColor(brushColor);
    tempView->setPenTrickness(penSize);
    tempView->setActiveTool(activeTool);
    connect(tempView, SIGNAL(noActiveTool()), this, SLOT(allButtonUncheked()));
    connect(tempView, SIGNAL(sendScrolBarsPos(int,int)), this, SLOT(setScrollBarsPos(int,int)));
    connect(tempView, SIGNAL(sendPenColor(QColor)), colorSet, SLOT(setColor(QColor)));
    connect(tempView, SIGNAL(sendBrushColor(QColor)), brushSet, SLOT(setColor(QColor)));
    connect(tempView, SIGNAL(sendPenColor(QColor)), this, SLOT(setPenColor(QColor)));
    connect(tempView, SIGNAL(sendBrushColor(QColor)), this, SLOT(setBrush(QColor)));
    connect(tempView, SIGNAL(sendWait(bool)), this , SLOT(setWait(bool)));
    connect(tempView, SIGNAL(sendURButton(bool,bool)), this, SLOT(setURButtons(bool,bool)));
    statBar->showMessage(tr("Image created"), 2000);
    if(tabBar->count() != 0)
        allSetEnabled(true);
    tempView->setURButtons();
    QApplication::restoreOverrideCursor();
}

/*!
 \brief метод вызывающий форма для ресайза изображения

*/
void MainWindow::resizeImage()
{
    QApplication::setOverrideCursor(Qt::WaitCursor);
    QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
    CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
    resizeImWindow = new FResizeImage(tempView->width(), tempView->height(), this);
    connect(resizeImWindow, SIGNAL(newSize(int, int)), this, SLOT(setNewSize(int, int)));
    resizeImWindow->show();
    resizeImWindow->raise();
    resizeImWindow->activateWindow();
    QApplication::restoreOverrideCursor();
}

/*!
 \brief метод для установки полученного из resizeImage размера

 \param width ширина
 \param height высота
*/
void MainWindow::setNewSize(int width, int height)
{
    QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
    CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
    tempView->setResizeImage(width, height);
}

/*!
 \brief метод вызывающий форму для ресайза зоны

*/
void MainWindow::resizeArea()
{
    QApplication::setOverrideCursor(Qt::WaitCursor);
    QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
    CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
    resizeImWindow = new FResizeImage(tempView->width(), tempView->height(), this);
    connect(resizeImWindow, SIGNAL(newSize(int, int)), this, SLOT(setNewAreaSize(int, int)));
    resizeImWindow->show();
    resizeImWindow->raise();
    resizeImWindow->activateWindow();
    QApplication::restoreOverrideCursor();
}

/*!
 \brief метод для установки полученного из resizeArea размера

 \param width ширина
 \param height высота
*/
void MainWindow::setNewAreaSize(int width, int height)
{
    QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
    CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
    tempView->setResizeArea(width, height);
}

/*!
 \brief метод вызывающий форму для ресайза истории

*/
void MainWindow::resizeStored()
{
    QApplication::setOverrideCursor(Qt::WaitCursor);
    QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
    CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
    resizeStoredWindow = new FStoredSet(tempView->getStored(), this);
    connect(resizeStoredWindow, SIGNAL(newSize(int)), this, SLOT(setNewStoreSize(int)));
    resizeStoredWindow->show();
    resizeStoredWindow->raise();
    resizeStoredWindow->activateWindow();
    QApplication::restoreOverrideCursor();
}

/*!
 \brief метод для установки нового размера истории

 \param size новый размер
*/
void MainWindow::setNewStoreSize(int size)
{
    QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
    CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
    tempView->setStored(size);
}

/*!
 \brief метод для проверки надобности созранить изображение перед его закрытием

 \param index номер таба
 \return bool флаг для индикации выполненого действия
*/
bool MainWindow::toContinue(int index)
{
    QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->widget(index));
    CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
    if(tempView->getModifiedFlag())
    {
        int flag = QMessageBox::warning(this, tr("easyPaint"),
                                        tr("The image has been modified.\n Do you want to save your changes?"),
                                        QMessageBox::Yes | QMessageBox::No | QMessageBox::Cancel);
        if(flag == QMessageBox::Yes)
        {
            saveFile();
            return true;
        }
        else
        {
            if(flag == QMessageBox::No)
            {
                tempView->setModifiedFlag(false);
                return true;
            }
            else
            {
                if(flag == QMessageBox::Cancel)
                    return false;
            }
        }
    }
    return true;
}

/*!
 \brief перегрузка closeEvent, позволяет проверить перед закрытием программы
        надобность сохранить изображение

 \param event эвент для обработки
*/
void MainWindow::closeEvent(QCloseEvent *event)
{
    for(int i(0); i < tabBar->count(); i++)
    {
        if(toContinue(i))
        {
//            closeFile(i);
            event->accept();
        }
        else
        {
            event->ignore();
            break;
        }
    }
}

/*!
 \brief описание программы

*/
void MainWindow::about()
{
    QMessageBox::about(this, tr("About easyPaint"),
            tr("<h2>easyPaint 0.6.0</h2>"
               "<p><a href=\"mailto:grin.minsk@gmail.com"
               "?subject:easyPaint project\">"
               "grin.minsk@gmail.com</a>"
               "<p><a href=\"mailto:leshka.popzov@gmail.com"
               "?subject:easyPaint project\">"
               "leshka.popzov@gmail.com</a>"));
}

/*!
 \brief метод отправляющий новый размер пера

 \param size размер пера
*/
void MainWindow::setPenTrickness(int size)
{
    penSize = size;
    QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
    CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
    tempView->setPenTrickness(size);
}

/*!
 \brief метод откравляющий новый цвет пера

 \param color цвет пера
*/
void MainWindow::setPenColor(const QColor &color)
{
    penColor = color;
    QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
    CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
    tempView->setPenColor(color);
}

/*!
 \brief метод отправляющий новый цвет заливки

 \param color цвет заливки
*/
void MainWindow::setBrush(const QColor &color)
{
    brushColor = color;
    QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
    CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
    tempView->setBrushColor(color);
}

/*!
 \brief метод вызывающий отмену действий

*/
void MainWindow::setUndo()
{
    QApplication::setOverrideCursor(Qt::WaitCursor);
    QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
    CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
    tempView->setPreviousAction();
    QApplication::restoreOverrideCursor();
}

/*!
 \brief метод вызывающий повтор действий

*/
void MainWindow::setRedo()
{
    QApplication::setOverrideCursor(Qt::WaitCursor);
    QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
    CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
    tempView->setNextAction();
    QApplication::restoreOverrideCursor();
}

/*!
 \brief метод вызывающий установку черно-белого эффекта

*/
void MainWindow::setBlackWhite()
{
    QApplication::setOverrideCursor(Qt::WaitCursor);
    QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
    CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
    tempView->setBlackWhite();
    QApplication::restoreOverrideCursor();
}

/*!
 \brief метод вызывающий утсановку эффекта негатива

*/
void MainWindow::setNegative()
{
    QApplication::setOverrideCursor(Qt::WaitCursor);
    QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
    CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
    tempView->setNegative();
    QApplication::restoreOverrideCursor();
}

/*!
 \brief метод вызывающий поворот изображения

*/
void MainWindow::setRotate()
{
    QApplication::setOverrideCursor(Qt::WaitCursor);
    QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
    CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
    tempView->setRotate();
    QApplication::restoreOverrideCursor();
}

/*!
 \brief установка инструмента лупа

 \param state флаг отжатия/зажатия
*/
void MainWindow::setLoupe(bool state)
{
    QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
    CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
    if(state)
    {
        buttonPen->setChecked(false);
        buttonLine->setChecked(false);
        buttonRectangle->setChecked(false);
        buttonEllipse->setChecked(false);
        buttonCursor->setChecked(false);
        buttonLastic->setChecked(false);
        buttonFill->setChecked(false);
        buttonSpray->setChecked(false);
        buttonPipet->setChecked(false);
        buttonHand->setChecked(false);
        buttonSelect->setChecked(false);
        buttonLoupe->setChecked(true);
        tempView->setActiveTool(LOUPE);
        activeTool = LOUPE;
        zoomSlider->setDisabled(true);
    }
    else
    {
        tempView->setActiveTool(NONE);
        activeTool = NONE;
        buttonLoupe->setChecked(false);
        zoomSlider->setDisabled(false);
    }
}

/*!
 \brief установка инструмента спрей

 \param state флаг отжатия/зажатия
*/
void MainWindow::setSpray(bool state)
{
    QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
    CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
    if(state)
    {
        buttonPen->setChecked(false);
        buttonLine->setChecked(false);
        buttonRectangle->setChecked(false);
        buttonEllipse->setChecked(false);
        buttonCursor->setChecked(false);
        buttonLastic->setChecked(false);
        buttonFill->setChecked(false);
        buttonLoupe->setChecked(false);
        buttonPipet->setChecked(false);
        buttonHand->setChecked(false);
        buttonSelect->setChecked(false);
        buttonSpray->setChecked(true);
        tempView->setActiveTool(SPRAY);
        activeTool = SPRAY;
    }
    else
    {
        tempView->setActiveTool(NONE);
        activeTool = NONE;
        buttonSpray->setChecked(false);
    }
}

/*!
 \brief установка инструмента заливка

 \param state флаг отжатия/зажатия
*/
void MainWindow::setFill(bool state)
{
    QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
    CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
    if(state)
    {
        buttonPen->setChecked(false);
        buttonLine->setChecked(false);
        buttonRectangle->setChecked(false);
        buttonEllipse->setChecked(false);
        buttonCursor->setChecked(false);
        buttonLastic->setChecked(false);
        buttonSpray->setChecked(false);
        buttonLoupe->setChecked(false);
        buttonPipet->setChecked(false);
        buttonHand->setChecked(false);
        buttonSelect->setChecked(false);
        buttonFill->setChecked(true);
        tempView->setActiveTool(FILL);
        activeTool = FILL;
    }
    else
    {
        tempView->setActiveTool(NONE);
        activeTool = NONE;
        buttonFill->setChecked(false);
    }
}

/*!
 \brief установка инструмента ластик

 \param state флаг отжатия/зажатия
*/
void MainWindow::setLastic(bool state)
{
    QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
    CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
    if(state)
    {
        buttonPen->setChecked(false);
        buttonLine->setChecked(false);
        buttonRectangle->setChecked(false);
        buttonEllipse->setChecked(false);
        buttonCursor->setChecked(false);
        buttonFill->setChecked(false);
        buttonSpray->setChecked(false);
        buttonLoupe->setChecked(false);
        buttonPipet->setChecked(false);
        buttonHand->setChecked(false);
        buttonSelect->setChecked(false);
        buttonLastic->setChecked(true);
        tempView->setActiveTool(LASTIC);
        activeTool = LASTIC;

    }
    else
    {
        tempView->setActiveTool(NONE);
        activeTool = NONE;
        buttonLastic->setChecked(false);
    }
}

/*!
 \brief установка инструмента курсор

 \param state флаг отжатия/зажатия
*/
void MainWindow::setCursor(bool state)
{
    QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
    CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
    if(state)
    {
        buttonPen->setChecked(false);
        buttonLine->setChecked(false);
        buttonRectangle->setChecked(false);
        buttonEllipse->setChecked(false);
        buttonLastic->setChecked(false);
        buttonPipet->setChecked(false);
        buttonFill->setChecked(false);
        buttonSpray->setChecked(false);
        buttonLoupe->setChecked(false);
        buttonHand->setChecked(false);
        buttonSelect->setChecked(false);
        buttonCursor->setChecked(true);
        tempView->setActiveTool(NONE);
        activeTool = NONE;
    }
    else
    {
        tempView->setActiveTool(NONE);
        activeTool = NONE;
        buttonCursor->setChecked(false);
    }
}

/*!
 \brief установка инструмента карандаш

 \param state флаг отжатия/зажатия
*/
void MainWindow::setPen(bool state)
{
    QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
    CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
    if(state)
    {
        buttonLine->setChecked(false);
        buttonRectangle->setChecked(false);
        buttonEllipse->setChecked(false);
        buttonCursor->setChecked(false);
        buttonLastic->setChecked(false);
        buttonPipet->setChecked(false);
        buttonFill->setChecked(false);
        buttonSpray->setChecked(false);
        buttonLoupe->setChecked(false);
        buttonHand->setChecked(false);
        buttonSelect->setChecked(false);
        buttonPen->setChecked(true);
        tempView->setActiveTool(PEN);
        activeTool = PEN;

    }
    else
    {
        tempView->setActiveTool(NONE);
        activeTool = NONE;
        buttonPen->setChecked(false);
    }
}

/*!
 \brief установка инструмента линия

 \param state флаг отжатия/зажатия
*/
void MainWindow::setLine(bool state)
{
    QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
    CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
    if(state)
    {
        buttonPen->setChecked(false);
        buttonRectangle->setChecked(false);
        buttonEllipse->setChecked(false);
        buttonCursor->setChecked(false);
        buttonLastic->setChecked(false);
        buttonPipet->setChecked(false);
        buttonFill->setChecked(false);
        buttonSpray->setChecked(false);
        buttonLoupe->setChecked(false);
        buttonHand->setChecked(false);
        buttonSelect->setChecked(false);
        buttonLine->setChecked(true);
        tempView->setActiveTool(LINE);
        activeTool = LINE;
    }
    else
    {
        tempView->setActiveTool(NONE);
        activeTool = NONE;
        buttonLine->setChecked(false);
    }
}

/*!
 \brief установка инструмента прямоугольник

 \param state флаг отжатия/зажатия
*/
void MainWindow::setRectangle(bool state)
{
    QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
    CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
    if(state)
    {
        buttonPen->setChecked(false);
        buttonLine->setChecked(false);
        buttonEllipse->setChecked(false);
        buttonCursor->setChecked(false);
        buttonLastic->setChecked(false);
        buttonPipet->setChecked(false);
        buttonFill->setChecked(false);
        buttonSpray->setChecked(false);
        buttonLoupe->setChecked(false);
        buttonHand->setChecked(false);
        buttonSelect->setChecked(false);
        buttonRectangle->setChecked(true);
        tempView->setActiveTool(RECTANGLE);
        activeTool = RECTANGLE;
    }
    else
    {
        tempView->setActiveTool(NONE);
        activeTool = NONE;
        buttonRectangle->setChecked(false);
    }
}

/*!
 \brief установка инструмента эллипс

 \param state флаг отжатия/зажатия
*/
void MainWindow::setElliple(bool state)
{
    QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
    CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
    if(state)
    {
        buttonPen->setChecked(false);
        buttonLine->setChecked(false);
        buttonRectangle->setChecked(false);
        buttonCursor->setChecked(false);
        buttonLastic->setChecked(false);
        buttonPipet->setChecked(false);
        buttonFill->setChecked(false);
        buttonSpray->setChecked(false);
        buttonLoupe->setChecked(false);
        buttonHand->setChecked(false);
        buttonSelect->setChecked(false);
        buttonEllipse->setChecked(true);
        tempView->setActiveTool(ELLIPSE);
        activeTool = ELLIPSE;
    }
    else
    {
        tempView->setActiveTool(NONE);
        activeTool = NONE;
        buttonEllipse->setChecked(false);
    }
}

/*!
 \brief установка инструмента пипетка

 \param state флаг отжатия/зажатия
*/
void MainWindow::setPipet(bool state)
{
    QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
    CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
    if(state)
    {
        buttonEllipse->setChecked(false);
        buttonPen->setChecked(false);
        buttonLine->setChecked(false);
        buttonRectangle->setChecked(false);
        buttonCursor->setChecked(false);
        buttonLastic->setChecked(false);
        buttonFill->setChecked(false);
        buttonSpray->setChecked(false);
        buttonLoupe->setChecked(false);
        buttonHand->setChecked(false);
        buttonSelect->setChecked(false);
        buttonPipet->setChecked(true);
        tempView->setActiveTool(PIPET);
        activeTool = PIPET;
    }
    else
    {
        tempView->setActiveTool(NONE);
        activeTool = NONE;
        buttonPipet->setChecked(false);
    }
}


/*!
 \brief установка инструмента рука

 \param state флаг отжатия/зажатия
*/
void MainWindow::setHand(bool state)
{
    QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
    CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
    if(state)
    {
        buttonEllipse->setChecked(false);
        buttonPen->setChecked(false);
        buttonLine->setChecked(false);
        buttonRectangle->setChecked(false);
        buttonCursor->setChecked(false);
        buttonLastic->setChecked(false);
        buttonFill->setChecked(false);
        buttonSpray->setChecked(false);
        buttonLoupe->setChecked(false);
        buttonPipet->setChecked(false);
        buttonSelect->setChecked(false);
        buttonHand->setChecked(true);
        tempView->setActiveTool(HAND);
        activeTool = HAND;
    }
    else
    {
        tempView->setActiveTool(NONE);
        activeTool = NONE;
        buttonHand->setChecked(false);
    }
}

/*!
 \brief установка инструмена выделение

 \param state флаг отжатия/зажатия
*/
void MainWindow::setSelect(bool state)
{
    QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
    CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
    if(state)
    {
        buttonEllipse->setChecked(false);
        buttonPen->setChecked(false);
        buttonLine->setChecked(false);
        buttonRectangle->setChecked(false);
        buttonCursor->setChecked(false);
        buttonLastic->setChecked(false);
        buttonFill->setChecked(false);
        buttonSpray->setChecked(false);
        buttonLoupe->setChecked(false);
        buttonPipet->setChecked(false);
        buttonHand->setChecked(false);
        buttonSelect->setChecked(true);
        tempView->setActiveTool(SELECT);
        activeTool = SELECT;
    }
    else
    {
        tempView->setActiveTool(NONE);
        activeTool = NONE;
        buttonSelect->setChecked(false);
    }
}


/*!
 \brief обработка слайдера для ресайза изображения

 \param factor величина ресайза
*/
void MainWindow::setZoom(int factor)
{
    QApplication::setOverrideCursor(Qt::WaitCursor);
    if(factor > 0)
        buttonLoupe->setEnabled(false);
    else
        buttonLoupe->setEnabled(true);
    QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
    CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
    tempView->setZoom(factor);
    QApplication::restoreOverrideCursor();
}

/*!
 \brief установка курсора в значение "ожидание"

 \param needTime флаг надобности установки
*/
void MainWindow::setWait(bool needTime)
{
    if(needTime)
        QApplication::setOverrideCursor(Qt::WaitCursor);
    else
        QApplication::restoreOverrideCursor();
}

/*!
 \brief перегрузка dropEvent

 \param event обрабатываемый эвент
*/
void MainWindow::dropEvent(QDropEvent *event)
{
    const QMimeData *mimeData = event->mimeData();

    if(mimeData->hasUrls() && checkUrl(mimeData))
    {
        event->acceptProposedAction();
        QUrl url = mimeData->urls().first();
        QString fileName;
#if defined(Q_OS_WIN32)
        fileName = url.path().right(url.path().size() - 1);
#else
        fileName = url.path();
#endif
        open(fileName);
    }
    else
        event->ignore();
}

/*!
 \brief перегрузка dragEnterEvent

 \param event обрабатываемый эвент
*/
void MainWindow::dragEnterEvent(QDragEnterEvent *event)
{
    const QMimeData *mimeData = event->mimeData();
    if(mimeData->hasUrls())
        event->acceptProposedAction();
    else
        event->ignore();
}


/*!
 \brief перегрузка dragMoveEvent

 \param event обрабатываемый эвент
*/
void MainWindow::dragMoveEvent(QDragMoveEvent *event)
{
    const QMimeData *mimeData = event->mimeData();
    if(mimeData->hasUrls() && checkUrl(mimeData))
        event->acceptProposedAction();
    else
        event->ignore();
}

/*!
 \brief перегрузка dragLeaveEvent

 \param event обрабатываемы эвент
*/
void MainWindow::dragLeaveEvent(QDragLeaveEvent *event)
{
    event->accept();
}

/*!
 \brief вспомогательный метод для drag&drop

 \param mimeData данные для обработки на совпадение
 \return bool флаг для обозначения совпадения
*/
bool MainWindow::checkUrl(const QMimeData *mimeData)
{
    QUrl url = mimeData->urls().first();
    QString fileName,format;
    fileName = url.path();
    format = fileName.right(fileName.size() - 1 - fileName.indexOf("."));
    if(format == "png" || format == "bmp" || format == "jpg" ||
            format == "gif" || format == "jpeg" || format == "pbm" ||
            format == "pgm" || format == "ppm" || format == "xbm" || format == "xpm")
        return true;
    else
        return false;
}

/*!
 \brief вспомогательные метод для открытия изображения

 \param fileName имя изображения
*/
void MainWindow::open(QString fileName)
{
    QApplication::setOverrideCursor(Qt::WaitCursor);
    QScrollArea *imageArea = new QScrollArea();
    imageArea->setAttribute(Qt::WA_DeleteOnClose);
    imageArea->setWidget(new CImageCV());
    imageArea->setBackgroundRole(QPalette::Dark);
    CImageCV *tempView = qobject_cast<CImageCV*>(imageArea->widget());
    if(tempView->setLoadImage(fileName))
    {
        fileName = tempView->setFilePath(fileName);
        tempView->setPenColor(penColor);
        tempView->setBrushColor(brushColor);
        tempView->setPenTrickness(penSize);
        tempView->setActiveTool(activeTool);
        connect(tempView, SIGNAL(noActiveTool()), this, SLOT(allButtonUncheked()));
        connect(tempView, SIGNAL(sendScrolBarsPos(int,int)), this, SLOT(setScrollBarsPos(int,int)));
        connect(tempView, SIGNAL(sendPenColor(QColor)), colorSet, SLOT(setColor(QColor)));
        connect(tempView, SIGNAL(sendBrushColor(QColor)), brushSet, SLOT(setColor(QColor)));
        connect(tempView, SIGNAL(sendPenColor(QColor)), this, SLOT(setPenColor(QColor)));
        connect(tempView, SIGNAL(sendBrushColor(QColor)), this, SLOT(setBrush(QColor)));
        connect(tempView, SIGNAL(sendWait(bool)), this, SLOT(setWait(bool)));
        connect(tempView, SIGNAL(sendURButton(bool,bool)), this, SLOT(setURButtons(bool,bool)));
        tabBar->addTab(imageArea, fileName);
        tabBar->resize(tempView->size());
        tabBar->setCurrentIndex(tabBar->count() - 1);
        statBar->showMessage(tr("Image loaded"), 2000);
        if(tabBar->count() != 0)
            allSetEnabled(true);
    }
    else
    {
        QApplication::restoreOverrideCursor();
        statBar->showMessage(tr("Error when opening image"), 2000);
        QMessageBox::warning(this, tr("easyPaint"),
                             tr("Error when loading image"));
    }
    tempView->setURButtons();
    QApplication::restoreOverrideCursor();
}

/*!
 \brief слот устанавливающий включение\выключение undo\redo экшенов

 \param flagU undo флаг
 \param flagR redo флаг
*/
void MainWindow::setURButtons(bool flagU, bool flagR)
{
    undoAction->setEnabled(flagU);
    redoAction->setEnabled(flagR);
}

/*!
 \brief метод вызывающий печать изображения

*/
void MainWindow::printImage()
{
    QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
    CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
    tempView->printImage();
}

/*!
 \brief метод для вызова вырезания изображения

*/
void MainWindow::cutImage()
{
    QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
    CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
    QClipboard *clipboard = QApplication::clipboard();
    QImage tempImage;
    if(tempView->cut(tempImage))
        clipboard->setImage(tempImage);
}

/*!
 \brief метод для вызова копирования изображения

*/
void MainWindow::copyImage()
{
    QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
    CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
    QClipboard *clipboard = QApplication::clipboard();
    QImage tempImage;
    if(tempView->copy(tempImage))
        clipboard->setImage(tempImage);
}

/*!
 \brief метод для вызова вставки изображения

*/
void MainWindow::pasteImage()
{
    QClipboard *clipboard = QApplication::clipboard();
    if(clipboard->mimeData()->hasImage())
    {
        int flag = QMessageBox::question(this, tr("easyPaint"),
                                         tr("Do you want to pase this image in new tab?"),
                                         QMessageBox::Yes | QMessageBox::No | QMessageBox::Cancel);
        if(flag == QMessageBox::Yes)
        {
            QImage newImg = clipboard->image();
            this->setNew(newImg.width(), newImg.height());
            QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
            CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
            tempView->setImage(newImg);
        }
        if(flag == QMessageBox::No)
        {
            QScrollArea *tempArea = qobject_cast<QScrollArea*>(tabBar->currentWidget());
            CImageCV *tempView = qobject_cast<CImageCV*>(tempArea->widget());
            tempView->paste(clipboard->image());
        }
        if(flag == QMessageBox::Cancel)
            return;
    }
}

/*!
 \brief метод для unchek'а всех кнопок

*/
void MainWindow::allButtonUncheked()
{
    buttonEllipse->setChecked(false);
    buttonPen->setChecked(false);
    buttonLine->setChecked(false);
    buttonRectangle->setChecked(false);
    buttonCursor->setChecked(false);
    buttonLastic->setChecked(false);
    buttonFill->setChecked(false);
    buttonSpray->setChecked(false);
    buttonLoupe->setChecked(false);
    buttonPipet->setChecked(false);
    buttonHand->setChecked(false);
    buttonSelect->setChecked(false);
    activeTool = NONE;
}
