/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, I1.HR                           *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Feb 92                                                   *
*  Last Update : Aug 92                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : movement1.c                                              *
*                                                                         *
*  Function    : Operations for moving array sections of distr. arrays    *
*                                                                         *
*              T[...,t_low:t_up]  =  S[...,s_low:s_up]                    *
*                                                                         *
*  Export : FORTRAN Interface                                             *
*                                                                         *
*  void dalib_move_define__ (t_N, t_low, t_up, s_N, s_low, s_up)           *
*  int *t_N, *t_low, *t_up;                                               *
*  int *s_N, *s_low, *s_up;                                               *
*                                                                         *
*  void dalib_move_source1__ (a, size, N1, x1, y1)                         *
*  unsigned char *a;                                                      *
*  int *size;                                                             *
*  int *N1, *x1, *y1;                                                     *
*                                                                         *
*  void dalib_move_source2__ (a_dsp, a, x1, y1, x2, y2)                    *
*  int *a_dsp;                                                            *
*  unsigned char *a;                                                      *
*  int *x1, *y1, *x2, *y2;                                                *
*                                                                         *
*   .......                                                               *
*                                                                         *
*  void dalib_move_source4__ (a_dsp, a, x1, y1, x2, y2, x3, y3, x4, y4)    *
*  int *a_dsp;                                                            *
*  unsigned char *a;                                                      *
*  int *x1, *y1, *x2, *y2, *x3, *y3, *x4, *y4;                            *
*                                                                         *
*  void dalib_move_target1__ (a_dsp, a, x1, y1)                            *
*  int *a_dsp;                                                            *
*  unsigned char *a;                                                      *
*  int *x1, *y1;                                                          *
*                                                                         *
*  void dalib_move_target2__ (a_dsp, a, x1, y1, x2, y2)                    *
*  int *a_dsp;                                                            *
*  unsigned char *a;                                                      *
*  int *x1, *y1, *x2, *y2;                                                *
*                                                                         *
*   .......                                                               *
*                                                                         *
*  void dalib_move_target4__ (a_dsp, a, x1, y1, x2, y2, x3, y3, x4, y4)    *
*  int *a_dsp;                                                            *
*  unsigned char *a;                                                      *
*  int *x1, *y1, *x2, *y2, *x3, *y3, *x4, *y4;                            *
*                                                                         *
**************************************************************************/

#include "system.h"

#undef DEBUG

  /*********************************************************
  *                                                        *
  *  Global data for a movement                            *
  *                                                        *
  *    - size of source and target                         *
  *    - global range of source and target                 *
  *    - local range of source and target                  *
  *    - kind of movement:                                 *
  *                                                        *
  *        a(1:5) = b(7:11)   many <-- many    KIND = 0    *
  *      a(1:5,4) = b(7:11)   one  <-- many    KIND = 1    *
  *        a(1:5) = b(7:11,3) many <-- one     KIND = 2    *
  *      a(1:5,4) = b(7:11,3) one  <-- one     KIND = 3    *
  *                                                        *
  *********************************************************/

int target_size, source_size;

int target_low, target_up;
int source_low, source_up;
int local_source_low, local_source_up; /* range of source that will be send  */
int local_target_low, local_target_up; /* range of target that will be recvd */

int move_kind, move_constant;

     /*********************************************************
     *                                                        *
     *  Definition of a movement                              *
     *                                                        *
     *********************************************************/

void dalib_move_define__ (t_N, t_low, t_up, s_N, s_low, s_up)
int *t_N, *t_low, *t_up;
int *s_N, *s_low, *s_up;

{ /* maps [s_low:s_up] of source to [t_low:t_up] of target */
  
#ifdef DEBUG
  printf ("Process %d calls move_define (%d:%d)/%d <- (%d:%d)/%d\n", 
           pcb.i, *t_low, *t_up, *t_N, *s_low, *s_up, *s_N          ); 
#endif

  target_size       = *t_N;
  target_low        = *t_low;
  target_up         = *t_up;
  source_size       = *s_N;
  source_low        = *s_low;
  source_up         = *s_up;

  move_kind = 0;
  if (source_low == source_up) move_kind += 2;
  if (target_low == target_up) move_kind += 1;

  move_constant = target_low - source_low;

  /* set local_source : elements to send */

  dalib_local_slice__ (&source_size, &source_low, &source_up,
                                    &local_source_low, &local_source_up);
        
  /* set local_target : elements to recv */

  dalib_local_slice__ (&target_size, &target_low, &target_up,
                                    &local_target_low, &local_target_up);
}

     /*********************************************************
     *                                                        *
     *  pid_low - pid_up : Processes that will get elements   *
     *                     from the source array              *
     *                                                        *
     *********************************************************/

void dalib_move_source_pids (pid_low, pid_up)
int *pid_low, *pid_up;

{ /* find pids where source has to send elements to */

  int def_target_low, def_target_up;

  if (local_source_low > local_source_up)
    { /* this process has no elements for other processes */
      *pid_low = 2;
      *pid_up  = 1;
    }
   else if (move_kind == 0) 
    { /* many <-> many */

      /* get the range of target that is defined by my local source */
      def_target_low = local_source_low + move_constant;
      def_target_up  = local_source_up + move_constant;
      *pid_low = dalib_where (target_size, def_target_low);
      *pid_up = dalib_where (target_size, def_target_up);
    }
   else if (move_kind == 2)
    { /* one -> many, my elements are for all other */
      *pid_low = dalib_where (target_size, target_low);
      *pid_up  = dalib_where (target_size, target_up);
    }
   else 
    { /* my element(s) are only for one processor , target_low == target_up */
      *pid_low = dalib_where (target_size, target_low);
      *pid_up  = *pid_low;
    }

  /* 
  printf ("Process %d will send processes %d - %d\n",
          pcb.i, *pid_low, *pid_up);
  */
}

     /*********************************************************
     *                                                        *
     *  t_pid : Process that gets elements from my source     *
     *                                                        *
     *  low : up : Interval of my source that has to be sent  *
     *                                                        *
     *********************************************************/

void dalib_move_source_range (t_pid, low, up)
int t_pid, *low, *up;

{ /* finds source range that has to be sent to t_pid */
  int n;
  int a_low, a_up;
  int s_low, s_up;
 
  n = pcb.p;

  dalib_local_extensions (t_pid, target_size, &a_low, &a_up);

  /* a_low:a_up is local range of target of t_pid */

  if (move_kind == 0)
     { /* many <- many, b(7:11) = a(8:12) */
       s_low = a_low - move_constant;
       s_up  = a_up  - move_constant;
       /* s_low .. s_up is range of source for t_pid, give it if I have */
       if (s_low < local_source_low) s_low = local_source_low;
       if (s_up  > local_source_up)  s_up  = local_source_up;
     }
   else if (move_kind == 2)
     { /* many <- one,  b(7:11) = a(8:12,4) */
       /* restrict full range of a to the slice */
       if (a_low < target_low) a_low = target_low;
       if (a_up  > target_up)  a_up  = target_up;
       /* normalize it for splitting up other dimension */
       s_low = a_low - target_low;
       s_up  = a_up  - target_low;
     }
   else 
     { /* all my elements are for one processor */
       s_low = local_source_low;
       s_up  = local_source_up;
     }
  *low = s_low;
  *up  = s_up;
}

     /*********************************************************
     *                                                        *
     *  pid_low - pid_up : Processes that have sent elements  *
     *                     from the source array              *
     *                                                        *
     *********************************************************/

void dalib_move_target_pids (pid_low, pid_up)
int *pid_low, *pid_up;

{ /* find pids where target will receive elements from */

  int def_source_low, def_source_up;

  if (local_target_low > local_target_up)
    { /* this process needs no elements from other processes */
      *pid_low = 2;
      *pid_up  = 1;
    }
   else if (move_kind == 0)
    { /* many <-> many */
      /* get the range of source that is needed for my local target */
      def_source_low = local_target_low - move_constant;
      def_source_up  = local_target_up  - move_constant;
      *pid_low = dalib_where (source_size, def_source_low);
      *pid_up = dalib_where (source_size, def_source_up);
    }
   else if (move_kind == 1)
    { /* many -> one, I get elements from all other */
      *pid_low = dalib_where (source_size, source_low);
      *pid_up  = dalib_where (source_size, source_up);
    }
   else
    { /* get elements only from one processor, source_low == source_up */
      *pid_low = dalib_where (source_size, source_low);
      *pid_up  = *pid_low;
    }

  /*
  printf ("Process %d will receive from processes %d - %d\n",
          pcb.i, *pid_low, *pid_up);
  */
}

     /*********************************************************
     *                                                        *
     *  t_pid : Process that has elements for my target       *
     *                                                        *
     *  low : up : Interval of my target that has to be recvd *
     *                                                        *
     *********************************************************/

void dalib_move_target_range (t_pid, low, up)
int t_pid, *low, *up;

{ /* finds target range that I will get from t_pid */
  int n;
  int a_low, a_up;
  int t_low, t_up;
 
  n = pcb.p;
  dalib_local_extensions (t_pid, source_size, &a_low, &a_up);

  /* a_low .. a_up is local range of source of t_pid */

  if (move_kind == 0)
    { /* many <- many, e.g. b(7:11) = a(8:12) */
      t_low = a_low + move_constant;
      t_up  = a_up  + move_constant;
      if (t_low < local_target_low) t_low = local_target_low;
      if (t_up  > local_target_up ) t_up  = local_target_up ;
    }
   else if (move_kind == 1)
    { /* one <- many, e.g. b(8:12,4) = a(7:11) */
      /* restrict full range of a to the range */
      if (a_low < source_low) a_low = source_low;
      if (a_up  > source_up ) a_up  = source_up;
      /* normalize it to have later access */
      t_low = a_low - source_low;
      t_up  = a_up  - source_low;
    }
   else 
    { /* many <- one, one <- one , I get all elements from one processor */
      t_low = local_target_low;
      t_up  = local_target_up;
    }

  *low = t_low;
  *up  = t_up;
}

     /*********************************************************
     *                                                        *
     *  Global sending of A[x1:y1]                            *
     *                                                        *
     *********************************************************/

void dalib_move_source1__ (a, size, N1, x1, y1)
unsigned char *a;
int *size;
int *N1, *x1, *y1;

{ int low_pid, up_pid, pid;
  int hx1, hy1;
  int from;
 
  from = dalib_pid_ ();

  if (*y1 < *x1) goto End;

  dalib_move_source_pids (&low_pid, &up_pid);

  /* I have to send local sections to low_pid .. up_pid */

  for (pid = low_pid; pid <= up_pid; pid++)
    { dalib_move_source_range (pid, &hx1, &hy1);
#ifdef DEBUG
      printf ("%d sends range1 %d - %d to %d\n", from, hx1, hy1, pid);
#endif 
      dalib_setup_section1 (*size, *N1, hx1, hy1);
      dalib_send_section1 (pid, a);
    }
   End: ;
}

     /*********************************************************
     *                                                        *
     *  Global sending of A[x1:y1,x2:y2]                      *
     *                                                        *
     *********************************************************/

void dalib_move_source2__ (a, size, N1, x1, y1, N2, x2, y2)
unsigned char *a;
int *size;
int *N1, *x1, *y1, *N2, *x2, *y2;

{ int low_pid, up_pid, pid;
  int hx2, hy2;
  int from;
  int dim;
 
  if (*y1 < *x1) goto End;
  if (*y2 < *x2) goto End;

  from = dalib_pid_ ();

  dalib_move_source_pids (&low_pid, &up_pid);

  /* I have to send local sections to low_pid .. up_pid */

  if (move_kind == 2)
     { /* many <- one, b(7:11) = a(8:12,4) so the problem is to
          find the dimension of a for which sections are splitted */
       dim = 1;
     }
    else
     dim = 2;

  for (pid = low_pid; pid <= up_pid; pid++)
    { dalib_move_source_range (pid, &hx2, &hy2);
      if (dim == 2)
        {
#ifdef DEBUG
          printf ("%d sends range2 %d - %d, %d - %d to %d\n",
                   from, *x1, *y1, hx2, hy2, pid);
#endif
          dalib_setup_section2 (*size, *N1, *x1, *y1, 
                                       *N2, hx2, hy2);
        }
       else
        {
#ifdef DEBUG
          printf ("%d sends 1->n range2 %d - %d, %d - %d to %d\n",
                   from, *x1+hx2, *x1+hy2, *x2, *y2, pid);
#endif
          dalib_setup_section2 (*size, *N1, *x1 + hx2, *x1 + hy2, 
                                       *N2, *x2, *y2);
        }
      dalib_send_section2 (pid, a);
    }
  End:;
}

     /*********************************************************
     *                                                        *
     *  Global sending of A[x1:y1,x2:y2,x3:y3]                *
     *                                                        *
     *********************************************************/

void dalib_move_source3__ (a, size, N1, x1, y1, N2, x2, y2, N3, x3, y3)
unsigned char *a;
int *size;
int *N1, *x1, *y1, *N2, *x2, *y2, *N3, *x3, *y3;

{ int low_pid, up_pid, pid;
  int hx3, hy3;
  int from;
  int dim;
 
  if (*y1 < *x1) goto End;
  if (*y2 < *x2) goto End;
  if (*y3 < *x3) goto End;

  from = dalib_pid_ ();

  dalib_move_source_pids (&low_pid, &up_pid);

  if (move_kind == 2)
     { /* many <- one, b(7:11) = a(8:12,..,4) so the problem is to
          find the dimension of a for which sections are splitted */
       if (*x2 != *y2)
          dim = 2;
        else
          dim = 1;
     }
    else
     dim = 3;

  /* I have to send local sections to low_pid .. up_pid */

  for (pid = low_pid; pid <= up_pid; pid++)
  /* I have to send local sections to low_pid .. up_pid */

  for (pid = low_pid; pid <= up_pid; pid++)
    { dalib_move_source_range (pid, &hx3, &hy3);
      if (dim == 3)
        {
#ifdef DEBUG
          printf ("%d sends range3 %d - %d, %d - %d, %d - %d to %d\n",
                   from, *x1, *y1, *x2, *y2, hx3, hy3, pid);
#endif
          dalib_setup_section3 (*size, *N1, *x1, *y1, 
                                       *N2, *x2, *y2, 
                                       *N3, hx3, hy3);
        }
       else if (dim == 2)
        {
#ifdef DEBUG
          printf ("%d sends 1->n range3 %d - %d, %d - %d, %d - %d to %d\n",
                   from, *x1, *y1, *x2+hx3, *x2+hy3, *x3, *y3, pid);
#endif
          dalib_setup_section3 (*size, *N1, *x1,*y1,
                                       *N2, *x2+hx3,*x2+hy3,
                                       *N3, *x3,*y3);
        }
       else 
        {
#ifdef DEBUG
          printf ("%d sends 1->n range3 %d - %d, %d - %d, %d - %d to %d\n",
                   from, *x1+hx3,*x1+hy3, *x2, *y2, *x3, *y3, pid);
#endif
          dalib_setup_section3 (*size, *N1, *x1+hx3, *x1+hy3,
                                       *N2, *x2,     *y2,
                                       *N3, *x3,     *y3);
        }
      dalib_send_section3 (pid, a);
    }
  End:;
}

     /*********************************************************
     *                                                        *
     *  Global sending of A[x1:y1,x2:y2,x3:y3,x4:y4]          *
     *                                                        *
     *********************************************************/

void dalib_move_source4__ (a, size, N1, x1, y1, N2, x2, y2, 
                                   N3, x3, y3, N4, x4, y4)
unsigned char *a;
int *size;
int *N1, *x1, *y1, *N2, *x2, *y2, *N3, *x3, *y3, *N4, *x4, *y4;

{ int low_pid, up_pid, pid;
  int hx4, hy4;
  int from;
  int dim;

  if (*y1 < *x1) goto End;
  if (*y2 < *x2) goto End;
  if (*y3 < *x3) goto End;
  if (*y4 < *x4) goto End;

  from = dalib_pid_ ();

  dalib_move_source_pids (&low_pid, &up_pid);

  if (move_kind == 2)
     { /* many <- one, b(7:11) = a(8:12,..,4) so the problem is to
          find the dimension of a for which sections are splitted */
       if (*x3 != *y3)
          dim = 3;
        else if (*x2 != *y2)
          dim = 2;
        else
          dim = 1;
     }
    else
     dim = 4;

  /* I have to send local sections to low_pid .. up_pid */

  for (pid = low_pid; pid <= up_pid; pid++)
    { dalib_move_source_range (pid, &hx4, &hy4);
      if (dim == 4)
        { /* usual case */
#ifdef DEBUG
          printf ("%d sends range4 %d - %d, %d - %d, %d - %d, %d - %d to %d\n",
                   from, *x1, *y1, *x2, *y2, *x3, *y3, hx4, hy4, pid);
#endif
          dalib_setup_section4 (*size, *N1, *x1, *y1, 
                                       *N2, *x2, *y2, 
                                       *N3, *x3, *y3, 
                                       *N4, hx4, hy4);
        }
       else if (dim == 3)
        {
#ifdef DEBUG
          printf ("%d sends 1->n range4 %d-%d,%d-%d,%d-%d,%d-%d to %d\n",
                   from, *x1,*y1,*x2,*y2,*x3+hx4,*x4+hy4,*x4,*y4, pid);
#endif
          dalib_setup_section4 (*size, *N1, *x1,*y1,
                                       *N2, *x2,*y2,
                                       *N3, *x3+hx4,*x3+hy4,
                                       *N4, *x4,*y4);
        }
       else if (dim == 2)
        {
#ifdef DEBUG
          printf ("%d sends 1->n range4 %d-%d,%d-%d,%d-%d,%d-%d to %d\n",
                   from, *x1,*y2,*x2+hx4,*x2+hy4,*x3,*y3,*x4,*y4, pid);
#endif
          dalib_setup_section4 (*size, *N1, *x1,*y1,
                                       *N2, *x2+hx4,*x2+hy4,
                                       *N3, *x3,*y3,
                                       *N4, *x4,*y4);
        }
       else 
        {
#ifdef DEBUG
          printf ("%d sends 1->n range4 %d-%d,%d-%d,%d-%d,%d-%d to %d\n",
                   from, *x1+hx4,*x1+hy4,*x2,*y2,*x3,*y3,*x4,*y4, pid);
#endif
          dalib_setup_section4 (*size, *N1, *x1+hx4,*x1+hy4,
                                       *N2, *x2,*y2,
                                       *N3, *x3,*y3,
                                       *N4, *x4,*y4);
        }
#ifdef DEBUG
      printf ("%d sends range4 %d - %d to %d\n", from, hx4, hy4, pid);
#endif
      dalib_send_section4 (pid, a);
    }
  End:;
}

     /*********************************************************
     *                                                        *
     *  Global receiving of A[x1:y1]                          *
     *                                                        *
     *********************************************************/

void dalib_move_target1__ (a, size, N1, x1, y1)
unsigned char *a;
int *size;
int *N1, *x1, *y1;

{ int low_pid, up_pid, pid;
  int hx1, hy1;
  int to;
 
  if (*y1 < *x1) goto End;

  dalib_move_target_pids (&low_pid, &up_pid);

  to = dalib_pid_();

  /* I have to receive local section from low_pid .. up_pid */

  for (pid = low_pid; pid <= up_pid; pid++)
    { dalib_move_target_range (pid, &hx1, &hy1);
#ifdef DEBUG
      printf ("%d recvs range1 %d - %d from %d\n", to, hx1, hy1, pid);
#endif
      dalib_setup_section1   (*size, *N1, hx1, hy1);
      dalib_recv_section1   (pid, a);
    }
  End:;
}

     /*********************************************************
     *                                                        *
     *  Global receiving of A[x1:y1,x2:y2]                    *
     *                                                        *
     *********************************************************/

void dalib_move_target2__ (a, size, N1, x1, y1, N2, x2, y2)
unsigned char *a;
int *size;
int *N1, *x1, *y1, *N2, *x2, *y2;

{ int low_pid, up_pid, pid;
  int hx2, hy2;
  int to;
  int dim;
 
  if (*y1 < *x1) goto End;
  if (*y2 < *x2) goto End;
 
  dalib_move_target_pids (&low_pid, &up_pid);

  to = dalib_pid_();

  /* I have to receive local section from low_pid .. up_pid */

  if (move_kind == 1)
     { /* one <- many, b(7:11,4) = a(8:12) so the problem is to
          find the dimension of b for which sections are splitted */
       dim = 1;
     }
   else
     dim = 2;

  for (pid = low_pid; pid <= up_pid; pid++)
    { dalib_move_target_range (pid, &hx2, &hy2);
      if (dim == 2)
        {
#ifdef DEBUG
          printf ("%d recvs range2 %d - %d, %d - %d from %d\n",
                   to, *x1, *y1, hx2, hy2, pid);
#endif
          dalib_setup_section2  (*size, *N1, *x1, *y1, 
                                        *N2, hx2, hy2);
        }
        else
        {
#ifdef DEBUG
          printf ("%d recvs n->1 range2 %d - %d, %d - %d from %d\n",
                   to, *x1+hx2, *x1+hy2, *x2, *y2, pid);
#endif
          dalib_setup_section2  (*size, *N1, *x1+hx2, *x1+hy2, 
                                        *N2, *x2, *y2);
        }
      dalib_recv_section2   (pid, a);
    }
  End:;
}

     /*********************************************************
     *                                                        *
     *  Global receiving of A[x1:y1,x2:y2,x3:y3]              *
     *                                                        *
     *********************************************************/

void dalib_move_target3__ (a, size, N1, x1, y1, N2, x2, y2, N3, x3, y3)
unsigned char *a;
int *size;
int *N1, *x1, *y1, *N2, *x2, *y2, *N3, *x3, *y3;

{ int low_pid, up_pid, pid;
  int hx3, hy3;
  int to;
  int dim;
 
  if (*y1 < *x1) goto End;
  if (*y2 < *x2) goto End;
  if (*y3 < *x3) goto End;
 
  dalib_move_target_pids (&low_pid, &up_pid);

  to = dalib_pid_();

  if (move_kind == 1)
     { /* one <- many, b(7:11,...,4) = a(8:12) so the problem is to
          find the dimension of b for which sections are splitted */
       if (*x2 != *y2)
          dim = 2;
       else
          dim = 1;
     }
   else
     dim = 3;

  /* I have to receive local section from low_pid .. up_pid */

  for (pid = low_pid; pid <= up_pid; pid++)
    { dalib_move_target_range (pid, &hx3, &hy3);
      if (dim == 3)
        { /* usual case */
#ifdef DEBUG
          printf ("%d recvs range3 %d - %d, %d - %d, %d - %d from %d\n",
                   to, *x1, *y1, *x2, *y2, hx3, hy3, pid);
#endif
          dalib_setup_section3  (*size, *N1, *x1, *y1, 
                                        *N2, *x2, *y2, 
                                        *N3, hx3, hy3);
        }
        else if (dim == 2)
        {
#ifdef DEBUG
          printf ("%d recvs n->1 range3 %d-%d,%d-%d,%d-%d from %d\n",
                   to, *x1,*y1,*x2+hx3,*x2+hy3,*x3,*y3, pid);
#endif
          dalib_setup_section3  (*size, *N1, *x1,*y1,
                                        *N2, *x2+hx3,*x2+hy3,
                                        *N3, *x3,*y3);
        }
        else
        {
#ifdef DEBUG
          printf ("%d recvs n->1 range3 %d-%d,%d-%d,%d-%d from %d\n",
                   to, *x1,*y1,*x2,*y2,*x3,*y3, pid);
#endif
          dalib_setup_section3  (*size, *N1, *x1,*y1,
                                        *N2, *x2,*y2,
                                        *N3, *x3,*y3);
        }
      dalib_recv_section3   (pid, a);
    }
  End:;
}

     /*********************************************************
     *                                                        *
     *  Global receiving of A[x1:y1,x2:y2,x3:y3,x4:y4]        *
     *                                                        *
     *********************************************************/

void dalib_move_target4__ (a, size, N1, x1, y1, N2, x2, y2, 
                                   N3, x3, y3, N4, x4, y4)
unsigned char *a;
int *size;
int *N1, *x1, *y1, *N2, *x2, *y2, *N3, *x3, *y3, *N4, *x4, *y4;

{ int low_pid, up_pid, pid;
  int hx4, hy4;
  int to;
  int dim;

  if (*y1 < *x1) goto End;
  if (*y2 < *x2) goto End;
  if (*y3 < *x3) goto End;
  if (*y4 < *x4) goto End;
 
  dalib_move_target_pids (&low_pid, &up_pid);

  to = dalib_pid_();

  if (move_kind == 1)
     { /* one <- many, b(7:11,...,4) = a(8:12) so the problem is to
          find the dimension of b for which sections are splitted */
       if (*x3 != *y3)
          dim = 3;
       else if (*x2 != *y2)
          dim = 2;
       else
          dim = 1;
     }
   else
     dim = 4;

  /* I have to receive local section from low_pid .. up_pid */

  for (pid = low_pid; pid <= up_pid; pid++)
    { dalib_move_target_range (pid, &hx4, &hy4);
      if (dim == 4)
        { /* usual case */
#ifdef DEBUG
          printf ("%d recvs range4 %d-%d, %d-%d, %d-%d, %d-%d from %d\n",
                   to, *x1, *y1, *x2, *y2, *x3, *y3, hx4, hy4, pid);
#endif
          dalib_setup_section4  (*size, *N1, *x1, *y1, 
                                        *N2, *x2, *y2, 
                                        *N3, *x3, *y3, 
                                        *N4, hx4, hy4);
        }
        else if (dim == 3)
        {
#ifdef DEBUG
          printf ("%d recvs n->1 range4 %d-%d,%d-%d,%d-%d,%d-%d from %d\n",
                   to, *x1,*y1,*x2,*y2,*x3+hx4,*x3+hy4,*x4,*y4, pid);
#endif
          dalib_setup_section4  (*size, *N1, *x1,*y1,
                                        *N2, *x2,*y2,
                                        *N3, *x3+hx4,*x3+hy4,
                                        *N4, *x4,*y4);
        }
        else if (dim == 2)
        {
#ifdef DEBUG
          printf ("%d recvs n->1 range4 %d-%d,%d-%d,%d-%d,%d-%d from %d\n",
                   to, *x1,*y1,*x2+hx4,*x2+hy4,*x3,*y3,*x4,*y4, pid);
#endif
          dalib_setup_section4  (*size, *N1, *x1,*y1,
                                        *N2, *x2+hx4,*x2+hy4,
                                        *N3, *x3,*y3,
                                        *N4, *x4,*y4);
        }
        else 
        {
#ifdef DEBUG
          printf ("%d recvs n->1 range4 %d-%d,%d-%d,%d-%d,%d-%d from %d\n",
                   to, *x1+hx4,*x1+hy4,*x2,*y2,*x3,*y3,*x4,*y4, pid);
#endif
          dalib_setup_section4  (*size, *N1, *x1+hx4,*x1+hy4,
                                        *N2, *x2,*y2,
                                        *N3, *x3,*y3,
                                        *N4, *x4,*y4);
        }
      dalib_recv_section4   (pid, a);
    }
  End:;
}
