/*
 * Copyright 2010-2012 Marcel Mueller
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *    1. Redistributions of source code must retain the above copyright notice,
 *       this list of conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 *    3. The name of the author may not be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */


#include "playbackworker.h"
//#include <os2.h>
#include <math.h>
#include <stdint.h>
// For some reason the above include does not work.
# define UINT32_MAX     (4294967295U)

#include <cpp/cpputil.h>
#include <stdarg.h>
#include <debuglog.h>


void PlaybackWorker::BackupBuffer::Reset()
{ Mutex::Lock lock(Mtx);
  BufferLow = 0;
  BufferHigh = 0;
  MaxWriteIndex = 0;
  BufferQueue[countof(BufferQueue)-1].Data = 0;
  memset(DataBuffer, 0, sizeof DataBuffer);
}

size_t PlaybackWorker::BackupBuffer::FindByWriteIndex(uint64_t wi) const
{ DEBUGLOG(("PlaybackWorker::BackupBuffer::FindByWriteIndex(%Lu) - [%u,%u[ %Lu\n", wi, BufferLow, BufferHigh, MaxWriteIndex));
  // Binary search in ring buffer
  size_t l = BufferLow;
  size_t r = BufferHigh;
  while (l != r)
  { size_t m = (l + r + countof(BufferQueue)*(l > r)) / 2 % countof(BufferQueue);
    const Entry* ep = BufferQueue + m;
    if (ep->WriteIndex > wi)
    { // too far
      r = m;
    } else if (ep->WriteIndex == wi)
    { // exact match
      DEBUGLOG(("PlaybackWorker::BackupBuffer::FindByWriteIndex: match %u\n", m));
      return m;
    } else
    { // before
      l = (m + 1) % countof(BufferQueue);
    }
  }
  DEBUGLOG(("PlaybackWorker::BackupBuffer::FindByWriteIndex: inexact match %u -> %Lu -> %u\n", l, BufferQueue[l].WriteIndex, BufferQueue[l].Data));
  return l;
}

/*size_t PlaybackWorker::BackupBuffer::FindByTime(pa_usec_t time) const
{ DEBUGLOG(("PlaybackWorker::BackupBuffer::FindByTime(%Lu) - [%u,%u[\n", time, BufferLow, BufferHigh));
  // Binary search in ring buffer
  size_t l = BufferLow;
  size_t r = BufferHigh;
  while (l != r)
  { size_t m = (l + r + (sizeof BufferQueue/sizeof *BufferQueue)*(l > r)) / 2 % (sizeof BufferQueue/sizeof *BufferQueue);
    const Entry* ep = BufferQueue + m;
    if (ep->Time > time)
    { // too far
      r = m;
    } else if (ep->Time == time)
    { // exact match
      DEBUGLOG(("PlaybackWorker::BackupBuffer::FindByTime: match %u\n", m));
      return m;
    } else
    { // before
      l = (m + 1) % (sizeof BufferQueue/sizeof *BufferQueue);
    }
  }
  DEBUGLOG(("PlaybackWorker::BackupBuffer::FindByTime: no match %u\n", l));
  return l;
}*/

void PlaybackWorker::BackupBuffer::StoreData(uint64_t wi, PM123_TIME pos, int channels, int rate, const float* data, size_t count)
{ DEBUGLOG(("PlaybackWorker::BackupBuffer::StoreData(%Lu, %g, %i, %i, %p, %u) - [%u,%u[\n",
    wi, pos, channels, rate, data, count, BufferLow, BufferHigh));
  Mutex::Lock lock(Mtx);
  if (wi <= MaxWriteIndex)
  { // Discard any data beyond the current write index.
    BufferHigh = FindByWriteIndex(wi);
  }
  MaxWriteIndex = wi;
  size_t lastbufhigh = (BufferHigh + countof(BufferQueue)-1) % countof(BufferQueue);
  Entry* ep = BufferQueue + lastbufhigh;
  // current write location in the ring buffer
  size_t datahigh = ep->Data;

  // Ensure enough space
  if (BufferLow != BufferHigh)
    while ( BufferLow != lastbufhigh
      && (BufferQueue[BufferLow].Data + countof(DataBuffer) - datahigh) % countof(DataBuffer) <= count )
      BufferLow = (BufferLow + 1) % countof(BufferQueue);
  // Join descriptors?
  if (BufferLow == BufferHigh)
    datahigh = 0;
  else if ( ep->Format.channels == channels && ep->Format.samplerate == rate
         && fabs(ep->Pos + count - pos) < 1E-6 )
    goto join;
  { // new buffer
    size_t newbufhigh = (BufferHigh + 1) % countof(BufferQueue);
    if (newbufhigh == BufferLow)
    { // Overflow => discard one old buffer
      BufferLow = (BufferLow + 1) % countof(BufferQueue);
    }
    ep = BufferQueue + BufferHigh;
    BufferHigh = newbufhigh;
    ep->Format.channels = channels;
    ep->Format.samplerate = rate;
  }
 join:
  ep->WriteIndex = wi;
  ep->Pos = pos;
  ep->Data = datahigh + count;
  if (ep->Data >= countof(DataBuffer))
  { ep->Data -= countof(DataBuffer);
    // memory wrap
    memcpy(DataBuffer + datahigh, data, (countof(DataBuffer) - datahigh) * sizeof(float));
    memcpy(DataBuffer, data + countof(DataBuffer) - datahigh, ep->Data * sizeof(float));
  } else
    memcpy(DataBuffer + datahigh, data, count * sizeof(float));
  DEBUGLOG(("PlaybackWorker::BackupBuffer::StoreData: [%u,%u[ time = %f\n", BufferLow, BufferHigh, ep->Pos));
}

PM123_TIME PlaybackWorker::BackupBuffer::GetPosByWriteIndex(uint64_t wi)
{ Mutex::Lock lock(Mtx);
  if (wi > MaxWriteIndex)
  { // Error: negative latency???
    DEBUGLOG(("PlaybackWorker::BackupBuffer::GetPosByWriteIndex(%Lu): negative Latency", wi));
    return 0;
  }
  const Entry& e = BufferQueue[FindByWriteIndex(wi)];
  return e.Pos - (PM123_TIME)(e.WriteIndex - wi) / (sizeof(float) * e.Format.channels * e.Format.samplerate);
}

/*PM123_TIME PlaybackWorker::BackupBuffer::GetPosByTimeIndex(pa_usec_t time) const
{ size_t p = FindByTime(time);
  if (p == BufferHigh)
  { // Error: negative latency???
    DEBUGLOG(("PlaybackWorker::BackupBuffer::GetPosByTimeIndex(%Lu): negative Latency", time));
    return 0;
  }
  const Entry* ep = BufferQueue + p;
  return ep->Pos - (ep->Time - time) / 1E6;
}*/

bool PlaybackWorker::BackupBuffer::GetDataByWriteIndex(uint64_t wi, OUTPUT_PLAYING_BUFFER_CB cb, void* param)
{ DEBUGLOG(("PlaybackWorker::BackupBuffer::GetDataByWriteIndex(%Lu, %p, %p)\n", wi, cb, param));
  Mutex::Lock lock(Mtx);
  if (wi > MaxWriteIndex)
    return false; // Error: negative latency???
  size_t p = FindByWriteIndex(wi);
  ASSERT(p != BufferHigh);
  const Entry* ep = BufferQueue + p;
  size_t before = (ep->WriteIndex - wi) / sizeof(float); // Requested samples start at /before/ values before ep->Data.
  size_t datastart = (ep->Data - before) % countof(DataBuffer);
  if (p == BufferLow)
  { size_t datahigh = BufferHigh == 0 ? BufferQueue[countof(BufferQueue)-1].Data : BufferQueue[BufferHigh-1].Data;
    if (before > (ep->Data - datahigh - 1) % countof(DataBuffer) + 1)
      datastart = datahigh;
  }
  for (;;)
  { p = (p + 1) % countof(BufferQueue);
    BOOL done = p == BufferHigh;
    if (datastart > ep->Data)
    { size_t count = (ep->Data - datastart + countof(DataBuffer)) / ep->Format.channels;
      size_t count2 = ep->Data / ep->Format.channels;
      DEBUGLOG(("PlaybackWorker::BackupBuffer::GetDataByWriteIndex: from %u to %u, count = %u, count2 = %u\n", datastart, ep->Data, count, count2));
      (*cb)(param, &ep->Format, DataBuffer + datastart, count - count2, ep->Pos - (PM123_TIME)count / ep->Format.samplerate, &done);
      if (done)
        break;
      (*cb)(param, &ep->Format, DataBuffer, count2, ep->Pos - (PM123_TIME)count2 / ep->Format.samplerate, &done);
    } else
    { size_t count = (ep->Data - datastart) / ep->Format.channels;
      if (count == 0)
        count = countof(DataBuffer);
      DEBUGLOG(("PlaybackWorker::BackupBuffer::GetDataByWriteIndex: from %u to %u, count = %u\n", datastart, ep->Data, count));
      (*cb)(param, &ep->Format, DataBuffer + datastart, count, ep->Pos - (PM123_TIME)count / ep->Format.samplerate, &done);
    }
    if (done)
      break;
    // next
    before = 0;
    datastart = ep->Data;
    ep = BufferQueue + p;
  }
  return true;
}


PlaybackWorker::OUTPUT_STRUCT() throw()
: Volume(PA_VOLUME_INVALID)
, DrainOpDeleg(DrainOp.Completion(), *this, &PlaybackWorker::DrainOpCompletion)
{ DEBUGLOG(("PlaybackWorker(%p)::PlaybackWorker()\n", this));
}

ULONG PlaybackWorker::Init(const xstring& server, const xstring& sink, const xstring& port, int minlatency, int maxlatency) throw()
{ DEBUGLOG(("PlaybackWorker(%p)::Init(%s)\n", this, server.cdata()));
  Server = server;
  Sink = sink;
  Port = port;
  MinLatency = minlatency;
  MaxLatency = maxlatency;
  try
  { if (!Server || !*Server)
      throw PAException(PLUGIN_ERROR,
        "You need to configure a destination server before using pulse123.\n"
        "Go to \"Plug-ins/PulseAudio for PM123\" from the main context menu.");
    Context.Connect("PM123", Server);
    return PLUGIN_OK;
  } catch (const PAException& ex)
  { Error(ex.GetMessage());
    return ex.GetError();
  }
}

PlaybackWorker::~OUTPUT_STRUCT()
{ DEBUGLOG(("PlaybackWorker(%p)::~PlaybackWorker()\n", this));
}

void PlaybackWorker::PopulatePropertyList(const char* uri, const volatile META_INFO& meta)
{ Proplist.clear();
  xstring tmp; // Temporary object for strongly thread safe access to *info.
  char buf[1024];
  Proplist[PA_PROP_MEDIA_FILENAME]  = ToUTF8(buf, sizeof buf, uri);
  Proplist[PA_PROP_MEDIA_NAME]      = ToUTF8(buf, sizeof buf, tmp = meta.album);
  Proplist[PA_PROP_MEDIA_TITLE]     = ToUTF8(buf, sizeof buf, tmp = meta.title);
  Proplist[PA_PROP_MEDIA_ARTIST]    = ToUTF8(buf, sizeof buf, tmp = meta.artist);
  Proplist[PA_PROP_MEDIA_COPYRIGHT] = ToUTF8(buf, sizeof buf, tmp = meta.copyright);
}

ULONG PlaybackWorker::Open(const char* uri, const INFO_BUNDLE_CV* info, PM123_TIME pos,
                           void DLLENTRYP(output_event)(void* w, OUTEVENTTYPE event), void* w) throw()
{ DEBUGLOG(("PlaybackWorker(%p)::Open(%s, {{%i, %i},}, %f, %p, %p)\n", this,
    uri, info->tech->channels, info->tech->samplerate, pos, output_event, w));

  OutputEvent = output_event;
  W = w;

  try
  { PopulatePropertyList(uri, *info->meta);

    switch (Stream.GetState())
    {default:
      // Update properties of existing stream
      try
      { Stream.ProplistUpdate(Proplist);
      } catch (const PAException& ex)
      { // Error here are non-fatal
      }
      return PLUGIN_OK;

     case PA_STREAM_UNCONNECTED:
     case PA_STREAM_FAILED:
     case PA_STREAM_TERMINATED:;
    }

    SS.format = PA_SAMPLE_FLOAT32LE;
    SS.channels = info->tech->channels;
    SS.rate = info->tech->samplerate;

    // The context is automatically connected at Init,
    // but at this point we have to synchronize the connection process.
    Context.WaitReady();
    if (Port)
    { PABasicOperation op;
      Context.SetSinkPort(op, Sink, Port);
      op.EnsureSuccess();
    }
    Stream.Connect(Context, "Out", &SS, NULL, Proplist,
                   Sink, PA_STREAM_INTERPOLATE_TIMING|PA_STREAM_NOT_MONOTONIC|PA_STREAM_AUTO_TIMING_UPDATE|PA_STREAM_VARIABLE_RATE, Volume);

    LastBuffer = NULL;
    TimeOffset = pos;
    WriteIndexOffset = 0;
    TrashFlag = true;
    FlushFlag = false;
    LowWater = false;
    Buffer.Reset();

    Stream.WaitReady();
    return PLUGIN_OK;
  } catch (const PAException& ex)
  { Error(ex.GetMessage());
    return ex.GetError();
  }
}

ULONG PlaybackWorker::Close() throw()
{ DEBUGLOG(("PlaybackWorker(%p)::Close()\n", this));
  Stream.Disconnect();
  return 0;
}

ULONG PlaybackWorker::SetVolume(float volume) throw()
{ DEBUGLOG(("PlaybackWorker(%p)::SetVolume(%f)\n", this, volume));
  try
  { // pseudo logarithmic scale
    volume /= 1 + 3.16227766*(1-volume);
    Volume = pa_sw_volume_from_linear(volume);
    if (Stream.GetState() == PA_STREAM_READY)
      Stream.SetVolume(Volume);
    return 0;

  } catch (const PAException& ex)
  { Error(ex.GetMessage());
    return ex.GetError();
  }
}

ULONG PlaybackWorker::SetPause(bool pause) throw()
{ DEBUGLOG(("PlaybackWorker(%p)::SetPause(%u)\n", this, pause));
  try
  { Stream.Cork(pause);
    if (LowWater)
    { LowWater = false;
      (*OutputEvent)(W, OUTEVENT_HIGH_WATER);
    }
    return 0;

  } catch (const PAException& ex)
  { Error(ex.GetMessage());
    return ex.GetError();
  }
}

ULONG PlaybackWorker::TrashBuffers(PM123_TIME pos) throw()
{ DEBUGLOG(("PlaybackWorker(%p)::TrashBuffers(%f)\n", this, pos));
  ULONG ret = 0;
  try
  { FlushFlag = false;
    Stream.Flush();
    //Buffer.Reset();
  } catch (const PAException& ex)
  { // We ignore errors here
    DEBUGLOG(("PlaybackWorker::TrashBuffers %s\n", ex.GetMessage().cdata()));
    ret = ex.GetError();
  }
  LastBuffer = NULL; // Discard any currently requested buffer
  TimeOffset = pos;
  TrashFlag = true;
  if (!LowWater)
  { LowWater = true;
    (*OutputEvent)(W, OUTEVENT_LOW_WATER);
  }
  return ret;
}

void PlaybackWorker::DrainOpCompletion(const int& success)
{ DEBUGLOG(("PlaybackWorker(%p)::DrainOpCompletion(%u)\n", this, success));
  if (FlushFlag)
  { // We raise the finish event even in case of an error to prevent
    // PM123 from hanging at the end of the song.
    RaiseOutputEvent(OUTEVENT_END_OF_DATA);
  }
}

int PlaybackWorker::RequestBuffer(const FORMAT_INFO2* format, float** buf) throw()
{ DEBUGLOG(("PlaybackWorker(%p)::RequestBuffer({%i, %i}, )\n", this,
    format ? format->channels : -1, format ? format->samplerate : -1));
  // some plug-ins didn't catch it ASSERT(LastBuffer == NULL);
  try
  { if ((FlushFlag = (buf == NULL)) != false)
    { // flush request
      Stream.RunDrain(DrainOp);
      return 0;
    }
    DrainOp.Cancel();

    // format changes?
    if (format->channels != SS.channels)
    { DEBUGLOG(("PlaybackWorker::RequestBuffer channels changed from {%u, %u}\n", SS.channels, SS.rate));
      // PulseAudio can't change the number of channels on the fly.
      // So we need to reopen the stream. To avoid to handle multiple concurrent
      // streams at once we wait until the last samples of the current stream are played
      // until the new stream is opened. Of course, this prevents from gapless playback.
      // But everything else is simply too much effort.
      Stream.RunDrain(DrainOp);
      // Wait for completion
      DrainOp.Wait();
      // reopen stream
      Stream.Disconnect();
      SS.channels = format->channels;
      SS.rate = format->samplerate;
      Stream.Connect(Context, "Out", &SS, NULL, Proplist,
                     Sink, PA_STREAM_INTERPOLATE_TIMING|PA_STREAM_NOT_MONOTONIC|PA_STREAM_AUTO_TIMING_UPDATE|PA_STREAM_VARIABLE_RATE, Volume);
      // Fetch samples written so far.
      WriteIndexOffset = Buffer.GetMaxWriteIndex();
      // Wait for stream to become ready
      Stream.WaitReady();
      // notify decoder about low buffer
      if (!LowWater)
      { LowWater = true;
        (*OutputEvent)(W, OUTEVENT_LOW_WATER);
      }
    } else if ((unsigned)format->samplerate != SS.rate)
    { DEBUGLOG(("PlaybackWorker::RequestBuffer sample rate changed from {%u, %u}\n", SS.channels, SS.rate));
      // PulseAudio is unable to change the sample rate at a certain location in the stream.
      // so we need to wait for all buffers to be played before we proceed.
      Stream.RunDrain(DrainOp);
      DrainOp.Wait();
      // Change sample rate of the stream.
      SS.rate = format->samplerate;
      Stream.UpdateSampleRate(SS.rate);
      // notify decoder about low buffer
      if (!LowWater)
      { LowWater = true;
        (*OutputEvent)(W, OUTEVENT_LOW_WATER);
      }
    }

    size_t len = (size_t)-1;
    *buf = LastBuffer = (float*)Stream.BeginWrite(len);
    len /= sizeof(float) * format->channels;
    DEBUGLOG(("PlaybackWorker::RequestBuffer: %i\n", len));
    return len;

  } catch (const PAStreamEndException& ex)
  { return 0;
  } catch (const PAException& ex)
  { Error(ex.GetMessage());
    RaiseOutputEvent(OUTEVENT_PLAY_ERROR);
    return -ex.GetError();
  }
}
void PlaybackWorker::CommitBuffer(int len, PM123_TIME pos) throw()
{ float* buffer = LastBuffer;
  DEBUGLOG(("PlaybackWorker(%p)::CommitBuffer(%i, %f) - %p\n", this, len, pos, buffer));
  LastBuffer = NULL;
  if (!buffer || len <= 0)
    return;
  try
  { len *= SS.channels;
    uint64_t wi = Stream.Write(buffer, len * sizeof(float)) + WriteIndexOffset;
    Buffer.StoreData(wi, pos + (PM123_TIME)len/SS.rate, SS.channels, SS.rate, buffer, len);
    TrashFlag = false;
  } catch (const PAStreamEndException& ex)
  { return;
  } catch (const PAException& ex)
  { Error(ex.GetMessage());
    RaiseOutputEvent(OUTEVENT_PLAY_ERROR);
  }
}

BOOL PlaybackWorker::IsPlaying() throw()
{ return Stream.GetState() == PA_STREAM_READY;
}

PM123_TIME PlaybackWorker::GetPosition() throw()
{ // get time
  try
  { if (!TrashFlag && Stream.GetState() == PA_STREAM_READY)
    { double tmp = Stream.GetTime()/1E6 * sizeof(float) * SS.channels * SS.rate;
      tmp = Buffer.GetPosByWriteIndex((uint64_t)tmp + WriteIndexOffset);
      DEBUGLOG(("PlaybackWorker::GetPosition: %f\n", tmp));
      // Check for buffer level
      const pa_timing_info* ti = Stream.GetTimingInfo();
      if (ti)
      { double ms = pa_bytes_to_usec(ti->write_index, &SS) / 1000. - tmp * 1000;
        DEBUGLOG2(("PlaybackWorker::GetPosition Timing info: {%i, %Lu,%Lu,%Lu, %i, %i,%Li, %i,%Li,} %u, %u, %u\n",
          ti->synchronized_clocks, ti->sink_usec, ti->source_usec, ti->transport_usec,
          ti->playing, ti-> write_index_corrupt, ti->write_index, ti->read_index_corrupt, ti->read_index,
          ms, MinLatency, MaxLatency));
        if (LowWater)
        { if (ms > MinLatency + MaxLatency)
          { LowWater = false;
            (*OutputEvent)(W, OUTEVENT_HIGH_WATER);
          }
        } else
        { if (ms < MinLatency)
          { LowWater = true;
            (*OutputEvent)(W, OUTEVENT_LOW_WATER);
          }
        }
      }
      return tmp;
    }
  } catch (const PAException& ex)
  { // We ignore errors here
    DEBUGLOG(("PlaybackWorker::GetPosition %s\n", ex.GetMessage().cdata()));
  }
  // Error
  return TimeOffset;
}

ULONG PlaybackWorker::GetCurrentSamples(PM123_TIME offset, OUTPUT_PLAYING_BUFFER_CB cb, void* param) throw()
{ DEBUGLOG(("PlaybackWorker(%p)::GetCurrentSamples(%f, %p, %p)\n", this, offset, cb, param));
  try
  { if (TrashFlag || Stream.GetState() != PA_STREAM_READY)
      return 1;
    double tmp = Stream.GetTime()/1E6 * (sizeof(float) * SS.channels * SS.rate);
    return !Buffer.GetDataByWriteIndex((uint64_t)tmp + WriteIndexOffset, cb, param);
  } catch (const PAException& ex)
  { // We ignore errors here
    DEBUGLOG(("PlaybackWorker::GetCurrentSamples: error %s\n", ex.GetMessage().cdata()));
    return 1;
  }
}


void PlaybackWorker::Error(const char* fmt, ...) throw()
{ va_list va;
  va_start(va, fmt);
  xstring err;
  err.vsprintf(fmt, va);
  (*Ctx.plugin_api->message_display)(MSG_ERROR, err);
  va_end(va);
}
