/*****************************************************************************/
/*                                                                           */
/*  Programmname  : BMPDiff main.c                                           */
/*  Verwendung    : Calculates the difference between 2 BMP's                */
/*  Bibliotheken  :                                                          */
/*  Autor         : Dipl.Ing. Jrgen Dittmer                                 */
/*  Speichermodell: FLAT                                                     */
/*  Compiler      : WATCOM C/C++ v. 10.0                                     */
/*  System        : OS/2 4.0                                                 */
/*  Erstellung    : 19 Sep 1999                                              */
/*  nderung      :                                                          */
/*                                                                           */
/*****************************************************************************/

#define  INCL_PM                    /* required to use PM APIs.            */
#define  INCL_DOSFILEMGR            /* File System values */
#define  INCL_OS2MM                 /* required for MCI and MMIO headers   */
#define  INCL_MMIOOS2               /* required for MMVIDEOHEADER          */
#define  INCL_MMIO_CODEC            /* required for circular, secondary    */

#include <os2.h>
#include <os2me.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

PMMIMAGEHEADER pImageHdr, pImageHdr1, pImageHdr2;
PBYTE pImage, pImage1, pImage2; 
LONG             lNumBytesRead = 0;


/****************************************************************************/
/*                                                                          */
/*  FUNCTION   : main(int argc, char* const argv[])                         */
/*                                                                          */
/****************************************************************************/
int main(int argc, char* const argv[])
{
int  i;
int  rc;
char achBMP1[128] = ""; /* Filename Source BMP 1 */
char achBMP2[128] = ""; /* Filename Source BMP 2 */
char achDiff[128] = ""; /* Filename Result BMP   */
int  iWid = 0;          /* Blurr Width           */
BOOL SaveDiff  = FALSE; /* Save Diff BMP         */


	if (argc>1) 
		for (i=0;i<argc;i++)
	    {
			if ((argv[i][0]=='-') || (argv[i][0] == '/'))
	        {
		        switch (argv[i][1])
				{
				case '1': 
					strcpy(achBMP1,&(argv[i][3]));
					if (!strrchr(achBMP1,'.')) strcat(achBMP1,".BMP");
					break;
				case '2':
					strcpy(achBMP2,&(argv[i][3]));
					if (!strrchr(achBMP2,'.')) strcat(achBMP2,".BMP");
					break;
				case 'd':
					strcpy(achDiff,&(argv[i][3]));
					if (!strrchr(achDiff,'.')) strcat(achDiff,".BMP");
					SaveDiff = TRUE;
					break;
				case 'b':
					iWid  = atoi(&(argv[i][3]));
					break;
				}
			}
	    }
	else 
	{
		printf("BMPDiff v1.0, J. Dittmer 22 Sep 1999, adapted from ICOMPARE c't 19/1999\n"
			   "Useage: BMPDiff -1:<Image1> -2:<Image2> [-d:<DiffImage>] [-b:<Blurrwidth>]\n\n"
			   "  <Image1> and <Image2> are the filenames of the images to compare\n"			   
			   "  <DiffImage> is the filename of the result image\n"
			   "  <Blurrwidth> width of the blurr filter.\n"
			   "  - Blurrwidth has to be odd and between 3 and 15.\n"
			   "  - Blurrwidth and DiffImage are optional parameters.\n");
	    return (0);
	}

	rc = InitDIB(achBMP1);
    pImage1 = pImage;
    pImageHdr1 = pImageHdr;
	if (rc < 0) return (-100 + rc);
    printf("%s: Width=%d Heigth=%d BitsPerPixel=%d\n", achBMP1,
        pImageHdr1->mmXDIBHeader.BMPInfoHeader2.cx,
        pImageHdr1->mmXDIBHeader.BMPInfoHeader2.cy,
        pImageHdr1->mmXDIBHeader.BMPInfoHeader2.cBitCount);

	rc = InitDIB(achBMP2);
    pImage2 = pImage;
    pImageHdr2 = pImageHdr;
	if (rc < 0) return (-200 + rc);
    printf("%s: Width=%d Heigth=%d BitsPerPixel=%d\n", achBMP2,
        pImageHdr2->mmXDIBHeader.BMPInfoHeader2.cx,
        pImageHdr2->mmXDIBHeader.BMPInfoHeader2.cy,
        pImageHdr2->mmXDIBHeader.BMPInfoHeader2.cBitCount);

    if ((pImageHdr1->mmXDIBHeader.BMPInfoHeader2.cx != pImageHdr2->mmXDIBHeader.BMPInfoHeader2.cx) ||
        (pImageHdr1->mmXDIBHeader.BMPInfoHeader2.cy != pImageHdr2->mmXDIBHeader.BMPInfoHeader2.cy)) {
        printf("Both Images must have same format!\n");
        return (-300 + rc);
    } /* endif */

    if ((pImageHdr1->mmXDIBHeader.BMPInfoHeader2.cBitCount != 24) ||
        (pImageHdr2->mmXDIBHeader.BMPInfoHeader2.cBitCount != 24)) {
        printf("Images must have 24 BitsPerPixel!\n");
        return (-300 + rc);
    } /* endif */



    if (iWid) {
       printf("Blurring %s...\n", achBMP1);
	   rc = ApplyBlurr(pImage1,
                       pImageHdr1->mmXDIBHeader.BMPInfoHeader2.cx,
                       pImageHdr1->mmXDIBHeader.BMPInfoHeader2.cy,
                       iWid);
	   if (rc < 0) return (-100 + rc);

       printf("Blurring %s...\n", achBMP2);
	   rc = ApplyBlurr(pImage2,
                       pImageHdr2->mmXDIBHeader.BMPInfoHeader2.cx,
                       pImageHdr2->mmXDIBHeader.BMPInfoHeader2.cy,
                       iWid);
	   if (rc < 0) return (-200 + rc);
    }

	rc = CalcDifference (pImage1, pImage2, 
                         pImageHdr1->mmXDIBHeader.BMPInfoHeader2.cx,
                         pImageHdr1->mmXDIBHeader.BMPInfoHeader2.cy);

    printf("Difference=%d\n", rc);

    if (SaveDiff){
        printf("Writing Diff Image %s\n", achDiff);
        WriteDIB(achDiff);
    }

    if(pImageHdr1 != NULL) free(pImageHdr1);
    if(pImageHdr2 != NULL) free(pImageHdr2);
    if(pImage1 != NULL) free(pImage1);
    if(pImage2 != NULL) free(pImage2);

    if (rc < 0) printf("Error, RC=%d\n", rc);
    return (rc);
}


/******************************************************************************/
/*                                                                            */
/*  FUNCTION   : ApplyBlurr(PBYTE pImage, ULONG cx, ULONG cy, INT iBwid)      */
/*                                                                            */
/*  PURPOSE    : Given a Pointer to a DIB structure it will apply a blurr     */
/*               Filter to the DIB. NOTE : Only 24Bit DIBS are supported.     */
/*                                                                            */
/*  RETURNS    : The maximum brightness in the resulting DIB                  */
/*               Negative Results mark errors.								  */
/*                                                                            */
/******************************************************************************/

INT ApplyBlurr (PBYTE pImage, ULONG cx, ULONG cy, INT iBwid)
{
int  dwX, dwY;
int  iNoIdx;
int  i,j;
int  iHalfBwid;
long rgbwtAve[3];
int  maxVal = 0;
int  tVal;

    if (!(iBwid & 1)) iBwid++;  /* Muss ungerade sein */       
	if (iBwid > 15) iBwid = 15;    /* Max 15 */
    iNoIdx = iBwid * iBwid;
    iHalfBwid = iBwid / 2;

        for (dwY = 0; dwY < cy; dwY++) 
		{
            for (dwX = 0; dwX < cx; dwX++) 
			{
				rgbwtAve[0] = 0;
				rgbwtAve[1] = 0;
				rgbwtAve[2] = 0;

 	 			for (i = 0; i < iBwid; i++)
					for (j = 0; j < iBwid; j++) 
					{
						if ((dwX+i-iHalfBwid < 0) || (dwX+i-iHalfBwid >= cx) ||
							(dwY+j-iHalfBwid < 0) || (dwY+j-iHalfBwid >= cy)) 
						{
							rgbwtAve[0] += 127L;
							rgbwtAve[1] += 127L;
							rgbwtAve[2] += 127L;
						}
						else
						{
							rgbwtAve[0] += *(pImage + 3 * (dwX+i-iHalfBwid + cx*(dwY+j-iHalfBwid)));
							rgbwtAve[1] += *(pImage + 3 * (dwX+i-iHalfBwid + cx*(dwY+j-iHalfBwid)) + 1);
							rgbwtAve[2] += *(pImage + 3 * (dwX+i-iHalfBwid + cx*(dwY+j-iHalfBwid)) + 2);
						}
				}
				*(pImage + 3 * (dwX + cx * dwY))     = (rgbwtAve[0] / iNoIdx ) % 256;
				*(pImage + 3 * (dwX + cx * dwY) + 1) = (rgbwtAve[1] / iNoIdx ) % 256;
				*(pImage + 3 * (dwX + cx * dwY) + 2) = (rgbwtAve[2] / iNoIdx ) % 256;
				tVal = *(pImage + 3 * (dwX + cx * dwY)) +
                       *(pImage + 3 * (dwX + cx * dwY) + 1) +
                       *(pImage + 3 * (dwX + cx * dwY) + 2);
				if (tVal > maxVal) maxVal = tVal;
            }
		}
    return maxVal;
}


/***********************/
/* Write a BMP to disk */
/***********************/

int WriteDIB(const char* FileName)
{
HFILE             hBMP;
BITMAPFILEHEADER2 bmph;
char              achFileName[128];
ULONG             ulAction;
ULONG             cBytes;    

    strcpy(achFileName,FileName);
    memset (&bmph, 0, sizeof(BITMAPFILEHEADER2));
    bmph.usType               = 0x4D42;    /* BM */
    bmph.cbSize               = sizeof(BITMAPFILEHEADER2);
    bmph.offBits              = sizeof(BITMAPFILEHEADER2);
    bmph.bmp2.cbFix           = sizeof(BITMAPINFOHEADER2);
    bmph.bmp2.cPlanes         = 1;
    bmph.bmp2.cBitCount       = 24;
    bmph.bmp2.cx              = pImageHdr2->mmXDIBHeader.BMPInfoHeader2.cx;
    bmph.bmp2.cy              = pImageHdr2->mmXDIBHeader.BMPInfoHeader2.cy;
    bmph.bmp2.ulCompression   = BCA_UNCOMP;
    bmph.bmp2.usRecording     = BRA_BOTTOMUP;
    bmph.bmp2.usRendering     = BRH_NOTHALFTONED;
    bmph.bmp2.ulColorEncoding = BCE_RGB;

    DosOpen (achFileName, &hBMP, &ulAction, 0, FILE_NORMAL,     
                   FILE_CREATE | OPEN_ACTION_REPLACE_IF_EXISTS,
                   OPEN_ACCESS_WRITEONLY | OPEN_SHARE_DENYWRITE,
                   0L);
    DosWrite (hBMP, &bmph, sizeof(BITMAPFILEHEADER2), &cBytes);
    DosWrite (hBMP, (PVOID)pImage2, bmph.bmp2.cx * bmph.bmp2.cy * 3, &cBytes);
    DosClose (hBMP);

    return (0);
}


int CalcDifference (PBYTE pImage1, PBYTE pImage2, ULONG cx, ULONG cy)
{
ULONG isize = cx * cy * 3;  /* Image size in bytes */
ULONG i;
int   maxDiff = 0;
int   minDiff = 1000;
int   tDiff;
int   actDiff;

    for (i = 0; i < isize; i+=3) {
        actDiff=0;
        tDiff = *(pImage1 + i) - *(pImage2 + i);
        if (tDiff < 0) actDiff -= tDiff; else actDiff += tDiff;
        tDiff = *(pImage1 + i + 1) - *(pImage2 + i + 1);
        if (tDiff < 0) actDiff -= tDiff; else actDiff += tDiff;
        tDiff = *(pImage1 + i + 2) - *(pImage2 + i + 2);
        if (tDiff < 0) actDiff -= tDiff; else actDiff += tDiff;

        if (actDiff < minDiff) minDiff = actDiff;
        if (actDiff > maxDiff) maxDiff = actDiff;

        *(pImage2 + i)     = actDiff/3;
        *(pImage2 + i + 1) = actDiff/3;
        *(pImage2 + i + 2) = actDiff/3;
    } /* endfor */

    return (maxDiff-minDiff);
}


int InitDIB(const char* FileName)
{
ULONG ulRc;
LONG             lRc = 0;
HMMIO            hmmioFrom = 0;
MMIOINFO         mmioinfo;
PCHAR            pBuffer = 0;
LONG             lLen = 0;
//LONG             lNumBytesRead = 0;
char             achFileName[128];


   strcpy(achFileName,FileName);

   /* Open the FROM file with header translation  */
   memset(&mmioinfo, '\0', sizeof(MMIOINFO));
   mmioinfo.ulTranslate = MMIO_TRANSLATEHEADER+MMIO_TRANSLATEDATA;
   mmioinfo.aulInfo[3] = MMIO_MEDIATYPE_IMAGE;
   hmmioFrom = mmioOpen(achFileName, &mmioinfo, MMIO_READ);
   /* Check for error */
   if (!hmmioFrom) {
      /* error */
      printf("Error, mmioOpen failed.\n");
      return (-1);
   } 

   /* Query the length of the header */
   ulRc = mmioQueryHeaderLength(hmmioFrom, &lLen, 0,0);
   if (ulRc) {
      /* error */
      printf("Error, mmioQueryHeaderLength failed.\n");
      return (-1);
   }

   /* Allocate size of header */
   pBuffer = (PCHAR) malloc (lLen);
   pImageHdr = (PMMIMAGEHEADER)pBuffer;   

   /* Get the movie header */
   ulRc = mmioGetHeader(hmmioFrom, pBuffer, lLen, &lNumBytesRead, 0,0);
   /* Check for error */
   if (ulRc) {
      /* error */
      printf("Error, mmioGetHeader failed.\n");
      return (-1);
   }

   /* Allocate image buffer */
   lLen = (LONG) pImageHdr->mmXDIBHeader.BMPInfoHeader2.cbImage;
   pImage = (PBYTE) malloc (lLen);

   /* Get the image from the FROM file */
   lRc = mmioRead(hmmioFrom, pImage, lLen);
   
   /* Check for error */
   if (lRc == MMIO_ERROR) {
      /* error */
      printf("Error, mmioRead failed.\n");
      return (-1);
   } 

   return (0);
} 
