/*****************************************************************************
 *                                                                          
 * This software module was originally developed by 
 *
 * J. Ignacio Ronda (UPM-GTI / ACTS-MoMuSys)
 *                                                      
 * and edited by                                        
 *                                                      
 * Martina Eckert (UPM-GTI / ACTS-MoMuSys)
 * Fernando Jaureguizar (UPM-GTI / ACTS-MoMuSys)
 *
 * in the course of development of the MPEG-4 Video (ISO/IEC 14496-2) standard.
 * This software module is an implementation of a part of one or more MPEG-4 
 * Video (ISO/IEC 14496-2) tools as specified by the MPEG-4 Video (ISO/IEC 
 * 14496-2) standard. 
 *
 * ISO/IEC gives users of the MPEG-4 Video (ISO/IEC 14496-2) standard free 
 * license to this software module or modifications thereof for use in hardware
 * or software products claiming conformance to the MPEG-4 Video (ISO/IEC 
 * 14496-2) standard. 
 *
 * Those intending to use this software module in hardware or software products
 * are advised that its use may infringe existing patents. The original 
 * developer of this software module and his/her company, the subsequent 
 * editors and their companies, and ISO/IEC have no liability for use of this 
 * software module or modifications thereof in an implementation. Copyright is 
 * not released for non MPEG-4 Video (ISO/IEC 14496-2) Standard conforming 
 * products. 
 *
 * ACTS-MoMuSys partners retain full right to use the code for his/her own 
 * purpose, assign or donate the code to a third party and to inhibit third 
 * parties from using the code for non MPEG-4 Video (ISO/IEC 14496-2) Standard
 * conforming products. This copyright notice must be included in all copies or
 * derivative works. 
 *
 * Copyright (c) 1997
 *
 *****************************************************************************/

/***********************************************************HeaderBegin*******
 *                                                                         
 * File:        rc_if.c
 *
 * Author:      J. Ignacio Ronda, UPM-GTI
 * Created:     18-06-97
 *                                                                         
 * Description: Generic rate control interface
 *
 * Flags:       -D_RC_DEBUG_  -  RC debugging   
 *
 * Modified:
 *      11.11.97  Martina Eckert: Headers, comments, cleaning
 *      27.11.97  M. Eckert: Changes for independent frame type control Q2
 *      15.12.97  M. Eckert: Extraction of RC parameters from one single string
 *      27.01.98  M. Eckert and F. Jaureguizar: RC_QuantAdjust() modifications.
 *
 ***********************************************************HeaderEnd*********/

/************************    INCLUDE FILES    ********************************/

#include "momusys.h"
#include "mom_vop.h"
#include "vm_config.h"
#include "mom_vol.h"
#include "mom_structs.h"
#include "mom_vo.h"
#include "vm_enc_defs.h"

#include "rc.h"

/************************ Static variables ************************/

static Int rc_algorithm;     /* Rate control algorithm */
static Int rc_rate_model;    /* Statistical model for rate pred. */
static Int rc_dist_model;    /* Statistical model for distortion pred. */
static Int mode;
static Int rc_init;          /* Init flag */
static Int R_glob_target;    /* Global target bitrate */           

static BUFFER      glob_buf; /* Rate control buffer */
static RC_TB_PARS  pars;     /* Number of bits for next frame */

/*static Int RCQ2_SepMotShapeText;*/


/***********************************************************CommentBegin******
 *
 * -- RC_init_global --
 *
 * Author : 
 *      J. Ignacio Ronda, UPM-GTI     
 *
 * Created :            
 *      18-06-97
 *
 * Purpose :
 *
 *      Initializes module.
 * 
 * Arguments in :       
 *      VOConfig *list              -  VO data 
 *      Int       rc_algorithm_use  -  RC algorithm 
 *      Int       rc_rate_model_use -  RC model 
 *      Int       rc_dist_model_use -  RC model 
 *      Int       R                 -  Global target rate 
 *      Int       size              -  Global buffer size 
 *
 * Arguments in/out :   
 *      -
 *
 * Arguments out :      
 *      -
 *
 * Return values :      
 *      -
 *
 * Side effects :       
 *      -
 *
 * Description :        
 *
 *
 * See also :
 *
 *
 * Modified : 
 *      15.12.97 M. Eckert: Extraction of RC parameters from one single string
 *
 *
 ***********************************************************CommentEnd********/

void RC_init_global(
                    VOConfig *list,               /* VO data */
		    Char     *rc_ctl_string,
		    Int      *rc_algorithm_used
		    )
{
   VOConfig  *vo_config;
   VolConfig *vol_config;
   Int    bits_frame;
   Int    R_target, glob_buffer_size;
   /*   Int    mode;*/

#ifdef _RC_DEBUG_
   fprintf(stdout, "RC: RC_IF: --> RC_init_global\n");
   fflush(stdout);
#endif

   vo_config = list;

   rc_init_ctl_parameters(rc_ctl_string, 
			  &rc_algorithm, 
			  &rc_rate_model, 
			  &rc_dist_model,
			  &R_target,
			  &glob_buffer_size,
			  &mode);

#ifdef _RC_DEBUG_
   fprintf(stdout, "alg = %d, rate_mo = %d, dist_mo = %d, R_t = %d, \
buf = %d, mode = %d\n",rc_algorithm, 
			 rc_rate_model, 
			 rc_dist_model,
			 R_target,
			 glob_buffer_size,
			 mode);
#endif

   *rc_algorithm_used = rc_algorithm;

  if (rc_init)
     error_exit("RC_init_global: Error: Already initialized\n");
  rc_init = 1;


  vol_config = GetVOConfigLayers(vo_config);

  /*  RCQ2_SepMotShapeText = GetVolConfigSepMotShapeText(vol_config);*/


  bits_frame = (Int)(R_target / GetVolConfigFrameRate(vol_config));

#ifdef _RC_DEBUG_
  fprintf(stdout,"bit_rate = %d,  frame rate = %f \n", 
          R_target, GetVolConfigFrameRate(vol_config));
  fprintf(stdout,">>>>> Init global bit_rate_frame = %d \n",
          bits_frame); fflush(stdout);
  fprintf(stdout,">>>>> Init global buffer size = %d \n", glob_buffer_size); fflush(stdout);
#endif

  /*  buffer_init(&glob_buf, glob_buffer_size, glob_buffer_size/2, bits_frame);*/
   buffer_init(&glob_buf, glob_buffer_size, 0, bits_frame);

  
  switch (rc_algorithm)
     {
     case RC_ALGORITHM_Q2:
       RCQ2_init(list);
       break;
     case RC_ALGORITHM_UPM1: 
#ifdef _RC_DEBUG_
       fprintf(stdout,">>>>> Init total bits/frame = %d \n", bits_frame); fflush(stdout);
#endif
       rc_total_bits_init_upm(&pars, bits_frame, glob_buffer_size);
       RC_UPM_init(list, rc_rate_model, rc_dist_model, 
		   rc_algorithm, bits_frame); 
       break;
     case RC_ALGORITHM_UPM2: 
       rc_total_bits_init_upm(&pars, bits_frame, glob_buffer_size);
       RC_UPM_init(list, rc_rate_model, rc_dist_model, 
		   rc_algorithm, bits_frame); 
       break;
     default:
       error_exit("RC_init_global: Error: Mode not supported\n");
     }

#ifdef _RC_DEBUG_
   fprintf(stdout, "RC: RC_IF: <-- RC_init_global\n");
   fflush(stdout);
#endif

}

/***********************************************************CommentBegin******
 *
 * -- RC_QuantAdjust --
 *
 * Author : 
 *      J. Ignacio Ronda, UPM-GTI     
 *
 * Created :            
 *      18-06-97
 *
 * Purpose :
 *
 *      Interface function for call of different adjust algorithms
 * 
 * Arguments in :       
 *      Int      vo_id        -  VO Id
 *      Int      vol_id       -  VOL Id
 *      VOConfig *vo_list     -  Access to all VOLs
 *      VolConfig *vol_config -  To save calculated QP
 *      Float    mad          -  Current MAD
 *      Vop      *curr        -  Current VOP
 *      Vop      *error_vop   -  Error prediction VOP
 *      UInt     num_pels_vop -  Number of pixels in object 
 *      Int      frame_type   -  I=0, P=1, B=2
 *
 * Arguments in/out :   
 *      -
 *
 * Arguments out :      
 *      -
 *
 * Return values :      
 *      Int    qp  -  Quantization parameter
 *
 * Side effects :       
 *      -
 *
 * Description :        
 *
 *
 * See also :
 *
 *
 * Modified : 
 *      27.11.97  M. Eckert: Changes for independent frame type control
 *      05.12.97  M. Eckert: rch_set_process_flag --> Quant_Adjust
 *      27.01.98  F. Jaureguizar and M. Eckert: New parameters error_vop and 
 *                vol_config. Now the new QP is set in the curr and error_vop
 *                VOPs and in the vol_config inside this routine. This setting
 *                is necessary in other higher level routines:
 *                  The QP from error_vop is used to code the MBs.
 *                  The QP from curr is used to send to the bitstream.
 *                  The QP from vol_config is used to print the value.
 *
 ***********************************************************CommentEnd********/

Int RC_QuantAdjust(
                   Int       vo_id,
                   Int       vol_id,
                   VOConfig  *vo_list,    /* Global RC */
                   VolConfig *vol_config,
                   Float     mad,
                   Vop       *curr,
                   Vop       *error_vop,
                   UInt      num_pels_vop, /* Number of pixels in object */
                   Int       frame_type
                   )
{
   Int       qp=2;

#ifdef _RC_DEBUG_
   fprintf(stdout, "RC: RC_IF: --> RC_QuantAdjust: mad= %f\n", mad);
   fflush(stdout);
#endif


   if (!rc_init)
      error_exit("RC_QuantAdjust: Error: Module not initialized\n");

   /* For global control: */

  if (rc_algorithm != RC_ALGORITHM_Q2)
   {

     /*
       R_glob_target = rc_total_bits_upm(&pars, buffer_get_occup(&glob_buf));
       */
     R_glob_target = rc_total_bits_upm(&pars, vo_list, 
				       buffer_get_occup(&glob_buf)); 
   
#ifdef _RC_DEBUG_
      fprintf(stdout,">>>>> R_glob_target = %d \n", R_glob_target ); 
      fflush(stdout);
#endif
   }

  switch (rc_algorithm)
     {
     case RC_ALGORITHM_Q2:
        qp = RCQ2_QuantAdjust(vo_id, vol_id, 
			      (Double)mad, 
			      num_pels_vop,
			      frame_type);  /* RC-BIP */
        break;

     case RC_ALGORITHM_UPM1: /* Minimization of SUM alpha[i] * D[i] */

        fprintf(stdout,"Global RC: Minimization Algorithm \n");

        qp = RC_UPM1_QuantAdjust(vo_id, vol_id, 
                                 vo_list, 
                                 (Double) R_glob_target, 
                                 (Double)mad, 
                                 num_pels_vop, 
                                 rc_rate_model, 
                                 rc_dist_model,
				 mode,
				 frame_type);
        break;
       
     case RC_ALGORITHM_UPM2: /* n VOP's, ensuring minimum quality for the most 
                                important one.
                                A list of importance is provided by the user 
                                and used as indices. */
        fprintf(stdout,"Global RC: Priority Algorithm \n");

        qp = RC_UPM2_QuantAdjust(vo_id, vol_id, 
                                 vo_list, 
                                 (Double) R_glob_target, 
                                 (Double)mad, 
                                 num_pels_vop, 
                                 rc_rate_model, 
                                 rc_dist_model,
				 mode, 
				 frame_type);
        break;
         
     case RC_ALGORITHM_UPM3:    /* Constant D_i ratios (srl) */
        fprintf(stdout,"Algorithm UPM3 not yet supported \n");
        exit(1);
        break;
        
     default:
        error_exit("RC_QuantAdjust: Error: Mode not supported\n");
     }

#ifdef _RC_DEBUG_
  fprintf(stdout, "RC: RC_IF: <-- RC_QuantAdjust: qp= %d\n", qp);
  fflush(stdout);
#endif

  if (frame_type==I_VOP)
    {
      PutVopIntraQuantizer(qp, curr);
      PutVolConfigIntraQuantizer(qp, vol_config);
    }
  else if (frame_type==P_VOP)
    {
      PutVopQuantizer(qp, curr);
      PutVopQuantizer(qp, error_vop);
      PutVolConfigQuantizer(qp, vol_config);
    }
  else if (frame_type==B_VOP)
    {
      PutVopBQuantizer(qp, curr);
      PutVopBQuantizer(qp, error_vop);
      PutVolConfigBQuantizer(qp, vol_config);
    }
  
  return qp;
}

/***********************************************************CommentBegin******
 *
 * -- RC_FrameSkipping --
 *
 * Author : 
 *      J. Ignacio Ronda, UPM-GTI     
 *
 * Created :            
 *      18-06-97
 *
 * Purpose :
 *      Interface function for different frame skipping functions
 * 
 * Arguments in :       
 *      Int      vo_id        -  VO Id
 *      Int      vol_id       -  VOL Id
 *
 * Arguments in/out :   
 *      -
 *
 * Arguments out :      
 *      -
 *
 * Return values :      
 *      Int     ret           -  Skip flag
 *
 * Side effects :       
 *      -
 *
 * Description :        
 *
 *
 * See also :
 *
 *
 * Modified : 
 *
 *
 ***********************************************************CommentEnd********/

Int RC_FrameSkipping(
                     Int vo_id,
                     Int vol_id
                     )
{

   Int ret=-1;

#ifdef _RC_DEBUG_
   fprintf(stdout, "RC: RC_IF: --> RC_FrameSkipping\n");
   fflush(stdout);
#endif

  if (!rc_init)
     error_exit("RC_FrameSkipping: Error: Module not initialized\n");

  switch (rc_algorithm)
     {
     case RC_ALGORITHM_Q2:
        ret = RCQ2_FrameSkipping(vo_id, vol_id);
        break;
     case RC_ALGORITHM_UPM1:   
        ret = 0;
        /* It's done globally in vm_enc_main (RC_GlobalFrameSkipping) */
        break;
     case RC_ALGORITHM_UPM2:   
        ret = 0;
        /* It's done globally in vm_enc_main (RC_GlobalFrameSkipping) */
        break;
     default:
       error_exit("RC_FrameSkipping: Error: Mode not supported\n");
     }

#ifdef _RC_DEBUG_
   fprintf(stdout, "RC: RC_IF: <-- RC_FrameSkipping: ret= %d\n", ret);
   fflush(stdout);
#endif

return ret;
}

/***********************************************************CommentBegin******
 *
 * -- RC_UpdateAfterSkipping --
 *
 * Author : 
 *      J. Ignacio Ronda, UPM-GTI     
 *
 * Created :            
 *      18-06-97
 *
 * Purpose :
 *      Interface function for different model update functions
 * 
 * Arguments in :       
 *      Int      vo_id        -  VO Id
 *      Int      vol_id       -  VOL Id
 *      Int      frame_type   -  I=0, P=1, B=2
 *
 * Arguments in/out :   
 *      -
 *
 * Arguments out :      
 *      -
 *
 * Return values :      
 *      -
 *
 * Side effects :       
 *      -
 *
 * Description :        
 *
 *
 * See also :
 *
 *
 * Modified : 
 *      27.11.97  M. Eckert: Changes for independent frame type control
 *
 *
 ***********************************************************CommentEnd********/

void RC_UpdateAfterSkipping(
                            Int vo_id,
                            Int vol_id,
			    Int frame_type
                            )
{
#ifdef _RC_DEBUG_
   fprintf(stdout, "RC: RC_IF: --> RC_UpdateAfterSkipping\n");
   fflush(stdout);
#endif

  if (!rc_init)
     error_exit("RC_UpdateAfterSkipping: Error: Module not initialized\n");

  switch (rc_algorithm)
     {
     case RC_ALGORITHM_Q2:
        RCQ2_UpdateAfterSkipping(vo_id, vol_id, frame_type);
        break;
     case RC_ALGORITHM_UPM1:   
        /* Update in RC_GlobalFrameSkipping, called in vm_enc_main */
        break;
     case RC_ALGORITHM_UPM2:   
        /* Update in RC_GlobalFrameSkipping, called in vm_enc_main */
        break;
     default:
        error_exit("RC_UpdateAfterSkipping: Error: Mode not supported\n");
     }

#ifdef _RC_DEBUG_
   fprintf(stdout, "RC: RC_IF: <-- RC_UpdateAfterSkipping\n");
   fflush(stdout);
#endif
}

/***********************************************************CommentBegin******
 *
 * -- RC_ExcludeIFrame --
 *
 * Author : 
 *      J. Ignacio Ronda, UPM-GTI     
 *
 * Created :            
 *      18-06-97
 *
 * Purpose :
 *      Interface function for different exclude functions for I-Frames
 * 
 * Arguments in :       
 *      Int      vo_id        -  VO Id
 *      Int      vol_id       -  VOL Id
 *      Int      vopbits      -  Bits needed to code this VOP
 *
 * Arguments in/out :   
 *      -
 *
 * Arguments out :      
 *      -
 *
 * Return values :      
 *      -
 *
 * Side effects :       
 *      -
 *
 * Description :        
 *
 *
 * See also :
 *
 *
 * Modified : 
 *
 *
 ***********************************************************CommentEnd********/

void RC_ExcludeIFrame(
                      Int vo_id,
                      Int vol_id,
                      Int vopbits
                      )
{
  if (!rc_init)
     error_exit("RC_ExcludeIFrame: Error: Module not initialized\n");

  switch (rc_algorithm)
     {
     case RC_ALGORITHM_Q2:
        RCQ2_ExcludeIFrame(vo_id, vol_id, vopbits);
        break;
     case RC_ALGORITHM_UPM1:
        break;
     case RC_ALGORITHM_UPM2:
        break;
     default:
        error_exit("RC_ExcludeIFrame: Error: Mode not supported\n");
     }
}

/***********************************************************CommentBegin******
 *
 * -- RC_UpdateModel --
 *
 * Author : 
 *      J. Ignacio Ronda, UPM-GTI     
 *
 * Created :            
 *      18-06-97
 *
 * Purpose :
 *      Interface function for different update functions for 
 *      rate and distortion models.
 * 
 * Arguments in :       
 *      Int      vo_id        -  VO Id
 *      Int      vol_id       -  VOL Id
 *      Int      vopbits      -  Total number of bits
 *      Int      vopheader    -  Total minus texture
 *      Double   dist         -  Distortion of VOP
 *      Int      frame_type   -  I=0, P=1, B=2
 *
 * Arguments in/out :   
 *      -
 *
 * Arguments out :      
 *      -
 *
 * Return values :      
 *      -
 *
 * Side effects :       
 *      -
 *
 * Description :        
 *
 *
 * See also :
 *
 *
 * Modified : 
 *      27.11.97  M. Eckert: Changes for independent frame type control
 *      05.12.97  M. Eckert: Pass rc_algorithm to UPM_Update_model to access
 *                           history data
 *
 *
 ***********************************************************CommentEnd********/

void RC_UpdateModel(
                    Int vo_id,
                    Int vol_id,
                    Int vopbits,   /* Total number of bits */
                    Int vopheader, /* Total minus texture */
                    Double dist,
		    Int frame_type
                    )
{

  if (!rc_init)
     error_exit("RC_UpdateModel: Error: Module not initialized\n");
  if (rc_algorithm == RC_ALGORITHM_Q2) {
   /*
    if (RCQ2_SepMotShapeText)
        RCQ2_Update2OrderModel(vo_id, vol_id, vopbits, vopheader, dist, 
			       frame_type);
     else
     */
  RCQ2_Update2OrderModel(vo_id, vol_id, vopbits, 0, dist, frame_type);
       
  }
  else
  {
     switch (rc_rate_model)
        {
        case RC_MODEL_UPM1: 
           RC_UPM_UpdateModel(vo_id, vol_id, vopbits, vopheader, dist, 
			      rc_algorithm, rc_rate_model, 1, frame_type); 
           break;
#ifdef _RC_UPM2_
        case RC_MODEL_UPM2: 
           RC_UPM_UpdateModel(vo_id, vol_id, vopbits, vopheader, dist, 
			      rc_algorithm, rc_rate_model, 1, frame_type); 
           break;
#endif
        default:
           error_exit("RC_UpdateModel: Error: Mode not supported\n");
        }

     switch (rc_dist_model)
        {
        case RC_MODEL_UPM1: 
	   RC_UPM_UpdateModel(vo_id, vol_id, vopbits, vopheader, dist, 
			      rc_algorithm, rc_dist_model, 2, frame_type); 
           break;
#ifdef _RC_UPM2_
        case RC_MODEL_UPM2: 
           RC_UPM_UpdateModel(vo_id, vol_id, vopbits, vopheader, dist, 
			      rc_algorithm, rc_dist_model, 2, frame_type); 
           break;
#endif
	default:
           error_exit("RC_UpdateModel: Error: Mode not supported\n");
        }
  }
}

/***********************************************************CommentBegin******
 *
 * -- RC_UpdateBuffer --
 *
 * Author : 
 *      J. Ignacio Ronda, UPM-GTI     
 *
 * Created :            
 *      18-06-97
 *
 * Purpose :
 *     Actualization of global rate control buffer. Counts sum of bits for all
 *     VOLs of one frame.
 * 
 * Arguments in :       
 *      VOConfig *vo_list      -  VO list
 *      BitCount num_bits[][]  -  number of bits for VO's
 *
 * Arguments in/out :   
 *      -
 *
 * Arguments out :      
 *      -
 *
 * Return values :      
 *      -
 *
 * Side effects :       
 *      -
 *
 * Description :        
 *
 *
 * See also :
 *
 *
 * Modified : 
 *
 *
 ***********************************************************CommentEnd********/

Int RC_UpdateBuffer(
                    VOConfig *vo_list,
                    BitCount num_bits[MAX_NUM_VOS][MAX_NUM_VOLS]
                    )
{
   VOConfig  *curr_vo;
   VolConfig *curr_vol;
   Int vo_i, vol_i, bit_sum=0;

#ifdef _RC_DEBUG_
   Int frame;
#endif

   curr_vo = vo_list;

   while(curr_vo != NULL)
   {
      vo_i = GetVOConfigId(curr_vo);
      curr_vol =  GetVOConfigLayers(curr_vo);
#ifdef _RC_DEBUG_
      /* just for debug print: */ 
      frame = curr_vol->frame + GetVolConfigM(curr_vol)*curr_vol->frame_skip-2;
#endif
      while(curr_vol != NULL)
      {
         vol_i = GetVolConfigId(curr_vol);
         bit_sum += (Int)num_bits[vo_i][vol_i].vop;

         /* Reset bit counts for the next time this VOL is coded: */
         num_bits[vo_i][vol_i].vop = 0;

         curr_vol = GetVolConfigNext(curr_vol);
      }
      curr_vo = GetVOConfigNext(curr_vo);
   }

#ifdef _RC_DEBUG_
   WRITE_INT_INT("bits_sum.b", frame, bit_sum);
#endif

   buffer_put(&glob_buf, bit_sum);

   return 0;
}

/***********************************************************CommentBegin******
 *
 * -- RC_GlobalFrameSkipping --
 *
 * Author : 
 *      Martina Eckert, UPM-GTI     
 *
 * Created :            
 *      18-06-97
 *
 * Purpose :
 *      Frame skipping for global rate control
 *
 * Arguments in :       
 *      VOConfig *vo_list     -  VO list
 *
 * Arguments in/out :   
 *      -
 *
 * Arguments out :      
 *      -
 *
 * Return values :      
 *      Int     ret           -  Skip flag
 *
 * Side effects :       
 *      -
 *
 * Description :        
 *
 *
 * See also :
 *      Sets Skip_flag for all VOLs at the beginning of one frame if the buffer 
 *      is full.
 *      Resets flag for update after skipping.
 *
 * Modified : 
 *
 *
 ***********************************************************CommentEnd********/

Int RC_GlobalFrameSkipping(
                           VOConfig *vo_list
                           )
{
   VOConfig  *curr_vo;
   VolConfig *curr_vol;
   Int vo_i,  vol_i;
   BUFFER    *buffer = &glob_buf;
   Int        frame=0;
   Float      time_inc, new_time;
 
   if (buffer->occup > SKIP_MARGIN * buffer->size)
   {

#ifdef _RC_DEBUG_
      fprintf(stdout,
              "RC: RC_GlobalFrameSkipping: Buffer fullness: %d of %d\n",
              buffer->occup, buffer->size);
      fprintf(stdout,"Set Skip flag for all VOLs! \n");
      fflush(stdout);
#endif

      /* Increment frame and time count for all VOLs: */
      curr_vo = vo_list;
      while(curr_vo != NULL)
      {
         vo_i = GetVOConfigId(curr_vo);
         curr_vol = GetVOConfigLayers(curr_vo);
         while(curr_vol != NULL)
         {
            vol_i = GetVolConfigId(curr_vol);
            frame = GetVolConfigFrame(curr_vol) + GetVolConfigFrameSkip(curr_vol);
            PutVolConfigFrame(frame, curr_vol);

            time_inc = GetVolConfigTimeInc(curr_vol);
            new_time = curr_vol->next_coding_time;
            new_time += time_inc*(Float)GetVolConfigM(curr_vol);
            new_time = RoundTime(new_time);
            curr_vol->next_coding_time = new_time;

            rcd_inc_skipped(vo_i, vol_i);

            curr_vol = GetVolConfigNext(curr_vol);
         }
         curr_vo = GetVOConfigNext(curr_vo);
      }
      fprintf(stdout,
              "....................Frame Skipping: newframe =%d\n",frame);

      buffer_put(&glob_buf, 0);

      return 1;

   }

   return 0;

}
/***********************************************************CommentBegin******
 *
 * -- RC_Free --
 *
 * Purpose :
 *      Free reserved memory for RC
 * 
 * Arguments in :       
 *	     VOConfig  *vo_list      - List of VOs
 *	     Int       rc_algorithm  - type of algorithm
 *
 * Created : 
 *      03.02.98   
 *
 * Modified: 19.02.98 M.Eckert: Inclusion of Q2
 *
 ***********************************************************CommentEnd********/

Void RC_Free(
	     VOConfig *vo_list,
	     Int      rc_algorithm
)
{
  switch (rc_algorithm)
    {
    case RC_ALGORITHM_Q2: 
      RCQ2_Free(vo_list);
      break;
    case RC_ALGORITHM_UPM1: 
    case RC_ALGORITHM_UPM2: 
      RC_UPM_Free(vo_list, rc_algorithm);
      break;
    default:
      error_exit("RC_Free: Error: Mode not supported\n");
    }

}

/*********************************************************** End of file ***/
