//
//                     TxWin, Textmode Windowing Library
//
//   Original code Copyright (c) 1995-2021 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   TxLib, released under MIT License
//
//   Copyright (c) 1995-2021  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on TxWin licensing can be directed to: info@dfsee.com
//
// ==========================================================================
//
// TxLib Unicode string handling
//
// JvW  28-02-2021 LICENSING: Changed from LGPL to the more liberal MIT license

#include <txlib.h>                              // TxLib interface

/*****************************************************************************/
// Convert a Mac-Unicode Pascal-style string to an ASCII 'C' string
// Will replace non-Ascii into '*' and 0x0000 into '.' (HFS+ special files ?)
/*****************************************************************************/
char *TxMacUniStr2Ascii                         // RET   ptr to ASCII string
(
   S_MACUNI_STRING    *macuni,                  // IN    Mac-Unicode string part
   char               *asc                      // INOUT ASCII string
)
{
   char               *pos = asc;               // no append, start at begin
   USHORT              len = TxBE16( macuni->length);
   USHORT              i;
   USHORT              uch;                     // unicode character

   for ( i = 0; i < len; i++)                   // always handle ALL characters
   {
      uch = TxBE16( macuni->unicode[i]);
      if (uch > 0xff)
      {
         if (uch != 0xffff)
         {
            *pos++ = '*';                       // to be refined
         }
      }
      else if (uch == 0)                        // Zero, possibly leading ?
      {
         *pos++ = '.';                          // replace by DOT
      }
      else                                      // ASCII subset
      {
         *pos++ = (char) (uch & 0xff);
      }
   }
   *pos = '\0';                                 // terminate the string
   return (asc);
}                                               /* end TxMacUnicode2Ascii    */
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Convert an ASCII 'C' string into a Mac-Unicode Pascal-style string
/*****************************************************************************/
USHORT TxAscii2MacUniStr                        // RET   length of MAC string
(
   char               *asc,                     // IN    ASCII string
   S_MACUNI_STRING    *macuni                   // INOUT Mac-Unicode string part
)
{
   USHORT              len = min( strlen( asc), MACUNI_MAX_LEN);
   USHORT              i;
   USHORT              uch;                     // unicode character

   macuni->length = TxBE16( len);
   for ( i = 0; i < len; i++)                   // upto calculated length
   {
      uch = asc[i];
      macuni->unicode[i] = TxBE16( uch);
   }

   return (len);
}                                               /* end TxAscii2MacUniStr    */
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Append translated Unicode string-fragment to an ASCII string, detect BE/LE
// For now it only processes ASCII equivalents where one byte is always ZERO!
/*****************************************************************************/
char *TxUnicAppend                              // RET   ptr to ASCII string
(
   USHORT             *uni,                     // IN    Unicode string part
   char               *asc,                     // INOUT ASCII string
   USHORT              len                      // IN    maximum Unicode length
)
{
   char               *pos = asc + strlen( asc); // append position
   USHORT              i;

   for (i = 0; i < len && uni[i] != 0; i++)     // until len or zero-unicode
   {
      if (uni[i] > 0xff)
      {
         if (uni[i] & 0xff)                     // both bytes non-zero,
         {
            if (uni[i] != 0xffff)
            {
               *pos++ = '*';                    // to be refined, use tables ?
            }
         }
         else                                   // assume big-endian
         {
            *pos++ = (char) (uni[i] >> 8);      // take upper byte
         }
      }
      else                                      // assume little-endian
      {
         *pos++ = (char) (uni[i] & 0xff);       // take lower byte
      }
   }
   *pos = '\0';                                 // terminate the string
   return (asc);
}                                               /* end TxUnicAppend          */
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Detect if input is Unicode or ASCII, Append fragment to an ASCII string
/*****************************************************************************/
char *TxUnicAsciiAppend                         // RET   ptr to ASCII string
(
   USHORT             *uni,                     // IN    Unicode or ASCII string
   char               *asc,                     // INOUT ASCII string
   USHORT              len                      // IN    max INPUT length, bytes
)
{
   if (TxUnicAsciiLength(   uni, len) > 3)
   {
      TXTM    txt;
      TxCopy( txt, (char *) uni, len);
      strcat( asc, txt);
   }
   else
   {
      TxUnicAppend( uni, asc,    len / 2);      // each unicode char is 2 bytes!
   }
   return (asc);
}                                               /* end TxUnicAsciiAppend     */
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Perform a Unicode to ASCII case-sensitive stringcompare (aka strncmp)
/*****************************************************************************/
int TxUnicStrncmp                               // RET   compare result
(
   char               *uni,                     // IN    Unicode string
   char               *asc,                     // IN    ASCII   string
   size_t              len                      // IN    max ASCII comp-length
)
{
   size_t              i;
   int                 rc = 0;

   for ( i = 0; (rc == 0) && (i < len) && *asc; i++, asc++)
   {
      rc = *uni - *asc;                         // compare 1st byte, CASE SENSITIVE
      uni++;
      if (rc == 0)                              // first byte equal (ascii)
      {
         rc = *uni;                             // second byte should be ZERO
      }
      uni++;
   }
   return (rc);
}                                               /* end TxUnicStrncmp         */
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Perform a Unicode to ASCII case-insensitive stringcompare (aka strncasecmp)
/*****************************************************************************/
int TxUnicStrnicmp                              // RET   compare result
(
   char               *uni,                     // IN    Unicode string
   char               *asc,                     // IN    ASCII   string
   size_t              len                      // IN    max ASCII comp-length
)
{
   size_t              i;
   int                 rc = 0;

   for ( i = 0; (rc == 0) && (i < len) && *asc; i++, asc++)
   {
      rc = ((*uni) & 0xdf) - ((*asc) & 0xdf);   // compare 1st byte, CASE INSENSITIVE
      uni++;
      if (rc == 0)                              // first byte equal (ascii)
      {
         rc = *uni;                             // second byte should be ZERO
      }
      uni++;
   }
   return (rc);
}                                               /* end TxUnicStrnicmp        */
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get length of possible Unicode string that is PURE ASCIIT, no zeroes etc
// Useful to determine how to display an unknown string (test if > 5 or so)
/*****************************************************************************/
int TxUnicAsciiLength                           // RET   length pure ASCII
(
   USHORT             *uni,                     // IN    Unicode string part
   USHORT              len                      // IN    maximum Unicode length
)
{
   char               *s = (char *) uni;
   int                 i = 0;

   while ((*s) && (i < (2 * len)))
   {
      if (TxIsPureAscii( *s++))
      {
         i++;
      }
      else
      {
         break;
      }
   }
   return (i);
}                                               /* end TxUnicAsciiLength     */
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Auto Copy/Translate Unicode or ASCII string-fragment to an ASCII string, LE
/*****************************************************************************/
char *TxUnicAscii2Ascii                         // RET   ptr to ASCII string
(
   USHORT             *uni,                     // IN    Unicode string part
   char               *asc,                     // INOUT ASCII string
   USHORT              len                      // IN    max INPUT length, bytes
)
{
   asc[0] = 0;                                  // start with empty result
   return( TxUnicAsciiAppend( uni, asc, len));
}                                               /* end TxUnicAscii2Ascii     */
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate Unicode string-fragment to an ASCII string (unicode little endian)
/*****************************************************************************/
char *TxUnic2Ascii                              // RET   ptr to ASCII string
(
   USHORT             *uni,                     // IN    Unicode string part
   char               *asc,                     // INOUT ASCII string
   USHORT              len                      // IN    maximum Unicode length
)
{
   asc[0] = 0;                                  // start with empty result
   return( TxUnicAppend( uni, asc, len));
}                                               /* end TxUnic2Ascii          */
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate ASCII string to Unicode string             (unicode little endian)
/*****************************************************************************/
USHORT *TxAscii2Unic                            // RET   ptr to Unicode string
(
   char               *asc,                     // IN    ASCII string
   USHORT             *uni,                     // INOUT Unicode string part
   USHORT              len                      // IN    maximum Unicode length
)
{
   int           i;
   char          ch;

   for (i = 0; i < len; i++)
   {
      ch = asc[ i];                             // ASCII character from name
      uni[ i] = (USHORT) ch;                    // write as UNICODE

      if (ch == 0)                              // end of ASCII string
      {
         break;
      }
   }
   return( uni);
}                                               /* end TxAscii2Unic          */
/*---------------------------------------------------------------------------*/


