//
//                     TxWin, Textmode Windowing Library
//
//   Original code Copyright (c) 1995-2021 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   TxLib, released under MIT License
//
//   Copyright (c) 1995-2021  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on TxWin licensing can be directed to: info@dfsee.com
//
// ==========================================================================
//
// TX Windowed text, default window procedures
//
// Author: J. van Wijk
//
// JvW  15-07-1998 Initial version, split off from txwind
// JvW  05-09-2001 Added DefDlgProc and other Dlg functions
// JvW  01-05-2018 Use proper SetWindowPtr for dialog cData (incl 64bit support)
// JvW  28-02-2021 LICENSING: Changed from LGPL to the more liberal MIT license

#include <txlib.h>                              // public interface
#include <txwpriv.h>                            // private window interface

// Generic window procedure, for any window-class including dialogs
static ULONG txwGenericWinProc                  // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
);

static char txwam_move[]  = "Use arrow keys to MOVE window; insert=SIZE, ENTER=end";
static char txwam_size[]  = "Use arrow keys to SIZE window; insert=MOVE, ENTER=end";
static char txwam_color[] = "Use arrow keys to change;  ENTER=end";
static TXTM txwam_custom;

/*****************************************************************************/
// Default window procedure, for any window-class except dialogs
/*****************************************************************************/
ULONG txwDefWindowProc                          // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
)
{
   ULONG               rc  = NO_ERROR;
   TXWINBASE          *wnd;
   TXWINDOW           *win;

   ENTER();
   TRCMSG( hwnd, msg, mp1, mp2);
   if ((wnd = txwValidateHandle( hwnd, &win)) != NULL)
   {
      TRCLAS( "DefWindowProc", hwnd);
      switch (msg)
      {
         case TXWM_CLOSE:
            if (wnd == txwa->desktop)           // quit application
            {
               rc = txwPostMsg( hwnd, TXWM_QUIT, 0, 0);
            }
            else                                // delegate to owner
            {
               txwPostMsg((wnd->owner) ? (TXWHANDLE) wnd->owner
                                       :  TXHWND_DESKTOP, msg, mp1, mp2);
            }
            break;

         default:
            rc = txwGenericWinProc( hwnd, msg, mp1, mp2);
            break;
      }
   }
   else
   {
      rc = TX_INVALID_HANDLE;
   }
   RETURN( rc);
}                                               // end 'txwDefWindowProc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Default window procedure, for dialogs including their control windows
/*****************************************************************************/
ULONG txwDefDlgProc                             // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
)
{
   ULONG               rc  = NO_ERROR;
   TXWINBASE          *wnd;
   TXWINDOW           *win;

   ENTER();
   TRCMSG( hwnd, msg, mp1, mp2);
   if ((wnd = txwValidateHandle( hwnd, &win)) != NULL)
   {
      USHORT  focusid = txwQueryWindowUShort( (TXWHANDLE) txwa->focus, TXQWS_ID);

      TRCLAS( "DefDlgProc", hwnd);
      switch (msg)
      {
         case TXWM_CHAR:
            switch ((ULONG) mp2)                // key-value
            {
               case TXk_ESCAPE:
                  txwPostMsg( hwnd, TXWM_CLOSE, 0, 0);
                  break;

               case TXk_F4:                       //- unless captured higher up
                  txwDismissDlg( hwnd, TXDID_OK); //- handle as OK button
                  break;

               case TXk_ENTER:
                  if (txwIsMinimized( hwnd, TRUE) || // unless a parent or
                      txwIsMinimized( hwnd, FALSE) ) // this is minimized
                  {
                     rc = txwGenericWinProc( hwnd, msg, mp1, mp2); // restore
                  }
                  else                          // End the dialog, returning
                  {                             // the proper dialog-RC
                     switch (focusid)
                     {
                        case TXDID_CANCEL:
                           txwPostMsg( hwnd, TXWM_CLOSE, 0, 0);
                           break;

                        default:
                           txwDismissDlg( hwnd, focusid);
                           break;
                     }
                  }
                  break;

               case TXk_UP:
               case TXk_DOWN:
               case TXk_LEFT:
               case TXk_RIGHT:
               case TXk_PGUP:
               case TXk_PGDN:
                  if (!txwIsMinimized( hwnd, TRUE) && // when NO parent AND
                      !txwIsMinimized( hwnd, FALSE) ) // this is NOT minimized
                  {
                     rc = txwGenericWinProc( hwnd, msg, mp1, mp2);  //- handle as usual, otherwise
                     break;                                         //- fall through and pass to SB
                  }

               case TXc_UP:                     // special keys bounced by
               case TXc_DOWN:                   // control to owner (SB)
               case TXc_LEFT:
               case TXc_RIGHT:
               case TXc_PGUP:
               case TXc_PGDN:
               case TXc_HOME:
               case TXc_END:
               case TXa_UP:
               case TXa_DOWN:
               case TXa_LEFT:
               case TXa_RIGHT:
               case TXa_PGUP:
               case TXa_PGDN:
               case TXa_HOME:
               case TXa_END:
               case TXs_UP:
               case TXs_DOWN:
               case TXs_LEFT:
               case TXs_RIGHT:
               case TXs_PGUP:
               case TXs_PGDN:                   // relay to scroll-buffer
               case TXs_HOME:
               case TXs_END:
               case TXk_HOME:
               case TXk_END:
               case TXk_F8:
               case TXk_F7:
               case TXa_F7:
               case TXa_1:                      // reverse find
               case TXa_2:                      // forward find
               case TXa_3:                      // basic/expert
               case TXa_4:
               case TXc_R:                      // Reverse find (again)
               case TXc_F:                      // Find (again)
               case TXc_N:
               case TXc_P:
                  txwPostMsg((TXWHANDLE) txwa->sbview, TXWM_CHAR, mp1, mp2);
                  break;

               default:
                  rc = txwGenericWinProc( hwnd, msg, mp1, mp2);
                  break;
            }
            break;

         case TXWM_CLOSE:
            txwDismissDlg( hwnd, TXDID_CANCEL);
            break;

         case TXWM_DESTROY:                     // keep position for next time
            {                                   // if position struct was set
               TXRECT  deskpos;
               TXRECT *initial = (TXRECT *) txwQueryWindowPtr( hwnd, TXQWP_DATA);

               TRECTA( "DLG border on destroy", (&win->border));
               if (initial != NULL)
               {
                  txwQueryWindowPos( TXHWND_DESKTOP, FALSE, &deskpos);
                  initial->left = win->border.left - deskpos.left; //- calculate new pos
                  initial->top  = win->border.top  - deskpos.top;  //- relative to desktop
               }
            }
            break;

         case TXWM_COMMAND:
            switch ((ULONG) mp2)                // command source
            {
               case TXCMDSRC_PUSHBUTTON:              //- if not captured by higher
                  txwDismissDlg( hwnd, (ULONG) mp1);  //- level, dismiss dialog on
                  break;                              //- every pushbutton!

               default:                         // otherwise ignore the message
                  break;
            }
            break;

         case TXWM_INITDLG:
            if (mp1 != 0)
            {
               txwSetFocus( (TXWHANDLE) mp1);
               txwPostMsg(   hwnd, TXWM_ACTIVATE, (TXWMPARAM) TRUE, 0);
            }
            break;

         default:
            rc = txwGenericWinProc( hwnd, msg, mp1, mp2);
            break;
      }
   }
   else
   {
      rc = TX_INVALID_HANDLE;
   }
   RETURN( rc);
}                                               // end 'txwDefDlgProc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Default window procedure, for any window-class including dialogs
/*****************************************************************************/
static ULONG txwGenericWinProc                  // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
)
{
   ULONG               rc  = NO_ERROR;
   TXWINBASE          *wnd = (TXWINBASE *) hwnd;
   TXWINDOW           *win = wnd->window;
   TXWHANDLE           parent = (TXWHANDLE) wnd->parent;
   TXWHANDLE           owner  = (TXWHANDLE) wnd->owner;
   USHORT              flags  = txwQueryWindowUShort( hwnd, TXQWS_FLAGS);
   USHORT              wid    = txwQueryWindowUShort( hwnd, TXQWS_ID);
   TXWHANDLE           target;
   TXSELIST           *list;
   ULONG               altcol = 0;
   ULONG               key = (ULONG) mp2;
   static short        lastMoveRow = 0;         // vertical-move-only, for list 'drag-select'
   short               sx = 1;

   ENTER();
   TRCMSG( (TXWHANDLE) wnd, msg, mp1, mp2);
   TRCLAS( "GenericWinProc", hwnd);

   if (win != NULL)
   {
      sx = win->client.right - win->client.left +1;
   }
   switch (msg)
   {
      case TXWM_ACTIVATE:
         if ((BOOL) mp1)                        // set window active ?
         {
            flags |= TXFF_ACTIVE;
         }
         else
         {
            flags &= ~TXFF_ACTIVE;
         }
         txwSetWindowUShort( hwnd, TXQWS_FLAGS, flags);
         txwInvalidateBorder( hwnd);
         break;

      case TXWM_BORDER:
         if (txwIsWindowShowing( hwnd))
         {
            txwPaintBorder( wnd, (wnd == txwa->focus));
         }
         break;

      case TXWM_CHAR:
         if (txwa->arrowMode != TXW_ARROW_STD)  // special arrow handling
         {
            TRACES(("Handling arrow mode: %8.8lx\n", txwa->arrowMode));
            if ((key == TXk_UP)   || (key == TXk_RIGHT) ||
                (key == TXk_DOWN) || (key == TXk_LEFT)  ||
                (key == TXk_PGUP) || (key == TXk_PGDN)  ||
                (key == TXk_HOME) || (key == TXk_END )  ||
                (key == TXk_INSERT))
            {
               switch (txwa->arrowMode)
               {
                  case TXW_ARROW_MOVE:
                  case TXW_ARROW_SIZE:
                     if (win->style & TXWS_MOVEABLE) // move & resize allowed
                     {
                        TXRECT  psize;          // parent pos & size
                        TXRECT  wsize;          // window pos & size
                        short   delta;

                        txwQueryWindowPos( parent, FALSE, &psize);
                        txwQueryWindowPos( hwnd,   FALSE, &wsize);

                        if (txwa->arrowMode == TXW_ARROW_MOVE)
                        {
                           switch (key)
                           {
                              case TXk_UP:
                                 txwSetWindowPos( hwnd,  0, -1, 0, 0, TXSWP_MOVE | TXSWP_RELATIVE);
                                 break;

                              case TXk_DOWN:
                                 txwSetWindowPos( hwnd,  0, +1, 0, 0, TXSWP_MOVE | TXSWP_RELATIVE);
                                 break;

                              case TXk_LEFT:
                                 txwSetWindowPos( hwnd, -1,  0, 0, 0, TXSWP_MOVE | TXSWP_RELATIVE);
                                 break;

                              case TXk_RIGHT:
                                 txwSetWindowPos( hwnd, +1,  0, 0, 0, TXSWP_MOVE | TXSWP_RELATIVE);
                                 break;

                              case TXk_PGUP:
                                 delta = psize.top - wsize.top;
                                 txwSetWindowPos( hwnd, 0, delta, 0, 0, TXSWP_MOVE | TXSWP_RELATIVE);
                                 break;

                              case TXk_PGDN:
                                 delta = psize.bottom - wsize.bottom;
                                 txwSetWindowPos( hwnd, 0, delta, 0, 0, TXSWP_MOVE | TXSWP_RELATIVE);
                                 break;

                              case TXk_HOME:
                                 delta = psize.left - wsize.left;
                                 txwSetWindowPos( hwnd, delta, 0, 0, 0, TXSWP_MOVE | TXSWP_RELATIVE);
                                 break;

                              case TXk_END:
                                 delta = psize.right - wsize.right;
                                 txwSetWindowPos( hwnd, delta, 0, 0, 0, TXSWP_MOVE | TXSWP_RELATIVE);
                                 break;

                              default:
                                 txwa->arrowMode = TXW_ARROW_SIZE;
                                 break;
                           }
                        }
                        else                    // size window
                        {
                           switch (key)
                           {
                              case TXk_UP:
                                 txwSetWindowPos( hwnd, 0,  0,  0, -1, TXSWP_SIZE | TXSWP_RELATIVE);
                                 break;

                              case TXk_DOWN:
                                 txwSetWindowPos( hwnd, 0,  0,  0, +1, TXSWP_SIZE | TXSWP_RELATIVE);
                                 break;

                              case TXk_LEFT:
                                 txwSetWindowPos( hwnd, 0,  0,  -2, 0, TXSWP_SIZE | TXSWP_RELATIVE);
                                 break;

                              case TXk_RIGHT:
                                 txwSetWindowPos( hwnd, 0,  0,  +2, 0, TXSWP_SIZE | TXSWP_RELATIVE);
                                 break;

                              case TXk_PGUP:
                                 delta = wsize.top - wsize.bottom;
                                 txwSetWindowPos( hwnd, 0,  0,  0, delta, TXSWP_SIZE | TXSWP_RELATIVE);
                                 break;

                              case TXk_PGDN:
                                 delta = psize.bottom - wsize.bottom;
                                 txwSetWindowPos( hwnd, 0,  0,  0, delta, TXSWP_SIZE | TXSWP_RELATIVE);
                                 break;

                              case TXk_HOME:
                                 delta = wsize.left - wsize.right + 7;
                                 txwSetWindowPos( hwnd, 0,  0,  delta, 0, TXSWP_SIZE | TXSWP_RELATIVE);
                                 break;

                              case TXk_END:
                                 delta = psize.right - wsize.right;
                                 txwSetWindowPos( hwnd, 0,  0,  delta, 0, TXSWP_SIZE | TXSWP_RELATIVE);
                                 break;

                              default:
                                 txwa->arrowMode = TXW_ARROW_MOVE;
                                 break;
                           }
                        }
                        if (txwa->arrowMode == TXW_ARROW_MOVE)
                        {
                           txwPostMsg( TXHWND_DESKTOP, TXWM_SETFOOTER, (TXWMPARAM) txwam_move, 0);
                        }
                        else
                        {
                           txwPostMsg( TXHWND_DESKTOP, TXWM_SETFOOTER, (TXWMPARAM) txwam_size, 0);
                        }
                     }
                     else
                     {
                        txwPostMsg( parent, msg, mp1, mp2);
                     }
                     break;

                  case TXW_ARROW_COLOR:
                     switch (key)
                     {
                        case TXk_LEFT:          // cycle through Scrollbuffer
                        case TXk_RIGHT:         // color alternatives (8)
                           if (txwa->sbview)
                           {
                              TXWINDOW  *sbwin = txwa->sbview->window;
                              if (key == TXk_LEFT)
                              {
                                 sbwin->sb.altcol--;
                              }
                              else
                              {
                                 sbwin->sb.altcol++;
                              }
                              sbwin->sb.altcol &= TXSB_COLOR_MASK;
                              txwInvalidateWindow((TXWHANDLE) txwa->sbview, FALSE, FALSE);
                           }
                           break;

                        default:
                           switch (key)
                           {
                              case TXk_UP:
                                 txwColorScheme( TXWCS_PREV_SCHEME,  NULL);
                                 break;

                              case TXk_DOWN:
                                 txwColorScheme( TXWCS_NEXT_SCHEME,  NULL);
                                 break;

                              case TXk_HOME:
                                 txwColorScheme( TXWCS_FIRST_SCHEME, NULL);
                                 break;

                              case TXk_END:
                                 txwColorScheme( TXWCS_LAST_SCHEME,  NULL);
                                 break;

                              case TXk_PGUP:
                                 if ((txwcs->linestyle--) == 0)
                                 {
                                    txwcs->linestyle = TXW_CS_LAST;
                                 }
                                 break;

                              case TXk_PGDN:
                                 if ((txwcs->linestyle++) == TXW_CS_LAST)
                                 {
                                    txwcs->linestyle = 0;
                                 }
                                 break;
                           }
                           txwInvalidateAll();  // redraw all windows
                           break;
                     }
                     if (txwa->sbview)
                     {
                        TXWINDOW  *sbwin = txwa->sbview->window;
                        altcol = sbwin->sb.altcol;
                     }
                     sprintf( txwam_custom, "Colors: %19.19s %u/%u - %s",
                              txwcs->name, altcol, txwcs->linestyle, txwam_color);
                     txwPostMsg( TXHWND_DESKTOP, TXWM_SETFOOTER, (TXWMPARAM) txwam_custom, 0);
                     break;

                  default:
                     break;
               }
            }
            else                                // end arrow mode
            {
               txwa->arrowMode = TXW_ARROW_STD;
               txwPostMsg( TXHWND_DESKTOP, TXWM_SETFOOTER, 0, 0);
               if (key != TXk_ENTER)
               {
                  txwPostMsg(hwnd, msg, mp1, mp2); // exec end-key if not ENTER
               }
            }
         }
         else
         {
            switch (key)                        // key-value
            {
               case TXk_F3:                     // Close Owner (Dlg) or self
                  switch (win->class)
                  {
                     case TXW_SBVIEW:           // should go to cmdline ...
                        txwPostMsg( hwnd, msg, mp1, (TXWMPARAM) TXk_TAB);
                        break;

                     default:
                        txwPostMsg((owner != 0) ? owner : hwnd, TXWM_CLOSE, 0, 0);
                        break;
                  }
                  break;

               case TXa_F3:                     // application exit
               case TXa_BACKQUOTE:              // emergency exit
                  txwPostMsg( hwnd, TXWM_QUIT, 0, 0);
                  break;

               case TXs_F1:                     // HELP alternative, Linux GNOME
                  txwPostMsg( hwnd, TXWM_HELP, 0, 0);
                  break;

               case TXs_F12:
               case TXa_F12:
                  //- to be refined, move code to a WM_COMMAND so it can be
                  // called from the menu too (using an accelerator ?)
                  txwa->arrowMode = TXW_ARROW_COLOR;
                  if (txwa->sbview)
                  {
                     TXWINDOW  *sbwin = txwa->sbview->window;
                     altcol = sbwin->sb.altcol;
                  }
                  sprintf( txwam_custom, "Colors: %19.19s %u/%u - %s",
                           txwcs->name, altcol, txwcs->linestyle, txwam_color);
                  txwPostMsg( TXHWND_DESKTOP, TXWM_SETFOOTER, (TXWMPARAM) txwam_custom, 0);
                  break;

               case TXa_F7:
               case TXs_F7:
                  txwa->arrowMode = TXW_ARROW_MOVE;
                  txwPostMsg( TXHWND_DESKTOP, TXWM_SETFOOTER, (TXWMPARAM) txwam_move, 0);
                  break;

               case TXa_F8:
               case TXs_F8:
                  txwa->arrowMode = TXW_ARROW_SIZE;
                  txwPostMsg( TXHWND_DESKTOP, TXWM_SETFOOTER, (TXWMPARAM) txwam_size, 0);
                  break;

               case TXc_L:                      // (UNIX) Refresh screen
                  txwRefreshScreen();           // should work on all platforms
                  break;

               case TXa_SLASH:                  // toggle trace value
                  switch (TxTrLevel)            // and slowdown delay
                  {
                     case 0:
                        TxTrLevel    = 1;
                        break;

                     case 1:
                        TxTrLevel    = 100;
                        if (TxQueryLogFile( NULL, NULL) != NULL)
                        {
                           TXLN line;

                           sprintf( line, "Tracing to: %s ", TxQueryLogName());
                           txwSetDesktopTopLine( line, cSchemeColor);
                        }
                        else if (TxTrLogOnly == FALSE)
                        {
                           txwSetDesktopTopLine( "Tracing to SCREEN ...      ",
                                                  cSchemeColor);
                        }
                        else
                        {
                           txwSetDesktopTopLine( "Tracing to nowhere ...     ",
                                                  cSchemeColor);
                        }
                        break;

                     default:
                        TxTrLevel    = 0;
                        TxTrSlowDown = 0;
                        txwSetDesktopTopLine( "Tracing stopped, no delay ... ",
                                               cSchemeColor);
                        break;
                  }
                  break;

               case TXk_F12:                    // Minimize win to title only
               case TXa_F9:                     // (or undo that)
                  if (win->style & TXWS_MOVEABLE) // move & resize allowed
                  {
                     if (flags & TXFF_MINIMIZED)
                     {
                        TRACES(("Set MINIMIZED OFF for window:%8.8lx\n", hwnd))
                        flags &= ~TXFF_MINIMIZED;
                        txwSetWindowUShort( hwnd, TXQWS_FLAGS, flags);
                        txwSetWindowPos( hwnd, wnd->restore.left,    wnd->restore.top,
                                               wnd->restore.right  - wnd->restore.left +1,
                                               wnd->restore.bottom - wnd->restore.top  +1,
                                               TXSWP_SIZE | TXSWP_MOVE | TXSWP_ABSOLUTE);
                     }
                     else                       // set to vertical size 0 or 1
                     {
                        TRACES(("Set MINIMIZED ON  for window:%8.8lx\n", hwnd))
                        flags |= TXFF_MINIMIZED;
                        txwSetWindowUShort( hwnd, TXQWS_FLAGS, flags);

                        wnd->restore = win->border; // new restore size/pos

                        txwSetWindowPos( hwnd, 0,  0,
                                         win->border.right - win->border.left  +1,
                                        (win->style & TXWS_TITLEBORDER) ? 1 : 0,
                                         TXSWP_SIZE | TXSWP_ABSOLUTE);
                     }
                     txwInvalidateAll();
                  }
                  else
                  {
                     TRACES(("window NOT movable, delegate MINIMIZE to parent ...\n"));
                     txwPostMsg( parent, msg, mp1, mp2);
                  }
                  break;

               case TXa_F10:                    // Maximize to parent
               case TXs_F10:                    // Maximize to parent
               case TXa_F5:                     // restore size/position
               case TXs_F5:                     // restore size/position
                  if (win->style & TXWS_MOVEABLE) // move & resize allowed
                  {
                     TXRECT  b    = win->border; // current border
                     TXRECT  rect = wnd->restore; // previous pos/size
                     BOOL    update_restore = TRUE;

                     TRECTA( "restore", (&rect));
                     TRECTA( "border ", (&b));

                     if (flags & TXFF_MINIMIZED)
                     {
                        TRACES(("Set MINIMIZED OFF for window:%8.8lx\n", hwnd))
                        flags &= ~TXFF_MINIMIZED;
                        txwSetWindowUShort( hwnd, TXQWS_FLAGS, flags);
                        update_restore = FALSE; // never restore to MINIMIZED
                     }
                     if ((key == TXa_F10) || (key == TXs_F10)) // maximize to parent size
                     {
                        if ( (win->class == TXW_CANVAS)          &&   //- canvas class
                            ((win->style & TXCS_NO_MAXIMIZE) != 0))   //- probably menu-bar
                        {
                           TRACES(("Do NOT maximize the MENU-BAR, but close it\n"));
                           txwPostMsg( hwnd, TXWM_CHAR, 0, (TXWMPARAM) TXk_ESCAPE);
                        }
                        else
                        {
                           txwQueryWindowRect((TXWHANDLE) wnd->parent, FALSE, &rect);

                           TRECTA( "parent ", (&rect));

                           if (wnd != txwa->desktop)
                           {
                              if ((rect.bottom != (b.bottom - b.top  +1)) ||
                                  (rect.right  != (b.right  - b.left +1))  )
                              {
                                 TRACES(("MAXIMIZE to parent for window:%8.8lx\n", hwnd))
                                 if (update_restore)
                                 {
                                    wnd->restore = b; // new restore size/pos
                                 }
                                 txwSetWindowPos( hwnd, 0, 0, rect.right, rect.bottom,
                                                  TXSWP_SIZE | TXSWP_MOVE);
                              }
                           }
                        }
                     }
                     else                       // restore to previous pos/size
                     {
                        if (update_restore)
                        {
                           wnd->restore = b;    // to toggle back and forth ...
                        }
                        txwSetWindowPos( hwnd, rect.left,    rect.top,
                                               rect.right  - rect.left +1,
                                               rect.bottom - rect.top  +1,
                                               TXSWP_SIZE | TXSWP_MOVE | TXSWP_ABSOLUTE);
                     }
                  }
                  else
                  {
                     TRACES(("window NOT movable, delegate to parent ...\n"));
                     txwPostMsg( parent, msg, mp1, mp2);
                  }
                  break;

               case TXk_UP:
               case TXk_DOWN:
                  if (txwIsMinimized( hwnd, TRUE) || // when minimized, let
                      txwIsMinimized( hwnd, FALSE) ) // the SB handle the key
                  {
                     txwPostMsg((TXWHANDLE) txwa->sbview, TXWM_CHAR, mp1, mp2);
                     break;
                  }
               default:
                  if (flags & TXFF_MINIMIZED)   // un-collapse on other keys
                  {                             // (does NOT work on child!)
                     TRACES(("Set MINIMIZED OFF for window:%8.8lx\n", hwnd))
                     flags &= ~TXFF_MINIMIZED;
                     txwSetWindowUShort( hwnd, TXQWS_FLAGS, flags);
                     txwSetWindowPos( hwnd, wnd->restore.left,    wnd->restore.top,
                                            wnd->restore.right  - wnd->restore.left +1,
                                            wnd->restore.bottom - wnd->restore.top  +1,
                                            TXSWP_SIZE | TXSWP_MOVE | TXSWP_ABSOLUTE);
                     txwInvalidateAll();
                  }
                  else
                  {
                     if (txwIsMinimized( hwnd, TRUE)) // any parent minimized ?
                     {
                        txwPostMsg( parent, msg, mp1, mp2);
                     }
                     else                       // non move/size/exit keys
                     {
                        switch (key)
                        {
                           case TXk_TAB:        // move to other window
                           case TXs_TAB:
                              target = (key == TXs_TAB)
                                     ? txwFindPrevFocus( hwnd, TRUE)
                                     : txwFindNextFocus( hwnd, TRUE);
                              if (target != hwnd)
                              {
                                 txwSetFocus( target);
                                 break;
                              }
                              else if (win->style & TXWS_TAB_2_OWNER)
                              {
                                 txwPostMsg( owner, msg, mp1, mp2);
                                 break;
                              }                 // when no TAB action, allow
                           default:             // class specific usage

                              //- to be refined, replace this by class-specific
                              //- processing on the HIGHEST level (see HexEdit)

                              switch (win->class)
                              {
                                 case TXW_TEXTVIEW:
                                    rc = txwIkeyTextview( hwnd, mp1, mp2);
                                    break;

                                 case TXW_SBVIEW:
                                    rc = txwSbViewWinProc( hwnd, msg, mp1, mp2);
                                    break;

                                 case TXW_ENTRYFIELD:
                                    rc = txwIkeyEntryfield( hwnd, mp1, mp2);
                                    break;

                                 case TXW_BUTTON:
                                    rc = txwIkeyButton( hwnd, mp1, mp2);
                                    break;

                                 case TXW_LISTBOX:
                                    rc = txwIkeyListBox( hwnd, mp1, mp2);
                                    break;

                                 default:
                                    txwPostMsg( owner, msg, mp1, mp2);
                                    break;
                              }
                              break;
                        }
                     }
                  }
                  break;
            }
         }
         break;

      case TXWM_BUTTONDOWN:                     // Mouse button clicked
      case TXWM_BUTTONDBLCLK:                   // or double-clicked
         txwMouseButtonDown( hwnd, msg, mp1, mp2);
         break;

      case TXWM_BUTTONUP:
         TRACES(( "BtnUP:%4.4hx/%4.4hx on %8.8lx at % 3hd,%-3hd %s '%s'\n",
                   TXSH1FROMMP(mp2),
                   TXSH2FROMMP(mp2),  hwnd,
                   TXSH1FROMMP(mp1) - win->client.left,
                   TXSH2FROMMP(mp1) - win->client.top,
                   txwClassDescription( hwnd),
                   (win->title) ? win->title  : ""));

         if (TXMOUSEKEYSCA() == TXm_KS_NONE)    // no shift/alt/ctrl keys
         {
            if (txwQueryCapture() != TXHWND_NULL)
            {
               if      (txwa->mDragFlags & TXSWP_MOVE)
               {
                  txwSetWindowPos( hwnd, TXMOUSECOL() - txwa->mDragCol,
                                         TXMOUSEROW() - txwa->mDragRow,
                                   0, 0, TXSWP_MOVE | TXSWP_RELATIVE);
               }
               else if (txwa->mDragFlags & TXSWP_SIZE)
               {
                  txwSetWindowPos( hwnd, 0, 0,
                                         TXMOUSECOL() - txwa->mDragCol,
                                         TXMOUSEROW() - txwa->mDragRow,
                                         TXSWP_SIZE | TXSWP_RELATIVE);
               }
               else if (txwa->mDragFlags & TXSWP_MARK)
               {
                  //- to be refined, set the now completed mark-area ?
                  //- should already be done by the last MOUSEMOVE ...
                  TRACES(("Ending drag-MARK\n"));
               }
               txwSetCapture( TXHWND_NULL);
               txwa->mDragFlags = 0;
            }
         }
         break;

      case TXWM_STARTDRAG:                      // leaving a 'click' location with button-down
         if (win->class == TXW_LISTBOX)         // simulated clicks, one per line
         {
            lastMoveRow = 0;                    // reset vertical-move-only filtering
            break;                              // and NO action (done by initial click)
         }
         else
         {
            //- fall through, so first-click location becomes the anchorpoint for dragging
         }
      case TXWM_MOUSEMOVE:                      // with button down (drag)
         TRACES(( "mMove:%4.4hx/%4.4hx on %8.8lx at % 3hd,%-3hd %s '%s'\n",
                   TXSH1FROMMP(mp2),
                   TXSH2FROMMP(mp2),  hwnd,
                   TXSH1FROMMP(mp1) - win->client.left,
                   TXSH2FROMMP(mp1) - win->client.top,
                   txwClassDescription( hwnd),
                   (win->title) ? win->title  : ""));

         if (TXMOUSEKEYSCA() == TXm_KS_NONE)    // no shift/alt/ctrl keys
         {
            if ((TXMOUSEBUTTON() & TXm_BUTTON2) == 0) // no drag on help
            {
               short  col = TXMOUSECOL();       // absolute mouse position!
               short  row = TXMOUSEROW();

               if (txwQueryCapture() == TXHWND_NULL)
               {
                  if ( txwIsDescendant( hwnd, txwa->modality) && // active
                      ((row <  win->client.top)    || // on window title line
                      ((row >= win->client.bottom) &&
                       (col >= win->client.right )))) // or lower-right corner
                  {
                     if (row <= win->client.top) // window move drag operation
                     {
                        txwa->mDragFlags = TXSWP_MOVE;
                     }
                     else                       // window resize drag operation
                     {
                        txwa->mDragFlags = TXSWP_SIZE;
                        col &= 0xfffe;          // size to EVEN column pos
                     }
                     if (win->style & TXWS_MOVEABLE) // frame itself movable
                     {
                        txwSetCapture( hwnd);
                        txwa->mDragCol = col;
                        txwa->mDragRow = row;
                     }
                     else if (((win->class == TXW_LISTBOX)   ||
                               (win->class == TXW_TEXTVIEW)) &&
                               (wnd->owner->window->style & TXWS_MOVEABLE))
                     {
                        txwSetCapture( owner);
                        txwa->mDragCol = col;
                        txwa->mDragRow = row;
                     }
                  }
                  else                          // no title-bar drag / window size
                  {
                     switch (win->class)
                     {
                        case TXW_LISTBOX:       // simulated clicks, one per line
                           if (row != lastMoveRow) // filters out horizontal movement!
                           {
                              //- basically the same processing as regular clicks
                              //- but can be filtered further there on 'msg'
                              txwMouseButtonDown( hwnd, msg, mp1, mp2);
                              lastMoveRow = row;
                           }
                           break;

                        case TXW_SBVIEW:
                        case TXW_HEXEDIT:
                        case TXW_TEXTVIEW:
                        case TXW_ENTRYFIELD:
                           txwa->mDragFlags = TXSWP_MARK;
                           txwSetCapture(  hwnd);
                           txwa->mDragCol = col; // initial mark anchor position
                           txwa->mDragRow = row;
                           if (txwSetDragMark( hwnd, col, row))
                           {
                              txwInvalidateWindow( hwnd, FALSE, FALSE); // repaint to show mark
                           }
                           break;

                        default:
                           break;
                     }
                  }
               }
               else                             // drag-window or drag-marking in progress
               {
                  TRACES(("Captured hwnd: %p  buttons: %x  DragFlags: %x\n", hwnd, TXMOUSEBUTTON(), txwa->mDragFlags));
                  if ((TXMOUSEBUTTON() & TXm_BUTTON1)) // full-window drag or drag-marking
                  {

                     if      (txwa->mDragFlags & TXSWP_MOVE)
                     {
                        txwSetWindowPos( hwnd, col - txwa->mDragCol,
                                               row - txwa->mDragRow,
                                         0, 0, TXSWP_MOVE | TXSWP_RELATIVE);
                        txwa->mDragCol = col;   // set new reference position
                        txwa->mDragRow = row;
                     }
                     else if (txwa->mDragFlags & TXSWP_SIZE)
                     {                          // to allow 1/2 step move/size
                        col &= 0xfffe;          // size to EVEN column pos only
                        txwSetWindowPos( hwnd, 0, 0,
                                               col - txwa->mDragCol,
                                               row - txwa->mDragRow,
                                               TXSWP_SIZE | TXSWP_RELATIVE);
                        txwa->mDragCol = col;   // set new reference position
                        txwa->mDragRow = row;
                     }
                     else if (txwa->mDragFlags & TXSWP_MARK)
                     {
                        //- set mark, relative to existing anchor in mDragCol/Row
                        if (txwSetDragMark( hwnd, col, row))
                        {
                           txwInvalidateWindow( hwnd, FALSE, FALSE); // repaint to show mark
                        }
                     }
                  }
               }
            }
         }
         break;

      case TXWM_CURSORVISIBLE:
         wnd->curvisible = (BOOL) mp1;
         txwSetCursorStyle( hwnd, txwa->insert);
         break;

      case TXWM_CREATE:
         txwSetCursorStyle( hwnd, txwa->insert);
         switch (win->class)
         {
            case TXW_SBVIEW:
               break;

            case TXW_ENTRYFIELD:
               if (win->style & TXES_MAIN_CMDLINE)
               {
                  txwa->maincmd  = hwnd;        // ENTRYFIELD commandline
                  TRACES(( "setting MAINCMD  to: %8.8lx\n", txwa->maincmd));
               }
               break;

            default:
               break;
         }
         break;

      case TXWM_DESTROY:                        // window will be destroyed
         switch (win->class)
         {
            case TXW_SBVIEW:
               txwa->sbview = NULL;             // SBVIEW for debug status
               break;

            case TXW_ENTRYFIELD:
               if (win->style & TXES_MAIN_CMDLINE)
               {
                  txwa->maincmd  = TXHWND_NULL; // ENTRYFIELD for commandline
                  TRACES(( "setting MAINCMD  to: %8.8lx\n", txwa->maincmd));
               }
               break;

            default:
               break;
         }
         break;                                 // notification

      case TXWM_HELP:                           // context-sensitive help
         {
            ULONG      helpid = TXWH_USE_WIN_HELP;

            switch (win->class)
            {
               case TXW_LISTBOX:
                  if ((list = win->lb.list) != NULL)
                  {
                     ULONG      li;
                     TXS_ITEM  *item;

                     if ((win->style & TXLS_DROP_MENU) == 0) // no heading
                     {
                        li = (ULONG) (list->top + win->lb.cpos);

                        if (li < list->count)   // avoid empty list trap
                        {
                           item = list->items[li];

                           if ((helpid = item->helpid) == TXWH_USE_CMD_CODE)
                           {
                              helpid = item->value; // use cmd value
                              TRACES(("Using cmdcode %8.8lx as helpid\n", helpid));
                           }
                           else if (helpid == TXWH_USE_OWNER_HELP)
                           {
                              TRACES(("Delegate help to owner\n"));
                              txwPostMsg( owner, msg, mp1, mp2);
                           }
                        }
                     }
                  }
                  break;

               default:
                  break;
            }
            TRACES(("Class specific helpid: %8.8lx = %lu (1 = USE_WIN)\n", helpid, helpid));
            if (helpid != TXWH_USE_OWNER_HELP)  // not delegated to owner ?
            {
               if (helpid == TXWH_USE_WIN_HELP) // not changed to specific ?
               {
                  TRACES(("Window specific helpid: %8.8lx = %lu\n", helpid, helpid));
                  helpid = win->helpid;         // use default window help
               }
               txwHelpDialog( helpid, 0, "");   // context-sensitive help
               TxCancelAbort();                 // don't set global abort
            }
         }
         break;                                 // after simple help screen

      case TXWM_MOVE:                           // window has been moved
         break;                                 // notification

      case TXWM_PAINT:                          // window completely visible
         if (txwIsWindowShowing( hwnd) ||       // or scrollbuf, painting a
             (win->class == TXW_SBVIEW))        // partly covered window
         {
            rc = txwPaintWindow( wnd, (ULONG) mp1); // mp1 is border paint
         }
         break;

      case TXWM_SELECTED:
         if ((BOOL) mp1)                        // set window selected ?
         {
            flags |= TXFF_SELECTED;
         }
         else
         {
            flags &= ~TXFF_SELECTED;
         }
         txwSetWindowUShort( hwnd, TXQWS_FLAGS, flags);
         txwInvalidateBorder( hwnd);
         break;

      case TXWM_SETFOCUS:
         switch (win->class)                    // FOCUS processing for specific classes
         {
            case TXW_SBVIEW:                    // needs to set dedicated footer ...
               rc = txwSbViewWinProc( hwnd, msg, mp1, mp2);
               break;

            case TXW_ENTRYFIELD:                // set insert mode when required
               if (((BOOL) mp1) &&              // got focus, will be on top,
                   (txwa->autoInsert > 1))      // AUTO insert mode is active
               {
                  txwa->insert = (sx > txwa->autoInsert); // INSERT on large entryfields (visible area)

                  TRACES(("AUTO INSERT (%d) field length: %hd, insert: %s\n", txwa->autoInsert, sx, (txwa->insert) ? "ON" : "OFF"));
               }
               break;

            case TXW_LISTBOX:
               if ((list = win->lb.list) != NULL)
               {
                  TRACES(("Drop-menu:  %s\n", (win->style & TXLS_DROP_MENU ) ? "YES" : "NO"));
                  TRACES(("Auto-drop:  %s\n", (win->style & TXLS_AUTO_DROP ) ? "YES" : "NO"));

                  if ((BOOL) mp1)               // got focus
                  {
                     TRACES(("LB:%8.8lx L:%8.8lx f:%4.4x t:'%s' F:'%s'\n",
                              win, list, flags, (win->title)  ? win->title  : "none",
                                                (win->footer) ? win->footer : "none"));

                     if (win->style & TXLS_DROP_MENU)
                     {
                        txwPostMsg( TXHWND_DESKTOP, TXWM_SETFOOTER,
                                    (TXWMPARAM) win->footer, 0);

                        if ((flags & TXFF_ACTIVE) == 0) // unless now dropped
                        {
                           if (win->style & TXLS_AUTO_DROP)
                           {
                              if (flags & TXFF_AUTODROPNEXT) // delay drop to next time
                              {
                                 TRACES(("Autodrop menu delayed until next time ...\n"));
                                 flags &= ~TXFF_AUTODROPNEXT;
                                 txwSetWindowUShort( hwnd, TXQWS_FLAGS, flags);
                              }
                              else
                              {
                                 txwPostMsg( hwnd, TXWM_CHAR, 0, (TXWMPARAM) TXa_ENTER); // forced drop
                              }
                           }
                        }
                     }
                     else                       // signal select on get-focus
                     {
                        txwPostMsg( owner, TXWM_CONTROL,
                                    TXMPFROM2SH(wid,TXLN_SETFOCUS),
                                    (TXWMPARAM) (win->lb.list));
                     }
                  }
                  else                          // lost focus
                  {
                     txwPostMsg( TXHWND_DESKTOP, TXWM_SETFOOTER, 0, 0); // reset description
                  }
                  if (list->flags & TXSL_MULTI_QUICK)
                  {
                     if (strlen( txwa->listmatch) != 0)
                     {
                        txwSetSbviewStatus( "", cSchemeColor);
                        strcpy( txwa->listmatch, ""); // reset string
                     }
                  }
               }
               txwPostMsg( hwnd, TXWM_PAINT, FALSE, 0); // repaint, list-bar
               break;

            default:
               break;
         }

         if (((BOOL) mp1) &&                    // got focus, will be on top,
             (win->style & TXWS_FOCUS_PAINT))   // so can be repainted
         {
            txwInvalidateWindow( hwnd, TRUE, TRUE);
         }
         else                                   // just update status part
         {                                      // and borders
            if (txwIsWindowShowing( hwnd))
            {
               if (win->style & TXWS_FOCUS_PAINT)
               {
                  txwInvalidateWindow( hwnd, TRUE, TRUE);
               }
               else
               {
                  txwPaintBorder( wnd, (BOOL) mp1);
               }
            }
            txwPaintWinStatus( wnd, NULL, 0);   // checks 'Showing' for status!
         }
         break;

      case TXWM_SIZE:                           // window has been sized
         switch (win->class)                    // additional SIZE processing
         {                                      // for specific classes
            case TXW_LISTBOX:
               if ((list = win->lb.list) != NULL)
               {
                  if      (win->style & TXLS_DROP_VALUE)
                  {
                  }
                  else if (win->style & TXLS_DROP_MENU)
                  {
                  }
                  else                          // multi-line or spin
                  {
                     list->vsize = (ULONG) (win->client.bottom - win->client.top +1);
                     TRACES(("Updated list->vsize to %hu lines\n",
                              win->client.bottom - win->client.top +1));
                  }
               }
               break;

            default:
               break;
         }
         break;                                 // notification

      case TXWM_SHOW:                           // window visibility change
         break;                                 // notification

      case TXWM_STATUS:                         // checks own visibility!
         rc = txwPaintWinStatus( wnd, (char *) mp1, (BYTE) mp2);
         break;

      case TXWM_SETFOOTER:                      // set/reset footer text
         if (mp1)
         {
            win->footer = (char *) mp1;         // set new footer text
         }
         else
         {
            win->footer = wnd->oldFooter;       // reset to original
         }
         txwPaintBorder( wnd,                   // repaint will NOT check
                        (wnd == txwa->focus));  // for border visibility!
         break;

      default:
         if (win->class == TXW_SBVIEW)          // can handle user messages!
         {
            rc = txwSbViewWinProc( hwnd, msg, mp1, mp2);
         }
         else
         {
            TRACES(("Warning: unhandled message in lowest level WinProc!\n"));
         }
         break;
   }
   RETURN( rc);
}                                               // end 'txwGenericWinProc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Generic MOUSE BUTTONDOWN/BUTTONDBLCLK processing, class-independant stuff
/*****************************************************************************/
ULONG txwMouseButtonDown                        // RET   TX_PENDING if not done
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    msg param 1
   TXWMPARAM           mp2                      // IN    msg param 2
)
{
   ULONG               rc  = NO_ERROR;
   TXWINBASE          *wnd = (TXWINBASE *) hwnd;
   TXWINDOW           *win = wnd->window;
   TXWHANDLE           owner  = (TXWHANDLE) wnd->owner;
   USHORT              flags  = txwQueryWindowUShort( hwnd, TXQWS_FLAGS);
   USHORT              wid    = txwQueryWindowUShort( hwnd, TXQWS_ID);

   ENTER();

   TRACES(( "Click:%4.4hx/%4.4hx on hwnd:%8.8lx at % 3hd,%-3hd %s '%s' owner hwnd:%8.8lx\n",
             TXSH1FROMMP(mp2), TXSH2FROMMP(mp2),  hwnd,
             TXSH1FROMMP(mp1) - win->client.left, TXSH2FROMMP(mp1) - win->client.top,
             txwClassDescription( hwnd), (win->title) ? win->title  : "", owner));

   #if defined (USE_CTRL_CLICK_TO_CLOSE_WINDOWS)
   if ((TXMOUSEKEYSCA() & TXm_KS_CTRL) != 0)    // ctrl-click any button, close window / cancel
   {                                            // except on desktop windows (no app quit)
      TRACES(("txwa->desktop:%8.8lx\n", txwa->desktop));
      if ((hwnd != (TXWHANDLE) txwa->desktop) && (owner != (TXWHANDLE) txwa->desktop))
      {
         txwPostMsg( hwnd, TXWM_CLOSE, 0, 0);
      }
   }
   else                                         // accept None, Shift and Alt for TX use
   #endif                                       // accept ALL (Ctrl/Alt-click for clipboard use)

   {
      short  col = TXMOUSECOL() - win->client.left;  //- relative to
      short  row = TXMOUSEROW() - win->client.top;   //- the window!
      short  sy  = win->client.bottom - win->client.top  +1;
      short  sx  = win->client.right  - win->client.left +1;

      TRACES(( "col:% 3hd row:% 3hd sy:% 3hd\n", col, row, sy));

      if      (TXMOUSEBUTTON() == TXm_BUTTON2)
      {
         if (txwa->rmb2Menu)                    // RMB click simulates <F10> for menu
         {
            txwPostMsg((TXWHANDLE) txwa->focus, TXWM_CHAR, 0, (TXWMPARAM) TXk_F10);
         }
      }
      else if (TXMOUSEBUTTON() == TXm_BUTTON3)  // present context sensitive help on window
      {
         TRACES(("Button-3, ignored on UNIX ...\n"));

         //- mouse-button 3 on UNIX is most often reserved for X clipboard pasting, don't interfere ...
         #ifndef UNIX
         if ((txwIsWindowEnabled(hwnd))     &&  // enabled and focus set ?
             (txwSetFocus( hwnd) == NO_ERROR))  // to window under mouse
         {
            TRACES(( "Changed focus to clicked window, show HELP\n"));
            //- class specific processing (mainly for lists, like menu-item help)

            switch (win->class)                 // additional positioning
            {
               case TXW_LISTBOX:
                  {
                     TXLISTBOX *dat  = &win->lb;
                     TXSELIST  *list = dat->list;

                     TRACES(("Drop-value: %s\n", (win->style & TXLS_DROP_VALUE) ? "YES" : "NO"));
                     TRACES(("Drop-menu:  %s\n", (win->style & TXLS_DROP_MENU ) ? "YES" : "NO"));

                     if ((list != NULL) && (list->count != 0))
                     {
                        if      (((win->style & TXLS_DROP_VALUE) == 0) && // no drop-down
                                 ((win->style & TXLS_DROP_MENU ) == 0)  ) // and no menu heading
                        {             // so it will be a multi-line, including menu
                           ULONG      li = (ULONG) (list->top + row);

                           TRACES(("Listbox style:%8.8lx li:%lu\n", win->style, li));

                           if (li < list->count)
                           {
                              TRACES(("Item: '%s' = '%s'\n", list->items[li]->text, list->items[li]->desc));

                              //- always ENTER even on disabled (for error message)
                              dat->cpos = row; //- Visual, and list position update
                           }
                        }
                     }
                  }
                  break;

               default:
                  break;
            }
            txwPostMsg( hwnd, TXWM_HELP, 0, 0);
         }
         else
         {
            txwPostMsg( hwnd, TXWM_HELP, 0, 0);
         }
         #endif
      }
      else                                      // perform BUTTON-1 default actions
      {
         TRACES(("Button-1, special and default (FOCUS) actions ...\n"));

         TRACES(( "FOCUS:%8.8lx owner:%8.8lx modality:%8.8lx maincmd:%8.8lx mainmenu:%8.8lx\n",
                   txwa->focus, txwa->focus->owner, txwa->modality, txwa->maincmd, txwa->mainmenu));

         if ((win->class == TXW_LISTBOX) &&     // is this a menu-heading ?
             (win->style & TXLS_DROP_MENU))     // close any open (sub)menu
         {
            TRACES(( "Click on menu-heading, re-open menu at: '%s'\n",
                     (win->title) ? win->title  : ""));

            txwa->reopenMenu = txwQueryWindowUShort( hwnd, TXQWS_ID);
            txwPostMsg((TXWHANDLE) txwa->focus, TXWM_CHAR, 0, (TXWMPARAM) TXk_MENU);
         }

         //- Close active popup-listbox when clicking outside of it
         else if ((txwa->focus->window->class == TXW_LISTBOX) &&
                  (txwa->focus->owner->window->style & TXCS_LIST_POPUP) &&
                  (txwa->focus->owner == (TXWINBASE *) txwa->modality)  &&
                  (txwIsDescendant( hwnd, txwa->modality) == FALSE))
         {
            TRACES(( "Close active listbox popup, and re-issue the message\n"));

            txwSendMsg((TXWHANDLE) txwa->focus->owner, TXWM_CLOSE, 0, 0);
            txwPostMsg( hwnd, TXWM_BUTTONDOWN, mp1, mp2);   // re-post message
         }

         //- toggle the main menu ON or OFF, on a mouse click near its normal position (or below cmd-entryfield)
         else if (((hwnd == (TXWHANDLE) txwa->desktop))               ||  //- desktop border, except leftside
                  ((hwnd == (TXWHANDLE) txwa->sbview ) && ((row < 0)  ||  //- sbview title-border, if any
                            (TXMOUSEROW() == 0)) && ((col + 3) < sx)) ||  //- sbview topline if no borders (-f-)
                  ((hwnd == txwa->maincmd) && (txwa->mainmenu  != 0)) ||  //- cmd-entryfield, with menu now up
                   (hwnd == txwa->mainmenu))                              //- menu-canvas itself
         {
            //- first CLOSE the existing menu, before asking for a menu-cycle!
            TRACES(( "Toggle main menu, click on desktop/scrollbuf-top etc\n"));
            txwPostMsg((TXWHANDLE) txwa->focus, TXWM_CHAR, 0, (TXWMPARAM) TXk_MENU);

            if (txwa->mainmenu != 0)            // there IS a main menu
            {
               short deco = (short) txwQueryWindowUShort( hwnd, TXQWS_Y);

               if ((TXMOUSECOL() > (win->client.right - deco)) && // Over btnName 'hot' area
                   (TXMOUSECOL() < (win->client.right - 3   ))  ) // but BEFORE the (fake) [X]
               {
                  TRACES(("Toggle to alternate menu (MENU_CYCLE), click on menu-name\n"));
                  txwPostMsg( txwQueryWindow( hwnd, TXQW_OWNER), TXWM_COMMAND,
                             (TXWMPARAM) TXCMD_MENU_CYCLE, (TXWMPARAM) TXCMDSRC_MENU);
               }
            }
         }

         //- Close active dialog frame/canvas/textview on click closebutton [X] or menu-cycle
         else if ( (txwIsDescendant( hwnd, txwa->modality)) &&
                   (row < 0) && (TXMOUSECOL() > (win->client.right -20)) &&
                  (((win->class == TXW_FRAME)     ||
                    (win->class == TXW_CANVAS)    ||
                    (win->class == TXW_TEXTVIEW)) &&
                   ((win->style  & TXCS_CLOSE_BUTTON))))
         {
            if (flags & TXFF_MINIMIZED)
            {
               TRACES(( "Undo minimized window [F12] button\n"));

               txwPostMsg( hwnd, TXWM_CHAR, 0, (TXWMPARAM) TXk_F12);
            }
            else
            {
               TRACES(( "Close dialog on click on [X] close button\n"));
               txwPostMsg( hwnd, TXWM_CLOSE, 0, 0);
               if (TXMOUSECOL() > (win->client.right -3))
               {
                  txwPostMsg( hwnd, TXWM_CLOSE, 0, 0);
               }
            }
         }

         //- Translate click on 'scroll-bar' to UP/DOWN or PGUP/PGDN keys
         else if ( (txwIsDescendant( hwnd, txwa->modality)) && // active window
                   (TXMOUSECOL() >= (win->client.right))    && // at right-side
                   ((win->class  == TXW_TEXTVIEW)           || // all textviews
                   ((win->class  == TXW_LISTBOX)            && // all lists scroll
                   ((win->style  &  TXLS_DROP_VALUE) == 0)) )) // except dropdown
         {
            ULONG   key = TXk_PGDN;

            if      (TXMOUSEROW() == win->client.top)    key = TXk_UP;
            else if (TXMOUSEROW() == win->client.bottom) key = TXk_DOWN;
            else if (row < (sy / 2))                     key = TXk_PGUP;

            TRACES(( "BORDER Scrollbar click to key: %s\n", txwKeyDescription(key)));
            txwPostMsg( hwnd, TXWM_CHAR, 0, (TXWMPARAM) key);
         }

         //- Translate click in 'scroll-bar' area scrollbuffer to UP/DOWN or PGUP/PGDN keys
         else if ((hwnd == (TXWHANDLE) txwa->sbview ) && (col + 3 >= sx))
         {
            ULONG   key = TXk_PGDN;

            if      (row < 4)        key = TXk_UP;
            else if (row + 6 > sy)   key = TXk_DOWN;
            else if (row < (sy / 2)) key = TXk_PGUP;

            TRACES(( "SCROLLBUF 'scrollbar' click to key: %s\n", txwKeyDescription(key)));
            txwPostMsg( hwnd, TXWM_CHAR, 0, (TXWMPARAM) key);
         }
         else
         {
            BOOL       bProbeMarkClick = FALSE; // check for SBVIEW mark actions

            TRACES(("No special actions performed, try to set FOCUS, and post-process ...\n"));

            //- other cases, try to switch focus, if OK, perform post-processing
            if (txwIsWindowEnabled(hwnd))       // enabled ?
            {
               if (txwSetFocus( hwnd) == NO_ERROR) // to window under mouse
               {
                  TRACES(( "Changed focus to clicked window, post-process ...\n"));

                  //- to be refined, below processing should be moved to the
                  //- the class specific procedures, that call this function
                  //- and on _PENDING do their own thing ... (see HexEdit)

                  switch (win->class)           // additional positioning
                  {
                     case TXW_ENTRYFIELD:       // set cursor to mouse pos
                        {
                           TXENTRYFIELD *dat = &win->ef;

                           if (dat->buf && strlen( dat->buf))
                           {
                              dat->curpos = min( col, txSlen(dat->buf) - dat->leftcol);
                              txwInvalidateWindow( hwnd, TRUE, TRUE);
                           }
                           bProbeMarkClick = TRUE; // alow copy/paste/unmark in entryfields
                        }
                        break;

                     case TXW_LISTBOX:
                        {
                           TXLISTBOX *dat  = &win->lb;
                           TXSELIST  *list = dat->list;

                           TRACES(("Drop-value: %s\n", (win->style & TXLS_DROP_VALUE) ? "YES" : "NO"));
                           TRACES(("Drop-menu:  %s\n", (win->style & TXLS_DROP_MENU ) ? "YES" : "NO"));

                           if ((list != NULL) && (list->count != 0))
                           {
                              if      (win->style & TXLS_DROP_VALUE)
                              {
                                 if (TXMOUSECOL() > (win->client.right -2))
                                 {
                                    //- when at end of control, force a drop-down
                                    txwPostMsg( hwnd, TXWM_CHAR, 0, (TXWMPARAM) TXa_ENTER);
                                 }
                              }
                              else if (win->style & TXLS_DROP_MENU) // menu heading
                              {
                              }
                              else              // multi-line, including menu
                              {
                                 ULONG      li = (ULONG) (list->top + row);

                                 TRACES(("Listbox style:%8.8lx li:%lu\n", win->style, li));

                                 if (li < list->count)
                                 {
                                    TRACES(("Item: '%s' = '%s'\n", list->items[li]->text,
                                                                   list->items[li]->desc));

                                    //- always ENTER even on disabled (for error message)
                                    dat->cpos = row; //- Visual, and list position update
                                    TxSelSetSelected( list, li, TX_TOGGLE, TXSF_MARK_STD);
                                    txwListMark2trhText( hwnd); //- update count in title

                                    txwPostMsg( owner, TXWM_CONTROL,
                                                TXMPFROM2SH(wid,TXLN_SELECT),
                                                (TXWMPARAM) (list));

                                    txwInvalidateWindow( hwnd, TRUE, TRUE);

                                    //- Don't execute on single click when NEEDDBLCLK is set
                                    if (((win->style & TXLS_NEEDDBLCLK) == 0) ||
                                         (msg == TXWM_BUTTONDBLCLK))
                                    {
                                       //- execute the selected line in list
                                       txwPostMsg( hwnd, TXWM_CHAR, 0, (TXWMPARAM) TXk_ENTER);
                                    }
                                 }
                              }
                           }
                        }
                        break;

                     case TXW_BUTTON:           // push the button
                        txwPostMsg( hwnd, TXWM_CHAR, 0, (TXWMPARAM) TXk_SPACE);
                        break;

                     case TXW_SBVIEW:
                     case TXW_TEXTVIEW:
                        bProbeMarkClick = TRUE;
                        break;

                     default:
                        rc = TX_PENDING;
                        break;
                  }
               }
               else
               {
                  bProbeMarkClick = TRUE;
               }
            }
            else                                // Window not-enabled (like text area on Msg)
            {
               switch (win->class)              // Marked-area / Clip-board handling
               {
                  case TXW_LISTBOX:       // could be a menu-heading, with inactive SBVIEW under it
                  case TXW_SBVIEW:
                  case TXW_TEXTVIEW:
                     bProbeMarkClick = TRUE;
                     break;

                  default:
                     break;
               }

            }
            if (bProbeMarkClick)   //- ACTIVE windows that handle clipboard, or inactive ones relaying to SBVIEW
            {
               TRACES(("Probing MARK click, to translate to key\n"));
               if ((win->class == TXW_TEXTVIEW)   || // active text-view of inactive one in msgbox etc
                   (win->class == TXW_SBVIEW)     || // active output-window
                   (win->class == TXW_ENTRYFIELD) || // active entryfield
                   ((txwa->sbview) && (txwa->sbview->window->sb.sbdata->markSize)))
               {
                  //- there is a mark in Output-Window, has priority over active/focus window!
                  if      ((TXMOUSEKEYSCA() & TXm_KS_ALT) != 0) // Alt-click is COPY mark (or default area) to clipboard
                  {
                     txwPostMsg( hwnd, TXWM_CHAR, 0, (TXWMPARAM) TXa_C); // simulate keyboard c-copy action
                  }
                  else if ((TXMOUSEKEYSCA() & TXm_KS_CTRL) != 0) // Ctrl-click is PASTE from clipboard
                  {
                     txwPostMsg( hwnd, TXWM_CHAR, 0, (TXWMPARAM) TXa_V); // simulate keyboard c-paste action
                  }
                  else                          // unmark on click/doubleclick TEXT/SB etc
                  {
                     txwPostMsg( hwnd, TXWM_CHAR, 0, (TXWMPARAM) TXa_U); // simulate keyboard unmark action
                  }
               }
            }
         }
      }
   }
   RETURN( rc);
}                                               // end 'txwMouseButtonDown'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Dismiss dialog and pass-on result code
/*****************************************************************************/
ULONG txwDismissDlg
(
   TXWHANDLE           hwnd,                    // IN    dialog handle
   ULONG               result                   // IN    result code
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXWINBASE          *wnd;
   TXWINDOW           *win;

   ENTER();
   TRACES(("Dismiss dlg %8.8lx with RC:%8.8lx = %lu\n", hwnd, result, result));

   if ((wnd = txwValidateHandle( hwnd, &win)) != NULL)
   {
      win->dlgResult        = result;
      wnd->us[TXQWS_FLAGS] |= TXFF_DLGDISMISSED;

      txwShowWindow( hwnd, FALSE);              // hide the window
   }
   else
   {
      rc = TX_INVALID_HANDLE;
   }
   #if defined (DEV32)
      txwInvalidateAll();                       // avoid VIO 64Kb buffer bug
   #endif
   RETURN (rc);
}                                               // end 'txwDismissDlg'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create a Dialog from memory-structures (created frame + linked controls)
/*****************************************************************************/
TXWHANDLE txwCreateDlg
(
   TXWHANDLE           parent,                  // IN    parent (ignored!)
   TXWHANDLE           owner,                   // IN    owner  window
   TXWINPROC           dlgproc,                 // IN    dialog procedure
   TXWHANDLE           dlg,                     // IN    dialog frame window
   PVOID               cData                    // IN    user control data
)
{
   TXWHANDLE           rc = 0;                  // function return
   TXWINBASE          *wnd;                     // frame window (dialog)
   TXWINDOW           *win;

   ENTER();

   TRACES(( "Frame hwnd:%8.8lx  dlgproc:%8.8lx  cData:%8.8lx\n", dlg, dlgproc, cData));

   //- to be refined (long term) create dialog from template instead of
   //- complete dialog frame + controls setup with CreateWindow
   //- until then, the parent specified here will be ignored (position)

   if ((wnd = txwValidateHandle( dlg, &win)) != NULL)
   {
      TXWHANDLE        focus;
      TXWINBASE       *vwnd;                    // validated winbase

      if ((focus = txwWindowFromID( dlg, win->dlgFocusID)) == TXHWND_NULL)
      {
         focus = dlg;                           // focus to dlg window itself
      }                                         // single control, no frame

      //- owner and dlgproc specified here overrule the template
      //- must be specified as 0 to CreateWindow (owner and winproc)

      if ((vwnd = txwValidateHandle( owner, NULL)) != NULL)
      {
         wnd->owner     = vwnd;
      }
      else
      {
         wnd->owner     = txwa->desktop;
      }
      wnd->winproc      = dlgproc;              // attach dialog procedure

      win->style            &= ~TXWS_VISIBLE;       //- set invisible
      wnd->us[TXQWS_FLAGS]  &= ~TXFF_DLGDISMISSED;  //- clear dismissed flag

      if (cData != NULL)                        // optional, USER PTR could be set already
      {
         txwSetWindowPtr( dlg, TXQWP_USER,  cData); // make user data available
      }
      txwPostMsg( (TXWHANDLE) wnd, TXWM_INITDLG, (TXWMPARAM) focus, (TXWMPARAM) cData);

      rc = dlg;
   }
   RETURN (rc);
}                                               // end 'txwCreateDlg'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Process a (modal) dialog using a local msg-loop
/*****************************************************************************/
ULONG txwProcessDlg                             // RET   dialog rc (dismiss)
(
   TXWHANDLE           hwnd                     // IN    dialog handle
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXWINBASE          *wnd;                     // frame window (dialog)
   TXWINDOW           *win;

   ENTER();
   TRWINS("ProcessDlg");

   if ((wnd = txwValidateHandle( hwnd, &win)) != NULL)
   {
      TXWQMSG          qmsg;
      TXWHANDLE        oldModality;             // previous modality window

      //- Following is a workarround for focus/loose-focus paint problems
      oldModality    = txwa->modality;          // remember previous ...
      txwa->modality = TXHWND_DESKTOP;          // temporary to desktop
      txwSetFocus(     TXHWND_DESKTOP);         // remove focus from current
      txwSendMsg(      TXHWND_DESKTOP,          // and from desktop too!
                       TXWM_SETFOCUS, (TXWMPARAM) FALSE, 0);

      txwa->modality = hwnd;                    // and make dialog modal

      txwShowWindow( hwnd, TRUE);               // make the dialog visible

      win->dlgResult = TXDID_CANCEL;            // if not dismissed properly

      while (((wnd->us[TXQWS_FLAGS] & TXFF_DLGDISMISSED) == 0) &&
             (txwGetMsg(  &qmsg)) )
      {
         txwDispatchMsg( &qmsg);
      }

      txwa->modality = oldModality;             // restore previous

      if      (qmsg.msg == TXWM_QUIT)           // signal QUIT to desktop message loop
      {
         txwPostMsg( TXHWND_DESKTOP, TXWM_QUIT, 0, 0);
      }
      else if (qmsg.msg == TXWM_SCREENRESIZE)   // signal RESIZE to desktop message loop
      {
         txwPostMsg( TXHWND_DESKTOP, TXWM_SCREENRESIZE, 0, 0);
      }
      rc = win->dlgResult;                      // return dialog result
   }
   else
   {
      rc = TX_INVALID_HANDLE;
   }
   RETURN (rc);
}                                               // end 'txwProcessDlg'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create, Process and Destroy a (modal) dialog
/*****************************************************************************/
ULONG txwDlgBox                                 // RET   dialog rc (dismiss)
(
   TXWHANDLE           parent,                  // IN    parent (ignored!)
   TXWHANDLE           owner,                   // IN    owner  window
   TXWINPROC           dlgproc,                 // IN    dialog procedure
   TXWHANDLE           dlg,                     // IN    dialog frame window
   PVOID               cData                    // IN    user control data
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXWINBASE          *wnd;                     // frame window (dialog)
   TXWINDOW           *win;

   ENTER();

   if ((wnd = txwValidateHandle( dlg, &win)) != NULL)
   {
      txwCreateDlg( parent, owner, dlgproc, dlg, cData);

      rc = txwProcessDlg( dlg);

      txwDestroyWindow( dlg);
   }
   else
   {
      rc = TX_INVALID_HANDLE;
   }
   RETURN (rc);
}                                               // end 'txwDlgBox'
/*---------------------------------------------------------------------------*/

