
// First include all needed standard header files
#include <stdlib.h>
#include <stdio.h>
#include <process.h>
#include <string.h>
#include <math.h>
#include <float.h>
#include <time.h>

#ifndef M_PI
#define M_PI 3.14159
#endif

// Then the needed OS/2 API stuffs
#define INCL_TYPES
#define INCL_WIN
#define INCL_DOS
#define INCL_DOSERRORS
#define INCL_GPI
#include <os2.h>

#include <cairo.h>
#include <cairo-os2.h>

#define DEBUG_BUILD

int bShutdown = 0;
int bNeedRedraw = 0;
int bAnalogueClock = 0;

// --------------- Functions: -----------------------------------------------

#ifdef DEBUG_BUILD
void MorphToPM()
{
  PPIB pib;
  PTIB tib;

  DosGetInfoBlocks(&tib, &pib);

  // Change flag from VIO to PM:
  if (pib->pib_ultype==2) pib->pib_ultype = 3;
}
#endif

cairo_surface_t *surface = NULL;
cairo_t *cr = NULL;
double line_width;
cairo_font_extents_t font_extents;

static void inline DisableFPUException()
{
  unsigned short usCW;

  // Some OS/2 PM API calls modify the FPU Control Word,
  // but forget to restore it.

  // This can result in XCPT_FLOAT_INVALID_OPCODE exceptions,
  // so to be sure, we always disable Invalid Opcode FPU exception
  // before using FPU stuffs with Cairo or from this application.

  usCW = _control87(0, 0);
  usCW = usCW | EM_INVALID | 0x80;
  _control87(usCW, MCW_EM | 0x80);
}

void DrawPage_SimpleColors(int iDrawCounter)
{
  // Draw three color bars
  cairo_set_source_rgb(cr, 1.0, 0, 0);
  cairo_rectangle(cr,
                  0.2, 0.1,
                  0.2, 0.8);
  cairo_fill(cr);

  cairo_set_source_rgb(cr, 0, 1.0, 0);
  cairo_rectangle(cr,
                  0.4, 0.1,
                  0.2, 0.8);
  cairo_fill(cr);

  cairo_set_source_rgb(cr, 0, 0, 1.0);
  cairo_rectangle(cr,
                  0.6, 0.1,
                  0.2, 0.8);
  cairo_fill(cr);

  // Text
  cairo_set_source_rgb(cr, 0.8, 0.8 ,0.8);
  cairo_select_font_face(cr, "Times", CAIRO_FONT_SLANT_NORMAL,
                         CAIRO_FONT_WEIGHT_BOLD);
  cairo_set_font_size(cr, 0.10);

  cairo_move_to(cr, 0.02, 0.96);
  cairo_show_text(cr, "Simple colors");
}

void DrawPage_AlphaColors(int iDrawCounter)
{
  // Draw three color bars
  cairo_set_source_rgba(cr, 1.0, 0, 0, iDrawCounter/10.0);
  cairo_rectangle(cr,
                  0.2, 0.1,
                  0.2, 0.8);
  cairo_fill(cr);

  cairo_set_source_rgba(cr, 0, 1.0, 0, 1-iDrawCounter/10.0);
  cairo_rectangle(cr,
                  0.4, 0.1,
                  0.2, 0.8);
  cairo_fill(cr);

  cairo_set_source_rgba(cr, 0, 0, 1.0, 0.5+iDrawCounter/20.0);
  cairo_rectangle(cr,
                  0.6, 0.1,
                  0.2, 0.8);
  cairo_fill(cr);

  // Text
  cairo_set_source_rgb(cr, 0.8, 0.8 ,0.8);
  cairo_select_font_face(cr, "Times", CAIRO_FONT_SLANT_NORMAL,
                         CAIRO_FONT_WEIGHT_BOLD);
  cairo_set_font_size(cr, 0.10);

  cairo_move_to(cr, 0.02, 0.96);
  cairo_show_text(cr, "Alpha colors");
}

void DrawPage_LinearGradients(int iDrawCounter)
{
  cairo_pattern_t *pattern1;
  cairo_pattern_t *pattern2;
  cairo_pattern_t *pattern3;

  pattern1 = cairo_pattern_create_linear(0+iDrawCounter/10.0, 0, 1, 1);
  pattern2 = cairo_pattern_create_linear(0.5, 0+iDrawCounter/10.0, 1, 0.5);
  pattern3 = cairo_pattern_create_linear(1, 1, 0+iDrawCounter/10.0, 0);

  cairo_pattern_add_color_stop_rgb(pattern1, 0,   1, 0, 0);
  cairo_pattern_add_color_stop_rgb(pattern1, 0.5, 1, 1, 0);
  cairo_pattern_add_color_stop_rgb(pattern1, 1,   0+iDrawCounter/10.0, 1, 0);

  cairo_pattern_add_color_stop_rgb(pattern2, 0,   0, 1, 0);
  cairo_pattern_add_color_stop_rgb(pattern2, 0.5, 0, 1+iDrawCounter/10.0, 1);
  cairo_pattern_add_color_stop_rgb(pattern2, 1,   0, 0, 1);

  cairo_pattern_add_color_stop_rgb(pattern3, 0,   0, 0+iDrawCounter/10.0, 1);
  cairo_pattern_add_color_stop_rgb(pattern3, 0.5, 1, 0, 1);
  cairo_pattern_add_color_stop_rgb(pattern3, 1,   1, 0, 0);

  // Draw three color bars
  cairo_set_source(cr, pattern1);
  cairo_rectangle(cr,
                  0.2, 0.1,
                  0.2, 0.8);
  cairo_fill(cr);

  cairo_set_source(cr, pattern2);
  cairo_rectangle(cr,
                  0.4, 0.1,
                  0.2, 0.8);
  cairo_fill(cr);

  cairo_set_source(cr, pattern3);
  cairo_rectangle(cr,
                  0.6, 0.1,
                  0.2, 0.8);
  cairo_fill(cr);

  cairo_pattern_destroy(pattern3);
  cairo_pattern_destroy(pattern2);
  cairo_pattern_destroy(pattern1);


  // Text
  cairo_set_source_rgb(cr, 0.8, 0.8 ,0.8);
  cairo_select_font_face(cr, "Times", CAIRO_FONT_SLANT_NORMAL,
                         CAIRO_FONT_WEIGHT_BOLD);
  cairo_set_font_size(cr, 0.10);

  cairo_move_to(cr, 0.02, 0.96);
  cairo_show_text(cr, "Linear gradients");
}

void DrawPage_LinearGradientsWithAlpha(int iDrawCounter)
{
  cairo_pattern_t *pattern1;
  cairo_pattern_t *pattern2;
  cairo_pattern_t *pattern3;

  pattern1 = cairo_pattern_create_linear(0+iDrawCounter/10.0, 0, 1, 1);
  pattern2 = cairo_pattern_create_linear(0.5, 0+iDrawCounter/10.0, 1, 0.5);
  pattern3 = cairo_pattern_create_linear(1, 1, 0+iDrawCounter/10.0, 0);

  cairo_pattern_add_color_stop_rgba(pattern1, 0,   1, 0, 0, iDrawCounter/10.0);
  cairo_pattern_add_color_stop_rgba(pattern1, 0.5, 1, 1, 0, 1-iDrawCounter/10.0);
  cairo_pattern_add_color_stop_rgba(pattern1, 1,   0+iDrawCounter/10.0, 1, 0, iDrawCounter/10.0);

  cairo_pattern_add_color_stop_rgba(pattern2, 0,   0, 1, 0, iDrawCounter/10.0);
  cairo_pattern_add_color_stop_rgba(pattern2, 0.5, 0, 1+iDrawCounter/10.0, 1, 1-iDrawCounter/10.0);
  cairo_pattern_add_color_stop_rgba(pattern2, 1,   0, 0, 1, 1-iDrawCounter/10.0);

  cairo_pattern_add_color_stop_rgba(pattern3, 0,   0, 0+iDrawCounter/10.0, 1, 1-iDrawCounter/10.0);
  cairo_pattern_add_color_stop_rgba(pattern3, 0.5, 1, 0, 1, iDrawCounter/10.0);
  cairo_pattern_add_color_stop_rgba(pattern3, 1,   1, 0, 0, iDrawCounter/10.0);

  // Draw three color bars
  cairo_set_source(cr, pattern1);
  cairo_rectangle(cr,
                  0.2, 0.1,
                  0.2, 0.8);
  cairo_fill(cr);

  cairo_set_source(cr, pattern2);
  cairo_rectangle(cr,
                  0.4, 0.1,
                  0.2, 0.8);
  cairo_fill(cr);

  cairo_set_source(cr, pattern3);
  cairo_rectangle(cr,
                  0.6, 0.1,
                  0.2, 0.8);
  cairo_fill(cr);

  cairo_pattern_destroy(pattern3);
  cairo_pattern_destroy(pattern2);
  cairo_pattern_destroy(pattern1);


  // Text
  cairo_set_source_rgb(cr, 0.8, 0.8 ,0.8);
  cairo_select_font_face(cr, "Times", CAIRO_FONT_SLANT_NORMAL,
                         CAIRO_FONT_WEIGHT_BOLD);
  cairo_set_font_size(cr, 0.06);

  cairo_move_to(cr, 0.02, 0.96);
  cairo_show_text(cr, "Linear gradients w/alpha");
}

void DrawPage_RadialGradients(int iDrawCounter)
{
  cairo_pattern_t *pattern1;
  cairo_pattern_t *pattern2;
  cairo_pattern_t *pattern3;

  pattern1 = cairo_pattern_create_radial(0.5, 0.5, 0.1+iDrawCounter/10.0, 0.5, 0.5, 0.5);
  pattern2 = cairo_pattern_create_radial(0.4, 0.4+iDrawCounter/40.0, 0.1, 0.5, 0.5, 0.5);
  pattern3 = cairo_pattern_create_radial(0.2, 0.2, 0.4, 0.2, 0.4, 0.6+iDrawCounter/30.0);

  cairo_pattern_add_color_stop_rgb(pattern1, 0,   1, 0, 0+iDrawCounter/10.0);
  cairo_pattern_add_color_stop_rgb(pattern1, 0.5, 1, 1, 0);
  cairo_pattern_add_color_stop_rgb(pattern1, 1,   0+iDrawCounter/10.0, 1, 0);

  cairo_pattern_add_color_stop_rgb(pattern2, 0,   0, 1, 0);
  cairo_pattern_add_color_stop_rgb(pattern2, 0.5, 0, 1-iDrawCounter/10.0, 1);
  cairo_pattern_add_color_stop_rgb(pattern2, 1,   0, 0, 1);

  cairo_pattern_add_color_stop_rgb(pattern3, 0,   0, 0, 1);
  cairo_pattern_add_color_stop_rgb(pattern3, 0.5, 1-iDrawCounter/10.0, 0, 1);
  cairo_pattern_add_color_stop_rgb(pattern3, 1,   1, 0+iDrawCounter/10.0, 0);

  // Draw three color bars
  cairo_set_source(cr, pattern1);
  cairo_rectangle(cr,
                  0.2, 0.1,
                  0.2, 0.8);
  cairo_fill(cr);

  cairo_set_source(cr, pattern2);
  cairo_rectangle(cr,
                  0.4, 0.1,
                  0.2, 0.8);
  cairo_fill(cr);

  cairo_set_source(cr, pattern3);
  cairo_rectangle(cr,
                  0.6, 0.1,
                  0.2, 0.8);
  cairo_fill(cr);

  cairo_pattern_destroy(pattern3);
  cairo_pattern_destroy(pattern2);
  cairo_pattern_destroy(pattern1);

  // Text
  cairo_set_source_rgb(cr, 0.8, 0.8 ,0.8);
  cairo_select_font_face(cr, "Times", CAIRO_FONT_SLANT_NORMAL,
                         CAIRO_FONT_WEIGHT_BOLD);
  cairo_set_font_size(cr, 0.10);

  cairo_move_to(cr, 0.02, 0.96);
  cairo_show_text(cr, "Radial gradients");
}

void DrawPage_ImagePattern(int iDrawCounter)
{
  cairo_surface_t *image;
  cairo_pattern_t *pattern;
  cairo_matrix_t   matrix;
  int width, height;

  cairo_save(cr);

  image = cairo_image_surface_create_from_png("image.png");
  width = cairo_image_surface_get_width(image);
  height= cairo_image_surface_get_height(image);

  pattern = cairo_pattern_create_for_surface(image);
  cairo_pattern_set_extend(pattern, CAIRO_EXTEND_REPEAT);


  cairo_matrix_init_scale(&matrix, width*(0.1+iDrawCounter/5.0), height * (0.1 + iDrawCounter/5.0));
  cairo_matrix_rotate(&matrix, 3.1415/30*iDrawCounter);
  cairo_pattern_set_matrix(pattern, &matrix);

  // Draw three color bars
  cairo_set_source(cr, pattern);
  cairo_rectangle(cr,
                  0.2, 0.1,
                  0.2, 0.8);
  cairo_fill(cr);


  cairo_matrix_init_scale(&matrix, width*(0.5 + iDrawCounter/10.0), height * 0.5);
  cairo_pattern_set_matrix(pattern, &matrix);

  cairo_set_source(cr, pattern);
  cairo_rectangle(cr,
                  0.4, 0.1,
                  0.2, 0.8);
  cairo_fill(cr);


  cairo_matrix_init_scale(&matrix, width*1.0, height * (0.2 + iDrawCounter/10.0));
  cairo_pattern_set_matrix(pattern, &matrix);

  cairo_set_source(cr, pattern);
  cairo_rectangle(cr,
                  0.6, 0.1,
                  0.2, 0.8);
  cairo_fill(cr);

  cairo_pattern_destroy(pattern);
  cairo_surface_destroy(image);

  cairo_restore(cr);
  // Text
  cairo_set_source_rgb(cr, 0.8, 0.8 ,0.8);
  cairo_select_font_face(cr, "Times", CAIRO_FONT_SLANT_NORMAL,
                         CAIRO_FONT_WEIGHT_BOLD);
  cairo_set_font_size(cr, 0.10);

  cairo_move_to(cr, 0.02, 0.96);
  cairo_show_text(cr, "Image pattern");
}

void DrawImage(int iDrawCounter, int iWidth, int iHeight)
{
  int iPageCounter;
  int x, y;
  DisableFPUException();

  cairo_save(cr);
  cairo_scale(cr, iWidth, iHeight);

  // Clear background with black
  cairo_set_source_rgb(cr, 0, 0, 0);
  cairo_rectangle(cr,
                  0, 0,
                  1, 1);
  cairo_fill(cr);
  // Draw checkers
  for (y=0; y<10; y++)
    for (x=0; x<5; x++)
    {
      cairo_set_source_rgb(cr, 0.3, 0.3, 0.3);
      cairo_rectangle(cr,
                      x*0.2 + (y%2)*0.1, y*0.1,
                      0.1, 0.1);
      cairo_fill(cr);
    }


  // Draw the real meat
  iPageCounter = iDrawCounter / 10;
  switch (iPageCounter % 6)
  {
    default:
    case 0:
      DrawPage_SimpleColors(iDrawCounter % 10);
      break;
    case 1:
      DrawPage_AlphaColors(iDrawCounter % 10);
      break;
    case 2:
      DrawPage_LinearGradients(iDrawCounter % 10);
      break;
    case 3:
      DrawPage_LinearGradientsWithAlpha(iDrawCounter % 10);
      break;
    case 4:
      DrawPage_RadialGradients(iDrawCounter % 10);
      break;
    case 5:
      DrawPage_ImagePattern(iDrawCounter % 10);
      break;
  }

  /* Draw horizontal and vertical lines */
  cairo_set_line_width(cr, 0.05);
  switch ((iDrawCounter / 7) % 5)
  {
    case 0:
      x = CAIRO_OPERATOR_ATOP;
      break;
    case 1:
      x = CAIRO_OPERATOR_ADD;
      break;
    case 2:
      x = CAIRO_OPERATOR_XOR;
      break;
    case 3:
      x = CAIRO_OPERATOR_DEST_OVER;
      break;
    case 4:
      x = CAIRO_OPERATOR_SOURCE;
      break;
  }

  cairo_set_source_rgba(cr, 0.5, 0.5, 0.5, 1);
  cairo_set_operator(cr, x);
  cairo_move_to(cr, 0, 0.05 + x*0.06);
  cairo_line_to(cr, 1, 0.05 + x*0.06);
  cairo_move_to(cr, 0.05 + x*0.06, 0);
  cairo_line_to(cr, 0.05 + x*0.06, 1);
  cairo_stroke(cr);

  /* Show operator name */
  cairo_set_source_rgba(cr, 0, 0, 0.9, 1);
  cairo_move_to(cr, 0, 0.065 + x*0.06);
  cairo_set_operator(cr, CAIRO_OPERATOR_ATOP);
  cairo_select_font_face(cr, "Times", CAIRO_FONT_SLANT_NORMAL,
                         CAIRO_FONT_WEIGHT_BOLD);
  cairo_set_font_size(cr, 0.05);

  switch (x)
  {
    case CAIRO_OPERATOR_CLEAR:
      cairo_show_text(cr, "CLEAR");
      break;
    case CAIRO_OPERATOR_SOURCE:
      cairo_show_text(cr, "SOURCE");
      break;
    case CAIRO_OPERATOR_OVER:
      cairo_show_text(cr, "OVER");
      break;
    case CAIRO_OPERATOR_IN:
      cairo_show_text(cr, "IN");
      break;
    case CAIRO_OPERATOR_OUT:
      cairo_show_text(cr, "OUT");
      break;
    case CAIRO_OPERATOR_ATOP:
      cairo_show_text(cr, "ATOP");
      break;
    case CAIRO_OPERATOR_DEST:
      cairo_show_text(cr, "DEST");
      break;
    case CAIRO_OPERATOR_DEST_OVER:
      cairo_show_text(cr, "DEST_OVER");
      break;
    case CAIRO_OPERATOR_DEST_IN:
      cairo_show_text(cr, "DEST_IN");
      break;
    case CAIRO_OPERATOR_DEST_OUT:
      cairo_show_text(cr, "DEST_OUT");
      break;
    case CAIRO_OPERATOR_DEST_ATOP:
      cairo_show_text(cr, "DEST_ATOP");
      break;
    case CAIRO_OPERATOR_XOR:
      cairo_show_text(cr, "XOR");
      break;
    case CAIRO_OPERATOR_ADD:
      cairo_show_text(cr, "ADD");
      break;
    case CAIRO_OPERATOR_SATURATE:
      cairo_show_text(cr, "SATURATE");
      break;
    default:
      cairo_show_text(cr, "Unknown");
      break;
  }

  cairo_set_operator(cr, x);

  // Example group text
  cairo_set_source_rgba(cr, 0.8, 0.7, 0.9, 0.5);
  cairo_select_font_face(cr, "Sans", CAIRO_FONT_SLANT_NORMAL,
                         CAIRO_FONT_WEIGHT_BOLD);
  cairo_set_font_size(cr, 0.15);

  cairo_move_to(cr, 0.02, 0.15);
  cairo_text_path(cr, "Operators");
  cairo_fill_preserve(cr);
  cairo_set_source_rgba(cr, 0.5, 0.5, 0.5, 0.4);
  cairo_set_line_width(cr, 0.01);
  cairo_close_path (cr);
  cairo_stroke (cr);

  cairo_restore(cr);
}


MRESULT EXPENTRY MainWindowProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
{
  switch (msg)
  {
    case WM_PAINT:
      {
        HPS hpsBeginPaint;
        RECTL rclRect;

        hpsBeginPaint = WinBeginPaint(hwnd, NULL, &rclRect);

        if (surface)
        {
#ifdef DEBUG_BUILD
//          printf("[MainWindowProc] : Repaint\n");
#endif
          if (!bNeedRedraw)
          {
            // Repaint the stuffs
            cairo_os2_surface_refresh_window(surface, hpsBeginPaint, &rclRect);
          } else
          {
            // It was resized or something, so we better show black image
            WinFillRect(hpsBeginPaint, &rclRect, CLR_BLACK);
          }
        }

        // All done!
        WinEndPaint(hpsBeginPaint);
        return (MRESULT) FALSE;
      }
    case WM_SIZE:
#ifdef DEBUG_BUILD
//        printf("[MainWindowProc] : In WM_SIZE\n");
#endif
        if (surface)
        {
          cairo_os2_surface_set_size(surface,
                                     SHORT1FROMMP(mp2), // New CX
                                     SHORT2FROMMP(mp2), // New CY
                                     6000);
          bNeedRedraw = 1;
        }
        break;
  }
  return WinDefWindowProc(hwnd, msg, mp1, mp2);
}

MRESULT EXPENTRY NewFrameProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
{
  PFNWP pOldFrameProc;
  MRESULT result;
  PTRACKINFO ti;

  pOldFrameProc = (PFNWP) WinQueryWindowULong(hwnd, QWL_USER);

  result = (*pOldFrameProc)(hwnd, msg, mp1, mp2);

  if (msg == WM_QUERYTRACKINFO)
  {
    ti = (PTRACKINFO) mp2;
    ti->ptlMinTrackSize.x = 300;
    ti->ptlMinTrackSize.y = 200;
  }
  return result;
}

void fnDrawThread(void *pParm)
{
  HWND hwndClientWindow = (HWND) pParm;
  HPS hpsClientWindow;
  SWP swpTemp;
  int iDelayCounter;
  int iDrawCounter;

  // Create Cairo surface
  hpsClientWindow = WinGetPS(hwndClientWindow);
  WinQueryWindowPos(hwndClientWindow, &swpTemp);

  surface = cairo_os2_surface_create(hpsClientWindow,
                                     swpTemp.cx,
                                     swpTemp.cy);
  cr = cairo_create(surface);
  cairo_os2_surface_set_manual_window_refresh(surface,
                                              TRUE);

  iDrawCounter = 0;

  while (!bShutdown)
  {
    // Draw image into cairo surface
    WinQueryWindowPos(hwndClientWindow, &swpTemp);
    DrawImage(iDrawCounter, swpTemp.cx, swpTemp.cy);
    iDrawCounter++;
    // Make changes visible
    bNeedRedraw = 0;
    WinInvalidateRect(hwndClientWindow, NULL, FALSE);

    // Wait a bit
    iDelayCounter = 0;
//    while ((iDelayCounter<90) && (!bNeedRedraw) && (!bShutdown))
//    {
      DosSleep(32);
//      iDelayCounter++;
//    }
  }

  // Destroy Cairo surface
  if (cr)
    cairo_destroy (cr);
  if (surface)
    cairo_surface_destroy (surface);

  WinReleasePS(hpsClientWindow);

  _endthread();
}

void PMTest()
{
  HAB hab;
  HMQ hmq;
  QMSG msg;
  ULONG flCreaFlags, flFrameStyle, flClientStyle;
  SWP swap;
  PFNWP pOldFrameProc;
  HWND hwndFrame;
  HWND hwndClient;
  unsigned long tidDrawThread;


#ifdef DEBUG_BUILD
  printf("[PMThread] : Starting\n");
#endif
  // Setup this thread for using PM (sending messages etc...)
  hab = WinInitialize(0);
  hmq = WinCreateMsgQueue(hab, 0);

  // Create main window
  WinRegisterClass(hab, "PrivateClassName",  // Register a private class, so
		   MainWindowProc,           // we can use our own window procedure for frame
		   0, 0);

  // Set defaults for main window:
  flCreaFlags = 

    FCF_TASKLIST | FCF_SYSMENU | FCF_TITLEBAR | FCF_SIZEBORDER |
    FCF_DLGBORDER | FCF_MINMAX |
    FCF_SHELLPOSITION | FCF_NOBYTEALIGN | FCF_AUTOICON;
  flFrameStyle = 
    FS_NOBYTEALIGN | FS_BORDER;
  flClientStyle =
      WS_CLIPCHILDREN;

  hwndFrame = WinCreateStdWindow(HWND_DESKTOP,
				 flFrameStyle,       // Frame window style
				 &flCreaFlags,
                                 "PrivateClassName",
                                 "TestApp for CAIRO.DLL",
				 flClientStyle,//0,   // Client window style
				 NULLHANDLE,
				 0L,
                                 &hwndClient);

  pOldFrameProc = WinSubclassWindow(hwndFrame, NewFrameProc); // Subclass window to be able
                                                              // to handle resize events
  // Store old window proc in frame window's ULONG
  WinSetWindowULong(hwndFrame, QWL_USER, (ULONG) pOldFrameProc);

  // Move the window more down if it would be out of the screen
  WinQueryWindowPos(hwndFrame, &swap);
  if (swap.y+swap.cy > WinQuerySysValue(HWND_DESKTOP, SV_CYSCREEN))
  {
    WinSetWindowPos(hwndFrame, HWND_TOP, swap.x,
		    WinQuerySysValue(HWND_DESKTOP, SV_CYSCREEN) - swap.cy, 0, 0, SWP_MOVE);
    WinQueryWindowPos(hwndFrame, &swap);
  }
  if (swap.x+swap.cx > WinQuerySysValue(HWND_DESKTOP, SV_CXSCREEN))
  {
    if (WinQuerySysValue(HWND_DESKTOP, SV_CXSCREEN) - swap.cx >= 0)
      WinSetWindowPos(hwndFrame, HWND_TOP, 
		      WinQuerySysValue(HWND_DESKTOP, SV_CXSCREEN) - swap.cx, swap.y,  0, 0, SWP_MOVE);
  }

  WinSetWindowPos(hwndFrame, HWND_TOP, 0, 0, 0, 0, SWP_ZORDER | SWP_SHOW | SWP_ACTIVATE);

  // Start drawing
  bShutdown = 0;
  tidDrawThread = _beginthread(fnDrawThread,
                               0,
                               1024*1024,
                               (void *) hwndClient);

#ifdef DEBUG_BUILD
  printf("[PMThread] : Reached WinProcessDlg\n");
#endif
  while (WinGetMsg(hab, &msg, 0, 0, 0))
    WinDispatchMsg(hab, &msg);
#ifdef DEBUG_BUILD
  printf("[PMThread] : Shutting down!\n");
#endif

  // Stop drawing
  bShutdown = 1;
  DosWaitThread(&tidDrawThread, DCWW_WAIT);

  WinDestroyWindow(hwndFrame); // Destroy windows

  // Uninitialize PM
  WinDestroyMsgQueue(hmq);
  WinTerminate(hab);

#ifdef DEBUG_BUILD
  printf("[PMThread] : Stopped\n");
#endif
}



// ---------- Main function -------------------------------------------------

int main(int argc, char *argv[])
{
#ifdef DEBUG_BUILD
  MorphToPM();
  setbuf(stdout, NULL);
  setbuf(stderr, NULL);
#ifdef __WATCOMC__
#if __WATCOMC__<1200
  printf("Compiled with WatcomC v%d.%02d\n", (__WATCOMC__)/100, (__WATCOMC__)%100);
#else
  printf("Compiled with OpenWatcom v%d.%02d\n", (__WATCOMC__-1100)/100, (__WATCOMC__-1100)%100);
#endif
#endif
  fflush(stdout);
#endif

  if (argc>1)
    bAnalogueClock = 1;
  else
    bAnalogueClock = 0;

  PMTest();

#ifdef DEBUG_BUILD
  printf("Everything done! Bye!\n");
#endif

  return 0;
}
