/**
 * PciAc5.c - A device driver for the Opto22 I/O card
 *
 * This file is part of the Drv16 Device Driver Development Kit.
 *
 * Copyright (c) 2013-2014 David Azarewicz david@88watts.net
 *
 * The following source code is provided to you solely for the purpose of
 * assisting you in developing your own OS/2 device drivers. You may use
 * this software in your device drivers free of charge. This Copyright
 * statement may not be removed.
 *
 * PciAc5 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *
 * Drv16 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
#include "Dev16lib.h"
#include "ioctl.h"
#include "version.h"

/* global variables */
char far *BaseAdr;

void StrategyInit(PREQPACKET prp);

/**
 * Return information from this driver
 */
void IoctlInfo(AC5INFO far *pInfo)
{
  pInfo->Data = 0x11223344;
}

/**
 * Read data from an IO port
 */
USHORT IoctlIoRW(AC5INFO far *pInfo)
{
  USHORT usStatus;

  usStatus = 0;

  switch (pInfo->Command)
  {
  case IOCTL_FCN_READ_DATA_0:
    pInfo->Data = ReadDword(BaseAdr+0x00000300);
    break;
  case IOCTL_FCN_READ_DATA_1:
    pInfo->Data = ReadDword(BaseAdr+0x00000700);
    break;
  case IOCTL_FCN_WRITE_DATA_0:
    WriteDword(BaseAdr+0x00000008, pInfo->Data);
    break;
  case IOCTL_FCN_WRITE_DATA_1:
    WriteDword(BaseAdr+0x00000018, pInfo->Data);
    break;
  case IOCTL_FCN_READ_CONFIG_0:
    pInfo->Data = ReadDword(BaseAdr+0x00000100);
    break;
  case IOCTL_FCN_READ_CONFIG_1:
    pInfo->Data = ReadDword(BaseAdr+0x00000500);
    break;
  case IOCTL_FCN_WRITE_CONFIG_0:
    WriteDword(BaseAdr+0x00000004, pInfo->Data);
    break;
  case IOCTL_FCN_WRITE_CONFIG_1:
    WriteDword(BaseAdr+0x00000014, pInfo->Data);
    break;
  default:
    usStatus = RPERR | RPERR_BADCOMMAND;
    break;
  }

  return usStatus;
}

void StrategyIoctl(PREQPACKET prp)
{
  if (prp->ioctl.bCategory != IOCTL_CATEGORY) {
    prp->usStatus |= RPERR | RPERR_BADCOMMAND;
    return;
  }

  if (prp->ioctl.usDataLen != sizeof(AC5INFO)) {
    prp->usStatus |= RPERR | RPERR_BADCOMMAND;
    return;
  }

  switch (prp->ioctl.bFunction) {
  case IOCTL_FUNCTION_INFO:
    IoctlInfo(prp->ioctl.pvData);
    break;
  case IOCTL_FUNCTION_IORW:
    prp->usStatus |= IoctlIoRW(prp->ioctl.pvData);
    break;
  default:
    prp->usStatus |= RPERR | RPERR_BADCOMMAND;
    return;
  }
}

#pragma aux StrategyHandler parm [es bx];
void far StrategyHandler(PREQPACKET prp)
{
  prp->usStatus = RPDONE;

  switch (prp->bCommand) {
  case STRATEGY_INIT:
    StrategyInit(prp);
    break;
  case STRATEGY_OPEN:
     break;
  case STRATEGY_CLOSE:
     break;
  case STRATEGY_GENIOCTL:
    StrategyIoctl(prp);
    break;
  case STRATEGY_DEINSTALL:
    break;
  case STRATEGY_INITCOMPLETE:
    Drv16InitComplete();
    break;
  case STRATEGY_SAVERESTORE:
    break;
  default:
    prp->usStatus = RPDONE | RPERR | RPERR_GENERAL;
  }
}

#pragma code_seg ("_inittext");
#pragma data_seg ("_initdata","endds");

static DRIVERSTRUCT drs =
{
  (PSZ)DFILE,
  (PSZ)"Pci Ac5 I/O Driver",
  (PSZ)DVENDOR,
  CMVERSION_MAJOR,
  CMVERSION_MINOR,
  {
    (DDATE / 10000),
    (DDATE % 10000) / 100,
    (DDATE % 100)
  },
  DRF_STATIC,
  DRT_UNDEFINED,
  DRS_UNDEFINED,
  NULL
};

static ADJUNCT adj =
{
  NULL, sizeof(ADJUNCT), ADJ_ADAPTER_NUMBER
};

static ADAPTERSTRUCT ads =
{
  (PSZ)"PCIAC5_# OPTO22 IO adapter",
  AS_NO16MB_ADDRESS_LIMIT,
  AS_BASE_COMM,
  AS_SUB_OTHER,
  AS_INTF_GENERIC,
  AS_HOSTBUS_PCI,
  AS_BUSWIDTH_32BIT,
  &adj,
  0
};

char cDevName[9] = "PCIAC5$";
int iVerbose;

int ParseCmdParms(PSZ pszCmdLine) {
  int iStatus;

  if (!pszCmdLine) return 0;

  iStatus = 0;
  while (*pszCmdLine)
  {
    if (*pszCmdLine++ != '/') continue; /* Ignore anything that doesn't start with '/' */
    /* pszCmdLine now points to first char of argument */

    if (ArgCmp(pszCmdLine, "V"))
    {
      pszCmdLine += 1;
      iVerbose = 1;
      continue;
    }
    cprintf("Unrecognized switch: %Fs\n", pszCmdLine-1);
    iStatus = 1; /* unrecognized argument */
  }

  return(iStatus);
}

void StrategyInit(PREQPACKET prp)
{
  int rc;
  short Success;
  static PCI_DEVICEINFO PciDevInfo;

  UtSetDriverName(cDevName);
  ParseCmdParms(prp->init_in.szArgs); /* Process command line parameters. */

  Success = 0;

  do {
    int iAdapterNumber;
    USHORT PciBusDevFunc;

    if (Drv16Init(prp)) break; /* fail */

    /* Create the resource manager entry for this driver */
    if (Rm1CreateDriver(&drs)) break; /* fail if we can't. */

    /* Find an adapter we can control. Instead of controlling multiple
     * adapters in one driver, we only support one adapter per driver
     * and require multiple instances of the driver, one for each adapter.
     * We support up to 10 instances.
     */
    rc = 1; /* start with fail status */
    for (iAdapterNumber=0; iAdapterNumber<10; iAdapterNumber++)
    {
      if (iAdapterNumber)
      {
        UtModifyName(cDevName, iAdapterNumber, iAdapterNumber>1);
        UtSetDriverName(cDevName);
      }

      /* look for this specific vendor and device ID */
      PciBusDevFunc = PciFindDevice(0x148a, 0xac05, iAdapterNumber);
      if (PciBusDevFunc == 0xffff) break; /* can't find a device. Stop looking. */

      if (PciGetDeviceInfo(PciBusDevFunc, &PciDevInfo)) break; /* Stop if this fails */

      if (PciDevInfo.bars[0].io != 0) break; /* Stop if not memory address */

      /* Try to acquire the resources for this card. If acquiring a resourcd fails,
       * that means that another driver has already claimed that resource.
       */
      do {
        rc = Rm1AddMem(PciDevInfo.bars[0].start, PciDevInfo.bars[0].size);
        if (rc) break;

        rc = Rm1AddIrq(PciDevInfo.irq, PciDevInfo.ipin, 0);
        if (rc) break;

        adj.Adapter_Number = iAdapterNumber;

        rc = Rm1CreateAdapter(&ads);
      } while (0);

      if (!rc) break; /* We found an adapter */

      Rm1Destroy(0); /* Undo all the resources we might have added */
    }
    if (rc) break; /* couldn't find an adapter */

    if (iVerbose) cprintf("PCIAC5 found card at %lx irq %d\n", PciDevInfo.bars[0].start, PciDevInfo.irq);
    /* We found an adapter and allocated its resources. Now map the memory.
     * Only map 1 page even though the device reports a bigger size.
     */
    BaseAdr = MapPhysToVirt(PciDevInfo.bars[0].start, 0x1000);
    if (!BaseAdr) break;

    Success = 1;
  } while (0);


  if (Success)
  {
    prp->usStatus = RPDONE;
    prp->init_out.usCodeEnd = (USHORT) _TextEnd;     // set end of code
    prp->init_out.usDataEnd = (USHORT) &_DataEnd;    //  and data pointers
    if (iVerbose) cprintf("PCIAC5 driver %s loaded successfully.\n", cDevName);
  }
  else
  {
    /* We failed to install. Undo everything we did. */
    Rm1Destroy(1);
    prp->usStatus = RPDONE | RPERR;
    prp->init_out.usCodeEnd = 0;     // set end of code
    prp->init_out.usDataEnd = 0;    //  and data pointers
    if (iVerbose) cprintf("PCIAC5 initialization failed. %s not loaded.\n", cDevName);
  }
}
