/* -------------------------------------------------------------------------- */
/* The jSyncManager Project -- Source File.                                   */
/* Copyright (c) 1998 - 2003 Brad BARCLAY <bbarclay@jsyncmanager.org>         */
/* -------------------------------------------------------------------------- */
/* OSI Certified Open Source Software                                         */
/* -------------------------------------------------------------------------- */
/*                                                                            */
/* This library is free software; you can redistribute it and/or modify it    */
/* under the terms of the GNU Lesser General Public License as published      */
/* by the Free Software Foundation; either version 2.1 of the License, or     */
/* (at your option) any later version.                                        */
/*                                                                            */
/* This library is distributed in the hope that it will be useful, but        */
/* WITHOUT ANY WARRANTY; without even the implied warranty of                 */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU          */
/* Lesser General Public License for more details.                            */
/*                                                                            */
/* You should have received a copy of the GNU Lesser General Public           */
/* License along with this library; if not, write to the Free Software        */
/* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA    */
/*                                                                            */
/* -------------------------------------------------------------------------- */
/* $Id: JSerial.c,v 1.1 2003/10/31 05:03:53 yaztromo Exp $                                                                       */
/* -------------------------------------------------------------------------- */


/* The jSerial SerialPeer native implementation for OS/2.
 * @author Brad BARCLAY &lt;bbarclay@jsyncmanager.org&gt;
 * <br>Last modified by: $Author: yaztromo $ on $Date: 2003/10/31 05:03:53 $.
 * @version $Revision: 1.1 $
 */

#define INCL_DOSFILEMGR
#define INCL_DOSDEVIOCTL
#define INCL_DOSDEVICES
#include <os2.h>                                   /* Required for accessing OS/2 API functions */
#include <jni.h>                                   /* Required for Java Interfacing             */
#include <bsedev.h>                                /* Required for accessing OS/2 device codes  */
#include <string.h>                                /* Required for string manipulation          */
#include "org_jSyncManager_JSerial_SerialPeer.h"   /* The generated Java Native Interface header*/

/* ========================================================================== */

/* Global variable definitions                                                */

   HFILE hf=0;                                  /* OS/2 file handle           */
   jint buffer[1]={0};                          /* Input buffer               */
   ULONG numbytes=0;                            /* Used to store the number of*/
                                                /* bytes in the buffer        */

/* -------------------------------------------------------------------------- */

/*
 * Class:     org_jSyncManager_JSerial_SerialPeer
 * Method:    grabPort
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_org_jSyncManager_JSerial_SerialPeer_grabPort
   (JNIEnv *env, jobject obj) {
   /* This function will attempt to gain control of the given serial port.    */

   /* Variable definitions.                                                   */

   jclass cls=(*env)->GetObjectClass(env, obj); /* The Java class handle      */
   jfieldID fid;                                /* The Java Field ID handle   */
   jbyte portnum;                               /* The Java byte storage      */
   APIRET rc;                                   /* OS/2 API call result       */
   char portname[5]="COMx";                     /* Used to store the dev name */
   ULONG action;                                /* Action performed by DosOpen*/
   jclass excp;                                 /* Used to hold an exception  */
   char err_msg[32]="unable to grab serial port "; /* Used to send error msgs.*/
   

   /* Start by getting the port number from the SerialPeer class.             */

   fid = (*env)->GetFieldID(env, cls, "port", "B");   /* Gets the FieldID     */
   portnum = (*env)->GetByteField(env, obj, fid);     /* Grab the byte        */

   /* Next, we'll format the port number as a standard OS/2 port ID string    */

   portname[3]=portnum+0x31;  /* NOTE:  portnum=0 corresponds to COM1         */

   /* Next we'll attempt to open the port                                     */

   rc = DosOpen(portname, &hf, &action, 0, FILE_NORMAL, FILE_OPEN, 
                OPEN_ACCESS_READWRITE | OPEN_SHARE_DENYNONE, (PEAOP2)NULL);

   if (rc!=0) {
      /* If the result code is anything but zero, and error occured           */
      excp=(*env)->FindClass(env, "org/jSyncManager/JSerial/SerialGrabException");
      if (excp==0) {
         /* In case the Java Environment can't find the custom exception      */
         excp=(*env)->FindClass(env, "java/lang/NullPointerException");
      } /* endif */
      strcat(err_msg, portname);
      (*env)->ThrowNew(env, excp,  err_msg);
   } /* endif */

   /* The device should now be open!                                          */
   return;
}

/* -------------------------------------------------------------------------- */

/*
 * Class:     org_jSyncManager_JSerial_SerialPeer
 * Method:    setSpeed
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_org_jSyncManager_JSerial_SerialPeer_setSpeed
  (JNIEnv *env, jobject obj, jint speed) {
   /* This function will set the speed of the modem to the value set in the   */
   /* speed variable.                                                         */

   jclass excp;                           /* Used for Java exception throwing */
   char err_msg[]="Unable to initialize serial port to specified speed";
                                          /* Used as an error message         */
   APIRET rc;                             /* Used for API return codes        */

   struct {
      ULONG bitrate;
      UCHAR fraction;                     /* No need to use fractions         */
   } speedParam;                          /* Used to set the speed parameter  */

   ULONG paramLen=sizeof(speedParam);     /* Used for the paramater length    */


   if (speed<0) {
      /* Throw an exception if the speed is a negative number                 */
      excp=(*env)->FindClass(env, "org/jSyncManager/JSerial/SerialSpeedException");
      if (excp==0) {
         /* In case the Java Environment can't find the custom exception      */
         excp=(*env)->FindClass(env, "java/lang/NullPointerException");
      } /* endif */
      (*env)->ThrowNew(env, excp,  err_msg);
   } /* endif */

   /* Next we'll try to set the baud rate                                     */

   speedParam.bitrate=speed;              /* I Hope the conversion works...   */
   speedParam.fraction=0;

   rc=DosDevIOCtl(hf, IOCTL_ASYNC, ASYNC_EXTSETBAUDRATE, (PULONG)&speedParam,
                  sizeof(speedParam), &paramLen, NULL, 0, NULL);

   if (rc!=0) {
      /* Throw an exception if the requested rate is invalid or unsupported   */
      excp=(*env)->FindClass(env, "org/jSyncManager/JSerial/SerialSpeedException");
      if (excp==0) {
         /* In case the Java Environment can't find the custom exception      */
         excp=(*env)->FindClass(env, "java/lang/NullPointerException");
      } /* endif */
      (*env)->ThrowNew(env, excp,  err_msg);

   } /* endif */

   /* Speed should now be set!                                                */
   return;
}

/* -------------------------------------------------------------------------- */

/*
 * Class:     org_jSyncManager_JSerial_SerialPeer
 * Method:    setDataBits
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_org_jSyncManager_JSerial_SerialPeer_setDataBits
  (JNIEnv *env, jobject obj, jint bits) {
   /* This function will set the number of data bits per transmission byte    */

   struct {
      UCHAR databits;
      UCHAR parity;
      UCHAR stopbits;
   } params;

   ULONG size2;

   LINECONTROL current;          /* Used to read the current settings         */
   ULONG size1=sizeof(current);
   APIRET rc;                    /* Used to store API return result codes     */

   jclass excp;                           /* Used for Java exception throwing */
   char err_msg[]="Unable to initialize serial port to specified bit rate";

   /* First we should read the current settings for the given port            */

   rc=DosDevIOCtl(hf, IOCTL_ASYNC, ASYNC_GETLINECTRL, NULL, 0, NULL, (PVOID)&current, 
                  sizeof(current), &size1);

   if (rc!=0) {
      /* An error has occured - throw a SerialDataBitException                */
      excp=(*env)->FindClass(env, "org/jSyncManager/JSerial/SerialDataBitException");
      if (excp==0) {
         /* In case the Java Environment can't find the custom exception      */
         excp=(*env)->FindClass(env, "java/lang/NullPointerException");
      } /* endif */
      (*env)->ThrowNew(env, excp,  "Unable to read existing settings.");
      return;
   } /* endif */

   /* Otherwise, if everything is okay, we'll set the bitrate                 */

   params.parity=current.bParity;
   params.stopbits=current.bStopBits;
   params.databits=(UCHAR)bits;

   rc=DosDevIOCtl(hf, IOCTL_ASYNC, ASYNC_SETLINECTRL, (PULONG)&params, 
                  sizeof(params), &size2, NULL, 0, NULL);

   if (rc!=0) {
      /* If the bitrate is incorrect, throw an exception                      */
      excp=(*env)->FindClass(env, "org/jSyncManager/JSerial/SerialDataBitException");
      if (excp==0) {
         /* In case the Java Environment can't find the custom exception      */
         excp=(*env)->FindClass(env, "java/lang/NullPointerException");
      } /* endif */
      (*env)->ThrowNew(env, excp,  err_msg);
   } /* endif */

   /* Otherwise, we've been successful, and will return to the calling class  */
   return;

}

/* -------------------------------------------------------------------------- */

/*
 * Class:     org_jSyncManager_JSerial_SerialPeer
 * Method:    setParity
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_org_jSyncManager_JSerial_SerialPeer_setParity
  (JNIEnv *env, jobject obj, jint parity) {
   /* This function will set parity type used                                 */

   struct {
      UCHAR databits;
      UCHAR parity;
      UCHAR stopbits;
   } params;

   ULONG size2=sizeof(params);

   LINECONTROL current;          /* Used to read the current settings         */
   ULONG size1=sizeof(current);
   APIRET rc;                    /* Used to store API return result codes     */

   jclass excp;                           /* Used for Java exception throwing */
   char err_msg[]="Unable to initialize serial port to specified parity";

   /* First we should read the current settings for the given port            */

   rc=DosDevIOCtl(hf, IOCTL_ASYNC, ASYNC_GETLINECTRL, NULL, 0, NULL, (PVOID)&current, 
                  sizeof(current), &size1);

   if (rc!=0) {
      /* An error has occured - throw a SerialParityException                 */
      excp=(*env)->FindClass(env, "org/jSyncManager/JSerial/SerialParityException");
      if (excp==0) {
         /* In case the Java Environment can't find the custom exception      */
         excp=(*env)->FindClass(env, "java/lang/NullPointerException");
      } /* endif */
      (*env)->ThrowNew(env, excp,  "Error reading old parity value");
      return;
   } /* endif */

   /* Otherwise, if everything is okay, we'll set the parity                  */

   params.parity=(UCHAR)parity;
   params.stopbits=current.bStopBits;
   params.databits=current.bDataBits;

   rc=DosDevIOCtl(hf, IOCTL_ASYNC, ASYNC_SETLINECTRL, (PULONG)&params, 
                  sizeof(params), &size2, NULL, 0, NULL);

   if (rc!=0) {
      /* If the parity rate is incorrect, throw an exception                  */
      excp=(*env)->FindClass(env, "org/jSyncManager/JSerial/SerialParityException");
      if (excp==0) {
         /* In case the Java Environment can't find the custom exception      */
         excp=(*env)->FindClass(env, "java/lang/NullPointerException");
      } /* endif */
      (*env)->ThrowNew(env, excp,  err_msg);
   } /* endif */

   /* Otherwise, we've been successful, and will return to the calling class  */
   return;

}

/* -------------------------------------------------------------------------- */

/*
 * Class:     org_jSyncManager_JSerial_SerialPeer
 * Method:    setStopBits
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_org_jSyncManager_JSerial_SerialPeer_setStopBits
  (JNIEnv *env, jobject obj, jint stopbits) {

   /* This function will set the number of stop bits per transmission byte    */

   struct {
      UCHAR databits;
      UCHAR parity;
      UCHAR stopbits;
   } params;

   ULONG size2=sizeof(params);

   LINECONTROL current;          /* Used to read the current settings         */
   ULONG size1=sizeof(current);
   APIRET rc;                    /* Used to store API return result codes     */

   jclass excp;                           /* Used for Java exception throwing */
   char err_msg[]="Unable to initialize serial port to specified ratio of stop bits";

   /* First we should read the current settings for the given port            */

   rc=DosDevIOCtl(hf, IOCTL_ASYNC, ASYNC_GETLINECTRL, NULL, 0, NULL, (PVOID)&current, 
                  sizeof(current), &size1);

   if (rc!=0) {
      /* An error has occured - throw a SerialStopBitException                */
      excp=(*env)->FindClass(env, "org/jSyncManager/JSerial/SerialStopBitException");
      if (excp==0) {
         /* In case the Java Environment can't find the custom exception      */
         excp=(*env)->FindClass(env, "java/lang/NullPointerException");
      } /* endif */
      (*env)->ThrowNew(env, excp,  "Unable to read current settings");
      return;
   } /* endif */

   /* Otherwise, if everything is okay, we'll set the number of stop bits     */

   params.parity=current.bParity;
   params.stopbits=(UCHAR)stopbits;
   params.databits=current.bDataBits;

   rc=DosDevIOCtl(hf, IOCTL_ASYNC, ASYNC_SETLINECTRL, (PULONG)&params, 
                  sizeof(params), &size2, NULL, 0, NULL);

   if (rc!=0) {
      /* If the stopbit value is incorrect, throw an exception                */
      excp=(*env)->FindClass(env, "org/jSyncManager/JSerial/SerialStopBitException");
      if (excp==0) {
         /* In case the Java Environment can't find the custom exception      */
         excp=(*env)->FindClass(env, "java/lang/NullPointerException");
      } /* endif */
      (*env)->ThrowNew(env, excp,  err_msg);
   } /* endif */

   /* Otherwise, we've been successful, and will return to the calling class  */
   return;

}

/* -------------------------------------------------------------------------- */

/*
 * Class:     org_jSyncManager_JSerial_SerialPeer
 * Method:    getChar
 * Signature: ()I
 */
JNIEXPORT jint JNICALL Java_org_jSyncManager_JSerial_SerialPeer_getChar
  (JNIEnv *env, jobject obj) {
   /* This function will read a character from the COM buffer, and pass it    */
   /* back to the calling method.                                             */

   APIRET rc;                 /* DosRead call result code storage             */
   jclass excp;               /* Used to throw exceptions                     */
   
   rc=DosRead(hf, (PVOID)buffer, (ULONG)1, (PULONG)&numbytes);
                                                /* Reads the next buffer      */
   
   if (rc==232) {
      /* There were no bytes to be read                                       */
      return -1;
   } /* endif */

   if (rc!=0) {
      /* Any other error and we throw an exception                            */
      excp=(*env)->FindClass(env, "org/jSyncManager/JSerial/SerialStopBitException");
      if (excp==0) {
         /* In case the Java Environment can't find the custom exception      */
         excp=(*env)->FindClass(env, "java/lang/NullPointerException");
      } /* endif */
      (*env)->ThrowNew(env, excp,  "Unable to get character");
   } /* endif */

   return (jint)buffer[0];    /* Return the value read                        */
} /* end-function */

/* -------------------------------------------------------------------------- */

/*
 * Class:     org_jSyncManager_JSerial_SerialPeer
 * Method:    putChar
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_org_jSyncManager_JSerial_SerialPeer_putChar
  (JNIEnv *env, jobject obj, jint out_char) {
   /* This function puts a character into the output buffer                   */

   APIRET rc;                 /* Used to store the result code from DosWrite  */
   jclass excp;               /* Used for throwing Java exceptions            */
   ULONG bytes_written=0;     /* Used to store the number of bytes written    */
   UCHAR test_send[1];

   test_send[0]=(UCHAR)(out_char);

   rc=DosWrite(hf, (PVOID)test_send, sizeof(test_send), &bytes_written);

   if (rc!=0) {
      /* Any other error and we throw an exception                         */
      excp=(*env)->FindClass(env, "java/io/IOException");
      if (excp==0) {
         /* In case the Java Environment can't find the custom exception   */
         excp=(*env)->FindClass(env, "java/lang/NullPointerException");
      } /* endif */
      (*env)->ThrowNew(env, excp,  "Unable to put char into buffer.");
   } /* endif */
}

/* -------------------------------------------------------------------------- */

/*
 * Class:     org_jSyncManager_JSerial_SerialPeer
 * Method:    getPlatformName
 * Signature: ()Ljava/lang/String;
 */
JNIEXPORT jstring JNICALL Java_org_jSyncManager_JSerial_SerialPeer_getPlatformName
  (JNIEnv * env, jclass obj) {
  /* This function returns a Java String containing the platform's name       */

   return (*env)->NewStringUTF(env, "Serial Library for OS/2 WARP, written by Brad Barclay.");
}
/* -------------------------------------------------------------------------- */

/*
 * Class:     org_jSyncManager_JSerial_SerialPeer
 * Method:    getVersion
 * Signature: ()Ljava/lang/String;
 */
JNIEXPORT jstring JNICALL Java_org_jSyncManager_JSerial_SerialPeer_getVersion
  (JNIEnv *env, jclass obj) {

   return (*env)->NewStringUTF(env, "Version 0.1 Beta");
}

/* -------------------------------------------------------------------------- */

/*
 * Class:     org_jSyncManager_JSerial_SerialPeer
 * Method:    releasePort
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_org_jSyncManager_JSerial_SerialPeer_releasePort
  (JNIEnv *env, jobject obj) {

   APIRET rc;              /* Used to store returned result codes             */

   rc=DosClose(hf);
}

/* -------------------------------------------------------------------------- */

/*
 * Class:     org_jSyncManager_JSerial_SerialPeer
 * Method:    checkPort
 * Signature: (B)Z
 */
JNIEXPORT jboolean JNICALL Java_org_jSyncManager_JSerial_SerialPeer_checkPort
  (JNIEnv *env, jobject obj, jbyte port_num) {
   /* This function determines wether or not a given port is valid            */
   /* For now, just report that the given port exists                         */
   return JNI_TRUE;

}

/* -------------------------------------------------------------------------- */

/*
 * Class:     org_jSyncManager_JSerial_SerialPeer
 * Method:    setRTSCTS
 * Signature: (Z)V
 */
JNIEXPORT void JNICALL Java_org_jSyncManager_JSerial_SerialPeer_setRTSCTS
  (JNIEnv *env, jobject obj, jboolean state) {

   /* This method will enable RTS/CTS hardware flow control.                  */

   DCBINFO devinfo;           /* Used to store the state of the serial device */
   APIRET rc;                 /* Used to store the result code from the call  */
   ULONG size=sizeof(devinfo);/* Used to store the size of the parameter      */
   jclass excp;               /* Used for throwing exceptions.                */

   /* First we'll read the current device state                               */

   rc=DosDevIOCtl(hf, IOCTL_ASYNC, ASYNC_GETDCBINFO, NULL, 
                  0, NULL, (PVOID)&devinfo, sizeof(devinfo), &size);

   if (rc!=0) {
      /* Any other error and we throw an exception                         */
      excp=(*env)->FindClass(env, "org.jSyncManager.JSerial.SerialHandshakingException");
      if (excp==0) {
         /* In case the Java Environment can't find the custom exception   */
         excp=(*env)->FindClass(env, "java/lang/NullPointerException");
      } /* endif */
      (*env)->ThrowNew(env, excp,  "Unable to get current device control line settings.");
   } /* endif */

   /* Next we'll modify the existing device settings...                    */

   if (state==JNI_TRUE) {
      devinfo.fbFlowReplace&=(255-64);     /* Sets bit 6 to OFF (RTS)             */
      devinfo.fbFlowReplace|=128;          /* Sets bit 7 to ON  (RTS)             */
      devinfo.fbCtlHndShake|=8;            /* Sets bit 3 to ON  (CTS)             */
   } else {
      devinfo.fbFlowReplace&=63;           /* Set bits 6 and 7 to OFF (RTS)       */
      devinfo.fbCtlHndShake&=247;          /* Sets bit 3 to OFF (CTS)             */
   } /* endif */

   /* Now we'll attempt to set the new control line settings.              */

   rc=DosDevIOCtl(hf, IOCTL_ASYNC, ASYNC_SETDCBINFO, (PVOID)&devinfo, sizeof(devinfo),
                  &size, NULL, 0, NULL);

   if (rc!=0) {
      /* Any other error and we throw an exception                         */
      excp=(*env)->FindClass(env, "org.jSyncManager.JSerial.SerialHandshakingException");
      if (excp==0) {
         /* In case the Java Environment can't find the custom exception   */
         excp=(*env)->FindClass(env, "java/lang/NullPointerException");
      } /* endif */
      (*env)->ThrowNew(env, excp,  "Unable to set new device control line settings.");
   } /* endif */

}

/* -------------------------------------------------------------------------- */

/*
 * Class:     org_jSyncManager_JSerial_SerialPeer
 * Method:    setDSRDTR
 * Signature: (Z)V
 */
JNIEXPORT void JNICALL Java_org_jSyncManager_JSerial_SerialPeer_setDSRDTR
  (JNIEnv *env, jobject obj, jboolean state) {

   /* This method will enable DSR/DTR hardware flow control.                  */

   DCBINFO devinfo;           /* Used to store the state of the serial device */
   APIRET rc;                 /* Used to store the result code from the call  */
   ULONG size=sizeof(devinfo);/* Used to store the size of the parameter      */
   jclass excp;               /* Used for throwing exceptions.                */

   /* First we'll read the current device state                               */

   rc=DosDevIOCtl(hf, IOCTL_ASYNC, ASYNC_GETDCBINFO, NULL, 
                  0, NULL, (PVOID)&devinfo, sizeof(devinfo), &size);

   if (rc!=0) {
      /* Any other error and we throw an exception                         */
      excp=(*env)->FindClass(env, "org.jSyncManager.JSerial.SerialHandshakingException");
      if (excp==0) {
         /* In case the Java Environment can't find the custom exception   */
         excp=(*env)->FindClass(env, "java/lang/NullPointerException");
      } /* endif */
      (*env)->ThrowNew(env, excp,  "Unable to get current device control line settings.");
   } /* endif */

   /* Next we'll modify the existing device settings...                    */

   if (state==JNI_TRUE) {
      devinfo.fbCtlHndShake&=(255-1);      /* Sets bit 0 to OFF (DTR)             */
      devinfo.fbCtlHndShake|=2;            /* Sets bit 1 to ON  (DTR)             */
      devinfo.fbCtlHndShake|=16;           /* Sets bit 4 to ON  (DSR)             */
   } else {
      devinfo.fbCtlHndShake&=252;          /* Set bits 0 and 1 to OFF (RTS)       */
      devinfo.fbCtlHndShake&=239;          /* Sets bit 4 to OFF (CTS)             */
   } /* endif */

   /* Now we'll attempt to set the new control line settings.              */

   rc=DosDevIOCtl(hf, IOCTL_ASYNC, ASYNC_SETDCBINFO, (PVOID)&devinfo, sizeof(devinfo),
                  &size, NULL, 0, NULL);

   if (rc!=0) {
      /* Any other error and we throw an exception                         */
      excp=(*env)->FindClass(env, "org.jSyncManager.JSerial.SerialHandshakingException");
      if (excp==0) {
         /* In case the Java Environment can't find the custom exception   */
         excp=(*env)->FindClass(env, "java/lang/NullPointerException");
      } /* endif */
      (*env)->ThrowNew(env, excp,  "Unable to set new device control line settings.");
   } /* endif */

}

/* -------------------------------------------------------------------------- */

/*
 * Class:     org_jSyncManager_JSerial_SerialPeer
 * Method:    setXONXOFF
 * Signature: (Z)V
 */
JNIEXPORT void JNICALL Java_org_jSyncManager_JSerial_SerialPeer_setXONXOFF
  (JNIEnv *env, jobject obj, jboolean state) {

   /* This method will enable XON/XOFF software flow control.                 */

   DCBINFO devinfo;           /* Used to store the state of the serial device */
   APIRET rc;                 /* Used to store the result code from the call  */
   ULONG size=sizeof(devinfo);/* Used to store the size of the parameter      */
   jclass excp;               /* Used for throwing exceptions.                */

   /* First we'll read the current device state                               */

   rc=DosDevIOCtl(hf, IOCTL_ASYNC, ASYNC_GETDCBINFO, NULL, 
                  0, NULL, (PVOID)&devinfo, sizeof(devinfo), &size);

   if (rc!=0) {
      /* Any other error and we throw an exception                         */
      excp=(*env)->FindClass(env, "org.jSyncManager.JSerial.SerialHandshakingException");
      if (excp==0) {
         /* In case the Java Environment can't find the custom exception   */
         excp=(*env)->FindClass(env, "java/lang/NullPointerException");
      } /* endif */
      (*env)->ThrowNew(env, excp,  "Unable to get current device control line settings.");
   } /* endif */

   /* Next we'll modify the existing device settings...                    */

   if (state==JNI_TRUE) {
      devinfo.fbFlowReplace|=3;            /* Sets bit 1 and 0 to  ON (XON/XOFF)  */
   } else {
      devinfo.fbFlowReplace&=252;          /* Set bits 0 and 1 to OFF (XON/XOFF)  */
   } /* endif */

   /* Now we'll attempt to set the new control line settings.              */

   rc=DosDevIOCtl(hf, IOCTL_ASYNC, ASYNC_SETDCBINFO, (PVOID)&devinfo, sizeof(devinfo),
                  &size, NULL, 0, NULL);

   if (rc!=0) {
      /* Any other error and we throw an exception                         */
      excp=(*env)->FindClass(env, "org.jSyncManager.JSerial.SerialHandshakingException");
      if (excp==0) {
         /* In case the Java Environment can't find the custom exception   */
         excp=(*env)->FindClass(env, "java/lang/NullPointerException");
      } /* endif */
      (*env)->ThrowNew(env, excp,  "Unable to set new device control line settings.");
   } /* endif */

}

/* -------------------------------------------------------------------------- */

/*
 * Class:     org_jSyncManager_JSerial_SerialPeer
 * Method:    startBreak
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_org_jSyncManager_JSerial_SerialPeer_startBreak
  (JNIEnv *env, jobject obj) {

   /* Sends a BREAK character.                                                */

   APIRET rc;              /* Used for storing return codes.                  */
   USHORT err;             /* Used to store COM error results.                */
   ULONG size=sizeof(err); /* Used to store the size of the data packet       */

   rc=DosDevIOCtl(hf, IOCTL_ASYNC, ASYNC_SETBREAKON, NULL, 0, NULL,
                  (PVOID)err, sizeof(err), &size);

}

/* -------------------------------------------------------------------------- */

/*
 * Class:     org_jSyncManager_JSerial_SerialPeer
 * Method:    stopBreak
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_org_jSyncManager_JSerial_SerialPeer_stopBreak
  (JNIEnv *env, jobject obj) {

   /* Stops sending BREAK characters.                                         */

   APIRET rc;              /* Used for storing return codes.                  */
   USHORT err;             /* Used to store COM error results.                */
   ULONG size=sizeof(err); /* Used to store the size of the data packet       */

   rc=DosDevIOCtl(hf, IOCTL_ASYNC, ASYNC_SETBREAKOFF, NULL, 0, NULL,
                  (PVOID)err, sizeof(err), &size);

}

/* -------------------------------------------------------------------------- */

/*
 * Class:     org_jSyncManager_JSerial_SerialPeer
 * Method:    getRTSCTSState
 * Signature: ()Z
 */
JNIEXPORT jboolean JNICALL Java_org_jSyncManager_JSerial_SerialPeer_getRTSCTSState
  (JNIEnv *env, jobject obj) {

   /* This method will determine the RTS/CTS state                            */

   DCBINFO devinfo;           /* Used to store the state of the serial device */
   APIRET rc;                 /* Used to store the result code from the call  */
   ULONG size=sizeof(devinfo);/* Used to store the size of the parameter      */
   jclass excp;               /* Used for throwing exceptions.                */

   /* First we'll read the current device state                               */

   rc=DosDevIOCtl(hf, IOCTL_ASYNC, ASYNC_GETDCBINFO, NULL, 
                  0, NULL, (PVOID)&devinfo, sizeof(devinfo), &size);

   if (rc!=0) {
      /* Any other error and we throw an exception                            */
      excp=(*env)->FindClass(env, "org.jSyncManager.JSerial.SerialHandshakingException");
      if (excp==0) {
         /* In case the Java Environment can't find the custom exception      */
         excp=(*env)->FindClass(env, "java/lang/NullPointerException");
      } /* endif */
      (*env)->ThrowNew(env, excp,  "Unable to get current device control line settings.");
   } /* endif */

   /* Next we'll query device settings...                                     */

   if ((devinfo.fbFlowReplace&=128)==128 & (devinfo.fbFlowReplace&=64)==0 & (devinfo.fbCtlHndShake&=8)==8) {
      return JNI_TRUE;
   } else {
      return JNI_FALSE;
   } /* endif */
}

/* -------------------------------------------------------------------------- */

/*
 * Class:     org_jSyncManager_JSerial_SerialPeer
 * Method:    getDSRDTRState
 * Signature: ()Z
 */
JNIEXPORT jboolean JNICALL Java_org_jSyncManager_JSerial_SerialPeer_getDSRDTRState
  (JNIEnv *env, jobject obj) {

   /* This method will determine the DSR/DTR state                            */

   DCBINFO devinfo;           /* Used to store the state of the serial device */
   APIRET rc;                 /* Used to store the result code from the call  */
   ULONG size=sizeof(devinfo);/* Used to store the size of the parameter      */
   jclass excp;               /* Used for throwing exceptions.                */

   /* First we'll read the current device state                               */

   rc=DosDevIOCtl(hf, IOCTL_ASYNC, ASYNC_GETDCBINFO, NULL, 
                  0, NULL, (PVOID)&devinfo, sizeof(devinfo), &size);

   if (rc!=0) {
      /* Any other error and we throw an exception                            */
      excp=(*env)->FindClass(env, "org.jSyncManager.JSerial.SerialHandshakingException");
      if (excp==0) {
         /* In case the Java Environment can't find the custom exception      */
         excp=(*env)->FindClass(env, "java/lang/NullPointerException");
      } /* endif */
      (*env)->ThrowNew(env, excp,  "Unable to get current device control line settings.");
   } /* endif */

   /* Next we'll query device settings...                                     */

   if ((devinfo.fbCtlHndShake&=1)==0 & (devinfo.fbCtlHndShake&=2)==2 & (devinfo.fbCtlHndShake&=16)==16) {
      return JNI_TRUE;
   } else {
      return JNI_FALSE;
   } /* endif */
}

/* -------------------------------------------------------------------------- */

/*
 * Class:     org_jSyncManager_JSerial_SerialPeer
 * Method:    getXONXOFFState
 * Signature: ()Z
 */
JNIEXPORT jboolean JNICALL Java_org_jSyncManager_JSerial_SerialPeer_getXONXOFFState
  (JNIEnv *env, jobject obj) {

   /* This method will determine the XON/XOFF state                           */

   DCBINFO devinfo;           /* Used to store the state of the serial device */
   APIRET rc;                 /* Used to store the result code from the call  */
   ULONG size=sizeof(devinfo);/* Used to store the size of the parameter      */
   jclass excp;               /* Used for throwing exceptions.                */

   /* First we'll read the current device state                               */

   rc=DosDevIOCtl(hf, IOCTL_ASYNC, ASYNC_GETDCBINFO, NULL, 
                  0, NULL, (PVOID)&devinfo, sizeof(devinfo), &size);

   if (rc!=0) {
      /* Any other error and we throw an exception                            */
      excp=(*env)->FindClass(env, "org.jSyncManager.JSerial.SerialHandshakingException");
      if (excp==0) {
         /* In case the Java Environment can't find the custom exception      */
         excp=(*env)->FindClass(env, "java/lang/NullPointerException");
      } /* endif */
      (*env)->ThrowNew(env, excp,  "Unable to get current device control line settings.");
   } /* endif */

   /* Next we'll query device settings...                                     */

   if ((devinfo.fbFlowReplace&=3)==3) {
      return JNI_TRUE;
   } else {
      return JNI_FALSE;
   } /* endif */
}

/* ========================================================================== */

