/*******************************************************************
 *
 *  ttmemory.c                                               1.1
 *
 *    Memory management component (body).
 *
 *  Copyright 1996, 1997 by
 *  David Turner, Robert Wilhelm, and Werner Lemberg.
 *
 *  This file is part of the FreeType project, and may only be used
 *  modified and distributed under the terms of the FreeType project
 *  license, LICENSE.TXT. By continuing to use, modify or distribute
 *  this file you indicate that you have read the license and
 *  understand and accept it fully.
 *
 *  NOTE:
 *
 *    The FreeType library uses a simple and fast growing heap
 *    known as a 'Font Pool' to manage all its data. This version
 *    does not manage multiple pools (a feature that may reveal
 *    useful for a FreeType based font server to be able to manage
 *    several open fonts).
 *
 *  Differences between 1.0 and 1.1 :
 *
 *   - Added 'Mem_Allocated' and 'Mem_Free' functions.
 *
 ******************************************************************/

#ifdef DEBUG_MEMORY
#include <stdlib.h>
#endif

#include "tterror.h"
#include "ttmemory.h"

  Bool  Font_Pool_Allocated = FALSE;


#define Mark_Magic      0xBABE0007L

  static unsigned char*  Buffer;

  static long            SizeBuff;
  static long            CurBuff;


/*******************************************************************
 *
 *  Function    :  Mem_Allocated
 *
 *  Description :  Returns amount of heap memory in use.
 *
 *  Input  :  none
 *
 *  Output :  heap memory in use
 *
 ******************************************************************/

  long  Mem_Allocated()
  {
    return CurBuff;
  }


/*******************************************************************
 *
 *  Function    :  Mem_Free
 *
 *  Description :  Returns amount of free heap memory.
 *
 *  Input  :  none
 *
 *  Output :  free heap memory
 *
 ******************************************************************/

  long  Mem_Free()
  {
    return SizeBuff - CurBuff;
  }


/*******************************************************************
 *
 *  Function    :  Init_FontPool
 *
 *  Description :  Initializes font pool management.
 *
 *  Input  :  Buff      pointer to a buffer for the heap
 *            Size      size of buffer Buff
 *
 *  Output :  none.
 *
 ******************************************************************/

  void  Init_FontPool( void*  Buff, long  Size )
  {
    Buffer              = (unsigned char*)Buff;
    SizeBuff            = Size;
    CurBuff             = 0;
    Font_Pool_Allocated = TRUE;
  }


/*******************************************************************
 *
 *  Function    :  Alloc
 *
 *  Description :  Allocates memory from the heap buffer.
 *
 *  Input  :  Size      size of the memory to be allocated
 *            P         pointer to a buffer pointer
 *
 *  Output :  SUCCESS if memory could be allocated,
 *            FAILURE otherwise.
 *
 ******************************************************************/


  Bool  Alloc( long  Size, void**  P ) 
  {

#ifdef DEBUG_MEMORY

    if ( Size )
    {
      *P = malloc( Size );
      return SUCCESS;
    }
    else
    {
      *P = NULL;
      Error = TT_Err_Out_Of_Memory;
      return FAILURE;
    }

#else

    long  L;
  
    L = CurBuff + ( ( Size + 3 ) & ( -4 ) );

    if ( L >= SizeBuff )
    {
      *P = NULL;
      Error = TT_Err_Out_Of_Memory;
      return FAILURE;
    }
    else
    {
      *P      = &Buffer[CurBuff];
      CurBuff = L;
      return SUCCESS;
    }

#endif

  }


/*******************************************************************
 *
 *  Function    :  Mark
 *
 *  Description :  Gets the current heap top position and puts it
 *                 into a mark record.
 *
 *  Input  :  M         pointer to a TMarkRecord structure
 *
 *  Output :  none
 *
 ******************************************************************/


  void  Mark( PMarkRecord  M )
  {
    M->Magic = Mark_Magic;
    M->Top   = CurBuff;
  }


/*******************************************************************
 *
 *  Function    :  Release
 *
 *  Description :  Sets the heap top to a previously saved position
 *                 with Mark().
 *
 *  Input  :  M         pointer to a TMarkRecord structure
 *
 *  Output :  SUCCESS on success. FAILURE if the mark is invalid
 *            or has already been released.
 *
 ******************************************************************/

  Bool  Release( PMarkRecord  M )
  {
    if ( M->Magic == Mark_Magic )
    {
      M->Magic = 0;
      CurBuff  = M->Top;
      M->Top   = -1;
      return SUCCESS;
    } 
    else
      return FAILURE;
  }


/* End */
