(*******************************************************************
 *
 *  TTFile.Pas                                                1.1
 *
 *    File I/O Component (specification)
 *
 *  Copyright 1996 David Turner, Robert Wilhelm and Werner Lemberg
 *
 *  This file is part of the FreeType project, and may only be used
 *  modified and distributed under the terms of the FreeType project
 *  license, LICENSE.TXT. By continuing to use, modify or distribute
 *  this file you indicate that you have read the license and
 *  understand and accept it fully.
 *
 *  NOTES :
 *
 *   The 1.1 version includes a change of the API as well as the
 *   "support" of multiple streams, useful for multiple font servers.
 *
 *   Note that it is not reentrant, which means that other components
 *   should use the TT_Lock_Access and TT_Release_Access functions
 *   before and after accessing the font file directly.
 *
 *  Changes from 1.0 to 1.1 :
 *
 *  - defined the type TT_Stream for file handles
 *  - renamed ( and cleaned ) the API.
 *
 *  - caching and memory-mapped files use the same API :
 *
 *      TT_Access_Frame to notify
 *
 *  - only the interface was really rewritten. This component still
 *    only supports one opened file at a time.
 *
 ******************************************************************)

Unit TTFile;

interface

uses FreeType,
     TTTypes,
     TTError;

{
function Open_Font_File( AName : String ) : boolean;

function Read_Font_File( var ABuff; ACount : Int ) : boolean;

function Read_Short ( var S : Short ) : boolean;
function Read_Long  ( var L : Long  ) : boolean;

function Seek_Font_File( APos : LongInt ) : boolean;

function Skip_Font_File( ADist : LongInt ) : boolean;

function Read_At_Font_File( APos : Long; var ABuff; ACount : Int ) : boolean;

procedure Close_Font_File;
}

function TT_Open_File( aName : String; var aStream : TT_Stream ) : Boolean;
(* Opens a file and returns a stream handle to it *)

function TT_Lock_Access( aStream : TT_Stream ) : Boolean;
(* Acquire the file mutex *)

function TT_Release_Access( aStream : TT_Stream ) : Boolean;
(* Release the file mutex *)

function TT_Read_File( var ABuff; ACount : Int ) : boolean;
(* Read a chunk of bytes directly from the file *)

function TT_Seek_File( APos : LongInt ) : boolean;
(* Seek a new file position *)

function TT_Skip_File( ADist : LongInt ) : boolean;
(* Skip to a new file position *)

function TT_Read_At_File( APos : Long; var ABuff; ACount : Int ) : boolean;
(* Seek and read a chunk of bytes *)

function TT_Access_Frame( aSize : Int ) : Boolean;
(* Access the next aSize bytes *)

function TT_Check_And_Access_Frame( aSize : Int ) : Boolean;
(* Access the next min(aSize,file_size-file_pos) bytes *)

function TT_Forget_Frame :  Boolean;
(* Forget the previously cached frame *)

function TT_Close_File( aStream : TT_Stream ) : boolean;

function TT_File_Size : Longint;

function TT_File_Pos  : Longint;

(* The four following functions should only be used after a TT_Access_Frame *)
(* and before a TT_Forget_Frame                                             *)

(* They do not provide error handling, intentionnaly, and are much faster *)
(* moreover, they could be converted to MACROS in the C version           *)

function GET_Byte   : Byte;
function GET_Char   : ShortInt;
function GET_Short  : Short;
function GET_UShort : UShort;
function GET_Long   : Long;
function GET_ULong  : ULong;
function GET_Tag4 : ULong;

implementation

var
  font_file  : FILE;

  num_files  : Int;

  current_frame : PByte;
  frame_cursor  : Longint;
  frame_size    : LongInt;


  function  TT_File_Size : Longint;
  begin
    TT_File_Size := FileSize( font_file );
  end;

  function TT_File_Pos : Longint;
  begin
    TT_File_Pos := FilePos( font_file );
  end;

(*******************************************************************
 *
 *  Function    :  TT_Open_File
 *
 *  Description :  opens the font file and reads it into a memory
 *                 buffer ( to ease development and speed debug ).
 *
 *  Input  :  aName   pathname of the file to open
 *
 *  Output :  True on sucess.
 *
 ******************************************************************)

 function TT_Open_File( aName : String; var aStream : TT_Stream ) : boolean;
 begin
   TT_Open_File := False;

   aStream := -1;

   if num_files > 0 then exit;
   (* Currently, we only support one opened font file at a time *)

   Assign( Font_File, AName );
   {$I-}
   Reset( Font_File, 1 );
   {$I+}
   if IOResult <> 0 then
     begin
       Error := TT_ErrMsg_File_Error;
       exit;
     end;

   aStream := 0;

   TT_Open_File := True;
 end;


(*******************************************************************
 *
 *  Function    : TT_Lock_Access
 *
 *  Description : Acquire the file mutex (blocking call)
 *
 *  Input  :  None
 *
 *  Output :  True on success. False on failure
 *
 ******************************************************************)

 function TT_Lock_Access( aStream : TT_Stream ) : Boolean;
 begin
   (* only one stream supported now *)
   (* return immediately            *)

   TT_Lock_Access := True;
 end;

(*******************************************************************
 *
 *  Function    : TT_Release_Access
 *
 *  Description : Release the file mutex
 *
 *  Input  :  None
 *
 *  Output :  True on success. False on failure
 *
 ******************************************************************)

 function TT_Release_Access( aStream : TT_Stream ) : Boolean;
 begin
   (* nothing to do currently *)

   TT_Release_Access := True;
 end;

(*******************************************************************
 *
 *  Function    : TT_Close_File
 *
 *  Description : Closes the font file and releases memory buffer
 *
 *  Input  :  None
 *
 *  Output :  True ( always )
 *
 ******************************************************************)

 function TT_Close_File( aStream : TT_Stream ) : Boolean;
 begin
   Close( Font_File );
   TT_Close_File := True;
 end;

(*******************************************************************
 *
 *  Function    : TT_Seek_File
 *
 *  Description : Seek the file cursor to a different position
 *
 *  Input  :  APos     new position on file
 *
 *  Output :  True on success. False if out of range
 *
 *  Notes  :  Does not set the error variable
 *
 ******************************************************************)

function TT_Seek_File( APos : LongInt ) : boolean;
begin
  {$I-}
  Seek( Font_File, APos );
  {$I+}
  if IOResult <> 0 then
    begin
      Error          := TT_ErrMsg_File_Error;
      TT_Seek_File := False;
      exit;
    end;

  TT_Seek_File := True;
end;

(*******************************************************************
 *
 *  Function    : TT_Skip_File
 *
 *  Description : Skip forward the file cursor
 *
 *  Input  :  ADist    number of bytes to skip
 *
 *  Output :  see Seek_Font_File
 *
 ******************************************************************)

function TT_Skip_File( ADist : LongInt ) : boolean;
begin
  TT_Skip_File := TT_Seek_File( FilePos(Font_File)+ADist );
end;

(*******************************************************************
 *
 *  Function    : TT_Read_File
 *
 *  Description : Reads a chunk of the file and copy it to memory
 *
 *  Input  :  ABuff     target buffer
 *            ACount    length in bytes to read
 *
 *  Output :  True if success. False if out of range
 *
 *  Notes  :  Current version prints an error message even if the
 *            debug state isn't on.
 *
 ******************************************************************)

function TT_Read_File( var ABuff; ACount : Int ) : boolean;
begin
  TT_Read_File := False;
  {$I-}
  BlockRead( Font_File, ABuff, ACount );
  {$I+}

  if IOResult <> 0 then
    begin
      Error := TT_ErrMsg_File_Error;
      exit;
    end;

  TT_Read_File := True;
end;

(*******************************************************************
 *
 *  Function    : TT_Read_At_File
 *
 *  Description : Read file at a specified position
 *
 *  Input  :  APos     position to seek to before read
 *            ABuff    target buffer
 *            ACount   number of bytes to read
 *
 *  Output :  True on success. False if error.
 *
 *  Notes  :  prints an error message if seek failed.
 *
 ******************************************************************)

function TT_Read_At_File( APos : Long; var ABuff; ACount : Int ) : boolean;
begin
  TT_Read_At_File := False;

  if not TT_Seek_File( APos ) or
     not TT_Read_File( ABuff, ACount ) then exit;

  TT_Read_At_File := True;
end;

(*******************************************************************
 *
 *  Function    : Read_Short
 *
 *  Description : Read a 2-bytes word from the file
 *
 *  Input  :  S   target short
 *
 *  Output :  True on success. False on overflow
 *
 *  Notes  :  Use the READ_ macros instead of this function
 *
 ******************************************************************)

function Read_Short( var S : Short ) : boolean;
var
  Cs : array[0..1] of Byte;
begin
  Read_Short := False;

  if not TT_Read_File( Cs, 2 ) then exit;

  S := ( Cs[0] shl 8 ) or Cs[1];

  Read_Short := True;
end;

function Read_UShort( var U : UShort ) : boolean;
var
  Cs : array[0..1] of Byte;
begin
  Read_UShort := False;

  if not TT_Read_File( Cs, 2 ) then exit;

  U := ( Cs[0] shl 8 ) or Cs[1];

  Read_UShort := True;
end;

(*******************************************************************
 *
 *  Function    : Read_Long
 *
 *  Description : Read a 4-bytes word from the file
 *
 *  Input  : L   target long
 *
 *  Output : True on success. False if overflow
 *
 *  Notes  : Use the READ_ macros instead of this function
 *
 ******************************************************************)

function Read_Long( var L : Long ) : boolean;
var
  Cs : array[0..3] of Byte;
begin
  Read_Long := False;

  if not TT_Read_File( Cs, 4 ) then exit;

  L := ( Cs[0] shl 24 ) or ( Cs[1] shl 16 ) or ( Cs[2] shl 8 ) or Cs[3];

  Read_Long := True;
end;

function Read_ULong( var U : ULong ) : boolean;
var
  Cs : array[0..3] of Byte;
begin
  Read_ULong := False;

  if not TT_Read_File( Cs, 4 ) then exit;

  U := ( Cs[0] shl 24 ) or ( Cs[1] shl 16 ) or ( Cs[2] shl 8 ) or Cs[3];

  Read_ULong := True;
end;

(*******************************************************************
 *
 *  Function    :  TT_Access_Frame
 *
 *  Description :  Notifies the component that we're going to read
 *                 aSize bytes from the current file position.
 *                 This function should load/cache/map these bytes
 *                 so that they will be addressed by the GET_xxx
 *                 functions easily.
 *
 *  Input  :  aSize   number of bytes to access.
 *
 *  Output :  True on success. False on failure
 *
 *            The function fails is the byte range is not within the
 *            the file, or if there is not enough memory to cache
 *            the bytes properly ( which usually means that aSize is
 *            too big in both cases ).
 *
 *            It will also fail if you make two consecutive calls
 *            to TT_Access_Frame, without a TT_Forget_Frame between
 *            them.
 *
 ******************************************************************)

 function TT_Access_Frame( aSize : Int ) : Boolean;
 var
   readBytes : Longint;
 begin
   TT_Access_Frame := True;

   if current_frame <> nil then exit;
   (* We already are accessing one frame *)

   GetMem( current_frame, aSize );
   if current_frame = nil then exit;
   (* Not enough memory to load frame or empty frame *)

   if not TT_Read_File( current_frame^, aSize ) then
   begin
     FreeMem( current_frame, aSize );
     current_frame := nil;
     exit;
   end;

   frame_size   := aSize;
   frame_cursor := 0;

   TT_Access_Frame := True;
 end;

(*******************************************************************
 *
 *  Function    :  TT_Check_And_Access_Frame
 *
 *  Description :  Notifies the component that we're going to read
 *                 aSize bytes from the current file position.
 *                 This function should load/cache/map these bytes
 *                 so that they will be addressed by the GET_xxx
 *                 functions easily.
 *
 *  Input  :  aSize   number of bytes to access.
 *
 *  Output :  True on success. False on failure
 *
 *            The function fails is the byte range is not within the
 *            the file, or if there is not enough memory to cache
 *            the bytes properly ( which usually means that aSize is
 *            too big in both cases ).
 *
 *            It will also fail if you make two consecutive calls
 *            to TT_Access_Frame, without a TT_Forget_Frame between
 *            them.
 *
 *
 * NOTE :  The only difference with TT_Access_Frame is that we check
 *         that the frame is within the current file.  We otherwise
 *         truncate it..
 *
 ******************************************************************)

 function TT_Check_And_Access_Frame( aSize : Int ) : Boolean;
 var
   readBytes : Longint;
 begin
   TT_Check_And_Access_Frame := True;

   if current_frame <> nil then exit;
   (* We already are accessing one frame *)

   readBytes := TT_File_Size - TT_File_Pos;
   if aSize > readBytes then aSize := readBytes;

   GetMem( current_frame, aSize );
   if current_frame = nil then exit;
   (* Not enough memory to load frame, or empty frame *)

   if not TT_Read_File( current_frame^, aSize ) then
   begin
     FreeMem( current_frame, aSize );
     current_frame := nil;
     exit;
   end;

   frame_size   := aSize;
   frame_cursor := 0;

   TT_Check_And_Access_Frame := True;
 end;

(*******************************************************************
 *
 *  Function    :  TT_Forget_Frame
 *
 *  Description :  Releases a cached frame after reading
 *
 *  Input  :  None
 *
 *  Output :  True on success. False on failure
 *
 ******************************************************************)

 function TT_Forget_Frame : boolean;
 begin
   TT_Forget_Frame := False;

   if current_frame = nil then exit;

   FreeMem( current_frame, frame_size );

   frame_size    := 0;
   current_frame := nil;
   frame_cursor  := 0;
 end;

(*******************************************************************
 *
 *  Function    :  GET_Byte
 *
 *  Description :  Extracts a byte from the current file frame
 *
 *  Input  :  None
 *
 *  Output :  Extracted Byte.
 *
 *  NOTES : We consider that the programmer is intelligent enough
 *          not to try to get a byte that is out of the frame. Hence,
 *          we provide no bounds check here. (A misbehaving client
 *          could easily page fault using this call).
 *
 ******************************************************************)

 function GET_Byte : Byte;
 begin
   GET_Byte := current_frame^[frame_cursor];
   inc( frame_cursor );
 end;

(*******************************************************************
 *
 *  Function    :  GET_Char
 *
 *  Description :  Extracts a signed byte from the current file frame
 *
 *  Input  :  None
 *
 *  Output :  Extracted char.
 *
 *  NOTES : We consider that the programmer is intelligent enough
 *          not to try to get a byte that is out of the frame. Hence,
 *          we provide no bounds check here. (A misbehaving client
 *          could easily page fault using this call).
 *
 ******************************************************************)

 function GET_Char : ShortInt;
 begin
   GET_Char := ShortInt( current_frame^[frame_cursor] );
   inc( frame_cursor );
 end;

(*******************************************************************
 *
 *  Function    :  GET_Short
 *
 *  Description :  Extracts a short from the current file frame
 *
 *  Input  :  None
 *
 *  Output :  Extracted short.
 *
 *  NOTES : We consider that the programmer is intelligent enough
 *          not to try to get a byte that is out of the frame. Hence,
 *          we provide no bounds check here. (A misbehaving client
 *          could easily page fault using this call).
 *
 ******************************************************************)

 function GET_Short : Short;
 begin
   GET_Short := Short( (current_frame^[ frame_cursor ] shl 8) or
                        current_frame^[frame_cursor+1]      );
   inc( frame_cursor, 2 );
 end;

(*******************************************************************
 *
 *  Function    :  GET_UShort
 *
 *  Description :  Extracts an unsigned  short from the frame
 *
 *  Input  :  None
 *
 *  Output :  Extracted ushort.
 *
 *  NOTES : We consider that the programmer is intelligent enough
 *          not to try to get a byte that is out of the frame. Hence,
 *          we provide no bounds check here. (A misbehaving client
 *          could easily page fault using this call).
 *
 ******************************************************************)

 function GET_UShort : UShort;
 begin
   GET_UShort := (UShort(current_frame^[ frame_cursor ]) shl 8) or
                  UShort(current_frame^[frame_cursor+1]);
   inc( frame_cursor, 2 );
 end;

(*******************************************************************
 *
 *  Function    :  GET_Long
 *
 *  Description :  Extracts a long from the frame
 *
 *  Input  :  None
 *
 *  Output :  Extracted long.
 *
 *  NOTES : We consider that the programmer is intelligent enough
 *          not to try to get a byte that is out of the frame. Hence,
 *          we provide no bounds check here. (A misbehaving client
 *          could easily page fault using this call).
 *
 ******************************************************************)

 function GET_Long : Long;
 begin
   GET_Long := (Long(current_frame^[ frame_cursor ]) shl 24) or
               (Long(current_frame^[frame_cursor+1]) shl 16) or
               (Long(current_frame^[frame_cursor+2]) shl 8 ) or
               (Long(current_frame^[frame_cursor+3])       );
   inc( frame_cursor, 4 );
 end;

(*******************************************************************
 *
 *  Function    :  GET_ULong
 *
 *  Description :  Extracts an unsigned long from the frame
 *
 *  Input  :  None
 *
 *  Output :  Extracted ulong.
 *
 *  NOTES : We consider that the programmer is intelligent enough
 *          not to try to get a byte that is out of the frame. Hence,
 *          we provide no bounds check here. (A misbehaving client
 *          could easily page fault using this call).
 *
 ******************************************************************)

 function GET_ULong : ULong;
 begin
   GET_ULong := Long((current_frame^[ frame_cursor ] shl 24) or
                     (current_frame^[frame_cursor+1] shl 16) or
                     (current_frame^[frame_cursor+2] shl 8 ) or
                      current_frame^[frame_cursor+3]       );
   inc( frame_cursor, 4 );
 end;

(*******************************************************************
 *
 *  Function    :  GET_Tag4
 *
 *  Description :  Extracts a Tag from the frame
 *
 *  Input  :  None
 *
 *  Output :  Extracted 4 byte Tag.
 *
 *  NOTES : We consider that the programmer is intelligent enough
 *          not to try to get a byte that is out of the frame. Hence,
 *          we provide no bounds check here. (A misbehaving client
 *          could easily page fault using this call).
 *
 ******************************************************************)

 function GET_Tag4 : ULong;
 var
   C : array[0..3] of Byte;
 begin
   move ( current_frame^[frame_cursor], c, 4 );
   inc( frame_cursor, 4 );

   GET_Tag4 := ULong(C);
end;

begin
  num_files     := 0;
  current_frame := nil;
  frame_cursor  := 0;
  frame_size    := 0;
end.
