{***************************************************************************}
{*                                                                         *}
{*  FreeType Glyph Viewer.                                                 *}
{*                                                                         *}
{*                                                                         *}
{*    This small program will load a TrueType font file and allow          *}
{*    you to view/scale/rotate its glyphs. Glyphs are in the order         *}
{*    found within the 'glyf' table.                                       *}
{*                                                                         *}
{*  NOTE : This version displays a magnified view of the glyph             *}
{*         along with the pixel grid.                                      *}
{*                                                                         *}
{*  This source code has been compiled and run under both Virtual Pascal   *}
{*  on OS/2 and Borland's BP7.                                             *}
{*                                                                         *}
{***************************************************************************}

program Zoom3;

uses Crt,

{$IFDEF OS2}
     Use32,
{$ENDIF}

     FullScr,
     FreeType,
     TTTypes,
     TTMemory,
     TTFile,
     TTCalc,
     TTTables,
     TTRaster,
     TTIns,
     TTError;

{&PMTYPE NOVIO}

{$DEFINE DEBUG}

const
  Precis  = 64;
  Precis2 = Precis div 2;

  PrecisAux = 1024;

  Screen_Width  = 640;
  Screen_Height = 480;
  Screen_Cols   = Screen_Width div 8;
  Screen_Size   = Screen_Cols * Screen_Height;

  Grid_Width  = Screen_Width div 16;
  Grid_Height = Screen_Height div 16;
  Grid_Cols   = Grid_Width div 8;
  Grid_Size   = Grid_Cols * Grid_Height;

  Screen_Center_X = Screen_Width div 2;
  Screen_Center_Y = Screen_Height div 2;

  Grid_Center_X = Grid_Width div 2;
  Grid_Center_Y = Grid_Height div 2;

  Profile_Buff_Size = 64000;

var

  Font_Buffer : PStorage;

  num_pts : word;
  num_ctr : word;

  glyfArray : word;

  epts_ctr : PShort;

  xCoord : TT_PCoordinates;
  yCoord : TT_PCoordinates;
  Flag   : TT_PTouchTable;

  ymin, ymax, xmax, xmin, xsize : longint;
  res, old_res                  : int;

  numPoints, numContours : int;

  Bitmap_small : TRasterBlock;
  Bitmap_big   : TRasterBlock;

  Rotation : int;  (* Angle modulo 1024 *)

  num_glyphs : int;

  stream    : TT_Stream;
  resident  : PResidentLeading;
  instance  : PInstanceRecord;

  display_outline : boolean;
  hint_glyph      : boolean;
  scan_type       : byte;

  old_glyph : int;
  glyph     : int;


(*******************************************************************
 *
 *  Function    : InitRows
 *
 *  Description : Allocates the target bitmaps
 *
 *****************************************************************)

Procedure InitRows;
var
  P: Pointer;
begin

  (* The big bitmap will contain the grid, the glyph contours and *)
  (* the magnified bitmap                                         *)

  Bitmap_big.rows  := Screen_Height;
  Bitmap_big.cols  := Screen_Cols;
  Bitmap_big.width := Screen_Width;
  Bitmap_big.flow  := TTFlowDown;
  Bitmap_big.size  := Screen_Size;

  GetMem( Bitmap_big.buffer, Bitmap_big.size );
  if Bitmap_big.buffer = NIL then
   begin
    Writeln('ERREUR:InitRows:Not enough memory to allocate big BitMap');
    halt(1);
   end;

  (* The small bitmap contains the rendered glyph, and is then later *)
  (* magnified into the big bitmap                                   *)

  Bitmap_small.rows  := Grid_Height;
  Bitmap_small.cols  := Grid_Cols;
  Bitmap_small.width := Grid_Width;
  Bitmap_small.flow  := TTFlowDown;
  Bitmap_small.size  := Grid_Size;

  GetMem( Bitmap_small.buffer, Bitmap_small.size );
  if Bitmap_small.buffer = NIL then
   begin
    Writeln('ERREUR:InitRows:Not enough memory to allocate big BitMap');
    halt(1);
   end;

  (* Note that the render pool should be allocated from the font pool *)
  (* for various debugging reasons, and because we're still in alpha, *)
  (* we don't do it yet..                                             *)

  GetMem( P, Profile_Buff_Size );
  if P=nil then
   begin
    writeln('ERREUR:InitRows:Not enough memory to allocate render pool');
    Halt(2);
   end;

  Init_Rasterizer( P,
                   Profile_Buff_Size,
                   nil,
                   0 );

  FillChar( Bitmap_big.Buffer^, Bitmap_big.Size, 0 );
  FIllChar( Bitmap_small.Buffer^, Bitmap_small.size, 0 );
end;

(*******************************************************************
 *
 *  Function    :  ClearData
 *
 *  Description :  Clears the bitmaps
 *
 *****************************************************************)

Procedure ClearData;
var i: integer;
begin
  FillChar( Bitmap_big.Buffer^, Bitmap_big.Size, 0 );

  FIllChar( Bitmap_small.Buffer^, Bitmap_small.size, 0 );
end;


(*******************************************************************
 *
 *  Function    :  LoadTrueTypeChar
 *
 *  Description :  Loads a single glyph into the xcoord, ycoord and
 *                 flag arrays, from the instance data.
 *
 *****************************************************************)

Function LoadTrueTypeChar( index      : integer;
                           resolution : integer;
                           center_x   : integer;
                           center_y   : integer ) : boolean;

var
  off    : longint;
  x, y   : Longint;
  i      : integer;
  j      : word;
  EM     : Word;

  ResR   : Real;
  CR, SR : Real;

begin
  LoadtrueTypeChar := FALSE;

  (* TT_Reset_Glyph( instance ); *)

  if not TT_Load_Glyph( instance, index ) then exit;

  numPoints   := instance^.pts.n;
  numContours := instance^.numContours;

  if (numPoints <= 0) or (numContours <= 0) then exit;

  for j := 0 to numPoints-1 do with instance^, pts do
  begin
    x := MulDiv_Round( org_x^[j], Scale1, Scale2 );
    y := MulDiv_Round( org_y^[j], Scale1, Scale2 );

    org_x^[j] := x;
    org_y^[j] := y;
    cur_x^[j] := x;
    cur_y^[j] := y;
    touch^[j] := touch^[j] and 1;
  end;

  if hint_glyph then
  begin

    Set_CodeRange( instance, 3, instance^.glyphIns, instance^.glyphSize );

    if not Goto_CodeRange( instance, TT_CodeRange_Glyph, 0 )
    then exit;

    TT_Glyph_Defaults( instance );

    with instance^, pts do
    begin
      cur_x^[n-2] := (org_x^[n-2]+32) and -64;
      cur_x^[n-1] := (org_x^[n-1]+32) and -64;
    end;

    if not RunIns( instance ) then exit;

  end;

  dec( numPoints, 2 );  (* remove phantom points *)

  for j := 0 to numPoints-1 do with instance^, pts do
  begin
    xCoord^[j] := cur_x^[j] + 32;
    yCoord^[j] := cur_y^[j] + 32 + 64;
    Flag^[j]   := touch^[j] and 1;
  end;

  {

  xMin := instance^.xMin;  (* Bounding box in EM values  *)
  xMax := instance^.xMax;  (* Note that these values are *)
  yMin := instance^.yMin;  (* completely random in CJK   *)
  yMax := instance^.yMax;  (* font files                 *)

  EM := instance^.fontres^.fontHeader.Units_Per_EM;

  dec( xMax, xMin );
  dec( yMax, yMin );

  dec ( resolution );
  resR := resolution/EM;

  xmax := trunc( (xmax * resR + 1)/2 );
  ymax := trunc( (ymax * resR + 1)/2 );

  (* Note that the center of the bounding box is displayed at the  *)
  (* center of our screen/bitmap with this computation.            *)
  (* this obviously is _not_ the right way to do it, but it's good *)
  (* enough for us                                                 *)

  CR := Cos( Rotation*Pi/512 );
  SR := Sin( Rotation*Pi/512 );

  for j:=0 to numPoints-1 do
   begin

    x := instance^.pts.cur_x^[j] * ( resolution / EM );
    y := instance^.pts.cur_y^[j] * ( resolution / EM );

    off := Trunc( Precis*( CR*(x-xmax) + SR*(y-ymax) ) );

    XCoord^[j] := Precis * center_X + off;
    (*
    XCoord^[j] := Precis*( Center_X + off div Precis ) + Precis2;
    *)
    off := Trunc( Precis*( - SR*(x-xmax) + CR*(y-ymax) ) );

    YCoord^[j] := Precis* center_Y + off;
    (*
    YCoord^[j] := Precis*( Center_Y + off div Precis ) + Precis2;
    *)
    Flag^[j] := instance^.pts.touch^[j] and 1;
   end;

  }

  LoadTrueTypeChar := TRUE;
end;


function Render_Magnified( glyph : integer ): boolean;
label
  Exit_1;
type
  TBlock = array[0..7] of Byte;
  PBlock = ^TBlock;
const
{
  Grid_Empty : TBlock
             = ( $10, $10, $10, $FF, $10, $10, $10, $10 );
}
  Grid_Pixel2 : TBlock
              = ( $FE, $FE, $FE, $FE, $FE, $FE, $FE, $00 );

  Pixel_Center_X = 3;
  Pixel_Center_Y = 3;

  Grid_Empty : TBlock
             = ( $00, $00, $00, $10, $00, $00, $00, $00 );

  Grid_Pixel1 : TBlock
              = ( $00, $00, $38, $38, $38, $00, $00, $00 );

  Big_Center_X = Grid_Center_X*16 + Pixel_Center_X;
  Big_Center_Y = Grid_Center_Y*16 + Pixel_Center_Y;

var
  r, w, w2, u, v, b, c : integer;

  x,y : Long;

  block : PBlock;
  G     : TGlyphRecord;

  pixel,
  empty : PBlock;
begin
  Render_Magnified := False;

  if not LoadTrueTypeChar( glyph, res div 8, Grid_Center_X, Grid_Center_Y )
    then exit;

  (* We begin rendering the glyph within the small bitmap *)

  G.numConts  := instance^.numContours;
  G.endPoints := instance^.endContours;
  G.Points    := numPoints;
  G.XCoord    := xCoord;
  G.YCoord    := yCoord;
  G.Flag      := Flag;

  if not Render_Glyph ( G, @Bitmap_small, scan_type ) then goto Exit_1;

  (* Then, we render the glyph outline in the bit bitmap *)

  for r := 0 to numPoints-1 do
  begin
    x := (xcoord^[r]-32);
    y := (ycoord^[r]-32);

    x := (x - Precis*Grid_Center_X)*16 + Big_Center_X*Precis;
    y := (y - Precis*Grid_Center_Y)*16 + Big_Center_Y*Precis;

    xcoord^[r] := x + 8*64;
    ycoord^[r] := y + 8*64;
  end;

   (* first compute the magnified coordinates *)

  G.numConts  := instance^.numContours;
  G.endPoints := instance^.endContours;
  G.Points    := numPoints;
  G.XCoord    := XCoord;
  G.YCoord    := YCoord;
  G.Flag      := Flag;

  if display_outline then
    if not Render_Glyph ( G, @Bitmap_big, scan_type ) then goto Exit_1;

  (* Now, magnify the small bitmap, XORing it to the big bitmap *)

  r := 0;
  w := 0;
  b := 0;

  empty := @Grid_Empty;

  if display_outline then pixel := @Grid_Pixel1
                     else pixel := @Grid_Pixel2;

  for y := 0 to Grid_Height-1 do
  begin

    for x := 0 to Grid_Width-1 do
    begin

      w2 := w;
      b  := b shr 1;

      if b = 0 then
      begin
        c := PByte(Bitmap_small.Buffer)^[r];
        b := $80;
        inc( r );
      end;

      if c and b <> 0 then block := pixel
                      else block := empty;

      for v := 0 to 7 do
      begin
        PByte(Bitmap_Big.Buffer)^[w2] := PByte(Bitmap_Big.Buffer)^[w2]
                                         xor block^[v];
        inc( w2, Bitmap_Big.cols );
      end;

      inc( w, 2 );

    end;

    inc( w, 15*Screen_Cols );

  end;


  (* Display the resulting big bitmap *)

  Display( Bitmap_big.Buffer^, 450, 80  );

Exit_1:
  (* Clear the bitmaps *)

  ClearData;
end;


procedure Usage;
begin
    Writeln('Simple TrueType Glyphs viewer - part of the FreeType project' );
    Writeln;
    Writeln('Usage : ',paramStr(0),' FontName[.TTF]');
    Halt(1);
end;



var i: integer;
    heure,
    min1,
    min2,
    sec1,
    sec2,
    cent1,
    cent2  :
{$IFDEF OS2}
    longint;
{$ELSE}
    word;
{$ENDIF}

    C : Char;

    Filename : String;

label Fin;

var
  Fail     : Int;
  glyphStr : String[4];

begin
  TextMode( co80+Font8x8 );

  GetMem       ( Font_Buffer, 512000 );
  Init_FontPool( Font_Buffer^, 512000 );

  num_pts   := 0;
  num_ctr   := 0;

  xCoord  := NIL;
  yCoord  := NIL;
  Flag    := NIL;

  if ParamCount = 0 then Usage;

  Filename := ParamStr(1);

  if Pos('.',FileName) = 0 then FileName:=FileName+'.TTF';

  if not TT_Open_File( FileName, stream ) then
    begin
      Writeln('ERROR: Could not open ', FileName );
      halt(1);
    end;

  if not TT_Load_Resident_Table( stream, resident ) then
   begin
    Writeln('ERROR: Could not load data from ', FileName );
    Halt(1);
   end;

  i := resident^.MaxProfile.maxPoints;

  GetMem( XCoord, SizeOf(Fixed) * i );
  GetMem( YCoord, SizeOf(Fixed) * i );
  GetMem( Flag, i );

  if not TT_Load_Instance_Data( resident, instance ) then
   begin
    Writeln('ERROR: Could not open instance from ', FileName );
    Halt(1);
   end;

  Rotation := 0;
  Fail     := 0;
  res      := 8;

  TT_Reset_Instance( instance, res, 96 );

  if not Goto_CodeRange( instance, TT_CodeRange_Font, 0 ) then
  begin
    Writeln('Could not go to font program' );
    halt(1);
  end;

  if not RunIns( instance ) then
  begin
    Writeln('Error while running font program' );
    halt(1);
  end;

  if not Goto_CodeRange( instance, TT_CodeRange_Cvt, 0 ) then
  begin
    Writeln('Could not go to CVT program' );
    halt(1);
  end;

  if not RunIns( instance ) then
  begin
    Writeln('Error while running CVT program' );
    Writeln('> ', TT_ErrorStr, ' <' );
    halt(1);
  end;

  TT_Set_Glyph_Defaults( instance );

  InitRows;

  SetGraphScreen( FS_Graphics_Mono );

  display_outline := true;
  hint_glyph      := false;
  scan_type       := 2;

  num_glyphs := instance^.fontres^.numGlyphs;

  old_glyph := -1;
  old_res   := res;
  glyph     := 71;

  Repeat

    if not Render_Magnified( glyph ) then inc( Fail );

    goto_xy( 54, 8 );
    Write_Str('glyph ');
    Str( glyph:3, glyphStr );
    Write_Str( glyphStr );

    goto_xy( 54, 9 );
    Write_Str('pts ');
    Str( instance^.pointSize div 64:4, glyphStr );
    Write_Str( glyphStr );

    goto_xy( 54, 10 );
    Write_Str('Hinting (''z'')   : ' );
    if hint_glyph then Write_Str('on ')
                  else Write_Str('off');

    goto_xy( 54, 11 );
    Write_Str('Outline (''t'')   : ' );
    if display_outline then Write_Str('on ')
                       else Write_Str('off');

    goto_xy( 54, 13 );
    Write_Str('scan type(''e'')  : ' );
    case scan_type of
      0 : Write_Str('none   ');
      1 : Write_Str('level 1');
      2 : Write_Str('level 2');
      4 : Write_Str('level 4');
      5 : Write_Str('level 5');
    end;

    C := Readkey;
    Case C of

     #27 : goto Fin;
     { ESC Key }

     'z' : hint_glyph := not hint_glyph;

     'e' : begin
             inc( scan_type );
             if scan_type = 6 then scan_type := 0;
             if scan_type = 3 then scan_type := 4;
           end;

     't' : display_outline := not display_outline;

     'x' : Rotation := ( Rotation-1 ) and 1023;
     'c' : Rotation := ( Rotation+1 ) and 1023;
     'v' : Rotation := ( Rotation-16) and 1023;
     'b' : Rotation := ( Rotation+16) and 1023;

     '+' : if res < 1040 then inc( res, 10 ) else res := 1050;
     '-' : if res > 11 then dec( res, 10 ) else res := 1;

     'i' : if glyph > 10 then dec( glyph, 10 ) else i := 0;

     'o' : if glyph < num_glyphs-11 then inc( glyph, 10 )
                                    else glyph := num_glyphs-1;

     'k' : if glyph > 0 then dec(glyph);
     'l' : if glyph < num_glyphs-1 then inc(glyph);

     'u' : if res > 0 then dec(res);
     'j' : if res < 1040 then inc(res);

    end;

    if old_res <> res then
    begin

      TT_Reset_Instance( instance, res, 96 );

      if not Goto_CodeRange( instance, TT_CodeRange_Cvt, 0 ) then
      begin
        Writeln('Could not go to CVT program' );
        halt(1);
      end;

      if not RunIns( instance ) then
      begin
        Writeln('Error while running CVT program' );
        Writeln('> ', TT_ErrorStr, ' <' );
        halt(1);
      end;

      TT_Set_Glyph_Defaults( instance );

      old_res := res;
    end;

  Until false;

 Fin:
  RestoreScreen;
  TT_Close_File(stream);

  Writeln('Echecs : ', Fail );
end.

