/*******************************************************************
 *
 *  freetype.h
 *
 *    High-level interface specification.
 *
 *  Copyright 1996, 1997 by
 *  David Turner, Robert Wilhelm, and Werner Lemberg.
 *
 *  This file is part of the FreeType project, and may only be used
 *  modified and distributed under the terms of the FreeType project
 *  license, LICENSE.TXT. By continuing to use, modify or distribute 
 *  this file you indicate that you have read the license and
 *  understand and accept it fully.
 *
 *  Notes:
 *
 *    This is the only file that should be included by client            
 *    application sources for the final release. All other types
 *    and functions defined in the "tt*.h" files are library  
 *    internals, and should not be included (except of course
 *    during development, as now).
 *
 *    FreeType is still in alpha!
 *
 ******************************************************************/

#ifndef FREETYPE_H
#define FREETYPE_H

#ifdef __cplusplus
extern "C" {
#endif

  /*******************************************************************/
  /*                                                                 */
  /*  FreeType types definitions.                                    */
  /*                                                                 */
  /*  All these begin with a 'TT_' prefix.                           */
  /*                                                                 */
  /*******************************************************************/

  typedef signed long     TT_Fixed;   /* Signed Fixed 16.16 Float */

  typedef signed short    TT_FWord;   /* Distance in FUnits */
  typedef unsigned short  TT_UFWord;  /* Unsigned distance */

  typedef signed short    TT_Short;
  typedef unsigned short  TT_UShort;
  typedef signed long     TT_Long;
  typedef unsigned long   TT_ULong;

  typedef signed short    TT_F2Dot14; /* Signed fixed float 2.14 used for */
                                      /* unary vectors, with layout:      */
                                      /*                                  */
                                      /*  s : 1  -- sign bit              */
                                      /*  m : 1  -- mantissa bit          */
                                      /*  f : 14 -- unsigned fractional   */
                                      /*                                  */
                                      /*  's:m' is the 2-bit signed int   */
                                      /*  value to which the positive     */
                                      /*  fractional part should be       */
                                      /*  added.                          */
                                      /*                                  */
  typedef signed long     TT_F26Dot6; /* 26.6 fixed float, used for       */
                                      /* glyph points pixel coordinates.  */

  typedef struct  _TT_UnitVector      /* guess what...  */
  { 
    TT_F2Dot14  x;
    TT_F2Dot14  y;
  } TT_UnitVector;

  typedef struct  _TT_Vector          /* Simple vector type */
  {
    TT_F26Dot6  x;
    TT_F26Dot6  y;
  } TT_Vector;


  typedef struct  _TT_Matrix                       
  {
    TT_Fixed  xx, xy;
    TT_Fixed  yx, yy;
  } TT_Matrix;

  /* A simple 2x2 matrix used for transforms */
  /* you should use 16.16 fixed floats       */
  /*                                         */
  /*  x' = xx*x + xy*y                       */
  /*  y' = yx*x + yy*y                       */
  /*                                         */

  /* A structure used to describe the source glyph to the renderer. */
  typedef struct  _TT_Glyph_Outline
  {
    unsigned int     contours;   /* number of contours in glyph          */
    unsigned short*  conStarts;  /* points to an array of each contour's */
                                 /* start point index                    */
    unsigned int     points;     /* number of points in the glyph        */
    long*            xCoord;     /* table of x coordinates               */
    long*            yCoord;     /* table of y coordinates               */
    unsigned char*   flag;       /* table of flags                       */

    char             dropout_mode;  /* dropout mode */
  } TT_Glyph_Outline;


  /* A structure used to return glyph metrics. */
  typedef struct  _TT_Glyph_Metrics
  {
    TT_F26Dot6  leftSideBearing;
    TT_F26Dot6  advanceWidth;

    TT_F26Dot6  xMin, yMin, xMax, yMax;
  } TT_Glyph_Metrics;


  /* A structure used to return instance metrics. */
  typedef struct  _TT_Instance_Metrics
  {
    int  pointSize;
    /* pointsize in points ( 1 point = 1/72 inch ) */

    int  x_ppem;  /* horizontal pixels per EM square */
    int  y_ppem;  /* vertical pixels per EM square   */

    int  x_resolution;  /* device horizontal resolution in dpi */
    int  y_resolution;  /* device vertical resolution in dpi   */

  } TT_Instance_Metrics;


  /* Flow constants:                                            */
  /*                                                            */
  /* The flow of a bitmap refers to the way lines are oriented  */
  /* within the bitmap data, i.e., the orientation of the Y     */
  /* coordinate axis.                                           */

  /* for example, if the first bytes of the bitmap pertain to   */
  /* its top-most line, then the flow is 'down'. if these bytes */
  /* pertain to its lowest line, the the flow is 'up'.          */

#define TT_Flow_Down  -1  /* bitmap is oriented from top to bottom */
#define TT_Flow_Up     1  /* bitmap is oriented from bottom to top */
#define TT_Flow_Error  0  /* an error occured during rendering     */


  /* A structure used to describe the target bitmap or pixmap to the   */
  /* renderer. Note that there is nothing in this structure that gives */
  /* the nature of the buffer.                                         */

  /* IMPORTANT NOTE :                                                  */
  /*                                                                   */
  /*   A pixmap's width _must_ be a multiple of 4. Clipping            */
  /*   problems will arise otherwise, if not page faults !             */
  /*                                                                   */
  /*   The typical settings are :                                      */
  /*                                                                   */
  /*   - for an WxH bitmap :                                           */
  /*                                                                   */
  /*       rows  = H                                                   */
  /*       cols  = (W+7)/8                                             */
  /*       width = W                                                   */
  /*       flow  = your_choice                                         */
  /*                                                                   */
  /*   - for an WxH pixmap                                             */
  /*                                                                   */
  /*       rows  = H                                                   */
  /*       cols  = (W+3) & ~3                                          */
  /*       width = cols                                                */
  /*       flow  = your_choice                                         */

  typedef struct  _TT_Raster_Map
  {
    int     rows;    /* number of rows                    */
    int     cols;    /* number of columns (bytes) per row */
    int     width;   /* number of pixels per line         */
    int     flow;    /* bitmap orientation                */

    void*   bitmap;  /* bit/pixmap buffer                 */
    long    size;    /* bit/pixmap size in bytes          */
  } TT_Raster_Map;





  /* ------- The font header TrueType table structure ----- */

  typedef struct  _TT_Header
  {
    TT_Fixed   Table_Version;
    TT_Fixed   Font_Revision;

    TT_Long    CheckSum_Adjust;
    TT_Long    Magic_Number;

    TT_UShort  Flags;
    TT_UShort  Units_Per_EM;

    TT_Long    Created [2];
    TT_Long    Modified[2];

    TT_FWord   xMin;
    TT_FWord   yMin;
    TT_FWord   xMax;
    TT_FWord   yMax;

    TT_UShort  Mac_Style;
    TT_UShort  Lowest_Rec_PPEM;

    TT_Short   Font_Direction;
    TT_Short   Index_To_Loc_Format;
    TT_Short   Glyph_Data_Format;

  } TT_Header;

  /* ------- The horizontal header TrueType table structure ----- */

  typedef struct _TT_Horizontal_Header
  {
    TT_Fixed   Version;
    TT_FWord   Ascender;
    TT_FWord   Descender;
    TT_FWord   Line_Gap;

    TT_UFWord  advance_Width_Max;

    TT_FWord   min_Left_Side_Bearing;
    TT_FWord   min_Right_Side_Bearing;
    TT_FWord   xMax_Extent;
    TT_FWord   caret_Slope_Rise;
    TT_FWord   caret_Slope_Run;

    TT_Short   Reserved[5];

    TT_Short   metric_Data_Format;
    TT_UShort  number_Of_HMetrics;

  } TT_Horizontal_Header;


  /* ----------- OS/2 Table ----------------------------- */

  typedef struct _TT_OS2
  {
    TT_UShort  version;                /* 0x0001 */
    TT_FWord   xAvgCharWidth;
    TT_UShort  usWeightClass;
    TT_UShort  usWidthClass;
    TT_Short   fsType;
    TT_FWord   ySubscriptXSize;
    TT_FWord   ySubscriptYSize;
    TT_FWord   ySubscriptXOffset;
    TT_FWord   ySubscriptYOffset;
    TT_FWord   ySuperscriptXSize;
    TT_FWord   ySuperscriptYSize;
    TT_FWord   ySuperscriptXOffset;
    TT_FWord   ySuperscriptYOffset;
    TT_FWord   yStrikeoutSize;
    TT_FWord   yStrikeoutPosition;
    TT_Short   sFamilyClass;

    char       panose[10];

    TT_ULong   ulUnicodeRange1;        /* Bits 0-31   */
    TT_ULong   ulUnicodeRange2;        /* Bits 32-63  */
    TT_ULong   ulUnicodeRange3;        /* Bits 64-95  */
    TT_ULong   ulUnicodeRange4;        /* Bits 96-127 */

    char       achVendID[4];

    TT_UShort  fsSelection;
    TT_UShort  usFirstCharIndex;
    TT_UShort  usLastCharIndex;
    TT_UShort  sTypoAscender;
    TT_UShort  sTypoDescender;
    TT_UShort  sTypoLineGap;
    TT_UShort  usWinAscent;
    TT_UShort  usWinDescent;

    /* only version 1 tables: */

    TT_ULong   ulCodePageRange1;       /* Bits 0-31   */
    TT_ULong   ulCodePageRange2;       /* Bits 32-63  */

  } TT_OS2;

  /* ----------- Postscript table ------------------------ */

  typedef struct _TT_Postscript
  {
    TT_Fixed  FormatType;
    TT_Fixed  italicAngle;
    TT_FWord  underlinePosition;
    TT_FWord  underlineThickness;
    TT_ULong  isFixedPitch;
    TT_ULong  minMemType42;
    TT_ULong  maxMemType42;
    TT_ULong  minMemType1;
    TT_ULong  maxMemType1;

    /* glyph names follow in the file, but we don't */
    /* load them by default..                       */

  } TT_Postscript;



  /* A structure used to describe face properties. */
  typedef struct  _TT_Face_Properties
  {
    int  num_Glyphs;   /* number of glyphs in face              */
    int  max_Points;   /* maximum number of points in a glyph   */
    int  max_Contours; /* maximum number of contours in a glyph */

    int  max_Faces;    /* 0 for normal TrueType files, and the  */
                       /* number of embedded faces minus 1 for  */
                       /* TrueType collections                  */

    TT_Header*             header;        /* TrueType header table      */
    TT_Horizontal_Header*  horizontal;    /* TrueType horizontal header */
    TT_OS2*                os2;           /* TrueType OS/2 table        */
    TT_Postscript*         postscript;    /* TrueType Postscript table  */

  } TT_Face_Properties;


  /* Here are the definitions of the handle types used for FreeType's */
  /* most common objects accessed by the client application. We use   */
  /* a simple trick there:                                            */
  /*                                                                  */
  /*   Each handle type is a structure that only contains one         */
  /*   pointer. The advantage of structures is that there are         */
  /*   mutually exclusive types. We could have defined the            */
  /*   following types:                                               */
  /*                                                                  */
  /*     typedef void*  TT_Stream;                                    */
  /*     typedef void*  TT_Face;                                      */
  /*     typedef void*  TT_Instance;                                  */
  /*     typedef void*  TT_Glyph;                                     */
  /*     typedef void*  TT_CharMap;                                   */
  /*                                                                  */
  /*   but these would have allowed lines like:                       */
  /*                                                                  */
  /*      stream = instance;                                          */
  /*                                                                  */
  /*   in the client code this would be a severe bug, unnoticed       */
  /*   by the compiler!                                               */
  /*                                                                  */
  /*   Thus, we enforce type checking with a simple language          */
  /*   trick...                                                       */
  /*                                                                  */
  /*   NOTE:  Some macros are defined in tttypes.h to perform         */
  /*          automatic type conversions for library hackers..        */
  /*                                                                  */

  struct _TT_Stream   { void*  z; };
  struct _TT_Face     { void*  z; };
  struct _TT_Instance { void*  z; };
  struct _TT_Glyph    { void*  z; };
  struct _TT_CharMap  { void*  z; };

  typedef struct _TT_Stream    TT_Stream;    /* stream handle type        */
  typedef struct _TT_Face      TT_Face;      /* face handle type          */
  typedef struct _TT_Instance  TT_Instance;  /* instance handle type      */
  typedef struct _TT_Glyph     TT_Glyph;     /* glyph handle type         */
  typedef struct _TT_CharMap   TT_CharMap;   /* character map handle type */

  typedef int  TT_Error;

  /*******************************************************************/
  /*                                                                 */
  /*  FreeType API                                                   */
  /*                                                                 */
  /*  All these begin with a 'TT_' prefix.                           */
  /*                                                                 */
  /*  Most of them are implemented in the 'ttapi.c' source file.     */
  /*                                                                 */
  /*******************************************************************/

  /* Initialize the engine. */
  TT_Error  TT_Init_FreeType();

  /* Finalize the engine, and release all allocated objects. */
  TT_Error  TT_Done_FreeType();

  /* Set the gray level palette. This is an array of 5 bytes used */
  /* to produce the font smoothed pixmaps. By convention :        */
  /*                                                              */
  /*  palette[0] = background (white)                             */
  /*  palette[1] = light                                          */
  /*  palette[2] = medium                                         */
  /*  palette[3] = dark                                           */
  /*  palette[4] = foreground (black)                             */
  /*                                                              */
  TT_Error  TT_Set_Raster_Gray_Palette( char*  palette );
  

  /* Open a new TrueType font file, and returns a handle for */
  /* it in variable '*face'.                                 */

  /* Note : the file can be either a TrueType file (*.ttf) or  */
  /*        a TrueType collection (*.ttc), in this case, only  */
  /*        the first face is opened. The number of faces in   */
  /*        the same collection can be obtained in the face's  */
  /*        properties, through TT_Get_Face_Properties and the */
  /*        'max_Faces' field.                                 */

  TT_Error  TT_Open_Face( const char*  fontpathname,
                          TT_Face*     face );

  /* Open a TrueType font file located inside a collection */
  /* The font is designed by its index in 'fontIndex'      */

  TT_Error  TT_Open_Collection( const char*  collectionpathname,
                                int          fontIndex,
                                TT_Face*     face );

  /* Return face properties in the 'properties' structure. */
  TT_Error  TT_Get_Face_Properties( TT_Face              face,
                                    TT_Face_Properties*  properties );

  /* Close a given font object, destroying all associated     */
  /* instances.                                               */
  TT_Error  TT_Close_Face( TT_Face  face );

  /* Open a new font instance and returns an instance handle   */
  /* for it in '*instance'.                                    */
  TT_Error  TT_New_Instance( TT_Face       face,
                             TT_Instance*  instance );

  /* Set device resolutions for a given instance. the values are     */
  /* given in dpi (Dots Per Inch). Default is 96 in both directions. */

  /* NOTE:  y_resolution is currently ignored, and the lib assumes   */
  /*        square pixels.                                           */
  TT_Error  TT_Set_Instance_Resolution( TT_Instance  instance,
                                        int          x_resolution,
                                        int          y_resolution );

  /* Set the pointsize for a given instance. Default is 10. */
  TT_Error  TT_Set_Instance_PointSize( TT_Instance  instance,
                                       int          pointsize );

  TT_Error  TT_Set_Instance_Transforms( TT_Instance instance,
                                        int         rotated,
                                        int         stretched );
  /* We do not provide direct support for rotation or stretching, */
  /* which means that you must perform these operations through   */
  /* a call to TT_Apply_Outline_Transform yourself before calling */
  /* TT_Get_Glyph_Bitmap.. However, the engine needs to know what */
  /* kind of text you're displaying. The following boolean flags  */
  /* notice the interpreter that :                                */
  /*                                                              */
  /*   rotated  : the glyphs will be rotated                      */
  /*   stretched: the glyphs will be stretched                    */
  /*                                                              */
  /* These setting only affect the hinting process !              */
  /*                                                              */

  /* Return instance metrics in 'metrics'. */
  TT_Error  TT_Get_Instance_Metrics( TT_Instance           instance,
                                     TT_Instance_Metrics*  metrics );

  /* Close a given instance object, destroying all associated */
  /* data.                                                    */
  TT_Error  TT_Done_Instance( TT_Instance  instance );

  /* Create a new glyph object related to the given 'face'.   */
  TT_Error  TT_New_Glyph( TT_Face    face,
                          TT_Glyph*  glyph );

  /* Discard (and destroy) a given glyph object. */
  TT_Error  TT_Done_Glyph( TT_Glyph  glyph );


#define TTLOAD_SCALE_GLYPH  1
#define TTLOAD_HINT_GLYPH   2

#define TTLOAD_DEFAULT  (TTLOAD_SCALE_GLYPH | TTLOAD_HINT_GLYPH)


  /* load and process (scale/transform and hint) a glyph from the */
  /* given 'instance'. The glyph and instance handles must be     */
  /* related to the same face object. The glyph index can be      */
  /* computed with a call to TT_Char_Index                        */

  /* the 'load_flags' argument is a combination of the macros     */
  /* TTLOAD_SCALE_GLYPH and TTLOAD_HINT_GLYPH. Hinting will be    */
  /* applied only if the scaling is selected..                    */

  /* When scaling is off (i.e. load_flags = 0), the returned      */
  /* outlines are in EM square coordinates (also called FUnits),  */
  /* extracted directly from the font with no hinting.            */
  /* Other glyph metrics are also in FUnits..                     */

  /* When scaling is on, the returned outlines are in fractional  */
  /* pixel units (i.e. TT_F26Dot6 = 26.6 fixed floats).           */

  /* NOTE : the glyph index must be in the range 0..num_glyphs-1  */
  /*        where 'num_glyphs' is the total number of glyphs in   */
  /*        the font file (given in the face properties)..        */
  TT_Error  TT_Load_Glyph( TT_Instance  instance,
                           TT_Glyph     glyph,
                           int          glyph_index,
                           int          load_flags );

  /* Return glyph outline pointers in 'outline'. Note that the returned */
  /* pointers are owned by the glyph object, and will be destroyed with */
  /* it. The client application should _not_ change the pointers.       */
  TT_Error  TT_Get_Glyph_Outline( TT_Glyph           glyph,
                                  TT_Glyph_Outline*  outline );

  /* Copy the glyph metrics into 'metrics'. */
  TT_Error  TT_Get_Glyph_Metrics( TT_Glyph           glyph,
                                  TT_Glyph_Metrics*  metrics );

  /* Render the glyph into a bitmap, with given position offsets. */
  TT_Error  TT_Get_Glyph_Bitmap( TT_Glyph        glyph,
                                 TT_Raster_Map*  map,
                                 TT_F26Dot6      x_offset,
                                 TT_F26Dot6      y_offset );

  /* Render the glyph into a pixmap, with given position offsets. */
  TT_Error  TT_Get_Glyph_Pixmap( TT_Glyph        glyph,
                                 TT_Raster_Map*  map,
                                 TT_F26Dot6      x_offset,
                                 TT_F26Dot6      y_offset );

  TT_Error  TT_Apply_Outline_Matrix( TT_Glyph_Outline*  outline,
                                     TT_Matrix*         matrix );
  /* Apply a transform to a glyph outline */

  TT_Error  TT_Apply_Outline_Translation( TT_Glyph_Outline*  outline,
                                          TT_F26Dot6         x_offset,
                                          TT_F26Dot6         y_offset );
  /* Apply a translation to a glyph outline */

  /* ----------------- character mappings support ------------- */

  /* Return the number of character mappings found in this file */
  /* Returns -1 in case of failure (invalid face handle)        */
  int  TT_Get_CharMap_Count( TT_Face  face );

  /* Return the ID of charmap number 'charmapIndex' of a given face */
  /* used to enumerate the charmaps present in a TrueType file      */
  TT_Error   TT_Get_CharMap_ID( TT_Face  face,
                                int      charmapIndex,
                                short*   platformID,
                                short*   encodingID );

  /* Look up the character maps found in 'face' and return a handle */
  /* for the one matching 'platformID' and 'platformEncodingID'     */
  /* (see the TrueType specs relating to the 'cmap' table for       */
  /* information on these ID numbers). Returns an error code.       */
  /* In case of failure, the handle is set to NULL and is invalid.  */
  TT_Error  TT_Get_CharMap( TT_Face      face,
                            int          charmapIndex,
                            TT_CharMap*  charMap );

  /* Translate a character code through a given character map   */
  /* and return the corresponding glyph index to be used in     */
  /* a TT_Load_Glyph call. This function returns -1 in case of  */
  /* failure.                                                   */
  int  TT_Char_Index( TT_CharMap      charMap,
                      unsigned short  charCode );

  /* --------------------- names table support ------------------- */

  /* Return the number of name strings found in the name table */
  /* Returns -1 in case of failure (invalid face handle)       */
  int  TT_Get_Name_Count( TT_Face  face );

  /* Return the ID of the name number 'nameIndex' of a given face */
  /* used to enumerate the charmaps present in a TrueType file    */
  TT_Error  TT_Get_Name_ID( TT_Face  face,
                            int      nameIndex,
                            short*   platformID,
                            short*   encodingID,
                            short*   languageID,
                            short*   nameID );

  /* Return the address and length of the name number 'nameIndex' */
  /* of a given face. The string is part of the face object and   */
  /* shouldn't be written to or released.                         */

  /* Note that if the string platformID is not in the range 0..3  */
  /* A null pointer will be returned !                            */
  TT_Error  TT_Get_Name_String( TT_Face  face,
                                int      nameIndex,
                                char**   stringPtr, /* address of pointer     */
                                int*     length );  /* address of str. length */

  /************************ error codes declaration **************/
  
  /* The error codes are grouped in 'classes' used to indicate the */
  /* 'level' at which the error happened.                          */
  /* The class is given by an error code's high byte..             */

  /* ------------- Success is always 0 -------- */

  #define TT_Err_Ok                      0
  
  /* -------- High-level API error codes ------ */
  
  #define TT_Err_Invalid_Face_Handle      0x001
  #define TT_Err_Invalid_Instance_Handle  0x002
  #define TT_Err_Invalid_Glyph_Handle     0x003
  #define TT_Err_Invalid_CharMap_Handle   0x004
  #define TT_Err_Invalid_Result_Address   0x005
  #define TT_Err_Invalid_Glyph_Index      0x006
  #define TT_Err_Invalid_Argument         0x007
  #define TT_Err_Could_Not_Open_File      0x008
  #define TT_Err_File_Is_Not_Collection   0x009
  
  #define TT_Err_Table_Missing            0x00A
  #define TT_Err_Invalid_Horiz_Metrics    0x00B
  #define TT_Err_Invalid_CharMap_Format   0x00C

  #define TT_Err_Invalid_File_Format      0x010
  
  /* -------- Memory component error codes ---- */

  /* this error indicates that an operation cannot */
  /* be performed due to memory exhaustion         */

  #define TT_Err_Out_Of_Memory            0x100
  
  /* -------- File component error codes ------ */
  
  /* these error codes indicate that the file could */
  /* not be accessed properly. Usually, this means  */
  /* a broken font file !                           */

  #define TT_Err_Invalid_File_Offset      0x200
  #define TT_Err_Invalid_File_Read        0x201
  #define TT_Err_Invalid_Frame_Access     0x202
  
  /* -------- Glyph loader error codes -------- */
  
  /* Produced only by the glyph loader, these error */
  /* codes indicate a broken glyph in a font file   */

  #define TT_Err_Too_Many_Points          0x300
  #define TT_Err_Too_Many_Contours        0x301
  #define TT_Err_Invalid_Composite        0x302
  #define TT_Err_Too_Many_Ins             0x303
  
  /* --- bytecode interpreter error codes ----- */
  
  /* These error codes are produced by the TrueType */
  /* bytecode interpreter. They usually indicate a  */
  /* broken font file, a broken glyph within a font */
  /* file, or a bug in the interpreter !            */

  #define TT_Err_Invalid_Opcode           0x400
  #define TT_Err_Too_Few_Arguments        0x401
  #define TT_Err_Stack_Overflow           0x402
  #define TT_Err_Code_Overflow            0x403
  #define TT_Err_Bad_Argument             0x404
  #define TT_Err_Divide_By_Zero           0x405
  #define TT_Err_Storage_Overflow         0x406
  #define TT_Err_Cvt_Overflow             0x407
  #define TT_Err_Invalid_Reference        0x408
  #define TT_Err_Invalid_Distance         0x409
  #define TT_Err_Interpolate_Twilight     0x40A
  #define TT_Err_Debug_OpCode             0x40B
  #define TT_Err_ENDF_In_Exec_Stream      0x40C
  #define TT_Err_Out_Of_CodeRanges        0x40D
  #define TT_Err_Nested_DEFS              0x40E 
  #define TT_Err_Invalid_Coderange        0x40F
  
  /* ------ internal failures error codes ----- */

  /* These error codes are produced when an incoherent */
  /* library state has been detected. These reflect a  */
  /* severe bug in the engine ! (or a major overwrite  */
  /* of your application into the library's data)      */
  
  #define TT_Err_Nested_Frame_Access      0x500
  #define TT_Err_Invalid_Cache_List       0x501
  #define TT_Err_Could_Not_Find_Context   0x502
  #define TT_Err_Unlisted_Object          0x503
  
  /* ---- scan-line converter error codes ----- */
  
  /* These error codes are produced by the raster component   */
  /* They indicate that an outline structure was incoherently */
  /* setup, or that you're trying to render an horribly       */
  /* complex glyph !                                          */

  #define TT_Err_Raster_Pool_Overflow     0x600
  #define TT_Err_Raster_Negative_Height   0x601
  #define TT_Err_Raster_Invalid_Value     0x602
  #define TT_Err_Raster_Not_Initialized   0x603
  

#ifdef __cplusplus
}
#endif

#endif /* FREETYPE_H */


/* End */
