/*******************************************************************
 *
 *  ttobjs.h                                                     1.0
 *
 *    Objects definition unit.
 *
 *  Copyright 1996, 1997 by
 *  David Turner, Robert Wilhelm, and Werner Lemberg.
 *
 *  This file is part of the FreeType project, and may only be used
 *  modified and distributed under the terms of the FreeType project
 *  license, LICENSE.TXT. By continuing to use, modify or distribute
 *  this file you indicate that you have read the license and
 *  understand and accept it fully.
 *
 ******************************************************************/

#ifndef TTOBJS_H
#define TTOBJS_H

#include "ttcommon.h"
#include "ttengine.h"
#include "tterror.h"
#include "ttmutex.h"
#include "ttcache.h"
#include "tttables.h"
#include "ttcmap.h"

  #ifdef __cplusplus
  extern "C" {
  #endif

/*                                                                       */
/*  This files contains the definitions and methods of the four          */
/*  kinds of objects managed by the FreeType engine. These are :         */
/*                                                                       */
/*                                                                       */
/*   Face objects :                                                      */
/*                                                                       */
/*     There is always one face object per opened TrueType font          */
/*     file, and only one. The face object contains data that is         */
/*     independent of current transform/scaling/rotation and             */
/*     pointsize, or glyph index. This data is made of several           */
/*     critical tables that are loaded on face object creation.          */
/*                                                                       */
/*     A face object tracks all active and recycled objects of           */
/*     the instance and execution context classes. Destroying a face     */
/*     object will automatically destroy all associated instances        */
/*                                                                       */
/*                                                                       */
/*   Instance objects :                                                  */
/*                                                                       */
/*     An instance object always relates to a given face object,         */
/*     known as its 'parent' or 'owner', and contains only the           */
/*     data that is specific to one given pointsize/transform of         */
/*     the face. You can only create an instance from a face object.     */
/*                                                                       */
/*     An instance's current transform/pointsize can be changed          */
/*     any time using a singly high-level API call.                      */
/*     ( TT_Reset_Instance )                                             */
/*                                                                       */
/*   Execution Context objects :                                         */
/*                                                                       */
/*     An execution context (or context in short) relates to a face      */
/*     It contains the data and tables that are necessary to load        */
/*     and hint (i.e. execute the glyph instructions of) one glyph.      */
/*     A context is a transient object that is queried/created on        */
/*     the fly : client applications never deal with them directly.      */
/*                                                                       */
/*                                                                       */
/*   Glyph objects :                                                     */
/*                                                                       */
/*     A glyph object contains only the minimal glyph information        */
/*     needed to render one glyph correctly. This means that a glyph     */
/*     object really contains tables that are sized to hold the          */
/*     contents of _any_ glyph of a given face. A client application     */
/*     can usually create one glyph object for a given face, then use    */
/*     it for all subsequent loads.                                      */
/*                                                                       */
/*   Here is an example of a client application :                        */
/*   ( NOTE : No error checking performed here !! )                      */
/*                                                                       */
/*                                                                       */
/*     TT_Face       face;         -- face handle                        */
/*     TT_Instance   ins1, ins2;   -- two instance handles               */
/*     TT_Glyph      glyph;        -- glyph handle                       */
/*                                                                       */
/*     TT_Init_FreeType();                                               */
/*                                                                       */
/*     -- Initialize the engine. This must be done prior to _any_        */
/*     -- operation.                                                     */
/*                                                                       */
/*     TT_Open_Face( "/some/face/name.ttf", &face );                     */
/*                                                                       */
/*     -- create the face object. This call opens the font file          */
/*                                                                       */
/*     TT_New_Instance( face, &ins1 );                                   */
/*     TT_New_Instance( face, &ins2 );                                   */
/*                                                                       */
/*     TT_Set_Instance_PointSize( ins1, 8 );                             */
/*     TT_Set_Instance_PointSize( ins2, 12 );                            */
/*                                                                       */
/*     -- creates two distinct instances of the same face                */
/*     -- ins1  is pointsize 8 at resolution 96 dpi                      */
/*     -- ins2  is pointsize 12 at resolution 96 dpi                     */
/*                                                                       */
/*     TT_New_Glyph( face, &glyph );                                     */
/*                                                                       */
/*     -- create a new glyph object which will receive the contents      */
/*     -- of any glyph of 'face'                                         */
/*                                                                       */
/*     TT_Load_Glyph( ins1, glyph, 64, DEFAULT_GLYPH_LOAD );             */
/*                                                                       */
/*     -- load glyph indexed 64 at pointsize 8 in the 'glyph' object     */
/*     -- NOTE : This call will fail if the instance and the glyph       */
/*     --        do not relate to the same face object.                  */
/*                                                                       */
/*     TT_Get_Outline( glyph, &outline );                                */
/*                                                                       */
/*     -- extracts the glyph outline from the object and copies it       */
/*     -- to the 'outline' record..                                      */
/*                                                                       */
/*     TT_Get_Metrics( glyph, &metrics );                                */
/*                                                                       */
/*     -- extracts the glyph metrics and put them into the 'metrics'     */
/*     -- record                                                         */
/*                                                                       */
/*     TT_Load_Glyph( ins2, glyph, 64, DEFAULT_GLYPH_LOAD );             */
/*                                                                       */
/*     -- load the same glyph at pointsize 12 in the 'glyph' object      */
/*                                                                       */
/*                                                                       */
/*     TT_Close_Face( &face );                                           */
/*                                                                       */
/*     -- destroys the face object. This will destroy 'ins1' and         */
/*     -- 'ins2'. However, the glyph object will still be available      */
/*                                                                       */
/*     TT_Done_FreeType();                                               */
/*                                                                       */
/*     -- Finalize the engine. This will also destroy all pending        */
/*     -- glyph objects (here 'glyph').                                  */
/*                                                                       */
/*                                                                       */

  struct _TFace;
  struct _TInstance;
  struct _TExecution_Context;
  struct _TGlyph;

  typedef struct _TFace  TFace, 
                        *PFace;

  typedef struct _TInstance  TInstance, 
                            *PInstance;

  typedef struct _TExecution_Context  TExecution_Context, 
                                     *PExecution_Context;

  typedef struct _TGlyph  TGlyph,
                         *PGlyph;

  /*************************************************************/
  /*                                                           */
  /*  ADDITIONAL SUBTABLES                                     */
  /*                                                           */
  /*  These tables are not precisely defined by the specs      */
  /*  but their structures is implied by the TrueType font     */
  /*  file layout..                                            */
  /*                                                           */
  /*************************************************************/

  /* Graphics State                            */
  /*                                           */
  /* The Graphics State (GS) is managed by the */
  /* instruction field, but does not come from */
  /* the font file. Thus, we can use 'int's    */
  /* where needed.                             */

  typedef struct  _TGraphicsState
  {
    Int         rp0;
    Int         rp1;
    Int         rp2;

    TT_UnitVector  dualVector;
    TT_UnitVector  projVector;
    TT_UnitVector  freeVector;

    Long        loop;
    TT_F26Dot6  minimum_distance;
    Int         round_state;

    Bool        auto_flip;
    TT_F26Dot6  control_value_cutin;
    TT_F26Dot6  single_width_cutin;
    TT_F26Dot6  single_width_value;
    Int         delta_base;
    Int         delta_shift;

    Byte        instruct_control;
    Bool        scan_control;
    Int         scan_type;

    Int         gep0;
    Int         gep1;
    Int         gep2;

  } TGraphicsState;

  extern const TGraphicsState  Default_GraphicsState;

  /*************************************************************/
  /*                                                           */
  /*  EXECUTION SUBTABLES                                      */
  /*                                                           */
  /*  These sub-tables relate to instruction execution         */
  /*                                                           */
  /*************************************************************/

  #define MAX_CODE_RANGES   3

  /* There can only be 3 active code ranges at once:   */
  /*   - the Font Program                              */
  /*   - the CVT Program                               */
  /*   - a glyph's instructions set                    */

  #define TT_CodeRange_Font  1
  #define TT_CodeRange_Cvt   2
  #define TT_CodeRange_Glyph 3

  typedef struct  _TCodeRange
  {
    PByte  Base;
    Int    Size;
  } TCodeRange;

  typedef TCodeRange*  PCodeRange;

  /* Defintion of a code range                                       */
  /*                                                                 */
  /* Code ranges can be resident to a glyph (i.e. the Font Program)  */
  /* while some others are volatile (Glyph instructions).            */
  /* Tracking the state and presence of code ranges allows function  */
  /* and instruction definitions within a code range to be forgotten */
  /* when the range is discarded.                                    */

  typedef TCodeRange  TCodeRangeTable[MAX_CODE_RANGES];

  /* defines a function/instruction definition record */

  typedef struct  _TDefRecord
  {
    Int   Range;      /* in which code range is it located ? */
    Int   Start;      /* where does it start ?               */
    Byte  Opc;        /* function #, or instruction code     */
    Bool  Active;     /* is it active ?                      */
  } TDefRecord;

  typedef TDefRecord*  PDefArray;

  /* defines a call record, used to manage function calls. */
  typedef struct  _TCallRecord
  {
    Int  Caller_Range;
    Int  Caller_IP;
    Int  Cur_Count;
    Int  Cur_Restart;
  } TCallRecord;

  /* defines a simple call stack */
  typedef TCallRecord*  PCallStack;

  
#ifndef TT_STATIC_INTEPRETER  /* indirect implementation */

  #define EXEC_OPS   PExecution_Context exc,
  #define EXEC_OP    PExecution_Context exc
  #define EXEC_ARGS  exc,
  #define EXEC_ARG   exc
  
#else                          /* static implementation */

  #define EXEC_OPS   /* void */
  #define EXEC_OP    /* void */
  #define EXEC_ARGS  /* void */
  #define EXEC_ARG   /* void */
  
#endif

  typedef  TT_F26Dot6 (*TRound_Function)( EXEC_OPS TT_F26Dot6 distance,
                                                   TT_F26Dot6 compensation );
  /* Rounding function, as used by the interpreter */
 
  typedef  void (*TMove_Function)( EXEC_OPS PVecRecord  zone,
                                            Int         point,
                                            TT_F26Dot6  distance );
  /* Point displacement along the freedom vector routine, as */
  /* used by the interpreter                                 */
 
  typedef  TT_F26Dot6 (*TProject_Function)( EXEC_OPS TT_F26Dot6 Vx,
                                                     TT_F26Dot6 Vy );
  /* Distance projection along one of the proj. vectors, as used */
  /* by the interpreter                                          */
  
  struct _TGlyph_Transform
  {
    TT_Fixed    xx, xy; /* transofrmation */
    TT_Fixed    yx, yy; /*     matrix     */
    TT_F26Dot6  ox, oy; /*    offsets     */
  };
  /* subglyph transformation record */

  typedef struct _TGlyph_Transform  TGlyph_Transform;
  typedef TGlyph_Transform         *PGlyph_Transform;

  struct _TSubglyph_Record
  {
    Int           index;      /* subglyph index */
    Bool          is_scaled;  /* is the subglyph scaled ? */
    Bool          is_hinted;  /* should it be hinted ?    */

    Int           xMin, xMax, yMin, yMax;

    Long          file_offset;

    Int           n_contours; /* number of contours                        */
    Int           n_points;   /* number of points (excluding phantom ones) */

    PCoordinates  org_x;
    PCoordinates  org_y;
    PCoordinates  cur_x;
    PCoordinates  cur_y;
    PTouchTable   flag;
    PUShort       contours;

    Int           arg1;  /* first argument  */
    Int           arg2;  /* second argument */

    Int           element_flag;  /* loaded element flag */

    TGlyph_Transform   transform;  /* transform */

    Int           leftSideBearing; /* in FUnits */
    Int           advanceWidth;    /* in FUnits */
  };
  /* subglyph loading record. Used to load composite components */

  typedef struct _TSubglyph_Record  TSubglyph_Record;
  typedef TSubglyph_Record*         PSubglyph_Record;
  typedef TSubglyph_Record*         PSubglyph_Stack;


  struct _TIns_Metrics
  {
    TT_F26Dot6   pointSize;      /* point size. 1 point = 1/72 inch. */

    Int          x_resolution;   /* device horizontal resolution in dpi. */
    Int          y_resolution;   /* device vertical   resolution in dpi. */

    Int          x_ppem;         /* horizontal pixels per EM */
    Int          y_ppem;         /* vertical   pixels per EM */

    Long         x_scale1;
    Long         x_scale2; /* used to scale FUnits to fractional pixels */

    Long         y_scale1;
    Long         y_scale2; /* used to scale FUnits to fractional pixels */

    TT_F26Dot6   compensations[4];  /* device-specific compensations */

    Bool         rotated;        /* is the glyph rotated ? flag   */
    Bool         stretched;      /* is the glyph stretched ? flag */
  };
  typedef struct _TIns_Metrics  TIns_Metrics;
  /* metrics used by the instance and execution context objects */

  /***********************************************************************/
  /*                                                                     */
  /*                         FreeType Face Type                          */
  /*                                                                     */
  /***********************************************************************/

  struct _TFace
  {
    PList_Element  element;
    /* list element node, used by the library to track */
    /* all opened faces, as well as recycled resident  */
    /* records..                                       */

    TT_Stream      stream;
    /* i/o stream */

    TMutex         lock;
    /* used only by the threaded builds of the library */

    TTTCHeader     ttcHeader;
    /* TrueType collection header, if any was found */

    TMaxProfile    maxProfile;
    /* maximum profile table, as found in the TrueType file */

    /* Note :                                         */
    /*  it seems that some maximum values cannot be   */
    /*  taken directly from this table, but rather by */
    /*  combining some of its fields ( e.g. the max.  */
    /*  number of points seems to be given by         */
    /*   MAX( maxPoints, maxCompositePoints )         */
    /*                                                */
    /*  For this reason, we define later our own      */
    /*  max values that are used to load and allocate */
    /*  further tables..                              */

    TT_Header             fontHeader;
    /* the font header, as found in the trueType file */

    TT_Horizontal_Header  horizontalHeader;
    /* the horizontal header */

    TT_OS2                os2;
    /* 'OS/2' table */

    TT_Postscript         postscript;
    /* 'Post' table */

    TName_Table       nameTable;
    /* name table */

    Int               numTables;
    PTableDirEntry    dirTables;
    /* The directory of TrueType tables for this typeface */

    Int          numCMaps;
    PCMapTable   cMaps;
    /* The directory of character mappings table for */
    /* this typeface                                 */

    Int       numLocations;
    PStorage  glyphLocations;
    /* The glyph locations table */

    PTableHorMetrics  longHMetrics;
    PShort            shortMetrics;
    /* The HMTX table data, used to compute both left */
    /* side bearing and advance width for all glyphs  */

    Int    fontPgmSize;
    PByte  fontProgram;
    /* the font program, if any */

    Int    cvtPgmSize;
    PByte  cvtProgram;
    /* the cvt program, if any */

    Int    cvtSize;
    PShort cvt;
    /* the original, unscaled, control value table */

    TGasp  gasp;
    /* the 'gasp' table */

    /* The following values _must_ be set by the */
    /* maximum progile loader                    */

    Int   numGlyphs;
    /* the face's total number of glyphs */

    Int   maxPoints;
    /* max glyph points number, simple and composite */

    Int   maxContours;
    /* max glyph contours number, simple and composite */

    Int   maxComponents;
    /* max components in a composite glyph */

    /* the following are object caches to track active */
    /* and recycled instances and execution contexts   */
    /* objects. See 'ttcache.h'                        */

    TCache  instances;
    /* current instances for this face */

    TCache  contexts;
    /* current execution contexts for this face */

    void*   extension;
    /* A typeless pointer to the face object extensions defined */
    /* in the 'c/lib/extend/ttext.*' files.                     */

    /* Use extensions to provide additional capabilities to the */
    /* engine. Read the developer's guide in the documentation  */
    /* dirctory to known how to do that..                       */
  };


  /***********************************************************************/
  /*                                                                     */
  /*                       FreeType Instance Type                        */
  /*                                                                     */
  /***********************************************************************/


  struct  _TInstance
  {
    PFace            owner;     /* face object */

    Bool             valid;

    TIns_Metrics     metrics;

    Int              numFDefs; /* number of function definitions */
    PDefArray        FDefs;    /* table of FDefs entries         */

    Int              numIDefs; /* number of instruction definitions */
    PDefArray        IDefs;    /* table if IDefs entries            */

    TCodeRangeTable  codeRangeTable;

    TGraphicsState   GS;
    TGraphicsState   default_GS;

    Int              cvtSize;
    PLong            cvt;
    /* the scaled control value table */

    /* debugging variables */

    Bool                debug;
    PExecution_Context  context;

    /* When using the debugger, we must keep the */
    /* execution context tied to the instance    */
    /* object rather than asking it on demand    */
  };

  /***********************************************************************/
  /*                                                                     */
  /*                  FreeType Execution Context Type                    */
  /*                                                                     */
  /***********************************************************************/

  struct _TExecution_Context 
  {
    PFace       owner;
    PInstance   instance;

    /* instructions state */
 
    Int          error;     /* last execution error */
  
    Int          curRange;  /* current code range number   */
    PByte        code;      /* current code range          */
    Int          IP;        /* current instruction pointer */
    Int          codeSize;  /* size of current range       */
  
    Byte         opcode;    /* current opcode              */
    Int          length;    /* length of current opcode    */
  
    Bool         step_ins;  /* true if the interpreter must */
                            /* increment IP after ins. exec */
  
    Int          numFDefs;  /* number of function defs */
    TDefRecord*  FDefs;     /* table of FDefs entries  */
  
    Int          numIDefs;  /* number of instruction defs */
    TDefRecord*  IDefs;     /* table of IDefs entries     */

    PByte        glyphIns;  /* glyph instructions buffer */
    Int          glyphSize; /* glyph instructions buffer size */
  
    Int          callTop,    /* top of call stack during execution */
                 callSize;   /* size of call stack */
    TCallRecord* callStack;  /* call stack */
  
    TCodeRangeTable codeRangeTable;  /* table of valid coderanges */
                                     /* useful for the debugger   */
  
    Int          storeSize;  /* size of current storage */
    PStorage     storage;    /* storage area            */
  
    Int          stackSize;  /* size of exec. stack */
    Int          top;        /* top of exec. stack  */
    PStorage     stack;      /* current exec. stack */
  
    Int          args,
                 new_top;    /* new top after exec.    */
  
    TT_F26Dot6   period;     /* values used for the */
    TT_F26Dot6   phase;      /* 'SuperRounding'     */
    TT_F26Dot6   threshold;

    TIns_Metrics metrics;       /* instance metrics */

    Int          cur_ppem;       /* ppem along the current proj vector */
    Long         scale1;         /* scaling values along the current   */
    Long         scale2;         /* projection vector too..            */
    Bool         cached_metrics; /* the ppem is computed lazily. used  */
                                  /* to trigger computation when needed */

    TVecRecord   zp0,  /* zone records */
                 zp1,
                 zp2,
                 pts,
                 twilight;
  
    Int          numContours; /* number of contours in current glyph */
    PUShort      endContours; /* array of contours' end-points       */
  
    Bool         instruction_trap;  /* If True, the interpreter will */
                                    /* exit after each instruction   */
 
    TGraphicsState  GS;          /* current graphics state */
 
    TGraphicsState  default_GS;  /* graphics state resulting from */
                                 /* the prep program              */
    Int          cvtSize;
    PLong        cvt;
  
    /* latest interpreter additions */
  
    Long         F_dot_P; /* dot product of freedom and projection vectors */
  
    TRound_Function   func_round;     /* current rounding function */
    
    TProject_Function func_project,   /* current projection function */
                      func_dualproj,  /* current dual proj. function */
                      func_freeProj;  /* current freedom proj. func  */
 
    TMove_Function    func_move;      /* current point move function */

    PSubglyph_Stack   loadStack;      /* loading subglyph stack */
  };

  /***********************************************************************/
  /*                                                                     */
  /*                  FreeType Glyph Object Type                         */
  /*                                                                     */
  /***********************************************************************/

  struct _TGlyph
  {
    PFace       face;

    TT_F26Dot6  xMin;  /* glyph bounding box */
    TT_F26Dot6  yMin;
    TT_F26Dot6  xMax;
    TT_F26Dot6  yMax;

    TT_F26Dot6  leftSideBearing;
    TT_F26Dot6  advanceWidth;

    Int         scan_type;
    Int         high_precision;

    Int           num_points;
    Int           num_contours;
    PCoordinates  x_coord;
    PCoordinates  y_coord;
    PTouchTable   touch;
    PUShort       endContours;
  };


  /* The following type is used to load a font from a collection */
  /* See Face_Create in ttobjs.c                                 */

  typedef struct _TFont_Input
  {
    TT_Stream  stream;     /* input stream                */
    Int        fontIndex;  /* index of font in collection */

  } TFont_Input;


  /********************************************************************/
  /*                                                                  */
  /*   Code Range Functions                                           */
  /*                                                                  */
  /********************************************************************/
 
  TT_Error  Goto_CodeRange( PExecution_Context exec, Int range, Int IP );
  /* Goto a specified coderange */
 
  PCodeRange  Get_CodeRange( PExecution_Context exec, Int range );
  /* Return a pointer to a given coderange record */
  /* Used only by the debugger                    */
 
  TT_Error  Set_CodeRange( PExecution_Context exec,
                           Int                range,
                           void*              base,
                           Int                length );
  /* Set a given code range properties */
 
  TT_Error  Clear_CodeRange( PExecution_Context exec, Int range );
  /* Clear a given coderange */




  PExecution_Context  New_Context( PFace  face );

  TT_Error  Done_Context( PExecution_Context  exec );


  TT_Error  Context_Load( PExecution_Context  exec,
                          PInstance           ins );

  TT_Error  Context_Save( PExecution_Context  exec,
                          PInstance           ins );

  TT_Error  Context_Run( PExecution_Context  exec,
                         Bool                debug );

  TT_Error  Instance_Init( PInstance  ins );

  TT_Error  Instance_Reset( PInstance  ins,
                            Bool       debug );
 
  /********************************************************************/
  /*                                                                  */
  /*   Component Initializer/Finalizer                                */
  /*                                                                  */
  /*   Called from 'freetype.c'                                       */
  /*   The component must create and register the face, instance and  */
  /*   execution context cache classes before any object can be       */
  /*   managed.                                                       */
  /*                                                                  */
  /********************************************************************/

  TT_Error  TTObjs_Init();
  TT_Error  TTObjs_Done();

  #ifdef __cplusplus
  }
  #endif

#endif /* TTOBJS_H */
