/*
 * Copyright 1996, 1997 Computing Research Labs, New Mexico State University
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE COMPUTING RESEARCH LAB OR NEW MEXICO STATE UNIVERSITY BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT
 * OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR
 * THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#ifndef lint
static char rcsid[] = "$Id: ttf2bdf.c,v 1.2 1997/10/01 18:24:55 mleisher Exp $";
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "freetype.h"

/*
 * This is needed to typecast the TT_Face field so the advance width
 * in the horizontal metrics can be used.
 */
#include "ttobjs.h"

/**************************************************************************
 *
 * Macros.
 *
 **************************************************************************/

/*
 * Set the default values used to generate a BDF font.
 */
#ifndef DEFAULT_PLATFORM_ID
#define DEFAULT_PLATFORM_ID 3
#endif

#ifndef DEFAULT_ENCODING_ID
#define DEFAULT_ENCODING_ID 1
#endif

#ifndef DEFAULT_POINT_SIZE
#define DEFAULT_POINT_SIZE 12
#endif

#ifndef DEFAULT_RESOLUTION
#define DEFAULT_RESOLUTION 100
#endif

/*
 * Used as a fallback for XLFD names where the character set/encoding can not
 * be determined.
 */
#ifndef DEFAULT_XLFD_CSET
#define DEFAULT_XLFD_CSET "-FontSpecific-0"
#endif

/*
 * nameID macros for getting strings from the TT font.
 */
#define TTF_COPYRIGHT 0
#define TTF_TYPEFACE  1
#define TTF_PSNAME    6

/**************************************************************************
 *
 * General globals set from command line.
 *
 **************************************************************************/

/*
 * The program name.
 */
static char *prog;

/*
 * The flag indicating whether messages should be printed or not.
 */
static int verbose = 0;

/*
 * Flags used when loading glyphs.
 */
static int load_flags = TTLOAD_SCALE_GLYPH | TTLOAD_HINT_GLYPH;

/*
 * The default platform and encoding ID's.
 */
static int pid = DEFAULT_PLATFORM_ID;
static int eid = DEFAULT_ENCODING_ID;

/*
 * Default point size and resolutions.
 */
static int point_size = DEFAULT_POINT_SIZE;
static int hres = DEFAULT_RESOLUTION;
static int vres = DEFAULT_RESOLUTION;

/*
 * The user supplied typeface name to use in the XLFD name.
 */
static char *face_name = 0;

/*
 * The user supplied weight name to use in the XLFD name.
 */
static char *weight_name = 0;

/*
 * The user supplied slant name to use in the XLFD name.
 */
static char *slant_name = 0;

/*
 * The user supplied spacing (p = proportional, c = character cell,
 * m = monospace).
 */
static int spacing = 0;

/**************************************************************************
 *
 * Internal globals.
 *
 **************************************************************************/

/*
 * The Units Per Em value used in numerous places.
 */
static TT_UShort upm;

/*
 * A flag indicating if a CMap was found or not.
 */
static TT_UShort nocmap;

/*
 * The scaling factor needed to compute the SWIDTH (scalable width) value
 * for BDF glyphs.
 */
static double swscale;

/*
 * Mac encoding names used when creating the BDF XLFD font name.
 */
static char *mac_encodings[] = {
    "-MacRoman-0",    "-MacJapanese-0",   "-MacChinese-0",   "-MacKorean-0",
    "-MacArabic-0",   "-MacHebrew-0",     "-MacGreek-0",     "-MacRussian-0",
    "-MacRSymbol-0",  "-MacDevanagari-0", "-MacGurmukhi-0",  "-MacGujarati-0",
    "-MacOriya-0",    "-MacBengali-0",    "-MacTamil-0",     "-MacTelugu-0",
    "-MacKannada-0",  "-MacMalayalam-0",  "-MacSinhalese-0", "-MacBurmese-0",
    "-MacKhmer-0",    "-MacThai-0",       "-MacLaotian-0",   "-MacGeorgian-0",
    "-MacArmenian-0", "-MacMaldivian-0",  "-MacTibetan-0",   "-MacMongolian-0",
    "-MacGeez-0",     "-MacSlavic-0",     "-MacVietnamese-0","-MacSindhi-0",
    "-MacUninterp-0"
};
static int num_mac_encodings = sizeof(mac_encodings) /
                               sizeof(mac_encodings[0]);

/*
 * ISO encoding names used when creating the BDF XLFD font name.
 */
static char *iso_encodings[] = {
    "-ASCII-0", "-ISO10646-0", "-ISO8859-1"
};
static int num_iso_encodings = sizeof(iso_encodings) /
                               sizeof(iso_encodings[0]);

/*
 * Microsoft encoding names used when creating the BDF XLFD font name.
 */
static char *ms_encodings[] = {
    "-Symbol-0", "-Unicode-2.0", "-ShiftJIS-0", "-GB2312.1980-0", "-Big5-0",
    "-KSC5601.1987-0", "-KSC5601.1992-0"
};
static int num_ms_encodings = sizeof(ms_encodings) /
                              sizeof(ms_encodings[0]);

/*
 * The propery names for all the XLFD properties.
 */
static char *xlfd_props[] = {
    "FOUNDRY",
    "FAMILY_NAME",
    "WEIGHT_NAME",
    "SLANT",
    "SETWIDTH_NAME",
    "ADD_STYLE_NAME",
    "PIXEL_SIZE",
    "POINT_SIZE",
    "RESOLUTION_X",
    "RESOLUTION_Y",
    "SPACING",
    "AVERAGE_WIDTH",
    "CHARSET_REGISTRY",
    "CHARSET_ENCODING",
};

/**************************************************************************
 *
 * Freetype globals.
 *
 **************************************************************************/

static TT_Face face;
static TT_Face_Properties properties;

static PFace facep;

static TT_Instance instance;

static TT_Glyph glyph;
static TT_Glyph_Metrics metrics;
static TT_Glyph_Outline outline;
static TT_Instance_Metrics imetrics;

static TT_Raster_Map raster;

static TT_CharMap cmap;

/**************************************************************************
 *
 * Freetype related code.
 *
 **************************************************************************/

/*
 * A generic routine to get a name from the TT name table.  This routine
 * always looks for English language names and checks three possibilities:
 * 1. English names with the MS Unicode encoding ID.
 * 2. English names with the MS unknown encoding ID.
 * 3. English names with the Apple Unicode encoding ID.
 *
 * The particular name ID mut be provided (e.g. nameID = 0 for copyright
 * string, nameID = 6 for Postscript name, nameID = 1 for typeface name.
 *
 * If the `dash_to_space' flag is set, all dashes (-) in the name will be
 * replaced with spaces.
 *
 * Returns the number of bytes added.
 */
static int
#ifdef __STDC__
ttf_get_english_name(char *name, int nameID, int dash_to_space)
#else
ttf_get_english_name(name, nameID, dash_to_space)
char *name;
int nameID, dash_to_space;
#endif
{
    int i, j, encid, slen, nrec;
    unsigned char *s;
    TNameRec *nr;

    nrec = facep->nameTable.numNameRecords;

    for (encid = 1, j = 0; j < 2; j++, encid--) {
        nr = facep->nameTable.names;
        /*
         * Locate one of the MS English font names.
         */
        for (i = 0; i < nrec; i++, nr++) {
            if (nr->platformID == 3 &&
                nr->encodingID == encid &&
                nr->nameID == nameID &&
                (nr->languageID == 0x0409 || nr->languageID == 0x0809 ||
                 nr->languageID == 0x0c09 || nr->languageID == 0x1009 ||
                 nr->languageID == 0x1409 || nr->languageID == 0x1809)) {
                s = nr->string;
                slen = nr->stringLength;
                break;
            }
        }

        if (i < nrec) {
            /*
             * Found one of the MS English font names.  The name is by
             * definition encoded in Unicode, so copy every second byte into
             * the `name' parameter, assuming there is enough space.
             */
            for (i = 1; i < slen; i += 2) {
                if (dash_to_space)
                  *name++ = (s[i] != '-') ? s[i] : ' ';
                else
                  *name++ = s[i];
            }
            *name = 0;
            return (slen >> 1);
        }
    }

    /*
     * No MS English name found, attempt to find an Apple Unicode English
     * name.
     */
    for (i = 0; i < nrec; i++, nr++) {
        if (nr->platformID == 0 && nr->languageID == 0 &&
            nr->nameID == nameID) {
            s = nr->string;
            slen = nr->stringLength;
            break;
        }
    }

    if (i < nrec) {
        /*
         * Found the Apple Unicode English name.  The name is by definition
         * encoded in Unicode, so copy every second byte into the `name'
         * parameter, assuming there is enough space.
         */
        for (i = 1; i < slen; i += 2) {
            if (dash_to_space)
              *name++ = (s[i] != '-') ? s[i] : ' ';
            else
              *name++ = s[i];
        }
        *name = 0;
        return (slen >> 1);
    }

    return 0;
}

/**************************************************************************
 *
 * General code.
 *
 **************************************************************************/

/*
 * Create an XLFD name.  Assumes there is enough space in the string passed
 * to fit a reasonably long XLFD name into, up to the 256 byte maximum.
 */
static void
#ifdef __STDC__
make_xlfd_name(char *name)
#else
make_xlfd_name(name)
char *name;
#endif
{
    TT_Long i;
    TT_UShort ismono, nhtmx, aw;
    TT_ULong val;
    double dr, dp;

    /*
     * Default the foundry name to "FreeType" in honor of the project and
     * because the foundry name is too difficult to automatically determine
     * from the names in TT fonts.
     */
    (void) strcpy(name, "-FreeType");
    name += 9;

    /*
     * Add the typeface name from the font.  The fallback default will be
     * "Unknown".
     */
    *name++ = '-';
    if ((i = ttf_get_english_name(name, TTF_TYPEFACE, 1)))
      name += i;
    else {
        if (face_name != 0) {
            (void) strcpy(name, face_name);
            name += strlen(face_name);
        } else {
            (void) strcpy(name, "Unknown");
            name += 7;
        }
    }

    /*
     * Add the weight name.  The default will be "Medium".
     */
    if (weight_name != 0) {
        sprintf(name, "-%s", weight_name);
        name += strlen(weight_name) + 1;
    } else {
        (void) strcpy(name, "-Medium");
        name += 7;
    }

    /*
     * Add the 
     */
    if (slant_name) {
        sprintf(name, "-%s", slant_name);
        name += strlen(slant_name) + 1;
    } else {
        *name++ = '-';
        *name++ = 'R';
    }

    /*
     * Default the setwidth name to "Normal".
     */
    (void) strcpy(name, "-Normal");
    name += 7;

    /*
     * Default the additional style name to NULL.
     */
    *name++ = '-';

    /*
     * Determine the pixel size from the point size and resolution.  Set the
     * global `swscale' factor at the same time so it can be used when the
     * glyph bitmaps are generated.
     */
    dr = (double) vres;
    dp = (double) (point_size * 10);
    val = (unsigned long) (((dp * dr) / 722.7) + 0.5);

    swscale = (dp * dr) / 10.0;

    /*
     * Set the pixel size, point size, and resolution.
     */
    sprintf(name, "-%ld-%d-%d-%d", val, point_size * 10, hres, vres);
    name += strlen(name);

    switch (spacing) {
      case 'p': case 'P': spacing = 'P'; break;
      case 'm': case 'M': spacing = 'M'; break;
      case 'c': case 'C': spacing = 'C'; break;
      default: spacing = 0; break;
    }
    /*
     * Go through the HTMX record and determine if the font is
     * monowidth and calculate an average width at the same time.
     * This average width is probably not exactly correct, but will
     * provide a number which can be used by programs that require
     * the pitch (characters per inch) of monowidth fonts.
     */
    ismono = 1;
    nhtmx = properties.horizontal->number_Of_HMetrics;
    for (val = 0, i = 0; i < properties.num_Glyphs; i++) {
        if (i < nhtmx)
          aw = facep->longHMetrics[i].advance_Width;
        else
          aw = facep->longHMetrics[nhtmx - 1].advance_Width;
        val += aw;

        /*
         * Check to see if this advance width is different than the
         * first one.  If so, then the font is proportional.  But
         * only do the check if the `ismono' flag is still set.
         */
        if (ismono && aw != facep->longHMetrics[0].advance_Width)
          ismono = 0;
    }

    /*
     * Adjust the average width and convert it to pixels.
     */
    aw = (TT_UShort) (val / properties.num_Glyphs);
    aw = ((aw * imetrics.x_ppem) / upm) * 10;

    /*
     * Set the spacing.
     */
    if (!spacing)
      spacing = (ismono) ? 'M' : 'P';
    *name++ = '-';
    *name++ = spacing;

    /*
     * Add the average width.
     */
    sprintf(name, "-%hd", aw);
    name += strlen(name);

    /*
     * If the cmap for the platform and encoding id was not found, or the
     * platform id is unknown, assume the character set registry and encoding
     * are the XLFD default.
     */
    if (nocmap || pid > 3)
      (void) strcpy(name, DEFAULT_XLFD_CSET);
    else {
        /*
         * Finally, determine the character set registry and encoding from the
         * platform and encoding ID.
         */
        switch (pid) {
          case 0:
            /*
             * Apple Unicode platform, so "Unicode-2.0" is the default.
             */
            (void) strcpy(name, "-Unicode-2.0");
            break;
          case 1:
            /*
             * Macintosh platform, so choose from the Macintosh encoding
             * strings.
             */
            if (eid < 0 || eid >= num_mac_encodings)
              (void) strcpy(name, DEFAULT_XLFD_CSET);
            else
              (void) strcpy(name, mac_encodings[eid]);
            break;
          case 2:
            /*
             * ISO platform, so choose from the ISO encoding strings.
             */
            if (eid < 0 || eid >= num_iso_encodings)
              (void) strcpy(name, DEFAULT_XLFD_CSET);
            else
              (void) strcpy(name, iso_encodings[eid]);
            break;
          case 3:
            /*
             * Microsoft platform, so choose from the MS encoding strings.
             */
            if (eid < 0 || eid >= num_ms_encodings)
              (void) strcpy(name, DEFAULT_XLFD_CSET);
            else
              (void) strcpy(name, ms_encodings[eid]);
            break;
        }
    }
}

static int
#ifdef __STDC__
generate_bitmaps(FILE *out, char *oname)
#else
generate_bitmaps(out, oname)
FILE *out;
char *oname;
#endif
{
    TT_Short maxx, maxy, minx, miny, xoff, yoff, dwidth, swidth;
    TT_UShort sx, sy, ex, ey;
    TT_Long code, idx, ng;
    unsigned char *bmap;
    double dw;

    /*
     * Calculate the font bounding box again so enough storage for the largest
     * bitmap can be allocated.
     */
    minx = (properties.header->xMin * imetrics.x_ppem) / upm;
    miny = (properties.header->yMin * imetrics.y_ppem) / upm;
    maxx = (properties.header->xMax * imetrics.x_ppem) / upm;
    maxy = (properties.header->yMax * imetrics.y_ppem) / upm;

    maxx -= minx;
    maxy -= miny;

    /*
     * Use the upward flow because the version of FreeType being used when
     * this was written did not support TT_Flow_Down.  This insures that this
     * routine will not mess up if TT_Flow_Down is implemented at some point.
     */
    raster.flow = TT_Flow_Up;
    raster.width = maxx;
    raster.rows = maxy;
    raster.cols = (maxx + 7) >> 3;
    raster.size = raster.cols * raster.rows;
    raster.bitmap = (void *) malloc(raster.size);

    for (ng = code = 0; ng < properties.num_Glyphs && code <= 0xffff; code++) {
        if (nocmap) {
            if (code >= properties.num_Glyphs)

              /*
               * At this point, all the glyphs are done.
               */
              break;
            idx = code;
        } else
          idx = TT_Char_Index(cmap, code);

        /*
         * If the glyph could not be loaded for some reason, just continue.
         */
        if (idx <= 0 || TT_Load_Glyph(instance, glyph, idx, load_flags))
          continue;

        (void) TT_Get_Glyph_Outline(glyph, &outline);
        (void) TT_Get_Glyph_Metrics(glyph, &metrics);

        /*
         * Clear the raster bitmap.
         */
        (void) memset((char *) raster.bitmap, 0, raster.size);

        /*
         * Determine the x and y offsets that will force the bitmap to
         * fit all the way onto the bitmap.
         */
        xoff = (metrics.xMin < 0) ? -metrics.xMin : 0;
        yoff = (metrics.yMin < 0) ? -metrics.yMin : 0;

        /*
         * If the bitmap cannot be generated, simply continue.
         */
        if (TT_Get_Glyph_Bitmap(glyph, &raster, xoff, yoff))
          continue;

        /*
         * Determine the DWIDTH (device width, or advance width in TT
         * terms) and the SWIDTH (scalable width) values.
         */
        dwidth = (metrics.advanceWidth >> 6) + 1;
        dw = (double) dwidth;
        swidth = (TT_Short) ((dw * 72000.0) / swscale);

        /*
         * Determine the actual bounding box of the glyph bitmap.  Do not
         * forget that the glyph is rendered upside down!
         */
        sx = ey = 0xffff;
        sy = ex = 0;
        bmap = (unsigned char *) raster.bitmap;
        for (miny = 0; miny < raster.rows; miny++) {
            for (minx = 0; minx < raster.width; minx++) {
                if (bmap[(miny * raster.cols) + (minx >> 3)] &
                    (0x80 >> (minx & 7))) {
                    if (minx < sx)
                      sx = minx;
                    if (minx > ex)
                      ex = minx;
                    if (miny > sy)
                      sy = miny;
                    if (miny < ey)
                      ey = miny;
                }
            }
        }

        /*
         * If the glyph is actually an empty bitmap, ignore it.
         */
        if (sx == 0xffff && ey == 0xffff && sy == 0 && ex == 0)
          sx = ex = sy = ey = 0;

        ng++;

        /*
         * Print the bitmap header.
         */
        fprintf(out, "STARTCHAR %04lX\nENCODING %ld\n", code, code);
        fprintf(out, "SWIDTH %hd 0\n", swidth);
        fprintf(out, "DWIDTH %hd 0\n", dwidth);
        fprintf(out, "BBX %hd %hd %ld %ld\n", (ex - sx) + 1, (sy - ey) + 1,
                (metrics.xMin * imetrics.x_ppem) / upm,
                (metrics.yMin * imetrics.y_ppem) / upm);
        fprintf(out, "BITMAP\n");

        /*
         * Now collect the bits so they can be printed.
         */
        for (miny = sy; miny >= ey; miny--) {
            for (idx = 0, minx = sx; minx <= ex; minx++) {
                if (minx > sx && ((minx - sx) & 7) == 0) {
                    /*
                     * Print the next byte.
                     */
                    fprintf(out, "%02lX", idx & 0xff);
                    idx = 0;
                }
                if (bmap[(miny * raster.cols) + (minx >> 3)] &
                    (0x80 >> (minx & 7)))
                  idx |= 0x80 >> ((minx - sx) & 7);
            }
            /*
             * Because of the structure of the loop, the last byte should
             * always be printed.
             */
            fprintf(out, "%02lX\n", idx & 0xff);
        }
        fprintf(out, "ENDCHAR\n");
    }

    /*
     * End the font and do memory cleanup on the glyph and raster structures.
     */
    fprintf(out, "ENDFONT\n");

    if (raster.size > 0)
      free((char *) raster.bitmap);

    return 0;
}

static int
#ifdef __STDC__
generate_bdf(FILE *out, char *iname, char *oname)
#else
generate_bdf(out, iname, oname)
FILE *out;
char *iname, *oname;
#endif
{
    TT_Long i;
    TT_Short maxx, maxy, minx, miny, p, e;
    char *xp, xlfd[256];

    /*
     * Get the requested cmap.
     */
    for (i = 0; i < TT_Get_CharMap_Count(face); i++) {
        if (!TT_Get_CharMap_ID(face, i, &p, &e) &&
            p == pid && e == eid)
          break;
    }
    if (i == TT_Get_CharMap_Count(face) && pid == 3 && eid == 1) {
        /*
         * Make a special case when this fails with pid == 3 and eid == 1.
         * Change to eid == 0 and try again.  This captures the two possible
         * cases for MS fonts.  Some other method should be used to cycle
         * through all the alternatives later.
         */
        for (i = 0; i < TT_Get_CharMap_Count(face); i++) {
            if (!TT_Get_CharMap_ID(face, i, &p, &e) &&
                p == pid && e == 0)
              break;
        }
        if (i < TT_Get_CharMap_Count(face)) {
            if (!TT_Get_CharMap(face, i, &cmap))
              eid = 0;
            else
              nocmap = 1;
        }
    } else {
        /*
         * A CMap was found for the platform and encoding IDs.
         */
        if (i < TT_Get_CharMap_Count(face) && TT_Get_CharMap(face, i, &cmap))
          nocmap = 1;
        else
          nocmap = 0;
    }

    if (nocmap && verbose) {
        fprintf(stderr,
                    "%s: no character map for platform %d encoding %d.  ",
                    prog, pid, eid);
        fprintf(stderr, "Generating all glyphs.\n");
    }

    /*
     * Generate the XLFD name.
     */
    make_xlfd_name(xlfd);

    fprintf(out, "STARTFONT 2.1\n");

    /*
     * Add the vanity comments.
     */
    fprintf(out, "COMMENT\n");
    fprintf(out, "COMMENT Converted from TrueType font \"%s\" by \"%s\".\n",
            iname, prog);
    fprintf(out, "COMMENT\n");

    fprintf(out, "FONT %s\n", xlfd);
    fprintf(out, "SIZE %d %d %d\n", point_size, hres, vres);

    /*
     * Calculate the font bounding box.
     */
    minx = (properties.header->xMin * imetrics.x_ppem) / upm;
    miny = (properties.header->yMin * imetrics.y_ppem) / upm;
    maxx = (properties.header->xMax * imetrics.x_ppem) / upm;
    maxy = (properties.header->yMax * imetrics.y_ppem) / upm;

    fprintf(out, "FONTBOUNDINGBOX %hd %hd %hd %hd\n",
            maxx - minx, maxy - miny, minx, miny);

    /*
     * Determine the font ascent and descent for the properties.
     */
    maxy = (properties.horizontal->Ascender * imetrics.y_ppem) / upm;
    miny = (properties.horizontal->Descender * imetrics.y_ppem) / upm;

    fprintf(out, "STARTPROPERTIES 19\n");

    /*
     * Print all the properties from the XLFD name first.
     */
    for (maxx = 0, xp = xlfd; maxx < 14; maxx++) {
        /*
         * Print the XLFD property name.
         */
        fprintf(out, "%s ", xlfd_props[maxx]);

        /*
         * Make sure the ATOM properties are wrapped in double quotes.
         */
        if (maxx < 6 || maxx == 10 || maxx > 11)
          putc('"', out);

        /*
         * Skip the leading '-' in the XLFD name.
         */
        xp++;

        /*
         * Skip until the next '-' or NULL.
         */
        for (; *xp && *xp != '-'; xp++)
          putc(*xp, out);

        /*
         * Make sure the ATOM properties are wrapped in double quotes.
         */
        if (maxx < 6 || maxx == 10 || maxx > 11)
          putc('"', out);

        putc('\n', out);
    }

    /*
     * Make sure to add the FONT_ASCENT and FONT_DESCENT properties
     * because X11 can not live without them.
     */
    fprintf(out, "FONT_ASCENT %hd\nFONT_DESCENT %hd\n", maxy, -miny);

    /*
     * Get the copyright string from the font.
     */
    (void) ttf_get_english_name(xlfd, TTF_COPYRIGHT, 0);
    fprintf(out, "COPYRIGHT \"%s\"\n", xlfd);

    /*
     * Last, print the two user-defined properties _TTF_FONTFILE and
     * _TTF_PSNAME.  _TTF_FONTFILE provides a reference to the original TT
     * font file which some systems can take advantage of, and _TTF_PSNAME
     * provides the Postscript name of the font if it exists.
     */
    (void) ttf_get_english_name(xlfd, TTF_PSNAME, FALSE);
    fprintf(out, "_TTF_FONTFILE \"%s\"\n_TTF_PSNAME \"%s\"\n", iname, xlfd);

    fprintf(out, "ENDPROPERTIES\nCHARS %d\n", properties.num_Glyphs);

    /*
     * Now go through and generate the glyph bitmaps themselves.
     */
    return generate_bitmaps(out, oname);
}

static void
#ifdef __STDC__
usage(int eval)
#else
usage(eval)
int eval;
#endif
{
    fprintf(stderr, "Usage: %s [options below] font.ttf\n", prog);
    fprintf(stderr, "-h\t\tThis message.\n");
    fprintf(stderr, "-v\t\tPrint warning messages during conversion.\n");
    fprintf(stderr, "-n\t\tTurn off glyph hinting.\n");
    fprintf(stderr,
            "-c c\t\tSet the character spacing (default: from font).\n");
    fprintf(stderr,
            "-t name\t\tSet the typeface name (default: from font).\n");
    fprintf(stderr, "-w name\t\tSet the weight name (default: Medium).\n");
    fprintf(stderr, "-s name\t\tSet the slant name (default: R).\n");
    fprintf(stderr,
            "-pid id\t\tSet the platform ID for encoding (default: %d).\n",
            DEFAULT_PLATFORM_ID);
    fprintf(stderr,
            "-eid id\t\tSet the encoding ID for encoding (default: %d).\n",
            DEFAULT_ENCODING_ID);
    fprintf(stderr, "-p n\t\tSet the point size (default: %dpt).\n",
            DEFAULT_POINT_SIZE);
    fprintf(stderr, "-r n\t\tSet the horizontal and vertical resolution ");
    fprintf(stderr, "(default: %ddpi).\n", DEFAULT_RESOLUTION);
    fprintf(stderr, "-rh n\t\tSet the horizontal resolution ");
    fprintf(stderr, "(default: %ddpi)\n", DEFAULT_RESOLUTION);
    fprintf(stderr, "-rv n\t\tSet the vertical resolution ");
    fprintf(stderr, "(default: %ddpi)\n", DEFAULT_RESOLUTION);
    fprintf(stderr,
            "-o outfile\tSet the output filename (default: stdout).\n");
    exit(eval);
}

void
#ifdef __STDC__
main(int argc, char *argv[])
#else
main(argc, argv)
int argc;
char *argv[];
#endif
{
    int res;
    char *infile, *outfile, *iname, *oname;
    FILE *out;

    if ((prog = strrchr(argv[0], '/')))
      prog++;
    else
      prog = argv[0];

    out = stdout;
    infile = outfile = 0;

    argc--;
    argv++;

    while (argc > 0) {
        if (argv[0][0] == '-') {
            switch (argv[0][1]) {
              case 'v': case 'V':
                verbose = 1;
                break;
              case 'n': case 'N':
                load_flags &= ~TTLOAD_HINT_GLYPH;
                break;
              case 'c': case 'C':
                argc--;
                argv++;
                spacing = argv[0][0];
                break;
              case 't': case 'T':
                argc--;
                argv++;
                face_name = argv[0];
                break;
              case 'w': case 'W':
                argc--;
                argv++;
                weight_name = argv[0];
                break;
              case 's': case 'S':
                argc--;
                argv++;
                slant_name = argv[0];
                break;
              case 'p': case 'P':
                res = argv[0][2];
                argc--;
                argv++;
                if (res == 'i' || res == 'I')
                  /*
                   * Set the platform ID.
                   */
                  pid = atoi(argv[0]);
                else
                  /*
                   * Set the point size.
                   */
                  point_size = atoi(argv[0]);
                break;
              case 'e': case 'E':
                /*
                 * Set the encoding ID.
                 */
                argc--;
                argv++;
                eid = atoi(argv[0]);
                break;
              case 'r':
                /*
                 * Set the horizontal and vertical resolutions.
                 */
                if (argv[0][2] == 'h')
                  hres = atoi(argv[1]);
                else if (argv[0][2] == 'v')
                  vres = atoi(argv[1]);
                else
                  hres = vres = atoi(argv[1]);
                argc--;
                argv++;
                break;
              case 'o':
                /*
                 * Set the output file name.
                 */
                argc--;
                argv++;
                outfile = argv[0];
                break;
              default:
                usage(1);
            }
        } else
          /*
           * Set the input file name.
           */
          infile = argv[0];

        argc--;
        argv++;
    }

    /*
     * Validate the values passed on the command line.
     */
    if (infile == 0) {
        fprintf(stderr, "%s: no input file provided.\n", prog);
        usage(1);
    } else {
        /*
         * Set the input filename that will be passed to the generator
         * routine.
         */
        if ((iname = strrchr(infile, '/')))
          iname++;
        else
          iname = infile;
    }

    /*
     * Check the platform and encoding IDs.
     */
    if (pid < 0 || pid > 255) {
        fprintf(stderr, "%s: invalid platform ID '%d'.\n", prog, pid);
        exit(1);
    }
    if (eid < 0 || eid > 65535) {
        fprintf(stderr, "%s: invalid encoding ID '%d'.\n", prog, eid);
        exit(1);
    }

    /*
     * Arbitrarily limit the point size to a minimum of 7pt and maximum of
     * 256pt.
     */
    if (point_size < 7 || point_size > 256) {
        fprintf(stderr, "%s: invalid point size '%dpt'.\n", prog, point_size);
        exit(1);
    }

    /*
     * Arbitrarily limit the resolutions to a minimum of 50dpi and a maximum
     * of 1200dpi.
     */
    if (hres < 50 || hres > 1200) {
        fprintf(stderr, "%s: invalid horizontal resolution '%ddpi'.\n",
                prog, hres);
        exit(1);
    }
    if (vres < 50 || vres > 1200) {
        fprintf(stderr, "%s: invalid vertical resolution '%ddpi'.\n",
                prog, vres);
        exit(1);
    }

    /*
     * Open the output file if specified.
     */
    if (outfile != 0) {
        /*
         * Attempt to open the output file.
         */
        if ((out = fopen(outfile, "w")) == 0) {
            fprintf(stderr, "%s: unable to open the output file '%s'.\n",
                    prog, outfile);
            exit(1);
        }
        /*
         * Set the output filename to be passed to the generator routine.
         */
        if ((oname = strrchr(outfile, '/')))
          oname++;
        else
          oname = outfile;
    } else
      /*
       * Set the default output file name to <stdout>.
       */
      oname = "<stdout>";

    /*
     * Intialize Freetype.
     */
    if ((res = TT_Init_FreeType())) {
        /*
         * Close the output file.
         */
        if (out != stdout) {
            fclose(out);
            (void) unlink(outfile);
        }
        fprintf(stderr, "%s[%d]: unable to initialize renderer.\n",
                prog, res);
        exit(1);
    }

    /*
     * Open the input file.
     */
    if ((res = TT_Open_Face(infile, &face))) {
        if (out != stdout) {
            fclose(out);
            (void) unlink(outfile);
        }
        fprintf(stderr, "%s[%d]: unable to open input file '%s'.\n",
                prog, res, infile);
        exit(1);
    }

    /*
     * Create a new instance.
     */
    if ((res = TT_New_Instance(face, &instance))) {
        (void) TT_Close_Face(face);
        if (out != stdout) {
            fclose(out);
            (void) unlink(outfile);
        }
        fprintf(stderr, "%s[%d]: unable to create instance.\n",
                prog, res);
        exit(1);
    }

    /*
     * Set the instance resolution and point size and the relevant
     * metrics.
     */
    (void) TT_Set_Instance_Resolution(instance, hres, vres);
    (void) TT_Set_Instance_PointSize(instance, point_size);
    (void) TT_Get_Instance_Metrics(instance, &imetrics);

    /*
     * Get the face properties and set the global units per em value for
     * convenience.
     */
    (void) TT_Get_Face_Properties(face, &properties);
    upm = properties.header->Units_Per_EM;

    /*
     * Create a new glyph container.
     */
    if ((res = TT_New_Glyph(face, &glyph))) {
        (void) TT_Done_Instance(instance);
        (void) TT_Close_Face(face);
        if (out != stdout) {
            fclose(out);
            (void) unlink(outfile);
        }
        fprintf(stderr, "%s[%d]: unable to create glyph.\n",
                prog, res);
        exit(1);
    }

    /*
     * Typecast a pointer to the face record because it is used in a number of
     * places.
     */
    facep = (PFace) face.z;

    /*
     * Generate the BDF font from the TrueType font.
     */
    res = generate_bdf(out, iname, oname);

    /*
     * Close the input and output files.
     */
    (void) TT_Close_Face(face);
    if (out != stdout) {
        fclose(out);
        if (res != 0)
          /*
           * An error occured when generating the font, so delete the
           * output file.
           */
          (void) unlink(outfile);
    }

    /*
     * Shut down the renderer.
     */
    (void) TT_Done_FreeType();

    exit(res);
}
