/* */

/* Constants */
#define DEFDIV 96	/* Default divisor (if no valid Mthd found) */
#define MAXDIVIDER	8192	// Maximum divider

#define UTIMEFMASK	0x1FFF	// Fractions mask (13 bits)
#define UTIMEBSHIFT	13	// Beats shift    (5 bits)
#define UTIMEBMASK	0x1F	// Beats mask
#define UTIMEMSHIFT	18	// Measures shift (14 bits)

#define UTIME(m,b,f)	(((m)<<UTIMEMSHIFT)|(((b)&UTIMEBMASK)<<UTIMEBSHIFT)|((f)&UTIMEFMASK))
#define UTIMEM(u)	((u)>>UTIMEMSHIFT)
#define UTIMEB(u)	(((u)>>UTIMEBSHIFT)&UTIMEBMASK)
#define UTIMEF(u)	((u)&UTIMEFMASK)

/* Event header. Event data must follow */
typedef struct {
  unsigned long index;
  unsigned long flags;		/* User flags */
  unsigned long mtime;		/* MIDI time */
  unsigned long rtime;		/* Real time or -1 if unknown */
  unsigned long utime;		/* Musical time */
  short track;			/* Track No */
  void *pprev;			/* previous event data or NULL */
  void *pnext;			/* next event data or NULL */
  void *linked;			// Linked event
  int port;
  int tbank;
  int dbank;
  unsigned long len;		// Event data length
  unsigned char status;		/* Status byte */
  unsigned char type;		// Type for metaevents
  unsigned char data[0];
  } EVENTHDR;

/* Unknown chunk header. A chunk must follow */
typedef struct {
  void *pprev;			// previous chunk or NULL
  void *pnext;			/* next chunk or NULL */
  unsigned long len;		/* Data length */
  unsigned char data[0];
  } UNKHDR;

/* Track data */
typedef struct {
   int flags;
   int mute;
   int port;			// Used only by MIDIIndex
   int tbank;
   int dbank;
   } TRACKDATA;

/* Parsed MIDI data format */
typedef struct {
  UNKHDR *pufirst;	  /* First unknown chunk header */
  UNKHDR *pulast;		  /* Last unknown chunk header */

  short fmt;		  /* 0/1/2 */
  short trks;		  /* Number of tracks */
  short div;		  /* Division */
  unsigned long	events;		  /* Total number of events */

  unsigned long warns;		  /* Warning flags */

  EVENTHDR *pfirst;		  /* First event header */
  EVENTHDR *plast;		  /* Last event header */
  EVENTHDR **index;		 // MIDI index
  unsigned long flags;		// User flags


  TRACKDATA *trackdata;		// Ptr to track data array
  } MIDIDATA;

// Packed MIDI data header
typedef struct {
   short fmt;
   short trks;			// Somehow check no of tracks / channels ?
   short div;
   unsigned long events;
   unsigned long size;		// Current size
   unsigned long allocated;	// Currenlty allocated when creating data
				// Current ptr when getting data
   } PKMIDIDATA;

// Packed MIDI event
typedef struct {
   unsigned long mtime;		//
   short track;			//
   unsigned long len;		// Event data length
   unsigned char status;	// Status byte 
   unsigned char type;		// Type for metaevents
   } PKEVENT;

/* Error codes */
#define ME_OK		0	/* Ok */
#define ME_NODATA	1	/* No data found */


/* Warnings (for MIDIData.warns) */
#define MW_BADTEMPO	1	/* Bad tempo meta event */
#define MW_BADMTHD	2	/* Bad MThd */
#define MW_BADFMT0	4	/* >1 track for fmt 0 */
#define MW_BADSTATUS	8	/* Status byte F1-FE */
#define MW_BADRSTATUS	0x10	/* Illegal running status */
#define MW_BADMTRK	0x20	/* Bad MTrk */
#define MW_BADCHUNK	0x40	/* Invalid chunk length */
#define MW_NOMTHD	0x80	/* No MThd chunk */
#define MW_BADTRKCNT	0x100	/* Number of tracks mismatch */
#define MW_EXTRAMTHD	0x200	/* more than 1 MThd in a file */
#define MW_BADTIMESIGN	0x400	/* Bad time signature */

#define PLAY_EVENT	1	// EVENTHDR flag to show the event marked
				// for playing by MIDIRescan

/* Prototypes */

/* Parse MIDI data from *File buffer Size bytes long into *MIDIData */
int MIDIParse(unsigned char *File, unsigned long Size, MIDIDATA *MIDIData);
/* Sort MIDI data by time */
void MIDISort(MIDIDATA *MIDIData);
/* Resolve MIDI time into real time (MIDIData must be sorted by time) */
int MIDITime(MIDIDATA *MIDIData);
/* Free all memory associated with MIDIData */
void MIDIDrop(MIDIDATA *MIDIData);

/* Creates MIDI index */
void MIDIIndex(MIDIDATA *MIDIData);

/* Return tempo at the specified event */
unsigned long MIDIGetTempo(MIDIDATA *MIDIData, EVENTHDR *hp);

// Return Time Sign at the specified event
void MIDIGetTimeSign(MIDIDATA *MIDIData, EVENTHDR *eh, unsigned int *num, unsigned int *denom);

/* Replace *Old MIDI event with *New. Free *Old */
void MIDIReplace(MIDIDATA *MIDIData,EVENTHDR *Old,EVENTHDR *New);

/* Remove *Event, free *Event */
void MIDIRemove(MIDIDATA *MIDIData,EVENTHDR *Event);

/* Find an event preceeeding the specified MIDI time */
EVENTHDR *MIDIPrevM(MIDIDATA *MIDIData, unsigned long MTime);

/* Convert MIDI time into real time (Basic version)
   hp - event before the MTime */
unsigned long MIDIbMtoR(MIDIDATA *MIDIData, EVENTHDR *hp,unsigned long MTime);

/* Convert MIDI time into real time (Full version ) */
unsigned long MIDIMtoR(MIDIDATA *MIDIData, unsigned long MTime);

/* Convert MIDI time into musical time (Basic version)
   hp - event before the MTime */
unsigned long MIDIbMtoU(MIDIDATA *MIDIData, EVENTHDR *hp,unsigned long MTime);

/* Convert MIDI time into musical time (Full version ) */
unsigned long MIDIMtoU(MIDIDATA *MIDIData, unsigned long MTime);

/* Find an event preceeeding the specified real time */
EVENTHDR *MIDIPrevR(MIDIDATA *MIDIData, unsigned long RTime);

/* Convert Real time into MIDI time (basic version)
   hp - event before the RTime */
unsigned long MIDIbRtoM(MIDIDATA *MIDIData, EVENTHDR *hp,unsigned long RTime);

/* Convert Real time into MIDI time (Full version) */
unsigned long MIDIRtoM(MIDIDATA *MIDIData, unsigned long RTime);

/* Find an event preceeeding the specified musical time */
EVENTHDR *MIDIPrevU(MIDIDATA *MIDIData, unsigned long UTime);

/* Convert musical time into MIDI time (basic version)
   hp - event before the musical */
unsigned long MIDIbUtoM(MIDIDATA *MIDIData, EVENTHDR *hp,unsigned long UTime);

/* Convert musical time into MIDI time (Full version) */
unsigned long MIDIUtoM(MIDIDATA *MIDIData, unsigned long UTime);

// Same as UTIME macro, but parms may be denormalized. UNFINISHED !!!
unsigned long MIDIUNorm(MIDIDATA *MIDIData, unsigned int Meas, unsigned int Beat, unsigned int Fr);

/* Insert event after hp */
void MIDIInsert(MIDIDATA *MIDIData,EVENTHDR *hp,EVENTHDR *Event);

/* Insert event (by MIDI Time), calculate real time and musical time */
void MIDIInsertM(MIDIDATA *MIDIData,EVENTHDR *Event);

/* Unparse MIDIData into a MIDI file image. Return file size. Required memory
   is allocated and ptr is returned in Buffer. */
unsigned long MIDIUnparse(MIDIDATA *MIDIData,char **Buffer);

// Create an event (do not include it in any list)
EVENTHDR *MIDINewEvent(unsigned long len);

// Resize an event (so that it will fit len bytes of data)
EVENTHDR *MIDIResizeEvent(EVENTHDR *eh,unsigned long len);

// Remove event itself (no link upates, etc)
void MIDIFreeEvent(EVENTHDR *eh);

//Insert n tracks at Pos.
void MIDIInsertTrks(MIDIDATA *MIDIData, int Pos, int n);

//Delete n tracks at Pos.
void MIDIDeleteTrks(MIDIDATA *MIDIData, int Pos, int n);


/* To add - musical time arithmetics:

unsigned long MIDIUAdd(MIDIDATA *MIDIData, unsigned long UBase,
                       unsigned int Meas, unsigned int Beat, unsigned int Fr);
unsigned long MIDIUSub(MIDIDATA *MIDIData, unsigned long UMinuend,unsigned long USubtrahend);

  */

// Start a new packed MIDI data. Returns ptr to packed data
PKMIDIDATA *MIDIPackedNew(int fmt, int trks, int div);

// Add an event to packed MIDI data
PKMIDIDATA *MIDIPackedAdd(PKMIDIDATA *p, EVENTHDR *eh, unsigned long mtime, int track);

// Finish creating packed MIDI data (realloc)
PKMIDIDATA *MIDIPackedDone(PKMIDIDATA *p);

// Drop packed MIDI data
void MIDIPackedDrop(PKMIDIDATA *p);

// Start getting events from packed MIDI data
void MIDIPackedStart(PKMIDIDATA *p);

// Get an event from packed MIDI data, returns NULL if no more data
EVENTHDR *MIDIPackedGet(PKMIDIDATA *p);

// Rescan MIDIData and mark events to be played with PLAY_EVENT flag
// Return FALSE if no events to be played after mtime are found
int MIDIRescan(MIDIDATA *MIDIData, unsigned long mtime);
