/*******************************************************************************
* FILE NAME: iframe.cpp                                                        *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in iframe.hpp.                                                             *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1995       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
// Priority INT_MIN (-2147483647 - 1) + 1024 + 512
#pragma priority( -2147482112 )
extern "C"
  {
  #define INCL_WINFRAMEMGR
  #define INCL_WINWINDOWMGR
  #define INCL_WINSWITCHLIST
  #define INCL_WINDIALOGS
  #define INCL_WINSYS
  #define INCL_WININPUT
  #define INCL_NLS
  #define INCL_DOSNLS
  #define INCL_WINERRORS
  #define INCL_SHLERRORS
  #define INCL_WINPOINTERS   // For WinSetPointer, WinQueryPointerPos.
  #include <iwindefs.h>
  }

#include <iframe.hpp>
#include <icconst.h>
#include <icolor.hpp>
#include <icoordsy.hpp>
#include <iexcept.hpp>
#include <iframext.hpp>
#include <iframhdr.hpp>
#include <iframprv.hpp>
#include <imenuprv.hpp>
#include <iprocadr.hpp>
#include <irecohdr.hpp>
#include <irect.hpp>
#include <istring.hpp>
#include <ithread.hpp>
#include <ititle.hpp>
#include <itrace.hpp>


// Segment definitions.
#ifdef IC_PAGETUNE
  #define _IFRAME_CPP_
  #include <ipagetun.h>
#endif


/*------------------------------------------------------------------------------
| Static style definitions.                                                    |
|                                                                              |
| These values exploit the specific values used for the PM toolkit             |
| WS_* and FS_* values.  Frame creation must support the "flags"               |
| corresponding to the WS_* values (except for certain flags that              |
| don't apply to frames), the FS_* values (except for certain flags that       |
| only apply to dialogs loaded from resource templates), and additional        |
| "attributes" corresponding to the particular FCF_* values for which there    |
| is no corresponding FS_* value.                                              |
|                                                                              |
| The FS_ and WS_ flags are placed in the lower DWORD, whereas the FCF_        |
| flags are placed in the low order word of the upper DWORD.                   |
| to their normal values.  All of the flags are documented below for clarity.  |
------------------------------------------------------------------------------*/
#pragma data_seg(ICLStaticConst)
const IFrameWindow::Style
#ifdef IC_WIN
  IFrameWindow::deferCreation      ( 0, IFS_DEFERCREATE ), // 0x00000100
  IFrameWindow::accelerator        ( 0, IFS_ACCELERATOR ), // 0x00000200
  IFrameWindow::menuBar            ( 0, IFS_MENU ),        // 0x00000400
  IFrameWindow::minimizedIcon      ( 0, IFS_MINICON ),     // 0x00000800
  IFrameWindow::alignNoAdjust      ( 0, IFS_ALIGNNOADJ ),  // 0x00001000
  IFrameWindow::shellPosition      ( 0, IFS_SHELLPOS ),    // 0x00002000

// May not be able to implement the below styles!
  IFrameWindow::animated           ( 0, IFS_ANIMATED ),    // 0x00000000
  IFrameWindow::appDBCSStatus      ( 0, IFS_APPDBCS ),     // 0x00000000
  IFrameWindow::dialogBackground   ( 0, IFS_DLGBACKGND ),  // 0x00000000
  IFrameWindow::hideButton         ( 0, IFS_HIDEBUTTON ),  // 0x00000000
  IFrameWindow::noMoveWithOwner    ( 0, IFS_NOMOVEOWNER ), // 0x00000000
  IFrameWindow::systemModal        ( 0, IFS_SYSMODAL ),    // 0x00000000
  IFrameWindow::windowList         ( 0, IFS_WINDOWLIST ),  // 0x00000000

  IFrameWindow::titleBar            = WS_CAPTION,          // 0x00C00000
  IFrameWindow::maximizeButton      = WS_MAXIMIZEBOX,      // 0x00010000
  IFrameWindow::minimizeButton      = WS_MINIMIZEBOX,      // 0x00020000
  IFrameWindow::sizingBorder        = WS_THICKFRAME,       // 0x00040000
  IFrameWindow::systemMenu          = WS_SYSMENU,          // 0x00080000
  IFrameWindow::horizontalScroll    = WS_HSCROLL,          // 0x00100000
  IFrameWindow::verticalScroll      = WS_VSCROLL,          // 0x00200000
  IFrameWindow::dialogBorder        = WS_DLGFRAME,         // 0x00400000 or WS_EX_DLGMODALFRAME?
  IFrameWindow::border              = WS_BORDER,           // 0x00800000
  IFrameWindow::maximized           = WS_MAXIMIZE,         // 0x01000000
                                   // WS_CLIPCHILDREN         0x02000000
                                   // WS_CLIPSIBLINGS         0x04000000
                                   // WS_DISABLED             0x08000000
                                   // WS_VISIBLE              0x10000000
  IFrameWindow::minimized           = WS_MINIMIZE,         // 0x20000000


  IFrameWindow::classDefaultStyle  ( 0x00CF0000ul, 0x00000000ul ),
                                  // 0x00C00000 titleBar
                                  // 0x00080000 systemMenu
                                  // 0x00020000 minimizeButton
                                  // 0x00010000 maximizeButton
                                  // 0x00040000 sizingBorder

  // styleBits is mask with just the style bits "on":
  IFrameWindow::styleBits           = 0xFFFF0000ul;       // 0x00000000
#else   // !IC_WIN
  IFrameWindow::minimizedIcon      = FS_ICON,                 // 0x00000001
  IFrameWindow::accelerator        = FS_ACCELTABLE,           // 0x00000002
  IFrameWindow::shellPosition      = FS_SHELLPOSITION,        // 0x00000004
  IFrameWindow::windowList         = FS_TASKLIST,             // 0x00000008
  IFrameWindow::alignNoAdjust      = FS_NOBYTEALIGN,          // 0x00000010
  IFrameWindow::noMoveWithOwner    = FS_NOMOVEWITHOWNER,      // 0x00000020
  IFrameWindow::systemModal        = FS_SYSMODAL,             // 0x00000040
  IFrameWindow::dialogBorder       = FS_DLGBORDER,            // 0x00000080
  IFrameWindow::border             = FS_BORDER,               // 0x00000100
                                  // Reserved                    0x00000200
                                  // Reserved                    0x00000400
  IFrameWindow::sizingBorder       = FS_SIZEBORDER,           // 0x00000800
  IFrameWindow::dialogBackground   = FS_AUTOICON,             // 0x00001000
                                  // Reserved                    0x00002000
                                  // Reserved                    0x00004000
  IFrameWindow::appDBCSStatus      = FS_DBE_APPSTAT,          // 0x00008000
                                  // Reserved                    0x00010000
                                  // Reserved                    0x00020000
                                  // Reserved                    0x00040000
                                  // Reserved                    0x00080000
                                  // Reserved                    0x00100000
                                  // Reserved                    0x00200000
  IFrameWindow::animated           = WS_ANIMATE,              // 0x00400000
  IFrameWindow::maximized          = WS_MAXIMIZED,            // 0x00800000
  IFrameWindow::minimized          = WS_MINIMIZED,            // 0x01000000
                                  // WS_SYNCPAINT                0x02000000
                                  // WS_SAVEBITS                 0x04000000
                                  // WS_PARENTCLIP               0x08000000
                                  // WS_CLIPSIBLINGS             0x10000000
                                  // WS_CLIPCHILDREN             0x20000000
                                  // WS_DISABLED                 0x40000000
                                  // WS_VISIBLE                  0x80000000

  IFrameWindow::titleBar           ( 0, FCF_TITLEBAR ),       // 0x00000001 *
  IFrameWindow::systemMenu         ( 0, FCF_SYSMENU ),        // 0x00000002 *
  IFrameWindow::menuBar            ( 0, FCF_MENU ),           // 0x00000004 *
  IFrameWindow::hideButton         ( 0, IFS_HIDEBUTTON ),     // 0x00000008 *
  IFrameWindow::minimizeButton     ( 0, FCF_MINBUTTON ),      // 0x00000010 *
  IFrameWindow::maximizeButton     ( 0, FCF_MAXBUTTON ),      // 0x00000020 *
  IFrameWindow::verticalScroll     ( 0, FCF_VERTSCROLL ),     // 0x00000040 *
  IFrameWindow::horizontalScroll   ( 0, FCF_HORZSCROLL ),     // 0x00000080 *
  IFrameWindow::deferCreation      ( 0, IFS_DEFERCREATE ),    // 0x00000100 *


                // 0x00000200 -> 0x00008000 reserved for future ICLUI usage
                // 0x00010000 -> 0x80000000 designated for customer usage

  IFrameWindow::classDefaultStyle  ( 0x00008808ul, 0x00000033ul ),
                                  // 0x00008000 appDBCSStatus
                                  // 0x00000800 sizingBorder
                                  // 0x00000008 windowList
                                                // 0x00000001 titleBar
                                                // 0x00000002 systemMenu
                                                // 0x00000010 minimizeButton
                                                // 0x00000020 maximizeButton

  // styleBits is mask with just the PM style bits "on":
  IFrameWindow::styleBits          = IFS_MASK | IWS_MASK;     // 0xFFFF89FFul
#endif  // !IC_WIN
#pragma data_seg()

/*------------------------------------------------------------------------------
| Default style for new objects (initial value).                               |
------------------------------------------------------------------------------*/
#pragma data_seg(ICLNonConst)
#ifdef IC_WIN
IFrameWindow::Style
  IFrameWindow::currentDefault ( 0x00CF0000ul, 0x00000000ul );
#else
IFrameWindow::Style
  IFrameWindow::currentDefault ( 0x00008808ul, 0x00000033ul );
#endif
  IWindow::SiblingOrder IFrameWindow::siblingCreateOrder =
                                                   IWindow::onTopOfSiblings;
#pragma data_seg()


#ifdef IC_WIN
extern "C" {
extern BOOL CALLBACK _pfnwpICDlgProc ( HWND, UINT, WPARAM, LPARAM );

static WNDPROC
  _wndClassFrameByte,
  _wndClassFrameByteAlign;

}

// icalcFrameRect performs conversion between a window rectangle and
// the window's client area rectangle.  The frameToClient argument controls
// which way the conversion is done.  If true, the input rectangle is
// the frame window rectangle, if false the rectangle is the client
// rectangle.  Similar in function to PM WinCalcFrameRect.
static IBase::Boolean icalcFrameRect ( IWindowHandle  hwnd,
                                       RECTL*         rect,
                                       IBase::Boolean frameToClient );
#endif  // IC_WIN


/*------------------------------------------------------------------------------
| IFrameHandler                                                                |
|                                                                              |
| This class wrappers an object pointer and destroys the pointer to the object |
| when the wrapper is destructed.                                              |
------------------------------------------------------------------------------*/
static struct DefaultHandler {
  operator IFrameHandler* ( );
IFrameHandler
 *operator -> ( )
  {
    return *this;
  }
 ~DefaultHandler ( )
  {
    if ( ptr )
    {
       delete ptr;
       ptr = 0;
    }
  }
IFrameHandler
 *ptr;
} defaultHandler;

DefaultHandler::operator IFrameHandler* ( )
{
  if ( !ptr )
    ptr = new IFrameHandler;
  return ptr;
}

/*------------------------------------------------------------------------------
| Static IFrameWindow members                                                  |
------------------------------------------------------------------------------*/
const unsigned
  IFrameWindowData::destroyOnClose    = 0x00000001,
  IFrameWindowData::hasDefaultHandler = 0x00000002,
  IFrameWindowData::modal             = 0x00000004,
  IFrameWindowData::needsUpdating     = 0x00000008,
  IFrameWindowData::fromDlgTemplate   = 0x00000010,
  IFrameWindowData::standardFrame     = 0x00000020,
  IFrameWindowData::hasBeenClosed     = 0x00000040,
#ifdef IC_WIN
  IFrameWindowData::needsClientSized  = 0x00000080,
  IFrameWindowData::exitModalLoop     = 0x00000100,
  IFrameWindowData::enableOwner       = 0x00000200,
#endif
  IFrameWindowData::modalAutoDelete   = 0x00000400;

/*------------------------------------------------------------------------------
| IFrameWindowData::IFrameWindowData                                           |
|                                                                              |
| This class contains the frame window data pointers.                          |
------------------------------------------------------------------------------*/
IFrameWindowData::IFrameWindowData ( Boolean flags )
  : pExtensions (0),
    flags(flags),
    rc(0),
    frameIcon(0),
    mousePtr(0),
    toolBarList(0),
    fpHelpList(0)
#ifdef IC_WIN
    ,
    logicalClientRect( )
#endif
#ifdef IC_PM
    ,
    disableBidiChildUpdate(false)
#endif
{ }

/*------------------------------------------------------------------------------
| IFrameWindowData::~IFrameWindowData                                          |
|                                                                              |
| Destructor here for page tuning.                                             |
------------------------------------------------------------------------------*/
IFrameWindowData::~IFrameWindowData ( )
{ }


/*------------------------------------------------------------------------------
| IFrameWindow::IFrameWindow                                                   |
|                                                                              |
| Constructor to create a frame window object:                                 |
| - from a dialog template,                                                    |
| - by creating a plain PM frame window with the default frame styles, or      |
| - from a dialog template, and if not found, by creating a plain PM frame     |
|   window with the default frame styles.                                      |
------------------------------------------------------------------------------*/
IFrameWindow::IFrameWindow ( unsigned long id,
                             FrameSource   source )
  : IWindow( ),
    fFrameWindowData(new IFrameWindowData(IFrameWindowData::destroyOnClose))
  {
  this -> tryToLoadDialog( IResourceId( id ), 0, 0, source );
  }

/*------------------------------------------------------------------------------
| IFrameWindow::IFrameWindow                                                   |
|                                                                              |
| Constructor to create a frame window object:                                 |
| - from a dialog template,                                                    |
| - by creating a plain PM frame window with the default frame styles, or      |
| - from a dialog template, and if not found, by creating a plain PM frame     |
|   window with the default frame styles.                                      |
------------------------------------------------------------------------------*/
IFrameWindow::IFrameWindow ( const IResourceId &resId,
                             IWindow           *owner,
                             FrameSource        source )
  : IWindow( ),
    fFrameWindowData(new IFrameWindowData(IFrameWindowData::destroyOnClose))
  {
  this -> tryToLoadDialog( resId, 0, owner, source );
  }

/*------------------------------------------------------------------------------
| IFrameWindow::IFrameWindow                                                   |
|                                                                              |
| Constructor to create a frame window object:                                 |
| - from a dialog template,                                                    |
| - by creating a plain PM frame window with the default frame styles, or      |
| - from a dialog template, and if not found, by creating a plain PM frame     |
|   window with the default frame styles.                                      |
------------------------------------------------------------------------------*/
IFrameWindow::IFrameWindow ( const IResourceId &resId,
                             IWindow           *parent,
                             IWindow           *owner,
                             FrameSource        source )
  : IWindow( ),
    fFrameWindowData(new IFrameWindowData(IFrameWindowData::destroyOnClose))
  {
  this -> tryToLoadDialog( resId, parent, owner, source );
  }

/*------------------------------------------------------------------------------
| IFrameWindow::IFrameWindow                                                   |
|                                                                              |
| Constructor to instantiate a frame window object from an existing window.    |
------------------------------------------------------------------------------*/
IFrameWindow::IFrameWindow ( const IWindowHandle &hwnd )
  : IWindow( ),
    fFrameWindowData(new IFrameWindowData(IFrameWindowData::destroyOnClose))
  {
  this -> setAutoDestroyWindow ( false );
  this -> start( hwnd );
  }

/*------------------------------------------------------------------------------
| IFrameWindow::IFrameWindow                                                   |
|                                                                              |
| Constructor to create a frame window object using styles and a given         |
| resource id.                                                                 |
------------------------------------------------------------------------------*/
IFrameWindow::IFrameWindow ( const IResourceId         &resId,
                             const IFrameWindow::Style &style )
  : IWindow( ),
    fFrameWindowData(new IFrameWindowData(IFrameWindowData::destroyOnClose))
  {
  // When the frame window rectangle is allowed to default, we use
  // the system recommended position without coordinate conversion.
  // This is so that resizing the frame will work as expected with the
  // application implemented in application coordinates.  To implement this
  // we convert the application coordinates returned by nextShellRect
  // back to native before calling initialize.
  IRectangle initRect = nextShellRect();
  if ( ICoordinateSystem::isConversionNeeded() )
     initRect = ICoordinateSystem::convertToNative( initRect,
                                                    desktopWindow()->size() );
  this -> initialize( resId, style, 0, 0, initRect );
  }

/*------------------------------------------------------------------------------
| IFrameWindow::IFrameWindow                                                   |
|                                                                              |
| Constructor to create a frame window object using styles and a given         |
| resource id.                                                                 |
------------------------------------------------------------------------------*/
IFrameWindow::IFrameWindow ( const IFrameWindow::Style &style,
                             const IResourceId         &resId )
  : IWindow( ),
    fFrameWindowData(new IFrameWindowData(IFrameWindowData::destroyOnClose))
  {
  // When the frame window rectangle is allowed to default, we use
  // the system recommended position without coordinate conversion.
  // This is so that resizing the frame will work as expected with the
  // application implemented in application coordinates.  To implement this
  // we convert the application coordinates returned by nextShellRect
  // back to native before calling initialize.
  IRectangle initRect = nextShellRect();
  if ( ICoordinateSystem::isConversionNeeded() )
     initRect = ICoordinateSystem::convertToNative( initRect,
                                                    desktopWindow()->size() );
  this -> initialize( resId, style, 0, 0, initRect );
  }

/*------------------------------------------------------------------------------
| IFrameWindow::IFrameWindow                                                   |
|                                                                              |
| Constructor to create a frame window object using the given title, resource  |
| id, and style.                                                               |
------------------------------------------------------------------------------*/
IFrameWindow::IFrameWindow ( const char                *title,
                             const IResourceId         &resId,
                             const IFrameWindow::Style &style )
  : IWindow( ),
    fFrameWindowData(new IFrameWindowData(IFrameWindowData::destroyOnClose))
  {
  // When the frame window rectangle is allowed to default, we use
  // the system recommended position without coordinate conversion.
  // This is so that resizing the frame will work as expected with the
  // application implemented in application coordinates.  To implement this
  // we convert the application coordinates returned by nextShellRect
  // back to native before calling initialize.
  IRectangle initRect = nextShellRect();
  if ( ICoordinateSystem::isConversionNeeded() )
     initRect = ICoordinateSystem::convertToNative( initRect,
                                                    desktopWindow()->size() );
  this -> initialize( resId, style, 0, 0, initRect, title );
  }

/*------------------------------------------------------------------------------
| IFrameWindow::IFrameWindow                                                   |
|                                                                              |
| Constructor to create a frame window object using the argument values.       |
------------------------------------------------------------------------------*/
IFrameWindow::IFrameWindow ( const IResourceId         &resId,
                             IWindow                   *parent,
                             IWindow                   *owner,
                             const IRectangle          &initRect,
                             const IFrameWindow::Style &style,
                             const char                *title )
  : IWindow( ),
    fFrameWindowData(new IFrameWindowData(IFrameWindowData::destroyOnClose))
  {
  this -> initialize( resId, style, parent, owner, initRect, title );
  }

/*------------------------------------------------------------------------------
| IFrameWindow::~IFrameWindow                                                  |
|                                                                              |
| Destructor must iterate extensions, delete each extension, and delete the    |
| collection.                                                                  |
------------------------------------------------------------------------------*/
IFrameWindow::~IFrameWindow ( )
  {
  IMODTRACE_DEVELOP( "IFrameWindow::dtor" );

  this -> removeDefaultHandler();      // Remove our handler.

  // Remove the recoordination handler.
  IRecoordHandler::defaultHandler()->stopHandlingEventsFor( this );

  IFrameExtensions                     // Process extensions...
   *exts = this->extensions();

  if ( exts )
    {                                  // Remove each one...
    IFrameExtensions::Cursor
      cursor( *exts );
    for( cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
      {
      IFrameExtension
       *ext = exts -> elementAt( cursor );
      delete ext;
      }

    delete exts;                       // Remove collection...
    }
  /********************************************************************/
  /* If a help id lookup table exists, get rid of it.                 */
  /********************************************************************/
  if ( fFrameWindowData->fpHelpList )
    IMenuPrivate::deleteHelpIdTable( this );

  delete fFrameWindowData;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::defaultStyle                                                   |
|                                                                              |
| Return the current default style (static member).                            |
------------------------------------------------------------------------------*/
IFrameWindow::Style IFrameWindow::defaultStyle ( )
  {
  return IFrameWindow::currentDefault;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::setDefaultStyle                                                |
|                                                                              |
| Set the current default static member to the argument style.                 |
------------------------------------------------------------------------------*/
void IFrameWindow::setDefaultStyle ( const Style &style )
  {
  IFrameWindow::currentDefault = style;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::convertToGUIStyle                                              |
|                                                                              |
| Returns base style for the control by default, or extended style if          |
| extended flag (bExtOnly) is set.                                             |
------------------------------------------------------------------------------*/
unsigned long IFrameWindow::convertToGUIStyle ( const IBitFlag& style,
                                                Boolean extendedOnly ) const
{
  // Obtain the style from the class (IWindow) that we inherit from
  unsigned long ulStyle =
             Inherited::convertToGUIStyle( style, extendedOnly );

  if (extendedOnly)
  {
    // Use mask to return extended styles in the user defined range and
    // FCF_ styles which have a partial one-to-one correspondence to our
    // extended style bits, so return FCF_ asis, masking out the unused bits.
    unsigned long ulTmp = extendedStyle();
    ulStyle |= ulTmp & IFS_EXTMASK;

#ifndef IC_WIN
    // Handle overlapping styles
    if (ulTmp & IFS_HIDEBUTTON)
      ulStyle |= FCF_HIDEBUTTON;

    // The next 2 lines hack around bugs in OS/2 PM
    if (style.asUnsignedLong() & dialogBackground.asUnsignedLong())
      ulStyle |= FCF_AUTOICON;
    if (style.asUnsignedLong() & appDBCSStatus.asUnsignedLong())
      ulStyle |= FCF_DBE_APPSTAT;
#endif
  }
  else
  {
    // FS_ styles have a one-to-one correspondence to our style bits, and
    // inhabit the lower word of the base GUI style.  Therefore, obtain
    // that portion asis, masking out the upper word.
    ulStyle |= style.asUnsignedLong() & IFS_MASK;

#ifdef IC_WIN
     //----------------------------------------------------------------------------
     // DEFECT 23816
     // Disable maximize button and remove system menu size 
     // when dialogBorder style is selected.
     //----------------------------------------------------------------------------
     // If a dialog border is wanted must not have WS_THICKFRAME on
     // to paint correctly on NT3.51 old shell
     IFrameWindow::Style& bitStyle = (IFrameWindow::Style &) style;
     if ( (bitStyle & IFrameWindow::dialogBorder ) != 0 )
     {
       ulStyle &=  ~(unsigned long)WS_THICKFRAME;
       // if dialogBorder is specified then don't allow the window to be maximized.
       // dialog windows are not resizeable, but can be minimized.
       ulStyle &=  ~(unsigned long)WS_MAXIMIZEBOX;
     }
     //----------------------------------------------------------------------------
   
     //----------------------------------------------------------------------------
     // DEFECT 23816
     // copy required style changes from porta-pak implementation
     //----------------------------------------------------------------------------
     // If either minimize or maximize button wanted, must have system menu
     if ( ((IPlatform::isWin95()) || (IPlatform::isNTNewShell())) &&
          ((bitStyle & IFrameWindow::minimizeButton) ||
           (bitStyle & IFrameWindow::maximizeButton)) )
       ulStyle |=  WS_SYSMENU;
   
     // If a system menu is wanted must have WS_CAPTION
     if ( bitStyle & IFrameWindow::systemMenu )
       ulStyle |=  WS_CAPTION;
   
     // If a sizing border is wanted must have WS_BORDER to paint
     // correctly on NT3.51 old shell
     if ( bitStyle & IFrameWindow::sizingBorder )
       ulStyle |= WS_BORDER;
   
     if ( !(bitStyle & IFrameWindow::titleBar) )
     {
       // WS_OVERLAPPED implies titlebar.  Make it a popup if we dont need
       // the titlebar area for buttons or system menu
       if ( !(/*(bitStyle & IFrameWindow::child) ||*/
              (bitStyle & IFrameWindow::minimized) ||
              (bitStyle & IFrameWindow::maximized) ||
              (bitStyle & IFrameWindow::systemMenu)) )
       {
         ulStyle |= WS_POPUP;
       }
     }
     //----------------------------------------------------------------------------
#endif   // IC_WIN
  }

  return( ulStyle );
}

/*------------------------------------------------------------------------------
| IFrameWindow::close                                                          |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::close ( )
  {
  this -> postEvent( WM_CLOSE );
  return *this;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::minimize                                                       |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::minimize ( )
  {
  this -> sendEvent( WM_SYSCOMMAND,
                     SC_MINIMIZE,
#ifdef IC_WIN
                     IEventParameter2( 0 ) );
#else   // !IC_WIN
                     IEventParameter2( CMDSRC_OTHER, false ) );
#endif  // !IC_WIN
  return *this;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::maximize                                                       |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::maximize ( )
  {
  this -> sendEvent( WM_SYSCOMMAND,
                     SC_MAXIMIZE,
#ifdef IC_WIN
                     IEventParameter2( 0 ) );
#else   // !IC_WIN
                     IEventParameter2( CMDSRC_OTHER, false ) );
#endif  // !IC_WIN
  return *this;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::restore                                                        |
|                                                                              |
| Restores a minimized or maximized window.                                    |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::restore ( )
  {
  this -> postEvent( WM_SYSCOMMAND,
                     SC_RESTORE,
#ifdef IC_WIN
                     IEventParameter2( 0 ) );
#else   // !IC_WIN
                     IEventParameter2( CMDSRC_OTHER, false ) );
#endif  // !IC_WIN
  return *this;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::moveSizeToClient                                               |
|                                                                              |
| Size and position the frame window around the client rectangle.              |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::moveSizeToClient ( const IRectangle& clientRect )
  {
  this->moveSizeTo( this->frameRectFor( clientRect ) );
  return *this;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::clientRectFor                                                  |
|                                                                              |
| Obtain the client window rectangle based on the size and position of         |
| the frame window and its' extensions.                                        |
------------------------------------------------------------------------------*/
IRectangle IFrameWindow::clientRectFor ( const IRectangle& frameRect ) const
  {
#ifdef IC_WIN
   // In Windows, the system has no support for adjusting the client
   // size for frame extensions.  Therefore we have to compute a logical
   // client rectangle.  We store this in the frame window data area.
   // The needClientSized flag is set in update() when needsUpdating is
   // seen.  The separate flag is needed because this function is called
   // during processing caused by update().
   if ( ( fFrameWindowData->flags & IFrameWindowData::needsUpdating) ||
        ( fFrameWindowData->flags & IFrameWindowData::needsClientSized ))
      {
      // clear the invalid flag
      fFrameWindowData->flags &= ~IFrameWindowData::needsClientSized;
      RECTL rc = frameRect.asRECTL();
      if ( frameRect != IRectangle() )
         {
         // convert from interface coordinates to system coordinates
         rc = ICoordinateSystem::convertToNative( frameRect,
                                                  parentSize() ).asRECTL();
         }

      if (!icalcFrameRect( this->handle(), &rc, true ) )
         {
         ITHROWGUIERROR("icalcFrameRect");
         }
      // store the updated rectangle
      // This is a client rectangle so it is expressed in client coordinates
      // Get the Windows client size so we can adjust the coordinates
      // of the logical client rectangle.
      RECT realClient;
      ICALCCLIENTRECT( handle(), &realClient );
      fFrameWindowData->logicalClientRect =
         ICoordinateSystem::convertToApplication(
                               IRectangle(rc),
                               ISize ( realClient.right - realClient.left,
                                       realClient.bottom - realClient.top ) );
      }  // needs client sized
   return fFrameWindowData->logicalClientRect;
#endif
#ifndef IC_WIN
  RECTL rc = ICoordinateSystem::convertToNative( frameRect,
                                                 parentSize() ).asRECTL();

  if (!ICALCCLIENTRECT( this->handle(), &rc ))
  {
    // If empty rectangle calculated (or error), return null rectangle
    return IRectangle();
  }
  return ICoordinateSystem::convertToApplication( IRectangle( rc ),
                                                  parentSize() );
#endif
  }

/*------------------------------------------------------------------------------
| IFrameWindow::frameRectFor                                                   |
|                                                                              |
| Obtain the frame window rectangle based on the size and position of          |
| the client rectangle and the frame windows extensions.                       |
------------------------------------------------------------------------------*/
IRectangle IFrameWindow::frameRectFor ( const IRectangle& clientRect ) const
  {
#ifdef IC_WIN
   // get the real client rectangle to adjust input coordinates which
   // are the logical client (after extensions)
   RECT realClient;
   ICALCCLIENTRECT( handle(), &realClient );

   RECTL rc =
      ICoordinateSystem::convertToNative(
         clientRect,
         ISize ( realClient.right - realClient.left,
                 realClient.bottom - realClient.top ) ).asRECTL();
   if ( icalcFrameRect( this->handle(), &rc, false) )
#endif
#ifndef IC_WIN
  RECTL rc = clientRect.asRECTL();   // client rect origin is 0,0 ... no recoord
  if (!WinCalcFrameRect( this->handle(), &rc, FALSE ))
#endif  // !IC_WIN
    {
    ITHROWGUIERROR("WinCalcFrameRect");
    }
  return ICoordinateSystem::convertToApplication( IRectangle( rc),
                                                  parentSize()  );
  }

/*------------------------------------------------------------------------------
| IFrameWindow::maximizeRect                                                   |
|                                                                              |
| Obtain the window's maximized position.                                      |
------------------------------------------------------------------------------*/
IRectangle IFrameWindow::maximizeRect ( ) const
  {
#ifdef IC_WIN
  WINDOWPLACEMENT
#else   // !IC_WIN
  SWP
#endif  // !IC_WIN
    posInfo = { 0 };

  if (! IQUERYMAXPOSITION( this->handle(), &posInfo ) )
  {
    ITHROWGUIERROR("IQUERYMAXPOSITION");
  }

  IRectangle result =
#ifdef IC_WIN
         IRectangle( IPoint( posInfo.ptMaxPosition ),
                     ISize ( GetSystemMetrics( SM_CXSCREEN ),
                             GetSystemMetrics( SM_CYSCREEN ) ) );
#else   // !IC_WIN
         IRectangle( IPoint( posInfo.x, posInfo.y ),
                     ISize( posInfo.cx, posInfo.cy ) );
#endif  // !IC_WIN
  return ICoordinateSystem::convertToApplication( result,
                                                  parentSize() );
  }

/*------------------------------------------------------------------------------
| IFrameWindow::minimizeRect                                                   |
|                                                                              |
| Obtain the window's minimized position.                     .                |
|                                                                              |
| Note: The WinGetMinPosition API turns on the WS_MINIMIZED style flags which  |
|       leaves the frame in an odd state.  Thus, we must turn the style back   |
|       off (if it was not on before the call).                                |
------------------------------------------------------------------------------*/
IRectangle IFrameWindow::minimizeRect ( ) const
  {
#ifdef IC_WIN
  WINDOWPLACEMENT
#else   // !IC_WIN
  Boolean
    wasMinimized = this->style() & WS_MINIMIZED;
  SWP
#endif  // !IC_WIN
    posInfo = { 0 };

  if (! IQUERYMINPOSITION( this->handle(), &posInfo ) )
  {
    ITHROWGUIERROR("IQUERYMINPOSITION");
  }

#ifdef IC_WIN
  IRectangle result( IPoint( posInfo.ptMinPosition ),
                     ISize ( GetSystemMetrics( SM_CXMIN ),
                             GetSystemMetrics( SM_CYMIN ) ) );
#else   // !IC_WIN
  // Don't really want to minimize so undo this style if need be...
  if ( !wasMinimized )
    // Have to cast away const-ness!
     ((IFrameWindow*)this) -> setStyle( this->style() &
                                         (unsigned long)~WS_MINIMIZED );

  IRectangle result( IPoint( posInfo.x, posInfo.y ),
                     ISize( posInfo.cx, posInfo.cy ) );
#endif  // !IC_WIN
  return ICoordinateSystem::convertToApplication( result,
                                                  parentSize() );
  }

/*------------------------------------------------------------------------------
| IFrameWindow::restoreRect                                                    |
|                                                                              |
| Obtain the restore rectangle position and size from the window words.        |
------------------------------------------------------------------------------*/
IRectangle IFrameWindow::restoreRect ( ) const
  {
#ifdef IC_WIN
  WINDOWPLACEMENT placeInfo = { 0 };
  if (! GetWindowPlacement( this->handle(), &placeInfo ) )
  {
     ITHROWGUIERROR("GetWindowPlacement");
  }

  IRectangle result( placeInfo.rcNormalPosition );
#else   // !IC_WIN
  IPoint
    pos( this->windowUShort( QWS_XRESTORE ),
         this->windowUShort( QWS_YRESTORE ) );
  ISize
    size( this->windowUShort( QWS_CXRESTORE ),
          this->windowUShort( QWS_CYRESTORE ) );

  IRectangle result( pos, size );
#endif  // !IC_WIN
  return ICoordinateSystem::convertToApplication( result,
                                                  parentSize() );
  }

/*------------------------------------------------------------------------------
| IFrameWindow::setRestoreRect                                                 |
|                                                                              |
| Set the restore position and size in the window words.                       |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::setRestoreRect ( const IRectangle &rect )
  {
  IRectangle nativeRect =
     ICoordinateSystem::convertToNative( rect, parentSize() );
#ifdef IC_WIN
#ifdef IC_NOTYET
  WINDOWPLACEMENT placeInfo = { sizeof(placeInfo), 0, SW_SHOWNA,
                                IPoint(),
// dab                          this->maximizeRect().position(),
                                this->maximizeRect().minXMinY(),
                                (RECT)nativeRect };
  if (! SetWindowPlacement( this->handle(), &placeInfo ) )
  {
     ITHROWGUIERROR("SetWindowPlacement");
  }
#endif  // IC_NOTYET
#else   // !IC_WIN
  this -> setWindowData( QWS_XRESTORE,  (unsigned short)nativeRect.minX() );
  this -> setWindowData( QWS_YRESTORE,  (unsigned short)nativeRect.minY() );
  this -> setWindowData( QWS_CXRESTORE, (unsigned short)nativeRect.width() );
  this -> setWindowData( QWS_CYRESTORE, (unsigned short)nativeRect.height() );
#endif  // !IC_WIN
  return *this;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::nextShellRect                                                  |
|                                                                              |
| Returns the system-recommended coordinates of the next main window.          |
------------------------------------------------------------------------------*/
IRectangle IFrameWindow::nextShellRect ( )
  {
  IRectangle
    result;
#ifdef IC_WIN
  STARTUPINFO
    sui;

  // Query the process start-up information to obtain this info
  GetStartupInfo( &sui );
  result = IRectangle( IPoint( sui.dwX, sui.dwY ),
                       ISize( sui.dwXSize, sui.dwYSize ) );
#else   // !IC_WIN
  SWP
    swp;
  unsigned long
    rc;
  rc = WinQueryTaskSizePos( IThread::current().anchorBlock(),
                            0,
                            &swp );

  // Note: If PMERR_INVALID_SESSION_ID is returned, the size and position
  //       values are still valid.  PM has a bug that returns this error
  //       if all of the frame windows for the current process have been
  //       removed from the task list.

  if (( rc == 0 ) || ( rc == PMERR_INVALID_SESSION_ID ))
    {
    result = IRectangle( IPoint( swp.x, swp.y ), ISize( swp.cx, swp.cy ) );
    }
  else
    {
    ITHROWGUIERROR("WinQueryTaskSizePos");
    }
#endif  // !IC_WIN
  if ( ICoordinateSystem::isConversionNeeded() )
     result = ICoordinateSystem::convertToApplication( result,
                                                       desktopWindow()->size() );
  return result;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::borderWidth                                                    |
|                                                                              |
| Returns the border width of the frame window.                                |
------------------------------------------------------------------------------*/
unsigned long IFrameWindow::borderWidth ( ) const
  {
  IMODTRACE_DEVELOP( "IFrameWindow::borderWidth" );

#ifdef IC_WIN
  int borderType = 0;   // Assume frame has no border.

  // Assume frame has only one border style.
  unsigned long style = this->style();
  if ( style & WS_THICKFRAME )
  {
     borderType = SV_CXSIZEBORDER;
  }
  else if ( style & WS_DLGFRAME )
  {
     borderType = SV_CXDLGFRAME;
  }
  else if ( style & WS_BORDER )
  {
     borderType = SV_CXBORDER;
  }
  else if( IQUERYWINDOWULONG( this->handle(), GWL_EXSTYLE ) &
                                           WS_EX_DLGMODALFRAME )
  {
     borderType = SV_CXDLGFRAME;
  }

  return borderType ? IQUERYSYSVALUE( borderType ) : 0;
#else   // !IC_WIN
  return this->borderSize().width();
#endif  // !IC_WIN
  }

/*------------------------------------------------------------------------------
| IFrameWindow::borderHeight                                                   |
|                                                                              |
| Returns the border height of the frame window.                               |
------------------------------------------------------------------------------*/
unsigned long IFrameWindow::borderHeight ( ) const
  {
  IMODTRACE_DEVELOP( "IFrameWindow::borderHeight" );

#ifdef IC_WIN
  int borderType = 0;   // Assume frame has no border.

  // Assume frame has only one border style.
  unsigned long style = this->style();
  if ( style & WS_THICKFRAME )
  {
     borderType = SV_CYSIZEBORDER;
  }
  else if ( style & WS_DLGFRAME )
  {
     borderType = SV_CYDLGFRAME;
  }
  else if ( style & WS_BORDER )
  {
     borderType = SV_CYBORDER;
  }
  else if( IQUERYWINDOWULONG( this->handle(), GWL_EXSTYLE ) &
                                           WS_EX_DLGMODALFRAME )
  {
     borderType = SV_CYDLGFRAME;
  }

  return borderType ? IQUERYSYSVALUE( borderType ) : 0;
#else   // !IC_WIN
  return this->borderSize().height();
#endif  // !IC_WIN
  }

/*------------------------------------------------------------------------------
| IFrameWindow::borderSize                                                     |
|                                                                              |
| Returns the border width and height of the frame window.                     |
------------------------------------------------------------------------------*/
ISize IFrameWindow::borderSize ( ) const
  {
  IMODTRACE_DEVELOP( "IFrameWindow::borderSize" );

#ifdef IC_WIN
  return ISize( this->borderWidth(), this->borderHeight() );
#else   // !IC_WIN
  POINTL ptl;

  this -> sendEvent( WM_QUERYBORDERSIZE,
                     IEventParameter1( (PWPOINT)&ptl ),
                     0 );

  return ISize( ptl.x, ptl.y );
#endif  // !IC_WIN
  }

/*------------------------------------------------------------------------------
| IFrameWindow::setBorderWidth                                                 |
|                                                                              |
| Set the border width of the frame window.                                    |
------------------------------------------------------------------------------*/
IFrameWindow& IFrameWindow::setBorderWidth ( unsigned long cx )
  {
  IMODTRACE_DEVELOP( "IFrameWindow::setBorderWidth" );

#ifdef IC_WIN
  // This is a problem, since it appears that Windows only lets you
  // change the system border sizes.  Border size doesn't look like
  // something that each frame keeps track of.
#else   // !IC_WIN
  IASSERTPARM( this->style() & IFrameWindow::sizingBorder.asUnsignedLong() );

  this -> sendEvent( WM_SETBORDERSIZE,
                     IEventParameter1( cx ),
                     IEventParameter2( this->borderHeight() ));

#endif  // !IC_WIN
  return *this;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::setBorderHeight                                                |
|                                                                              |
| Set the border height of the frame window.                                   |
------------------------------------------------------------------------------*/
IFrameWindow& IFrameWindow::setBorderHeight ( unsigned long cy )
  {
  IMODTRACE_DEVELOP( "IFrameWindow::setBorderHeight" );

#ifdef IC_WIN
  // This is a problem, since it appears that Windows only lets you
  // change the system border sizes.  Border size doesn't look like
  // something that each frame keeps track of.
#else   // !IC_WIN
  IASSERTPARM( this->style() & IFrameWindow::sizingBorder.asUnsignedLong() );

  this -> sendEvent( WM_SETBORDERSIZE,
                     IEventParameter1( this->borderWidth() ),
                     IEventParameter2( cy ));
#endif  // !IC_WIN
  return *this;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::setBorderSize                                                  |
|                                                                              |
| Set the border width and height of the frame window.                         |
------------------------------------------------------------------------------*/
IFrameWindow& IFrameWindow::setBorderSize ( const ISize& size )
  {
  IMODTRACE_DEVELOP( "IFrameWindow::setBorderSize" );

#ifdef IC_WIN
  // This is a problem, since it appears that Windows only lets you
  // change the system border sizes.  Border size doesn't look like
  // something that each frame keeps track of.
#else   // !IC_WIN
  IASSERTPARM( this->style() & IFrameWindow::sizingBorder.asUnsignedLong() );

  this -> sendEvent( WM_SETBORDERSIZE,
                     IEventParameter1( (unsigned long)size.width() ),
                     IEventParameter2( (unsigned long)size.height() ));
#endif  // !IC_WIN
  return *this;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::setBorderSize                                                  |
|                                                                              |
| Set the border width and height of the frame window.                         |
------------------------------------------------------------------------------*/
IFrameWindow& IFrameWindow::setBorderSize ( unsigned long cxcy )
  {
  IMODTRACE_DEVELOP( "IFrameWindow::setBorderSize" );

#ifdef IC_WIN
  // This is a problem, since it appears that Windows only lets you
  // change the system border sizes.  Border size doesn't look like
  // something that each frame keeps track of.
#else   // !IC_WIN
  IASSERTPARM( this->style() & IFrameWindow::sizingBorder.asUnsignedLong() );

  this -> sendEvent( WM_SETBORDERSIZE,
                     IEventParameter1( cxcy ),
                     IEventParameter2( cxcy ));
#endif  // !IC_WIN
  return *this;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::setBorderSize                                                  |
|                                                                              |
| Set the border width and height of the frame window.                         |
------------------------------------------------------------------------------*/
IFrameWindow& IFrameWindow::setBorderSize ( unsigned long cx,
                                            unsigned long cy )
  {
  IMODTRACE_DEVELOP( "IFrameWindow::setBorderSize" );

#ifdef IC_WIN
  // This is a problem, since it appears that Windows only lets you
  // change the system border sizes.  Border size doesn't look like
  // something that each frame keeps track of.
#else   // !IC_WIN
  IASSERTPARM( this->style() & IFrameWindow::sizingBorder.asUnsignedLong() );

  if ( this->style() & IFrameWindow::sizingBorder.asUnsignedLong() )
    this -> sendEvent( WM_SETBORDERSIZE,
                       IEventParameter1( cx ),
                       IEventParameter2( cy ));
#endif  // !IC_WIN
  return *this;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::showModally                                                    |
|                                                                              |
| Causes the frame to be displayed in application-modal mode.                  |
| Notes: If the frame's parent and owner are the same, throw exception as      |
|          this will result in the modal dialog being disabled, too.           |
|        Set the flag to mark this frame as being in modal display mode.       |
|        Next, clear the "dismissed" flag so PM will actually process the      |
|          window (WinProcessDlg simply returns if the flag is already set).   |
|        Then, call WinProcessDlg.                                             |
|        Return the "result" data member (which avoids the problem of PM       |
|          "thunking" the WinProcessDlg return code to a short).               |
------------------------------------------------------------------------------*/
unsigned long IFrameWindow::showModally ( )
{
   IMODTRACE_DEVELOP( "IFrameWindow::showModally" );
#ifndef IC_WIN
   // IC_NOTYET
   // For WS_POPUP style (recommended for dialogs),  GetParent returns
   // the same value as the owner.  Parent does not appear to be really
   // the owner window ... perhaps a bug in NT 3.1 ??
   IASSERTPARM( this->parent() != this->owner() );
#endif

   this -> update();

   fFrameWindowData->flags |= IFrameWindowData::modal;

#ifdef IC_WIN
   fFrameWindowData->flags &= ~IFrameWindowData::exitModalLoop;

   // Need to do stuff here to disable the owner for modal behavior
   IWindow* owner = this->owner();
   // if owner is desktop do not disable it
   if ( owner == IWindow::desktopWindow() )
      owner = 0;

   // if owner is already disabled then do nothing.  App is probably
   // broke but we don't want to change state of owner.
   fFrameWindowData->flags &= ~IFrameWindowData::enableOwner;
   if ( (owner) && (owner->isEnabled() ))
      {
      owner->disable();
      // set flag for use by dismiss to re-enable owner.
      fFrameWindowData->flags |= IFrameWindowData::enableOwner;
      }

   // show the modal window
   this -> show();

   // Enter a private event loop.
   QMSG
      msg;
   HWND hwndDlg = this->handle();
   Boolean done = fFrameWindowData->flags & IFrameWindowData::exitModalLoop;
   while ( !done )
      {
      //IC_NOTYET  second parameter maybe should be the hwnd of the dialog???
      done = !(GetMessage( &msg, 0, 0, 0)) ;
      if (!done)
         {
         if (!IsDialogMessage( hwndDlg, &msg ) )
            {
            TranslateMessage( &msg );
            DispatchMessage( &msg );
            }
         }
      done |= fFrameWindowData->flags & IFrameWindowData::exitModalLoop;
      }

   // If we disabled the owner then we need to reenable it.  The doc
   // on EnableWindow says that this should happen before the modal window
   // is destroyed to get focus back to the owner.   Therefore the
   // processing is done in IFrameWindow::dismiss, since we may
   // destroy the dialog before we get here.
#else   // !IC_WIN

  WinSetWindowUShort( this->handle(),
                      QWS_FLAGS,
                      (unsigned short)(WinQueryWindowUShort( this->handle(),
                                            QWS_FLAGS ) & ~FF_DLGDISMISSED ));

  WinProcessDlg( this->handle() );


#endif  // !IC_WIN
  fFrameWindowData->flags &= ~IFrameWindowData::modal;
  unsigned long result = this->result();

  // Now do auto-delete object processing.  The modalAutoDelete flag
  // is set by IFrameHandler during WM_DESTROY processing.
  if ( fFrameWindowData->flags & IFrameWindowData::modalAutoDelete )
  {
     delete this;
  }

  return result;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::result                                                         |
|                                                                              |
| Return the most recent result for this frame.                                |
------------------------------------------------------------------------------*/
unsigned long IFrameWindow::result ( ) const
  {
  return fFrameWindowData->rc;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::dismiss                                                        |
|                                                                              |
| Dismisses a frame window by hiding it, and, if modal, causes the completion  |
| of the showModally function call that triggered its display.                 |
| Notes: Set the frame "result" to the argument value.                         |
|        If the frame is modal, dismiss it.                                    |
|        Otherwise, simply hide the frame.                                     |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::dismiss ( unsigned long result )
{
  this -> setResult( result );

  if ( this->isModal() )
     {
#ifdef IC_WIN
     // Re-enable the owner.  This has to happen before destruction
     // of the modal window.  Do it before the hide to minimize flashing
     // in cases where focus was changed by user to another app during
     // lifetime of dialog.
     if (fFrameWindowData->flags & IFrameWindowData::enableOwner )
        {
        fFrameWindowData->flags &= ~IFrameWindowData::enableOwner;
        IWindow* owner = this->owner();
        if (owner)
           {
           owner->enable();
           // move owner to top of Z-order
           owner->positionOnSiblings();
           }
        }

     // hide the window
     this->hide();

     // Bail out of the event loop we started when the modal
     // frame was shown.   Set flag to cause exit.
     this->fFrameWindowData->flags |= IFrameWindowData::exitModalLoop;
#else
     IDISMISSDLG( this->handle(), (unsigned short)result );
#endif
     }
  else
     {
     this->hide();

     // Check if we are the active window, and if so set the focus
     // to the owner
     if ( IQUERYACTIVEWINDOW(desktopWindow()->handle()) == handle() )
        {
        if ( this->owner() )
          this->owner()->setFocus();
       else
          this->parent()->setFocus();
        }

     }

  return *this;
}

/*------------------------------------------------------------------------------
| IFrameWindow::setResult                                                      |
|                                                                              |
| Simply set the "rc" data member to the argument value.                       |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::setResult ( unsigned long result )
  {
  fFrameWindowData->rc = result;
  return *this;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::notifyOwner                                                    |
|                                                                              |
| Posts a command event to the frame's (or dialog's) owner.                    |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::notifyOwner ( unsigned long         id,
                                          ICommandEvent::Source source,
                                          Boolean               pointerDevice )
  {
  IWindow* pwinOwner = this->owner();
  if (pwinOwner)
  {                               // Have an owner to notify
     pwinOwner->postEvent( WM_COMMAND, id,
                           IEventData( source,
                                       (unsigned short)pointerDevice ) );
  } /* endif */
  return *this;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::icon                                                           |
------------------------------------------------------------------------------*/
IPointerHandle IFrameWindow::icon ( ) const
  {
#ifdef IC_WIN
  return IPointerHandle( (HICON)GetClassLong( this -> handle(), GCL_HICON ));
#else   // !IC_WIN
  return IPointerHandle( this -> sendEvent( WM_QUERYICON ) );
#endif  // !IC_WIN
  }

/*------------------------------------------------------------------------------
| IFrameWindow::setIcon                                                        |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::setIcon ( const IPointerHandle &icon )
  {
#ifdef IC_WIN
  SetClassLong( this -> handle(), GCL_HICON, (ULONG)((void *)icon) );
#else   // !IC_WIN
  this -> sendEvent( WM_SETICON, (unsigned long)icon );
#endif  // !IC_WIN
  fFrameWindowData->frameIcon = icon;
  return *this;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::setIcon                                                        |
|                                                                              |
| Set icon after loading it from resource library.                             |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::setIcon ( const IResourceId &iconResId )
  {
#ifndef IC_WIN
  if ( this->style() & IFrameWindow::minimizedIcon.asUnsignedLong() )
    this -> setIcon( iconResId.resourceLibrary().loadIcon( iconResId, false ));
  else
#endif  // !IC_WIN
    this -> setIcon( iconResId.resourceLibrary().loadIcon( iconResId ));
  return *this;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::setIcon                                                        |
|                                                                              |
| Set icon after loading it from default resource library.                     |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::setIcon ( unsigned long iconResId )
  {
  this -> setIcon( IResourceId( iconResId ) );
  return *this;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::mousePointer                                                   |
|                                                                              |
| Return the handle of the mouse pointer to use when the mouse is over the     |
| frame window.                                                                |
------------------------------------------------------------------------------*/
IPointerHandle IFrameWindow::mousePointer ( ) const
  {
  return fFrameWindowData->mousePtr;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::setMousePointer                                                |
|                                                                              |
| Set the handle of the mouse pointer to use when the mouse is over the        |
| frame window.                                                                |
------------------------------------------------------------------------------*/
IFrameWindow& IFrameWindow::setMousePointer (
                                            const IPointerHandle& mousePointer )
  {
  fFrameWindowData->mousePtr = mousePointer;

  if ( mousePointer )
  {
     // If the mouse pointer is within the frame window, consider changing it
     // immediately to the specified pointer (rather than wait for the change to
     // occur on the next mouse move event).
     unsigned long
       x, y;
     IQUERYPOINTERPOSITION( x, y );
     IPoint
       mousePosition( x, y );
     if ( this->nativeRect().contains( mousePosition ) )
     {
        // Don't change the mouse pointer if it is over a window that is
        // not related to the frame (the frame may be covered by a window
        // from another application).  Assume that if the window under the
        // mouse pointer is a descendant (child at any level) of the frame
        // window in the window hierarchy, it is ok to change the mouse pointer.
        POINTL
          pt( mousePosition.asPOINTL() );
        IWindowHandle
          frameHandle( this->handle() ),
          desktopHandle( IWindow::desktopWindow()->handle() ),
          handleWithMouse( IWINDOWFROMPOINT( desktopHandle, &pt, true ) );
        Boolean
          done = false;
        for ( ; ! done; handleWithMouse = IPARENTOF( handleWithMouse ) )
        {
           if ( frameHandle == handleWithMouse )
           {  // The window with the mouse is the frame or a descendant.
              ISETPOINTER( mousePointer );
              done = true;
           }
           else if ( handleWithMouse == 0  ||  handleWithMouse == desktopHandle )
           {  // We didn't find the frame after walking the parent chain.
              done = true;
           }
        }
     }
  }

  return *this;
  }


/*------------------------------------------------------------------------------
| IFrameWindow::clientHandle                                                   |
------------------------------------------------------------------------------*/
IWindowHandle IFrameWindow::clientHandle ( ) const
  {
  return IWindow::handleWithParent( FID_CLIENT, this->handle() );
  }

/*------------------------------------------------------------------------------
| IFrameWindow::client                                                         |
------------------------------------------------------------------------------*/
IWindow *IFrameWindow::client ( ) const
  {
  return IWindow::windowWithHandle( this->clientHandle() );
  }

/*------------------------------------------------------------------------------
| IFrameWindow::setClient                                                      |
|                                                                              |
| Sets one of the frame's children as the client area.                         |
| Notes: First, if there is a current client control (different than the       |
|          argument) then change the parent and the owner of that control to   |
|          HWND_OBJECT.                                                        |
|        Next, set the pClient member to the argument.  If it really is a      |
|          client (pointer not 0) then set its parent and owner to this frame  |
|          and set its window ID to FID_CLIENT.                                |
|        If the client changed, refresh the frame.                             |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::setClient ( IWindow *newClient )
{
  IMODTRACE_DEVELOP( "IFrameWindow::setClient" );

  IASSERTPARM ( newClient != this );

  IWindowHandle previousClient = this->clientHandle();
  IWindowHandle newClientHandle = 0;

  Boolean
    updateIt       = false;
  Boolean
    showing        = this->isVisible(),
#ifdef IC_WIN
    // Check for case where previous client is NULL
    clientHasFocus = ( previousClient && ( previousClient ==
                        IQUERYFOCUS( IWindow::desktopWindow()->handle() )));
#else
    clientHasFocus = ( previousClient ==
                         IQUERYFOCUS( IWindow::desktopWindow()->handle() ));
#endif

  if ( newClient )
     newClientHandle = newClient->handle();

  if ( previousClient != newClientHandle )
     {
     if ( showing )
        IENABLEWINDOWUPDATE( this->handle(), false );

     if ( clientHasFocus )
        this->setFocus();

     if ( previousClient )
        ISETPARENT( previousClient,
                    IWindow::objectWindow()->handle(),
                    false );

     if ( newClient )
        {
        newClient -> setParent( this );
        ISETIDOF( newClient->handle(), FID_CLIENT );
        if ( clientHasFocus )
           newClient -> setFocus();
        }
#ifdef IC_PM
     else
        {
        // To work around an OS/2 bug where the client rectangle is
        // not correct when we remove a client, we need to force an
        // extra update of the frame.  Defect 4008.
        fFrameWindowData->flags |= IFrameWindowData::needsUpdating;
        this->update();
        }
#endif

     updateIt = true;
     }
  else
     {
     // Check for the case where setClient was done with a window
     // whose original id was FID_CLIENT.  If the frame is showing
     // compare client rectangle to see if update needed.   If not
     // showing, always set needsUpdating.  In latter case cannot call
     // clientRectFor since rect may be null and cause WinCalcFrameRect
     // to fail.
     if ( showing )
        {
        IRectangle newClientRect = IRectangle();
        if (newClient)
           newClientRect = newClient->rect();
        if (clientRectFor( this->rect() )  != newClientRect )
           updateIt = true;
        }
     else
        {
        // Always set update if not showing.
        updateIt = true;
        }
     }

  if ( updateIt )
     {
     fFrameWindowData->flags |= IFrameWindowData::needsUpdating;

     if ( showing )
        {
#ifdef IC_WIN
        // Enable Window since show does not do this in Windows.
        IENABLEWINDOWUPDATE( this->handle(), true );
#endif
        this -> show();
        }
     }

  return *this;
}

/*------------------------------------------------------------------------------
| IFrameWindow::addExtension                                                   |
|                                                                              |
| Create and add new extensions.                                               |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::addExtension ( IWindow       *newExtension,
                                           Location       location,
                                           unsigned long  widthOrHeight,
                                           SeparatorType  separator )
  {
  return this -> addExtension( new IFrameExtension( newExtension,
                                                    location,
                                                    widthOrHeight,
                                                    separator ) );
  }

/*------------------------------------------------------------------------------
| IFrameWindow::addExtension                                                   |
|                                                                              |
| Convert width/height arg to unsigned long and use overloaded function.       |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::addExtension ( IWindow       *newExtension,
                                           Location       location,
                                           int            widthOrHeight,
                                           SeparatorType  separator )
  {
  return this -> addExtension( newExtension,
                               location,
                               (unsigned long)widthOrHeight,
                               separator );
  }

/*------------------------------------------------------------------------------
| IFrameWindow::addExtension                                                   |
|                                                                              |
| Create and add new extensions.                                               |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::addExtension ( IWindow       *newExtension,
                                           Location       location,
                                           double         percentage,
                                           SeparatorType  separator )
  {
  return this -> addExtension( new IFrameExtension( newExtension,
                                                    location,
                                                    percentage,
                                                    separator ) );
  }

/*------------------------------------------------------------------------------
| IFrameWindow::addExtension                                                   |
|                                                                              |
| Create and add new extensions.                                               |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::addExtension ( IWindow       *newExtension,
                                           Location       location,
                                           SeparatorType  separator )
  {
  return this -> addExtension( new IFrameExtension( newExtension,
                                                    location,
                                                    separator ) );
  }

/*------------------------------------------------------------------------------
| IFrameWindow::addExtension                                                   |
|                                                                              |
| If this is the first extension, create the collection.  Add the argument     |
| extension at end of sequence.                                                |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::addExtension ( IFrameExtension *extension )
  {
  IMODTRACE_DEVELOP( "IFrameWindow::addExtension" );
  Boolean
    alreadyUsed = false;

  IFrameExtensions
   *exts = this->extensions();

  if ( !exts )
     this -> setExtensions( exts = new IFrameExtensions );
  else
    {
    unsigned
      result = exts->numberOfElements();
    while ( result )
      if ( exts->elementAtPosition( result )->control()
                                    == extension->control() )
         {
         alreadyUsed = true;
         break;
         }
      else
         result--;
    }

  if ( alreadyUsed || extension->control()->handle() == this->clientHandle() )
     {
     ITHROWLIBRARYERROR( IC_FRAME_EXTENSION_IN_USE,
                         IErrorInfo::invalidParameter,
                         IException::recoverable );
     }

  exts -> addAsLast( extension );

  extension->control() -> setParent( this );
  fFrameWindowData->flags |= IFrameWindowData::needsUpdating;

  if ( this->isVisible() )
    this -> show();

  return *this;
  }

/*------------------------------------------------------------------------------
| IBase::Boolean IFrameWindow::isAnExtension                                   |
------------------------------------------------------------------------------*/
IBase::Boolean IFrameWindow::isAnExtension( const IWindow* window ) const
  {
  IFrameExtensions *exts = this->extensions();
  if ( exts )
     {
     unsigned
       result = exts->numberOfElements();
     while ( result )
       {
       if ( exts->elementAtPosition( result )->control() == window )
          return true;
       else
          result--;
       }
     }
  return false;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::findExtension                                                  |
|                                                                              |
| Iterate over extensions, looking for the one matching the argument IWindow   |
| pointer.  If not found, throw exception.                                     |
------------------------------------------------------------------------------*/
unsigned IFrameWindow::findExtension ( IWindow *window )
  {
  IFrameExtensions
   *exts = this->extensions();

  unsigned
    result = 0;

  if ( exts )
     {
     result = exts->numberOfElements();
     while ( result )
       if ( exts->elementAtPosition( result ) -> control() == window )
          break;
       else
          result--;
     }

  if ( !result )
     {
     ITHROWLIBRARYERROR( IC_FRAME_EXTENSION_NOT_FOUND,
                         IErrorInfo::invalidParameter,
                         IException::recoverable );
     }

  return result;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::removeExtension                                                |
|                                                                              |
| Search the extensions for argument window.  If found, remove this extension  |
| and delete it.  Otherwise, throw exception.                                  |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::removeExtension ( IWindow *extension,
                                              Boolean  updateDisplay )
  {
  IMODTRACE_DEVELOP( "IFrameWindow::removeExtension" );

  IFrameExtensions
   *exts = this->extensions();

  unsigned
    n = 0;

  if ( exts )
     {
     n = exts->numberOfElements();
     while ( n )
       if ( exts->elementAtPosition( n ) -> control() == extension )
          break;
       else
          n--;
     }

  if ( n )
     {
     IFrameExtension
      *ext = this->extensions() -> elementAtPosition( n );
     this->extensions() -> removeAtPosition( n );
     delete ext;

     if ( updateDisplay )
        {
        Boolean
          reShow = this -> isVisible();
        if ( reShow )
           IENABLEWINDOWUPDATE( this->handle(), false );
        if ( extension -> isValid() )
           {
           if ( extension -> hasFocus() )
              this->setFocus();
           extension -> setParent( IWindow::objectWindow() );
           }
        fFrameWindowData->flags |= IFrameWindowData::needsUpdating;
        if ( reShow )
           this -> show();
        }
     }

  return *this;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::setExtensionSize                                               |
|                                                                              |
| Locate extension and reset its size.                                         |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::setExtensionSize ( IWindow *extension,
                                               unsigned long widthOrHeight )
  {
  unsigned
    n = this -> findExtension( extension );
  if ( n )
     {
     this->extensions() -> elementAtPosition( n ) -> setSize( widthOrHeight );
     fFrameWindowData->flags |= IFrameWindowData::needsUpdating;
     if ( this->isVisible() )
        this->update();
     }
  return *this;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::setExtensionSize                                               |
|                                                                              |
| Convert width/height arg to unsigned long and use overloaded function.       |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::setExtensionSize ( IWindow *extension,
                                               int      widthOrHeight )
  {
  return this->setExtensionSize( extension, (unsigned long)widthOrHeight );
  }

/*------------------------------------------------------------------------------
| IFrameWindow::setExtensionSize                                               |
|                                                                              |
| Locate extension and reset its size.                                         |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::setExtensionSize ( IWindow *extension,
                                               double   widthOrHeight )
  {
  unsigned
    n = this -> findExtension( extension );
  if ( n )
     {
     this->extensions() -> elementAtPosition( n ) -> setSize( widthOrHeight );
     fFrameWindowData->flags |= IFrameWindowData::needsUpdating;
     if ( this->isVisible() )
        this->update();
     }
  return *this;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::useExtensionMinimumSize                                        |
|                                                                              |
| Locate extension and change its type to be based on its minimum size.        |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::useExtensionMinimumSize ( IWindow *extension )
{
  unsigned n = this->findExtension( extension );
  if ( n )
  {
     this->extensions()->elementAtPosition( n )->useMinimumSize();
     fFrameWindowData->flags |= IFrameWindowData::needsUpdating;
     if ( this->isVisible() )
        this->update();
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IFrameWindow::show                                                           |
|                                                                              |
| Update the frame (if necessary) then invoke inherited version.               |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::show ( Boolean showWindow )
  {
  if ( showWindow )
     this -> update();
  Inherited::show( showWindow );
  return *this;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::update                                                         |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::update ( )
  {
  if ( fFrameWindowData->flags & IFrameWindowData::needsUpdating )
     {
     fFrameWindowData->flags &= ~IFrameWindowData::needsUpdating;
#ifdef IC_WIN
     // Set the needsClientSized flag so we know to recalculate the stored
     // logical client rectangle during the update processing.
     fFrameWindowData->flags |= IFrameWindowData::needsClientSized;
     refresh();          // IC_NOTYET ... why do we need this?
     this -> sendEvent( IC_UM_UPDATEFRAME );
#endif
#ifndef IC_WIN
     this -> sendEvent( WM_UPDATEFRAME, ~0 );
#endif  // !IC_WIN
     }
  return *this;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::setDestroyOnClose                                              |
|                                                                              |
| Sets the destroy window flag on or off.                                      |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::setDestroyOnClose ( Boolean destroy )
  {
  if ( destroy )
     fFrameWindowData->flags |= IFrameWindowData::destroyOnClose;
  else
     fFrameWindowData->flags &= ~IFrameWindowData::destroyOnClose;

  return *this;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::willDestroyOnClose                                             |
|                                                                              |
| Returns the state of the destroy window flag.                                |
------------------------------------------------------------------------------*/
Boolean IFrameWindow::willDestroyOnClose ( ) const
  {
  if ( fFrameWindowData->flags & IFrameWindowData::destroyOnClose )
     return true;
  else
     return false;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::isFlashing                                                     |
------------------------------------------------------------------------------*/
Boolean IFrameWindow::isFlashing ( ) const
  {
#ifdef IC_WIN
  // Need internal flag to hold this since flashing in Windows must
  // be done manually using FlashWindow call and timer. - IC_NOTYET
  return false;
#else  // !IC_WIN
  unsigned short
    flags = this -> windowUShort( QWS_FLAGS );
  return ( flags & FF_FLASHWINDOW ) ? true : false;
#endif // IC_WIN
  }

/*------------------------------------------------------------------------------
| IFrameWindow::isMaximized                                                    |
------------------------------------------------------------------------------*/
Boolean IFrameWindow::isMaximized ( ) const
  {
  return( (this->style() & IFrameWindow::maximized.asUnsignedLong())
                 ?  true
                 :  false );
  }

/*------------------------------------------------------------------------------
| IFrameWindow::isMinimized                                                    |
------------------------------------------------------------------------------*/
Boolean IFrameWindow::isMinimized ( ) const
  {
  return( (this->style() & IFrameWindow::minimized.asUnsignedLong())
                 ?  true
                 :  false );
  }

/*------------------------------------------------------------------------------
| IFrameWindow::isModal                                                        |
------------------------------------------------------------------------------*/
Boolean IFrameWindow::isModal ( ) const
  {
  return fFrameWindowData->flags & IFrameWindowData::modal;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::addToWindowList                                                |
|                                                                              |
| Adds this frame's title as a window list entry.                              |
| Notes: Get title text from title bar.                                        |
|        If the frame already has a window list entry, change it.              |
|        Otherwise, add a new entry.                                           |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::addToWindowList ( )
{
   IMODTRACE_DEVELOP( "IFrameWindow::addToWindowList" );

   IWindowHandle                       // Get window title.
     hwndTitleBar = this -> handleFor( titleBar );

   if (hwndTitleBar == 0)
   {
      ITHROWGUIERROR( "" );
   } /* endif */

   ULONG uLen = IQUERYWINDOWTEXTLENGTH( hwndTitleBar );
   ++uLen;
   IString
     buffer( 0, MAXNAMEL + 1 );

   if ( uLen > 1 )
   {
      IQUERYWINDOWTEXT( this->handle(), MAXNAMEL + 1, buffer );
   } /* endif */
#ifdef IC_WIN
   // IC_NOTYET - Will depend on which Windows we ship with
#else   // !IC_WIN
   /*********************************************************/
   /* Add or change window list entry.                      */
   /*********************************************************/
   HSWITCH hsw = WinQuerySwitchHandle( this->handle(), 0 );
   if ( hsw == 0 )
   {
      SWCNTRL swctl;
      swctl.hwnd = this->handle();
      swctl.hwndIcon = this->icon();
      swctl.hprog = 0;
      swctl.idProcess = IApplication::current().id();
      swctl.idSession = 0;
      swctl.uchVisibility = SWL_VISIBLE;
      swctl.fbJump = SWL_JUMPABLE;
      strcpy( swctl.szSwtitle, buffer );

      // Create the switch entry.
      HSWITCH hswitch;
      hswitch = WinCreateSwitchEntry( IThread::current().anchorBlock(),
                                      &swctl );
      if ( hswitch == 0 )
      {
         ITHROWGUIERROR( "WinCreateSwitchEntry" );
      } /* endif */
   }
   else
   {
      SWCNTRL swctl;
      unsigned long rc = WinQuerySwitchEntry( hsw, &swctl );

      strcpy( swctl.szSwtitle, buffer );
      rc = WinChangeSwitchEntry( hsw, &swctl );

      if ( rc != 0 )
      {
         ITHROWGUIERROR( "WinChangeSwitchEntry" );
      } /* endif */
   } /* endif */
#endif  // !IC_WIN
   return *this;
}

/*------------------------------------------------------------------------------
| IFrameWindow::removeFromWindowList                                           |
|                                                                              |
| Remove this frame's window list entry.                                       |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::removeFromWindowList ( )
{
   IMODTRACE_DEVELOP( "IFrameWindow::removeFromWindowList" );

#ifdef IC_WIN
   // IC_NOTYET - Will depend on which Windows we ship with
#else   // !IC_WIN
   HSWITCH hswitch = WinQuerySwitchHandle( this->handle(), 0 );

   if ( hswitch != 0 )
   {
      if ( WinRemoveSwitchEntry( hswitch ) != 0 )
      {
         ITHROWGUIERROR( "WinRemoveSwitchEntry" );
      } /* endif */
   } /* endif */
#endif  // !IC_WIN
   return *this;
}

/*------------------------------------------------------------------------------
| IFrameWindow::handleFor                                                      |
|                                                                              |
| Return the IWindowHandle corresponding to a standard frame control.          |
| Notes: Determine the appropriate FID_* value for the requested control.      |
|        Then, issue handleWithId to get the corresponding window handle.      |
------------------------------------------------------------------------------*/
IWindowHandle IFrameWindow::handleFor ( const Style &standardControl ) const
  {
  IWindowHandle
    result(0);
#ifdef IC_WIN
   // IC_NOTYET - Will depend on which Windows we ship with
#else   // !IC_WIN
  unsigned long
    id = 0;
  if ( standardControl == hideButton
       ||
       standardControl == minimizeButton
       ||
       standardControl == maximizeButton )
    id = FID_MINMAX;
  else if ( standardControl == horizontalScroll )
    id = FID_HORZSCROLL;
  else if ( standardControl == verticalScroll )
    id = FID_VERTSCROLL;
  else if ( standardControl == menuBar )
    id = FID_MENU;
  else if ( standardControl == titleBar )
    id = FID_TITLEBAR;
  else if ( standardControl == systemMenu )
    id = FID_SYSMENU;
  if ( id )
    result = IWindow::handleWithParent( id, this->handle() );
#endif  // !IC_WIN
  return result;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::beginFlashing                                                  |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::beginFlashing ( )
  {
  Boolean bFlash = IFLASHWINDOW( this->handle(), true );
#ifndef IC_WIN
  if ( !bFlash )
     ITHROWGUIERROR( "WinFlashWindow" );
#endif
  return *this;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::endFlashing                                                    |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::endFlashing ( )
  {
  Boolean bFlash = IFLASHWINDOW( this->handle(), false );
#ifndef IC_WIN
  if ( !bFlash )
     ITHROWGUIERROR( "WinFlashWindow" );
#endif
  return *this;
  }

#ifndef IC_WIN
/*------------------------------------------------------------------------------
| IFrameWindow::shareParentDBCSStatus                                          |
|                                                                              |
| Causes a child frame to share the DBCS status control of its parent.         |
| Notes: Get DBCS environment info.                                            |
|        If in a DBCS environment, issue calls to share parent's DBCS status   |
|        area control.                                                         |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::shareParentDBCSStatus ( )
  {
  IMODTRACE_DEVELOP( "IFrameWindow::shareParentDBCSStatus" );

  COUNTRYCODE
    countryCode = { 0 };
  char
    buffer[12];

  // Get OS/2 DBCS info...
  DosQueryDBCSEnv( sizeof( buffer ), &countryCode, buffer );

  // See if DBCS environment is active...
  if ( buffer[0] || buffer[1] )
    // DBCS active, hook to parent DBCS status area...
    {
    IDynamicLinkLibrary
      dbcsLib( "pmnlsssc" );

    typedef int (_System * T1)( HAB, HWND, ULONG, PIMEMODE );

    IProcedureAddress< T1 >
      entryPt1( "WIN32DBCSIMECONTROL", dbcsLib );

    if ( entryPt1 )
      {
      IMODTRACE_DEVELOP( "Calling WIN32DBCSIMECONTROL" );
      IMEMODE
        imemode;
      imemode.lenIMEMODE = sizeof( imemode );
      imemode.fIMEMode = DBE_IMCTL_INHERIT_PFRAME;
      entryPt1( IThread::current().anchorBlock(),
                this->handle(),
                DBE_IMCTL_SET,
                &imemode );
      }

    typedef int (_System * T2)( HAB, HWND, ULONG, ULONG, PUSHORT );

    IProcedureAddress< T2 >
      entryPt2( "WIN32DBCSMODECONTROL", dbcsLib );

    if ( entryPt2 )
      {
      IMODTRACE_DEVELOP( "Calling WIN32DBCSMODECONTROL" );
      USHORT
        mode = DBE_MCTL_INHERIT_PFRAME;
        entryPt2( IThread::current().anchorBlock(),
                  this->handle(),
                  DBE_MCTL_SET,
                  DBE_MCTL_INPUTMODEREQUEST,
                  &mode );
      }
    }

  return *this;
  }
#endif  //!IC_WIN

/*------------------------------------------------------------------------------
| IFrameWindow::setDefaultOrdering                                             |
|                                                                              |
| Static function that sets the default placement of new windows.              |
------------------------------------------------------------------------------*/
void IFrameWindow::setDefaultOrdering ( IWindow::SiblingOrder ordering )
{
  IFrameWindow::siblingCreateOrder = ordering;

}

/*------------------------------------------------------------------------------
| IFrameWindow::defaultOrdering                                                |
|                                                                              |
| Static function that returns the default placement of new windows.           |
------------------------------------------------------------------------------*/
IWindow::SiblingOrder IFrameWindow::defaultOrdering ( )
{

  return IFrameWindow::siblingCreateOrder;
}

/*------------------------------------------------------------------------------
| IFrameWindow::tryToLoadDialog                                                |
|                                                                              |
| Attempt to load the dialog window and if it fails, calls initialize.         |
| Notes: Initialize PM before trying to load the resource.                     |
|        Attempt to load the dialog template.                                  |
|        Note: No attempt will be made to load a system menu icon, or          |
|              accelerator table for the dialog.                               |
|              It looks like WinNT will load a menu bar, while PM won't.       |
|        If that fails, try to create a plain frame by calling initialize().   |
|        Otherwise, "start" the dialog by calling "start".                     |
|        Some of these steps will be skipped, based on the value of the        |
|        passed-in FrameSource enumerator.                                     |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::tryToLoadDialog ( const IResourceId &resId,
                                              IWindow           *parent,
                                              IWindow           *owner,
                                              FrameSource        source )
{
  IMODTRACE_DEVELOP( "IFrameWindow::tryToLoadDialog" );

  if ( source != dialogResource  &&
       source != noDialogResource  &&
       source != tryDialogResource )
  {           // Validate the enumerator value.
     ITHROWLIBRARYERROR( IC_INVALIDENUMVALUE,
                         IErrorInfo::invalidParameter,
                         IException::recoverable );
  }

  IWindowHandle
    dialog( 0 );

  IThread::current().initializeGUI();

  IWindowHandle
    hParent = ( parent ) ? parent->handle() : IWindow::desktopWindow()->handle(),
    hOwner  = ( owner )  ? owner->handle()  : IWindowHandle( 0 );

  if ( source != noDialogResource )
  {           // Try to load dialog resource.
#ifdef IC_WIN
//   int tmp = DialogBoxParam( resId.resourceLibrary().handle(),
//                             MAKEINTRESOURCE( resId.id() ),
//                             hOwner,
//                             (DLGPROC)_pfnwpICDlgProc,
//                             (LPARAM)(void*)this );
//   return *this;
//
     dialog = ILOADDLG( hParent,
                        hOwner,
                        (DLGPROC)_pfnwpICDlgProc,
                        resId.resourceLibrary().handle(),
                        resId.id(),
                        0 );
#else   // !IC_WIN
     dialog = ILOADDLG( hParent,
                        hOwner,
                        WinDefDlgProc,
                        resId.resourceLibrary().handle(),
                        resId.id(),
                        0 );
#endif  // !IC_WIN
  }

  if ( dialog )
  {           // Dialog was loaded successfully.
     fFrameWindowData->flags |= IFrameWindowData::fromDlgTemplate;
     fFrameWindowData->flags &= ~IFrameWindowData::standardFrame;

#ifdef IC_WIN
     if ( hParent != hOwner )
     {
        // ILOADDLG (CreateDialogParm) ignores the parent and
        // appears to set it to the owner.
        if ( hParent != IWindow::desktopWindow()->handle() )
        {
           ISETPARENT( dialog, hParent, TRUE );
        }
     }

#endif  // IC_WIN
     this -> start( dialog );
  }
  else
  {           // Dialog was not loaded.
     IGUIErrorInfo loadDlgError( "ILOADDLG" );

     if ( source == noDialogResource  ||
          ( source == tryDialogResource  &&
#ifdef IC_WIN
            ( loadDlgError.errorId() == ERROR_RESOURCE_DATA_NOT_FOUND  ||
              loadDlgError.errorId() == ERROR_RESOURCE_TYPE_NOT_FOUND  ||
              loadDlgError.errorId() == ERROR_RESOURCE_NAME_NOT_FOUND )))
#else   // !IC_WIN
            ( loadDlgError.errorId() == PMERR_RESOURCE_NOT_FOUND )))
#endif  // !IC_WIN
     {         // Dialog was not found, try to create new frame window.
        // When the frame window rectangle is allowed to default, we use
        // the system recommended position without coordinate conversion.
        // This is so that resizing the frame will work as expected with
        // the application implemented in application coordinates.  To
        // implement this, we convert the application coordinates returned
        // by nextShellRect back to native before calling initialize.
        IRectangle initRect = IFrameWindow::nextShellRect();
        if ( ICoordinateSystem::isConversionNeeded() )
        {
           initRect = ICoordinateSystem::convertToNative(
                         initRect,
                         desktopWindow()->size() );
        }

        this -> initialize( resId,
                            IFrameWindow::defaultStyle(),
                            parent,
                            owner,
                            initRect,
                            0 );
     }
     else
     {         // Dialog was not found, error case.
        loadDlgError.throwError( IEXCEPTION_LOCATION() );
     }
  }

  return *this;
}

#ifdef IC_WIN
/*------------------------------------------------------------------------------
| _pfnwpICDlgProc                                                              |
|                                                                              |
| Dialog procedure for Windows to call.                                        |
------------------------------------------------------------------------------*/
BOOL CALLBACK _pfnwpICDlgProc( HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam )

{
   IMODTRACE_ALL( "_pfnwpICDlgProc" );
   ITRACE_ALL( IString( "msg=" ) +
               IString( msg ).d2x() +
               IString( " wparam=" ) +
               IString( wparam ).d2x() +
               IString( " lparam=" ) +
               IString( lparam ).d2x() );
   switch ( msg )
      {
      case WM_INITDLG:
         {
         return true;
         }
      case WM_COMMAND:
         {
         unsigned long notifyCode = HIWORD( wparam );
         IWindowHandle hwndCtl( (void*)lparam );
         if ( hwndCtl && ( notifyCode == BN_CLICKED ) )
            {
            // It is a control message indicating button clicked.
            // If it is an unwrappered pushbutton we need to generate
            // a WM_COMMAND to the dialog.  Wrappered pushbuttons handle
            // this with a default handler.
            unsigned long ulstyle = ISTYLEOF( hwndCtl ) & BS_PRIMARYSTYLES;
            if ( (ulstyle == BS_DEFPUSHBUTTON ) ||
                 (ulstyle == BS_PUSHBUTTON ) )
               {
               if ( 0 == IWindow::windowWithHandle( hwndCtl ) )
                  {
                  // Post the PM style WM_COMMAND.  We don't support
                  // the PM HELP or SYSCOMMAND styles on unwrappered buttons.
                  IWindowHandle hwndDlg = hwnd;
                  ITRACE_ALL( IString( "hwndDlg=" ) +
                              IString( hwndDlg.asUnsigned() ).d2x() +
                              IString( " hwnd=" ) +
                              IString( hwndCtl.asUnsigned() ).d2x() );
                  hwndDlg.postEvent( WM_COMMAND,
                                     IEventData( LOWORD( wparam ),
                                                 CMDSRC_PUSHBUTTON ),
                                     0 );
                  }  // unwrappered
               }  // is pushbutton
            }  // BN_CLICKED
         }  // WM_COMMAND
         break;
      default:
         break;
      }  // switch

   return false;
}
#endif // IC_WIN

/*------------------------------------------------------------------------------
| IFrameWindow::initialize                                                     |
|                                                                              |
| Specifies to use a common frame window construction.                         |
| Notes: Build arguments and call IFrameWindow::create.                        |
|        The style and frame creation flags come from the argument style.  If  |
|          shellPosition is desired, then determine the shell position and     |
|          turn off the flag (this results in better behavior for subsequent   |
|          size/position set/query calls).                                     |
|        If the IWindow::visible style is indicated, then store this fact,     |
|          create the window invisible, and do a show after the frame is       |
|          refreshed.  This is to overcome a quirk in the standard frame       |
|          whereby the first time the window is shown, the titlebar, etc. are  |
|          not drawn right.                                                    |
|        Likewise, if minimized/maximized are specified, then minimize/maximize|
|          the frame after construction.                                       |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::initialize ( const IResourceId &resId,
                                         const Style       &style,
                                         IWindow           *parent,
                                         IWindow           *owner,
                                         const IRectangle  &initRect,
                                         const char        *title )
  {
  IMODTRACE_DEVELOP( "IFrameWindow::initialize" );

  // Save the extended style to make sure we have a copy of it stored
  setExtendedStyle( extendedStyle() | style.asExtendedUnsignedLong() );

  // Do nothing if creation is deferred.
  if ( style & IFrameWindow::deferCreation )
  {
    return *this;
  }

  unsigned long
    frameId    = resId.id(),
    frameStyle = convertToGUIStyle( style );

  IWindowHandle
    hParent = ( parent ) ? parent->handle() : IWindow::desktopWindow()->handle(),
    hOwner  = ( owner )  ? owner->handle()  : IWindowHandle( 0 );

  IRectangle
    rect = initRect;

#ifdef IC_WIN
  // Create our window classes for the frame if necessary.
  ATOM frameClass = LOWORD( registerFrameClass( style, resId ));
#endif  // IC_WIN

#ifdef IC_WIN
  HINSTANCE
    hInst;
#else
  FRAMECDATA
    fcdata;
#endif

  Boolean
    isChild = ( hParent != IWindow::desktopWindow()->handle() );
#ifdef IC_WIN
  if ( isChild )
  {                     // This dialog is a child window.
     frameStyle |= WS_CHILD;
     // Remove conflicting styles.
  }
  else
  {                     // This dialog is not a child window.
     if ( style & IFrameWindow::titleBar )
     {
        frameStyle |= WS_OVERLAPPED;
     }
     else
     {
        frameStyle |= WS_POPUP;
     }
     // Remove conflicting styles.
  }
#endif  // IC_WIN

#ifdef IC_WIN
  hInst = (void*)resId.resourceLibrary().handle();
#else  // !IC_WIN
  fcdata.cb            = sizeof( FRAMECDATA );
  fcdata.flCreateFlags = convertToGUIStyle( style, true );
  fcdata.hmodResources = resId.resourceLibrary().handle(),
  fcdata.idResources   = (unsigned short)frameId;
#endif // IC_WIN

  // Find the parent frame window of this frame being created.
  IWindow* parentFrame = parent;
  while ( parentFrame  &&
          parentFrame->isFrameWindow() == false )
  {             // Go up the parent chain until a frame is found
     parentFrame = parentFrame->parent();
  }

#ifndef IC_WIN
  // If child window, and it can share its parent's DBCS status
  // area, then suppress its status area.
  if ( isChild  &&
       parentFrame  &&
       WinQueryWindowULong( parentFrame->handle(), QWL_STYLE ) &
                                                  FS_DBE_APPSTAT )
    {
    fcdata.flCreateFlags &= ~FCF_DBE_APPSTAT;
    // (But don't wipe out style which PM seems to ignore --
    // use the style as a flag that it uses a status area
    // for the above check.)
    }
#endif

  IString
    titleString;

  // See if title not provided...
  if ( style & IFrameWindow::titleBar && !title )
  {
     // Initialize GUI...
     IThread::current().initializeGUI();

     // Try to load from resource library...
     titleString = resId.resourceLibrary().tryToLoadString( resId.id() );
     if ( titleString.length() )
     {
        title = titleString;
     }
  }

  if ( style & IFrameWindow::shellPosition )
#ifdef IC_WIN
  // Emulate shell position functionality by setting up default
  // window rectangle if shell position style is set.
  {
     rect.moveTo( IPoint( CW_USEDEFAULT, CW_USEDEFAULT ) );
     rect.sizeTo( IPair( 0x7FFFFFFF, 0x7FFFFFFF ) );
  }
#else
    rect = IFrameWindow::nextShellRect();
#endif  // IC_WIN

  // Create the window...
  this -> create( frameId,
                  title,
#ifdef IC_WIN
                  frameStyle & ~WS_VISIBLE,
                  (const char*)frameClass,
#else
                  frameStyle & (unsigned long)~WS_VISIBLE
                             & (unsigned long)~FS_SHELLPOSITION,
                  WC_FRAME,
#endif
                  hParent,
                  hOwner,
                  rect,
#ifdef IC_WIN
                  // Use presParam field to pass instance handle to create()
                  0,
                  hInst );
#else
                  &fcdata,
                  0 );
#endif

  // If dialog background is specified, set the presentation parameter
  if ( style & IFrameWindow::dialogBackground )
  {
    this -> setBackgroundColor         ( IGUIColor(IGUIColor::dialogBgnd) );
    this -> setDisabledBackgroundColor ( IGUIColor(IGUIColor::dialogBgnd) );
  }

#ifdef IC_WIN
  // If one or both of the minimize/maximize buttons is missing and
  // there is still a system menu, disable the corresponding item in
  // the system menu.  Disabling rather than removing is what is recommended
  // in the SDK.
  if (frameStyle & WS_SYSMENU)
  {
     //-----------------------------------------------------
     // DEFECT 23816
     //-----------------------------------------------------
     // Disable maximize button and remove system menu size 
     // when dialogBorder style is selected. (copy code
     // implementation from portapak revisions.)
     //-----------------------------------------------------
     ISystemMenu sysMenu(this );

     if ( !(frameStyle & WS_MINIMIZEBOX) )
       sysMenu.disableItem( ISystemMenu::idMinimize );
     if ( !(frameStyle & WS_MAXIMIZEBOX) )
       sysMenu.disableItem( ISystemMenu::idMaximize );

     // if dialog frame, disable maximize menuitem.
     if ( exFrameStyle & WS_EX_DLGMODALFRAME)
       sysMenu.disableItem( ISystemMenu::idMaximize );
  
     // if we don't have a minimize or a maximize button then remove
     // the restore button ( Note that if the dialogBorder sytle is specified
     // then we don't have maximize button.
     if ( !(frameStyle & WS_MINIMIZEBOX) && 
          ( !(frameStyle & WS_MAXIMIZEBOX) ||
             (exFrameStyle & WS_EX_DLGMODALFRAME )) )
     {
       sysMenu.disableItem( ISystemMenu::idRestore );
     }
     // if dialog frame, or not resizable-border, disable size menuitem.
     if ( !(frameStyle & WS_THICKFRAME) ||
           (exFrameStyle & WS_EX_DLGMODALFRAME ) )
     {
       sysMenu.disableItem( ISystemMenu::idSize );
     }
     //-----------------------------------------------------
  }
#endif

  // Update the frame so controls are in place...
  fFrameWindowData->flags |= IFrameWindowData::needsUpdating;

#ifndef IC_WIN
  // Connect to parent's DBCS status field, if necessary...
  if ( isChild && (style & IFrameWindow::appDBCSStatus ))
    this -> shareParentDBCSStatus();
#endif

  // Handle max/min requests
  if ( style & IFrameWindow::minimized )
     this -> minimize();
  else if ( style & IFrameWindow::maximized )
     this -> maximize();

  // If requested, show the window...
  if ( style & IWindow::visible )
     this -> show();

  return *this;
  }


#ifdef IC_WIN
/*------------------------------------------------------------------------------
| IFrameWindow::registerFrameClass                                             |
------------------------------------------------------------------------------*/
unsigned long IFrameWindow::registerFrameClass ( const Style& style,
                                                 const IResourceId& resId )
{
  static ATOM
     frameClass = 0;
  if ( frameClass == 0 )
     {
     WNDCLASS
       wndclass;
     IString
       className = "IC_FRAME_CLASS" ;
     unsigned int
       classStyle = CS_DBLCLKS | CS_OWNDC;

     // Update class style to include byte align flag if requested
     if ( !( style & IFrameWindow::alignNoAdjust ) )
        {
        classStyle |= CS_BYTEALIGNWINDOW;
        }

     // Initialize instance handle and declare icon handle temp
     HINSTANCE
       hInstance = IApplication::current().userResourceLibrary().handle();
     HICON
       hIcon;

     // If minimized Icon style set, then load icon resource
     if ( style & IFrameWindow::minimizedIcon )
        {
        if ( !(hIcon = LoadIcon( hInstance, MAKEINTRESOURCE( resId.id() ))))
           {
           ITHROWGUIERROR( "LoadIcon" );
           }
        }
     // otherwise use default application icon
     else
        {
        hIcon = LoadIcon( (HINSTANCE)NULL, IDI_APPLICATION );
        }

     // If menu style set, then load menu resource
     LPCSTR lpszMenu = 0;
     if ( style & IFrameWindow::menuBar )
       lpszMenu = MAKEINTRESOURCE( resId.id() );

     // Initialize a window class structure with default information
     // in preparation of class registration
     wndclass.style       = classStyle;
     wndclass.lpfnWndProc = (WNDPROC)IDEFWINDOWPROC;
     wndclass.cbClsExtra  = 0;
     wndclass.cbWndExtra  = 4;          // 4 bytes for ID IGWL_ID
     wndclass.hInstance   = hInstance;
     wndclass.hIcon       = hIcon;
     wndclass.hCursor     = LoadCursor( (HINSTANCE)NULL, IDC_ARROW );
     wndclass.hbrBackground = (HBRUSH)(COLOR_BACKGROUND+1);
     wndclass.lpszMenuName  = lpszMenu;
     wndclass.lpszClassName = className;

     // Frame window class not already registered.
     if ( !(frameClass = RegisterClass( &wndclass )))
        {  // RegisterClass failed.
        ITHROWGUIERROR( "RegisterClass" );
        }
     else
        {
        AddAtom( className );
        }
     }   // if not registered
  return MAKELONG( frameClass, 0 );
}
#endif  // IC_WIN

/*------------------------------------------------------------------------------
| IFrameWindow::create                                                         |
------------------------------------------------------------------------------*/
IWindowHandle IFrameWindow::create ( unsigned long         id,
                                     const char           *title,
                                     unsigned long         style,
                                     const char           *windowClass,
                                     const IWindowHandle  &parent,
                                     const IWindowHandle  &owner,
                                     const IRectangle     &initRect,
                                     const void           *ctlData,
                                     const void           *presParams,
                                     IWindow::SiblingOrder ordering )
  {
  IMODTRACE_DEVELOP( "IFrameWindow::create" );

  IThread::current().initializeGUI();

#ifdef IC_PM
  // Don't pass text strings > 60 bytes to IWindow::create.  PM seems
  // to have a problem in this case (the end of the title is garbage).
  const char* createText = 0;
  if ( title  &&  strlen( title ) < MAXNAMEL )
  {
     createText = title;
  }
#endif

  IWindowHandle hwnd =
     Inherited::create(
#ifdef IC_WIN
                        (style & WS_CHILD) ? id : 0,
                        title,
#else   // !IC_WIN
                        id,
                        createText,
#endif  // !IC_WIN
                        style,
                        windowClass,
                        parent,
                        owner,
                        initRect,
                        ctlData,
                        presParams,
                        ordering );

  // OK: add frame handler (via start()).
  fFrameWindowData->flags |= IFrameWindowData::standardFrame;
  fFrameWindowData->flags &= ~IFrameWindowData::fromDlgTemplate;

#ifdef IC_WIN
  // set the extra window longword with the ID
  if (! (style & WS_CHILD) )
  {
    ISETIDOF( hwnd, id );
  }

  // Load accelerator table.  Once loaded, needs to be hooked into
  //   the window dispatch loop.
  // Load menu.
  // Load system menu icon.
#endif  // IC_WIN

  this->start( hwnd );

#ifdef IC_PM
  if ( createText != title )
  {      // Couldn't pass long title to IWindow::create, so add now.
     ITitle titleBar( this );
     titleBar.setTitleText( title );
  }
#endif // IC_PM

  return hwnd;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::start                                                          |
|                                                                              |
| Create a new frame handler, add it to this frame window, and initiate        |
| message handling for the window.                                             |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::start ( const IWindowHandle &hwnd )
  {
#ifdef IC_WIN
#ifdef IC_TRACE_ALL  // debugging code
  unsigned long ulstyle = ISTYLEOF( hwnd );
  ITRACE_ALL(IString("Frame Style=") + IString( ulstyle ).d2x() );
  HWND oldParent = IPARENTOF( hwnd );
  HWND oldOwner  = IOWNEROF( hwnd );
  ITRACE_ALL(IString( "GetParent=" ) + IString( (unsigned long)oldParent).d2x() );
  ITRACE_ALL(IString( "owner=" ) + IString( (unsigned long)oldOwner).d2x() );
  oldParent = (void*)GetWindowLong( hwnd, GWL_HWNDPARENT );
  ITRACE_ALL(IString( "GWL_HWNDPARENT=" ) + IString( (unsigned long)oldParent).d2x() );
#endif
#endif

  // Add the recoordination handler.
  IRecoordHandler::defaultHandler()->handleEventsFor( this );

  this -> addDefaultHandler( );
  this -> startHandlingEventsFor( hwnd );

  // If frame wasn't created via tryToLoadDialog or create, then
  // set type according to whether there's a client.
  if ( !( fFrameWindowData->flags & IFrameWindowData::fromDlgTemplate )
       &&
       !( fFrameWindowData->flags & IFrameWindowData::standardFrame ) )
     if ( this->clientHandle() )
        fFrameWindowData->flags |= IFrameWindowData::standardFrame;
     else
        fFrameWindowData->flags |= IFrameWindowData::fromDlgTemplate;

  return *this;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::addDefaultHandler                                              |
|                                                                              |
| Adds the default frame handler to a newly created frame.                     |
| Notes: If pHandler is null, create static handler.                           |
|        Add handler to frame and increment frame count.                       |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::addDefaultHandler ( )
  {
  this -> addHandler( defaultHandler );

  fFrameWindowData->flags |= IFrameWindowData::hasDefaultHandler;

  return *this;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::removeDefaultHandler                                           |
|                                                                              |
| Remove default handler and decrement frame count.  If frame count goes to    |
| zero, delete static handler and zero pointer.                                |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::removeDefaultHandler ( )
  {
  if ( fFrameWindowData->flags & IFrameWindowData::hasDefaultHandler )
     {
     this -> removeHandler( defaultHandler );

     fFrameWindowData->flags &= ~IFrameWindowData::hasDefaultHandler;
     }

  return *this;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::extensions                                                     |
|                                                                              |
| Return the "pointer to extensions" data member.                              |
------------------------------------------------------------------------------*/
IFrameExtensions *IFrameWindow::extensions ( ) const
  {
  return fFrameWindowData->pExtensions;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::setExtensions                                                  |
|                                                                              |
| Set the "pointer to extensions" data member.                                 |
------------------------------------------------------------------------------*/
IFrameWindow &IFrameWindow::setExtensions ( IFrameExtensions *extensions )
  {
  fFrameWindowData->pExtensions = extensions;
  return *this;
  }

/*------------------------------------------------------------------------------
| IFrameWindow::setLayoutDistorted                                             |
|                                                                              |
| If the minimum size of a frame extension that should be sized based on its   |
| minimum size has changed, update the size and position of all frame          |
| extensions and the client window.                                            |
------------------------------------------------------------------------------*/
IFrameWindow&
  IFrameWindow::setLayoutDistorted ( unsigned long layoutAttributesOn,
                                     unsigned long layoutAttributesOff )
{
  if ( layoutAttributesOn & IWindow::childMinimumSizeChanged )
  {                     // Minimum size of a child has changed.
     IFrameExtensions *exts = this->extensions();  // Get extensions.
     if ( exts )        // Check if a frame extension changed size.
     {                  // (Otherwise it may be the client window.)
        Boolean found = false;
        IFrameExtensions::Cursor cursor( *exts );
        for ( cursor.setToFirst();
              !found  &&  cursor.isValid();
              cursor.setToNext() )
        {
           IFrameExtension *ext = exts->elementAt( cursor );
           if ( ext->type() == IFrameExtension::minimumSize)
           {
              ISize
                sizMinimum = ext->control()->minimumSize(),
                sizActual = ext->control()->size();
              switch ( ext->location() )
              {
                case IFrameWindow::aboveClient:
                case IFrameWindow::belowClient:
                  // Horizontal frame extension.
                  if ( sizMinimum.height() != sizActual.height() )
                  {          // Frame extension not sized to min size.
                     found = true;
                  }
                  break;
                default:
                  // Vertical frame extension.
                  if ( sizMinimum.width() != sizActual.width() )
                  {          // Frame extension not sized to min size.
                     found = true;
                  }
                  break;
              }
           }
        }

        if ( found )
        {               // Need to update size of frame extension.
           fFrameWindowData->flags |= IFrameWindowData::needsUpdating;
           if ( this->isVisible() )
           {
              this->update();
           }
        }
     }
  }

  Inherited::setLayoutDistorted( layoutAttributesOn, layoutAttributesOff );
  return *this;
}

/*------------------------------------------------------------------------------
| IFrameWindow::setToolBarList                                                 |
|                                                                              |
| Stores the list of toolbars for a frame window.                              |
------------------------------------------------------------------------------*/
IFrameWindow& IFrameWindow::setToolBarList ( IToolBarList* toolBarList )
{
  fFrameWindowData->toolBarList = toolBarList;
  return *this;
}

/*------------------------------------------------------------------------------
| IFrameWindow::toolBarList                                                    |
|                                                                              |
| Returns the list of toolbars for a frame window.                             |
------------------------------------------------------------------------------*/
IToolBarList* IFrameWindow::toolBarList ( ) const
{
  return fFrameWindowData->toolBarList;
}

/*------------------------------------------------------------------------------
| IFrameWindow::backgroundColor                                                |
|                                                                              |
| Returns the background color for a frame window.                             |
------------------------------------------------------------------------------*/
IColor IFrameWindow::backgroundColor ( ) const
{
#ifndef IC_WIN
  IGUIColor guiColor( IGUIColor::windowBgnd );

  if ( (fFrameWindowData->flags & IFrameWindowData::fromDlgTemplate) ||
       ( this->usesDialogBackground() ))
  {
     guiColor = IGUIColor( IGUIColor::dialogBgnd );
  }
  return ( IWindow::color( PP_BACKGROUNDCOLOR, guiColor ));
#else
  return IColor( 0 );     // IC_NOTYET
#endif
}

/*------------------------------------------------------------------------------
| IFrameWindow::disabledBackgroundColor                                        |
|                                                                              |
| Returns the disabled background color for a frame window.                    |
------------------------------------------------------------------------------*/
IColor IFrameWindow::disabledBackgroundColor ( ) const
{
#ifndef IC_WIN
  IGUIColor guiColor( IGUIColor::windowBgnd );

  if ( (fFrameWindowData->flags & IFrameWindowData::fromDlgTemplate) ||
       ( this->usesDialogBackground() ))
  {
     guiColor = IGUIColor( IGUIColor::dialogBgnd );
  }
  return ( IWindow::color( PP_DISABLEDBACKGROUNDCOLOR, guiColor ));
#else
  return IColor( 0 );     // IC_NOTYET
#endif
}

/*------------------------------------------------------------------------------
| IFrameWindow::resetBackgroundColor                                           |
|                                                                              |
| Resets the background color by undoing a previous set.                       |
------------------------------------------------------------------------------*/
IFrameWindow& IFrameWindow::resetBackgroundColor ( )
{
  this -> setBackgroundColor( IGUIColor( IGUIColor::dialogBgnd ));
  return *this;
}

/*------------------------------------------------------------------------------
| IFrameWindow::resetDisabledBackgroundColor                                   |
|                                                                              |
| Resets the disabled background color by undoing a previous set.              |
------------------------------------------------------------------------------*/
IFrameWindow& IFrameWindow::resetDisabledBackgroundColor ( )
{
  this -> setDisabledBackgroundColor( IGUIColor( IGUIColor::dialogBgnd ));
  return *this;
}

/*------------------------------------------------------------------------------
| IFrameWindow::usesDialogBackground                                           |
|                                                                              |
| Returns if the frame has the FCF_AUTOICON frame creation flag set, by        |
| testing for a by-product of setting FCF_AUTOICON.                            |
------------------------------------------------------------------------------*/
Boolean IFrameWindow::usesDialogBackground ( ) const
{
#ifndef IC_WIN
  return (this->windowUShort( QWS_FLAGS ) & 0x100) ? true : false;
#else
  return false;         // IC_NOTYET
#endif
}

/*------------------------------------------------------------------------------
| IFrameHandler::numberOfExtensions                                            |
|                                                                              |
| Return the number of frame extensions that the argument frame possesses.     |
------------------------------------------------------------------------------*/
unsigned long
  IFrameHandler::numberOfExtensions ( const IFrameWindow* pFrame ) const
  {
  unsigned long
    result = 0;

  // Get pointer to frame extensions collection...
  const IFrameExtensions
   *extensions = pFrame->extensions();

  // If there are any, result is number of elements in collection...
  if ( extensions )
     result = extensions->numberOfElements();

  return result;
  }

#ifndef _IFRAMEVT_
  #include <iframevt.hpp>
#endif

#ifndef _ISWP_
  #include <iswp.hpp>
#endif

/*------------------------------------------------------------------------------
| IFrameHandler::format                                                        |
|                                                                              |
| Fills an array of window position structures, one for each frame control.    |
| This function does this for the frame extensions that have been added to the |
| frame using IFrameWindow::addExtension.                                      |
------------------------------------------------------------------------------*/
Boolean IFrameHandler::format ( IFrameFormatEvent &fmtEvent )
  {
  RECTL
    temp = { 0, 0 };

  IFrameFormatEvent
    localEvent = fmtEvent;
  if ( fmtEvent.parameter2() == 0 )
     {
     localEvent = IEvent( fmtEvent.handle(),
                          fmtEvent.eventId(),
                          fmtEvent.parameter1(),
                          &temp );
     }

  // Get count of standard controls (and thus index of extensions)...
  unsigned long
    controls = this -> defaultProcedure( localEvent );

  fmtEvent.setResult( this->positionExtensions( localEvent, controls ));

  return true;
  }

/*------------------------------------------------------------------------------
| IFrameHandler::positionExtensions                                            |
|                                                                              |
| Called from format.  This function will position and size the frame          |
| extensions after the size and position of the standard frame controls        |
| have been set in the ISWPArray in the IFrameFormatEvent.                     |
------------------------------------------------------------------------------*/
unsigned long IFrameHandler::positionExtensions( IFrameFormatEvent &formatEvent,
                                                 unsigned long numStdControls )
  {
  IMODTRACE_DEVELOP( "IFrameHandler::positionExtensions" );

  unsigned long
    index( numStdControls );

  // Construct "array of SWPs" object...
  ISWPArray
   swpArray( formatEvent.swpArray(),
             numStdControls + this->numberOfExtensions( formatEvent.frame() ) );

  // Iterate over the frame extensions (if there are any)...
  IFrameExtensions
   *extensions = formatEvent.frame()->extensions();
  if ( extensions )
     {
     IFrameExtensions::Cursor
       cursor( *extensions );

     for ( cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
       {
       // Access this extension...
       IFrameExtension
        &ext = *( extensions->elementAt( cursor ) );

       ISWP
        &extSWP = swpArray[ index++ ];

       // Locate standard control that this extension is attached to...
       unsigned long
         baseId = ext.attachedToId();

       unsigned
         baseIndex = swpArray.indexOf( baseId );

       // See if extension will show...
       if ( baseId == FID_CLIENT
            ||
            baseIndex < swpArray.size() )
          {
          ISWP
            clientSWP,
           &baseSWP = ( baseIndex < swpArray.size() ) ?
                        swpArray[ baseIndex ]
                      :
                        clientSWP;

          // Dummy swp has "empty" client area dimensions...
          clientSWP
            .setSize( formatEvent.clientRect().size() )
            .setPosition( formatEvent.clientRect().minXMinY() );

          // Update extension and base SWPs per location...
          extSWP = ext.attachTo( baseSWP );

          // Remember client adjustment (if one was made)...
          if ( baseId == FID_CLIENT )
            formatEvent.setClientRect( IRectangle( baseSWP.position(),
                                                   baseSWP.size() ) );
          }
       else
          // Base is absent, hide this extension...
          extSWP.setSize( ISize( 0, 0 ) );

       // Set hwnd of associated control...
       extSWP.hwnd = ext.control()->handle();
       // Always set flags to move/size/noadjust...
#ifdef IC_WIN
       extSWP.flags() = SWP_NOOWNERZORDER;  // Default is move/size.
#else  // !IC_WIN
       extSWP.flags() = SWP_MOVE | SWP_SIZE | SWP_NOADJUST;
#endif // !IC_WIN
       // Always place on "top" since no frame controls overlap...
       extSWP.behind = HWND_TOP;

       } // extension loop

     } // has extensions
  return index;
  }

/*------------------------------------------------------------------------------
| IFrameHandler::draw                                                          |
|                                                                              |
| Called when the frame window needs to be redrawn.  This function draws the   |
| separator lines between frame extensions and returns false.                  |
| Notes: Iterate extensions, drawing each via IFrameExtension::drawSeparator.  |
------------------------------------------------------------------------------*/
Boolean IFrameHandler::draw ( IFrameEvent &frameEvent )
  {
  IMODTRACE_DEVELOP( "IFrameHandler::draw" );

  // Get the basic frame drawing done...
  frameEvent.setResult( this -> defaultProcedure( frameEvent ) );

  // Iterate over the frame extensions (if there are any)...
  IFrameExtensions
   *extensions = frameEvent.frame()->extensions();

  if ( extensions )
     {
     // Get presentation space handle for drawing...
     IPresSpaceHandle
       psh = frameEvent.frame()->presSpace();

     IFrameExtensions::Cursor
       cursor( *extensions );

     // Iterate extensions, drawing separator for each...
     for ( cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
       // Draw separator for next extension...
       extensions->elementAt( cursor ) -> drawSeparator( psh );

     // Release presentation space...
     frameEvent.frame()->releasePresSpace( psh );
     }

  return true;
  }

/*------------------------------------------------------------------------------
| IFrameHandler::calcRect                                                      |
|                                                                              |
| Iterate extensions, expand frame rectangle by height and width of            |
| each extension if given a client rectangle, shrink client rectangle          |
| by height and width of each extension if given a frame rectangle.            |
------------------------------------------------------------------------------*/
Boolean IFrameHandler::calcRect ( IFrameEvent &frameEvent )
  {
  IMODTRACE_DEVELOP( "IFrameHandler::calcRect" );

  Boolean result = false;

  // Going from frame to client.
  if ( frameEvent.parameter2().number1() )
     {
#ifdef IC_WIN
     // In windows defaultProcedure will ignore and return false...
     // We instead get the client rectangle as known by Windows and
     // adjust it to screen coordinates.
     RECT* pRc = (RECT*)frameEvent.parameter1().asUnsignedLong();
     Boolean rc = GetClientRect( frameEvent.handle(), pRc );
     frameEvent.setResult( rc );
//     if ( rc )
//        {
//        // get result in screen coordinates
//        MapWindowPoints( frameEvent.handle(), HWND_DESKTOP, (POINT*)pRc, 2 );
//        }
#endif
#ifndef IC_WIN
     // Call default window procedure to subtract base controls
     frameEvent.setResult( this->defaultProcedure( frameEvent ));
#endif
     // Don't call default window procedure.
     result = true;
     }

  // Iterate over the frame extensions (if there are any)
  // adding or subtracting the frame extensions.
  IFrameExtensions
   *extensions = frameEvent.frame()->extensions();

  if ( extensions )
     {
     PRECTL     pRc = (PRECTL)frameEvent.parameter1().asUnsignedLong();
     IRectangle rectl( *pRc );
     IFrameExtensions::Cursor cursor( *extensions );

     // Frame --> client.
     if ( frameEvent.parameter2().number1() )
        {
        for ( cursor.setToLast(); cursor.isValid(); cursor.setToPrevious() )
           {
           // Access this extension...
           IFrameExtension
            &ext = *( extensions->elementAt( cursor ) );

           if ( ext.attachedToId() == FID_CLIENT )
              {
              rectl = ext.baseRectFor( rectl );
              }
           }
        }
     // Client --> frame.
     else
        {
        for ( cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
          {
          // Access this extension...
          IFrameExtension
           &ext = *( extensions->elementAt( cursor ) );

          if ( ext.attachedToId() == FID_CLIENT )
             {
             rectl = ext.totalRectFor( rectl );
             }
          }
#ifdef   IC_WIN
        // Do the stuff that the PM default procedure does.
        // Compute the rectangle for the frame in screen coordinates
        // including the standard controls present.
        ITRACE_ALL( IString("Client rect with extensions=") + rectl.asString() );
        RECTL winRect = rectl.asRECTL();
        // Get result in screen coordinates.
        MapWindowPoints( frameEvent.handle(), HWND_DESKTOP, (POINT*)&winRect, 2 );

        unsigned long ulStyle = ISTYLEOF( frameEvent.handle() );
        Boolean  hasMenu = false;
        if ( GetMenu( frameEvent.handle() ) )
           hasMenu = true;
        AdjustWindowRect( (RECT*)&winRect, ulStyle, hasMenu );
#ifdef IC_NOTYET
        // Need to account for title bar and border here.
#endif
        rectl = IRectangle( winRect );
        ITRACE_ALL( IString( "Frame rectangle=" ) + rectl.asString() );
#endif   //IC_WIN
        }
     *pRc = rectl.asRECTL();
     } // has extensions

  return result;
  }

#ifdef   IC_WIN
/*------------------------------------------------------------------------------
| IBase::Boolean icalcFrameRect                                                |
------------------------------------------------------------------------------*/
IBase::Boolean icalcFrameRect ( IWindowHandle  hwnd,
                                RECTL*         rect,
                                IBase::Boolean frameToClient )
{
   IMODTRACE_ALL( "icalcFrameRect" );
   RECT* prect = (RECT*)rect;   // RECT and RECTL identical in windef.h.
   unsigned long style = ISTYLEOF( hwnd );
   IBase::Boolean result = false;
   if (( style & (WS_CHILD | WS_POPUP) ))
      {
      // Not a frame window.
      if ( frameToClient )
         {
         result = GetClientRect( hwnd, prect );
//         if ( result )
//            {
//            // Get result in screen coordinates.
//            MapWindowPoints( hwnd, HWND_DESKTOP, (POINT*)prect, 2 );
//            }
         }
      else
         {
#ifdef IC_NOTYET
         // Need to be able to detect menu if one is possible with WS_POPUP.
#endif
         result = AdjustWindowRect( prect, style, false );
         }
      }
   else
      {
      // Is a frame window.  We need to take into account frame extensions.
      // Do this by sending a user message to the frame handler.
      if ( frameToClient )
         {
         // Set prect to the real window rectangle.
         GetWindowRect( hwnd, prect );
         }
      result = hwnd.sendEvent( IC_UM_CALCFRAMERECT,
                               IEventData( prect ),
                               IEventData( frameToClient ) );
      }
   ITRACE_ALL( IString( "rect=" ) + IRectangle( *rect ).asString() );
   return result;
}
#endif   //IC_WIN
