/*******************************************************************************
* FILE NAME: ientryfd.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in ientryfd.hpp.                                                           *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1995       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
// Priority INT_MIN (-2147483647 - 1) + 1024 + 512
#pragma priority( -2147482112 )

extern "C" {
  #define INCL_NLS                     //ES_ANY, ES_SBCS, ES_DBCS, ES_MIXED
  #define INCL_WINCLIPBOARD
  #define INCL_WINENTRYFIELDS
  #define INCL_WINERRORS               //PMERR_HEAP_MAX_SIZE_REACHED
  #define INCL_WININPUT                //WM_CHAR, etc.
  #define INCL_WINMESSAGEMGR           //WNDPARAM struct
  #define INCL_WINWINDOWMGR
  #define INCL_GPIPRIMITIVES
  #define INCL_WINSYS
  #define INCL_GPIBITMAPS
  #include <iwindefs.h>
}

#include <ientryfd.hpp>
#ifdef IC_PMWIN   // DEFECT 4624 : required for PM and WIN
  #include <iefpriv.hpp>
  #include <ikeyevt.hpp>
#endif
#include <icconst.h>
#include <icolor.hpp>
#include <iexcept.hpp>
#include <ifont.hpp>
#include <ihandle.hpp>
#include <inotifev.hpp>
#include <irect.hpp>
#include <ireslib.hpp>
#include <istring.hpp>
#include <ithread.hpp>
#include <itrace.hpp>
#include <iwcname.hpp>

// Segment definitions.
#ifdef IC_PAGETUNE
  #define _IENTRYFD_CPP_
  #include <ipagetun.h>
#endif

//-----------------------------------------------------------------------
// DEFECT 4624 : required for PM and WIN
//-----------------------------------------------------------------------
#ifdef IC_PMWIN
#ifndef IES_AUTOTAB
   #define IES_AUTOTAB            0x00000008ul   //IEntryField::autoTab
#endif
#endif // IC_PMWIN
//-----------------------------------------------------------------------

/*------------------------------------------------------------------------------
| Public entry field styles.                                                   |
------------------------------------------------------------------------------*/
#pragma data_seg(ICLStaticConst)
const IEntryField::Style
  IEntryField::readOnly          = ES_READONLY,
  IEntryField::leftAlign         ( 0, IES_LEFTALIGN ),
  IEntryField::centerAlign       = ES_CENTER,
  IEntryField::rightAlign        = ES_RIGHT,
  IEntryField::autoScroll        = ES_AUTOSCROLL,
  IEntryField::margin            = ES_MARGIN,
  IEntryField::unreadable        = ES_UNREADABLE,
#ifdef IC_PMWIN   // DEFECT 4624 : use extendedStyle for PM and WIN
  IEntryField::autoTab           ( 0, IES_AUTOTAB ),
#else
  IEntryField::autoTab           = ES_AUTOTAB,
#endif
#ifdef IC_MOTIFPM
  IEntryField::command           = ES_COMMAND,
  IEntryField::anyData           ( 0, IES_ANYDATA ),
  IEntryField::sbcsData          = ES_SBCS,
  IEntryField::dbcsData          = ES_DBCS,
  IEntryField::mixedData         ( 0, IES_MIXEDDATA ),
#endif
  IEntryField::border3D          ( 0, IWS_BORDER3D ),
#ifdef IC_WIN
  IEntryField::upperCase         = ES_UPPERCASE,         //simulate under PM?
  IEntryField::lowerCase         = ES_LOWERCASE,         //simulate under PM?
#endif
  IEntryField::classDefaultStyle ( (WS_VISIBLE    |
                                    ES_MARGIN     |
                                    ES_AUTOSCROLL),
                                                    (IES_LEFTALIGN |
                                                     IES_ANYDATA) );

const long IEntryField::end = -2;
#pragma data_seg()


/*------------------------------------------------------------------------------
| Default style for new objects (initial value).                               |
------------------------------------------------------------------------------*/
#pragma data_seg(ICLNonConst)
IEntryField::Style
 IEntryField::currentDefaultStyle ( (WS_VISIBLE    |
                                     ES_MARGIN     |
                                     ES_AUTOSCROLL),
                                                     (IES_LEFTALIGN |
                                                      IES_ANYDATA) );
#pragma data_seg()

#ifdef IC_PMWIN   // DEFECT 4624 : required for PM and WIN
/*------------------------------------------------------------------------------
| IEntryHandler                                                                |
|                                                                              |
| This class wrappers an object pointer and destroys the pointer to the object |
| when the wrapper is destructed.                                              |
------------------------------------------------------------------------------*/
static struct DefaultHandler {
  operator IEntryHandler* ();
IEntryHandler
 *operator -> ()
    {
    return *this;
    }
 ~DefaultHandler ( )
    {
    if (ptr)
      {
      delete ptr;
      ptr = 0;
      }
    }
IEntryHandler
 *ptr;
} defaultHandler;

DefaultHandler::operator IEntryHandler* ( )
  {
  if ( !ptr )
    ptr = new IEntryHandler;
  return ptr;
  }
#endif //IC_PMWIN

/*------------------------------------------------------------------------------
| IEntryField::IEntryField                                                     |
|                                                                              |
| Default constructor (needed by combo box).                                   |
------------------------------------------------------------------------------*/
IEntryField::IEntryField()
  : fEntryFieldData( 0 ),
    eReadStatus(doesntMatter),
    queryInsMode(false)
{
}


/*------------------------------------------------------------------------------
| IEntryField::IEntryField                                                     |
|                                                                              |
| Constructor to create an entry field control on a standard window.           |
------------------------------------------------------------------------------*/
IEntryField::IEntryField ( unsigned long ulId,
                           IWindow* parent,
                           IWindow* pwndOwner,
                           const IRectangle& rectInit,
                           const Style& style )
  : fEntryFieldData( 0 ),
    eReadStatus(doesntMatter),
    queryInsMode(false)
{
  // assertions on input parms
  IASSERTPARM(parent!=0);

  // Save the extended style to make sure we have a copy of it stored
  setExtendedStyle( extendedStyle() | style.asExtendedUnsignedLong() );

  /****************************************************************/
  /* Borland has bug with embedded ? operator so below is needed. */
  /****************************************************************/
#ifdef __BORLANDC__
  IWindowHandle owner;
  if (pwndOwner == 0)
    owner = IWindowHandle(0);
  else
    owner = pwndOwner->handle();
#else
  IWindowHandle owner = (pwndOwner == 0) ? IWindowHandle(0)
                                         : pwndOwner->handle();
#endif

  IWindowHandle whEntryField =
         this -> create( ulId,
                         0,
                         convertToGUIStyle( style ),
                         WC_ENTRYFIELD,
                         parent->handle(),
                         owner,
                         rectInit,
                         0,
                         0 );

#ifdef IC_PMWIN   // DEFECT 4624 : required for PM and WIN
  this->addHandler( defaultHandler );
#endif
  startHandlingEventsFor(whEntryField);
}


/*------------------------------------------------------------------------------
| IEntryField::IEntryField                                                     |
|                                                                              |
| Constructor to instantiate an entry field object for a dialog template.      |
------------------------------------------------------------------------------*/
IEntryField::IEntryField( unsigned long ulId,
                          IWindow*      parent )
  : fEntryFieldData( 0 ),
    eReadStatus(doesntMatter),
    queryInsMode(false)
{
  setAutoDestroyWindow(false);

#ifdef IC_PM
  // Work-around for window id breakage in Warp GA for the child
  // entry field of a spin button control (it should be the window
  // id of the parent spin button, but is 1).
  if (parent  &&
      parent->id() == ulId  &&
      IWindow::handleWithParent( ulId, parent->handle() ) == 0  &&
      IWindowClassName ( parent->handle() ) == WC_SPINBUTTON)
  {                // Can't find a child entry field of a spin button.
     ulId = 1;     // Replace the id with the one Warp GA uses.
  }
#endif  // IC_PM

#ifdef IC_PMWIN   // DEFECT 4624 : required for PM and WIN
  this->addHandler( defaultHandler );
#endif
  startHandlingEventsFor(ulId, parent);

  hasChanged();                //Reset the changed status to FALSE,
                               //status is set to TRUE on creation.
}


/*------------------------------------------------------------------------------
| IEntryField::IEntryField                                                     |
|                                                                              |
| Constructor to instantiate from an existing entry field control.             |
------------------------------------------------------------------------------*/
IEntryField::IEntryField( const IWindowHandle& wh )
  : fEntryFieldData( 0 ),
    eReadStatus(doesntMatter),
    queryInsMode(false)
{
  setAutoDestroyWindow(false);
#ifdef IC_PMWIN   // DEFECT 4624 : required for PM and WIN
  this->addHandler( defaultHandler );
#endif
  startHandlingEventsFor(wh);
}


/*------------------------------------------------------------------------------
| IEntryField::~IEntryField                                                    |
|                                                                              |
| Empty destructor here for page tuning.                                       |
------------------------------------------------------------------------------*/
IEntryField::~IEntryField()
{}


/*------------------------------------------------------------------------------
| IEntryField::defaultStyle                                                    |
|                                                                              |
| Return the default style for new entry field objects.                        |
------------------------------------------------------------------------------*/
IEntryField::Style IEntryField::defaultStyle()
{
  return( currentDefaultStyle );
}


/*------------------------------------------------------------------------------
| IEntryField::setDefaultStyle                                                 |
|                                                                              |
| Replace the default style for new entry field objects.                       |
------------------------------------------------------------------------------*/
void IEntryField::setDefaultStyle( const Style& style )
{
  currentDefaultStyle = style;
}


/*------------------------------------------------------------------------------
| IEntryField::convertToGUIStyle                                               |
|                                                                              |
| Returns base style for the control by default, or extended style if          |
| extended flag (bExtOnly) is set.                                             |
------------------------------------------------------------------------------*/
unsigned long IEntryField::convertToGUIStyle( const IBitFlag& guiStyle,
                                              Boolean bExtOnly ) const
{
  // Obtain the style from the class (ITextControl) that we inherit from.
  unsigned long ulStyle = Inherited::convertToGUIStyle( guiStyle, bExtOnly );

  if (bExtOnly)
  {
    // Use mask to only return extended styles in the user defined range.
    ulStyle |= extendedStyle() & IS_EXTMASK;
  }
  else
  {
    // Let the clean ES_ styles, flow thru.
    ulStyle |= guiStyle.asUnsignedLong() & IES_MASK;

#ifdef IC_MOTIFPM
    // ES_MIXED is the result of (ES_SBCS | ES_DBCS).  Therefore, we
    // define a unique bit for it in the extended styles.  Map the
    // extended style definition, back to the system definition,
    // prior to creating the control.
    unsigned long ulTmp = extendedStyle() & IES_EXTMASK;

    if (ulTmp & mixedData.asExtendedUnsignedLong())
    {
       ulStyle |= ES_MIXED;
    }
#endif
#ifdef IC_WIN
    // For Windows it is always a child and must be an MLE to get alignment,
    // and other required functions to work.
    ulStyle |= WS_CHILD | ES_MULTILINE;
#endif
  }

  return( ulStyle );
}


/*------------------------------------------------------------------------------
| IEntryField::setAlignment                                                    |
|                                                                              |
| Replaces the text alignment style of the entry field while preserving the    |
| marked area and cursor position.                                             |
------------------------------------------------------------------------------*/
IEntryField& IEntryField::setAlignment( Alignment alignType )
{
#ifndef IC_WIN
  unsigned long ulStyle = style();
  unsigned long ulOldStyle = ulStyle;

  //Remove existing alignment
  ulStyle &= ~(centerAlign.asUnsignedLong() | rightAlign.asUnsignedLong());

  switch (alignType)
  {
    case left:
      // No additional processing required
    break;

    case right:
      ulStyle |= rightAlign.asUnsignedLong();
    break;

    case center:
      ulStyle |= centerAlign.asUnsignedLong();
    break;

    default:
      ITHROWLIBRARYERROR(IC_INVALIDENUMVALUE,
                         IErrorInfo::invalidParameter,
                         IException::recoverable);
    break;
  }

  if (ulStyle != ulOldStyle)
  {
    setStyle(ulStyle);                   //Change style
    refresh();                           //Force repaint
  }
#endif
  return *this;
}


/*------------------------------------------------------------------------------
| IEntryField::alignment                                                       |
|                                                                              |
| Returns the current alignment value for the entry field.                     |
------------------------------------------------------------------------------*/
IEntryField::Alignment  IEntryField::alignment() const
{
  unsigned long ulStyle = style();
  if (ulStyle & rightAlign.asUnsignedLong())
      return( right );

  if (ulStyle & centerAlign.asUnsignedLong())
      return( center );

  return( left );
}


/*------------------------------------------------------------------------------
| IEntryField::setCharType                                                     |
|                                                                              |
| Replaces the text data type style of the entry field while preserving the    |
| marked area and cursor position.                                             |
------------------------------------------------------------------------------*/
IEntryField& IEntryField::setCharType( CharType charType )
{
  unsigned long ulStyle = style();
  unsigned long ulOldStyle = ulStyle;
#ifdef IC_PM
    ulStyle &= ~(sbcsData.asUnsignedLong() | dbcsData.asUnsignedLong() |
                 ES_MIXED);
#endif

  switch (charType)
  {
#ifdef IC_PM
     case sbcs:
       ulStyle |= sbcsData.asUnsignedLong();
     break;

     case dbcs:
       ulStyle |= dbcsData.asUnsignedLong();
     break;

     case any:
      // No additional processing required
     break;

     case mixed:
       ulStyle |= ES_MIXED;
     break;
#endif
     default:
      ITHROWLIBRARYERROR(IC_INVALIDENUMVALUE,
                         IErrorInfo::invalidParameter,
                         IException::recoverable);
     break;
  }

  if (ulStyle != ulOldStyle)
  {
    setStyle(ulStyle);                   //Change style
    refresh();                           //Force refresh
    notifyObservers(INotificationEvent(IEntryField::characterTypeId,
                                       *this, true, (void *)charType));
  }
  return( *this );
}


/*------------------------------------------------------------------------------
| IEntryField::charType                                                        |
|                                                                              |
| Returns the text contents type.                                              |
------------------------------------------------------------------------------*/
IEntryField::CharType  IEntryField::charType() const
{
#ifdef IC_PM
  unsigned long ulStyle = style();
  if ((ulStyle & sbcsData.asUnsignedLong()) &&
      (ulStyle & dbcsData.asUnsignedLong()))
     return( mixed );

  if (ulStyle & dbcsData.asUnsignedLong())
     return( dbcs );

  if (ulStyle & sbcsData.asUnsignedLong())
     return( sbcs );
#endif
  return( any );
}


/*------------------------------------------------------------------------------
| IEntryField::enableAutoScroll                                                |
|                                                                              |
| Sets the autoScroll style on the entry field control.                        |
------------------------------------------------------------------------------*/
IEntryField& IEntryField::enableAutoScroll( Boolean enable )
{
  unsigned long ulStyle = style();
  unsigned long ulOldStyle = ulStyle;

  if (enable)
  {
    ulStyle |= autoScroll.asUnsignedLong();
  }
  else
  {
    ulStyle &= ~autoScroll.asUnsignedLong();
  }

  if (ulStyle != ulOldStyle)
  {
    setStyle(ulStyle);                   //Change style
    refresh();                           //Force refresh
  }
  return *this;
}


/*------------------------------------------------------------------------------
| IEntryField::disableAutoScroll                                               |
------------------------------------------------------------------------------*/
IEntryField& IEntryField::disableAutoScroll()
{
  enableAutoScroll(false);
  return( *this );
}


/*------------------------------------------------------------------------------
| IEntryField::isAutoScroll                                                    |
------------------------------------------------------------------------------*/
Boolean IEntryField::isAutoScroll() const
{
  return( (style() & autoScroll.asUnsignedLong()) ? true : false );
}


/*------------------------------------------------------------------------------
| IEntryField::enableMargin                                                    |
------------------------------------------------------------------------------*/
IEntryField& IEntryField::enableMargin ( Boolean enable )
{
  unsigned long ulStyle = style();
  unsigned long ulOldStyle = ulStyle;

  if (enable)
  {
    ulStyle |= margin.asUnsignedLong();
  }
  else
  {
    ulStyle &= ~margin.asUnsignedLong();
  }

  if (ulStyle != ulOldStyle)
  {
    setStyle(ulStyle);                   //Change style
    refresh();                           //Force refresh
  }
  return *this;
}


/*------------------------------------------------------------------------------
| IEntryField::disableMargin                                                   |
------------------------------------------------------------------------------*/
IEntryField& IEntryField::disableMargin()
{
  enableMargin(false);
  return( *this );
}


/*------------------------------------------------------------------------------
| IEntryField::isMargin                                                        |
------------------------------------------------------------------------------*/
Boolean IEntryField::isMargin() const
{
  return( (style() & margin.asUnsignedLong()) ? true : false );
}


#ifdef IC_PM
/*------------------------------------------------------------------------------
| IEntryField::enableCommand                                                   |
|                                                                              |
| Enables or disbles the command style on an entry field control.              |
------------------------------------------------------------------------------*/
IEntryField& IEntryField::enableCommand( Boolean enable )
{
  unsigned long ulStyle = style();
  Boolean styleChanged = false;

  if (enable)
  {
    if (!(ulStyle & command.asUnsignedLong()))
    {
      ulStyle |= command.asUnsignedLong();
      styleChanged = true;
    }
  }
  else if (ulStyle & command.asUnsignedLong())
  {
    ulStyle &= ~command.asUnsignedLong();
    styleChanged = true;
  }

  if (styleChanged)
  {
    setStyle(ulStyle);                   //Change style
    refresh();                           //Force refresh
  }

  return( *this );
}


/*------------------------------------------------------------------------------
| IEntryField::disableCommand                                                  |
------------------------------------------------------------------------------*/
IEntryField& IEntryField::disableCommand()
{
  enableCommand(false);
  return( *this );
}


/*------------------------------------------------------------------------------
| IEntryField::isCommand                                                       |
------------------------------------------------------------------------------*/
Boolean IEntryField::isCommand() const
{
  return( (style() & command.asUnsignedLong()) ? true : false );
}
#endif // IC_PM


/*------------------------------------------------------------------------------
| IEntryField::enableAutoTab                                                   |
------------------------------------------------------------------------------*/
IEntryField& IEntryField::enableAutoTab( Boolean enable )
{
#ifdef IC_PMWIN   // DEFECT 4624 : required for PM and WIN
  unsigned long ulExtStyle = extendedStyle();
  unsigned long ulOldStyle = ulExtStyle;

  if (enable)
    ulExtStyle |= autoTab.asExtendedUnsignedLong();
  else
    ulExtStyle &= ~autoTab.asExtendedUnsignedLong();

  if (ulExtStyle != ulOldStyle)
    setExtendedStyle(ulExtStyle);
#else             // DEFECT 4624 : prior (default) code
  unsigned long ulStyle = style();
  unsigned long ulOldStyle = ulStyle;

  if (enable)
  {
    ulStyle |= autoTab.asUnsignedLong();
  }
  else
  {
    ulStyle &= ~autoTab.asUnsignedLong();
  }

  if (ulStyle != ulOldStyle)
  {
    setStyle(ulStyle);                   //Change style
    refresh();                           //Force refresh
  }
#endif // IC_PMWIN
  return *this;
}


/*------------------------------------------------------------------------------
| IEntryField::disableAutoTab                                                  |
------------------------------------------------------------------------------*/
IEntryField& IEntryField::disableAutoTab()
{
  enableAutoTab(false);
  return( *this );
}


/*------------------------------------------------------------------------------
| IEntryField::isAutoTab                                                       |
------------------------------------------------------------------------------*/
Boolean IEntryField::isAutoTab() const
{
#ifdef IC_PMWIN   // DEFECT 4624 : required for PM and WIN
  return( (extendedStyle() & autoTab.asExtendedUnsignedLong()) ? true : false );
#else             // DEFECT 4624 : prior (default) code
  return( (style() & autoTab.asUnsignedLong()) ? true : false );
#endif // IC_PMWIN
}


/*------------------------------------------------------------------------------
| IEntryField::cut                                                             |
|                                                                              |
| Removes the selected text from the entry field control and puts it in the    |
| clipboard.                                                                   |
------------------------------------------------------------------------------*/
IEntryField& IEntryField::cut( unsigned long timestamp )
{
  IASSERTSTATE(hasSelectedText());

  IEventResult evt = handle().sendEvent(EM_CUT,
                                        IEventParameter1(0),
                                        IEventParameter2(0));

  if (!(evt.asUnsignedLong()))
     ITHROWGUIERROR("PM = EM_CUT - WIN = WM_CUT");

  return( *this );
}


/*------------------------------------------------------------------------------
| IEntryField::copy                                                            |
|                                                                              |
| Copy marked text to the clipboard.                                           |
------------------------------------------------------------------------------*/
IEntryField& IEntryField::copy( unsigned long timestamp )
{
  IASSERTSTATE(hasSelectedText());

  IEventResult evt = handle().sendEvent(EM_COPY,
                                        IEventParameter1(0),
                                        IEventParameter2(0));

  if (!(evt.asUnsignedLong()))
     ITHROWGUIERROR("PM = EM_COPY - WIN = WM_COPY");

  return( *this );
}


/*------------------------------------------------------------------------------
| IEntryField::paste                                                           |
|                                                                              |
| Replaces the selected text with the text in the clipboard                    |
------------------------------------------------------------------------------*/
IEntryField& IEntryField::paste()
{
  IASSERTSTATE(clipboardHasTextFormat());

  IEventResult evt = handle().sendEvent(EM_PASTE,
                                        IEventParameter1(0),
                                        IEventParameter2(0));
  if (!(evt.asUnsignedLong()))
     ITHROWGUIERROR("PM = EM_PASTE - WIN = WM_PASTE");

  return( *this );
}


/*------------------------------------------------------------------------------
| IEntryField::clear                                                           |
|                                                                              |
| Replaces the selected text in the entry field with blanks.                   |
------------------------------------------------------------------------------*/
IEntryField& IEntryField::clear( unsigned long timestamp )
{
  IASSERTSTATE(hasSelectedText());
  IRange rangMark = selectedRange();
  IString strText  = text();

  // Compensate for 0-based index from selected range
  // and 1-based index used by IString.
  strText.overlayWith(IString(0,rangMark.upperBound() -
                                rangMark.lowerBound()+1),
                                rangMark.lowerBound()+1);
  setText((char *)strText);
  return( *this );
}


/*------------------------------------------------------------------------------
| IEntryField::discard                                                         |
|                                                                              |
| Deletes the selected text.  Compacts the marked area.                        |
------------------------------------------------------------------------------*/
IEntryField& IEntryField::discard()
{
  IASSERTSTATE(hasSelectedText());

  IEventResult evt = handle().sendEvent(EM_CLEAR,
                                        IEventParameter1(0),
                                        IEventParameter2(0));

  if (!(evt.asUnsignedLong()))
     ITHROWGUIERROR("PM = EM_CLEAR - WIN = WM_CLEAR");

  return( *this );
}


/*------------------------------------------------------------------------------
| IEntryField::removeAll                                                       |
|                                                                              |
| Deletes the entire contents of the Entryfield.                               |
------------------------------------------------------------------------------*/
IEntryField& IEntryField::removeAll()
{
  /*******************************************************************/
  /* Check to insure that Entryfield is not already empty            */
  /*******************************************************************/
  if (!isEmpty())
  {
    /*****************************************************************/
    /* Turn update off, select all of the text, then call discard()  */
    /* to remove all of the text from the entryfield.                */
    /*****************************************************************/
    enableUpdate(false);
    selectRange();
    discard();
    enableUpdate(true);
  }
  return *this;
}


/*------------------------------------------------------------------------------
| IEntryField::isEmpty                                                         |
------------------------------------------------------------------------------*/
Boolean IEntryField::isEmpty() const
{
  if (text().length())
    return( false );
  else
    return( true );
}


/*------------------------------------------------------------------------------
| IEntryField::hasChanged                                                      |
|                                                                              |
| Queries whether the entry field has changed since the last query.            |
------------------------------------------------------------------------------*/
Boolean IEntryField::hasChanged() const
{
  IEventResult evt = handle().sendEvent(EM_QUERYCHANGED,
                                        IEventParameter1(0),
                                        IEventParameter2(0));

#ifdef IC_PM
  if (evt.asUnsignedLong())
    return( true );
  else
    return( false );
#endif

#ifdef IC_WIN
  if (evt.asUnsignedLong())
  {
    handle().sendEvent(EM_SETMODIFY,                 //Must reset in Windows
                       IEventParameter1(false),
                       IEventParameter2(0));
    return( true );
  }
  else
    return( false );
#endif
}


/*------------------------------------------------------------------------------
| IEntryField::isWriteable                                                     |
------------------------------------------------------------------------------*/
Boolean IEntryField::isWriteable() const
{
#ifdef IC_WIN
  if ( style() & readOnly.asUnsignedLong() )
#else
  IEventResult evt = handle().sendEvent(EM_QUERYREADONLY,
                                        IEventParameter1(0),
                                        IEventParameter2(0));
  if (evt.asUnsignedLong())
#endif
    return( false );
  else
    return( true );
}


/*------------------------------------------------------------------------------
| IEntryField::hasSelectedText                                                 |
------------------------------------------------------------------------------*/
Boolean IEntryField::hasSelectedText() const
{
  if (selectedTextLength() == 0)
    return( false );
  else
    return( true );
}

/*------------------------------------------------------------------------------
| IEntryField::setChangedFlag                                                  |
------------------------------------------------------------------------------*/
IEntryField& IEntryField::setChangedFlag( Boolean changeFlag )
{
  if (changeFlag)
    ISETWINDOWTEXT(handle(), (PSZ)text());
  else
    hasChanged();
  return *this;
}

/*------------------------------------------------------------------------------
| IEntryField::setTextChangedFlag                                              |
------------------------------------------------------------------------------*/
IEntryField& IEntryField::setTextChangedFlag( Boolean changed )
{
#ifdef IC_PM
  if (changed)
    ISETWINDOWTEXT(handle(), (PSZ)text());
  else
    IEventResult evt = handle().sendEvent(EM_QUERYCHANGED,
                                          IEventParameter1(0),
                                          IEventParameter2(0));
#endif

#ifdef IC_WIN
  handle().sendEvent(EM_SETMODIFY, IEventParameter1(changed),
                                   IEventParameter2(0));
#endif

  return *this;
}

/*------------------------------------------------------------------------------
| IEntryField::resetTextChangedFlag                                            |
------------------------------------------------------------------------------*/
IEntryField& IEntryField::resetTextChangedFlag( )
{
  setTextChangedFlag( false );
  return *this;
}

/*------------------------------------------------------------------------------
| IEntryField::hasTextChanged                                                  |
|                                                                              |
| Queries whether the entry field has changed since the last query.            |
------------------------------------------------------------------------------*/
Boolean IEntryField::hasTextChanged( ) const
{
  //In PM, this message resets the change flag by default.
  //In Windows, this message does not reset the change flag by default.
  IEventResult evt = handle().sendEvent(EM_QUERYCHANGED,
                                        IEventParameter1(0),
                                        IEventParameter2(0));

  if (evt.asUnsignedLong())
  {
#ifdef IC_PM
    ((IEntryField* const)this)->setTextChangedFlag( true );
#endif
    return( true );
  }
  else
    return( false );
}

/*------------------------------------------------------------------------------
| IEntryField::disable                                                         |
------------------------------------------------------------------------------*/
IEntryField& IEntryField::disable()
{
  if (isWriteable())
  {
     eReadStatus = wasInputButReset;
     disableDataUpdate();
  }
  IWindow::disable();

  if (hasFocus())
  {
     /**********************************************************/
     /* Send tab message to self (this will be sent on to the  */
     /* owner to be processed).                                */
     /* NOTE: The only additional information sent with the    */
     /*       VK_TAB message is that it is a virtual key.      */
     /**********************************************************/
#ifdef IC_PM
     handle().sendEvent(WM_CHAR,
                        IEventParameter1((unsigned short)KC_VIRTUALKEY,0,0),
                        IEventParameter2((unsigned short)0,VK_TAB,0));
#endif

#ifdef IC_WIN
     handle().sendEvent(WM_KEYDOWN,
                        IEventParameter1(VK_TAB),
                        IEventParameter2(0));
#endif
  }

  return( *this );
}


/*------------------------------------------------------------------------------
| IEntryField::enable                                                          |
|                                                                              |
| Enables the entry field.  Remove the read-only (if turned on by disable()    |
| and reset by enableReadOnly()).                                              |
------------------------------------------------------------------------------*/
IEntryField& IEntryField::enable( Boolean enable )
{
  if (eReadStatus == wasInputButReset)
  {
     eReadStatus=doesntMatter;
     enableDataUpdate();
  }
  IENABLEWINDOW(handle(), enable);
  return( *this );
}


#ifdef IC_PM
/*------------------------------------------------------------------------------
| IEntryField::isInsertMode                                                    |
------------------------------------------------------------------------------*/
Boolean IEntryField::isInsertMode() const
{
  ((IEntryField*)this)->queryInsMode = true;
  IEventResult evt = handle().sendEvent(EM_SETINSERTMODE,
                                        IEventParameter1(true),
                                        IEventParameter2(0));
  if (evt.asUnsignedLong())
  {
    ((IEntryField*)this)->queryInsMode = false;
    return( true );
  }

  handle().sendEvent(EM_SETINSERTMODE,
                     IEventParameter1(false),
                     IEventParameter2(0));

  ((IEntryField*)this)->queryInsMode = false;
  return( false );
}


/*------------------------------------------------------------------------------
| IEntryField::enableInsertMode                                                |
------------------------------------------------------------------------------*/
IEntryField& IEntryField::enableInsertMode( Boolean insert )
{
  handle().sendEvent(EM_SETINSERTMODE,
                     IEventParameter1(insert),
                     IEventParameter2(0));
  return( *this );
}


/*------------------------------------------------------------------------------
| IEntryField::disableInsertMode                                               |
------------------------------------------------------------------------------*/
IEntryField& IEntryField::disableInsertMode()
{
  enableInsertMode(false);
  return( *this );
}
#endif


/*------------------------------------------------------------------------------
| IEntryField::enableDataUpdate                                                |
------------------------------------------------------------------------------*/
IEntryField& IEntryField::enableDataUpdate( Boolean update )
{
  if (update)
  {
    if ( !(this->isWriteable()) )
    {
      handle().sendEvent(EM_SETREADONLY,
                         IEventParameter1(false),
                         IEventParameter2(0));
    }
  }
  else if ( this->isWriteable() )
  {
    handle().sendEvent(EM_SETREADONLY,
                       IEventParameter1(true),
                       IEventParameter2(0));
  }
  return( *this );
}


/*------------------------------------------------------------------------------
| IEntryField::disableDataUpdate                                               |
------------------------------------------------------------------------------*/
IEntryField& IEntryField::disableDataUpdate()
{
  enableDataUpdate(false);
  return( *this );
}


/*------------------------------------------------------------------------------
| IEntryField::setLimit                                                        |
|                                                                              |
| Sets the maximum number of characters allowed in the entry field.            |
------------------------------------------------------------------------------*/
IEntryField& IEntryField::setLimit( unsigned long ulLimit )
{
  IASSERTSTATE(text().length() <= ulLimit);

  if (this->limit() != ulLimit)
  {
    /*************************************************************/
    /* Now set the new edit limit for the entry field.           */
    /*************************************************************/
    handle().sendEvent(EM_SETTEXTLIMIT,
                       IEventParameter1(ulLimit),
                       IEventParameter2(0));

    setLayoutDistorted(IWindow::minimumSizeChanged, 0);
  }

  return( *this );
}


/*------------------------------------------------------------------------------
| IEntryField::setLimit                                                        |
|                                                                              |
| Sets the maximum number of characters allowed in the entry field given a     |
| string resource-id that defines the limit.                                   |
------------------------------------------------------------------------------*/
IEntryField& IEntryField::setLimit( const IResourceId& residTextLimit )
{
  IString strTextLimit =
     residTextLimit.resourceLibrary().loadString(residTextLimit);
  setLimit(strTextLimit.asInt());
  return( *this );
}


/*------------------------------------------------------------------------------
| IEntryField::limit                                                       c   |
|                                                                              |
| Returns the maximum number of characters allowed to entered into the entry   |
| field.                                                                       |
------------------------------------------------------------------------------*/
unsigned long IEntryField::limit() const
{
#ifdef IC_PM
  WNDPARAMS     wprmData;
  ENTRYFDATA    efdCtlData;
  unsigned long ulClTextLimit = 0;

  wprmData.fsStatus = WPM_CTLDATA;          //Get control data
  wprmData.pCtlData = &efdCtlData;          //Area for control data
  wprmData.cbCtlData = sizeof(ENTRYFDATA);  //Size of control data

  /*******************************************************************/
  /* Because of an error in PMMERGE.DLL on an OS2J machine, do       */
  /* NOT check the return value of this call.  On an OS2J machine    */
  /* the return code from PM is 1, but we are getting 0 back here    */
  /* due to the EBX register getting trashed in PMMERGE.  Hence, we  */
  /* throw an exception when we shouldn't.                           */
  /*******************************************************************/
  handle().sendEvent(WM_QUERYWINDOWPARAMS,
                     IEventParameter1(&wprmData),
                     IEventParameter2(0));

  ulClTextLimit = efdCtlData.cchEditLimit;  // Edit limit
                                   // Note: wprmData.fsStatus does not get
                                   //       cleared on success, as advertised!
  return( ulClTextLimit );
#endif

#ifdef IC_WIN
  return( 30000 );  // Default in Windows is 30000 so use it for now
#endif
#ifdef IC_NOTYET
  Need to store the limit within this class for Windows in order to allow
  a query of it ...
#endif
}


/*------------------------------------------------------------------------------
| IEntryField::selectRange                                                     |
|                                                                              |
| Marks (selects) text in the entry field.  Index of characters to mark is     |
| assumed to be 0-based.                                                       |
------------------------------------------------------------------------------*/
IEntryField& IEntryField::selectRange( const IRange& range,
                                       unsigned long timestamp )
{
  //Make upper the character following the last selected one
  IRange temprange = IRange(range.lowerBound(), range.upperBound() + 1);

  if (range.upperBound() == end)
    temprange.setUpperBound(text().length() + 1);

  setCursorInfo(temprange);
  return( *this );
}


/*------------------------------------------------------------------------------
| IEntryField::selectedRange                                                   |
|                                                                              |
| Returns the range of the selected text.  Index of characters marked is       |
| assumed to be 0-based.                                                       |
------------------------------------------------------------------------------*/
IRange IEntryField::selectedRange() const
{
  IASSERTSTATE(hasSelectedText());
  IRange rangMarked = getCursorInfo();
  if (rangMarked.lowerBound() != rangMarked.upperBound())
     rangMarked.setUpperBound(rangMarked.upperBound() - 1);
                                //Make upper the last selected character
  return( rangMarked );
}


/*------------------------------------------------------------------------------
| IEntryField::selectedTextLength                                              |
|                                                                              |
| Returns the size of the user marked area in bytes.  Does not count '\0'.     |
------------------------------------------------------------------------------*/
unsigned long IEntryField::selectedTextLength() const
{
  IRange rangMarked = getCursorInfo();

  return (rangMarked.upperBound() - rangMarked.lowerBound());
}


/*------------------------------------------------------------------------------
| IEntryField::selectedText                                                    |
|                                                                              |
| Returns the selected text.                                                   |
------------------------------------------------------------------------------*/
IString IEntryField::selectedText() const
{
  long    lMinOffset;     //0-based offset of 1st char in selection
  long    lMaxOffset;     //0-based offset of 1st char AFTER selection

  IRange rangMarked = getCursorInfo();

  if (rangMarked.lowerBound() == rangMarked.upperBound())
  {                                    //No selected text
    IString str("\0");
    return( str );
  }
  else
  {
    lMinOffset = rangMarked.lowerBound();
    lMaxOffset = rangMarked.upperBound();

    IString str((char *)text() + lMinOffset,
                lMaxOffset - lMinOffset, '\0');
    return( str );
  }
}


/*------------------------------------------------------------------------------
| IEntryField::setLeftIndex                                                    |
|                                                                              |
| Set the first character displayed at the left edge of the entry field        |
| control.  The index is 0-based.                                              |
------------------------------------------------------------------------------*/
IEntryField& IEntryField::setLeftIndex( unsigned long ulIndex )
{
#ifndef IC_WIN
  IEventResult evt = handle().sendEvent(EM_SETFIRSTCHAR,
                                        IEventParameter1(ulIndex),
                                        IEventParameter2(0));
  if (!(evt.asUnsignedLong()))
     ITHROWGUIERROR("EM_SETFIRSTCHAR");
#else
  // Simulate by setting caret to character desired and then scroll into view
  handle().sendEvent( EM_SETSEL, IEventParameter1(ulIndex),
                      IEventParameter2(ulIndex));
  IEventResult evt = handle().sendEvent(EM_SCROLLCARET,
                                        IEventParameter1(0),
                                        IEventParameter2(0));
#endif
  return *this;
}


/*------------------------------------------------------------------------------
| IEntryField::leftIndex                                                       |
|                                                                              |
| Get the offset position of the first character displayed at the left edge    |
| of the entry field.  The index is 0-based.                                   |
------------------------------------------------------------------------------*/
unsigned long IEntryField::leftIndex() const
{
  IEventResult evt = handle().sendEvent(EM_QUERYFIRSTCHAR,
                                        IEventParameter1(0),
                                        IEventParameter2(0));

  return( evt.asUnsignedLong() );
}


/*------------------------------------------------------------------------------
| IEntryField::setStyle                                                        |
|                                                                              |
| Replaces the style of the entry field control, while preserving the          |
| cursor position and selected text.                                           |
------------------------------------------------------------------------------*/
IEntryField& IEntryField::setStyle( unsigned long ulStyle )
{
  IRange range = getCursorInfo();
  IWindow::setStyle(ulStyle);
  setCursorInfo(range);
  return( *this );
}


/*------------------------------------------------------------------------------
| IEntryField::calcMinimumSize                                                 |
|                                                                              |
| Calculate the minimum screen size needed by the control.  The height is      |
| based on the font height and the width is based on the average font width    |
| multiplied by the text limit.                                                |
------------------------------------------------------------------------------*/
ISize IEntryField::calcMinimumSize() const
{
  ISize sizChar = ((IEntryField *)this)->characterSize();
                                            //Cast to non-const object
  unsigned long ulWidth = sizChar.width();  //Average char width
  unsigned long ulHeight= sizChar.height();

  // Hack similar to the one in ITextControl::displaySize.  This will
  // keep entry fields with borders and static text controls, which use
  // the same font, about equal in height on a multi-cell or set canvas.
  ulHeight += 2;

  unsigned long ulClTextLimit = limit();

  /*****************************************************************/
  /* Use a step function to reflect the fact that you are more     */
  /* likely to exceed an average character width with an           */
  /* individual character, but a long string will likely be ok     */
  /* with this average value.                                      */
  /*****************************************************************/

  IFont font(this);
  if ( !font.isFixed() )
  {
     // DEFECT 8842 : earlier algorithm gave uneven length increases;
     // simplify to a smooth curve of lengths over text limit range.
     if (ulClTextLimit <= 2)
        ulWidth = (unsigned long)(ulWidth * 3);
     else if (ulClTextLimit <=16)
        ulWidth = (unsigned long)(ulWidth + 5);
     else if (ulClTextLimit <=24)
        ulWidth = (unsigned long)(ulWidth + 4);
     else if (ulClTextLimit <=32)
        ulWidth = (unsigned long)(ulWidth + 3);
     else if (ulClTextLimit <=40)
        ulWidth = (unsigned long)(ulWidth + 2);
     else if (ulClTextLimit <=48)
        ulWidth = (unsigned long)(ulWidth + 1);

     ulWidth = ulWidth * ulClTextLimit;
  }  // is proportional
  else
  {
     ulWidth = (unsigned long)(ulWidth * ulClTextLimit);
  }  // not proportional

  if ( ulWidth > ( sizChar.width() * 35 ))
  {
     ulWidth = sizChar.width() * 35;
  }

  if (isMargin())
  {
     ulWidth +=(3 * IQUERYSYSVALUE(SV_CXBORDER));
     ulHeight+=(3 * IQUERYSYSVALUE(SV_CYBORDER));
  } /* endif */

  ISize sizMin(ulWidth, ulHeight);
  return( sizMin );
}


/*------------------------------------------------------------------------------
| IEntryField::cursorPosition                                                  |
|                                                                              |
| Returns the character position of where the cursor currently is.             |
------------------------------------------------------------------------------*/
unsigned long IEntryField::cursorPosition() const
{
  IEventResult evt = handle().sendEvent(EM_QUERYSEL,
                                        IEventParameter1(0),
                                        IEventParameter2(0));

  return( evt.number2() );
}


/*------------------------------------------------------------------------------
| IEntryField::setCursorPosition                                               |
|                                                                              |
| Position the cursor in front of the character specified.                     |
------------------------------------------------------------------------------*/
IEntryField& IEntryField::setCursorPosition( unsigned long lCursorChar )
{
   IEventResult evt;

#ifdef IC_PM
   evt = handle().sendEvent(EM_SETSEL,
                      IEventParameter1((unsigned short)lCursorChar,
                                       (unsigned short)lCursorChar),
                      IEventParameter2(0));

   if (!(evt.asUnsignedLong()))
      ITHROWGUIERROR("EM_SETSEL");
#endif
#ifdef IC_WIN
   evt = handle().sendEvent(EM_SETSEL,
                      IEventParameter1(lCursorChar),
                      IEventParameter2(lCursorChar));
#endif

   //----------------------------------------------------------------
   // DEFECT 25702 : if not in view, scroll to 1st char position on line.
   //----------------------------------------------------------------
#ifndef IC_WIN
   unsigned long ulIndex;
   ulIndex = ( (lCursorChar > 0 ? lCursorChar-1 : 0) );
   evt = handle().sendEvent(EM_SETFIRSTCHAR,
                            IEventParameter1(ulIndex),
                            IEventParameter2(0));
#else
   evt = handle().sendEvent(EM_SCROLLCARET,
                            IEventParameter1(0),
                            IEventParameter2(0));
#endif
   //----------------------------------------------------------------

   return *this;
}


/*------------------------------------------------------------------------------
| IEntryField::getCursorInfo                                                   |
|                                                                              |
| Gets the bound of the marked or selected text, which also includes where     |
| the cursor is placed.  (The values in the returned range have not been       |
| adjusted to match the format used by selectedRange.)                         |
------------------------------------------------------------------------------*/
IRange IEntryField::getCursorInfo() const
{
  IEventResult evt = handle().sendEvent(EM_QUERYSEL,
                                        IEventParameter1(0),
                                        IEventParameter2(0));

  return( IRange(evt.number1(), evt.number2()) );
}


/*------------------------------------------------------------------------------
| IEntryField::setCursorInfo                                                   |
|                                                                              |
| Marks (selects) text in the entry field, and sets the position of the cursor.|
------------------------------------------------------------------------------*/
void IEntryField::setCursorInfo( const IRange& range )
/***************************************************************/
/* Marks (selects) texts in the entry field, and also sets     */
/* the position of the cursor.                                 */
/***************************************************************/
{
#ifdef IC_PM
  IEventResult evt = handle().sendEvent(EM_SETSEL,
                      IEventParameter1((unsigned short)range.lowerBound(),
                                       (unsigned short)range.upperBound()),
                      IEventParameter2(0));

  if (!(evt.asUnsignedLong()))
     ITHROWGUIERROR("EM_SETSEL");
#endif
#ifdef IC_WIN
  IEventResult evt = handle().sendEvent(EM_SETSEL,
                      IEventParameter1((int)range.lowerBound()),
                      IEventParameter2((int)range.upperBound() - 1));
#endif
}


#ifdef IC_PM
/*------------------------------------------------------------------------------
| IEntryField::foregroundColor                                                 |
|                                                                              |
| Returns the foreground color of the IEntryField.                             |
------------------------------------------------------------------------------*/
IColor IEntryField::foregroundColor () const
{
  IGUIColor guiColor(IGUIColor::windowText);

  if (!isWriteable())
  {
    guiColor = IGUIColor(IGUIColor::outputText);
  }
  return (IWindow::color(PP_FOREGROUNDCOLOR, guiColor));
}


/*------------------------------------------------------------------------------
| IEntryField::backgroundColor                                                 |
|                                                                              |
| Returns the background color of the IEntryField.                             |
------------------------------------------------------------------------------*/
IColor IEntryField::backgroundColor () const
{
  return (IWindow::color(PP_BACKGROUNDCOLOR,
                         IGUIColor(IGUIColor::entryFieldBgnd)));
}
#endif
#ifdef IC_NOTYET
  Will this color model work with Windows???
#endif


#ifdef IC_PMWIN   // DEFECT 4624 : required for PM and WIN
/*------------------------------------------------------------------------------
| IEntryHandler::IEntryHandler                                                 |
|                                                                              |
| Empty constructor here for page tuning.                                      |
------------------------------------------------------------------------------*/
IEntryHandler::IEntryHandler()
{
}

/*------------------------------------------------------------------------------
| IEntryHandler::~IEntryHandler                                                |
|                                                                              |
| Empty destructor here for page tuning.                                       |
------------------------------------------------------------------------------*/
IEntryHandler::~IEntryHandler()
{
}

 /*------------------------------------------------------------------------------
| IEntryHandler::dispatchHandlerEvent                                          |
|                                                                              |
| Intercepts enter key event and eats it                                       |
|                                                                              |
|    This handler returns false for all other events.                          |
------------------------------------------------------------------------------*/
Boolean IEntryHandler::dispatchHandlerEvent( IEvent& event )
{
  IMODTRACE_ALL("IEntryHandler::dispatchHandlerEvent");
  Boolean
    result = false;

  switch( event.eventId() )
  {
#ifdef IC_PM
    case WM_CHAR:
      {
         //--------------------------------------------------------------
         // DEFECT 4624 : Override auto-tabbing for PM
         //--------------------------------------------------------------
         IKeyboardEvent keyevt(event);
         IKeyboardEvent::VirtualKey vkKey;
         if (keyevt.isCharacter()  &&
             (keyevt.isVirtual() == false  ||
             ((vkKey = keyevt.virtualKey()) != IKeyboardEvent::tab  &&
            vkKey != IKeyboardEvent::backSpace  &&
            vkKey != IKeyboardEvent::esc)) )
         {
            IEntryField* ef = (IEntryField*)event.window();
            unsigned long efLimit = ef->limit() - 1;
            //--------------------------------------------------------
            // Defect 28756:  Modified the condition which determines
            //                whether an "autoTab" is to be sent.
            //                Now, instead of checking for textLength
            //                for equality to 'efLimit', we check
            //                for 'textLength - selectedTextLength'
            //                for equality to 'efLimit'.
            //---------------------------------------------------------
            if ( ef->isAutoTab() &&
                 ((ef->textLength() - ef->selectedTextLength()) == efLimit) &&
                 (ef->cursorPosition() == efLimit) )
            {
               ef->sendEvent(WM_CHAR,
                             IEventParameter1((unsigned short)KC_VIRTUALKEY,0,0),
                             IEventParameter2((unsigned short)0,VK_TAB,0));
            }
         }
         //--------------------------------------------------------------
      }
#endif   // IC_PM
#ifdef IC_WIN
    case WM_CHAR:
      {
         //--------------------------------------------------------------
         // DEFECT 4624 : Emulate auto-tabbing for Windows
         //--------------------------------------------------------------
         unsigned long ep1 = (unsigned long)event.parameter2();
         if ((ep1 != VK_BACKSPACE ) && (ep1 != VK_BREAK ))
         {
           // Check if autotab style is set, and if when this key was
           // pressed we are one character short of the limit (meaning that
           // after this key is processed, we will be at our limit and
           // that the cursor is in the last position in the field)
           IEntryField* ef = (IEntryField*)event.window();
           unsigned long efLimit = ef->limit() - 1;
           //--------------------------------------------------------
           // Defect 28756:  Modified the condition which determines
           //                whether an "autoTab" is to be sent.
           //                Now, instead of checking for textLength
           //                for equality to 'efLimit', we check
           //                for 'textLength - selectedTextLength'
           //                for equality to 'efLimit'.
           //---------------------------------------------------------
           if ( ef->isAutoTab() &&
                ((ef->textLength() - ef->selectedTextLength()) == efLimit) &&
                (ef->cursorPosition() == efLimit) )
           {
             ef->postEvent( WM_CHAR, VK_TAB );
           }
         }
         //--------------------------------------------------------------
         if ( (unsigned long)event.parameter1() == 0x000D )
         {
           return true;
         }
         //--------------------------------------------------------------
      }
#endif   // if IC_WIN
      if ( (unsigned long)event.parameter1() == 0x000D )
        return true;
      break;

    default:
      break;
  }
  return result;
}
#endif   // IC_PMWIN

