/*******************************************************************************
* FILE NAME: imle.cpp                                                          *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in imle.hpp.                                                               *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1995       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/

// Priority INT_MIN (-2147483647 - 1) + 1024 + 512
#pragma priority( -2147482112 )

extern "C" {
  #define INCL_WINMLE
  #define INCL_WINCLIPBOARD
  #define INCL_WININPUT             // WM_CHAR, etc.
  #define INCL_WINMESSAGEMGR        // for WNDPARAM struct
  #define INCL_WINWINDOWMGR         // WinQueryWindowULong
  #define INCL_GPIPRIMITIVES
  #define INCL_WINSYS
  #include <iwindefs.h>

  #include <sys\stat.h>
  #include <stdio.h>              // include standard IO routines
#ifdef IC_WIN
  #include <io.h>                 // include windows IO routines
  #include <fcntl.h>              // include windows IO routines
#endif
  #include <string.h>             // include string handling routines
  #include <stdlib.h>             // include memory handling routines
}

#include <imle.hpp>
#include <icconst.h>
#include <iexcept.hpp>
#include <irect.hpp>
#include <ireslib.hpp>
#include <istring.hpp>
#include <icolor.hpp>
#include <ifont.hpp>
#include <imphdr.hpp>

// Segment definitions
#ifdef IC_PAGETUNE
  #define _IMLE_CPP_
  #include <ipagetun.h>
#endif

/*------------------------------------------------------------------------------
| Public multiple-line entry field control styles                              |
------------------------------------------------------------------------------*/
#pragma data_seg(ICLStaticConst)
const IMultiLineEdit::Style
  IMultiLineEdit::wordWrap          = MLS_WORDWRAP,
#ifndef IC_WIN
  IMultiLineEdit::ignoreTab         = MLS_IGNORETAB,
#endif
#ifdef IC_WIN
  IMultiLineEdit::ignoreTab         = 0,             // not supported
#endif
  IMultiLineEdit::border3D          ( 0, IWS_BORDER3D ),
  IMultiLineEdit::border            = MLS_BORDER,
  IMultiLineEdit::verticalScroll    = MLS_VSCROLL,
  IMultiLineEdit::horizontalScroll  = MLS_HSCROLL,
  IMultiLineEdit::readOnly          = MLS_READONLY,
  IMultiLineEdit::classDefaultStyle ( WS_VISIBLE  |
                                      MLS_BORDER  |
                                      MLS_VSCROLL |
                                      MLS_WORDWRAP );

const long IMultiLineEdit::end = -2;
#pragma data_seg()

/*------------------------------------------------------------------------------
| Default style for new objects (initial value).                               |
------------------------------------------------------------------------------*/
#pragma data_seg(ICLNonConst)
IMultiLineEdit::Style
  IMultiLineEdit::currentDefaultStyle ( WS_VISIBLE  |
                                        MLS_BORDER  |
                                        MLS_VSCROLL |
                                        MLS_WORDWRAP );
#pragma data_seg()

/*------------------------------------------------------------------------------
| IMultiLineEdit::IMultiLineEdit                                               |
|                                                                              |
| Constructor to create a MLE control on a standard window.                    |
------------------------------------------------------------------------------*/
IMultiLineEdit::IMultiLineEdit( unsigned long ulId,
                                IWindow* pwndParent,
                                IWindow* pwndOwner,
                                const IRectangle& rectInit,
                                const Style& style )
                                 : bRefresh ( true ),
                                   fMultiLineEditData ( 0 )
#ifdef IC_WIN
                                 , fcharLimit( 30000 ),
                                   flimitSet( false )
#endif
{
  // assertions on input parms
  IASSERTPARM(pwndParent!=0);

  // Save the extended style to make sure we have a copy of it stored
  setExtendedStyle( extendedStyle() | style.asExtendedUnsignedLong() );

  /****************************************************************/
  /* Borland has bug with embedded ? operator so below is needed. */
  /****************************************************************/
#ifdef __BORLANDC__
  IWindowHandle owner;
  if (pwndOwner == 0)
    owner = IWindowHandle(0);
  else
    owner = pwndOwner->handle();
#else
  IWindowHandle owner = (pwndOwner == 0) ? IWindowHandle(0) : pwndOwner->handle();
#endif

#ifdef IC_WIN
  //-------------------------------------------------------------------
  // DEFECT 25510 : conversely, if MLS_HSCROLL, turn IMLS_WORDWRAP OFF
  //          (Windows won't support wordwrap and horiz scroll both)
  //-------------------------------------------------------------------
  if ( style & horizontalScroll ) {
     disableWordWrap(); // remove from stored extendedStyle()
  }
  //-------------------------------------------------------------------
#endif

  IWindowHandle whMLE(
      this -> create( ulId,
                      0,
                      convertToGUIStyle( style ),
                      WC_MLE,
                      pwndParent->handle(),
                      owner,
                      rectInit,
                      0,
                      0 ));

  startHandlingEventsFor(whMLE);

  setEditRegion();
  IMousePointerHandler::defaultHandler()->handleEventsFor(this);
}


/*------------------------------------------------------------------------------
| IMultiLineEdit::IMultiLineEdit                                               |
|                                                                              |
| Constructor to instantiate an object for a dialog template.                  |
------------------------------------------------------------------------------*/
IMultiLineEdit::IMultiLineEdit( unsigned long ulId,
                                IWindow* pwndParent )
                                 : bRefresh ( true ),
                                   fMultiLineEditData ( 0 )
#ifdef IC_WIN
                                 , fcharLimit( 30000 ),
                                   flimitSet( false )
#endif
{
  setAutoDestroyWindow(false);
  startHandlingEventsFor(ulId, pwndParent);

  setEditRegion();
  IMousePointerHandler::defaultHandler()->handleEventsFor(this);

#ifdef IC_WIN
  //-------------------------------------------------------------------
  // DEFECT 25510 : if MLS_HSCROLL, turn IMLS_WORDWRAP OFF
  //          (Windows won't support wordwrap and horiz scroll both)
  //-------------------------------------------------------------------
  if ( style() & MLS_HSCROLL )
     disableWordWrap();
  //-------------------------------------------------------------------
#endif
}


/*------------------------------------------------------------------------------
| IMultiLineEdit::IMultiLineEdit                                               |
|                                                                              |
| Constructor to instantiate from an existing MLE control.                     |
------------------------------------------------------------------------------*/
IMultiLineEdit::IMultiLineEdit ( const IWindowHandle& wh )
                                   : bRefresh(true),
                                     fMultiLineEditData ( 0 )
#ifdef IC_WIN
                                   , fcharLimit( 30000 ),
                                     flimitSet( false )
#endif
{
  setAutoDestroyWindow(false);
  startHandlingEventsFor(wh);
  IMousePointerHandler::defaultHandler()->handleEventsFor(this);

#ifdef IC_WIN
  //-------------------------------------------------------------------
  // DEFECT 25510 : if MLS_HSCROLL, turn IMLS_WORDWRAP OFF
  //          (Windows won't support wordwrap and horiz scroll both)
  //-------------------------------------------------------------------
  if ( style() & MLS_HSCROLL )
     disableWordWrap();
  //-------------------------------------------------------------------
#endif
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::~IMultiLineEdit                                              |
|                                                                              |
| Remove the default mouse pointer handler                                     |
------------------------------------------------------------------------------*/
IMultiLineEdit::~IMultiLineEdit()
{
  IMousePointerHandler::defaultHandler()->stopHandlingEventsFor(this);
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::Style  IMultiLineEdit::defaultStyle                          |
|                                                                              |
| Return the default style for new entry field objects.                        |
------------------------------------------------------------------------------*/
IMultiLineEdit::Style  IMultiLineEdit::defaultStyle()
{
  return currentDefaultStyle;
}


/*------------------------------------------------------------------------------
| IMultiLineEdit::setDefaultStyle                                              |
|                                                                              |
| Replace the default style for new entry field objects.                       |
------------------------------------------------------------------------------*/
void  IMultiLineEdit::setDefaultStyle(const Style& mlsStyle)
{
  currentDefaultStyle = mlsStyle;
}


/*------------------------------------------------------------------------------
| IMultiLineEdit::convertToGUIStyle                                            |
|                                                                              |
| Returns base style for the control by default, or extended style if          |
| extended flag (bExtOnly) is set.                                             |
------------------------------------------------------------------------------*/
unsigned long IMultiLineEdit :: convertToGUIStyle(const IBitFlag& guiStyle,
                                                  Boolean bExtOnly) const
{
  // Obtain the style from the class (ITextControl) that we inherit from
  unsigned long ulStyle = Inherited::convertToGUIStyle( guiStyle, bExtOnly );

  if (bExtOnly)
  {
    // Use mask to only return extended styles in the user defined range
    ulStyle |= extendedStyle() & IS_EXTMASK;
  }
  else
  {
    // Let the clean MLS_ styles, flow thru
    ulStyle |= guiStyle.asUnsignedLong() & IMLS_MASK;

#ifdef IC_WIN
    // For Windows it is always a child and always has multiline style
    ulStyle |= WS_CHILD | ES_MULTILINE;
#endif
  }

  return( ulStyle );
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::setChangedFlag                                               |
|                                                                              |
| Set a flag to indicate that the MLE contents have changed.                   |
------------------------------------------------------------------------------*/
IMultiLineEdit&   IMultiLineEdit:: setChangedFlag( Boolean changed )
{
   handle().sendEvent(MLM_SETCHANGED, IEventParameter1(changed),
                                      IEventParameter2(0));
   return *this;
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::resetChangedFlag                                             |
|                                                                              |
| Reset the changed flag to indicate no changes have been made.                |
------------------------------------------------------------------------------*/
IMultiLineEdit& IMultiLineEdit::resetChangedFlag()
{
  handle().sendEvent ( MLM_SETCHANGED,
                       IEventParameter1(false),
                       IEventParameter2(0) );
  return *this;
}


/*------------------------------------------------------------------------------
| IMultiLineEdit::isChanged                                                    |
|                                                                              |
| Queries whether the MLE text has changed since the last time the changed     |
| flag was cleared.                                                            |
------------------------------------------------------------------------------*/
Boolean  IMultiLineEdit::isChanged() const
{
  IEventResult evt(handle().sendEvent(MLM_QUERYCHANGED,
                                      IEventParameter1(0),
                                      IEventParameter2(0)));
  if (evt.asUnsignedLong())
     return true;
  else
     return false;
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::setTextChangedFlag                                           |
|                                                                              |
| Set a flag to indicate that the MLE contents have changed.                   |
------------------------------------------------------------------------------*/
IMultiLineEdit&   IMultiLineEdit::setTextChangedFlag( Boolean changed )
{
   handle().sendEvent(MLM_SETCHANGED, IEventParameter1(changed),
                                      IEventParameter2(0));
   return *this;
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::resetTextChangedFlag                                         |
|                                                                              |
| Reset the flag that indicates that the MLE contents have changed.            |
------------------------------------------------------------------------------*/
IMultiLineEdit&   IMultiLineEdit::resetTextChangedFlag( )
{
   setTextChangedFlag( false );
   return *this;
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::hasTextChanged                                               |
|                                                                              |
| Queries whether the MLE text has changed since the last time the changed     |
| flag was cleared.                                                            |
------------------------------------------------------------------------------*/
Boolean  IMultiLineEdit::hasTextChanged( ) const
{
  IEventResult evt(handle().sendEvent(MLM_QUERYCHANGED,
                                      IEventParameter1(0),
                                      IEventParameter2(0)));

  return ( evt.asUnsignedLong() != 0 );
}


/*------------------------------------------------------------------------------
| IMultiLineEdit::disableDataUpdate                                            |
|                                                                              |
| Set the MLE to read-only mode, preventing updates to the text.               |
------------------------------------------------------------------------------*/
IMultiLineEdit&   IMultiLineEdit::disableDataUpdate()
{
   enableDataUpdate(false);
   return *this;
}


/*------------------------------------------------------------------------------
| IMultiLineEdit::enableDataUpdate                                             |
|                                                                              |
| Sets the MLE to read-write mode which allows the user to update the MLE text.|
------------------------------------------------------------------------------*/
IMultiLineEdit&  IMultiLineEdit::enableDataUpdate ( Boolean update )
{
  Boolean writeable = this->isWriteable();

  /***************************/
  /* Only set if we need to. */
  /***************************/
  if ((update && !writeable) || (!update && writeable))
  {
    handle().sendEvent(MLM_SETREADONLY, IEventParameter1(!update),
                                        IEventParameter2(0));
  }
  return *this;
}


/*------------------------------------------------------------------------------
| IMultiLineEdit::isWriteable                                                  |
------------------------------------------------------------------------------*/
Boolean IMultiLineEdit::isWriteable() const
{
#ifdef IC_WIN
  if ( style() & readOnly.asUnsignedLong() )
#else
  IEventResult evt(handle().sendEvent(MLM_QUERYREADONLY,
                                      IEventParameter1(0),
                                      IEventParameter2(0)));
  if (evt.asUnsignedLong())
#endif
     return false;
  else
     return true;
}


/*------------------------------------------------------------------------------
| IMultiLineEdit::disableUpdate                                                |
|                                                                              |
| Disable screen updates.                                                      |
------------------------------------------------------------------------------*/
IMultiLineEdit& IMultiLineEdit::disableUpdate()
{
  enableUpdate( false );
  return *this;
}


/*------------------------------------------------------------------------------
| IMultiLineEdit::enableUpdate                                                 |
------------------------------------------------------------------------------*/
IMultiLineEdit& IMultiLineEdit::enableUpdate( Boolean update )
{
#ifndef IC_WIN  // IC_NOTYET
  if ( update )
  {
     IEventResult evt(handle().sendEvent(MLM_ENABLEREFRESH,
                                         IEventParameter1(0),
                                         IEventParameter2(0)));
     if (evt.asUnsignedLong())
        bRefresh=true;
  } else {
     IEventResult evt(handle().sendEvent(MLM_DISABLEREFRESH,
                                         IEventParameter1(0),
                                         IEventParameter2(0)));
     if (evt.asUnsignedLong())
        bRefresh=false;
  } /* endif */
#endif
  return *this;
}


/*------------------------------------------------------------------------------
| IMultiLineEdit::setText                                                      |
|                                                                              |
| Changes the control text given a string.                                     |
------------------------------------------------------------------------------*/
IMultiLineEdit&  IMultiLineEdit::setText(const char* pszText)
{
  setText((char*)pszText,0);
  return *this;

}

/*------------------------------------------------------------------------------
| IMultiLineEdit::setText                                                      |
|                                                                              |
| Changes the control text using the string table.                             |
------------------------------------------------------------------------------*/
IMultiLineEdit&  IMultiLineEdit::setText(const IResourceId& residText)
{
  IString str(residText.resourceLibrary().loadString(residText));
  setText((char*)str,0);
  return *this;
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::setText                                                      |
|                                                                              |
| Changes the control text to the text passed in the buffer.                   |
------------------------------------------------------------------------------*/
IMultiLineEdit&  IMultiLineEdit::setText( const char*   pszBuffer,
                                          unsigned long ulBufferSize )
{
#ifdef IC_WIN
   /**************************************************************/
   /* No import support in Windows so just set the text          */
   /* IC_NOTYET - What to do about ulBufferSize??                */
   /**************************************************************/
   Inherited::setText( pszBuffer );
#else
   removeAll();                   // First clear out the MLE text

   /**************************************************************/
   /* Handle setting the MLE text to NULL                        */
   /**************************************************************/
   if ( !pszBuffer )
     return *this;

   if (ulBufferSize == 0) {
      ulBufferSize = strlen(pszBuffer);
   } /* endif */

   /**************************************************************/
   /* Specify the origin of the text to be imported from the     */
   /* buffer as the current cursor position.                     */
   /**************************************************************/
   long ipt = -1;

   /**************************************************************/
   /* Specify the buffer for MLE object import operation.        */
   /**************************************************************/
   import(&ipt,ulBufferSize,pszBuffer);   // Import text from buffer
#endif
   return *this;
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::text                                                         |
|                                                                              |
| Returns the current contents of the MLE                                      |
------------------------------------------------------------------------------*/
IString IMultiLineEdit::text() const
{
#ifdef IC_WIN
   /**************************************************************/
   /* No export support in Windows so just get the text string   */
   /**************************************************************/
   return Inherited::text();
#else
   unsigned long numChars;             // Number of characters exported
   long ipt(0);

   unsigned long ulMyBufSize(textLengthAfterFormat() + 1);
                                  // Set buffer length

   char* pszNewBuf = new char[ulMyBufSize];
                                  // Create buffer

   numChars = ((IMultiLineEdit *)this)->export(&ipt,ulMyBufSize-1,pszNewBuf);
                                  // Export text to buffer

   *(pszNewBuf + numChars) = '\0';// Add a null terminator
   IString str( pszNewBuf );
   delete pszNewBuf;
   return str;
#endif
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::text                                                         |
|                                                                              |
| Returns the current contents of the MLE at the specified line number.        |
------------------------------------------------------------------------------*/
IString IMultiLineEdit::text(unsigned long lLineNumber) const
{
   LONG  numChars;                // Number of chars exported
   long  ipt(0);
   char* pszNewBuf;
   unsigned long ulMyBufSize;

   if (lLineNumber < numberOfLines())
   {
      ipt = (long)((unsigned long) handle().sendEvent(MLM_CHARFROMLINE,
                                 IEventParameter1(lLineNumber),
                                 IEventParameter2(0)));
   } else {
      return IString();
   }
   ulMyBufSize = (unsigned long) handle().sendEvent(MLM_QUERYLINELENGTH,
                              IEventParameter1((unsigned long)ipt),
                              IEventParameter2(0));

   pszNewBuf = new char[ulMyBufSize + 1]; // Create buffer
#ifdef IC_WIN
   // Windows GetLine function requires buffer size to be in first word
   // so set it there and pass line number and buffer for query
   pszNewBuf[0] = ulMyBufSize;
   numChars = handle().sendEvent( EM_GETLINE,
                                  IEventParameter1(lLineNumber),
                                  IEventParameter2(pszNewBuf) );
#else
   numChars = ((IMultiLineEdit *)this)->export(&ipt,ulMyBufSize,pszNewBuf);
                                      // Export text to buffer
#endif
   *(pszNewBuf + numChars) = '\0';    // Add a null terminator
   IString str( pszNewBuf );
   delete pszNewBuf;
   return str;
}

#ifndef IC_WIN  // IC_NOTYET
/*------------------------------------------------------------------------------
| IMultiLineEdit::addAtOffset                                                  |
|                                                                              |
| Insert the specified text into the MLE at the specified insertion point.     |
------------------------------------------------------------------------------*/
IMultiLineEdit& IMultiLineEdit::addAtOffset( const char*   pszBuffer,
                                             unsigned long lInsertionPoint,
                                             unsigned long ulBufferSize,
                                             EOLFormat     type )
{
   long lNumInserted;
   long ipt, iptCursor;

   if (ulBufferSize == 0)                 // If no size specified,
      ulBufferSize = strlen(pszBuffer);   // Assume null-terminated string

   ipt = lInsertionPoint;                 // Input to zero-based
   switch (type)
   {
     case cfText:
          handle().sendEvent(MLM_FORMAT, IEventParameter1(MLFIE_CFTEXT),
                                         IEventParameter2(0));
          break;
     case noTran:
          handle().sendEvent(MLM_FORMAT, IEventParameter1(MLFIE_NOTRANS),
                                         IEventParameter2(0));
          break;
     case MLEFormat:
          handle().sendEvent(MLM_FORMAT, IEventParameter1(MLFIE_WINFMT),
                                         IEventParameter2(0));
          break;
   } /* endswitch */

   /**************************************************************/
   /* Import text from the buffer into the MLE                   */
   /**************************************************************/
   lNumInserted = (long)import(&ipt,ulBufferSize,pszBuffer);
   IEventResult evt(handle().sendEvent(MLM_QUERYSEL,
                                       IEventParameter1(MLFQS_CURSORSEL),
                                       IEventParameter2(0)));
   iptCursor = evt.asUnsignedLong();
   if (iptCursor > lInsertionPoint) {      // If text was inserted before
      iptCursor += lNumInserted;           // the cursor location, reset the
      setCursorPosition( iptCursor );      // cursor position
   } /* endif */

   return *this;
}


/*------------------------------------------------------------------------------
| IMultiLineEdit::addAsLast                                                    |
|                                                                              |
| Insert the specified text into the MLE at the end of the existing text.      |
------------------------------------------------------------------------------*/
IMultiLineEdit& IMultiLineEdit::addAsLast( const char*   pszBuffer,
                                           unsigned long ulBufferSize,
                                           EOLFormat     type)
{
   if (ulBufferSize == 0)                 // If no size specified,
      ulBufferSize = strlen(pszBuffer);   // Assume null-terminated string
   long ipt(textLengthAfterFormat());

   switch (type)
   {
     case cfText:
          handle().sendEvent(MLM_FORMAT, IEventParameter1(MLFIE_CFTEXT),
                                         IEventParameter2(0));
          break;
     case noTran:
          handle().sendEvent(MLM_FORMAT, IEventParameter1(MLFIE_NOTRANS),
                                         IEventParameter2(0));
          break;
     case MLEFormat:
          handle().sendEvent(MLM_FORMAT, IEventParameter1(MLFIE_WINFMT),
                                         IEventParameter2(0));
          break;
   } /* endswitch */

   /**************************************************************/
   /* Import text from the buffer into the MLE                   */
   /**************************************************************/
   import(&ipt,ulBufferSize,pszBuffer);
   return *this;
}
#endif

/*------------------------------------------------------------------------------
| IMultiLineEdit::add                                                          |
|                                                                              |
| Insert the specified text into the MLE at the current cursor position.       |
------------------------------------------------------------------------------*/
IMultiLineEdit& IMultiLineEdit::add( const char*   pszBuffer,
                                     unsigned long ulBufferSize,
                                     EOLFormat     type )
{
   if (ulBufferSize == 0)                 // If no size specified,
      ulBufferSize = strlen(pszBuffer);   // Assume null-terminated string
   long ipt(-1);                          // Insert at current cursor position
#ifdef IC_WIN
   ipt = cursorPosition();
   handle().sendEvent( EM_REPLACESEL,
                       IEventParameter1(0),
                       IEventParameter2((char*)pszBuffer) );
   setCursorPosition(ipt+ulBufferSize);   // Position cursor after newly
                                          // inserted text
#else
   switch (type)
   {
     case cfText:
          handle().sendEvent(MLM_FORMAT, IEventParameter1(MLFIE_CFTEXT),
                                         IEventParameter2(0));
          break;
     case noTran:
          handle().sendEvent(MLM_FORMAT, IEventParameter1(MLFIE_NOTRANS),
                                         IEventParameter2(0));
          break;
     case MLEFormat:
          handle().sendEvent(MLM_FORMAT, IEventParameter1(MLFIE_WINFMT),
                                         IEventParameter2(0));
          break;
   } /* endswitch */

   /**************************************************************/
   /* Import text from the buffer into the MLE                   */
   /**************************************************************/
   import( &ipt, ulBufferSize, pszBuffer );

   setCursorPosition( ipt );     // Position cursor after newly inserted text
#endif
   return *this;
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::importFromFile                                               |
|                                                                              |
| Insert the contents of the specified file into the MLE at the current cursor |
| position.                                                                    |
| Notes: Prior to importing a file to the MLE, the file must exist (in binary  |
|          format).                                                            |
------------------------------------------------------------------------------*/
unsigned long  IMultiLineEdit::importFromFile( const char* pszFileName,
                                               EOLFormat   type )
{
   FILE*          filePtr;

   /***********************************************************************/
   /* Open the specified file.                                            */
   /***********************************************************************/
   struct stat buf;
   if (stat(pszFileName,&buf))
      buf.st_size=0;

#ifdef IC_WIN
   // If a limit has NOT been set specifically by application
   if (!flimitSet)
   {
     int fh;
     // Then get the overall size of the file
     if (( fh = open( pszFileName, O_RDONLY )) == -1 )
       ITHROWGUIERROR("_open");
     long length = filelength( fh );
     close( fh );

     // And if the size is greater than default, change the size
     if (length > fcharLimit)
     {
       IEventResult evt(handle().sendEvent( EM_LIMITTEXT,
                                            IEventParameter1( (int)length ),
                                            IEventParameter2( 0 ) ));
     }
   }
#endif
   if ((filePtr = fopen(pszFileName, "rb")) == (FILE *)0)
      ITHROWGUIERROR("fopen");
   IString data(0,buf.st_size+1);

   /***********************************************************************/
   /* Read a line from the file till the end of file                      */
   /***********************************************************************/
   Boolean bOldRefresh=bRefresh;
   if (bRefresh)
      disableUpdate();

   if (!(fread((char*)data,sizeof(char),buf.st_size-1,filePtr)))
      ITHROWGUIERROR("fread");
   add(data,buf.st_size-1,type);

// unsigned long  ulNumChars(0);
// char line[1025];
// int  i;
// int  j;
// for (j=0;j<buf.st_size ; )
// {
//    for (i=0;(i<=1023 && j+i<buf.st_size);i++)
//       if ((INT)(line[i]=fgetc(filePtr)) == 26)
//       {
//          i--;
//          j++;
//       }
//    if (i)
//    {
//       line[i] = 0;      // Null terminate (for Windows)
//       add(&line[0],i,type);
//    }
//    ulNumChars+=i;
//    j+=i;
// } /* endfor */

   if (bOldRefresh)
     enableUpdate();

   fclose(filePtr);
   return buf.st_size-1;
// return ulNumChars;
}

#ifndef IC_WIN  // IC_NOTYET
/*------------------------------------------------------------------------------
| IMultiLineEdit::exportToFile                                                 |
|                                                                              |
| Save the contents of the MLE to the specified file.                          |
------------------------------------------------------------------------------*/
unsigned long  IMultiLineEdit::exportToFile( const char* pszFileName,
                                             EOLFormat   type )
{
   if (textLengthAfterFormat())
   {
      switch (type)
      {
        case cfText:
             handle().sendEvent(MLM_FORMAT, IEventParameter1(MLFIE_CFTEXT),
                                            IEventParameter2(0));
             break;
        case noTran:
             handle().sendEvent(MLM_FORMAT, IEventParameter1(MLFIE_NOTRANS),
                                            IEventParameter2(0));
             break;
        case MLEFormat:
             handle().sendEvent(MLM_FORMAT, IEventParameter1(MLFIE_WINFMT),
                                            IEventParameter2(0));
             break;
      } /* endswitch */

      return exportFile(pszFileName, FALSE);

   } else {

      return 0;

   } /* endif */
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::exportSelectedTextToFile                                     |
|                                                                              |
| Export the marked area of MLE to a file.                                     |
------------------------------------------------------------------------------*/
unsigned long IMultiLineEdit::exportSelectedTextToFile ( const char* fileName,
                                                         EOLFormat   type)
{
  IASSERTSTATE(hasSelectedText());

  switch (type)
  {
    case cfText:
         handle().sendEvent(MLM_FORMAT, IEventParameter1(MLFIE_CFTEXT),
                                        IEventParameter2(0));
         break;
    case noTran:
         handle().sendEvent(MLM_FORMAT, IEventParameter1(MLFIE_NOTRANS),
                                        IEventParameter2(0));
         break;
    case MLEFormat:
         handle().sendEvent(MLM_FORMAT, IEventParameter1(MLFIE_WINFMT),
                                        IEventParameter2(0));
         break;
  } /* endswitch */

  return exportFile(fileName, TRUE);

}
#endif


/*------------------------------------------------------------------------------
| IMultiLineEdit::setLimit                                                     |
|                                                                              |
| Set the limit on the number of bytes that can be input into the MLE.         |
------------------------------------------------------------------------------*/
IMultiLineEdit& IMultiLineEdit::setLimit( unsigned long lNewLimit )
{
  /***************************/
  /* Only set if we need to. */
  /***************************/
  if (lNewLimit != this->limit())
  {
    IEventResult evt(handle().sendEvent(MLM_SETTEXTLIMIT,
                                        IEventParameter1(lNewLimit),
                                        IEventParameter2(0)));
#ifdef IC_WIN
    fcharLimit = lNewLimit;
    flimitSet = true;
#else
    // Check error code returned only on PM
    if (evt.asUnsignedLong())
    {
      ITHROWGUIERROR("MLM_SETTEXTLIMIT");
    }
#endif
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::limit                                                        |
|                                                                              |
| Returns the maximum number of bytes allowed that the MLE can hold.           |
------------------------------------------------------------------------------*/
unsigned long IMultiLineEdit::limit() const
{
#ifdef IC_PM
  IEventResult evt(handle().sendEvent(MLM_QUERYTEXTLIMIT,
                                      IEventParameter1(0),
                                      IEventParameter2(0)));

  if (evt.asLong() < 0)
     return ULONG_MAX;
  else
     return evt.asUnsignedLong();
#endif
#ifdef IC_WIN
  return fcharLimit;    // Must keep locally since Windows does not
                        // support querying of current limit set
#endif
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::selectRange                                                  |
|                                                                              |
| Specifies a range of characters.                                             |
| Notes: If upper range is not specified, default is last character in the MLE.|
------------------------------------------------------------------------------*/
IMultiLineEdit&  IMultiLineEdit::selectRange( const IRange& range,
                                              unsigned long timestamp )
{
   long start(range.lowerBound());
   long stop(range.upperBound());

   if (start == end)
      start = textLength()+1;

   if (stop == end)
      stop = textLength()+1;

   if ( stop  < 0)
      stop  = 0;

   if ( start < 0)
      start = 0;

   if (start > stop )
      start++;   //Need to move cursor past character so we can select it
   else
      stop++;    //Need to move cursor past character so we can select it

   handle().sendEvent(MLM_SETSEL, IEventParameter1((unsigned long)start),
                                  IEventParameter2((unsigned long)stop));

   return *this;
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::selectedRange                                                |
|                                                                              |
| Gets the bound of the marked or selected text.  Index of marked characters   |
| is returned as 0-based.                                                      |
------------------------------------------------------------------------------*/
IRange  IMultiLineEdit::selectedRange() const
{
  IASSERTSTATE(hasSelectedText());
#ifdef IC_WIN
  IEventResult minSel, maxSel;
  handle().sendEvent(EM_GETSEL,
                     IEventParameter1(&minSel),
                     IEventParameter2(&maxSel));
#else
  IEventResult minSel( handle().sendEvent(MLM_QUERYSEL,
                                          IEventParameter1(MLFQS_MINSEL),
                                          IEventParameter2(0)));
  IEventResult maxSel( handle().sendEvent(MLM_QUERYSEL,
                                          IEventParameter1(MLFQS_MAXSEL),
                                          IEventParameter2(0)));
#endif
  IRange rangMarked(minSel.asUnsignedLong(), maxSel.asUnsignedLong());
  if (rangMarked.lowerBound() != rangMarked.upperBound())
     rangMarked.setUpperBound(rangMarked.upperBound() - 1);
                                  // Make upper the last selected character
  return rangMarked;
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::selectedText                                                 |
|                                                                              |
| Gets the marked (selected) text.                                             |
------------------------------------------------------------------------------*/
IString IMultiLineEdit::selectedText() const
{
  char  *pchMarkedText;           // Source text

  unsigned long length = selectedTextLength();
  if (!length)
  {                               // No selected text
     IString str("\0");
     return str;
  } else {
     IString str(IString(0,length+1));
     pchMarkedText = (char *)str;
#ifndef IC_WIN
     // IC_NOTYET - Cannot determine easy way to query ONLY the selected
     //             text from a Windows Entry Field
     handle().sendEvent(MLM_QUERYSELTEXT,
                             IEventParameter1((unsigned long) pchMarkedText),
                             IEventParameter2(0));
#endif
     return str;
  } /*endif */
}


/*------------------------------------------------------------------------------
| IMultiLineEdit::hasSelectedText                                              |
|                                                                              |
| Query whether any text is currently marked.                                  |
------------------------------------------------------------------------------*/
Boolean IMultiLineEdit::hasSelectedText() const
{
  if (selectedTextLength() == 0)
    return( false );
  else
    return( true );
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::isUndoable                                                   |
|                                                                              |
| Queries whether any undoable actions have been performed on the contents of  |
| the MLE.                                                                     |
------------------------------------------------------------------------------*/
Boolean IMultiLineEdit::isUndoable() const
{
  IEventResult evt(handle().sendEvent(MLM_QUERYUNDO,
                                      IEventParameter1(0),
                                      IEventParameter2(0)));
  if (evt.asUnsignedLong())
     return true;
  else
     return false;
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::cut                                                          |
|                                                                              |
| Copy any marked text to the clipboard, and then delete the marked text from  |
| the MLE.                                                                     |
------------------------------------------------------------------------------*/
IMultiLineEdit& IMultiLineEdit::cut(unsigned long timestamp)
{
  IASSERTSTATE(hasSelectedText());

  IEventResult evt(handle().sendEvent(MLM_CUT,
                                      IEventParameter1(0),
                                      IEventParameter2(0)));

  if (!(evt.asUnsignedLong()))
     ITHROWGUIERROR("MLM_CUT");

  return *this;
}


/*------------------------------------------------------------------------------
| IMultiLineEdit::copy                                                         |
|                                                                              |
| Copy any marked text to the clipboard.                                       |
------------------------------------------------------------------------------*/
IMultiLineEdit& IMultiLineEdit::copy(unsigned long timestamp)
{
  IASSERTSTATE(hasSelectedText());

  IEventResult evt(handle().sendEvent(MLM_COPY,
                                      IEventParameter1(0),
                                      IEventParameter2(0)));

  if (!(evt.asUnsignedLong()))
     ITHROWGUIERROR("MLM_COPY");

  return *this;
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::paste                                                        |
|                                                                              |
| Paste text from the clipboard into the MLE at the current cursor position    |
| in the text.                                                                 |
------------------------------------------------------------------------------*/
IMultiLineEdit& IMultiLineEdit::paste()
{
  IASSERTSTATE(clipboardHasTextFormat());

  IEventResult evt(handle().sendEvent(MLM_PASTE,
                                      IEventParameter1(0),
                                      IEventParameter2(0)));

  if (!(evt.asUnsignedLong()))
     ITHROWGUIERROR("MLM_PASTE");

  return *this;
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::clear                                                        |
|                                                                              |
| Replace the marked area with blanks.                                         |
------------------------------------------------------------------------------*/
IMultiLineEdit& IMultiLineEdit::clear(unsigned long timestamp)
{
  char    *pchMarkedText;              // Source text
  ULONG    num_char, i;                // char and loop count

  IASSERTSTATE(hasSelectedText());
  IString str(selectedText());
  pchMarkedText = (char *)str;

  /***********************************************************************/
  /* Get the size of user marked area.                                   */
  /***********************************************************************/
  num_char = selectedTextLength();

  /******************************************************************/
  /* Replace the text with blanks, leaving CR and LF chars intact.  */
  /******************************************************************/
  for (i = 0; i < num_char; i++)
      if (*(pchMarkedText + i) != 0x0D && *(pchMarkedText +i) != 0x0A)
         *(pchMarkedText + i) = ' ';

  /******************************************************************/
  /* Replace the marked area with the blanked out text.             */
  /******************************************************************/
#ifdef IC_WIN
  handle().sendEvent( EM_REPLACESEL,
                      IEventParameter1(0),
                      IEventParameter2((unsigned long)pchMarkedText));
#else
  Boolean bOldRefresh=bRefresh;
  if (bRefresh)
     disableUpdate();

  handle().sendEvent( MLM_INSERT,
                      IEventParameter1((unsigned long)pchMarkedText),
                      IEventParameter2(0));

  if (bOldRefresh)
     enableUpdate();
#endif

  return *this;
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::discard                                                      |
|                                                                              |
| Erase the marked text.                                                       |
------------------------------------------------------------------------------*/
IMultiLineEdit& IMultiLineEdit::discard()
{
  IASSERTSTATE(hasSelectedText());

  IEventResult evt(handle().sendEvent(MLM_CLEAR,
                                      IEventParameter1(0),
                                      IEventParameter2(0)));

  if (!(evt.asUnsignedLong()))
     ITHROWGUIERROR("MLM_CLEAR");

  return *this;
}


/*------------------------------------------------------------------------------
| IMultiLineEdit::undo                                                         |
|                                                                              |
| Restores the MLE contents to the state they were in before the last change.  |
------------------------------------------------------------------------------*/
IMultiLineEdit& IMultiLineEdit::undo()
{
   /********************************************************************/
   /* Request PM to undo the last operation, if there was one.         */
   /********************************************************************/
   IEventResult evt(handle().sendEvent(MLM_UNDO,
                                       IEventParameter1(0),
                                       IEventParameter2(0)));

#ifndef IC_WIN
   if (!(evt.asUnsignedLong()))
      /*****************************************************************/
      /* Undo failed, so lets reset the MLE undo flag.                 */
      /*****************************************************************/
      handle().sendEvent(MLM_RESETUNDO, IEventParameter1(0),
                                        IEventParameter2(0));
#endif
   return *this;
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::removeAll                                                    |
|                                                                              |
| Deletes the entire contents of the MLE.                                      |
------------------------------------------------------------------------------*/
IMultiLineEdit& IMultiLineEdit::removeAll()
{
   Boolean bOldRefresh(bRefresh);

#ifdef IC_WIN
   // No delete all function in Windows so just set contents to NULL string
   setText((char*)IString(),0);
#else
   if (bRefresh)
      disableUpdate();

   handle().sendEvent(MLM_DELETE, IEventParameter1(0),
                                  IEventParameter2(textLengthAfterFormat()));

   if (bOldRefresh)
      enableUpdate(bOldRefresh);
#endif
   return *this;
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::setEditRegionWidth                                           |
|                                                                              |
| Set the width (in pixels) of the MLE text region.  The edit region defines   |
| the area in which text will be visible.  It can be less than the actual      |
| MLE window size.                                                             |
------------------------------------------------------------------------------*/
IMultiLineEdit& IMultiLineEdit::setEditRegionWidth(long sWidth)
{
#ifdef IC_WIN
   RECT mleRect;

   handle().sendEvent( EM_GETRECT,
                       IEventParameter1(0),
                       IEventParameter2((unsigned long)&mleRect) );
   // IC_NOTYET - Based on ref text, may need to adjust returned rect
   // by border size for set to work correctly.
   mleRect.right = mleRect.left + sWidth;

   // Note no exception thrown since message does not return value
   IEventResult evt( handle().sendEvent(EM_SETRECT,
                       IEventParameter1(0),
                       IEventParameter2((unsigned long)&mleRect)));

   //---------------------------------------------------------------------
   // DEFECT 25510
   //---------------------------------------------------------------------
   if ( !isWordWrap() ) {
     IString msg = IString("WARNING: setEditRegionWidth() has no effect") +
                   IString(" in a WINDOWS MLE without WordWrap.");
     ITRACE_DEVELOP( msg );
   }
   //---------------------------------------------------------------------
#else
   MLEFORMATRECT mleformrc;
   mleformrc.cxFormat = sWidth;
   Boolean bOldRefresh=bRefresh;

   if (bRefresh)
      disableUpdate();

   IEventResult evt(
           handle().sendEvent(MLM_SETFORMATRECT,
                              IEventParameter1((unsigned long)(&mleformrc)),
                              IEventParameter2((unsigned long)
                                                        MLFFMTRECT_LIMITHORZ)));
   if (bOldRefresh)
      enableUpdate();
   if (!(evt.asUnsignedLong()))
      ITHROWLIBRARYERROR(IC_INVALIDSIZE,IErrorInfo::invalidParameter,
                         IException::recoverable);
#endif
   return *this;
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::setEditRegionHeight                                          |
|                                                                              |
| Set the height of the text region inside of the MLE control.                 |
------------------------------------------------------------------------------*/
IMultiLineEdit& IMultiLineEdit::setEditRegionHeight(long sHeight)
{
#ifdef IC_WIN
   RECT mleRect;

   handle().sendEvent( EM_GETRECT,
                       IEventParameter1(0),
                       IEventParameter2((unsigned long)&mleRect) );
   // IC_NOTYET - Based on ref text, may need to adjust returned rect
   // by border size for set to work correctly.
   mleRect.bottom = mleRect.top + sHeight;

   // Note no exception thrown since message does not return value
   IEventResult evt( handle().sendEvent(EM_SETRECT,
                       IEventParameter1(0),
                       IEventParameter2((unsigned long)&mleRect)));
#else
   MLEFORMATRECT mleformrc;
   mleformrc.cyFormat = sHeight;
   Boolean bOldRefresh=bRefresh;

   if (bRefresh)
      disableUpdate();

   IEventResult evt(
           handle().sendEvent(MLM_SETFORMATRECT,
                              IEventParameter1((unsigned long)(&mleformrc)),
                              IEventParameter2((unsigned long)
                                                     MLFFMTRECT_LIMITVERT)));
   if (bOldRefresh)
      enableUpdate();
   if (!(evt.asUnsignedLong()))
      ITHROWLIBRARYERROR(IC_INVALIDSIZE,IErrorInfo::invalidParameter,
                         IException::recoverable);
#endif
   return *this;
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::setEditRegion                                                |
|                                                                              |
| Set the width and the height of the text region inside of the MLE control.   |
| Method returns TRUE if the text fits within new format-rectangle dimension.  |
| Otherwise, it returns FALSE, indicating that the format rectangle was not    |
| set.                                                                         |
------------------------------------------------------------------------------*/
IMultiLineEdit& IMultiLineEdit::setEditRegion( const ISize& sizeNew )
{
#ifdef IC_WIN
   RECT mleRect;

   handle().sendEvent( EM_GETRECT,
                       IEventParameter1(0),
                       IEventParameter2((unsigned long)&mleRect) );
   // IC_NOTYET - Based on ref text, may need to adjust returned rect
   // by border size for set to work correctly.

   mleRect.right = mleRect.left + (long)sizeNew.width();
   mleRect.bottom = mleRect.top + (long)sizeNew.height();

   // Note no exception thrown since message does not return value
   IEventResult evt( handle().sendEvent(EM_SETRECT,
                       IEventParameter1(0),
                       IEventParameter2((unsigned long)&mleRect)));
#else
   MLEFORMATRECT mleformrc;
   mleformrc.cxFormat = (long)sizeNew.width();
   mleformrc.cyFormat = (long)sizeNew.height();
   Boolean bOldRefresh=bRefresh;

   if (bRefresh)
      disableUpdate();

   IEventResult evt(
           handle().sendEvent(MLM_SETFORMATRECT,
                              IEventParameter1((unsigned long)(&mleformrc)),
                              IEventParameter2((unsigned long)
                                                  (MLFFMTRECT_LIMITVERT |
                                                   MLFFMTRECT_LIMITHORZ))));
   if (bOldRefresh)
      enableUpdate();

   if (!(evt.asUnsignedLong()))
      ITHROWLIBRARYERROR(IC_INVALIDSIZE,IErrorInfo::invalidParameter,
                         IException::recoverable);
#endif
   return *this;
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::setEditRegion                                                |
|                                                                              |
| Set the edit region to be the size of the entire MLE window (minus margins). |
------------------------------------------------------------------------------*/
IMultiLineEdit& IMultiLineEdit::setEditRegion()
{
#ifdef IC_WIN
   IRectangle mleRectangle = rect();
   RECTL mleRect = mleRectangle.asRECTL();

   // IC_NOTYET - Based on ref text, may need to adjust returned rect
   // by border size for set to work correctly.

   // Note no exception thrown since message does not return value
   IEventResult evt( handle().sendEvent(EM_SETRECT,
                     IEventParameter1(0),
                     IEventParameter2((unsigned long)&mleRect)));
#else
   MLEFORMATRECT mleformrcNew;
   Boolean bOldRefresh=bRefresh;
   if (bRefresh)
      disableUpdate();
   IEventResult evt(
           handle().sendEvent(MLM_SETFORMATRECT,
                              IEventParameter1((unsigned long)(&mleformrcNew)),
                              IEventParameter2((unsigned long)
                                                      MLFFMTRECT_MATCHWINDOW)));
   if (bOldRefresh)
      enableUpdate();

   if (!(evt.asUnsignedLong()))
      ITHROWLIBRARYERROR(IC_INVALIDSIZE,IErrorInfo::accessError,
                         IException::recoverable);
#endif
   return *this;
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::editRegionWidth                                              |
|                                                                              |
| Returns the width of the MLE editing area.                                   |
------------------------------------------------------------------------------*/
unsigned long IMultiLineEdit::editRegionWidth() const
{
#ifdef IC_WIN
   RECT mleRect;

   handle().sendEvent( EM_GETRECT,
                       IEventParameter1(0),
                       IEventParameter2((unsigned long)&mleRect) );
   // IC_NOTYET - Based on ref text, may need to adjust returned rect
   // by border size for set to work correctly.
   return (mleRect.right - mleRect.left);
#else
   MLEFORMATRECT mleformrc;
   unsigned long         ulFormatFlag =  MLFFMTRECT_LIMITHORZ;

   /***********************************************************************/
   /* Get the extent of the edit area.                                    */
   /***********************************************************************/
   handle().sendEvent(MLM_QUERYFORMATRECT,
                      IEventParameter1((unsigned long)(&mleformrc)),
                      IEventParameter2((unsigned long)(&ulFormatFlag)));

   return (mleformrc.cxFormat);
#endif
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::editRegionHeight                                             |
|                                                                              |
| Returns the height of the MLE editing area.                                  |
------------------------------------------------------------------------------*/
unsigned long IMultiLineEdit::editRegionHeight() const
{
#ifdef IC_WIN
   RECT mleRect;

   handle().sendEvent( EM_GETRECT,
                       IEventParameter1(0),
                       IEventParameter2((unsigned long)&mleRect) );
   // IC_NOTYET - Based on ref text, may need to adjust returned rect
   // by border size for set to work correctly.
   return (mleRect.bottom - mleRect.top);
#else
   MLEFORMATRECT mleformrc;
   unsigned long         ulFormatFlag =  MLFFMTRECT_LIMITVERT;

   /***********************************************************************/
   /* Get the extent of the edit area.                                    */
   /***********************************************************************/
   handle().sendEvent(MLM_QUERYFORMATRECT,
                      IEventParameter1((unsigned long)(&mleformrc)),
                      IEventParameter2((unsigned long)(&ulFormatFlag)));
   return (mleformrc.cyFormat);
#endif
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::setTab                                                       |
|                                                                              |
| Sets the interval at which tab stops are placed (expressed in pixels).       |
------------------------------------------------------------------------------*/
IMultiLineEdit& IMultiLineEdit::setTab(unsigned long lTabPixelInterval)
{
#ifdef IC_WIN
   // IC_NOTYET - Need logic here to convert from pixels to dialog box units
   unsigned long lTabDistance = lTabPixelInterval;
   IEventResult evt(
           handle().sendEvent(EM_SETTABSTOPS,
                              IEventParameter1(1),
                              IEventParameter2(lTabDistance)));

   if (evt.asLong() == 0)
      ITHROWGUIERROR("MLM_SETTABSTOP");
#else
   IEventResult evt(
           handle().sendEvent(MLM_SETTABSTOP,
                              IEventParameter1((unsigned long)
                                                       lTabPixelInterval),
                              IEventParameter2(0)));

   if (evt.asLong() < 0)
      ITHROWGUIERROR("MLM_SETTABSTOP");
#endif
   return *this;
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::isWordWrap                                                   |
------------------------------------------------------------------------------*/
Boolean IMultiLineEdit::isWordWrap() const
{
#ifdef IC_PM
   IEventResult evt(handle().sendEvent(MLM_QUERYWRAP,
                                       IEventParameter1(0),
                                       IEventParameter2(0)));
   if (evt.asUnsignedLong())
      return true;
   else
      return false;
#endif
#ifdef IC_WIN
  // Always true in Windows.
  return true;
#endif
}

#ifndef IC_WIN
/*------------------------------------------------------------------------------
| IMultiLineEdit::enableWordWrap                                               |
|                                                                              |
| Puts the MLE in word-wrap mode, which means lines will not be scrolled       |
| right or left.                                                               |
------------------------------------------------------------------------------*/
IMultiLineEdit& IMultiLineEdit::enableWordWrap( Boolean enable )
{
  Boolean wordWrap(this->isWordWrap());

  /***************************/
  /* Only set if we need to. */
  /***************************/
  if ((enable && !wordWrap) || (!enable && wordWrap))
  {
#ifdef IC_PM
    IEventResult evt(handle().sendEvent(MLM_SETWRAP,
                                        IEventParameter1(enable),
                                        IEventParameter2(0)));

    if (!(evt.asUnsignedLong()))
    {
      ITHROWGUIERROR("MLM_SETWRAP");
    }
#endif

#ifdef IC_WIN
    //---------------------------------------------------------------------
    // DEFECT 25510
    //---------------------------------------------------------------------
    if ( enable && (style() & MLS_HSCROLL) ) {
      ITRACE_DEVELOP("WARNING: WINDOWS cannot WordWrap on MLE with HorizScroll.\n\tenableWordWrap() ignored.");
    }
    else if ( !enable ) {
      ITRACE_DEVELOP("WARNING: WINDOWS cannot disable WordWrap on MLE.\n\tenableWordWrap() ignored.");
    }
    else {
      unsigned long extStyle = extendedStyle();
      extStyle &= ~IMLS_WORDWRAP;
      setExtendedStyle( extStyle );
    }
    //---------------------------------------------------------------------
#endif
  }

  return *this;
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::disableWordWrap                                              |
------------------------------------------------------------------------------*/
IMultiLineEdit& IMultiLineEdit::disableWordWrap()
{
   enableWordWrap(false);
   return *this;
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::foregroundColor                                              |
|                                                                              |
| Returns the foreground color of the IMultiLineEdit control.                  |
------------------------------------------------------------------------------*/
IColor IMultiLineEdit::foregroundColor() const
{
  IGUIColor guiColor(IGUIColor::windowText);

  if (!isWriteable())
  {
    guiColor = IGUIColor(IGUIColor::outputText);
  }
  return (IWindow::color(PP_FOREGROUNDCOLOR, guiColor));
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::backgroundColor                                              |
|                                                                              |
| Returns the background color of the IMultiLineEdit control.                  |
------------------------------------------------------------------------------*/
IColor IMultiLineEdit::backgroundColor() const
{
  return (IWindow::color(PP_BACKGROUNDCOLOR,
                         IGUIColor(IGUIColor::entryFieldBgnd)));
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::setFont                                                      |
|                                                                              |
| Change the font used for this MLE object.                                    |
| Notes: The font point size set by this routine is the best fit size          |
|          determined by the system instead of the size specified by the user. |
------------------------------------------------------------------------------*/
IMultiLineEdit& IMultiLineEdit::setFont(const IFont& fntm)
{
   _FATTRS* pfat = (_FATTRS*)fntm.fattrs();

   if (pfat->fsSelection & FATTR_SEL_OUTLINE)
   {
      pfat->fsFontUse |=FATTR_FONTUSE_OUTLINE;
   } /* endif */

   if (fntm.isVectorOnly()) {
      pfat->fsFontUse |=FATTR_FONTUSE_TRANSFORMABLE;
      long  lxFontResolution, lyFontResolution;
      HDC   hDC;
      HPS hps = WinGetPS(handle());
      hDC = GpiQueryDevice(hps);
      DevQueryCaps( hDC, CAPS_HORIZONTAL_FONT_RES,
                    (long)1, &lxFontResolution);
      DevQueryCaps( hDC, CAPS_VERTICAL_FONT_RES,
                    (long)1, &lyFontResolution);
      pfat->lMaxBaselineExt = fntm.pointSize() * lyFontResolution / 72;
      pfat->lAveCharWidth   = fntm.pointSize() * lxFontResolution / 72;
      WinReleasePS(hps);
      }

   IEventResult evt(handle().sendEvent(MLM_SETFONT,
                                       IEventParameter1((unsigned long)pfat),
                                       IEventParameter2(0)));

   if (!(evt.asUnsignedLong()))
      ITHROWGUIERROR("MLM_SETFONT");
   return *this;
}
#endif

/*------------------------------------------------------------------------------
| IMultiLineEdit::numberOfLines                                                |
|                                                                              |
| Query the number of lines of text in the MLE.                                |
------------------------------------------------------------------------------*/
unsigned long IMultiLineEdit::numberOfLines() const
{
   unsigned long ulRetVal,ipt;

   /******************************************************************/
   /* Query number of lines in the MLE                               */
   /******************************************************************/
   ulRetVal = (unsigned long) handle().sendEvent( MLM_QUERYLINECOUNT,
                                                  IEventParameter1(0),
                                                  IEventParameter2(0));

   /******************************************************************/
   /* Get the position of the first character on the last line       */
   /******************************************************************/
   ipt = (unsigned long) handle().sendEvent( MLM_CHARFROMLINE,
                                             IEventParameter1(ulRetVal),
                                             IEventParameter2(0) );

   /******************************************************************/
   /* Check last line, if line length is 0, subtract one from count  */
   /******************************************************************/
   if ((unsigned long) handle().sendEvent(MLM_QUERYLINELENGTH,
                  IEventParameter1(ipt), IEventParameter2(0)) != 0)
     return ulRetVal;
   else
     return ulRetVal-1;
}


/*------------------------------------------------------------------------------
| IMultiLineEdit::addLineAsLast                                                |
|                                                                              |
| Insert the specified NULL-terminated text at the last line.                  |
------------------------------------------------------------------------------*/
IMultiLineEdit& IMultiLineEdit::addLineAsLast( const char* pszLineText,
                                               EOLFormat   type )
{
   addLine(pszLineText,numberOfLines()+1,type);
   return *this;
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::addLine                                                      |
|                                                                              |
| Insert the specified line of text at the line location specified.            |
------------------------------------------------------------------------------*/
IMultiLineEdit& IMultiLineEdit::addLine( const char*   pszLineText,
                                         unsigned long lLineNumber,
                                         EOLFormat     type )
{
   long ipt;
   unsigned long  ulBuffSize;
   char * pszTempBuff;
   char pszEolChar[3];
   unsigned long ulFormat;
   Boolean bTextSel;
   IRange  selRange;

   strcpy( pszEolChar, "\r\n");
   switch (type)
   {
     case cfText:
        ulFormat = MLFIE_CFTEXT;
        break;
     case noTran:
        ulFormat = MLFIE_NOTRANS;
        strcpy( pszEolChar, "\n");
        break;
     case MLEFormat:
        ulFormat = MLFIE_WINFMT;
        break;
   } /* endswitch */

#ifdef IC_WIN
   handle().sendEvent( EM_FMTLINES,
                       IEventParameter1(ulFormat == MLFIE_WINFMT),
                       IEventParameter2(0));
#else
   handle().sendEvent( MLM_FORMAT,
                       IEventParameter1(ulFormat),
                       IEventParameter2(0));
#endif

   if (bTextSel = hasSelectedText())
     selRange = selectedRange();

   // If line number to add is before end of file, calculate insertion point
   if (lLineNumber < numberOfLines())
   {
      ipt = (long)((unsigned long) handle().sendEvent(MLM_CHARFROMLINE,
                              IEventParameter1(lLineNumber),
                              IEventParameter2(0)));
      ulBuffSize = strlen(pszLineText) + strlen(pszEolChar) + 1;
      pszTempBuff = new char[ulBuffSize];        // Allocate storage

      strcpy( pszTempBuff, pszLineText);
      strcat( pszTempBuff, pszEolChar);          // Add line terminator

#ifdef IC_WIN
      handle().sendEvent( EM_SETSEL,
                          IEventParameter1((unsigned long)ipt),
                          IEventParameter2((unsigned long)ipt) );
      handle().sendEvent( EM_REPLACESEL,
                          IEventParameter1(0),
                          IEventParameter2(pszTempBuff) );
#else
      import(&ipt,ulBuffSize,pszTempBuff);                   // Import text,

      handle().sendEvent( MLM_DELETE,
                          IEventParameter1((unsigned long)ipt-1),
                          IEventParameter2(1));
#endif
      if (bTextSel)
        selectRange( selRange );
      delete pszTempBuff;

   }
   // otherwise line is to be added at the end of the file
   else
   {
      ipt = textLength();
      if (ipt!=0)
      {
#ifdef IC_WIN
         handle().sendEvent( EM_SETSEL,
                             IEventParameter1((unsigned long)ipt),
                             IEventParameter2((unsigned long)ipt) );
         handle().sendEvent( EM_REPLACESEL,
                             IEventParameter1(0),
                             IEventParameter2(pszEolChar) );
#else
         import(&ipt, strlen(pszEolChar)+1, pszEolChar );    // Import an eol char
         handle().sendEvent(MLM_DELETE,
                                 IEventParameter1((unsigned long)ipt-1),
                                 IEventParameter2(1));
#endif
         ipt--;
      } /* endif */

#ifdef IC_WIN
      handle().sendEvent( EM_SETSEL,
                          IEventParameter1((unsigned long)ipt),
                          IEventParameter2((unsigned long)ipt) );
      handle().sendEvent( EM_REPLACESEL,
                          IEventParameter1(0),
                          IEventParameter2((char*)pszLineText) );
#else
      import(&ipt,strlen(pszLineText),pszLineText);  // import text,
#endif
      if (bTextSel)
        selectRange( selRange );
   } /* endif */

   return *this;
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::removeLine                                                   |
|                                                                              |
| Deletes the text on the specified line of the MLE.                           |
------------------------------------------------------------------------------*/
IMultiLineEdit& IMultiLineEdit::removeLine( unsigned long lLineNumber )
{
   unsigned long ipt;
   unsigned long ulNumChars;

   if (lLineNumber <= numberOfLines())
      ipt = (unsigned long) handle().sendEvent(MLM_CHARFROMLINE,
                                 IEventParameter1(lLineNumber),
                                 IEventParameter2(0));
   else
      return *this;

   ulNumChars = (unsigned long) handle().sendEvent(MLM_QUERYLINELENGTH,
                                  IEventParameter1(ipt),
                                  IEventParameter2(0));
#ifndef IC_WIN
// IC_NOTYET - Cannot figure out how to delete the line without either
//             selecting text and deleting selection OR copying the entire
//             buffer over, deleting the text and then resetting the entire
//             text of the control to the new buffer.
   handle().sendEvent(MLM_DELETE, IEventParameter1(ipt),
                                  IEventParameter2(ulNumChars));
#endif
   return *this;
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::setTop                                                       |
|                                                                              |
| Make the specified line be the first visible line of the MLE on the screen.  |
------------------------------------------------------------------------------*/
IMultiLineEdit& IMultiLineEdit::setTop( unsigned long lLineNumber )
{

#ifdef IC_WIN
   unsigned long scrollAmount = top() - lLineNumber;

   handle().sendEvent( EM_LINESCROLL,
                       IEventParameter1(0),
                       IEventParameter2(scrollAmount) );
#else
   IEventResult evt(handle().sendEvent(MLM_CHARFROMLINE,
                                       IEventParameter1(lLineNumber),
                                       IEventParameter2(0)));

   IEventResult evt2(handle().sendEvent(MLM_SETFIRSTCHAR,
                                        IEventParameter1(evt.asUnsignedLong()),
                                        IEventParameter2(0)));

   if (!(evt2.asUnsignedLong()))
      ITHROWGUIERROR("MLM_SETFIRSTCHAR");
#endif

   return *this;
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::top                                                          |
|                                                                              |
| Returns the line number of the line currently displayed as the first visible |
| line on the MLE.                                                             |
------------------------------------------------------------------------------*/
unsigned long IMultiLineEdit::top() const
{
#ifdef IC_WIN
   IEventResult evt2(handle().sendEvent(EM_GETFIRSTVISIBLELINE,
                                        IEventParameter1(0),
                                        IEventParameter2(0)));
#else
   IEventResult evt(handle().sendEvent(MLM_QUERYFIRSTCHAR,
                                       IEventParameter1(0),
                                       IEventParameter2(0)));

   IEventResult evt2(handle().sendEvent(MLM_LINEFROMCHAR,
                                        IEventParameter1(evt.asUnsignedLong()),
                                        IEventParameter2(0)));
#endif
   return evt2.asUnsignedLong();
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::setCursorLinePosition                                        |
|                                                                              |
| Position the cursor on the line specified.                                   |
------------------------------------------------------------------------------*/
IMultiLineEdit& IMultiLineEdit::setCursorLinePosition(unsigned long lLineNumber)
{
   IEventResult evt(handle().sendEvent(MLM_CHARFROMLINE,
                                       IEventParameter1(lLineNumber),
                                       IEventParameter2(0)));

   handle().sendEvent(MLM_SETSEL, IEventParameter1(evt.asUnsignedLong()),
                                  IEventParameter2(evt.asUnsignedLong()));
   return *this;
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::cursorLinePosition                                           |
|                                                                              |
| Returns the line number where the cursor is currently positioned.            |
------------------------------------------------------------------------------*/
unsigned long IMultiLineEdit::cursorLinePosition() const
{
   IEventResult evt(handle().sendEvent( MLM_LINEFROMCHAR,
                                        IEventParameter1(cursorPosition()),
                                        IEventParameter2(0)));

   return evt.asUnsignedLong();
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::cursorPosition                                               |
|                                                                              |
| Returns the character position of where the cursor currently is.             |
------------------------------------------------------------------------------*/
unsigned long IMultiLineEdit::cursorPosition() const
{
#ifdef IC_WIN
   unsigned long endPos;
   IEventResult evt(handle().sendEvent(MLM_QUERYSEL,
                                       IEventParameter1(0),
                                       IEventParameter2(&endPos)));
   return endPos;
#else
   IEventResult evt(handle().sendEvent(MLM_QUERYSEL,
                                       IEventParameter1(MLFQS_CURSORSEL),
                                       IEventParameter2(0)));
   return evt.asUnsignedLong();
#endif
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::setCursorPosition                                            |
|                                                                              |
| Position the cursor in front of the character specified.                     |
------------------------------------------------------------------------------*/
IMultiLineEdit& IMultiLineEdit::setCursorPosition( unsigned long lCursorChar )
{
   IEventResult evt;
   evt = handle().sendEvent(MLM_SETSEL, IEventParameter1(lCursorChar),
                                        IEventParameter2(lCursorChar));

   //----------------------------------------------------------------
   // DEFECT 25702 : if not in view, scroll to 1st char position on line.
   //----------------------------------------------------------------
#ifndef IC_WIN
   unsigned long ulIndex;
   ulIndex = ( (lCursorChar > 0 ? lCursorChar-1 : 0) );
   // must move to the correct line, then scroll char into view
   evt = handle().sendEvent(MLM_LINEFROMCHAR,
                            IEventParameter1(lCursorChar),
                            IEventParameter2(0));
   setTop( evt.asUnsignedLong() );
   evt = handle().sendEvent(EM_SETFIRSTCHAR,
                            IEventParameter1(ulIndex),
                            IEventParameter2(0));
#else
   evt = handle().sendEvent(EM_SCROLLCARET,
                            IEventParameter1(0),
                            IEventParameter2(0));
#endif
   //----------------------------------------------------------------

   return *this;
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::visibleLines                                                 |
|                                                                              |
| Returns the number of visible lines that will fit in the MLE using the       |
| current font.                                                                |
------------------------------------------------------------------------------*/
unsigned long IMultiLineEdit::visibleLines() const
{
   long lRetVal=0;
#ifndef IC_WIN
// IC_NOTYET - Need to get Font code working first
   FATTRS fattrs;

   handle().sendEvent(MLM_QUERYFONT,
                      IEventParameter1((unsigned long)(PFATTRS(&fattrs))),
                      IEventParameter2(0));

   long lRegionHeight = editRegionHeight();
   long lCharHeight = fattrs.lMaxBaselineExt;

   if ((lRegionHeight==0) || (lCharHeight==0))
      return 0;

   lRetVal = ( lRegionHeight / lCharHeight );
#endif
   return lRetVal;
}


#ifdef IC_PM
/*------------------------------------------------------------------------------
| Private methods                                                              |
------------------------------------------------------------------------------*/
/*------------------------------------------------------------------------------
| IMultiLineEdit::exportFile                                                   |
|                                                                              |
| Exports either the entire file, or only the marked area, to a file.          |
------------------------------------------------------------------------------*/
long IMultiLineEdit::exportFile( const char* pszFileName,
                                 Boolean     flMarkedAreaOnly )
{
   unsigned long      ulBufSize;        // Buffer size
   long        iptOrigin;               // Origin of the MLE object
   FILE        *filePtr;                // File pointer of specified file
   unsigned long      ulNumChars;       // Number of chars exported by MLE
   int         numItems=1;

   /***********************************************************************/
   /* Open file in binary mode.  If file exists, erase its contents.  If  */
   /* file does not exist, create it for writing or reading.              */
   /* If file cannot be opened for writing (i.e. readonly) return 0 as    */
   /* number of bytes written                                             */
   /***********************************************************************/
   if ((filePtr = fopen(pszFileName, "wb")) == (FILE *)0)
     return 0;

   /***********************************************************************/
   /* Check if we need to export all text or only selected text           */
   /***********************************************************************/
   if (flMarkedAreaOnly == FALSE)
   {
     ulNumChars = ulBufSize  = textLengthAfterFormat();
     iptOrigin = 0;
   }
   /***********************************************************************/
   /* If selected, determine area the user has marked area for exporting. */
   /***********************************************************************/
   else
   {
     IRange rang = selectedRange();
     iptOrigin = rang.lowerBound();
     ulNumChars = ulBufSize = selectedTextLength();
   }

   /***********************************************************************/
   /* Allocate Maximum buffer for the import/export operation.            */
   /***********************************************************************/
   IString pchBuf(0,ulBufSize+1);

   /***********************************************************************/
   /* Export MLE to the specified buffer - it returns number of characters*/
   /* actually copied to the buffer including CR and LF.                  */
   /***********************************************************************/
   ulNumChars = export( &iptOrigin, ulBufSize, (char*)pchBuf );

   /***********************************************************************/
   /* Write number of chars from the buffer to the specified file.        */
   /***********************************************************************/
   if (ulNumChars)
     numItems = fwrite((char*)pchBuf, ulNumChars, 1, filePtr);

   /***********************************************************************/
   /* Write out an end of file character and close the file               */
   /***********************************************************************/
   long endfile=26;
   fwrite((char*) &endfile,1,1,filePtr);
   fclose(filePtr);

   /***********************************************************************/
   /* If the original write failed, then return 0 as count written        */
   /***********************************************************************/
   if (numItems != 1)
     return 0;

   return ulNumChars;                  // return number of chars exported
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::export                                                       |
|                                                                              |
| Export the block of text from the insertion point passed into the specified  |
| buffer                                                                       |
------------------------------------------------------------------------------*/
unsigned long IMultiLineEdit::export( long*          ipt,
                                      unsigned long  length,
                                      const char*    buffer )
{
  char* exportBuffer;
  char* tempBuffer = (char*)buffer;
  unsigned long numberToExport, origExport, totalExported=0;

  /********************************************************************/
  /* Allocate the memory to be used to transfer from MLE to buffer    */
  /* MLE can only export in chunks of less than 64K                   */
  /********************************************************************/
  if ( DosAllocMem( (PPVOID)&exportBuffer, 63005,
                     PAG_WRITE | PAG_COMMIT | OBJ_TILE ))
    ITHROWGUIERROR("DosAllocMem");

  /********************************************************************/
  /* Point the export to our recently allocated block of temp storage */
  /********************************************************************/
  handle().sendEvent( MLM_SETIMPORTEXPORT,
                      IEventParameter1(exportBuffer),
                      IEventParameter2(63000));

  /********************************************************************/
  /* As long as there is text to export, do the following             */
  /********************************************************************/
  while (length != 0)
  {
    /******************************************************************/
    /* If remainder is less than largest buffer, use it as export size*/
    /******************************************************************/
    numberToExport = 63000;
    if ( length < 63000 )
      numberToExport = length;

    /******************************************************************/
    /* Save original export size since MLM_EXPORT modifies it and then*/
    /* export the text into the buffer                                */
    /******************************************************************/
    origExport = numberToExport;
    IEventResult evt = handle().sendEvent( MLM_EXPORT,
                                   IEventParameter1(ipt),
                                   IEventParameter2(&numberToExport));

    /******************************************************************/
    /* Now copy the data out of the temp buffer to the final buffer   */
    /******************************************************************/
    memcpy( tempBuffer, exportBuffer, origExport );

    /******************************************************************/
    /* Update external pointer to where to write next block of text   */
    /******************************************************************/
    tempBuffer += origExport;

    /******************************************************************/
    /* Modify counts of text left to export and total exported        */
    /******************************************************************/
    length -= origExport;
//  totalExported += evt.asUnsignedLong();  // Bug in PM causes the return
                                            // value to be one less than
                                            // actual number exported
    totalExported += origExport;            // so use original requested #
  } /* endwhile */
  /********************************************************************/
  /* Free the memory allocated above                                  */
  /********************************************************************/
  DosFreeMem( exportBuffer );

  /********************************************************************/
  /* Return count of text exported                                    */
  /********************************************************************/
  return totalExported;
}


/*------------------------------------------------------------------------------
| IMultiLineEdit::import                                                       |
|                                                                              |
| Import the specified block of text from a buffer into an MLE at the          |
| desired insertion point.                                                     |
------------------------------------------------------------------------------*/
unsigned long IMultiLineEdit::import( long*         ipt,
                                      unsigned long length,
                                      const char*   buffer )
{
  char* importBuffer;
  char* tempBuffer = (char*)buffer;
  unsigned long numberToImport=63000, totalImported=0;

  /********************************************************************/
  /* Verify that we will not exceed the currently set limit on the    */
  /* MLE before trying to import text                                 */
  /********************************************************************/
  if (length+textLength() <= limit())
  {
     /*****************************************************************/
     /* Allocate the memory to be used to transfer from buffer to MLE */
     /* MLE can only import in chunks of less than 64K that are       */
     /* segment aligned                                               */
     /*****************************************************************/
     if ( DosAllocMem( (PPVOID)&importBuffer, 63005,
                       PAG_WRITE | PAG_COMMIT | OBJ_TILE ))
       ITHROWGUIERROR("DosAllocMem");

     /*****************************************************************/
     /* Set up the buffer to import from to the newly allocated one   */
     /*****************************************************************/
     handle().sendEvent( MLM_SETIMPORTEXPORT,
                         IEventParameter1(importBuffer),
                         IEventParameter2(63005));

     /*****************************************************************/
     /* As long as there is data to import, bring it in               */
     /*****************************************************************/
     while (length != 0)
     {
        /**************************************************************/
        /* If smaller than the buffer, change from max to smaller val */
        /**************************************************************/
        if ( length < 63000 )
          numberToImport = length;

        /**************************************************************/
        /* Copy from the external buffer to the segment aligned buffer*/
        /**************************************************************/
        memcpy(importBuffer,tempBuffer,numberToImport);

        /**************************************************************/
        /* Update the external pointer to the next block of text      */
        /**************************************************************/
        tempBuffer += numberToImport;

        /**************************************************************/
        /* Import the text from the segment aligned block to the MLE  */
        /**************************************************************/
        IEventResult evt = handle().sendEvent( MLM_IMPORT,
                                      IEventParameter1(ipt),
                                      IEventParameter2(numberToImport));

        /**************************************************************/
        /* Modify counts of text left to import and total imported    */
        /**************************************************************/
        length -= numberToImport;
        totalImported += evt.asUnsignedLong();
     }

     /*****************************************************************/
     /* Free the memory allocated above                               */
     /*****************************************************************/
     DosFreeMem( importBuffer );
  }
  /********************************************************************/
  /* If limit exceeded, than throw exception stating this             */
  /********************************************************************/
  else
  {
     ITHROWLIBRARYERROR( IC_INVALIDSIZE_LIMIT,
                         IErrorInfo::invalidParameter,
                         IException::recoverable );
  } /* endif */
  /********************************************************************/
  /* Return count of text imported                                    */
  /********************************************************************/
  return totalImported;
}
#endif

/*------------------------------------------------------------------------------
| IMultiLineEdit::textLength                                                   |
|                                                                              |
| Returns the current length in bytes (containing CR/LF) of the MLE text.      |
------------------------------------------------------------------------------*/
unsigned long IMultiLineEdit::textLength() const
{
#ifdef IC_WIN
   return Inherited::textLength();
#else
   IEventResult evt(handle().sendEvent(MLM_QUERYTEXTLENGTH,
                                       IEventParameter1(0),
                                       IEventParameter2(0)));
   return evt.asUnsignedLong();
#endif
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::textLengthAfterFormat                                        |
|                                                                              |
| Returns current length in bytes (containing CR/LF) of the MLE text.          |
------------------------------------------------------------------------------*/
unsigned long IMultiLineEdit::textLengthAfterFormat() const
{
#ifdef IC_WIN
// IC_NOTYET - Approximate format count by adding an additional CR per line
//  since we cannot query current format

   return (textLength() + numberOfLines());
#else
   IEventResult evt(handle().sendEvent(MLM_QUERYFORMATTEXTLENGTH,
                                 IEventParameter1(0),
                                 IEventParameter2((unsigned long) 0xFFFFFFFF)));
   return evt.asUnsignedLong();
#endif
}

/*------------------------------------------------------------------------------
| IMultiLineEdit::selectedTextLength                                           |
|                                                                              |
| Returns the size of the user marked area in bytes (containing CR/LF). Does   |
| not count '\0'.                                                              |
------------------------------------------------------------------------------*/
unsigned long IMultiLineEdit::selectedTextLength() const
{
#ifdef IC_WIN
   unsigned long startPt;                          // Start point of marked area
   unsigned long endPt;                            // End point of marked area

   handle().sendEvent( EM_GETSEL,
                       IEventParameter1( &startPt ),
                       IEventParameter2( &endPt ) );
#else
   IEventResult minSel( handle().sendEvent(MLM_QUERYSEL,
                                           IEventParameter1(MLFQS_MINSEL),
                                           IEventParameter2(0)));
   IEventResult maxSel( handle().sendEvent(MLM_QUERYSEL,
                                           IEventParameter1(MLFQS_MAXSEL),
                                           IEventParameter2(0)));
   unsigned long startPt(minSel.asUnsignedLong());   // Start point of marked area
   unsigned long endPt(maxSel.asUnsignedLong());     // End point of marked area
#endif

   if (startPt == endPt)
     return 0;
   else
   {
#ifdef IC_WIN
// IC_NOTYET - Can calculate characters but not bytes by getting line numbers
//  of selection start and end, counting all of the characters on all of those
//  lines and then subtracting out the unselected characters (as returned by
//  EM_LINELENGTH passing minus one    OR get character index for start point
//  and end point and subtract them
   IEventResult minLine( handle().sendEvent( EM_LINEFROMCHAR,
                                             IEventParameter1(startPt),
                                             IEventParameter2(0)));
   IEventResult maxLine( handle().sendEvent( EM_LINEFROMCHAR,
                                             IEventParameter1(endPt),
                                             IEventParameter2(0)));
   // Approximate byte count for now by using 1 for each character and
   // 2 per line in selected text block (for CR/LF)
   return ((endPt - startPt) +
            2*(maxLine.asUnsignedLong() - minLine.asUnsignedLong()));
#else
      IEventResult evt(handle().sendEvent(MLM_QUERYFORMATTEXTLENGTH,
                                          IEventParameter1(startPt),
                                          IEventParameter2(endPt - startPt)));
      return evt.asUnsignedLong();
#endif
   }
}
