/*
 * This file is part of the portable Forth environment written in ANSI C.
 * Copyright (C) 1993  Dirk Uwe Zoller
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This file is version 0.9.5 of 15-May-94
 * Check for the latest version of this package via anonymous ftp at
 *	roxi.rz.fht-mannheim.de
 *	/pub/unix/languages/pfe-VERSION.tar.gz
 * Please direct any comments via internet to
 *	duz@roxi.rz.fht-mannheim.de.
 * Thank You.
 */
/*
 * lpf83.c ---	Compatibility with Laxen&Perry's F83.
 *
 *		There are lots of useful words in F83 that do not appear
 *		in any standard. This file defines some of them.
 *
 * (duz 06Sep93)
 */

#include <string.h>
#include <ctype.h>

#include "config.h"
#include "forth.h"
#include "support.h"
#include "compiler.h"


/*****************************************************************************/
/* from KERNEL86.BLK							     */
/*****************************************************************************/

Code (bounds)			/* BOUNDS */
{
  Cell h = sp [1];
  sp [1] += sp [0];
  sp [0] = h;
}

Code (perform)			/* PERFORM ( addr --- ) */
{				/* same as `@ EXECUTE' */
  EXECUTE (*(Xt *)*sp++);
}

Code (question_leave)		/* (?LEAVE) */
{
  if (*sp++)
    {
      ip = rp [2] - 1;
      rp += 3;
      BRANCH;
    }
}

code (noop) {}

Code (r_p_fetch)		/* RP@ (--- addr) */
{				/* returns return stack pointer */
  *--sp = (Cell)rp;
}

Code (r_p_store)		/* RP! (addr ---) */
{				/* sets returns stack pointer */
  rp = (Xt **)*sp++;
}

Code (s_p_store)		/* SP! (x addr ---) */
{				/* sets stack pointer */
  sp = *(Cell **)sp;
}

Code (dash_rot)			/* -ROT */
{
  Cell h = sp [2];
  sp [2] = sp [0];
  sp [0] = sp [1];
  sp [1] = h;
}

Code (c_set)			/* CSET ( n addr --- ) */
{				/* set bits in byte at given address */
  *(char *)sp [0] |= (char)sp [1];
  sp += 2;
}

Code (c_reset)			/* CRESET ( n addr --- ) */
{				/* reset bits in byte at given address */
  *(char *)sp [0] &= ~(char)sp [1];
  sp += 2;
}

Code (c_toggle)			/* CTOGGLE ( n addr --- ) */
{				/* toggle bits in byte at given address */
  *(char *)sp [0] ^= (char)sp [1];
  sp += 2;
}

Code (off)			/* OFF */
{
  *(Cell *)*sp++ = FALSE;
}

Code (on)			/* ON */
{
  *(Cell *)*sp++ = TRUE;
}

Code (three_dup)		/* 3DUP */
{
  sp -= 3;
  sp [0] = sp [3];
  sp [1] = sp [4];
  sp [2] = sp [5];
}

Code (four_dup)			/* 4DUP */
{
  sp -= 4;
  sp [0] = sp [4];
  sp [1] = sp [5];
  sp [2] = sp [6];
  sp [3] = sp [7];
}

Code (upc)			/* UPC ( c1 --- c2 ) */
{				/* convert single character to upper case */
  *sp = toupper (*sp);
}

Code (upper)			/* UPPER ( addr cnt --- ) */
{				/* convert string to upper case */
  to_upper ((char *)sp [1], sp [0]);
  sp += 2;
}

Code (skip)			/* SKIP ( addr cnt c --- addr' cnt' ) */
{				/* skip leading characters c */
  char *p = (char *)sp [2];
  Cell n = sp [1];
  char c = (char)*sp++;

  while (n && *p == c)
    n--, p++;
  sp [1] = (Cell)p;
  sp [0] = n;
}

Code (scan)			/* SCAN ( addr cnt c --- addr' cnt' ) */
{				/* scan for first occurence of c in string */
  char *p = (char *)sp [2];
  Cell n = sp [1];
  char c = (char)*sp++;

  while (n && *p != c)
    n--, p++;
  sp [1] = (Cell)p;
  sp [0] = n;
}

Code (place)			/* PLACE ( addr1 len addr2 --- ) */
{				/* store string addr1/len at addr2 */
  Byte *p = (Byte *)sp [0];
  *p = sp [1];
  memcpy ((Byte *)sp [2], p + 1, *p);
  sp += 3;
}

Code (ascii)			/* state smart version of CHAR/[CHAR] */
{
  char *p;
  uCell n;

  parse (' ', &p, &n);
  if (n == 0)
    tHrow (THROW_INVALID_NAME);
  if (STATE)
    {
      compile1 ();
      COMMA (*(Byte *)p);
    }
  else
    *--sp = *(Byte *)p;
}
COMPILES (ascii, literal_execution,
	  SKIPS_CELL, DEFAULT_STYLE);

Code (control)			/* like ASCII but returns char - '@' */
{
  char *p;
  uCell c;
  uCell n;

  parse (' ', &p, &n);
  if (n == 0)
    tHrow (THROW_INVALID_NAME);
  c = *(Byte *)p;
  if ('@' <= c && c <= '_')
    c -= '@';
  if (STATE)
    {
      compile1 ();
      COMMA (c);
    }
  else
    *--sp = c;
}
COMPILES (control, literal_execution,
	  SKIPS_CELL, DEFAULT_STYLE);

Code (number_question)		/* NUMBER? ( addr --- d flag ) */
{				/* convert counted string to number */
  char *p = (char *)*sp;
  sp -= 2;
  sp [0] = number_question (p + 1, *(Byte *)p, (dCell *)&sp [1]);
}

/*****************************************************************************/
/* from EXTEND86.BLK							     */
/*****************************************************************************/

Code (vocs)			/* VOCS */
{
  Wordl *wl = VOC_LINK;

  while (wl != NULL)
    {
      dot_name (to_name (BODY_FROM (wl)));
      wl = wl->prev;
    }
}


LISTWORDS (lpf83) =
{
  OC ("BS",		'\b'),
  CO ("BOUNDS",		bounds),
  CO ("PERFORM",	perform),
  CO ("?LEAVE",		question_leave),
  CO ("NOOP",		noop),
  CO ("RP@",		r_p_fetch),
  CO ("RP!",		r_p_store),
  CO ("SP!",		s_p_store),
  CO ("-ROT",		dash_rot),
  CO ("CSET",		c_set),
  CO ("CRESET",		c_reset),
  CO ("CTOGGLE",	c_toggle),
  CO ("OFF",		off),
  CO ("ON",		on),
  CO ("3DUP",		three_dup),
  CO ("4DUP",		four_dup),
  CO ("UPC",		upc),
  CO ("UPPER",		upper),
  CO ("SKIP",		skip),
  CO ("SCAN",		scan),
  CO ("PLACE",		place),
  CS ("ASCII",		ascii),
  CS ("CONTROL",	control),
  CO ("NUMBER?",	number_question),
  CO ("VOCS",		vocs),
};
COUNTWORDS (lpf83, "L&P F83 compatiblity");
