/*
 * This file is part of the portable Forth environment written in ANSI C.
 * Copyright (C) 1993  Dirk Uwe Zoller
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This file is version 0.9.5 of 15-May-94
 * Check for the latest version of this package via anonymous ftp at
 *	roxi.rz.fht-mannheim.de
 *	/pub/unix/languages/pfe-VERSION.tar.gz
 * Please direct any comments via internet to
 *	duz@roxi.rz.fht-mannheim.de.
 * Thank You.
 */
/*
 * sysdep.c ---	put here any definitions missing in your system
 *		most of this file contributed by Marko Teiste
 * (duz 05Aug93)
 */

#include <stdlib.h>
#include <unistd.h>
#ifdef EMX
# include <io.h>
#endif

#include "config.h"
#undef exit

#if HAVE_POLL
# include <sys/types.h>
# include <poll.h>
#elif HAVE_SELECT
# include <sys/time.h>
#endif

#include "forth.h"


/* ========================================================================= */
#if !defined (HAVE_AH_TRIG)
/* ========================================================================= */

/*
 * Simple acosh(), asinh(), atanh() for those unfortunates who don't
 * have them. These are oversimplified routines (no error or boundry
 * checking). !!! DONT TRUST THESE ROUTINES !!!
 */
#include <math.h>

double acosh (double n)
	{ return log (n + sqrt (n * n - 1)); }
double asinh (double n)
	{ return (n < 0 ? -1.0 : 1.0) * log (fabs (n) + sqrt (n * n + 1)); }
double atanh (double n)
	{ return log (1.0 + ((2.0 * n) / (1.0 - n))) * 0.5; }

/* ========================================================================= */
#endif
#if !defined (HAVE_STRDUP)
/* ========================================================================= */

#include <stdlib.h>
#include <string.h>

char *
strdup (const char *s)
{
  char *p = malloc (strlen (s) + 1);
  return strcpy (p, s);
}

/* ========================================================================= */
#endif
#if !defined (HAVE_MEMMOVE)
/* ========================================================================= */

void
memmove (char *d, const char *s, unsigned n)
{
  if (n)
    if (s > d)
      do
	*d++ = *s++;
      while (--n > 0);
    else
      do
	{
	  --n;
	  d [n] = s [n];
	}
      while (n > 0);
}

/* ========================================================================= */
#endif
#if !defined (HAVE_RENAME)
/* ========================================================================= */

#include <stdio.h>

#if defined (XNX386)
#include <sys/ndir.h>
#define PATH_MAX MAXNAMLEN
#endif

int
rename (const char *source, const char *target)
/*
 * Rename file,
 * This is not a foolproof routine, one of those
 * "I'll do it better when I have more time" things. -mte
 */
{
  char save_name [PATH_MAX+1];	/* Name of saved file */

  if (access(target, 0) == 0)
    {
      sprintf (save_name, "%s~", target);
      if (access (save_name, 0) == 0 ||
	  unlink (save_name) == -1 ||
	  link (target, save_name) == -1 ||
	  unlink (target) == -1)
	return -1;
    }
  if (link (source, target) == -1)
    {
      if (save_name != NULL)
	if (link (save_name, target) == -1)
	  return -1;
	else
	  unlink (save_name);
      return -1;
    }
  if (unlink (source) == -1)
    return -1;
  if (save_name != NULL)
    unlink (save_name);
  return 0;
}

/* ========================================================================= */
#endif
#if !defined (HAVE_ATEXIT)
/* ========================================================================= */

static atexit_fp atexitfun [10];
static int atexitfuns = 0;

int
atexit (atexit_fp fun)
{
  atexitfun [atexitfuns++] = fun;
  return 0;
}

void
trick_exit (int n)
{
  int i;

  for (i = atexitfuns; --i >= 0; )
    (*atexitfun [i]) ();
  exit (n);
}

/* ========================================================================= */
#endif
/* ========================================================================= */

void
millisec (int ms)
/*
 * Somehow wait ms milli-seconds.
 * Versions using poll and select according to Stevens'
 * "Advanced Programming in the UNIX Environment" p.705
 */
{
#if HAVE_USLEEP
  usleep (ms * 1000);
#elif HAVE_POLL
  struct pollfd dummy;
  poll (&dummy, 0, ms);
#elif HAVE_SELECT
  struct timeval tval;
  tval.tv_sec = ms / 1000;
  tval.tv_usec = ms % 1000 * 1000;
  select (0, NULL, NULL, NULL, &tval);
#elif defined (EMX)
  _sleep2 (ms);
#else
  sleep ((ms + 999) / 1000);
#endif
}

int
file_access (const char *fn)
/* return best possible access method, */
/* 0 if no access but file exists, -1 if file not existing (for you) */
{
  if (access (fn, F_OK) != 0)
    return -1;
  if (access (fn, R_OK | W_OK) == 0)
    return FMODE_RW;
  if (access (fn, R_OK) == 0)
    return FMODE_RO;
  if (access (fn, W_OK) == 0)
    return FMODE_WO;
  return 0;
}
